/*-
 * Copyright (c) 2000-2004 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipastat.c,v 1.3 2011/01/23 18:42:35 simon Exp $";
#endif /* !lint */

#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <grp.h>
#include <locale.h>
#include <pwd.h>
#include <regex.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifdef WITH_PTHREAD
# include <pthread.h>
#endif

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "pathnames.h"
#include "memfunc.h"

#include "ipastat_conf.h"
#include "ipastat_log.h"
#include "ipastat_main.h"
#include "ipastat_rules.h"
#include "ipastat_st.h"
#include "ipastat_time.h"

static const char *envprogname;

static void	exit_err(const char *, ...) ATTR_NORETURN
		    ATTR_FORMAT(printf, 1, 2);
static void	exit_errx(const char *, ...) ATTR_NORETURN
		    ATTR_FORMAT(printf, 1, 2);

/*
 * Output the program name, a message, an error message and exit.
 */
static void
exit_err(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	vlogmsg(IPA_LOG_ERR, format, ap);
	va_end(ap);
	exit(EXIT_FAILURE);
}

/*
 * Output the program name, a message and exit.
 */
static void
exit_errx(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	vlogmsgx(IPA_LOG_ERR, format, ap);
	va_end(ap);
	exit(EXIT_FAILURE);
}

/*
 * Output version number (-v and -h switches).
 */
static void
show_version(void)
{
	printf(IPASTAT_NAME ", version " PACKAGE_VERSION "\nRuntime settings: "
#ifdef SYM_PREFIX
	"symbol prefix \"" SYM_PREFIX "\", "
#else
	"symbol prefix is not used, "
#endif
#ifdef WITH_PTHREAD
	"thread-safe mode, "
#else
	"single-threaded mode, "
#endif
#ifdef USE_LIBLTDL
	"using libtool's ltdl library, "
#else
	"using dlopen interface, "
#endif
#ifdef WITH_MEMFUNC_DEBUG
	"memfunc debugging is enabled"
#else
	"memfunc debugging is disabled"
#endif
	"\nSupports: rules"
#ifdef WITH_LIMITS
	", limits"
#endif
#ifdef WITH_THRESHOLDS
	", thresholds"
#endif
	".\n");
}

/*
 * Set supplementary groups for process, gid is a GID from getpwnam(),
 * NGROUPS_MAX is defined in SUSv3.
 */
static int
setsuppgids(const char *user, gid_t gid)
{
	gid_t gids[NGROUPS_MAX + 1];
	const struct group *grp;
	char *const *gr_user;
	int i, ngids;

	gids[0] = gid;
	ngids = 1;

	errno = 0;
	while ((grp = getgrent()) != NULL) {
		for (i = 0; i < ngids; ++i)
			if (grp->gr_gid == gids[i])
				goto next;
		for (gr_user = grp->gr_mem; *gr_user != NULL; ++gr_user)
			if (strcmp(user, *gr_user) == 0)
				if (ngids < NGROUPS_MAX + 1) {
					gids[ngids++] = grp->gr_gid;
					break;
				}
next:		;
	}
	if (errno != 0) {
		logmsg(IPA_LOG_ERR, "setsuppgids: getgrent");
		return (-1);
	}
	if (setgroups(ngids, gids) < 0) {
		logmsg(IPA_LOG_ERR, "setsuppgids: setgroups");
		return (-1);
	}
	return (0);
}

/*
 * Output help message (-h switch).
 */
static void
usage(void)
{
	show_version();
	printf("\
Usage: %s [options]\n\
 Statistics viewer\n\
 Options are:\n\
  -c dir\t\tSet the directory "IPASTAT_NAME" should chroot(2) into\n\
\t\t\timmediately, working directory is not changed\n\
  -f conf_file\t\tUse given configuration file instead of using default\n\
\t\t\tconfiguration file %s\n\
  -t\t\t\tCheck the configuration file (two switches mimic\n\
\t\t\treal configuration regime) and exit\n",
	    envprogname, IPASTAT_CONF_FILE);
	printf("\
  -q options\t\tQuery statistics from modules\n\
  -u user\t\tChange UID of the running copy of "IPASTAT_NAME" to user\n\
  -g group\t\tChange GID of the running copy of "IPASTAT_NAME" to group\n\
  -h\t\t\tOutput this help message and exit\n\
  -v\t\t\tShow version number and exit\n");
	printf(" Options for the -q switch (details in the manual page):\n\
  -a rules\t\tOutput all rules\n"
#ifdef WITH_LIMITS
"  -a limits\t\tOutput all limits (-r)\n"
#endif
#ifdef WITH_THRESHOLDS
"  -a thresholds\t\tOutput all thresholds (-r)\n"
#endif
"  -s stat\t\tUse given statistics system\n\
  -r rule\t\tSpecify a rule name\n"
#ifdef WITH_LIMITS
"  -l limit\t\tSpecify a limit name (-r)\n"
#endif
#ifdef WITH_THRESHOLDS
"  -t threshold\t\tSpecify a threshold name (-r)\n"
#endif
"  -i|I interval\t\tSpecify time interval\n\
  -x regexp\t\tFilter names with the given regular expression (-a)\n");
}

/*
 * Implement "-u user" and "-g group" options.
 */
static void
change_user(const char *u_name, const char *g_name)
{
	const struct passwd *pwd;
	const struct group *grp;
	char *endptr;
	uid_t uid;
	gid_t gid;

	if (u_name == NULL && g_name == NULL)
		return;

	if (g_name != NULL) {
		/* -g group */
		errno = 0;
		grp = getgrnam(g_name);
		if (grp == NULL) {
			if (errno != 0)
				exit_err("change_user: getgrnam(%s)", g_name);
			if (!isdigit((unsigned char)*g_name))
				exit_errx("cannot find group %s", g_name);
			errno = 0;
			gid = (gid_t)strtoul(g_name, &endptr, 10);
			if (errno != 0)
				exit_err("change_user: strtoul");
			if (g_name == endptr || *endptr != '\0')
				exit_errx("cannot find group %s", g_name);
		} else
			gid = grp->gr_gid;
		if (setgroups(1, &gid) < 0)
			exit_err("change_user: setgroups(1, [%lu])",
			    (unsigned long)gid);
	}

	if (u_name != NULL) {
		/* -u user */
		errno = 0;
		pwd = getpwnam(u_name);
		if (pwd == NULL) {
			if (errno != 0)
				exit_err("change_user: getpwnam(%s)", u_name);
			if (!isdigit((unsigned char)*u_name))
				exit_errx("cannot find user %s", u_name);
			if (g_name == NULL)
				exit_errx("argument in the -u option is not "
				    "a valid user name");
			errno = 0;
			uid = (uid_t)strtoul(u_name, &endptr, 10);
			if (errno != 0)
				exit_err("change_user: strtoul");
			if (u_name == endptr || *endptr != '\0')
				exit_errx("cannot find user %s", u_name);
		} else {
			uid = pwd->pw_uid;
			if (g_name == NULL) {
				gid = pwd->pw_gid;
				if (setsuppgids(pwd->pw_name, gid) < 0)
					exit_errx("change_user: cannot set all "
					    "groups for user %s", u_name);
			}
		}
	}

	if (setgid(gid) < 0)
		exit_err("change_user: setgid(%lu)", (unsigned long)gid);

	if (u_name != NULL && setuid(uid) < 0)
		exit_err("change_user: setuid(%lu)", (unsigned long)uid);

	/*
	 * To be sure, that descriptors used by getpw*() and getgr*()
	 * functions are closed.
	 */
	endpwent();
	endgrent();
}

#ifdef WITH_LIMITS
# define Q_OPTSTRING_LIMIT "l:"
#else
# define Q_OPTSTRING_LIMIT ""
#endif

#ifdef WITH_THRESHOLDS
# define Q_OPTSTRING_THRESHOLD "t:"
#else
# define Q_OPTSTRING_THRESHOLD ""
#endif

#define Q_OPTSTRING ":a:i:I:r:s:" Q_OPTSTRING_LIMIT Q_OPTSTRING_THRESHOLD "x:"

static void
parse_query(int argc, char *argv[])
{
#ifdef WITH_ANY_LIMITS
	const struct opt_rule *opt_rule;
	unsigned int type;
#endif
	const char *name;
	size_t len;
	int opt;

	while ((opt = getopt(argc, argv, Q_OPTSTRING)) != -1)
		switch (opt) {
		case ':':
			exit_errx("-q: option requires an argument -- %c",
			    optopt);
			/* NOTREACHED */
		case '?':
			exit_errx("-q: illegal option -- %c", optopt);
			/* NOTREACHED */
		case 'I':
			if (opt_tint_add(optarg, 1) < 0)
				exit_errx("-q: cannot add time interval");
			break;
		case 'a':
			if (a_flag != A_FLAG_ABSENT)
				exit_errx("-q: duplicated option -- a");
			len = strlen(optarg);
			if (strncmp(optarg, "rules", len) == 0)
				a_flag = A_FLAG_RULES;
			else
#ifdef WITH_LIMITS
			if (strncmp(optarg, "limits", len) == 0)
				a_flag = A_FLAG_LIMITS;
			else
#endif
#ifdef WITH_THRESHOLDS
			if (strncmp(optarg, "thresholds", len) == 0)
				a_flag = A_FLAG_THRESHOLDS;
			else
#endif
				exit_errx("-q: unknown option value \"%s\" "
				    "in the -a option", optarg);
			break;
		case 'i':
			if (opt_tint_add(optarg, 0) < 0)
				exit_errx("-q: cannot add time interval");
			break;
#ifdef WITH_LIMITS
		case 'l':
			if (cur_opt_rule == NULL)
				exit_errx("-q: the -l option can be used "
				    "only after the -r option");
			name = optarg;
			if (validate_name(name) < 0)
				exit_errx("-q: illegal %s name \"%s\"",
				    "limit", name);
			if (opt_limit_add(name) < 0)
				exit_errx("-q: cannot add limit");
			break;
#endif
		case 'r':
			name = optarg;
			if (validate_name(name) < 0)
				exit_errx("-q: illegal %s name \"%s\"",
				    "rule", name);
			if (opt_rule_add(name) < 0)
				exit_errx("-q: cannot add rule");
			break;
		case 's':
			if (opt_st_add(optarg) < 0)
				exit_errx("-q: cannot add statistics "
				    "systems names");
			break;
#ifdef WITH_THRESHOLDS
		case 't':
			if (cur_opt_rule == NULL)
				exit_errx("-q: the -t option can be used "
				    "only after the -r option");
			name = optarg;
			if (validate_name(name) < 0)
				exit_errx("-q: illegal %s name \"%s\"",
				    "threshold", name);
			if (opt_threshold_add(name) < 0)
				exit_errx("-q: cannot add threshold");
			break;
#endif
		case 'x':
			x_pat = optarg;
			x_reg_ptr = &x_reg;
			break;
		default:
			exit_errx("-q: unexpected option -- %c", optopt);
		}

	if (optind < argc)
		exit_errx("-q: non-switch argument \"%s\"", argv[optind]);

	switch (a_flag) {
	case A_FLAG_ABSENT:
		break;
	case A_FLAG_RULES:
		if (!STAILQ_EMPTY(&opt_rules_list) ||
		    !STAILQ_EMPTY(&opt_tint_list))
			exit_errx("-q: \"-a rules\" should be used alone");
		break;
#ifdef WITH_LIMITS
	case A_FLAG_LIMITS:
		if (STAILQ_EMPTY(&opt_rules_list))
			exit_errx("-q: \"-a limits\" requires the -r option");
		if (has_opt_limits || has_opt_thresholds)
			exit_errx("-q: \"-a limits\" requires only "
			    "the -r option");
		break;
#endif
#ifdef WITH_THRESHOLDS
	case A_FLAG_THRESHOLDS:
		if (STAILQ_EMPTY(&opt_rules_list))
			exit_errx("-q: \"-a thresholds\" requires "
			    "the -r option");
		if (has_opt_limits || has_opt_thresholds)
			exit_errx("-q: \"-a thresholds\" requires only "
			    "the -r option");
		break;
#endif
	}

#if defined(WITH_ANY_LIMITS) && defined(WITH_THRESHOLDS)
	if (has_opt_limits && has_opt_thresholds)
		exit_errx("-q: do not mix -l and -t options");
#endif

#ifdef WITH_ANY_LIMITS
	if (!STAILQ_EMPTY(&opt_rules_list)) {
		type = STAILQ_FIRST(&opt_rules_list)->type;
		STAILQ_FOREACH(opt_rule, &opt_rules_list, link)
			if (opt_rule->type != type)
				exit_errx("-q: incorrect combination of "
				    "-l or -t options with -r options");

# ifdef WITH_THRESHOLDS
		if (type == OPT_RULE_THRESHOLD && !STAILQ_EMPTY(&opt_tint_list))
			exit_errx("-q: do not use -i option with "
			    "the -t option");
# endif
	}
#endif /* WITH_ANY_LIMITS */

	if (x_pat != NULL) {
		int error;

		error = regcomp(&x_reg, x_pat, REG_EXTENDED|REG_NOSUB);
		if (error != 0)
			exit_errx("-q: regcomp(%s): %s", x_pat,
			    regerrbuf(error));
	}
}

int
main(int argc, char *argv[])
{
	const char *u_name;		/* -u user */
	const char *g_name;		/* -g group */
	int opt;			/* Current option. */
	int rv;				/* Return value */
	char q_flag;			/* Set if -q. */
	char t_flag;			/* Set if -t. */

	/* Save the program name. */
	envprogname = strrchr(argv[0], '/');
	if (envprogname != NULL)
		++envprogname;
	else
		envprogname = argv[0];

	mypid = (long)getpid();

	log_ident = envprogname;
	mvlogmsgx = mvlogmsgx_wrapper;
	memfunc_pre_init();
	if (memfunc_init() < 0)
		exit_errx("main: memfunc_init failed");

	m_anon = mem_type_new_local(MTYPE_NAME(anonymous),
	    "Anonymous memory", 0);
	if (m_anon == NULL)
		exit_errx("main: mem_type_new_local failed");

	if (init_time_data() < 0)
		exit_errx("main: init_time_data failed");

	u_name = g_name = NULL;
	q_flag = t_flag = 0;
	opterr = 0;
	while ((opt = getopt(argc, argv, ":c:f:g:hi:qtu:v")) != -1)
		switch (opt) {
		case ':':
			exit_errx("option requires an argument -- %c", optopt);
			/* NOTREACHED */
		case '?':
			exit_errx("illegal option -- %c", optopt);
			/* NOTREACHED */
		case 'c':
			if (*optarg != '/')
				exit_errx("path in the -c option should be "
				    "absolute");
			if (chroot(optarg) < 0)
				exit_err("main: chroot(%s)", optarg);
			break;
		case 'f':
			ipastat_conf_file = optarg;
			break;
		case 'g':
			g_name = optarg;
			break;
		case 'h':
			usage();
			return (EXIT_SUCCESS);
		case 'i':
			log_ident = optarg;
			break;
		case 'q':
			q_flag = 1;
			goto getopt_done;
		case 't':
			if (t_flag)
				mimic_real_config = 1;
			else
				t_flag = 1;
			break;
		case 'u':
			u_name = optarg;
			break;
		case 'v':
			show_version();
			return (EXIT_SUCCESS);
		default:
			exit_errx("unexpected option -- %c", optopt);
		}

	if (optind < argc)
		exit_errx("non-switch argument \"%s\"", argv[optind]);

getopt_done:
	change_user(u_name, g_name);

#ifdef HAVE_DL_INIT
	if (dl_init() != 0)
		exit_errx("main: dl_init failed");
#endif

	if (setlocale(LC_ALL, "") == NULL)
		exit_err("setlocale");

	if (t_flag) {
		if (configure(TEST_PARSING) < 0)
			return (EXIT_FAILURE);
		show_config();
		return (EXIT_SUCCESS);
	}

	if (!q_flag)
		exit_errx("statistics query is missed, use the -q option");

	parse_query(argc, argv);

	rv = EXIT_FAILURE;

	if (configure(CONFIG_PARSING) < 0) {
		logmsgx(IPA_LOG_ERR, "main: configure failed");
		goto done;
	}

	if (opt_st_parse() < 0) {
		logmsgx(IPA_LOG_ERR, "cannot parse statistics names "
		    "given in command line");
		goto done;
	}

	if (opt_rules_parse() < 0) {
		logmsgx(IPA_LOG_ERR, "cannot completely parse command line");
		goto done;
	}

	if (ipastat_main() < 0)
		logmsgx(IPA_LOG_ERR, "cannot perform query");
	else
		rv = EXIT_SUCCESS;

	if (deinit_all() < 0)
		rv = EXIT_FAILURE;

	opt_tint_free();
	opt_st_free();
	opt_rules_free();

	if (free_all() < 0)
		rv = EXIT_FAILURE;

done:
#ifdef HAVE_DL_EXIT
	if (dl_exit() != 0) {
		logmsgx(IPA_LOG_ERR, "main: dl_exit failed");
		rv = EXIT_FAILURE;
	}
#endif

	if (x_pat != NULL)
		regfree(&x_reg);

	return (rv);
}
