/*-
 * Copyright (c) 2003 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *   @(#)$Id: ipa_limits.h,v 1.2 2011/01/23 18:42:34 simon Exp $
 */

#ifndef IPA_LIMITS_H
#define IPA_LIMITS_H

#ifdef WITH_LIMITS

#ifndef LIMIT_NSIZE
# define LIMIT_NSIZE	30
#endif

#ifndef LIMIT_NALLOC
# define LIMIT_NALLOC	20
#endif

#ifdef WITH_SUBLIMITS

#ifndef SUBLIMIT_NSIZE
# define SUBLIMIT_NSIZE	LIMIT_NSIZE
#endif

#ifndef SUBLIMIT_NALLOC
# define SUBLIMIT_NALLOC LIMIT_NALLOC
#endif

struct rule;
struct limit;

#define SUBLIMIT_FLAG_REACHED	0x1	/* Sublimit is reached. */

#define SUBLIMIT_IS_REACHED(x)	   ((x)->subl_flags == SUBLIMIT_FLAG_REACHED)
#define SUBLIMIT_IS_NOTREACHED(x)  (!SUBLIMIT_IS_REACHED(x))

#define SUBLIMIT_SET_REACHED(x)	   ((x)->subl_flags = SUBLIMIT_FLAG_REACHED)
#define SUBLIMIT_SET_NOTREACHED(x) ((x)->subl_flags = 0)

/*
 * limit { sublimit {}} section.
 */
struct sublimit {
	STAILQ_ENTRY(sublimit) link;	/* Link for list of sublimits. */

	uint64_t	lim;		/* sublimit <limit> */

	unsigned int	subl_flags;	/* ORed SUBLIMIT_FLAG_xxx */

	struct cmds	reach;		/* { reach {}} */

	struct wpid	wpid;		/* Sublimit's wpid structure. */

	struct limit	*limit;		/* Pointer to sublimit's limit. */

	unsigned char	cnt_type;	/* Type of "sublimit" argument. */
	unsigned int	lim_pc;		/* sublimit <limit> if it is xx%. */
	char		*name;		/* sublimit <limit> string. */

	struct cmds_limit rc[2];	/* { startup } and { shutdown } */
};

/*
 * List of all sublimits in one limit.
 */
STAILQ_HEAD(sublimits_list, sublimit);

extern ipa_mzone *sublimit_mzone;

extern struct sublimit *sublimit_by_name(const struct limit *, const char *);

extern int	reach_sublimit(const struct rule *, const struct limit *,
		    struct sublimit *);

extern void	sublimit_init_cmds(struct sublimit *);

#endif /* WITH_SUBLIMITS */

/*
 * limit { expire {}} section.
 */
struct expire {
	struct texp	expire;		/* expire { expire } */
	struct cmds	cmds;		/* Commands. */
};

/*
 * limit { restart {}} section.
 */
struct restart {
	struct texp	restart;	/* restart { restart } */
	struct cmds	cmds;		/* Commands. */
};

#define LIMIT_FLAG_ACTIVE	0x01	/* Limit is active. */
#define LIMIT_FLAG_REACHED	0x02	/* Limit is reached. */
#define LIMIT_FLAG_SET		0x04	/* "limit" parameter was given. */
#define LIMIT_FLAG_INITED	0x08	/* Limit was initialized. */
#define LIMIT_FLAG_BUSY		0x10	/* Limit is busy. */

#define LIMIT_IS_ACTIVE(x)	((x)->lim_flags & LIMIT_FLAG_ACTIVE)
#define LIMIT_IS_INACTIVE(x)	(!LIMIT_IS_ACTIVE(x))
#define LIMIT_IS_REACHED(x)	((x)->lim_flags & LIMIT_FLAG_REACHED)
#define LIMIT_IS_NOTREACHED(x)	(!LIMIT_IS_REACHED(x))
#define LIMIT_IS_SET(x)		((x)->lim_flags & LIMIT_FLAG_SET)
#define LIMIT_IS_NOTSET(x)	(!LIMIT_IS_SET(x))
#define LIMIT_IS_INITED(x)	((x)->lim_flags & LIMIT_FLAG_INITED)
#define LIMIT_IS_BUSY(x)	((x)->lim_flags & LIMIT_FLAG_BUSY)

#define LIMIT_SET_ACTIVE(x)	((x)->lim_flags |= LIMIT_FLAG_ACTIVE)
#define LIMIT_SET_INACTIVE(x)	((x)->lim_flags &= ~LIMIT_FLAG_ACTIVE)
#define LIMIT_SET_REACHED(x)	((x)->lim_flags |= LIMIT_FLAG_REACHED)
#define LIMIT_SET_NOTREACHED(x)	((x)->lim_flags &= ~LIMIT_FLAG_REACHED)
#define LIMIT_SET_INITED(x)	((x)->lim_flags |= LIMIT_FLAG_INITED)
#define LIMIT_SET_BUSY(x)	((x)->lim_flags |= LIMIT_FLAG_BUSY)
#define LIMIT_SET_UNBUSY(x)	((x)->lim_flags &= ~LIMIT_FLAG_BUSY)

/*
 * rule { limit {}} section.
 */
struct limit {
	STAILQ_ENTRY(limit) link;	/* Link for list of limits. */

	unsigned int	no;		/* Limit ordinal number. */

	unsigned int	lim_flags;	/* ORed LIMIT_FLAG_xxx */

	uint64_t	lim;		/* { limit } */
	uint64_t	cnt;		/* Positive counter. */
	uint64_t	cnt_neg;	/* Negative counter. */

	const struct db_list *db_list;	/* { db_list } */

#ifdef WITH_SUBLIMITS
	struct sublimits_list sublimits;/* All { sublimit {}} */
#endif

	struct restart	restart;	/* { restart {}} */
	struct cmds	reach;		/* { reach {}} */
	struct expire	expire;		/* { expire {}} */
	const struct worktime *worktime;/* { worktime } */

	ipa_tm		event_tm;	/* Exact time when to check some
					   limit's event. */
	unsigned int	event_sec;	/* Time when to check some limit's
					   event. */
	unsigned int	event_date_set;	/* The same as in ipa_limit_state. */
	ipa_tm	event_date[IPA_LIMIT_EVENT_NUM]; /* See ipa_limit_state. */

	struct wpid	wpid;		/* Limit's wpid structure. */

	const struct rule *rule;	/* Pointer to limit's rule. */

	unsigned char	cnt_type;	/* Type of "limit" parameter. */
	signed char	load_limit;	/* { load_limit } */

	char		*name;		/* Name of this limit. */
	char		*info;		/* { info } */

	struct cmds_limit rc[2];	/* { startup } and { shutdown } */
};

/*
 * List of all limits in one rule.
 */
STAILQ_HEAD(limits_list, limit);

extern const char *const limit_event_msg[];

extern signed char global_debug_limit;
extern signed char global_debug_limit_init;
extern signed char global_load_limit;

extern ipa_mzone *limit_mzone;

extern int	limit_add_chunk(const struct rule *, struct limit *,
		    const uint64_t *);
extern int	limit_sub_chunk(const struct rule *, struct limit *,
		    const uint64_t *);
extern int	limits_add_chunk(const struct rule *, const uint64_t *);
extern int	limits_sub_chunk(const struct rule *, const uint64_t *);

extern int	init_limits(const struct rule *);
extern int	check_limits(const struct rule *, unsigned int *);
extern void	limit_set_event_sec(struct limit *);
extern int	limits_newday(struct rule *);

extern int	restart_limit(const struct rule *, struct limit *);
extern int	reach_limit(const struct rule *, struct limit *);
extern int	expire_limit(const struct rule *, struct limit *);

#define set_limit_inactive(r, l) mod_set_limit_active((r), (l), 0)
extern int	mod_set_limit_active(const struct rule *, struct limit *, int);

extern int	copy_limits(struct rule *, const struct limits_list *);
extern void	free_limits(unsigned int, struct limits_list *, int);

extern void	limit_init_cmds(struct limit *);
extern void	limit_inherit(struct limit *);

extern struct limit *limit_by_name(const struct rule *, const char *);

#endif /* WITH_LIMITS */

extern unsigned int nstatlimits;
extern unsigned int ndynlimits;

extern unsigned int nstatsublimits;
extern unsigned int ndynsublimits;

#endif /* !IPA_LIMITS_H */
