/*-
 * Copyright (c) 2003 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: confcommon.c,v 1.2.2.1 2011/11/15 18:12:29 simon Exp $";
#endif /* !lint */

#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <locale.h>
#include <regex.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "memfunc.h"
#include "parser.h"

#if (SIZEOF_LONG * CHAR_BIT) < 32
# error "'long' is less than 32 bits: too small"
#endif

#if (SIZEOF_LONG_LONG * CHAR_BIT) < 64
# error "'long long' is less than 64 bits: too small"
#endif

/* Number of spaces for indentation of outputted configuration. */
#ifndef CONF_INDENT
# define CONF_INDENT	4
#endif

/*
 * All strto_xxx() functions get strings with positive decimal
 * integers, so it is not necessary to check '-' in the first
 * character of a string inside strto_xxx() functions (remember,
 * that strtoul() and strtoull() work with negative values).
 */

#define LOG_BUF_SIZE	1024

const char *const conf_event_msg[] = {
	"GLOBAL_BEGIN",		/*  0 */
	"GLOBAL_END",		/*  1 */
	"RULE_BEGIN",		/*  2 */
	"RULE_END",		/*  3 */
	"LIMIT_BEGIN",		/*  4 */
	"LIMIT_END",		/*  5 */
	"THRESHOLD_BEGIN",	/*  6 */
	"THRESHOLD_END",	/*  7 */
	"AUTORULE_BEGIN",	/*  8 */
	"AUTORULE_END",		/*  9 */
	"RULEPAT_BEGIN",	/* 10 */
	"RULEPAT_END",		/* 11 */
	"CUSTOM_SECT_BEGIN",	/* 12 */
	"CUSTOM_SECT_END"	/* 13 */
};

int		value_units;	/* value_units parameter. */
char		got_arg_value;	/* Set if get_arg_value() was called. */
char		need_nl = 0;	/* Need a new line. */
static char	was_space = 0;	/* There was already space in output. */

const char	*curmodfile;	/* File name of the current module. */
const char	*curparam;	/* Current parameter name. */
const char	*cursect;	/* Current section name. */

static unsigned int conf_indent = 0; /* Current output indentation. */

void		*conf_sect_he_mzone;
void		*conf_param_he_mzone;

regex_t		re_time;
regex_t		re_bytes;

/* Return values from get_arg_*(). */
static char	*arg_string;
static char	*arg_misc;
static int	arg_int;
static int32_t	arg_int32;
static uint32_t	arg_uint32;
static int64_t	arg_int64;
static uint64_t	arg_uint64[2];

extern ipa_mem_type *m_anon;

/*
 * Just call regexec(3), ignoring result of match.
 */
int
regexec_simple(const regex_t *re, const char *str)
{
	return (regexec(re, str, 0, (regmatch_t *)NULL, 0));
}

/*
 * Form an error message in buffer according to code using
 * regerror(3) and return pointer to it.
 */
const char *
regerrbuf(int code)
{
	static char buf[128];

	regerror(code, (regex_t *)NULL, buf, sizeof(buf));
	return (buf);
}

/*
 * Print indentation spaces.
 */
static void
print_indent(void)
{
	unsigned int i;

	for (i = conf_indent; i > 0; --i)
		printf(" ");
}

/*
 * This function is called by modules to output their log messages
 * about errors occurred during parsing configuration file.
 * If code is not equal to zero, then output error string as well.
 */
void
mod_logconferr(const char *mod_name, int code, const char *format, va_list ap)
{
	char buf[LOG_BUF_SIZE];
	const struct parser_pb *pb;
	const char *err_pre, *err_msg;
	char *msg;
	int rv, msg_alloced;

	msg_alloced = 0;
	rv = vsnprintf(buf, sizeof(buf), format, ap);
	if (rv < 0)
		msg = "(mod_logconferr: vsnprintf failed)";
	else if (rv < sizeof(buf))
		msg = buf;
	else if (rv >= sizeof(buf)) {
		msg = mem_malloc(++rv, m_anon);
		if (msg == NULL)
			msg = "(mod_logconferr: mem_malloc failed)";
		else if (vsnprintf(msg, rv, format, ap) < 0) {
			mem_free(msg, m_anon);
			msg = "(mod_logconferr: vsnprintf failed)";
		} else
			msg_alloced = 1;
	}

	if (code == 0)
		err_pre = err_msg = "";
	else {
		err_pre = ": ";
		err_msg = strerror(code);
	}

	pb = parser_top_pb();
	if (pb != NULL) {
		const char *token_type, *token_name, *token_tail;

		if (curparam != NULL) {
			token_type = " parameter \"";
			token_name = curparam;
			token_tail = "\":";
		} else if (cursect != NULL) {
			token_type = " section \"";
			token_name = cursect;
			token_tail = "\":";
		} else
			token_type = token_name = token_tail = "\0";
		logconfe("%s:%u%s: MOD %s:%s%s%s %s%s%s",
		    pb->fname, pb->lineno,
		    pb->fp != NULL ? "" : "(in variable's value)",
		    mod_name, token_type, token_name, token_tail,
		    msg, err_pre, err_msg);
	} else
		logconfe("MOD %s: %s%s%s", mod_name, msg, err_pre, err_msg);

	if (msg_alloced)
		mem_free(msg, m_anon);
}

/*
 * Output an error message for some line in the configuration file.
 * If code is not zero, then output error string as well.
 * This function is called from places where pb is defined.
 */
static void
vlogconf(int code, const char *format, va_list ap)
{
	char buf[LOG_BUF_SIZE];
	const struct parser_pb *pb;
	const char *token_type, *token_name, *token_tail;
	const char *err_pre, *err_msg, *in_var_msg;
	char *msg;
	int rv, msg_alloced;

	if (curparam != NULL) {
		token_type = " parameter \"";
		token_name = curparam;
		token_tail = "\":";
	} else if (cursect != NULL) {
		token_type = " section \"";
		token_name = cursect;
		token_tail = "\":";
	} else
		token_type = token_name = token_tail = "\0";

	msg_alloced = 0;
	rv = vsnprintf(buf, sizeof(buf), format, ap);
	if (rv < 0)
		msg = "(vlogconf: vsnprintf failed)";
	else if (rv < sizeof(buf))
		msg = buf;
	else if (rv >= sizeof(buf)) {
		msg = mem_malloc(++rv, m_anon);
		if (msg == NULL)
			msg = "(vlogconf: mem_malloc failed)";
		else if (vsnprintf(msg, rv, format, ap) < 0) {
			mem_free(msg, m_anon);
			msg = "(vlogconf: vsnprintf failed)";
		} else
			msg_alloced = 1;
	}

	if (code == 0)
		err_pre = err_msg = "";
	else {
		err_pre = ": ";
		err_msg = strerror(code);
	}

	pb = parser_top_pb();
	in_var_msg = pb->fp != NULL ? "" : "(in variable's value)";
	if (curmodfile == NULL)
		logconfe("%s:%u%s:%s%s%s %s%s%s", pb->fname, pb->lineno,
		    in_var_msg, token_type, token_name, token_tail,
		    msg, err_pre, err_msg);
	else
		logconfe("%s:%u%s: module %s:%s%s%s %s%s%s", pb->fname,
		    pb->lineno, in_var_msg, curmodfile, token_type,
		    token_name, token_tail, msg, err_pre, err_msg);

	if (msg_alloced)
		mem_free(msg, m_anon);
}

/*
 * Log an error message during configuration,
 * use errno as error code.
 */
void
logconf(const char *format, ...)
{
	va_list ap;
	int errno_save;

	errno_save = errno;
	va_start(ap, format);
	vlogconf(errno_save, format, ap);
	va_end(ap);
}

/*
 * Log an error message during configuration,
 * do not use errno.
 */
void
logconfx(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	vlogconf(0, format, ap);
	va_end(ap);
}

/*
 * Convert string to uint32_t.
 */
int
strto_uint32(uint32_t *result, const char *nptr, char **endptr_ret)
{
	char *endptr;
	unsigned long val;

	errno = 0;
	val = strtoul(nptr, &endptr, 10);
	if (errno != 0) {
		logconf("strtoul");
		return (-1);
	}
#if (SIZEOF_LONG * CHAR_BIT) > 32
	if (val > UINT32_MAX) {
		logconfx("too big value for 'uint32_t' type");
		return (-1);
	}
#endif
	if (nptr == endptr) {
		logconfx("wrong number");
		return (-1);
	}
	*result = (uint32_t)val;
	if (endptr_ret != NULL)
		*endptr_ret = endptr;
	return (0);
}

#ifndef HAVE_STRTOULL
/*
 * On the system (on which I tried to build IPA) that does not
 * have strtoull(), ULLONG_MAX also was not defined.
 */
# ifndef ULLONG_MAX
#  define ULLONG_MAX	UINT64_MAX
# endif

/*
 * Since this function always is called for base 10, third argument
 * is not used, and since first argument always points to a string
 * with all digits checks and implementation can be simplified
 */
/* ARGSUSED2 */
static unsigned long long
strtoull(const char *nptr, char **endptr, int base ATTR_UNUSED)
{
	unsigned long long x;
	const char *ptr;
	char c;

	x = 0;
	for (ptr = nptr; *ptr != '\0'; ++ptr) {
		c = *ptr - '0';
		if (x > (ULLONG_MAX / 10) ||
		    (x == (ULLONG_MAX / 10) && c > (ULLONG_MAX % 10))) {
			errno = ERANGE;
			return (ULLONG_MAX);
		}
		x *= 10;
		x += c;
	}
	if (endptr != NULL)
		*endptr = (char *)ptr;
	return (x);
}
#endif /* !HAVE_STRTOULL */

/*
 * Convert string to uint64_t.
 */
int
strto_uint64(uint64_t *result, const char *nptr, char **endptr_ret)
{
	unsigned long long val;
	char *endptr;

	errno = 0;
	val = strtoull(nptr, &endptr, 10);
	if (errno != 0) {
		logconf("strtoull");
		return (-1);
	}
#if (SIZEOF_LONG_LONG * CHAR_BIT) > 64
	if (val > UINT64_MAX) {
		logconfx("too big value for 'uint64_t' type");
		return (-1);
	}
#endif
	if (nptr == endptr) {
		logconfx("wrong number");
		return (-1);
	}
	*result = (uint64_t)val;
	if (endptr_ret != NULL)
		*endptr_ret = endptr;
	return (0);
}

/*
 * Convert string to 'unsigned int'.
 */
int
strto_u_int(unsigned int *result, const char *nptr, char **endptr)
{
	unsigned long val;

	errno = 0;
	val = strtoul(nptr, endptr, 10);
	if (errno != 0) {
		logconf("strtoul");
		return (-1);
	}
#if ULONG_MAX > UINT_MAX
	if (val > UINT_MAX) {
		logconfx("too big value for 'unsigned int' type");
		return (-1);
	}
#endif
	*result = (unsigned int)val;
	return (0);
}

/*
 * Check whether the given string is an unsigned integer.
 */
static int
check_unsigned_integer(const char *ptr, const char *what)
{
	for (; *ptr != '\0'; ++ptr)
		if (!isdigit((unsigned char)*ptr)) {
			logconfx("argument should be %s integer", what);
			return (-1);
		}
	return (0);
}

/*
 * Convert string to uint32_t, allow a user to specify
 * a `+' sign before number.
 */
static int
get_arg_uint32(void *res)
{
	char *ptr;

	ptr = parser_args;
	if (*ptr == '+')
		++ptr;
	if (check_unsigned_integer(ptr, "a positive") < 0)
		return (-1);
	return (strto_uint32((uint32_t *)res, ptr, (char **)NULL));
}

/*
 * Convert string to int32_t, allow a user to specify
 * `-' and `+' signs before number.
 */
static int
get_arg_int32(void *res)
{
	char *ptr;
	uint32_t val;
	char sign;

	ptr = parser_args;
	switch (*ptr) {
	case '-':
		sign = 1;
		++ptr;
		break;
	case '+':
		++ptr;
		/* FALLTHROUGH */
	default:
		sign = 0;
	}

	if (check_unsigned_integer(ptr, "an") < 0)
		return (-1);
	if (strto_uint32(&val, ptr, (char **)NULL) < 0)
		return (-1);

	if (sign) {
		/* This can be wrong on system where |INT32_MIN| > INT32_MAX */
		if (val > INT32_MAX) {
			logconfx("too little value for 'int32_t' type");
			return (-1);
		}
		*(int32_t *)res = -(int32_t)val;
	} else {
		if (val > INT32_MAX) {
			logconfx("too big value for 'int32_t' type");
			return (-1);
		}
		*(int32_t *)res = (int32_t)val;
	}

	return (0);
}

/*
 * Convert string to int64_t, allow a user to specify
 * `-' and `+' signs before number.
 */
static int
get_arg_int64(void *res)
{
	uint64_t val;
	char *ptr;
	char sign;

	ptr = parser_args;
	switch (*ptr) {
	case '-':
		sign = 1;
		++ptr;
		break;
	case '+':
		++ptr;
		/* FALLTHROUGH */
	default:
		sign = 0;
	}

	if (check_unsigned_integer(ptr, "an") < 0)
		return (-1);
	if (strto_uint64(&val, ptr, (char **)NULL) < 0)
		return (-1);

	if (sign) {
		/* This can be wrong on system where |INT64_MIN| > INT64_MAX */
		if (val > INT64_MAX) {
			logconfx("too little value for 'int64_t' type");
			return (-1);
		}
		*(int64_t *)res = -(int64_t)val;
	} else {
		if (val > INT64_MAX) {
			logconfx("too big value for 'int64_t' type");
			return (-1);
		}
		*(int64_t *)res = (int64_t)val;
	}

	return (0);
}

/*
 * Convert string to uint64_t, allow a user to specify
 * a `+' sign before number.
 */
int
get_arg_uint64(void *res)
{
	char *ptr;

	ptr = parser_args;
	if (*ptr == '+')
		++ptr;
	if (check_unsigned_integer(ptr, "a positive") < 0)
		return (-1);
	return (strto_uint64((uint64_t *)res, ptr, (char **)NULL));
}

int
get_arg_time(void *res)
{
	uint64_t value, result;
	char *ptr, *endptr;
	char level, error, overflow;

	result = 0;
	ptr = parser_args;
	level = error = overflow = 0;

	for (;;) {
		if (strto_uint64(&value, ptr, &endptr) < 0)
			return (-1);
		ptr = endptr;
		switch (*ptr) {
		case 'h':
			if (level > 0)
				error = 1;
			else if (value > UINT64_MAX / SECONDS_IN_HOUR)
				overflow = 1;
			else {
				level = 1;
				value *= SECONDS_IN_HOUR;
			}
			break;
		case 'm':
			if (level > 1)
				error = 1;
			else if (value > UINT64_MAX / SECONDS_IN_MINUTE)
				overflow = 1;
			else {
				level = 2;
				value *= SECONDS_IN_MINUTE;
			}
			break;
		default: /* 's' */
			if (level > 2)
				error = 1;
			else
				level = 3;
		}
		if (error) {
			logconfx("wrong time format");
			return (-1);
		}
		if (overflow || result > UINT64_MAX - value) {
			logconfx("too big value for 'uint64_t' type");
			return (-1);
		}
		result += value;

		if (*++ptr == '\0') {
			/* EOL */
			break;
		}
		if (*ptr == ' ')
			++ptr;
	}
	*(uint64_t *)res = result;
	return (0);
}

/*
 * Parse a value for the boolean variable,
 * all comparisons are case insensitive.
 */
static int
get_arg_boolean(void *res)
{
	const char *ptr;

	ptr = parser_args;
	if (strcasecmp(ptr, "yes") == 0)
		*(int *)res = 1;
	else if (strcasecmp(ptr, "no") == 0)
		*(int *)res = 0;
	else {
		logconfx("argument should be \"yes\" or \"no\"");
		return (-1);
	}
	return (0);
}

static int
get_arg_string(void *res)
{
	char *ptr;

	if (!parser_arg_is_str()) {
		logconfx("argument should be a string");
		return (-1);
	}
	ptr = parser_strdup(parser_args, m_parser);
	if (ptr == NULL)
		return (-1);
	*(char **)res = ptr;
	return (0);
}

int
get_arg_bytes(void *res)
{
	uint64_t value, result;
	char *ptr, *endptr;
	char level, error, overflow;

	result = 0;
	ptr = parser_args;
	level = error = overflow = 0;

	for (;;) {
		if (strto_uint64(&value, ptr, &endptr) < 0)
			return (-1);
		ptr = endptr;
		switch (*ptr) {
		case 'T':
			if (level > 0)
				error = 1;
			else if (value > UINT64_MAX / TBYTE)
				overflow = 1;
			else {
				level = 1;
				value *= TBYTE;
			}
			break;
		case 'G':
			if (level > 1)
				error = 1;
			else if (value > UINT64_MAX / GBYTE)
				overflow = 1;
			else {
				level = 2;
				value *= GBYTE;
			}
			break;
		case 'M':
			if (level > 2)
				error = 1;
			else if (value > UINT64_MAX / MBYTE)
				overflow = 1;
			else {
				level = 3;
				value *= MBYTE;
			}
			break;
		case 'K':
			if (level > 3)
				error = 1;
			else if (value > UINT64_MAX / KBYTE)
				overflow = 1;
			else {
				level = 4;
				value *= KBYTE;
			}
			break;
		default: /* 'B' */
			if (level > 4)
				error = 1;
			else
				level = 5;
		}
		if (error) {
			logconfx("wrong format of an argument");
			return (-1);
		}
		if (overflow || result > UINT64_MAX - value) {
			logconfx("too big value for 'uint64_t' type");
			return (-1);
		}
		result += value;

		if (*++ptr == '\0') {
			/* EOL */
			break;
		}
		if (*ptr == ' ')
			++ptr;
	}
	*(uint64_t *)res = result;
	return (0);
}

int
get_arg_value(void *res)
{
	uint64_t type, val;
	char *ptr;

	ptr = parser_args;
	if (regexec_simple(&re_bytes, ptr) == 0) {
		if (get_arg_bytes(&val) < 0)
			return (-1);
		type = IPA_CONF_TYPE_BYTES;
	} else if (regexec_simple(&re_time, ptr) == 0) {
		if (get_arg_time(&val) < 0)
			return (-1);
		type = IPA_CONF_TYPE_TIME;
	} else {
		if (get_arg_uint64(&val) < 0)
			return (-1);
		type = IPA_CONF_TYPE_UINT64;
	}
	if (value_units > 0 && type != value_units) {
		logconfx("type of value is not the same as allowed "
		    "by the \"value_units\" parameter");
		return (-1);
	}
	*(uint64_t *)res = type;
	*((uint64_t *)res + 1) = val;
	got_arg_value = 1;
	return (0);
}

int
get_arg_per_cent(void *res)
{
	char *ptr;
	uint32_t pc;

	ptr = parser_args + parser_args_len - 1;
	if (*ptr != '%') {
		logconfx("wrong format of an argument");
		return (-1);
	}
	*ptr = '\0';
	if (get_arg_uint32(&pc) < 0)
		return (-1);
	if (pc > 100) {
		logconfx("per cent value should be <= 100%%");
		return (-1);
	}
	*(int *)res = pc;
	return (0);
}

/*
 * Simply point *res to read string from configuration file.
 */
static int
get_arg_misc(void *res)
{
	*(char **)res = parser_args;
	return (0);
}

static unsigned int
conf_hash_value(const char *name)
{
	unsigned int hash_value;

	for (hash_value = 0; *name != '\0'; ++name)
		hash_value += *name;
	return (hash_value);
}

#define sect_hash_value(x) conf_hash_value(x)
#define param_hash_value(x) conf_hash_value(x)

#define sect_hash_bucket(x) ((x) & (CONF_SECT_BUCKETS - 1))
#define param_hash_bucket(x) ((x) & (CONF_PARAM_BUCKETS - 1))

static int
init_conf_sect_tbl(const char *mod_file, int build_re, ipa_conf_sect *tbl,
    struct conf_sect_hash **hash_ptr)
{
	const char *errbuf;
	const ipa_conf_sect *sect;
	struct conf_sect_hash *hash;
	struct conf_sect_he *he;
	unsigned int i;
	int error;

	/* Allocate hash buckets. */
	hash = mem_malloc(CONF_SECT_BUCKETS * sizeof(*hash), m_anon);
	if (hash == NULL) {
		logconfe("init_conf_sect_tbls: mem_malloc failed");
		return (-1);
	}
	*hash_ptr = hash;

	/* Initialize head of each hash bucket. */
	for (i = 0; i < CONF_SECT_BUCKETS; ++i)
		SLIST_INIT(&hash[i]);

	/* Build regular expressions and hash entries. */
	for (sect = tbl, i = 0; sect->sect_name != NULL; ++i, ++sect) {
		if (sect->arg_pattern != NULL && build_re) {
			error = regcomp(sect->arg_regexp, sect->arg_pattern,
			    REG_EXTENDED|REG_NOSUB);
			if (error != 0) {
				errbuf = regerrbuf(error);
				if (mod_file == NULL)
					logconfe("init_conf_sect_tbls: regcomp"
					    "(\"%s\"): %s", sect->arg_pattern,
					    errbuf);
				else
					logconfe("init_conf_sect_tbls for "
					    "module %s: regcomp(\"%s\"): %s",
					    mod_file, sect->arg_pattern,
					    errbuf);
				return (-1);
			}
		}
		he = mzone_alloc(conf_sect_he_mzone);
		if (he == NULL) {
			logconfe("init_conf_sect_tbls: mzone_malloc failed");
			return (-1);
		}
		he->sect_name = sect->sect_name;
		he->idx = i;
		he->hash_value = sect_hash_value(he->sect_name);
		SLIST_INSERT_HEAD(
		    &hash[sect_hash_bucket(he->hash_value)], he, link);
	}
	return (0);
}

static int
init_conf_param_tbl(const char *mod_file, int build_re, ipa_conf_param *tbl,
    struct conf_param_hash **hash_ptr)
{
	const char *errbuf;
	const ipa_conf_param *param;
	struct conf_param_hash *hash;
	struct conf_param_he *he;
	unsigned int i;
	int error;

	/* Allocate hash buckets. */
	hash = mem_malloc(CONF_PARAM_BUCKETS * sizeof(*hash), m_anon);
	if (hash == NULL) {
		logconfe("init_conf_param_tbls: mem_malloc failed");
		return (-1);
	}
	*hash_ptr = hash;

	/* Initialize head of each hash bucket. */
	for (i = 0; i < CONF_PARAM_BUCKETS; ++i)
		SLIST_INIT(&hash[i]);

	/* Build regular expressions and hash entries. */
	for (param = tbl, i = 0; param->param_name != NULL; ++i, ++param) {
		if (param->arg_pattern != NULL && build_re) {
			error = regcomp(param->arg_regexp, param->arg_pattern,
			    REG_EXTENDED|REG_NOSUB);
			if (error != 0) {
				errbuf = regerrbuf(error);
				if (mod_file == NULL)
					logconfe("init_conf_param_tbls: regcomp"
					    "(\"%s\"): %s", param->arg_pattern,
					    errbuf);
				else
					logconfe("init_conf_param_tbls for "
					    "module %s: regcomp(\"%s\"): %s",
					    mod_file, param->arg_pattern,
					    errbuf);
				return (-1);
			}
		}
		he = mzone_alloc(conf_param_he_mzone);
		if (he == NULL) {
			logconfe("init_conf_param_tbls: mzone_alloc failed");
			return (-1);
		}
		he->param_name = param->param_name;
		he->idx = i;
		he->hash_value = param_hash_value(he->param_name);
		SLIST_INSERT_HEAD(
		    &hash[param_hash_bucket(he->hash_value)], he, link);
	}
	return (0);
}

/*
 * Compile regular expressions and initialize hash tables for
 * configuration sections and parameters.
 */
int
init_conf_tbls(const char *mod_file, int build_re,
    ipa_conf_sect *sect_tbl, struct conf_sect_hash **sect_hash_ptr,
    ipa_conf_param *param_tbl, struct conf_param_hash **param_hash_ptr)
{
	if (sect_tbl == NULL)
		*sect_hash_ptr = NULL;
	else if (init_conf_sect_tbl(mod_file, build_re, sect_tbl,
	    sect_hash_ptr) < 0)
		return (-1);

	if (param_tbl == NULL)
		*param_hash_ptr = NULL;
	else if (init_conf_param_tbl(mod_file, build_re, param_tbl,
	    param_hash_ptr) < 0)
		return (-1);

	return (0);
}

const ipa_conf_sect *
find_conf_sect(const ipa_conf_sect *tbl, const struct conf_sect_hash *hash,
    const char *sect_name)
{
	if (hash != NULL) {
		const struct conf_sect_hash *bucket;
		const struct conf_sect_he *he;
		unsigned int hash_value;

		hash_value = sect_hash_value(sect_name);
		bucket = &hash[sect_hash_bucket(hash_value)];

		SLIST_FOREACH(he, bucket, link)
			if (he->hash_value == hash_value &&
			    strcmp(he->sect_name, sect_name) == 0)
				return (&tbl[he->idx]);
	}
	return (NULL);
}

const ipa_conf_param *
find_conf_param(const ipa_conf_param *tbl, const struct conf_param_hash *hash,
    const char *param_name)
{
	if (hash != NULL) {
		const struct conf_param_hash *bucket;
		const struct conf_param_he *he;
		unsigned int hash_value;

		hash_value = param_hash_value(param_name);
		bucket = &hash[param_hash_bucket(hash_value)];

		SLIST_FOREACH(he, bucket, link)
			if (he->hash_value == hash_value &&
			    strcmp(he->param_name, param_name) == 0)
				return (&tbl[he->idx]);
	}
	return (NULL);
}

/*
 * Release memory hold by compiled regular expressions, if
 * free_regexp is non-zero, for configuration sections and
 * parameters, hash entries are freed all at once with one
 * mzone_deinit() call.
 */
void
deinit_conf_tbls(int free_re,
    ipa_conf_sect *sect_tbl, struct conf_sect_hash *sect_hash,
    ipa_conf_param *param_tbl, struct conf_param_hash *param_hash)
{
	/* Release memory hold by hash buckets. */
	mem_free(sect_hash, m_anon);
	mem_free(param_hash, m_anon);

	if (!free_re)
		return;

	/* Release memory hold by regular expressions. */
	if (sect_tbl != NULL) {
		const ipa_conf_sect *sect;

		for (sect = sect_tbl; sect->sect_name != NULL; ++sect)
			if (sect->arg_pattern != NULL)
				regfree(sect->arg_regexp);
	}
	if (param_tbl != NULL) {
		const ipa_conf_param *param;

		for (param = param_tbl; param->param_name != NULL; ++param)
			if (param->arg_pattern != NULL)
				regfree(param->arg_regexp);
	}
}

const char *
plural_form(unsigned int n)
{
	return (n != 1 ? "s" : "");
}

/*
 * All print_*() functions are trivial, just take care about
 * need_nl and was_space variables.
 */

struct time_conv {
	uint64_t	div;
	uint64_t	val;
	char		ch;
};

static struct time_conv time_conv_tbl[] = {
	{ SECONDS_IN_HOUR,	0, 'h' },
	{ SECONDS_IN_MINUTE,	0, 'm' },
	{ 1,			0, 's' }
};

#define TIME_CONV_TBL_SIZE (sizeof(time_conv_tbl) / sizeof(time_conv_tbl[0]))

void
print_time(const uint64_t *ptr)
{
	uint64_t a;
	int i, i1, i2;

	a = *ptr;
	i1 = i2 = -1;
	for (i = 0; i < TIME_CONV_TBL_SIZE; ++i) {
		if ((time_conv_tbl[i].val = a / time_conv_tbl[i].div) != 0) {
			if (i1 < 0)
				i1 = i;
			i2 = i;
		}
		a %= time_conv_tbl[i].div;
	}
	if (i1 >= 0) {
		for (i = i1;; ++i) {
			printf("%"PRIu64"%c", time_conv_tbl[i].val,
			    time_conv_tbl[i].ch);
			if (i == i2)
				break;
			printf(" ");
		}
	} else
		printf("0s");
}

struct byte_conv {
	uint64_t	div;
	unsigned int	val;
	char		ch;
};

static struct byte_conv byte_conv_tbl[] = {
	{ TBYTE,	0, 'T' },
	{ GBYTE,	0, 'G' },
	{ MBYTE,	0, 'M' },
	{ KBYTE,	0, 'K' },
	{ 1,		0, 'B' }
};

#define BYTE_CONV_TBL_SIZE (sizeof(byte_conv_tbl) / sizeof(byte_conv_tbl[0]))

void
print_bytes(const uint64_t *ptr)
{
	uint64_t a;
	int i, i1, i2;

	a = *ptr;
	i1 = i2 = -1;
	for (i = 0; i < BYTE_CONV_TBL_SIZE; ++i) {
		if ((byte_conv_tbl[i].val = a / byte_conv_tbl[i].div) != 0) {
			if (i1 < 0)
				i1 = i;
			i2 = i;
		}
		a %= byte_conv_tbl[i].div;
	}
	if (i1 >= 0) {
		for (i = i1;; ++i) {
			printf("%u%c", byte_conv_tbl[i].val,
			    byte_conv_tbl[i].ch);
			if (i == i2)
				break;
			printf(" ");
		}
	} else
		printf("0B");
}

void
print_value(const uint64_t *value, unsigned int value_type)
{
	switch (value_type) {
	case IPA_CONF_TYPE_UINT64:
		printf("%"PRIu64, *value);
		break;
	case IPA_CONF_TYPE_BYTES:
		print_bytes(value);
		break;
	case IPA_CONF_TYPE_TIME:
		print_time(value);
		break;
	}
}

void
print_boolean(int boolean)
{
	was_space = 0;
	if (boolean)
		printf("yes");
	else
		printf("no");
}

void
print_string(const char *str)
{
	was_space = 0;
	printf("\"");
	for (; *str != '\0'; ++str)
		switch (*str) {
		case '\t':
			printf("\\t");
			break;
		case '\n':
			printf("\\n");
			break;
		case '\"':
		case '\\':
			printf("\\");
			/* FALLTHROUGH */
		default:
			printf("%c", *str);
		}
	printf("\"");
}

void
mod_print_param_name(const char *prefix, const char *param_name)
{
	print_indent();
	if (prefix != NULL)
		printf("%s:", prefix);
	printf("%s = ", param_name);
	was_space = 1;
}

void
mod_print_param_end(void)
{
	print_param_end();
	need_nl = 1;
}

void
mod_print_args(const char *format, va_list ap)
{
	was_space = 0;
	vprintf(format, ap);
}

void
print_nl(void)
{
	printf("\n");
}

void
print_nl_cond(void)
{
	if (need_nl) {
		need_nl = 0;
		print_nl();
	}
}

void
mod_print_sect_name(const char *prefix, const char *sect_name)
{
	print_indent();
	if (prefix != NULL)
		printf("%s:", prefix);
	printf("%s ", sect_name);
	was_space = 1;
}

void
mod_print_sect_end(void)
{
	print_sect_end();
	need_nl = 1;
}

void
print_space(void)
{
	if (!was_space) {
		was_space = 1;
		printf(" ");
	}
}

void
print_param_name(const char *name)
{
	print_indent();
	printf("%s = ", name);
}

void
print_param_name0(const char *name)
{
	print_indent();
	printf("%s ", name);
}

void
print_param_end(void)
{
	printf(";\n");
}

void
print_sect_name(const char *name)
{
	print_indent();
	printf("%s ", name);
}

void
print_sect_begin(void)
{
	printf("{\n");
	conf_indent += CONF_INDENT;
}

void
print_sect_end(void)
{
	conf_indent -= CONF_INDENT;
	print_indent();
	printf("}\n");
}

struct conf_re {
	const char	*pat;	/* Regular expression pattern. */
	regex_t		*re;	/* Compiled regular expression. */
};

static struct conf_re conf_re_tbl[] = {
	{ PAT_BYTES,	&re_bytes	},
	{ PAT_TIME,	&re_time	}
};

#define CONF_RE_TBL_SIZE (sizeof(conf_re_tbl) / sizeof(conf_re_tbl[0]))

/*
 * Build regular expressions for parsing some IPA_CONF_TYPE_xxx.
 */
int
build_conf_re(void)
{
	static char called = 0;

	const struct conf_re *cre;
	unsigned int i;
	int error;

	if (called)
		return (0);
	called = 1;

	for (i = 0, cre = conf_re_tbl; i < CONF_RE_TBL_SIZE; ++cre, ++i) {
		error = regcomp(cre->re, cre->pat, REG_EXTENDED|REG_NOSUB);
		if (error != 0) {
			logconfe("build_conf_re: regcomp(\"%s\"): %s",
			    cre->pat, regerrbuf(error));
			return (-1);
		}
	}

	return (0);
}

void *
dl_lookup_sym(dl_handle handle, const char *sym)
{
	void *ptr;
#ifdef SYM_PREFIX
	char *psym;
#endif

#ifdef SYM_PREFIX
	if (mem_asprintf(m_anon, &psym, SYM_PREFIX"%s", sym) < 0) {
		logconfx("dl_lookup_sym: mem_asprintf failed");
		return (NULL);
	}
	ptr = dl_sym(handle, psym);
	mem_free(psym, m_anon);
	if (ptr != NULL)
		return (ptr);
#endif

	ptr = dl_sym(handle, sym);
	if (ptr == NULL)
		logconfx("dl_lookup_sym: dl_sym(\"%s\"): %s", sym, dl_error());

	return (ptr);
}

/*
 * Any symbol in any name must be letter, digit or punctuation (except
 * double quote and both slashes) from the ASCII character set.
 */
int
validate_name(const char *s)
{
	unsigned char c;

	if (*s == '\0')
		return (-1);
	do {
		c = (unsigned char)*s;
		if (!isascii(c) || !(isalnum(c) || ispunct(c)) ||
		    c == '\"' || c == '/' || c == '\\')
			return (-1);
	} while (*++s != '\0');
	return (0);
}

/*
 * A wrapper for validate_name() with logging during configuration parsing.
 */
int
conf_validate_name(const char *s)
{
	if (validate_name(s) < 0) {
		logconfx("illegal symbol in name");
		return (-1);
	}
	return (0);
}

/*
 * Get module's name from the file name "[/path/]foobar[-x.y.z][.so]".
 * Return "foobar" part from the given path name.
 */
char *
get_mod_name(char *file_name)
{
	char *ptr, *ptr2, *name;

	ptr = strrchr(file_name, '/');
	if (ptr != NULL)
		++ptr;			/* foobar[-x.y.z].[so] */
	else
		ptr = file_name;
	name = mem_strdup(ptr, m_anon);
	if (name == NULL) {
		logconfx("get_mod_name: mem_strdup failed");
		return (NULL);
	}
	ptr2 = strchr(name, '.');
	if (ptr2 != NULL)
		*ptr2 = '\0';		/* foobar[-x] */
	ptr = strrchr(name, '-');
	if (ptr != NULL) {
		for (ptr2 = ptr; *ptr2 != '\0'; ++ptr2)
			if (!isdigit((unsigned char)*ptr2))
				break;
		if (*ptr2 == '\0')
			*ptr = '\0';	/* foobar */
	}
	return (name);
}

/* get_arg_tbl[] must be ordered by IPA_CONF_TYPE_xxx values. */
const struct get_arg get_arg_tbl[] = {
	{ &arg_int32,	get_arg_int32,	  NULL		}, /* TYPE_INT32    */
	{ &arg_uint32,	get_arg_uint32,	  NULL		}, /* TYPE_UINT32   */
	{ &arg_int64,	get_arg_int64,	  NULL		}, /* TYPE_INT64    */
	{ arg_uint64,	get_arg_uint64,	  NULL		}, /* TYPE_UINT64   */
	{ &arg_string,	get_arg_string,	  NULL		}, /* TYPE_STRING   */
	{ arg_uint64,	get_arg_bytes,	  &re_bytes	}, /* TYPE_BYTES    */
	{ arg_uint64,	get_arg_time,	  &re_time	}, /* TYPE_TIME     */
	{ arg_uint64,	get_arg_value,	  NULL		}, /* TYPE_VALUE    */
	{ &arg_int,	get_arg_per_cent, NULL		}, /* TYPE_PER_CENT */
	{ &arg_int,	get_arg_boolean,  NULL		}, /* TYPE_BOOLEAN  */
	{ &arg_misc,	get_arg_misc,	  NULL		}  /* TYPE_MISC     */
};
