/*
 * HANDS.C - generate 30 (or so) bridge hands in a compact format
 *
 * 21 June 1994 -- Released into the public domain.  
 * Authors Morris Jones and Matt Clegg.
 *
 * usage: hands [ -s <seed number> ]
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#ifdef MSDOS
#include <time.h>
#else
#include <sys/time.h>
#endif
#define _BRIDGE_
#include "types.h"

static char SCCSID[] = "$Id: hands.c,v 1.4 1997/02/16 17:47:13 mojo Exp $";	/* RCS keywords */

/* --------------------------------------------------------------------- */

/* Globals */

char print_buffer[13][98];	/* array holds one row of hands for printing */
long seed;
int seed_provided = 0;
deal cards[36];				/* the decks */
int starting_hand = 0;		/* starting hand for this page */
int porter_deals = 0;		/* if true, print dealing info */
char header[98];			/* page header */
char footer[98];			/* page footer */
char nl[] = "\n";
char ff[] = "\f";
int ps = 0;					/* If true, generate PostScript output */

/* --------------------------------------------------------------------- */

/* Forward references */

int main(int argc,char *argv[],char *envp[]);
void shuffle_the_deck(deal cards);
void srand48(long seed);
long lrand48(void);
void print_hand(int print_offset,deal cards,int hand_number);
void print_suit(int suit,int player,deal cards,int offset);
void print_page(int starting_hand);
void bprint(char *out,char *in);
void print_usage(void);
int getopt(int, char **, char *);
long atol(char *);
time_t time(time_t *);
void exit(int);
void print_porter(deal,int);
void print_postscript_prolog(void);
void print_ps_header(void);
void print_ps_footer(void);
void ps_print_page(int starting_hand);
void ps_print_hand(int hand_number);
void ps_print_suit(int suit,int player,deal hand);
void ps_print_porter();

/* --------------------------------------------------------------------- */

/* Macros */

#if (defined(MSDOS) || defined(USE_RAND))
#define RANDOM rand
#define SRANDOM(seed) srand(seed)
#elif (defined(USE_RANDOM))
#define RANDOM random
#define SRANDOM(seed) srandom(seed)
#else
#define RANDOM lrand48
#define SRANDOM(seed) srand48(seed)
#endif
#define random_mod(n) ((RANDOM()/64) % (n))

/* --------------------------------------------------------------------- */

int
main(int argc,char *argv[],char *envp[])
{
	int hand_number = 0;
	int i, line;
	int c;
	extern char *optarg;
	extern int optind;
	
/* Process the command line (not many options currently) */

	while ((c=getopt(argc,argv,"apqh:f:s:")) != -1) {
		switch (c) {
			case 'a':
				ps = 1;
				break;
			case 's':
				seed_provided = 1;
				seed = atol(optarg);
				break;
			case 'p':
				porter_deals = 1;
				break;
			case 'q':
				porter_deals = 2;
				break;
			case 'h':
				memset(header,0,sizeof(header));
				strncpy(header,optarg,96);
				break;
			case 'f':
				memset(footer,0,sizeof(footer));
				strncpy(footer,optarg,96);
				break;
			case '?':
			default:
				print_usage();
				exit(1);
				break;
		}
	}

/* shuffle 36 decks */

	if (!seed_provided) (void)time(&seed);
	SRANDOM(seed);

	for (i=0;i<36;++i) shuffle_the_deck(cards[i]);

/* Generate PostScript prolog (if required) */

	if (ps) print_postscript_prolog();

/* Print the first page (15 hands) */

	hand_number = starting_hand;
	if (ps) {
		puts("%%Page: 1");
		puts("save");
	}
	if (header[0]) {
		if (ps) {
			print_ps_header();
		} else {
			fputs(header,stdout);
			fputs(nl,stdout);
			fputs(nl,stdout);
		}
	}

	(ps) ? ps_print_page(hand_number) : print_page(hand_number);

	if (footer[0]) {
		if (ps) {
			print_ps_footer();
		} else {
			fputs(nl,stdout);
			fputs(footer,stdout);
			fputs(nl,stdout);
		}
	}
	if (ps) {
		puts("restore");
		puts("showpage");
	} else {
		fputs(ff,stdout);
	}

/* Print the second page (15 hands) */

	hand_number += 15;
	if (ps) {
		puts("%%Page: 2");
		puts("save");
	}
	if (header[0]) {
		if (ps) {
			print_ps_header();
		} else {
			fputs(header,stdout);
			fputs(nl,stdout);
			fputs(nl,stdout);
		}
	}

	(ps) ? ps_print_page(hand_number) : print_page(hand_number);

/* Print the seed number at the bottom */

	if (footer[0]) {
		if (ps) {
			print_ps_footer();
		} else {
			fputs(nl,stdout);
			fputs(footer,stdout);
			fputs(nl,stdout);
		}
	}

	if (ps) {
		printf("TIMES setfont\n");
		printf("576 56 moveto\n");
		printf("(S = %lu)\n",seed);
		puts("dup stringwidth pop neg 0 rmoveto show");

	} else {
		printf("S = %lu\n",seed);
	}
	if (ps) {
		puts("restore");
		puts("showpage");
	} else {
		fputs(ff,stdout);
	}

/* Print Porter deals if requested */

	if (porter_deals) {
		if (ps) {
			ps_print_porter();
		} else {
			fputs(header,stdout);
			fputs(nl,stdout);
			for (i=0;i<30;i++) print_porter(cards[i],i);
		}
	}

	if (ps) puts("%%EOF");
	return 0;
}

/* --------------------------------------------------------------------- */

void
print_usage()
{
	fprintf(stderr,"usage: hands [ options ]\n");
	fprintf(stderr,"    -s seed   Use given number (long) for seed\n");
	fprintf(stderr,"    -p        Print Gary Porter style dealing info after hands\n");
	fprintf(stderr,"    -q        Print Gary Porter style, using ^>v<, not NeSw\n");
	fprintf(stderr,"    -h \"hdr\"  Add the header following\n");
	fprintf(stderr,"    -f \"ftr\"  Add the footer following\n");
	fprintf(stderr,"    -a        Generate PostScript (Adobe) output\n");
}

/* --------------------------------------------------------------------- */

/* Prints five rows of three hands.  Includes some magic numbers that
 * relate to the column locations, and boundaries of vertical and
 * horizontal rules.  I don't feel the urge to generalize this beyond
 * this point right now.  print_hand() is more general, but still
 * refers to the global print buffer array.
 */

void 
print_page(int hand_number)
{
	int line, i;

/* Initialize the print buffer, including vertical rules */

	for (line =0; line<5; ++line) {
		for (i=0;i<13;++i) {
			memset(print_buffer[i],' ',96);
			print_buffer[i][96] = '\n';
			print_buffer[i][97] = 0;
		}
		for (i=0;i<12;++i) {
			print_buffer[i][30] = '|';
			print_buffer[i][62] = '|';
		}

/* Put three hands in the print buffer, replacing existing spaces */

		print_hand(0,cards[hand_number],hand_number);
		hand_number++;
		print_hand(32,cards[hand_number],hand_number);
		hand_number++;
		print_hand(64,cards[hand_number],hand_number);
		hand_number++;

/* Feed the print buffer to stdout, and print a separator line if
 * this isn't the last row of hands
 */

		if (line != 4) {
			memset(print_buffer[12],'-',92);
			print_buffer[12][30] = '+';
			print_buffer[12][62] = '+';
			for (i=0;i<13;++i) {
				fputs(print_buffer[i],stdout);
			}
		}
		else {
			for (i=0;i<12;++i) {
				fputs(print_buffer[i],stdout);
			}
		}
	}
}

/* --------------------------------------------------------------------- */

/* print one hand at specified x offset
 *
 * This routine takes full advantage of the data structures designed
 * by Matt Clegg for bridge hands, suits, players, and vulnerabilities.
 * It references the print buffer as an x,y character positions, adding
 * the x offset specified.  (Future expansion might add a y offset as
 * well.)
 */

void 
print_hand(int print_offset,deal cards,int hand_number)
{
	int player, suit;
	int i;
	int x,y;
	char *p;
	char *dlr;
	char *vul;
	int vul_index;
	char buf[20];

/* Create strings for dealer and vulnerability */

	switch (dealer_list[hand_number % 16]) {
		case PLAYER_NORTH:
			dlr = "N";
			break;
		case PLAYER_EAST:
			dlr = "E";
			break;
		case PLAYER_SOUTH:
			dlr = "S";
			break;
		case PLAYER_WEST:
			dlr = "W";
			break;
	}

	vul_index = ns_vulnerability_list[hand_number % 16] + 
	  2 * ew_vulnerability_list[hand_number % 16];

	switch (vul_index) {
		case 0:
			vul = "None";
			break;
		case 1:
			vul = "N-S";
			break;
		case 2:
			vul = "E-W";
			break;
		case 3:
			vul = "Both";
			break;
	}

/* Install board number, dealer, and vulnerability in the print buffer */

	p = &print_buffer[0][0+print_offset];
	sprintf(buf,"Bd  %d",hand_number + 1);
	bprint(p,buf);

	p = &print_buffer[1][0+print_offset];
	sprintf(buf,"Dlr %s",dlr);
	bprint(p,buf);

	p = &print_buffer[2][0+print_offset];
	sprintf(buf,"Vul %s",vul);
	bprint(p,buf);

/* Loop on the players and the suits to install the cards themselves */

	for ( player = 0; player < 4 ; ++player) {
		for ( suit = 0; suit < 4; ++suit) {
			print_suit(suit,player,cards,print_offset);
		}
	}
}

/* --------------------------------------------------------------------- */

/* print one suit for one hand
 *
 * This routine references an array of x,y coordinates that is indexed
 * by player and suit.  suit_offsets[][] is defined in types.h
 */

void 
print_suit(int suit,int player,deal cards,int offset)
{
	int i;
	char *p;
	int x, y;
	char buf[10];

	x = suit_offsets[player][suit].x + offset;
	y = suit_offsets[player][suit].y;
	p = &print_buffer[y][x];

	*p++ = *suit_names[suit];
	p++;
	for ( i=12; i >= 0; --i ) {
		if (cards[i + 13*suit] == player) {
			*p++ = *rank_names[i];
		}
	}
}

/* --------------------------------------------------------------------- */

/* Use this routine instead of sprintf() to avoid having a NULL term */

void 
bprint(char *out,char *in)
{
	while (*in)  *out++ = *in++;
}

/* --------------------------------------------------------------------- */

/* Print hands in Gary Porter style:
 * Board 1  NeWssWNNW
 */

void
print_porter(deal cards,int board)
{
	int card;
	int suit;
	int marker[4];	/* four player markers */

	if (porter_deals == 1) {
		marker[PLAYER_NORTH] = 'N';
		marker[PLAYER_EAST] = 'e';
		marker[PLAYER_SOUTH] = 'S';
		marker[PLAYER_WEST] = 'w';
	}
	else if (porter_deals == 2) {
		marker[PLAYER_NORTH] = '^';
		marker[PLAYER_EAST] = '>';
		marker[PLAYER_SOUTH] = 'v';
		marker[PLAYER_WEST] = '<';
	}

	printf("Board %2d   ",board+1);

	for (suit = SUIT_SPADES; suit >= 0; --suit) {
		for (card = 12; card >= 0; --card) {
			switch (cards[card+13*suit]) {
				case PLAYER_NORTH:
					putchar(marker[PLAYER_NORTH]);
					break;
				case PLAYER_EAST:
					putchar(marker[PLAYER_EAST]);
					break;
				case PLAYER_SOUTH:
					putchar(marker[PLAYER_SOUTH]);
					break;
				case PLAYER_WEST:
					putchar(marker[PLAYER_WEST]);
					break;
			}
		}
		putchar(' ');
	}
	printf("\n\n\n");
}

/* --------------------------------------------------------------------- */

/* Almost direct from okbridge -- the seed is installed up in main()
 */

void 
shuffle_the_deck (deal cards)
/* Using the algorithm suggested by Douglas Foxvog.  Thanks, Doug! */
{
	int i, t, c;
	deal shuffle;

	for (i = 0; i < 52; i++) 
		shuffle [i] = i;
	for (i = 0; i < 51; i++) {
		c = random_mod (52 - i);
		t = shuffle[i+c]; 
		shuffle[i+c] = shuffle[i];
		shuffle[i] = t;
	};
	for (i = 0; i < 52; i++)
		cards[shuffle[i]] = (i % 4);
		
}

/* --------------------------------------------------------------------- */

/* Output the PostScript prolog
 */

void
print_postscript_prolog()
{
	puts("%!PS-Adobe-3.0");
	if (porter_deals) {
		puts("%%Pages: 4");
	} else {
		puts("%%Pages: 2");
	}
	puts("%%EndComments");
	fputs("\
%%BeginProlog\n\
\n\
/FIXED /Courier findfont 10 scalefont def\n\
/SYMS /Symbol findfont 10 scalefont def\n\
/TIMES /Times-Roman findfont 10 scalefont def\n\
\n\
/printsuits {\n\
TIMES setfont\n\
0 110 moveto (Bd) show\n\
0 100 moveto (Dlr) show\n\
0 90 moveto (Vul) show\n\
\n\
SYMS setfont\n\
60 110 moveto (\\252) show\n\
60 30 moveto (\\252) show\n\
0 70 moveto (\\252) show\n\
90 70 moveto (\\252) show\n\
\n\
.5 setlinewidth\n\
60 100 moveto (\\251) false charpath stroke\n\
90 60 moveto (\\251) false charpath stroke\n\
60 20 moveto (\\251) false charpath stroke\n\
0 60 moveto (\\251) false charpath stroke\n\
\n\
60 90 moveto (\\250) false charpath stroke\n\
90 50 moveto (\\250) false charpath stroke\n\
60 10 moveto (\\250) false charpath stroke\n\
0 50 moveto (\\250) false charpath stroke\n\
\n\
60 80 moveto (\\247) show\n\
90 40 moveto (\\247) show\n\
60 0 moveto (\\247) show\n\
0 40 moveto (\\247) show\n\
} def\n\
\n\
/drawlines {\n\
204 76 moveto\n\
204 716 lineto\n\
397 76 moveto\n\
397 716 lineto\n\
22 201 moveto\n\
598 201 lineto\n\
22 331 moveto\n\
598 331 lineto\n\
22 461 moveto\n\
598 461 lineto\n\
22 591 moveto\n\
598 591 lineto\n\
.5 setlinewidth\n\
stroke\n\
} def\n\
\n\
%%EndProlog\n\
",stdout);
}

/* --------------------------------------------------------------------- */

/* PostScript to display the header and footer
 */

void
print_ps_header()
{
	puts("306 726 moveto");
	puts("/Times-Roman findfont 12 scalefont setfont");
	fprintf(stdout,"(%s)\n",header);
	puts("dup stringwidth pop 2 div neg 0 rmoveto show");
}

void
print_ps_footer()
{
	puts("22 56 moveto");
	puts("/Times-Roman findfont 12 scalefont setfont");
	fprintf(stdout,"(%s) show\n",footer);
}

/* --------------------------------------------------------------------- */

/* PostScript version of the print page routine
 */

/* Origin points for each of 15 hands */

POINT origin[15] = {
	{ 22, 596 },
	{ 213, 596 },
	{ 406, 596 },
	{ 22, 466 },
	{ 213, 466 },
	{ 406, 466 },
	{ 22, 336 },
	{ 213, 336 },
	{ 406, 336 },
	{ 22, 206 },
	{ 213, 206 },
	{ 406, 206 },
	{ 22, 76 },
	{ 213, 76 },
	{ 406, 76 }
};

void
ps_print_page(int starting_hand)
{
	int i;

	puts("drawlines");

	for (i=0; i<15; ++i) {
		printf("gsave %d %d translate\n",origin[i].x,origin[i].y);
		ps_print_hand(starting_hand + i);
		puts("grestore");
	}
}

/* --------------------------------------------------------------------- */

/* Print one hand in PostScript
 */

POINT ps_suit_offsets[4][4] = {	/* [player][suit] */
	{
		{ 72, 80 },		/* [north][clubs]		*/
		{ 72, 90 },		/* [north][diamonds]	*/
		{ 72, 100 },	/* [north][hearts]		*/
		{ 72, 110 }		/* [north][spades]		*/
	}, {
		{ 102, 40 },	/* [east][clubs]		*/
		{ 102, 50 },	/* [east][diamonds]		*/
		{ 102, 60 },	/* [east][hearts]		*/
		{ 102, 70 }		/* [east][spades]		*/
	}, {
		{ 72, 0 },		/* [south][clubs]		*/
		{ 72, 10 },		/* [south][diamonds]	*/
		{ 72, 20 },		/* [south][hearts]		*/
		{ 72, 30 }		/* [south][spades]		*/
	}, {
		{ 12, 40 },		/* [west][clubs]		*/
		{ 12, 50 },		/* [west][diamonds]		*/
		{ 12, 60 },		/* [west][hearts]		*/
		{ 12, 70 }		/* [west][spades]		*/
	}
};

void
ps_print_hand(int hand_number)
{
	deal hand;
	int player, suit;
	int i;
	int x,y;
	char *dlr;
	char *vul;
	int vul_index;
	char buf[20];

	memcpy(&hand,&cards[hand_number],sizeof(deal));

/* Create strings for dealer and vulnerability */

    switch (dealer_list[hand_number % 16]) {
        case PLAYER_NORTH:
            dlr = "N";
            break;
        case PLAYER_EAST:
            dlr = "E";
            break;
        case PLAYER_SOUTH:
            dlr = "S";
            break;
        case PLAYER_WEST:
            dlr = "W";
            break;
    }

    vul_index = ns_vulnerability_list[hand_number % 16] +
      2 * ew_vulnerability_list[hand_number % 16];

    switch (vul_index) {
        case 0:
            vul = "None";
            break;
        case 1:
            vul = "N-S";
            break;
        case 2:
            vul = "E-W";
            break;
        case 3:
            vul = "Both";
            break;
    }

/* Put the board number, dealer, and vul strings */

	puts("printsuits");
	puts("FIXED setfont");
	printf("18 110 moveto (%d) show\n",hand_number+1);
	printf("18 100 moveto (%s) show\n",dlr);
	printf("18 90 moveto (%s) show\n",vul);
	
/* Print the suits */

	for ( player = 0; player < 4; ++player) {
		for ( suit = 0; suit < 4; ++suit) {
			ps_print_suit(suit,player,hand);
		}
	}
}

/* --------------------------------------------------------------------- */

/* Print one suit in PostScript
 */

void
ps_print_suit(int suit,int player,deal cards)
{
	int i;
	int x, y;
	char buf[10];

	x = ps_suit_offsets[player][suit].x;
	y = ps_suit_offsets[player][suit].y;

	printf("%d %d moveto (",x,y);
	
	for (i=12; i>=0; --i) {
		if (cards[i + 13*suit] == player) {
			fputs(rank_names[i],stdout);
		}
	}
	printf(") show\n");
}

/* --------------------------------------------------------------------- */

/* Print the Porter deals in PostScript
 */

void
ps_print_porter()
{
	int card;
	int suit;
	int marker[4];
	int xpos;
	deal hand;
	int i;

    if (porter_deals == 1) {
        marker[PLAYER_NORTH] = 'N';
        marker[PLAYER_EAST] = 'e';
        marker[PLAYER_SOUTH] = 'S';
        marker[PLAYER_WEST] = 'w';
    }
    else if (porter_deals == 2) {
        marker[PLAYER_NORTH] = '^';
        marker[PLAYER_EAST] = '>';
        marker[PLAYER_SOUTH] = 'v';
        marker[PLAYER_WEST] = '<';
    }

/* Feed the page for the Porter dealing guides */

	xpos = 726;
	puts("%%Page: 3");
	puts("save");
	printf("36 %d moveto",xpos);
	xpos -= 24;
	if (header[0]) {
		puts("/Times-Roman findfont 12 scalefont setfont");
		printf("(%s) show\n",header);
	}

	puts("/Courier findfont 12 scalefont setfont");
	for (i=0;i<15;++i) {
		memcpy(hand,cards[i],sizeof(deal));
		printf("36 %d moveto (%d) show\n",xpos,i+1);
		printf("72 %d moveto (",xpos);

		for (suit = SUIT_SPADES; suit >= 0; --suit) {
			for (card = 12; card >= 0; --card) {
				switch (hand[card+13*suit]) {
					case PLAYER_NORTH:
						putchar(marker[PLAYER_NORTH]);
						break;
					case PLAYER_EAST:
						putchar(marker[PLAYER_EAST]);
						break;
					case PLAYER_SOUTH:
						putchar(marker[PLAYER_SOUTH]);
						break;
					case PLAYER_WEST:
						putchar(marker[PLAYER_WEST]);
						break;
				}
			}
			putchar(' ');
		}
		
		puts(") show");
		xpos -= 24;
	}

	puts("restore");
	puts("showpage");

/* Print the second page of deals */

	xpos = 726;
	puts("%%Page: 4");
	puts("save");
	printf("36 %d moveto",xpos);
	xpos -= 24;
	if (header[0]) {
		puts("/Times-Roman findfont 12 scalefont setfont");
		printf("(%s) show\n",header);
	}

	puts("/Courier findfont 12 scalefont setfont");
	for (i=15;i<30;++i) {
		memcpy(hand,cards[i],sizeof(deal));
		printf("36 %d moveto (%d) show\n",xpos,i+1);
		printf("72 %d moveto (",xpos);

		for (suit = SUIT_SPADES; suit >= 0; --suit) {
			for (card = 12; card >= 0; --card) {
				switch (hand[card+13*suit]) {
					case PLAYER_NORTH:
						putchar(marker[PLAYER_NORTH]);
						break;
					case PLAYER_EAST:
						putchar(marker[PLAYER_EAST]);
						break;
					case PLAYER_SOUTH:
						putchar(marker[PLAYER_SOUTH]);
						break;
					case PLAYER_WEST:
						putchar(marker[PLAYER_WEST]);
						break;
				}
			}
			putchar(' ');
		}
		
		puts(") show");
		xpos -= 24;
	}

	puts("restore");
	puts("showpage");
}
