/* This program was written by Alexander Siegel in September of 1989   */
/* at Cornell University.  It may may copied freely for private use or */
/* public dispersion provided that this comment is not removed.  This  */
/* program, any portion of this program, or any derivative of this     */
/* program may not be sold or traded for financial gain.               */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h> 
#include <stdio.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/Xutil.h>
#include <errno.h>
#include <string.h>
#include <golddig.h>
/* Include all the bitmaps for the terrain blocks */
#include <bitmaps.h>

int savehighscore = 1; /* by default, do save highscores */
 
/* All in and out movements except up */
#define NOUPBITS   DLEAVE | LLEAVE | RLEAVE | HENTER | VENTER
/* All in and out movements */
#define MOVEBITS   NOUPBITS | ULEAVE
/* Standard bit pattern for empty space */
#define SPACEBITS  NOUPBITS | DIGUND | DFALL
/* Generic item which can be picked up */
#define ITEMBITS   SPACEBITS | PICKUP
/* Bit pattern used for dug holes */
#define HOLEBITS   SPACEBITS | STOPBAD | NODRAW
 
/* Structure describing all the characteristics of all blocks.  Refer */
/* to the defines and structure definition in golddig.h. */
struct symbs_s symbs[] = 
  {{SPACE,SPACE,"space",space_bits,SPACEBITS,XK_space,0, "black"},
   {'!','|',"escape",escape_bits,MOVEBITS | DIGUND | INACTIVE,XK_exclam,XK_1, "blue"},
   {BRICK,BRICK,"wall",wall_bits,CANDIG | KILLIN,XK_3,XK_numbersign, "firebrick"},
   {'$','$',"gold",gold_bits,ITEMBITS | TREASURE,XK_4,XK_dollar, "gold"},
   {'-','-',"rope",rope_bits,NOUPBITS | DIGUND,XK_minus,0, "wheat"},
   {HOLE1,HOLE1,"hole1",hole1_bits,HOLEBITS,0,0, "firebrick"},
   {HOLE1+1,HOLE1+1,"hole2",hole2_bits,HOLEBITS,0,0, "firebrick"},
   {HOLE1+2,HOLE1+2,"hole3",hole3_bits,HOLEBITS,0,0, "firebrick"},
   {HOLE1+3,HOLE1+3,"hole4",hole4_bits,HOLEBITS,0,0, "firebrick"},
   {HOLE1+4,HOLE1+4,"hole5",hole5_bits,HOLEBITS,0,0, "firebrick"},
   {HOLE1+5,HOLE1+5,"hole6",hole6_bits,HOLEBITS,0,0, "firebrick"},
   {HOLE1+6,HOLE1+6,"hole7",hole7_bits,HOLEBITS,0,0, "firebrick"},
   {'<','<',"left-arrow",larrow_bits,LLEAVE | HENTER | VENTER | LFALL | DIGUND,
      XK_comma,XK_less, "orange"},
   {'=','=',"tube",tube_bits,RLEAVE | LLEAVE | HENTER,XK_equal,0, "purple"},
   {'>','>',"right-arrow",rarrow_bits,RLEAVE | HENTER | VENTER | RFALL | 
      DIGUND,XK_period,XK_greater, "orange"},
   {STONE,STONE,"stone",stone_bits,KILLIN,XK_2,XK_at, "dimgrey"},
   {'^','^',"anti-space",anti_bits,ULEAVE | LLEAVE | RLEAVE | HENTER |
      VENTER | DIGUND | UFALL,XK_6,XK_asciicircum, "PaleGreen"},
   {'a','a',"armor",armor_bits,ITEMBITS | ARMOR,XK_A,XK_a, "yellow"},
   {BADGUY,BADGUY,"bad-guy",badguy_bits,SPACEBITS,XK_B,XK_b, "white"},
   {'c','c',"chute",chute_bits,DLEAVE | DFALL | VENTER,XK_C,XK_c, "purple"},
   {'d','d',"down-arrow",darrow_bits,DLEAVE | HENTER | VENTER | DIGUND |
      DFALL,XK_D,XK_d, "orange"},
   {'e','e',"reverse-monster",reverse_bits,ITEMBITS | REVERSE, XK_E,XK_e, "ForestGreen"},
   {'f','f',"up-arrow",uarrow_bits,ULEAVE | HENTER | VENTER | UFALL |
      DIGUND,XK_F,XK_f, "orange"},
   {'g',BRICK,"ghost-brick",ghost_bits,(SPACEBITS) & ~HENTER,XK_G,XK_g, "firebrick"},
   {'i',SPACE,"invisible-block",invis_bits,KILLIN,XK_I,XK_i,"firebrick"},
   {'j','j',"jump-pad",jump_bits,NOUPBITS | UPTWO,XK_J,XK_j, "IndianRed"},
   {'k','k',"kill-zone",kill_bits,HENTER | VENTER | DIGUND |
      KILLIN,XK_K,XK_k, "green"},
   {'l','l',"power-shovel",pshovel_bits,ITEMBITS | PSHOVEL,XK_L,XK_l, "SandyBrown"},
   {'m','m',"super-jump-pad",sjump_bits,NOUPBITS | UPALL,XK_M,XK_m, "red"},
   {'n','n',"nuclear-shovel",nshovel_bits,ITEMBITS | NSHOVEL,XK_N,XK_n, "tan"},
   {'o','o',"anchor",anchor_bits,ITEMBITS | ANCHOR,XK_O,XK_o, "DodgerBlue"},
   {PLAYER,PLAYER,"player",player_bits,SPACEBITS,XK_P,XK_p, "white"},
   {'q','q',"speed-boot",speed_bits,ITEMBITS | SPEED,XK_Q,XK_q, "sienna"},
   {'r','r',"parachute",parac_bits,ITEMBITS | STOPFALL,XK_R,XK_r, "LightCyan"},
   {'s','s',"step-ladder",steplad_bits,MOVEBITS | PICKUP,XK_S,XK_s, "turquoise"},
   {'t','t',"teleporter",teleport_bits,SPACEBITS | TELEPORT,XK_T,XK_t, "VioletRed"},
   {'u','u',"leave-level",uplevel_bits,SPACEBITS | UPLEVEL |
      INACTIVE,XK_U,XK_u, "brown"},
   {'v','v',"vertical-rope",vrope_bits,ULEAVE | DLEAVE |
      HENTER | VENTER,XK_V,XK_v, "wheat"},
   {'w','w',"window",window_bits,SPACEBITS | UPLEVEL,XK_W,XK_w, "cyan"},
   {'x','x',"extra-brick",rshovel_bits,ITEMBITS | RSHOVEL,XK_X,XK_x, "tomato"},
   {'y','y',"heavy-fog",fog_bits,SPACEBITS,XK_Y,XK_y, "DarkSlateGrey"},
   {'z','z',"time-stop",hourgl_bits,ITEMBITS | TIMESTOP,XK_Z,XK_z, "magenta"},
   {'|','|',"ladder",ladder_bits,MOVEBITS,XK_backslash,XK_bar, "CornflowerBlue"},
   {'~','-',"portable-rope",portable_bits,NOUPBITS | DIGUND |
      PICKUP,XK_asciitilde,XK_quoteleft, "wheat"},
 
   /* List terminator */
   {'\0','\0',(char *) 0,(char *) 0,0,0,0}};
 
Font scorefont;     /* Font used to display score */
XFontStruct *scorestruct;
int scoresize;      /* Size of score font */
GC scoregc;         /* GC used to draw score */
GC blackgc;         /* Simple black foreground GC */
GC livesgc;         /* For drawing player in score line */

/* These are the graphics cursors used for drawing the player at */
/* various times. */
GC standgc,angelgc,angelugc,angellgc,flygc,hang1gc,hang2gc,up1gc,up2gc;
GC left1gc,left2gc,right1gc,right2gc;
/* Graphics cursors for drawing the possible states of the badguys */
GC badguy1gc,badguy2gc,badguy3gc;

/* Manufaction a 16x16 graphics cursor used in a XFill... operation. */
/* The FillTiled style is used. */

static GC fill_gc(func, bits, item, color)
int func;       /* Drawing function such as GXcopy or GXor. */
char *bits;    /* Bits describing fill pattern.  Produced in an X11 */
                /* bitmap file usually. */
char *item, *color; /* item and color */

{
  XGCValues gcv;
  XColor got, hardw;
  Pixmap pmap;
 
  /* Build X11 bitmap from data in bits */

  if (DisplayCells(disp, scrn) <= 2) {

    /*
     * monochrome
     */

    pmap = XCreatePixmapFromBitmapData
      (disp,wind,bits,16,16,BlackPixel(disp,scrn),WhitePixel(disp,scrn),
       DisplayPlanes(disp,scrn));
    gcv.foreground = BlackPixel(disp,scrn);
    gcv.background = WhitePixel(disp,scrn);
    
  } else {
    
    /*
     * color
     */
    
    if (!XAllocNamedColor(disp, XDefaultColormap(disp, scrn), color,
			  &got, &hardw)) {
      printf("XAllocNamedColor failed for %s color %s!\n", item, color);
      printf("using WhitePixel\n");
      hardw.pixel = WhitePixel(disp, scrn);
    }
    
    pmap = XCreatePixmapFromBitmapData
      (disp,wind,bits,16,16, hardw.pixel, background,
       DisplayPlanes(disp,scrn));
    
    gcv.foreground = hardw.pixel;
    gcv.background = background;
    
  }

  /* Assign the graphics cursor parameters */
  gcv.function = func;
  gcv.tile = pmap;
  gcv.fill_style = FillTiled;
  /* Return the created graphics cursor */
  return(XCreateGC(disp,wind,GCFunction | GCForeground | GCBackground | 
                   GCTile | GCFillStyle,&gcv));
}
 
/* Manufaction a 16x16 graphics cursor used in a XFill... operation. */
/* The FillStippled style is used. */
static GC stip_gc(func, bits, pix)
int func, pix;
char bits[];
{
  XGCValues gcv;
  Pixmap pmap;
 
  /* Build X11 bitmap from data in bits */
  pmap = XCreateBitmapFromData(disp,wind,bits,16,16);
  /* Assign the graphics cursor parameters */
  gcv.function = func;
  gcv.foreground = pix;
  gcv.fill_style = FillStippled;
  gcv.stipple = pmap;
  /* Return the created graphics cursor */
  return(XCreateGC(disp,wind,GCFunction | GCFillStyle | 
		   GCForeground | GCStipple,&gcv));
}

/* Start X11 and do some basic initialization */
void xstart(evmask, argc, argv)
long evmask;    /* Event mask which will be used in XSelectInput */
int argc;
char **argv;
{
  register int i;
  XGCValues xgcv;
  XWMHints wmhints;
  XSetWindowAttributes xswa;
  unsigned int xswamask;
  XSizeHints hints;
  int x,y;
  unsigned int w,h;
  Window rootW;
  char *resource;
  XColor hardw, got;
  unsigned long tmp, score_fg, score_bg, mecol, badcol;
 
  /* Open up the display */
  disp = XOpenDisplay(NULL);
  /* Check to see if the open display succeeded */
  if(disp == NULL) {
    fprintf(stderr,"Display open failed.  Check DISPLAY environment variable.\n");
    exit(-1);
  }
 
  /* Set the default screen number */
  scrn = DefaultScreen(disp);
  /* Get the root window */
  rootW = RootWindow(disp,scrn);
  /* Get the window position, width, and height */
  x = y = w = h = 1;
  i = XParseGeometry (geom, &x, &y, &w, &h);
  
  /* Set values of fields in hints structure */
  if(i & WidthValue) hints.width = w;   else hints.width = 50 << 4;
  if(i & HeightValue) hints.height = h; else hints.height =  (30 << 4)
    + scoresize;
  if(i & XValue) {
    if(i & XNegative)
      hints.x = XDisplayWidth(disp,scrn) - hints.width - abs(x);
    else
      hints.x = x;
  }
  else
    hints.x = 20;
  if(i & YValue) {
    if(i & YNegative)
      hints.y = XDisplayHeight(disp,scrn) - hints.height - abs(y);
    else
      hints.y = y;
  }
  else
    hints.y = 20;
  hints.max_width = DisplayWidth(disp,scrn);
  hints.max_height = DisplayHeight(disp,scrn);
  hints.flags = PMaxSize | PPosition | PSize;
  
  /* Build window attributes */
  xswa.background_pixel = BlackPixel(disp,scrn);
  xswa.border_pixel = WhitePixel(disp,scrn);
  xswamask = CWBackPixel | CWBorderPixel;
  /* Create Window */
  wind = XCreateWindow(disp, rootW, hints.x, hints.y, hints.width,
                       hints.height, 2, 0, CopyFromParent,
                       CopyFromParent, xswamask, &xswa);
  /* Check to see if the open window succeeded */
  if(wind == 0) {
    fprintf(stderr,"Window open failed.\n");
    XCloseDisplay(disp);
    exit(-1);
  }
  /* Set other properties on window */
  XSetStandardProperties(disp,wind,"Golddig","Golddig",None,argv,argc,&hints);
  
  if (DisplayCells(disp, scrn) <= 2)
    background = WhitePixel(disp, scrn);
  else {

    resource = XGetDefault(disp, "Golddig", "background");
    if (resource == NULL)
      resource = "black";
    if (!XAllocNamedColor(disp, XDefaultColormap(disp, scrn), resource,
			  &got, &hardw)) {
      printf("XAllocNamedColor failed for background color %s!\n", resource);
      printf("using BlackPixel\n");
      hardw.pixel = BlackPixel(disp, scrn);
    }
    background = hardw.pixel;
  }
  
  /* Clear fast block type lookup table */
  for(i=0;i<256;++i) {
    fast_lookup[i].gc = NULL;
    /* Everything starts out looking like a space */
    fast_lookup[i].code = SPACEBITS;
  }
  /* Generate block type lookup table from symbs array defined above. */
  /* After this the symbs array will be used very rarely. */
  for(i=0;symbs[i].symb != '\0';++i) {
    fast_lookup[symbs[i].symb].gc  =
      fill_gc(GXcopy,symbs[i].bits, symbs[i].name, 
	((resource = XGetDefault(disp, "Golddig", symbs[i].name)) == NULL) ? 
	      symbs[i].dcolor: resource);
    fast_lookup[symbs[i].symb].code = symbs[i].code;
  }
  /* Load in the font used to display the score */
  scorefont = XLoadFont(disp,SCOREFONT);
  scorestruct = XQueryFont(disp, scorefont);
  scoresize = scorestruct->max_bounds.ascent +
    scorestruct->max_bounds.descent;

  /* Create GC which will be used from drawing score */

  xgcv.function = GXcopy;
  xgcv.font = scorefont;

  if (DisplayCells(disp, scrn) <= 2) {

    score_fg = xgcv.foreground = WhitePixel(disp,scrn);
    score_bg = xgcv.background = BlackPixel(disp,scrn);

  } else {

    resource = XGetDefault(disp, "Golddig", "score-fg");
    if (resource == NULL)
      resource = "black";
    if (!XAllocNamedColor(disp, XDefaultColormap(disp, scrn), resource,
			  &got, &hardw)) {
      printf("XAllocNamedColor failed for score-fg color %s!\n", resource);
      printf("using BlackPixel\n");
      hardw.pixel = BlackPixel(disp, scrn);
    }
    score_fg = xgcv.foreground = hardw.pixel;
    resource = XGetDefault(disp, "Golddig", "score-bg");
    if (resource == NULL)
      resource = "grey";
    if (!XAllocNamedColor(disp, XDefaultColormap(disp, scrn), resource,
			  &got, &hardw)) {
      printf("XAllocNamedColor failed for score-bg color %s!\n", resource);
      printf("using WhitePixel\n");
      hardw.pixel = WhitePixel(disp, scrn);
    }
    score_bg = xgcv.background = hardw.pixel;

  }

  scoregc = XCreateGC(disp,wind,
                      GCFunction | GCFont | GCForeground | GCBackground,
                      &xgcv);

  /* Create GC which will be used for clearing score line */

  xgcv.function = GXcopy;
  tmp = xgcv.foreground;
  xgcv.foreground = score_bg;
  xgcv.background = score_fg;
  blackgc = XCreateGC(disp,wind,
                      GCFunction | GCForeground | GCBackground,
                      &xgcv);
  /* Create GC used for drawing men in score line */
  livesgc = stip_gc(GXcopy,player_bits, score_fg);

  if (DisplayCells(disp, scrn) <= 2) {
    mecol = badcol = BlackPixel(disp, scrn);
  } else {
    resource = XGetDefault(disp, "Golddig", "mycolor");
    if (resource == NULL)
      resource = "white";
    if (!XAllocNamedColor(disp, XDefaultColormap(disp, scrn), resource,
			  &got, &hardw)) {
      printf("XAllocNamedColor failed for mycolor (color %s).\n", resource);
      printf("using WhitePixel\n");
      hardw.pixel = WhitePixel(disp, scrn);
    }
    mecol = hardw.pixel;

    resource = XGetDefault(disp, "Golddig", "badcol");
    if (resource == NULL)
      resource = "white";
    if (!XAllocNamedColor(disp, XDefaultColormap(disp, scrn), resource,
			  &got, &hardw)) {
      printf("XAllocNamedColor failed for badcol (color %s).\n", resource);
      printf("using WhitePixel\n");
      hardw.pixel = WhitePixel(disp, scrn);
    }
    badcol = hardw.pixel;
  }

  /* compute all the graphics cursors for drawing the player in his */
  /* various states. */
  standgc = stip_gc(GXcopy,player_bits, mecol);
  angelgc = stip_gc(GXcopy,angel_bits, mecol);
  angelugc = stip_gc(GXcopy,angelu_bits,mecol);
  angellgc = stip_gc(GXcopy,angell_bits,mecol);
  flygc = stip_gc(GXcopy,fly_bits,mecol);
  hang1gc = stip_gc(GXcopy,hang1_bits,mecol);
  hang2gc = stip_gc(GXcopy,hang2_bits,mecol);
  up1gc = stip_gc(GXcopy,up1_bits,mecol);
  up2gc = stip_gc(GXcopy,up2_bits,mecol);
  left1gc = stip_gc(GXcopy,left1_bits,mecol);
  left2gc = stip_gc(GXcopy,left2_bits,mecol);
  right1gc = stip_gc(GXcopy,right1_bits,mecol);
  right2gc = stip_gc(GXcopy,right2_bits,mecol);

  /* Generate graphics cursors for drawing bad guy */
  badguy1gc = stip_gc(GXcopy,badguy_bits,badcol);
  badguy2gc = stip_gc(GXcopy,badguy2_bits,badcol);
  badguy3gc = stip_gc(GXcopy,badguy3_bits,badcol);

  /* Tell the WM that we want input... */
  wmhints.input = True;
  wmhints.flags = InputHint;
  XSetWMHints(disp, wind, &wmhints);
 
  /* Select the interesting window events */
  XSelectInput(disp,wind,evmask);
 
  /* Name and raise the window */
  XMapRaised(disp,wind);
 
  /* Flush and synchronize the server */
  XFlush(disp);
  XSync(disp,False);
}
 
/* Gracefully shut X windows down.  It is not strictly necessary to */
/* call this function. */
void xend()
{
  XUnloadFont(disp,scorefont);
  XUnmapWindow(disp,wind);
  XDestroyWindow(disp,wind);
  XCloseDisplay(disp);
}
 
/* Draw a block from the level array in the output window. */
void draw_block(x,y)
int x,y;    /* Position of block in array */
{
  register unsigned char curchar;
  GC drawgc;
 
  /* Get the block character out of the level array */
  curchar = level[x*ysize + y];
  /* If there is gold left and this block is inactive, replace it with */
  /* a space. */
  if(goldleft > 0 && (fast_lookup[curchar].code & INACTIVE))
    curchar = SPACE;
  /* Get the graphics cursor */
  drawgc = fast_lookup[curchar].gc;
  /* Replace questionable characters with spaces */
  if(drawgc == NULL)
    drawgc = fast_lookup[SPACE].gc;
  /* Fill the block */
  XFillRectangle(disp,wind,drawgc,x << 4,y << 4,16,16);
}

/* Change a block character in the level array.  The block is redrawn. */
void setchar(x,y,ch)
int x,y;    /* Position of block character to change. */
char ch;    /* Character to change it to */
{
  if(level[x*ysize + y] != ch) {
    level[x*ysize + y] = ch;
    draw_block(x,y);
  }
}
 
/* Draw the score and level number */
void draw_score()
{
  char buf[50];
  int textwidth,i;
 
  /* Build the output string */
  sprintf(buf,"score: %d  level: %d  speed: %d%s",
    score, levelnum, speed,angelleft ? " You are dead! " : "");
  /* Clear the current score line */
  XFillRectangle(disp,wind,blackgc,0,ysize << 4,xsize << 4,scoresize);
  /* Actually draw the text */
  i = strlen(buf);
  XDrawString(disp,wind,scoregc,0,(ysize << 4) + scoresize - 1,buf, i);
  /* find size of string we just drew */
  textwidth = XTextWidth(scorestruct, buf, i);
  i = (textwidth + 16) & ~15;   /* starting point */
  textwidth = lives * 16;
  XFillRectangle(disp,wind,livesgc, i, (ysize << 4), textwidth, 16);
}

/* Redraw the entire level */
void draw_level()
{
  int x,y;
 
  /* Change the window size */
  XResizeWindow(disp,wind,xsize << 4,(ysize << 4) + scoresize);
  /* Draw the score and level number */
  draw_score();
  /* Iterate through each block position and draw it */
  for(x=0;x < xsize;++x)
    for(y=0;y < ysize;++y)
      draw_block(x,y);
}
 
/* Load a level out of a file.  The global variables worldname and */
/* levelnum are used to produce the file name which is stored in */
/* filename. */
void load_level()
{
  FILE *levelfile;
  register int i,j;
  int x,y;
  char buf[MAXPATHLEN];
 
  /* Manufaction the file name by starting with the world name and */
  /* appending the level number to it. */
  /* Try looking in ~/.golddig first, then in the global levels. */
  snprintf(buf, sizeof(buf), "%s/.golddig/%s%03d", getenv("HOME"),
	   worldname, levelnum);
  /* Open level file for reading */
  levelfile = fopen(buf,"r");

  if (levelfile == NULL) {
    /* If local file doesn't exist, try global one. */
    snprintf(buf, sizeof(buf), "%s/%s%03d", LIB, worldname, levelnum);
    /* Open level file for reading */
    levelfile = fopen(buf,"r");
  } else {
    printf("Using %s\n", buf);
    savehighscore = 0;
  }

  /* If level file does not exist, use the default level file. */
  if(levelfile == NULL) {
    /* Build the default level name */
    snprintf(buf, sizeof(buf), "%s/default", LIB);
    /* Open default level file for reading */
    levelfile = fopen(buf,"r");
    if(levelfile == NULL) {
      perror(buf);
      exit(1);
    }
  }
 
  /* Load the first line of the level file */
  if(fgets(buf,300,levelfile) == NULL) {
    x = 50;
    y = 30;
  }
  else {
    /* Extract the level size */
    sscanf(buf,"%d %d",&x,&y);
  }
  /* Change level size only if it is necessary */
  if(xsize == -1)
    xsize = x;
  if(ysize == -1)
    ysize = y;
  /* Carefully check the sanity of the size parameters */
  if(xsize < 5)
    xsize = 5;
  if(xsize > 250)
    xsize = 250;
  if(ysize < 5)
    ysize = 5;
  if(ysize > 250)
    ysize = 250;
  if(xsize * ysize > MAXLEVEL) {
    if(xsize > ysize)
      xsize = MAXLEVEL / ysize;
    else
      ysize = MAXLEVEL / xsize;
  }
  /* Iterate through each horizontal line */
  for(i=0;i<ysize;++i) {
    /* Load the next line from the file */
    if(fgets(buf,300,levelfile) != NULL) {
      /* Go through each horizontal position and copy the data into */
      /* the level array. */
      for(j=0;j<xsize;++j) {
        /* Break out if line ends prematurely */
        if(buf[j] == '\n' || buf[j] == '\0')
          break;
        level[j*ysize + i] = buf[j];
      }
    }
    else
      j = 0;
    /* Fill in rest of premature lines with spaces */
    for(;j<xsize;++j)
      level[j*ysize + i] = SPACE;
  }
  /* Close the level file */
  fclose(levelfile);
}
 
/* Save the current level back into a file.  The global variables */
/* worldname and levelnum are used to determine the file name. */
void save_level()
{
  FILE *levelfile;
  char buf[MAXPATHLEN+1];
  char dir[MAXPATHLEN+1];
  register int i,j;
  struct stat sb;

  /* Check existence of $HOME/.golddig; create if not existant. */
  snprintf(dir, sizeof(dir), "%s/.golddig", getenv("HOME"));
  if ((stat(dir, &sb) == -1)
      && (mkdir(dir, S_IRWXU|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH) == -1)) {
    perror("error accessing or creating ~/.golddig");
    exit(1);
  }

  /* Write levels in ~/.golddig. */
  snprintf(buf, sizeof(buf), "%s/%s%03d", dir, worldname, levelnum);
  /* Open level file for writing */
  levelfile = fopen(buf,"w");

  if (levelfile == NULL) {
    /* If local file doesn't exist, try global one. */
    snprintf(buf, sizeof(buf), "%s/%s%03d", LIB, worldname, levelnum);
    /* Open level file for reading */
    levelfile = fopen(buf,"r");
  }

  /* Open the data file */
  levelfile = fopen(buf,"w");
  if(levelfile == NULL) {
    perror(buf);
    exit(1);
  }
  /* Write out the size of the level.  Normal text is used so that */
  /* levels can be easily copied across architectures. */
  fprintf(levelfile,"%d %d\n",xsize,ysize);
  /* Terminate the lines for writing out the horizontal level lines */
  buf[xsize] = '\n';
  buf[xsize+1] = '\0';
  /* Iterate through each vertical position */
  for(i=0;i<ysize;++i) {
    /* Copy each horizontal line into the output buffer */
    for(j=0;j<xsize;++j)
      buf[j] = level[j*ysize + i];
    /* Write the line out to the file */
    fputs(buf,levelfile);
  }
  /* Close the data file */
  fclose(levelfile);
}
