/*
 * ginifeed.c - main source if GINIfeed
 * Copyright 2002, 2003 Krisztian Pifko <monsta@users.sourceforge.net>
 *
 * Author:
 *	Krisztian Pifko <monsta@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/signal.h>
#include <sys/stat.h>
#include <glib.h>
#ifdef ENV_LINUX
#include <time.h>
#endif

#include "gini.h"
#include "net.h"
#include "asf.h"
#include "avi.h"
#include "mov.h"
#include "mp3.h"
#include "ogg.h"
#include "real.h"
#include "util.h"
#include "getopt.h"

#define GINIFEED_NAME "GINIfeed"
#define RBUFSIZE 1024

int mainsock, titlesock = -1, relaysock = -1;
fd_set sockset, socksetw;
char rbuff[RBUFSIZE + 1];
unsigned long socketswritten;
int socketsinit;
FILE *f;
char *buf;
ssize_t readed;
//char secshow[4] = "|/-\\";
char *playlist = NULL;
int mainport = 8000;
char *host = NULL;
unsigned int brate = 0;
u_long bufsize = 0;
char *fname = NULL;
int randomplay = 0;
int skip_file = 0;
long last_ctrl_c = 0;
char *basedir = NULL;
char *suffix = NULL;
u_int seed = 0;
char *x_ac_name = NULL;
char *x_ac_genre = NULL;
char *x_ac_url = NULL;
int x_ac_public = 0;
char *x_ac_description = NULL;
int streamtype = 0;
int actual_seconds;
int actual_seconds_full;
int actual_bitrate;
int actual_nominal;
char *actual_artist = NULL;
char *actual_title = NULL;
ssize_t actual_size;
char *password = NULL;
char *mount = NULL;
int daemon_mode = 0;
int reread_playlist;
gchar **files;
int use_cue = 0;
u_char *current_dir;
u_int *track_order = NULL;
int title_streaming = 0;
int use_tag = 0;
int without_dir = 0;
int stream_relay = 0;
char *relay_protocol = NULL;
char *relay_host = NULL;
char *relay_port = NULL;
char *relay_location = NULL;

mp3info mp3_info;
aviinfo avi_info;

static char const shortopts[] = "m:w:n:g:u:D:H:P:p:d:s:b:crvhzS:tGW";
static struct option const longopts[] = {
  {"mount", required_argument, NULL, 'm'},
  {"password", required_argument, NULL, 'w'},
  {"name", required_argument, NULL, 'n'},
  {"genre", required_argument, NULL, 'g'},
  {"url", required_argument, NULL, 'u'},
  {"description", required_argument, NULL, 'D'},
  {"host", required_argument, NULL, 'H'},
  {"port", required_argument, NULL, 'P'},
  {"playlist", required_argument, NULL, 'p'},
  {"basedir", required_argument, NULL, 'd'},
  {"suffix", required_argument, NULL, 's'},
  {"bitrate", required_argument, NULL, 'b'},
  {"use-cue", no_argument, NULL, 'c'},
  {"random", no_argument, NULL, 'r'},
  {"version", no_argument, NULL, 'v'},
  {"help", no_argument, NULL, 'h'},
  {"background", no_argument, NULL, 'z'},
  {"seed", required_argument, NULL, 'S'},
  {"title-streaming", no_argument, NULL, 't'},
  {"use-tag", no_argument, NULL, 'G'},
  {"without-dir", no_argument, NULL, 'W'},
  {NULL, no_argument, NULL, 0}
};

static char const *const option_help[] = {
  "  -H    --host <host>           host to connect to. DEFAULT: 127.0.0.1",
  "  -P    --port <port>           port to connect to. DEFAULT: 8000",
  "  -p    --playlist <file>       specify playlist.",
  "  -r    --random                play songs in random order.",
  "  -d    --basedir <dir>         put this before filenames in playlist.",
  "  -s    --suffix <str>          put this after filenames in playlist.",
  "  -b    --bitrate <br>          specify bitrate. DEFAULT: autodetect",
  "  -n    --name <name>           name of stream.",
  "  -g    --genre <genre>         genre of stream.",
  "  -u    --url <url>             url of stream.",
  "  -D    --description <desc>    description of stream.",
  "  -m    --mount <mount>         description of stream.",
  "  -w    --password <pass>       password of server.",
  "  -S    --seed <num>            seed of random.",
  "  -c    --use-cue               generate cue file.",
  "  -t    --title-streaming       enable title streaming.",
  "  -G    --use-tag               use file tags for info.",
  "  -W    --without-dir           use filename without dirname for info.",
  "  -z    --background            run in the background.",
  "  -v    --version               output version info.",
  "  -h    --help                  output this help.",
  "",
  "Report bugs to <monsta@users.sourceforge.net>.",
  0
};

static void version ()
{
  printf ("%s %s %s\n%s\n", GINIFEED_NAME, VERSION, GINI_URL, GINI_COPYRIGHT);
}

static void usage ()
{
  printf ("\nusage: ginifeed [OPTIONS]\n");
  printf ("\n [OPTIONS] Try --help for more information\n");
  printf ("\n\n");
  return;
}

void get_some_switches (int argc, char *argv[], int *todo)
{
  register int optc;
	int i;
  char const *const *p;
  if (optind == argc)
    return;
  while ((optc = getopt_long (argc, argv, shortopts, longopts, (int *) 0))
	 != -1) {
    switch (optc) {
    case 'G':
      use_tag = 1;
      break;
    case 'W':
      without_dir = 1;
      break;
    case 't':
      title_streaming = 1;
      break;
    case 'c':
      use_cue = 1;
      break;
    case 'm':
      mount = g_strdup (optarg);
      break;
    case 'w':
      password = g_strdup (optarg);
			for (i=0;i<strlen(optarg);i++){
		  	optarg[i]='*';
			}
      break;
    case 'n':
      x_ac_name = g_strdup (optarg);
      break;
    case 'g':
      x_ac_genre = g_strdup (optarg);
      break;
    case 'u':
      x_ac_url = g_strdup (optarg);
      break;
    case 'D':
      x_ac_description = g_strdup (optarg);
      break;
    case 'H':
      host = g_strdup (optarg);
      break;
    case 'z':
      daemon_mode = 1;
      break;
    case 'r':
      randomplay = 1;
      break;
    case 'P':
      mainport = atoi (optarg);
      break;
    case 'S':
      seed = atoi (optarg);
      break;
    case 'b':
      brate = atoi (optarg);
      break;
    case 'p':
      playlist = g_strdup (optarg);
      break;
    case 'd':
      basedir = g_strdup (optarg);
      break;
    case 's':
      suffix = g_strdup (optarg);
      break;
    case 'v':
//      version ();
      exit (0);
      break;
    case 'h':
      usage (argv[0]);
      for (p = option_help; *p; p++)
	printf ("%s\n", *p);
      exit (0);
      break;
    default:
      usage (argv[0]);
    }
  }
}

void resetsockswritten (void)
{
  socketswritten = 0;
}

void closeallsockets (void)
{
  closesocket (mainsock);
  if (stream_relay){
	  closesocket (relaysock);
  }
}

void closeall (void)
{
  closeallsockets ();
  if (!stream_relay) {
    fclose (f);
  }
  g_free (buf);
  g_free (fname);
  g_strfreev (files);
  g_free (actual_artist);
  g_free (actual_title);
  g_free (current_dir);
  g_free (track_order);
  g_free (relay_protocol);
  g_free (relay_host);
  g_free (relay_port);
  g_free (relay_location);
}

void FD_SETALL (fd_set * set)
{
  FD_ZERO (set);
  FD_SET (mainsock, set);
}

void FD_RSETALL (fd_set * set)
{
  FD_ZERO (set);
  FD_SET (relaysock, set);
}

void FD_TSETALL (fd_set * set)
{
  FD_ZERO (set);
  FD_SET (titlesock, set);
}

void signalh (const int sig);

static void setup_signal_traps ()
{
  signal (SIGSEGV, signalh);
  signal (SIGPIPE, signalh);
  signal (SIGINT, signalh);
  signal (SIGQUIT, signalh);
  signal (SIGFPE, signalh);
  signal (SIGBUS, signalh);
  signal (SIGTERM, signalh);
  signal (SIGHUP, signalh);
// signal (SIGUSR1, signalh);
// signal (SIGUSR2, signalh);
// signal (SIGCHLD, signalh);
  signal (SIGIO, signalh);
#ifndef ENV_CYGWIN
  signal (SIGALRM, SIG_IGN);
#endif
}

void signalh (const int sig)
{
  struct timeval now;
  setup_signal_traps ();
  switch (sig) {
  case SIGHUP:
    reread_playlist = 1;
    break;
  case SIGSEGV:
  case SIGBUS:
  case SIGFPE:
  case SIGIO:
  case SIGTERM:
    printf ("\nEEEEK!\n");
    closeallsockets ();
    exit (0);
    break;
  case SIGINT:
    gettimeofday (&now, NULL);
    if (last_ctrl_c + 1 < now.tv_sec) {
      skip_file = 1;
      last_ctrl_c = now.tv_sec;
      break;
    }
  case SIGQUIT:
    printf ("\nMANUAL KILL!\n");
    closeallsockets ();
    exit (0);
    break;
  }
  setup_signal_traps ();
}

int get_stream_type (unsigned char *b)
{
  int ret = STREAM_UNKNOWN;

  if (!memcmp (b, "OggS", 4)) {
    ret = STREAM_OGG;
  } else if (!memcmp (b, "ID3", 3)) {
    ret = STREAM_MP3;
  } else if ((0xFF == b[0]) && ((b[1] & 0xE0) == 0xE0)) {
    ret = STREAM_MP3;
  } else if ((!memcmp (b, "RIFF", 4))&&(!memcmp (b+8, "AVI ",4))) {
    ret = STREAM_AVI;
  } else if (!memcmp (b+4, "moov", 4)) {
    ret = STREAM_MOV;
  } else if (!memcmp (b, "\x30\x26\xB2\x75\x8E\x66\xCF\x11\xA6\xD9\x00\xAA\x00\x62\xCE\x6C", 16)) {
    ret = STREAM_ASF;
  } else if (!memcmp (b, ".RMF", 4)) {
    ret = STREAM_REAL;
  }
  return ret;
}

int probe_media_type (FILE * file)
{
#define MAGIC_LEN 16
  unsigned char magic[MAGIC_LEN];
  int treaded, ret = STREAM_UNKNOWN;

  treaded = fread (magic, 1, MAGIC_LEN, file);
  if (MAGIC_LEN == treaded) {
    ret = get_stream_type (magic);
    rewind (file);
    if (STREAM_MP3 == ret) {
      if (get_mp3_info (&mp3_info, SCAN_FULL, 1, &actual_seconds, &actual_bitrate, &actual_nominal,
			&actual_artist, &actual_title)) {
	ret = STREAM_UNKNOWN;
      }
    } else if (STREAM_OGG == ret) {
      if (get_ogg_info
	  (file, &actual_seconds, &actual_bitrate, &actual_nominal, &actual_artist, &actual_title)) {
	ret = STREAM_UNKNOWN;
      }
    } else if (STREAM_AVI == ret) {
      if (get_avi_info (file, &avi_info, &actual_seconds, &actual_bitrate, &actual_nominal, &actual_artist,
			&actual_title)) {
	ret = STREAM_UNKNOWN;
      }
    } else if (STREAM_MOV == ret) {
      if (get_mov_info (file, &actual_seconds, &actual_bitrate, &actual_nominal, &actual_artist,
			&actual_title)) {
	ret = STREAM_UNKNOWN;
      }
    } else if (STREAM_ASF == ret) {
      if (get_asf_info (file, &actual_seconds, &actual_bitrate, &actual_nominal, &actual_artist,
			&actual_title)) {
	ret = STREAM_UNKNOWN;
      }
    } else if (STREAM_REAL == ret) {
      if (get_real_info (file, &actual_seconds, &actual_bitrate, &actual_nominal, &actual_artist,
			&actual_title)) {
	ret = STREAM_UNKNOWN;
      }
    }
  }

  rewind (file);
  return ret;
}

int http_probe (int sock)
{
  gchar *request_uri;
  int reqlen;
  int twritten, written;
  int treaded;
  char *tbitrate;
  int ret = 0;
  char *tb;
//int ifile;

  request_uri =
    g_strdup_printf ("GET %s HTTP/1.0\r\nUser-Agent: %s/%s\r\nHost: %s\r\n\r\n", relay_location,
		     GINIFEED_NAME, VERSION, relay_host);
  reqlen = strlen (request_uri);
  written = 0;
  twritten = send (sock, request_uri, reqlen, MSG_DONTWAIT);
  if (twritten > 0) {
    reqlen -= twritten;
    written += twritten;
    while ((reqlen > 0) && (twritten > 0)) {
      usleep (5);
      twritten = send (sock, request_uri + written, reqlen, 0);
      reqlen -= twritten;
      written += twritten;
    }
  }
  if (twritten > 0) {
    treaded = recv (sock, rbuff, RBUFSIZE, 0);
    if (treaded > 0) {
      if (0 == memcmp (rbuff, "HTTP/1.0 200 OK", 15)) {
	tbitrate = getheaders ("x-audiocast-bitrate:", rbuff);
	if (NULL != tbitrate) {
	  brate = atoi (tbitrate);
	  g_free (tbitrate);
	  readed = treaded;
	  tb = get_data_pos (rbuff);
	  if (NULL == tb) {
	    closesocket (sock);
	  } else {
			if (NULL == x_ac_name) {
				x_ac_name = getheaders ("x-audiocast-name:", rbuff);
			}
			if (NULL == x_ac_genre) {
				x_ac_genre = getheaders ("x-audiocast-genre:", rbuff);
			}
			if (NULL == x_ac_url) {
				x_ac_url = getheaders ("x-audiocast-url:", rbuff);
			}
			if (NULL == x_ac_description) {
				x_ac_description = getheaders ("x-audiocast-description:", rbuff);
			}
	    ret = get_stream_type (tb)+1;
	  }
	} else {
	  closesocket (sock);
	}
      } else {
	closesocket (sock);
      }
    } else {
      closesocket (sock);
    }
  } else {
    closesocket (sock);
  }
  return ret;
}

FILE *media_open (char *fn, const char *mode)
{
  FILE *ret = NULL;
  int stype, relport;
  struct stat st;

  if (!stream_relay) {
    if (stat (fn, &st) == 0) {
      actual_size = st.st_size;
    }
    ret = fopen (fn, mode);
    if (ret != NULL) {
      mp3_info.filename = g_strdup (fn);
      mp3_info.file = ret;
      avi_info.name = g_strdup (fn);
      g_free (actual_artist);
      g_free (actual_title);
      stype = probe_media_type (ret);
      actual_seconds_full = actual_seconds;
	  if (0==actual_bitrate){
		  actual_bitrate=((actual_size*8/actual_seconds)/1024);
//			actual_bitrate=actual_size/actual_seconds;
	  }
	  if (0==actual_nominal){
		  actual_nominal=((actual_size*8/actual_seconds)/1024);
//			actual_nominal=actual_size/actual_seconds;
	  }
      g_free (mp3_info.filename);
      g_free (avi_info.name);
      if (STREAM_UNKNOWN == stype) {
	ret = NULL;
      } else {
	if (0 == streamtype) {
	  streamtype = stype;
	  printf ("first file type is %s\n", stream_types[stype]);
	} else {
	  if (streamtype != stype) {
	    ret = NULL;
	  }
	}
      }
    }
  } else {
    if (urlparse (fn, &relay_protocol, &relay_host, &relay_port, &relay_location)) {
      relport = atoi (relay_port);
      relaysock = connectto2 (relay_host, relport);
      if (-1 != relaysock) {
	streamtype = http_probe (relaysock);
	if (streamtype) {
		streamtype--;
		printf ("relayed stream type is %s", stream_types[streamtype]);
		if (!streamtype){
			printf(", treating as %s",stream_types[STREAM_MP3]);
			streamtype=STREAM_MP3;
		}
		printf("\n");
	  ret = (FILE *) fn;
	}
      }
    }
  }
  return ret;
}

gchar **read_playlist (char *name)
{
  char *filelist;
  ssize_t plsize;
  struct stat plinfo;
  FILE *fil;
  gchar **ret = NULL;

  if (name != NULL) {
    if (stat (name, &plinfo) == 0) {
      plsize = plinfo.st_size;
      if ((fil = fopen (name, "r")) != NULL) {
	filelist = g_malloc (plsize + 1);
	if (fread (filelist, plsize, 1, fil)) {
	  filelist[plsize + 1] = '\0';
	  ret = g_strsplit (filelist, "\n", 0);
	  g_free (filelist);
	}
	fclose (fil);
      }
    }
  }
  return ret;
}

u_int count_files (gchar ** flist)
{
  u_int ret = 0;

  if (flist != NULL) {
    while (flist[ret]) {
      ret++;
    }
  }
  return ret;
}

gint *create_track_order (u_int number, int randomgen)
{
  u_int *to;
  u_int i, j, tmp, found;

  to = NULL;
  if (number) {
    to = g_malloc (sizeof (u_int) * number);
    if (randomgen) {
      i = 0;
      while (i < number) {
	found = 0;
	tmp = random () % number;
	for (j = 0; j < i; j++) {
	  if (to[j] == tmp) {
	    found = 1;
	  }
	}
	if (!found) {
	  to[i] = tmp;
	  i++;
	}
      }
    } else {
      for (i = 0; i < number; i++) {
	to[i] = i;
      }
    }
  }
  return to;
}

void update_cue (char *mountname, char *name, char *artist, char *title, int secs, int remsecs)
{
  FILE *cuefile;
  char *cuename;

  cuename = g_strconcat (current_dir, "/", mountname, ".cue", NULL);
  if ((cuefile = fopen (cuename, "w+"))) {
    fprintf (cuefile, "%s\n%s\n%s\n%02d:%02d\n%02d:%02d\n", name, artist, title, secs / 60, secs % 60,
	     remsecs / 60, remsecs % 60);
    fclose (cuefile);
  }
  g_free (cuename);
}

int main (int argc, char *argv[])
{
  struct timeval tv, et;
  int ret, ret2, ret3;
  long prevsec;
  int quit = 0;
  char *header;
  int headerlen;
  int blocksreaded = 0;
  int block_out = 0;
  u_int filecount;
  int current_song, prev_song;
  char *tmpfname;
  u_long allreaded=0;
  u_long thisreaded=0;
  u_long allsent=0;
  u_long thissent=0;
  u_int firstblock = 1;
	u_int firststreamblock = 1;
  u_int do_title_stream = 0;
  gchar *title_stream_buffer = NULL;
  u_int title_stream_written;
  gchar *encoded_song;
  gchar *tmp_song;
  char *tmpbuf;
	gchar *status_str;
	u_int status_len;
	int i;
//  int ifile,ofile;

  version ();
  get_some_switches (argc, argv, NULL);

  if (NULL == password) {
    password = g_strdup ("letmein");
  }
  if (NULL == mount) {
    mount = g_strdup ("default");
  }
//  tv.tv_sec = 0;
//  tv.tv_usec = 0;

  gettimeofday (&et, NULL);
  prevsec = et.tv_sec;
  socketsinit = 0;
  socketswritten = 0;

  if (playlist != NULL) {
    if (0 == memcmp (playlist, "http://", 7)) {
      stream_relay = 1;
    }
    if (!stream_relay) {
      files = read_playlist (playlist);
      filecount = count_files (files);
      if (randomplay) {
	if (seed) {
	  srandom (seed);
	} else {
	  srandom (time (NULL));
	}
      }
    }
    if ((filecount > 0) || (stream_relay)) {
      if (!stream_relay) {
	if (track_order != NULL) {
	  g_free (track_order);
	}
	track_order = create_track_order (filecount, randomplay);
	printf ("found %d file(s) in playlist\n", filecount);
	current_song = 0;
	tmpfname = files[track_order[current_song]];
	if (basedir != NULL) {
	  fname = g_strconcat (basedir, "/", tmpfname, suffix, NULL);
	} else {
	  fname = g_strconcat (tmpfname, suffix, NULL);
	}
      } else {
	fname = g_strdup (playlist);
      }
      if (NULL == host) {
	host = g_strdup ("127.0.0.1");
      }
//      mainsock = connectto (0x7f000001, mainport);
      mainsock = connectto2 (host, mainport);
      if (mainsock != -1) {
	setup_signal_traps ();
	if ((f = media_open (fname, "r")) != NULL) {
		if (NULL == x_ac_name) {
			x_ac_name = g_strdup ("default name");
		}
		if (NULL == x_ac_genre) {
			x_ac_genre = g_strdup ("default genre");
		}
		if (NULL == x_ac_url) {
			x_ac_url = g_strdup ("http://default-url/");
		}
		if (NULL == x_ac_description) {
			x_ac_description = g_strdup ("default description");
		}
	  if (!stream_relay) {
	    current_dir = g_get_current_dir ();
			
	    if ((title_streaming) && (STREAM_MP3 == streamtype)) {
	      do_title_stream = 1;
	      if (use_tag) {
		tmp_song = g_strdup_printf ("%s - %s", actual_artist, actual_title);
	      } else if (without_dir) {
		tmp_song = g_strdup (g_basename (fname));
	      } else {
		tmp_song = g_strdup (fname);
	      }
	      encoded_song = urlencode (tmp_song);
	      g_free (tmp_song);
	      title_stream_buffer =
		g_strdup_printf
		("GET /admin.cgi?pass=%s&mode=updinfo&mount=%%2F%s&song=%s&length=%d HTTP/1.0\r\nHost: %s:%d\r\nUser-Agent: %s/%s\r\n\r\n",
		 password, mount, encoded_song, actual_size, host, mainport, GINIFEED_NAME, VERSION);
	      g_free (encoded_song);
	      title_stream_written = 0;
	    }
	    if (use_cue) {
	      update_cue (mount, fname, actual_artist, actual_title, actual_seconds_full, actual_seconds);
	    }
	  }
#ifndef ENV_CYGWIN
#ifndef ENV_SOLARIS
#ifndef ENV_AIX
	  if (daemon_mode) {
	    daemon (0, 0);
	  }
#endif
#endif
#endif
	  
	  header =
	    g_strdup_printf
	    ("SOURCE %s /%s\r\nx-audiocast-name:%s\r\nx-audiocast-genre:%s\r\nx-audiocast-url:%s\r\nx-audiocast-public:0\r\nx-audiocast-bitrate:%d\r\nx-audiocast-description:%s\r\n\r\n",
	     password, mount, x_ac_name, x_ac_genre, x_ac_url, brate ? brate : actual_nominal,
	     x_ac_description);
	  headerlen = strlen (header);

	  if (!stream_relay) {
			printf ("feeding: `%s' [%02d:%02d:%02d] ",g_basename (fname),actual_seconds / 60 / 60,
				(actual_seconds - ((actual_seconds / 60 / 60)*60*60))/60, actual_seconds % 60);
			status_str=g_strdup_printf("(%02d:%02d:%02d)",actual_seconds / 60 / 60,
				(actual_seconds - ((actual_seconds / 60 / 60)*60*60))/60, actual_seconds % 60);
			printf("%s",status_str);
			status_len=strlen(status_str);
			g_free(status_str);
	  } else {
	    printf ("feeding: `%s' ", playlist);
			status_str=g_strdup_printf("in: %lu K, out: %lu K",allreaded/1024,allsent/1024);
			printf("%s",status_str);
			status_len=strlen(status_str);
			g_free(status_str);
	  }
//                              printf("\nBSIZE: %d\n",bufsize);
	  if (brate) {
	    bufsize = (u_long) (brate * 128) * 1.01;
	  } else {
	    bufsize = actual_size;
	    bufsize = (u_long) (bufsize/actual_seconds) * 1.01;
	  }
	  buf = g_malloc (sizeof (char) * bufsize);
	  if (!stream_relay) {
	    readed = fread (buf, 1, bufsize, f);
	  } else {
	    tmpbuf = get_data_pos (rbuff);
	    if (NULL != tmpbuf) {
	      readed = readed - (tmpbuf - rbuff);
	      memcpy (buf, tmpbuf, readed);
	    } else {
	      printf ("Error getting real stream beginning\n");
	      quit = 1;
	    }
	  }
	  allreaded += readed;
	  thisreaded += readed;
	  while (!quit) {

      if ((socketsinit==2)&&(stream_relay)&&((socketswritten==readed)||(firststreamblock))){
//	    if ((socketsinit == 2) && (stream_relay)) {
	      FD_RSETALL (&sockset);
	      resettimeout (&tv);
	      ret = select (relaysock + 1, &sockset, NULL, NULL, &tv);
	      if (ret > 0) {
		if (FD_ISSET (relaysock, &sockset)) {
			if (!firststreamblock){
		  	readed = recv (relaysock, buf, bufsize, MSG_DONTWAIT);
			}else{
				readed = recv (relaysock, buf+readed, bufsize-readed, MSG_DONTWAIT) + readed;
			}
		  if (readed > 0) {
				if (firststreamblock){
					firststreamblock=0;
				}
		    socketswritten = 0;
				allreaded += readed;
				thisreaded += readed;
		  } else {
		    closesocket (relaysock);
		    printf ("\nrelayed server closed socket\n");
		    quit = 1;
		  }
		}
	      }
	    }

	    resettimeout (&tv);
	    FD_SETALL (&sockset);
			if ((!firststreamblock)||(0==socketsinit)||(!stream_relay)){
	    	FD_SETALL (&socksetw);
	    	ret = select (mainsock + 1, &sockset, &socksetw, NULL, &tv);
			}else{
	    	ret = select (mainsock + 1, &sockset, NULL, NULL, &tv);
			}
	    if (ret > 0) {

	      if (FD_ISSET (mainsock, &sockset)) {
		ret2 = recv (mainsock, rbuff, RBUFSIZE, MSG_DONTWAIT);
		if (ret2 > 0) {
		  rbuff[ret2] = '\0';
		  if (1 == socketsinit && 'O' == rbuff[0] && 'K' == rbuff[1]) {
		    socketsinit++;
		  } else {
		    printf ("\n%s\n", rbuff);
		  }
		} else {
		  closesocket (mainsock);
		  printf ("\nremote end closed socket\n");
		  quit = 1;
		}
	      }

	      if (socketsinit != 1) {
					if ((!stream_relay) || (!firststreamblock) || (0==socketsinit)){
		ret3 = 0;
		if (FD_ISSET (mainsock, &socksetw)) {
		  if ((2 == socketsinit)) {
		    if (socketswritten < readed) {
		      ret3 = send (mainsock, buf + socketswritten, readed - socketswritten, MSG_DONTWAIT);
/*if ((ret3>0)&&(stream_relay)){
ofile=open("feed.out",O_WRONLY|O_CREAT|O_APPEND,S_IRWXU);
write(ofile,buf+socketswritten,ret3);
close(ofile);
}*/
		      allsent += ret3;
		      thissent += ret3;
		      if (firstblock) {
			firstblock = 0;
		      }
		    }
		  } else if (0 == socketsinit) {
		    ret3 = send (mainsock, header + socketswritten, headerlen - socketswritten, MSG_DONTWAIT);
/*if ((ret3>0)&&(stream_relay)){
ofile=open("feed.out",O_WRONLY|O_CREAT|O_APPEND,S_IRWXU);
write(ofile,header+socketswritten,ret3);
close(ofile);
}*/
		  }
		  if ((ret3 > 0) || ((socketswritten == readed) && (2 == socketsinit))) {
		    socketswritten += ret3;
		    if (!socketsinit) {
		      if (socketswritten == headerlen) {
			socketsinit = 1;
			socketswritten = 0;
		      }
		    } else if (2 == socketsinit) {
		      if (socketswritten == readed) {
			block_out = 1;
		      }
		    }
		  } else {
		    closesocket (mainsock);
		    printf ("\nremote end closed socket\n");
		    quit = 1;
		  }
		}
	}
	      }
	    }

	    if (!stream_relay) {
	      if ((do_title_stream) && (2 == socketsinit)) {
		if (titlesock == -1) {
		  titlesock = connectto2 (host, mainport);
		} else {
		  FD_TSETALL (&socksetw);
		  resettimeout (&tv);
		  ret = select (titlesock + 1, NULL, &socksetw, NULL, &tv);
		  if (ret > 0) {
		    if (FD_ISSET (titlesock, &socksetw)) {
		      if (title_stream_written < strlen (title_stream_buffer)) {
			ret3 = send (titlesock, title_stream_buffer + title_stream_written,
				     strlen (title_stream_buffer) - title_stream_written, MSG_DONTWAIT);
			title_stream_written += ret3;
			if (ret3 <= 0) {
			  closesocket (titlesock);
			  g_free (title_stream_buffer);
			  title_stream_written = 0;
			  do_title_stream = 0;
			  titlesock = -1;
			}
		      } else {
			closesocket (titlesock);
			g_free (title_stream_buffer);
			title_stream_written = 0;
			do_title_stream = 0;
			titlesock = -1;
		      }
		    }
		  }
		}
	      }
	    }

	    fflush (0);
	    usleep (10);
	    gettimeofday (&et, NULL);
	    if (prevsec < et.tv_sec) {
	      actual_seconds--;
	      prevsec = et.tv_sec;

	      if (!stream_relay) {

		if (use_cue) {
		  update_cue (mount, fname, actual_artist, actual_title, actual_seconds_full, actual_seconds);
		}
		if ((readed < bufsize) || (skip_file)) {
		  skip_file = 0;
			for (i=0;i<status_len;i++){
				printf("\b \b");
			}
/*			for (i=0;i<status_len;i++){
				printf(" ");
			}
			for (i=0;i<status_len;i++){
				printf("\b");
			}*/
		  printf ("done\n");
		  fclose (f);
		  thisreaded = 0;
		  thissent = 0;

		  if (reread_playlist) {
		    g_strfreev (files);
		    files = read_playlist (playlist);
		    filecount = count_files (files);
		    if (track_order != NULL) {
		      g_free (track_order);
		    }
		    track_order = create_track_order (filecount, randomplay);
		    reread_playlist = 0;
		  }

		  current_song++;
		  if (current_song == filecount) {
		    current_song = 0;
		  }
		  g_free (fname);
		  tmpfname = files[track_order[current_song]];
		  if (basedir != NULL) {
		    fname = g_strconcat (basedir, "/", tmpfname, suffix, NULL);
		  } else {
		    fname = g_strconcat (tmpfname, suffix, NULL);
		  }
		  f = media_open (fname, "r");
		  while (NULL == f) {
		    printf ("WARNING: cannot open: `%s'\n", g_basename (fname));
		    if ((randomplay) && (filecount > 1)) {
		      prev_song = current_song;
		      while (prev_song == current_song) {
			current_song = random () % filecount;
		      }
		    } else {
		      current_song++;
		    }
		    if (current_song == filecount) {
		      current_song = 0;
		    }
		    g_free (fname);
		    tmpfname = files[current_song];
		    if (basedir != NULL) {
		      fname = g_strconcat (basedir, "/", tmpfname, suffix, NULL);
		    } else {
		      fname = g_strconcat (tmpfname, suffix, NULL);
		    }
		    f = media_open (fname, "r");
		  }
		  if (brate) {
		    bufsize = (u_long) (brate * 128) * 1.01;
		  } else {
		    bufsize = actual_size;
		    bufsize = (u_long) (bufsize / actual_seconds) * 1.01;
		  }
//                              printf("\nBSIZE: %d\n",bufsize);
		  buf = g_realloc (buf, sizeof (char) * bufsize);
			printf ("feeding: `%s' [%02d:%02d:%02d] ",g_basename (fname),actual_seconds / 60 / 60,
				(actual_seconds - ((actual_seconds / 60 / 60)*60*60))/60, actual_seconds % 60);
			status_str=g_strdup_printf("(%02d:%02d:%02d)",actual_seconds / 60 / 60,
				(actual_seconds - ((actual_seconds / 60 / 60)*60*60))/60, actual_seconds % 60);
			printf("%s",status_str);
			status_len=strlen(status_str);
			g_free(status_str);

		  if ((title_streaming) && (STREAM_MP3 == streamtype)) {
		    if (titlesock != -1) {
		      closesocket (titlesock);
		      titlesock = -1;
		      g_free (title_stream_buffer);
		    }
		    do_title_stream = 1;
		    if (use_tag) {
		      tmp_song = g_strdup_printf ("%s - %s", actual_artist, actual_title);
		    } else if (without_dir) {
		      tmp_song = g_strdup (g_basename (fname));
		    } else {
		      tmp_song = g_strdup (fname);
		    }
		    encoded_song = urlencode (tmp_song);
		    g_free (tmp_song);
		    title_stream_buffer =
		      g_strdup_printf
		      ("GET /admin.cgi?pass=%s&mode=updinfo&mount=%%2F%s&song=%s&length=%d HTTP/1.0\r\nHost: %s:%d\r\nUser-Agent: %s/%s\r\n\r\n",
		       password, mount, encoded_song, actual_size, host, mainport, GINIFEED_NAME, VERSION);
		    g_free (encoded_song);
		    title_stream_written = 0;
		  }
		}
	      }

				for (i=0;i<status_len;i++){
					printf("\b");
				}
	      if (!stream_relay) {
					status_str=g_strdup_printf("(%02d:%02d:%02d)",actual_seconds / 60 / 60,
						(actual_seconds - ((actual_seconds / 60 / 60)*60*60))/60, actual_seconds % 60);
					printf("%s",status_str);
					status_len=strlen(status_str);
					g_free(status_str);
	      } else {
					status_str=g_strdup_printf("in: %lu K, out: %lu K",allreaded/1024,allsent/1024);
					printf("%s",status_str);
					status_len=strlen(status_str);
					g_free(status_str);
	      }
	      fflush (0);
	      if (!stream_relay) {
		if (!firstblock) {
		  readed = fread (buf, 1, bufsize, f);
		  blocksreaded++;
		  allreaded += readed;
		  thisreaded += readed;
		  resetsockswritten ();
		  block_out = 0;
		}
	      }
	    }
	  }
	  closeall ();
	} else {
	  printf ("cannot open: %s!\n", fname);
	}
	closeallsockets ();
      } else {
	printf ("cannot connect!\n");
      }
    } else {
      printf ("playlist is empty\n");
    }
  } else {
    printf ("playlist is not defined\n");
  }
  return 0;
}
