/*
 * gini.c - main source of GINI
 * Copyright 2002, 2003 Krisztian Pifko <monsta@users.sourceforge.net>
 *
 * Author:
 *	Krisztian Pifko <monsta@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/signal.h>
#include <sys/socket.h>
#include <glib.h>
#include <string.h>
#include <sys/stat.h>
#include <fcntl.h>
#ifdef ENV_LINUX
#include <time.h>
#endif


#include "gini.h"
#include "net.h"
#include "util.h"
#include "getopt.h"

#define GINI_NAME "GINI"
#define RBUFSIZE 1024

static feed_connection **feed_connections;
static temp_connection **temp_connections;
static recv_connection **recv_connections;
static term_connection **term_connections;
static cons_connection **cons_connections;
static int temp_connections_num, feed_connections_num, recv_connections_num,
	term_connections_num,cons_connections_num;

static int mainsock;
static fd_set sockset, socksetw;
static u_char rbuff[RBUFSIZE + 1];

static int max_recvs = 8;
static int max_feeds = 2;
static int max_conss = 2;
static int mainport = 8000;
static int daemon_mode = 0;
static int fallback = 0;
static u_char *x_ac_location = NULL, *x_ac_admin = NULL, *x_ac_server_url = NULL;
static u_char *password;
static u_char *status_log_name = NULL, *status_log_name2 = NULL, *access_log_name = NULL, *system_log_name = NULL;
static u_char *current_dir;
static FILE *status_log, *access_log, *system_log;
static u_char *tmp_password;
static u_char logtime[30];
static gchar **response_messages;

static u_long allseconds = 0;
static u_long this_sec_readed = 0, this_sec_sent = 0;

static int quit=0;

static char const shortopts[] = "S:Y:A:w:l:a:u:P:m:s:c:vhzf";
static struct option const longopts[] = {
  {"status-log", required_argument, NULL, 'S'},
  {"system-log", required_argument, NULL, 'Y'},
  {"access-log", required_argument, NULL, 'A'},
  {"password", required_argument, NULL, 'w'},
  {"location", required_argument, NULL, 'l'},
  {"admin", required_argument, NULL, 'a'},
  {"url", required_argument, NULL, 'u'},
  {"port", required_argument, NULL, 'P'},
  {"max-clients", required_argument, NULL, 'm'},
  {"max-sources", required_argument, NULL, 's'},
  {"max-consoles", required_argument, NULL, 'c'},
  {"fallback", no_argument, NULL, 'f'},
  {"version", no_argument, NULL, 'v'},
  {"help", no_argument, NULL, 'h'},
  {"background", no_argument, NULL, 'z'},
  {NULL, no_argument, NULL, 0}
};

static char const *const option_help[] = {
  " -P    --port <port>          port to connect to. DEFAULT: 8000",
  " -m    --max-clients <num>    maximum number of clients allowed. DEFAULT: 8",
  " -s    --max-sources <num>    maximum number of feeders allowed. DEFAULT: 2",
  " -c    --max-consoles <num>   maximum number of consoles allowed. DEFAULT: 2",
  " -l    --location <loc>       server location.",
  " -a    --admin <e-mail>       e-mail address of admin.",
  " -u    --server-url <url>     url of server.",
  " -w    --password <pass>      password expected from feeders.",
  " -f    --fallback             play default stream if requested in not found.",
  " -z    --background           run in the background.",
  " -S    --status-log <file>    log status info to this file. DEFAULT: status.log",
  " -Y    --system-log <file>    log system info to this file. DEFAULT: system.log",
  " -A    --access-log <file>    log access info to this file. DEFAULT: access.log",
  " -v    --version              output version info.",
  " -h    --help                 output this help.",
  "",
  "Report bugs to <monsta@users.sourceforge.net>.",
  0
};

static char const *const console_help[] = {
  " exit | quit                      exit remote console",
  " shutdown                         shutdown server",
	" uptime | up                      show uptime info",
	" ls recv | feed | cons            list receive, feed or console connections",
	" kick recv | feed | cons <#id>    kick the given connection numbers",
	" set [var]	[value]                query/set system variables",
	" help                             show this help",
  0
};

static void formatlogtime ()
{
  time_t t;
  struct tm *tt;

  time (&t);
  tt = localtime (&t);
  strftime (logtime, 29, "[%d/%b/%Y:%H:%M:%S %z]", tt);
}

static void version ()
{
  printf ("%s %s %s\n%s\n", GINI_NAME, VERSION, GINI_URL, GINI_COPYRIGHT);
}

static void usage ()
{
  printf ("usage: gini [OPTIONS]\n");
  printf (" [OPTIONS] Try --help for more information\n");
  printf ("\n");
  return;
}

void get_some_switches (int argc, char *argv[], int *todo)
{
  register int optc;
	int i;
  char const *const *p;
  if (optind == argc)
    return;
  while ((optc = getopt_long (argc, argv, shortopts, longopts, (int *) 0))
	 != -1) {
    switch (optc) {
    case 'S':
      status_log_name = g_strdup (optarg);
      break;
    case 'Y':
      system_log_name = g_strdup (optarg);
      break;
    case 'A':
      access_log_name = g_strdup (optarg);
      break;
    case 'l':
      x_ac_location = g_strdup (optarg);
      break;
    case 'a':
      x_ac_admin = g_strdup (optarg);
      break;
    case 'u':
      x_ac_server_url = g_strdup (optarg);
      break;
    case 'w':
      password = g_strdup (optarg);
		  for (i=0;i<strlen(optarg);i++){
		  	optarg[i]='*';
			}
      break;
    case 'z':
      daemon_mode = 1;
      break;
    case 'f':
      fallback = 1;
      break;
    case 'P':
      mainport = atoi (optarg);
      break;
    case 'm':
      max_recvs = atoi (optarg);
      break;
    case 's':
      max_feeds = atoi (optarg);
      break;
    case 'c':
      max_conss = atoi (optarg);
      break;
    case 'v':
//      version ();
      exit (0);
      break;
    case 'h':
      usage (argv[0]);
      for (p = option_help; *p; p++)
	printf ("%s\n", *p);
      exit (0);
      break;
    default:
      usage (argv[0]);
    }
  }
}

u_long bufsize_from_bitrate (int bitr)
{
  u_long ret;
  ret = bitr * 128;
//  ret = (u_long) ret *1.01;
  ret = (u_long) ret *1.5;
  return (ret);
}

temp_connection *new_temp_connection (int socket, u_long remote_ip, u_long bufsize)
{
  temp_connection *tc;

  tc = g_malloc (sizeof (temp_connection));
  if (tc != NULL) {
    tc->socket = socket;
    tc->remote_ip = remote_ip;
    tc->readed = 0;
    tc->buffer = g_malloc (bufsize);
  }
  return tc;
}

cons_connection *new_cons_connection (int socket, u_long remote_ip, u_long bufsize)
{
  cons_connection *cc;

  cc = g_malloc (sizeof (cons_connection));
  if (cc != NULL) {
    cc->socket = socket;
    cc->remote_ip = remote_ip;
		cc->init_state = 1;
    cc->readed = 0;
    cc->written = 0;
    cc->rbuffer = g_malloc (bufsize+1);
    cc->sbuffer = g_strdup ("");
		cc->sec_start = allseconds;
		cc->sec_last = allseconds;
  }
  return cc;
}

term_connection *new_term_connection (int socket, u_long remote_ip, gchar * response_buffer)
{
  term_connection *tc;

  tc = g_malloc (sizeof (term_connection));
  if (tc != NULL) {
    tc->socket = socket;
    tc->remote_ip = remote_ip;
    tc->response_buffer = g_strdup (response_buffer);
    tc->written = 0;
  }
  return tc;
}

recv_connection *new_recv_connection (int socket, u_long remote_ip, int fc, u_char * request,
				      u_char * referer, u_char * useragent, u_int titleport)
{
  recv_connection *rc;

  rc = g_malloc (sizeof (recv_connection));
  if (rc != NULL) {
    rc->socket = socket;
    rc->remote_ip = remote_ip;
    rc->feed_conn = fc;
    rc->init_state = 0;
    rc->written = 0;
    rc->allwritten = 0;
    rc->packets = 0;
    rc->title_port = titleport;
    rc->title_socket = -1;
		rc->sec_start = allseconds;
    if (request) {
      rc->request = g_strdup (request);
    } else {
      rc->request = g_strdup ("");
    }
    if (referer) {
      rc->referer = g_strdup (referer);
    } else {
      rc->referer = g_strdup ("-");
    }
    if (useragent) {
      rc->useragent = g_strdup (useragent);
    } else {
      rc->useragent = g_strdup ("-");
    }
  }
  return rc;
}

feed_connection *new_feed_connection (int socket, u_long remote_ip, int bitrate, u_char * mount,
				      u_char * x_ac_name, u_char * x_ac_genre, u_char * x_ac_url,
				      u_char * x_ac_description)
{
  feed_connection *fc;
  u_long bs;

  fc = g_malloc (sizeof (feed_connection));
  if (fc != NULL) {
    fc->socket = socket;
    fc->remote_ip = remote_ip;
    fc->init_state = 1;
    fc->stream_type = STREAM_UNKNOWN;
    fc->bitrate = bitrate;
    if (mount) {
      fc->mount = g_strdup (mount);
    } else {
      fc->mount = g_strdup ("/default");
    }
    fc->header = NULL;
    bs = bufsize_from_bitrate (bitrate);
    fc->buffer = g_malloc (bs);
    fc->skipfirst = 0;
    fc->readed = 0;
    fc->packets = 0;
    fc->bytes = 0;
    fc->media_header = NULL;
    fc->media_header_len = 0;
    fc->title_string = NULL;
    fc->file_size = 0;
    fc->udpseq = 0;
		fc->sec_start = allseconds;
    if (x_ac_name) {
      fc->x_ac_name = g_strdup (x_ac_name);
    } else {
      fc->x_ac_name = g_strdup ("default name");
    }
    if (x_ac_genre) {
      fc->x_ac_genre = g_strdup (x_ac_genre);
    } else {
      fc->x_ac_genre = g_strdup ("default genre");
    }
    if (x_ac_url) {
      fc->x_ac_url = g_strdup (x_ac_url);
    } else {
      fc->x_ac_url = g_strdup ("http://default-url/");
    }
    if (x_ac_description) {
      fc->x_ac_description = g_strdup (x_ac_description);
    } else {
      fc->x_ac_description = g_strdup ("default description");
    }
  }
  return fc;
}

void free_feed_connection (feed_connection * fc)
{
  if (fc != NULL) {
    g_free (fc->mount);
    g_free (fc->header);
    g_free (fc->buffer);
    g_free (fc->media_header);
    g_free (fc->x_ac_name);
    g_free (fc->x_ac_genre);
    g_free (fc->x_ac_url);
    g_free (fc->x_ac_description);
    g_free (fc->title_string);
    g_free (fc);
  }
}

void free_temp_connection (temp_connection * tc)
{
  if (tc != NULL) {
    g_free (tc->buffer);
    g_free (tc);
  }
}

void free_cons_connection (cons_connection * cc)
{
  if (cc != NULL) {
    g_free (cc->rbuffer);
    g_free (cc->sbuffer);
    g_free (cc);
  }
}

void free_term_connection (term_connection * tc)
{
  if (tc != NULL) {
    g_free (tc->response_buffer);
    g_free (tc);
  }
}

void free_recv_connection (recv_connection * rc)
{
  if (rc != NULL) {
    g_free (rc->request);
    g_free (rc->referer);
    g_free (rc->useragent);
    g_free (rc);
  }
}

int get_feed_connection_index_by_socket (int socket)
{
  int i, ret = -1;

  for (i = 0; i < feed_connections_num; i++) {
    if (feed_connections[i]->socket == socket) {
      ret = i;
    }
  }
  return ret;
}

int get_temp_connection_index_by_socket (int socket)
{
  int i, ret = -1;

  for (i = 0; i < temp_connections_num; i++) {
    if (temp_connections[i]->socket == socket) {
      ret = i;
    }
  }
  return ret;
}

int get_term_connection_index_by_socket (int socket)
{
  int i, ret = -1;

  for (i = 0; i < term_connections_num; i++) {
    if (term_connections[i]->socket == socket) {
      ret = i;
    }
  }
  return ret;
}

int get_cons_connection_index_by_socket (int socket)
{
  int i, ret = -1;

  for (i = 0; i < cons_connections_num; i++) {
    if (cons_connections[i]->socket == socket) {
      ret = i;
    }
  }
  return ret;
}

int get_recv_connection_index_by_socket (int socket)
{
  int i, ret = -1;

  for (i = 0; i < recv_connections_num; i++) {
    if (recv_connections[i]->socket == socket) {
      ret = i;
    }
  }
  return ret;
}

int get_feed_connection_index_by_mount (u_char * mount)
{
  int i, ret = -1;

  for (i = 0; i < feed_connections_num; i++) {
    if (feed_connections[i]->mount) {
      if (!strcmp (feed_connections[i]->mount, mount)) {
	ret = i;
      }
    }
  }
  return ret;
}

int find_mount (u_char * name, int fback)
{
  int ret = -1;

  ret = get_feed_connection_index_by_mount (name);
  if ((-1 == ret) && (fback)) {
    ret = 0;
  }
  return ret;
}

void reset_recv_connection_written (int feedsockid)
{
  recv_connections[feedsockid]->written = 0;
}

void null_temp_connections (void)
{
  int i;

  for (i = 0; i < max_recvs + max_feeds + max_conss; i++) {
    temp_connections[i] = NULL;
  }
}

void null_term_connections (void)
{
  int i;

  for (i = 0; i < max_recvs + max_feeds + max_conss; i++) {
    term_connections[i] = NULL;
  }
}

void null_recv_connections (void)
{
  int i;

  for (i = 0; i < max_recvs; i++) {
    recv_connections[i] = NULL;
  }
}

void null_cons_connections (void)
{
  int i;

  for (i = 0; i < max_conss; i++) {
    cons_connections[i] = NULL;
  }
}

void reset_feed_connections_readed (void)
{
  int i;

  for (i = 0; i < feed_connections_num; i++) {
    feed_connections[i]->readed = 0;
  }
}

int any_feed_connections_readed (void)
{
  int i, ret = 0;

  for (i = 0; i < feed_connections_num; i++) {
    ret += feed_connections[i]->readed;
  }
  return ret;
}

void null_feed_connections (void)
{
  int i;

  feed_connections_num = 0;
  for (i = 0; i < max_feeds; i++) {
    feed_connections[i] = NULL;
  }
}

int count_valid_feed_connections (void)
{
  int i, ret = 0;

  for (i = 0; i < feed_connections_num; i++) {
    if ((2 == feed_connections[i]->init_state) && (feed_connections[i]->stream_type != STREAM_UNKNOWN)) {
      ret++;
    }
  }
  return ret;
}

int count_feed_connections (void)
{
  int i, ret = 0;

  for (i = 0; i < feed_connections_num; i++) {
    if (feed_connections[i]->init_state != 0) {
      ret++;
    }
  }
  return ret;
}

void add_recv_connection (int sock, u_long remote_ip, int cfeed, u_char * request, u_char * referer,
			  u_char * useragent, u_int titleport)
{
  recv_connection *rc;

  if (sock > -1) {
    rc = new_recv_connection (sock, remote_ip, cfeed, request, referer, useragent, titleport);
    if (rc != NULL) {
      recv_connections[recv_connections_num] = rc;
      recv_connections_num++;
      if ((titleport) && (STREAM_MP3 == feed_connections[cfeed]->stream_type)) {
	rc->title_socket = connect_udp (remote_ip, titleport);
      }
    }
  }
  g_free (request);
  g_free (referer);
  g_free (useragent);
}

void add_temp_connection (int sock, u_long remote_ip)
{
  temp_connection *tc;

  if (sock > -1) {
    tc = new_temp_connection (sock, remote_ip, RBUFSIZE);
    if (tc != NULL) {
      temp_connections[temp_connections_num] = tc;
      temp_connections_num++;
    }
  }
}

void add_cons_connection (int sock, u_long remote_ip)
{
  cons_connection *cc;

  if (sock > -1) {
    cc = new_cons_connection (sock, remote_ip, RBUFSIZE);
    if (cc != NULL) {
      cons_connections[cons_connections_num] = cc;
      cons_connections_num++;
    }
  }
}

void add_term_connection (int socket, u_long remote_ip, gchar * response_buffer)
{
  term_connection *tc;

  if (socket > -1) {
    tc = new_term_connection (socket, remote_ip, response_buffer);
    if (tc != NULL) {
      term_connections[term_connections_num] = tc;
      term_connections_num++;
    }
  }
}

void add_feed_connection (int sock, u_long remote_ip, u_char * mount, u_char * ac_name, u_char * ac_genre,
			  u_char * ac_url, u_char * ac_desc, int brate)
{
  feed_connection *fc;

  if (sock > -1) {
    fc = new_feed_connection (sock, remote_ip, brate, mount, ac_name, ac_genre, ac_url, ac_desc);
    if (fc != NULL) {
      feed_connections[feed_connections_num] = fc;
      feed_connections_num++;
    }
  }
  g_free (mount);
  g_free (ac_name);
  g_free (ac_genre);
  g_free (ac_url);
  g_free (ac_desc);
}

void set_feed_connection_state (int sock, int state)
{
  int pos;

  pos = get_feed_connection_index_by_socket (sock);
  if (pos > -1) {
    feed_connections[pos]->init_state = state;
  }
}

void set_cons_connection_state (int sock, int state)
{
  int pos;

  pos = get_cons_connection_index_by_socket (sock);
  if (pos > -1) {
    cons_connections[pos]->init_state = state;
  }
}

void set_recv_connection_state (int sock, int state)
{
  int pos;

  pos = get_recv_connection_index_by_socket (sock);
  if (pos > -1) {
    recv_connections[pos]->init_state = state;
  }
}

void set_feed_connection_type (int sock, int type)
{
  int pos;
  feed_connection *fc;

  pos = get_feed_connection_index_by_socket (sock);
  fc = feed_connections[pos];
  if (pos > -1) {
    fc->stream_type = type;
		fc->header =
	g_strdup_printf
	("HTTP/1.0 200 OK\r\nServer: %s/%s\r\nContent-Type: %s\r\nx-audiocast-location: %s\r\nx-audiocast-admin: %s\r\nx-audiocast-server-url: %s\r\nx-audiocast-mount: %s\r\nx-audiocast-name: %s\r\nx-audiocast-description: %s\r\nx-audiocast-url: %s\r\nx-audiocast-genre: %s\r\nx-audiocast-bitrate: %d\r\nx-audiocast-public: 0\r\n\r\n",
	 GINI_NAME, VERSION, mime_types[type], x_ac_location, x_ac_admin, x_ac_server_url, fc->mount, fc->x_ac_name,
	 fc->x_ac_description, fc->x_ac_url, fc->x_ac_genre, fc->bitrate);
  }
}

void remove_recv_connection (int sock)
{
  int i, pos;
  recv_connection *rc;
  struct in_addr in_address;

  pos = get_recv_connection_index_by_socket (sock);
  if (pos > -1) {
    rc = recv_connections[pos];
    in_address.s_addr = htonl (rc->remote_ip);
    if (rc->title_socket != -1) {
      closesocket (rc->title_socket);
      rc->title_socket = -1;
    }
    fprintf (access_log, "%s - - %s \"%s\" 200 %lu \"%s\" \"%s\"\n", inet_ntoa (in_address), logtime,
	     rc->request, rc->allwritten, rc->referer, rc->useragent);
    free_recv_connection (recv_connections[pos]);

    for (i = pos; i < recv_connections_num - 1; i++) {
      recv_connections[i] = recv_connections[i + 1];
    }
    recv_connections_num--;
    recv_connections[recv_connections_num] = NULL;
  }
}

void remove_temp_connection (int sock)
{
  int i, pos;

  pos = get_temp_connection_index_by_socket (sock);
  if (pos > -1) {
    free_temp_connection (temp_connections[pos]);
    for (i = pos; i < temp_connections_num - 1; i++) {
      temp_connections[i] = temp_connections[i + 1];
    }
    temp_connections_num--;
    temp_connections[temp_connections_num] = NULL;
  }
}

void remove_term_connection (int sock)
{
  int i, pos;

  pos = get_term_connection_index_by_socket (sock);
  if (pos > -1) {
    free_term_connection (term_connections[pos]);
    for (i = pos; i < term_connections_num - 1; i++) {
      term_connections[i] = term_connections[i + 1];
    }
    term_connections_num--;
    term_connections[term_connections_num] = NULL;
  }
}

void remove_feed_connection (int sock)
{
  int i, pos, tsock;

  pos = get_feed_connection_index_by_socket (sock);
  if (pos > -1) {
    for (i = recv_connections_num - 1; i >= 0; i--) {
      if (recv_connections[i]->feed_conn == pos) {
	tsock = recv_connections[i]->socket;
	closesocket (tsock);
	remove_recv_connection (tsock);
      } else if (recv_connections[i]->feed_conn > pos) {
	recv_connections[i]->feed_conn--;
      }
    }
    free_feed_connection (feed_connections[pos]);
    for (i = pos; i < feed_connections_num - 1; i++) {
      feed_connections[i] = feed_connections[i + 1];
    }
    feed_connections_num--;
    feed_connections[feed_connections_num] = NULL;
  }
}

void remove_cons_connection (int sock)
{
  int i, pos;

  pos = get_cons_connection_index_by_socket (sock);
  if (pos > -1) {
    free_cons_connection (cons_connections[pos]);
    for (i = pos; i < cons_connections_num - 1; i++) {
      cons_connections[i] = cons_connections[i + 1];
    }
    cons_connections_num--;
    cons_connections[cons_connections_num] = NULL;
  }
}

void temp_to_term (int sock, gchar * response_buffer)
{
  int pos;

  pos = get_temp_connection_index_by_socket (sock);
  if (pos > -1) {
    add_term_connection (sock, temp_connections[pos]->remote_ip, response_buffer);
    remove_temp_connection (sock);
  }
}

void feed_to_term (int sock, gchar * response_buffer)
{
  int pos;

  pos = get_feed_connection_index_by_socket (sock);
  if (pos > -1) {
    add_term_connection (sock, feed_connections[pos]->remote_ip, response_buffer);
    remove_feed_connection (sock);
  }
}

int get_max_recv_socket (void)
{
  int i, max = -1;

  for (i = 0; i < recv_connections_num; i++) {
    if (recv_connections[i]->socket > max) {
      max = recv_connections[i]->socket;
    }
  }
  return max;
}

int get_max_cons_socket (void)
{
  int i, max = -1;

  for (i = 0; i < cons_connections_num; i++) {
    if (cons_connections[i]->socket > max) {
      max = cons_connections[i]->socket;
    }
  }
  return max;
}

int get_max_temp_socket (void)
{
  int i, max = -1;

  for (i = 0; i < temp_connections_num; i++) {
    if (temp_connections[i]->socket > max) {
      max = temp_connections[i]->socket;
    }
  }
  return max;
}

int get_max_term_socket (void)
{
  int i, max = -1;

  for (i = 0; i < term_connections_num; i++) {
    if (term_connections[i]->socket > max) {
      max = term_connections[i]->socket;
    }
  }
  return max;
}

int get_max_feed_socket (void)
{
  int i, max = -1;

  for (i = 0; i < feed_connections_num; i++) {
    if (feed_connections[i]->socket > max) {
      max = feed_connections[i]->socket;
    }
  }
  return max;
}

void closeallsockets (void)
{
  int i;

  for (i = 0; i < recv_connections_num; i++) {
    closesocket (recv_connections[i]->socket);
  }
  for (i = 0; i < temp_connections_num; i++) {
    closesocket (temp_connections[i]->socket);
  }
  for (i = 0; i < feed_connections_num; i++) {
    closesocket (feed_connections[i]->socket);
  }
  for (i = 0; i < term_connections_num; i++) {
    closesocket (term_connections[i]->socket);
  }
  for (i = 0; i < cons_connections_num; i++) {
    closesocket (cons_connections[i]->socket);
  }
  closesocket (mainsock);
}

void send_title_infos (void)
{
  int i;
  recv_connection *rc;
  gchar *title_info;

  for (i = 0; i < recv_connections_num; i++) {
    rc = recv_connections[i];
    if ((rc->title_socket != -1) && (feed_connections[rc->feed_conn]->title_string != NULL)) {
      title_info =
	g_strdup_printf
	("x-audiocast-udpseqnr: %lu\r\nx-audiocast-streamtitle: %s\r\nx-audiocast-streamurl: %s\r\nx-audiocast-streamlength: %lu\r\n",
	 feed_connections[rc->feed_conn]->udpseq, feed_connections[rc->feed_conn]->title_string,
	 feed_connections[rc->feed_conn]->x_ac_url, feed_connections[rc->feed_conn]->file_size);
      send (rc->title_socket, title_info, strlen (title_info), MSG_DONTWAIT);
//                      printf("%s",title_info);
      g_free (title_info);
    }
  }
}


void closeall (void)
{
  int i;

  closeallsockets ();

  for (i = 0; i < recv_connections_num; i++) {
    free_recv_connection (recv_connections[i]);
  }
  for (i = 0; i < temp_connections_num; i++) {
    free_temp_connection (temp_connections[i]);
  }
  for (i = 0; i < feed_connections_num; i++) {
    free_feed_connection (feed_connections[i]);
  }
  for (i = 0; i < term_connections_num; i++) {
    free_term_connection (term_connections[i]);
  }
  for (i = 0; i < cons_connections_num; i++) {
    free_cons_connection (cons_connections[i]);
  }
  g_free (recv_connections);
  g_free (temp_connections);
  g_free (feed_connections);
  g_free (term_connections);
  g_free (cons_connections);

//  fclose (status_log);
  fclose (system_log);
  fclose (access_log);

  g_free (system_log_name);
  g_free (access_log_name);
  g_free (status_log_name2);
  g_free (status_log_name);
  g_free (current_dir);
  g_free (x_ac_location);
  g_free (x_ac_admin);
  g_free (x_ac_server_url);
  g_free (password);

  for (i = 0; i < RESPONSES_NUM; i++) {
    g_free (response_messages[i]);
  }
  g_free (response_messages);
}

void FD_SETALL (fd_set * set)
{
  int i;

  FD_ZERO (set);
  for (i = 0; i < recv_connections_num; i++) {
    FD_SET (recv_connections[i]->socket, set);
  }
}

void FD_CCSETALL (fd_set * set)
{
  int i;

  FD_ZERO (set);
  for (i = 0; i < cons_connections_num; i++) {
    FD_SET (cons_connections[i]->socket, set);
  }
}

void FD_CCW1SETALL (fd_set * set)
{
  int i;

  FD_ZERO (set);
  for (i = 0; i < cons_connections_num; i++) {
		if (cons_connections[i]->init_state==1){
    	FD_SET (cons_connections[i]->socket, set);
		}
  }
}

void FD_CCW2SETALL (fd_set * set)
{
  int i;

  FD_ZERO (set);
  for (i = 0; i < cons_connections_num; i++) {
		if ((cons_connections[i]->init_state==2)&&(strlen(cons_connections[i]->sbuffer))){
    	FD_SET (cons_connections[i]->socket, set);
		}
  }
}

void FD_CSETALL (fd_set * set)
{

  FD_ZERO (set);
	FD_SET (0, set);
}

void FD_TSETALL (fd_set * set)
{
  int i;

  FD_ZERO (set);
  for (i = 0; i < temp_connections_num; i++) {
    FD_SET (temp_connections[i]->socket, set);
  }
}

void FD_RSETALL (fd_set * set)
{
  int i;

  FD_ZERO (set);
  for (i = 0; i < term_connections_num; i++) {
    FD_SET (term_connections[i]->socket, set);
  }
}

void FD_FSETALL (fd_set * set)
{
  int i;

  FD_ZERO (set);
  for (i = 0; i < feed_connections_num; i++) {
    FD_SET (feed_connections[i]->socket, set);
  }
}

void signalh (const int sig);

static void setup_signal_traps ()
{
  signal (SIGSEGV, signalh);
  signal (SIGPIPE, signalh);
  signal (SIGINT, signalh);
  signal (SIGQUIT, signalh);
  signal (SIGFPE, signalh);
  signal (SIGBUS, signalh);
  signal (SIGTERM, signalh);
// signal (SIGHUP, signalh);
// signal (SIGUSR1, signalh);
// signal (SIGUSR2, signalh);
// signal (SIGCHLD, signalh);
  signal (SIGIO, signalh);
#ifndef ENV_CYGWIN
  signal (SIGALRM, SIG_IGN);
#endif
}

void signalh (const int sig)
{
  setup_signal_traps ();
  switch (sig) {
// case SIGHUP:
// break;
  case SIGSEGV:
  case SIGBUS:
  case SIGFPE:
  case SIGIO:
  case SIGTERM:
//    printf ("\nEEEEK!\n");
	  quit=1;
    break;
  case SIGINT:
  case SIGQUIT:
    printf ("\n%s IS EXITING!\n", GINI_NAME);
	  quit=1;
    break;
  }
  setup_signal_traps ();
}

u_char *get_ogg_header (u_char * b, int len)
{
  
u_char * h;
  int found = 0;

  h = memfind ("OggS", b, len, 4);
  while ((h) && (found < 2)) {
    found++;
    h = memfind ("OggS", h + 4, len - (h - b) - 1 - 4, 4);
  }
  return h;
}

u_char *get_avi_header (u_char * b, int len)
{

  u_char *h;

  h = memfind ("movi", b, len, 4);
  return h;
}

u_char *get_mov_header (u_char * b, int len)
{

  u_char *h;

  h = memfind ("mdat", b, len, 4);
  return h;
}

u_char *get_asf_header (u_char * b, int len)
{

  u_char *h;

  h = memfind ("\x36\x26\xB2\x75", b, len, 4);
  return h;
}

u_char *get_real_header (u_char * b, int len)
{

  u_char *h;

  h = memfind ("DATA", b, len, 4);
  return h;
}

int count_ogg_headers (u_char * b, int len)
{
  u_char *h;
  int found = 0;

  h = memfind ("OggS", b, len, 4);
  while (h) {
    found++;
    h = memfind ("OggS", h + 4, len - (h - b) - 1 - 4, 4);
  }
  return found;
}

int check_ogg_header (int sock)
{
  u_char *firsth, *lasth, *b;
  int hcount, i, len;
	int ret=0;

  i = get_feed_connection_index_by_socket (sock);
  if (i > -1) {
    b = feed_connections[i]->buffer;
    len = feed_connections[i]->readed;
    hcount = count_ogg_headers (b, len);
    if (hcount > 2) {
      firsth = b;
//                      firsth = memfind ("OggS", b, len, 4);
      lasth = get_ogg_header (firsth, len);
			if (NULL!=lasth){
				if (feed_connections[i]->media_header != NULL) {
					g_free (feed_connections[i]->media_header);
				}
				feed_connections[i]->media_header_len = lasth - firsth + 4;
				feed_connections[i]->media_header = g_malloc (feed_connections[i]->media_header_len);
				memcpy (feed_connections[i]->media_header, b, feed_connections[i]->media_header_len);
				//              ogg_headers_set(i);
				fprintf (system_log, "%s: ogg header cought, len: %d\n", logtime,
					 feed_connections[i]->media_header_len);
				ret=1;
			}
    }
  }
	return ret;
}

int check_mov_header (int sock)
{
  u_char *firsth, *lasth, *b;
  int i, len;
	int ret=0;

  i = get_feed_connection_index_by_socket (sock);
  if (i > -1) {
    b = feed_connections[i]->buffer;
    len = feed_connections[i]->readed;
    firsth = b;
//                      firsth = memfind ("OggS", b, len, 4);
    lasth = get_mov_header (firsth, len);
		if (NULL!=lasth){
			if (feed_connections[i]->media_header != NULL) {
				g_free (feed_connections[i]->media_header);
			}
			feed_connections[i]->media_header_len = lasth - firsth + 4;
			feed_connections[i]->media_header = g_malloc (feed_connections[i]->media_header_len);
			memcpy (feed_connections[i]->media_header, b, feed_connections[i]->media_header_len);
			fprintf (system_log, "%s: mov header cought, len: %d\n", logtime, feed_connections[i]->media_header_len);
			ret=1;
		}
  }
	return ret;
}

int check_avi_header (int sock)
{
  u_char *firsth, *lasth, *b;
  int i, len;
	int ret=0;

  i = get_feed_connection_index_by_socket (sock);
  if (i > -1) {
    b = feed_connections[i]->buffer;
    len = feed_connections[i]->readed;
    firsth = b;
//                      firsth = memfind ("OggS", b, len, 4);
    lasth = get_avi_header (firsth, len);
		if (NULL!=lasth){
			if (feed_connections[i]->media_header != NULL) {
				g_free (feed_connections[i]->media_header);
			}
			feed_connections[i]->media_header_len = lasth - firsth + 4;
			feed_connections[i]->media_header = g_malloc (feed_connections[i]->media_header_len);
			memcpy (feed_connections[i]->media_header, b, feed_connections[i]->media_header_len);
			fprintf (system_log, "%s: avi header cought, len: %d\n", logtime, feed_connections[i]->media_header_len);
			ret=1;
		}
  }
	return ret;
}

void check_avi_frame (int sock)
{
  u_char *firsth, *lasth, *lasth2, *lasth3, *b;
  int i, len;

  i = get_feed_connection_index_by_socket (sock);
  if (i > -1) {
    b = feed_connections[i]->buffer;
    len = feed_connections[i]->readed;
    firsth = b;
    lasth = memfind ("00db", b, len, 4);
    lasth2 = memfind ("00dc", b, len, 4);
    lasth3 = memfind ("01wb", b, len, 4);
    if (lasth3 != NULL) {
      if (lasth3 < lasth2) {
	lasth2 = lasth3;
      }
    }
    if (NULL == lasth) {
      lasth = lasth2;
    } else {
      if (lasth2 < lasth) {
	lasth = lasth2;
      }
    }
    if (lasth != NULL) {
      feed_connections[i]->skipfirst = lasth - firsth;
    } else {
      feed_connections[i]->skipfirst = 0;
    }
  }
}

int check_asf_header (int sock)
{
  u_char *firsth, *lasth, *b;
  int i, len;
	int ret=0;

  i = get_feed_connection_index_by_socket (sock);
  if (i > -1) {
    b = feed_connections[i]->buffer;
    len = feed_connections[i]->readed;
    firsth = b;
//                      firsth = memfind ("OggS", b, len, 4);
    lasth = get_asf_header (firsth, len);
		if (NULL!=lasth){
			if (feed_connections[i]->media_header != NULL) {
				g_free (feed_connections[i]->media_header);
			}
			feed_connections[i]->media_header_len = lasth - firsth + 50;
			feed_connections[i]->media_header = g_malloc (feed_connections[i]->media_header_len);
			memcpy (feed_connections[i]->media_header, b, feed_connections[i]->media_header_len);
			fprintf (system_log, "%s: asf header cought, len: %d\n", logtime, feed_connections[i]->media_header_len);
			ret=1;
		}
  }
	return ret;
}

void check_asf_frame (int sock)
{
  u_char *firsth, *lasth, *b;
  int i, len;

  i = get_feed_connection_index_by_socket (sock);
  if (i > -1) {
    b = feed_connections[i]->buffer;
    len = feed_connections[i]->readed;
    firsth = b;
    lasth = memfind ("\x82\x00", b, len, 2);
    if (lasth != NULL) {
      feed_connections[i]->skipfirst = lasth - firsth;
    } else {
      feed_connections[i]->skipfirst = 0;
    }
  }
}

int check_real_header (int sock)
{
  u_char *firsth, *lasth, *b;
  int i, len;
	int ret=0;

  i = get_feed_connection_index_by_socket (sock);
  if (i > -1) {
    b = feed_connections[i]->buffer;
    len = feed_connections[i]->readed;
    firsth = b;
//                      firsth = memfind ("OggS", b, len, 4);
    lasth = get_real_header (firsth, len);
		if (NULL!=lasth){
			if (feed_connections[i]->media_header != NULL) {
				g_free (feed_connections[i]->media_header);
			}
			feed_connections[i]->media_header_len = lasth - firsth + 10;
			feed_connections[i]->media_header = g_malloc (feed_connections[i]->media_header_len);
			memcpy (feed_connections[i]->media_header, b, feed_connections[i]->media_header_len);
			fprintf (system_log, "%s: real header cought, len: %d\n", logtime, feed_connections[i]->media_header_len);
			ret=1;
		}
  }
	return ret;
}

void check_real_frame (int sock)
{
  u_char *firsth, *lasth, *b;
  int i, len;

  i = get_feed_connection_index_by_socket (sock);
  if (i > -1) {
    b = feed_connections[i]->buffer;
    len = feed_connections[i]->readed;
    firsth = b;
    lasth = memfind ("\x00\x00", b, len, 2);
    if (lasth != NULL) {
      feed_connections[i]->skipfirst = lasth - firsth;
    } else {
      feed_connections[i]->skipfirst = 0;
    }
  }
}

u_char *get_source_mount (u_char * data)
{
  u_char *first, *last, *crlf, *ret = NULL;

  first = strchr (data, ' ');
  if (first != NULL) {
    last = strchr (first + 1, ' ');
    if (last != NULL) {
      crlf = strstr (data, "\r\n");
      if (NULL == crlf) {
	crlf = strchr (data, '\n');
      
}
      if (crlf != NULL) {
	if (last < crlf) {
	  ret = g_malloc (sizeof (u_char) * (crlf - last));
	  ret = strncpy (ret, last + 1, crlf - last - 1);
	  ret[crlf - last - 1] = '\0';
	}
      }
    }
  }
  return ret;
}

u_char *get_request_mount (u_char * data)
{
  u_char *first, *last, *ret = NULL;

  first = strchr (data, ' ');
  if (first != NULL) {
    last = strchr (first + 1, ' ');
    if (last != NULL) {
      ret = g_malloc (sizeof (u_char) * (last - first));
      ret = strncpy (ret, first + 1, last - first - 1);
      ret[last - first - 1] = '\0';
    }
  }
  return ret;
}

u_char *get_request_request (u_char * data)
{
  u_char *first, *last, *ret = NULL;

  first = data;
  if (first != NULL) {
    last = strstr (first, "\r\n");
    if (NULL == last) {
      last = strchr (first, '\n');
    }
    if (last != NULL) {
      ret = g_malloc (sizeof (u_char) * (last - first + 1));
      ret = strncpy (ret, first, last - first);
      ret[last - first] = '\0';
    }
  }
  return ret;
}

u_char *get_source_password (u_char * data)
{
  u_char *first, *last, *ret = NULL;

  first = strchr (data, ' ');
  if (first != NULL) {
    last = strchr (first + 1, ' ');
    if (last != NULL) {
      ret = g_malloc (sizeof (u_char) * (last - first));
      ret = strncpy (ret, first + 1, last - first - 1);
      ret[last - first - 1] = '\0';
    }
  }
  return ret;
}

u_char *get_cons_password (u_char * data)
{
  u_char *first, *last, *ret = NULL;

  first = strchr (data, ' ');
  if (first != NULL) {
    last = strstr (first, "\r\n");
		if (NULL == last){
			last = strchr (first, '\n');
		}
    if (last != NULL) {
      ret = g_malloc (sizeof (u_char) * (last - first));
      ret = strncpy (ret, first + 1, last - first - 1);
      ret[last - first - 1] = '\0';
    }
  }
  return ret;
}

static gchar *parsecommand(gchar *comm, int socket)
{
	gchar *ret, *w, *tmp, *tmp2;
	gchar **words;
	int i,j;
	int sock;
	u_int log_day,log_hour,log_min;
  struct in_addr in_address;
	u_long dur_sec;
	float data;
	char *sizecode;
	
	ret=NULL;
//	ret=g_strdup("");
	w=g_strdup(comm);
	i=0;
	while (w[i]){
		if ('\t'==w[i]){
			w[i]=' ';
		}
		i++;
	}
	g_strstrip(w);
	words=g_strsplit(w," ",0);
	if (words[0]){
		g_strup(words[0]);
	i=0;
	while (words[i]){
		if (0==words[i][0]){
			for (j=i;words[j];j++){
				words[j]=words[j+1];
			}
		}else{
			i++;
		}
	}
/*	for (i=0;words[i];i++){
		
	}*/
//	ret=g_strjoinv("\" \"",words);
	if ((!strcmp(words[0],"EXIT"))||(!strcmp(words[0],"QUIT"))){
		if (-1==socket){
			ret=g_strdup("use shutdown for exit on local consoles");
		}
	} else if (!strcmp(words[0],"SHUTDOWN")){
		quit=1;
	} else if (!strcmp(words[0],"HELP")){
		i=0;
		ret=g_strdup("");
		while (console_help[i]){
			tmp=ret;
			ret=g_strconcat(tmp,console_help[i],"\n",NULL);
			g_free(tmp);
			i++;
		}
	} else if ((!strcmp(words[0],"UPTIME"))||(!strcmp(words[0],"UP"))){
		log_day = allseconds / 60 / 60 / 24;
	  log_hour = (allseconds - (log_day * 60 * 60 * 24)) / 60 / 60;
	  log_min = (allseconds - (log_day * 60 * 60 * 24) - (log_hour * 60 * 60)) / 60;
		ret=g_strdup_printf(" %02d:%02d:%02lu up %d days, %d clients on %d streams, %d admins, %lu kbps in, %lu kbps out",
			log_hour,log_min,allseconds%60,log_day,recv_connections_num,feed_connections_num,cons_connections_num,this_sec_readed * 8 / 1024, this_sec_sent * 8 / 1024);
	} else if (!strcmp(words[0],"LS")){
		if (words[1]){
			g_strup(words[1]);
			if (!strcmp(words[1],"RECV")){
				ret=g_strdup(" #\tip\t\tmount\t\tplayer\t\twritten\t time\n");
				for (i = 0; i < recv_connections_num; i++) {
					in_address.s_addr = htonl (recv_connections[i]->remote_ip);
					dur_sec=allseconds-recv_connections[i]->sec_start;
					log_day = dur_sec / 60 / 60 / 24;
					log_hour = (dur_sec - (log_day * 60 * 60 * 24)) / 60 / 60;
					log_min = (dur_sec - (log_day * 60 * 60 * 24) - (log_hour * 60 * 60)) / 60;
					data=convert_size(recv_connections[i]->allwritten,&sizecode);
					tmp=ret;
					tmp2 = g_strdup_printf (" %d\t%s\t%s\t\t%s\t%.1f %s\t %02d:%02d:%02lu, %dd\n",i,
						 inet_ntoa (in_address), feed_connections[recv_connections[i]->feed_conn]->mount,
						 recv_connections[i]->useragent, data, sizecode,
							log_hour,log_min,dur_sec%60,log_day);
					ret=g_strconcat(tmp,tmp2,NULL);
					g_free(tmp);
					g_free(tmp2);
				}
			} else if (!strcmp(words[1],"FEED")){
				ret=g_strdup(" #\tmount\tformat\t\tip\t\treaded\t time\n");
				for (i = 0; i < feed_connections_num; i++) {
					in_address.s_addr = htonl (feed_connections[i]->remote_ip);
					dur_sec=allseconds-feed_connections[i]->sec_start;
					log_day = dur_sec / 60 / 60 / 24;
					log_hour = (dur_sec - (log_day * 60 * 60 * 24)) / 60 / 60;
					log_min = (dur_sec - (log_day * 60 * 60 * 24) - (log_hour * 60 * 60)) / 60;
					data=convert_size(feed_connections[i]->bytes,&sizecode);
					tmp=ret;
					tmp2=g_strdup_printf(" %d\t%s\t%d kbps %s\t%s\t%.1f %s\t %02d:%02d:%02lu, %dd\n",i,
						 feed_connections[i]->mount, feed_connections[i]->bitrate,
						 stream_types[feed_connections[i]->stream_type], inet_ntoa (in_address),
						 data, sizecode,
							log_hour,log_min,dur_sec%60,log_day);
					ret=g_strconcat(tmp,tmp2,NULL);
					g_free(tmp);
					g_free(tmp2);
				}
			} else if (!strcmp(words[1],"CONS")){
				ret=g_strdup(" #\tip\t\t\tidle\n");
				for (i = 0; i < cons_connections_num; i++) {
					in_address.s_addr = htonl (cons_connections[i]->remote_ip);
					dur_sec=allseconds-cons_connections[i]->sec_last;
					log_day = dur_sec / 60 / 60 / 24;
					log_hour = (dur_sec - (log_day * 60 * 60 * 24)) / 60 / 60;
					log_min = (dur_sec - (log_day * 60 * 60 * 24) - (log_hour * 60 * 60)) / 60;
					tmp=ret;
					tmp2=g_strdup_printf(" %d\t%s\t\t%02d:%02d:%02lu, %dd\n",i,
						 inet_ntoa (in_address), log_hour,log_min,dur_sec%60,log_day);
					ret=g_strconcat(tmp,tmp2,NULL);
					g_free(tmp);
					g_free(tmp2);
				}
			} else {
				ret=g_strdup("ls what?");
			}
		}else{
			ret=g_strdup("ls what?");
		}
	} else if (!strcmp(words[0],"KICK")){
		if (words[1]){
			g_strup(words[1]);
			if (!strcmp(words[1],"RECV")){
				if (words[2]){
					j=atoi(words[2]);
					if (j<recv_connections_num){
						ret=g_strconcat(" kicking receiver: ",words[2],"\n",NULL);
						sock=recv_connections[j]->socket;
						remove_recv_connection (sock);
						closesocket (sock);
					}else{
						ret=g_strconcat(" no such receiver: ",words[2],"\n",NULL);
					}
				}else{
					ret=g_strdup("kick which receiver?");
				}
			} else if (!strcmp(words[1],"FEED")){
				if (words[2]){
					j=atoi(words[2]);
					if (j<feed_connections_num){
						ret=g_strconcat(" kicking feeder: ",words[2],"\n",NULL);
						sock=feed_connections[j]->socket;
						remove_feed_connection (sock);
						closesocket (sock);
					}else{
						ret=g_strconcat(" no such feeder: ",words[2],"\n",NULL);
					}
				}else{
					ret=g_strdup("kick which feeder?");
				}
			} else if (!strcmp(words[1],"CONS")){
				if (words[2]){
					j=atoi(words[2]);
					if (j<cons_connections_num){
						ret=g_strconcat(" kicking console: ",words[2],"\n",NULL);
						sock=cons_connections[j]->socket;
						remove_cons_connection (sock);
						closesocket (sock);
					}else{
						ret=g_strconcat(" no such console: ",words[2],"\n",NULL);
					}
				}else{
					ret=g_strdup("kick which console?");
				}
			} else {
				ret=g_strdup("kick what?");
			}
		}else{
			ret=g_strdup("kick what?");
		}
	} else if (!strcmp(words[0],"SET")){
		if (!words[1]){
			ret=g_strdup_printf(" max_recvs: %d\n max_feeds: %d\n x_ac_admin: %s\n x_ac_location: %s\n x_ac_server_url: %s\n",
			max_recvs,max_feeds,x_ac_admin,x_ac_location,x_ac_server_url);
		}else{
				ret=g_strdup("not implemented");
		}
	}	else {
		ret=g_strdup("unknown command");
	}
	}
	g_strfreev(words);
	g_free(w);
	return ret;
}

int main (int argc, char *argv[])
{
  struct timeval tv, et, start;
  struct in_addr in_address;
  int ret, ret2, ret3, ret4, i, j;
  int sock, tsock;
  u_char *br;
  u_char *header;
  int headerlen;
  u_long loopc;
  u_long prevsec;
//  u_char *tmpbuf = NULL;
  int feed_br;
  u_char *tmp_mount;
  int tmp_feed;
  u_int log_day, log_hour, log_min;
  u_long remote_ip;
  u_long total_connections = 0;
  u_long total_feed_connections = 0;
  u_long total_get_connections = 0;
	u_long total_cons_connections = 0;
  u_long total_bytes_received = 0;
  u_long total_bytes_sent = 0;
  u_long total_feed_bytes_received = 0;
  u_long total_feed_bytes_sent = 0;
  u_long total_cons_bytes_received = 0;
  u_long total_cons_bytes_sent = 0;
  u_char *temp_file_size;
  u_char *temp_title_port;
	int headercheck;
	gchar **commands;
	gchar *comm_response;
	gchar *tmp_command;

  version ();
  get_some_switches (argc, argv, NULL);

  temp_connections = g_malloc (sizeof (temp_connection *) * (max_recvs + max_feeds + max_conss));
  term_connections = g_malloc (sizeof (term_connection *) * (max_recvs + max_feeds + max_conss));
  feed_connections = g_malloc (sizeof (feed_connection *) * max_feeds);
  recv_connections = g_malloc (sizeof (recv_connection *) * max_recvs);
  cons_connections = g_malloc (sizeof (cons_connection *) * max_conss);

  if (NULL == x_ac_location) {
    x_ac_location = g_strdup ("default location");
  }
  if (NULL == x_ac_admin) {
    x_ac_admin = g_strdup ("admin@default");
  }
  if (NULL == x_ac_server_url) {
    x_ac_server_url = g_strdup ("http://defaul_server_url/");
  }
  if (NULL == password) {
    password = g_strdup ("letmein");
  }
  current_dir = g_get_current_dir ();
  if (NULL == status_log_name) {
    status_log_name = g_strdup ("status.log");
  }
  status_log_name2 = g_strconcat (current_dir, "/", status_log_name, NULL);
  if (NULL == system_log_name) {
    system_log_name = g_strdup ("system.log");
  }
  if (NULL == access_log_name) {
    access_log_name = g_strdup ("access.log");
  }

  null_recv_connections ();
  null_temp_connections ();
  null_feed_connections ();
  null_term_connections ();
  null_cons_connections ();

  resettimeout (&tv);

  header = NULL;

  gettimeofday (&et, NULL);
  prevsec = et.tv_sec;

  system_log = fopen (system_log_name, "a");
  access_log = fopen (access_log_name, "a");

  if ((system_log != NULL) && (access_log != NULL)) {
    mainsock = listenonport (mainport);
    if (mainsock != -1) {

#ifndef ENV_CYGWIN 
#ifndef ENV_SOLARIS
#ifndef ENV_AIX
      if (daemon_mode) {
	daemon (0, 0);
      }
#endif
#endif
#endif

			if (!daemon_mode){
				printf("%s",PROMPT);
				fflush(0);
			}
			
      response_messages = g_malloc (sizeof (gchar *) * RESPONSES_NUM);

      response_messages[RESPONSE_306] =
	g_strdup_printf
	("HTTP/1.0 306 Request messed up\r\nServer: %s/%s\r\nConnection: close\r\nContent-Type: text/html\r\n\r\n<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML//EN\">\r\n<html>\r\n<head>\r\n<title>306 Request all messed up</title>\r\n</head>\r\n<body>\r\n<h1>306 Request all messed up</h1>\r\nThe request your application sent to the server is unacceptable.\r\n</body>\r\n</html>\r\n",
	 GINI_NAME, VERSION);
      response_messages[RESPONSE_400] =
	g_strdup_printf
	("HTTP/1.0 400 Bad request\r\nServer: %s/%s\r\nConnection: close\r\nContent-Type: text/html\r\n\r\n<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML//EN\">\r\n<html>\r\n<head>\r\n<title>403 Forbidden</title>\r\n</head>\r\n<body>\r\n<h1>403 Forbidden</h1>\r\nThe request is bad.\r\n</body>\r\n</html>\r\n",
	 GINI_NAME, VERSION);
      response_messages[RESPONSE_403] =
	g_strdup_printf
	("HTTP/1.0 403 Forbidden\r\nServer: %s/%s\r\nConnection: close\r\nContent-Type: text/html\r\n\r\n<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML//EN\">\r\n<html>\r\n<head>\r\n<title>403 Forbidden</title>\r\n</head>\r\n<body>\r\n<h1>403 Forbidden</h1>\r\nYou don't have access to this entity (stream or file). Please go away\r\n</body>\r\n</html>\r\n",
	 GINI_NAME, VERSION);
      response_messages[RESPONSE_404] =
	g_strdup_printf
	("HTTP/1.0 404 Not found\r\nServer: %s/%s\r\nConnection: close\r\nContent-Type: text/html\r\n\r\n<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML//EN\">\r\n<html>\r\n<head>\r\n<title>404 Entity Not Found</title>\r\n</head>\r\n<body>\r\n<h1>404 Entity Not Found</h1>\r\nThe requested file or stream was not found on this server.\r\n</body>\r\n</html>\r\n",
	 GINI_NAME, VERSION);
      response_messages[RESPONSE_504] =
	g_strdup_printf
	("HTTP/1.0 504 Server full\r\nServer: %s/%s\r\nConnection: close\r\nContent-Type: text/html\r\n\r\n<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML//EN\">\r\n<html>\r\n<head>\r\n<title>404 Entity Not Found</title>\r\n</head>\r\n<body>\r\n<h1>504 Server full</h1>\r\nThe server is full, try again later.\r\n</body>\r\n</html>\r\n",
	 GINI_NAME, VERSION);
      response_messages[RESPONSE_BAD_PASSWORD] = g_strdup ("\r\nBAD PASSWORD\r\n");
      response_messages[RESPONSE_NO_BITRATE] = g_strdup ("\r\nNO BITRATE SPECIFIED\r\n");
      response_messages[RESPONSE_FEEDS_FULL] = g_strdup ("\r\nFEEDS FULL\r\n");
      response_messages[RESPONSE_UNKNOWN_TYPE] = g_strdup ("\r\nUNKNOWN STREAM TYPE\r\n");
      response_messages[RESPONSE_BROKEN_HEADER] = g_strdup ("\r\nBROKEN HEADER\r\n");
      response_messages[RESPONSE_KICKED] = g_strdup ("\r\nKICKED\r\n");
      response_messages[RESPONSE_CONSS_FULL] = g_strdup ("\r\nCONSOLES FULL\r\n");

      setup_signal_traps ();
      loopc = 0;
      gettimeofday (&et, NULL);
      gettimeofday (&start, NULL);
      formatlogtime ();
      while (!quit) {
	sock = acceptconnection (mainsock, &remote_ip);
	if (sock > -1) {
	  total_connections++;
	  in_address.s_addr = htonl (remote_ip);
	  if (recv_connections_num + feed_connections_num + cons_connections_num < max_recvs + max_feeds + max_conss) {
	    fprintf (system_log, "%s: new connection from: %s\n", logtime, inet_ntoa (in_address));
	    add_temp_connection (sock, remote_ip);
	  } else {
	    fprintf (system_log, "%s: connection refused: too many connections\n", logtime);
	    fprintf (system_log, "%s: terminating connection\n", logtime);
	    add_term_connection (sock, remote_ip, response_messages[RESPONSE_504]);
	  }
	}

////////////////////////////////////////////////////////////////////////////////

/* local console stuff */
	if (!daemon_mode){
      FD_CSETALL (&sockset);
      ret = select (1, &sockset, NULL, NULL, &tv);
      if (ret > 0) {
	  if (FD_ISSET (0, &sockset)) {
	    ret2 = read (0, rbuff, RBUFSIZE);
	    if (ret2 > 0) {
	      rbuff[ret2] = '\0';
				commands=g_strsplit(rbuff,"\n",0);
				i=0;
				while (commands[i]){
					comm_response=parsecommand(commands[i],-1);
					if (comm_response){
						printf("%s\n",comm_response);
						g_free(comm_response);
					}
					i++;
				}
				g_strfreev(commands);
				if (!quit){
	      	printf ("%s", PROMPT);
				}
	      fflush(0);
	    } else {
	      quit=1;
	    }
	  }
      }		
	}

////////////////////////////////////////////////////////////////////////////////
	
/* incoming unknown connections */
	FD_TSETALL (&sockset);
	resettimeout (&tv);
	ret = select (get_max_temp_socket () + 1, &sockset, NULL, NULL, BLOCK ? NULL : &tv);
	if (ret > 0) {
	  for (i = temp_connections_num - 1; i >= 0; i--) {
	    if (FD_ISSET (temp_connections[i]->socket, &sockset)) {
	      ret2 = recv (temp_connections[i]->socket, rbuff, RBUFSIZE, MSG_DONTWAIT);
	      total_bytes_received += ret2;
		  temp_connections[i]->justreaded = ret2;
	      if (temp_connections[i]->justreaded > 0) {
		rbuff[ret2] = '\0';

		temp_connections[i]->buffer =
		  g_realloc (temp_connections[i]->buffer, ret2 + temp_connections[i]->readed + 1);
		memcpy (temp_connections[i]->buffer + temp_connections[i]->readed, rbuff, ret2);
		temp_connections[i]->buffer[ret2 + temp_connections[i]->readed] = '\0';

		temp_connections[i]->readed += ret2;
		if ((strstr (temp_connections[i]->buffer, "\r\n\r\n") != NULL)
		    || (strstr (temp_connections[i]->buffer, "\n\n") != NULL)) {
		  if (!strncmp (temp_connections[i]->buffer, "SOURCE", 6)) {
		    total_feed_connections++;
		    if (feed_connections_num < max_feeds) {
		      tmp_password = get_source_password (temp_connections[i]->buffer);
		      if (tmp_password != NULL) {
			if (!strcmp (tmp_password, password)) {
			  br = getheaders ("x-audiocast-bitrate:", temp_connections[i]->buffer);
			  if (br != NULL) {
			    feed_br = atoi (br);
			    g_free (br);
			    tsock = temp_connections[i]->socket;
			    remote_ip = temp_connections[i]->remote_ip;
			    add_feed_connection (tsock, remote_ip,
						 get_source_mount (temp_connections[i]->buffer),
						 getheaders ("x-audiocast-name:",
							     temp_connections[i]->buffer),
						 getheaders ("x-audiocast-genre:",
							     temp_connections[i]->buffer),
						 getheaders ("x-audiocast-url:", temp_connections[i]->buffer),
						 getheaders ("x-audiocast-description:",
							     temp_connections[i]->buffer), feed_br);
			    remove_temp_connection (tsock);
			    fprintf (system_log, "%s: incoming feed %d kbps\n", logtime, feed_br);
			  } else {
			    tsock = temp_connections[i]->socket;
			    temp_to_term (tsock, response_messages[RESPONSE_NO_BITRATE]);
			    fprintf (system_log, "%s: incoming feed denied: no bitrate\n", logtime);
			  }
			} else {
			  tsock = temp_connections[i]->socket;
			  temp_to_term (tsock, response_messages[RESPONSE_BAD_PASSWORD]);
			  fprintf (system_log, "%s: incoming feed denied: incorrect password\n", logtime);
			}
			g_free (tmp_password);
		      } else {
			tsock = temp_connections[i]->socket;
			temp_to_term (tsock, response_messages[RESPONSE_BAD_PASSWORD]);
			fprintf (system_log, "%s: incoming feed denied: no password\n", logtime);
		      }
		    } else {
		      tsock = temp_connections[i]->socket;
		      temp_to_term (tsock, response_messages[RESPONSE_FEEDS_FULL]);
		      fprintf (system_log, "%s: incoming feed denied: feeds full\n", logtime);
		    }
		  } else if (!strncmp (temp_connections[i]->buffer, "GET /admin.cgi", 14)) {
		    if (count_valid_feed_connections ()) {
		      tmp_mount = urldecode (getparameter (temp_connections[i]->buffer, "mount="));
		      if (tmp_mount != NULL) {
			tmp_password = getparameter (temp_connections[i]->buffer, "pass=");
			if (tmp_password != NULL) {
			  if (!strcmp (tmp_password, password)) {
			    tmp_feed = find_mount (tmp_mount, fallback);
			    fprintf (system_log, "%s: incoming song info for: %s\n", logtime, tmp_mount);
			    if (tmp_feed != -1) {
			      if (feed_connections[tmp_feed]->remote_ip == temp_connections[i]->remote_ip) {
				if (feed_connections[tmp_feed]->title_string != NULL) {
				  g_free (feed_connections[tmp_feed]->title_string);
				}
				feed_connections[tmp_feed]->title_string =
				  urldecode (getparameter (temp_connections[i]->buffer, "song="));
				temp_file_size = getparameter (temp_connections[i]->buffer, "length=");
				if (temp_file_size != NULL) {
				  feed_connections[tmp_feed]->file_size = atol (temp_file_size);
				  g_free (temp_file_size);
				}
				feed_connections[tmp_feed]->udpseq++;
//                                                                      printf("%s / %lu\n",feed_connections[tmp_feed]->title_string,feed_connections[tmp_feed]->file_size);
				tsock = temp_connections[i]->socket;
				remove_temp_connection (tsock);
			      } else {
				tsock = temp_connections[i]->socket;
				remove_temp_connection (tsock);
				closesocket (tsock);
				fprintf (system_log,
					 "%s: incoming song info denied: different source ip\n", logtime);
			      }
			    } else {
			      tsock = temp_connections[i]->socket;
			      remove_temp_connection (tsock);
			      closesocket (tsock);
			      fprintf (system_log,
				       "%s: incoming song info denied: nonexistent mount point\n", logtime);
			    }
			  } else {
			    tsock = temp_connections[i]->socket;
			    remove_temp_connection (tsock);
			    closesocket (tsock);
			    fprintf (system_log,
				     "%s: incoming song info denied: invalid password\n", logtime);
			  }
			  g_free (tmp_password);
			} else {
			  tsock = temp_connections[i]->socket;
			  remove_temp_connection (tsock);
			  closesocket (tsock);
			  fprintf (system_log,
				   "%s: incoming song info denied: no password specified\n", logtime);
			}
			g_free (tmp_mount);
		      } else {
			
tsock = temp_connections[i]->socket;
			remove_temp_connection (tsock);
			closesocket (tsock);
			fprintf (system_log, "%s: incoming song info denied: no mount specified\n", logtime);
		      }
		    } else {
		      tsock = temp_connections[i]->socket;
		      remove_temp_connection (tsock);
		      closesocket (tsock);
		      fprintf (system_log, "%s: incoming song info denied: no feed\n", logtime);
		    }
		  } else if (!strncmp (temp_connections[i]->buffer, "GET", 3)) {
		    total_get_connections++;
		    if (recv_connections_num < max_recvs) {
		      if (count_valid_feed_connections ()) {
			tmp_mount = get_request_mount (temp_connections[i]->buffer);
			if (tmp_mount != NULL) {
			  tmp_feed = find_mount (tmp_mount, fallback);
			  fprintf (system_log, "%s: incoming stream request for: %s\n", logtime, tmp_mount);
			  g_free (tmp_mount);
			  if (tmp_feed != -1) {
			    tsock = temp_connections[i]->socket;
			    remote_ip = temp_connections[i]->remote_ip;
			    temp_title_port =
			      getheaders ("x-audiocast-udpport:", temp_connections[i]->buffer);
			    add_recv_connection (tsock, remote_ip, tmp_feed,
						 get_request_request (temp_connections[i]->buffer),
						 getheaders ("Referer:", temp_connections[i]->buffer),
						 getheaders ("User-Agent:", temp_connections[i]->buffer),
						 NULL == temp_title_port ? 0 : atoi (temp_title_port));
			    g_free (temp_title_port);
			    remove_temp_connection (tsock);
			  } else {
			    tsock = temp_connections[i]->socket;
			    temp_to_term (tsock, response_messages[RESPONSE_404]);
			    fprintf (system_log,
				     "%s: incoming stream request denied: nonexistent mount point\n",
				     logtime);
			  }
			} else {
			  tsock = temp_connections[i]->socket;
			  temp_to_term (tsock, response_messages[RESPONSE_306]);
			  fprintf (system_log, "%s: incoming stream request denied: no mount specified\n",
				   logtime);
			}
		      } else {
			tsock = temp_connections[i]->socket;
			temp_to_term (tsock, response_messages[RESPONSE_404]);
			fprintf (system_log, "%s: incoming stream request denied: no feed\n", logtime);
		      }
		    } else {
		      tsock = temp_connections[i]->socket;
		      temp_to_term (tsock, response_messages[RESPONSE_504]);
		      fprintf (system_log, "%s: incoming stream request denied: max clients reached\n",
			       logtime);
		    }
		  } else if(!strncmp (temp_connections[i]->buffer, "CONSOLE", 7)) {
				total_cons_connections++;
		    if (cons_connections_num < max_conss) {
		      tmp_password = get_cons_password (temp_connections[i]->buffer);
		      if (tmp_password != NULL) {
						if (!strcmp (tmp_password, password)) {
							tsock = temp_connections[i]->socket;
							remote_ip = temp_connections[i]->remote_ip;
							add_cons_connection (tsock, remote_ip);
							remove_temp_connection (tsock);
							fprintf (system_log, "%s: incoming console\n", logtime);
						}else{
							tsock = temp_connections[i]->socket;
							temp_to_term (tsock, response_messages[RESPONSE_BAD_PASSWORD]);
							fprintf (system_log, "%s: incoming console denied: invalid password\n", logtime);
						}
					}else{
						tsock = temp_connections[i]->socket;
						temp_to_term (tsock, response_messages[RESPONSE_BAD_PASSWORD]);
						fprintf (system_log, "%s: incoming console denied: no password\n", logtime);
					}
				}else{
					tsock = temp_connections[i]->socket;
					temp_to_term (tsock, response_messages[RESPONSE_FEEDS_FULL]);
					fprintf (system_log, "%s: incoming console denied: consoles full\n", logtime);
				}
			
			}	else {
		    tsock = temp_connections[i]->socket;
		    temp_to_term (tsock, response_messages[RESPONSE_306]);
		    fprintf (system_log, "%s: unknown request denied\n", logtime);
		  }
//              g_free (tmpbuf);
		}
	      } else {
		tsock = temp_connections[i]->socket;
		remove_temp_connection (tsock);
		closesocket (tsock);
		fprintf (system_log, "%s: remote end closed before identification\n", logtime);
	      }
	    }
	  }
	}

////////////////////////////////////////////////////////////////////////////////
		
	if (cons_connections_num) {
/* remote console stuff */
	  FD_CCSETALL (&sockset);
	  FD_CCW1SETALL (&socksetw);
	  resettimeout (&tv);
	  ret = select (get_max_cons_socket () + 1, &sockset, &socksetw, NULL, BLOCK ? NULL : &tv);
	  if (ret > 0) {
	    for (i = cons_connections_num - 1; i >= 0; i--) {
	      if (1 == cons_connections[i]->init_state) {
		if (FD_ISSET (cons_connections[i]->socket, &socksetw)) {
		  ret4 = send (cons_connections[i]->socket, "OK\r\n\r\n", 6, MSG_DONTWAIT);
		  total_bytes_sent += ret4;
		  cons_connections[i]->justwritten = ret4;
		}
		if (cons_connections[i]->justwritten > 0) {
		  set_cons_connection_state (cons_connections[i]->socket, 2);
		} else {
		  tsock = cons_connections[i]->socket;
		  closesocket (tsock);
		  remove_cons_connection (tsock);
		  fprintf (system_log, "%s: remote console closed\n", logtime);
		}
	      } else if (2==cons_connections[i]->init_state) {
		if (FD_ISSET (cons_connections[i]->socket, &sockset)) {
//					printf("ovasnank\n");
					fflush(0);
		  ret4 =
		    recv (cons_connections[i]->socket, cons_connections[i]->rbuffer,
			  RBUFSIZE, MSG_DONTWAIT);
		  total_bytes_received += ret4;
		  total_cons_bytes_received += ret4;
		  cons_connections[i]->justreaded = ret4;
			this_sec_readed += ret4;

		  if (cons_connections[i]->justreaded > 0) {
				cons_connections[i]->rbuffer[cons_connections[i]->justreaded]='\0';
		    cons_connections[i]->readed = ret4;
				commands=g_strsplit(cons_connections[i]->rbuffer,"\n",0);
				cons_connections[i]->sec_last = allseconds;
//				printf("\n'%s'\n",cons_connections[i]->rbuffer);
				j=0;
				while (commands[j]){
					comm_response=parsecommand(commands[j],cons_connections[i]->socket);
					if (comm_response){
						tmp_command=cons_connections[i]->sbuffer;
						cons_connections[i]->sbuffer=g_strconcat(cons_connections[i]->sbuffer,comm_response,"\n",NULL);
						g_free(comm_response);
						g_free(tmp_command);
					}else{
						tsock = cons_connections[i]->socket;
						closesocket (tsock);
						remove_cons_connection (tsock);
						fprintf (system_log, "%s: remote console closing\n", logtime);
						break;
					}
//					printf("\n%s\n",commands[j]);
					fflush(0);
					j++;
				}
				g_strfreev(commands);
		  } else {
		    tsock = cons_connections[i]->socket;
		    closesocket (tsock);
		    remove_cons_connection (tsock);
		    fprintf (system_log, "%s: remote console closed\n", logtime);
		  }
		}
		}
	    }
	  }
	}

	if (cons_connections_num) {
/* remote console stuff */
	  FD_CCW2SETALL (&socksetw);
	  resettimeout (&tv);
	  ret = select (get_max_cons_socket () + 1, NULL, &socksetw, NULL, BLOCK ? NULL : &tv);
	  if (ret > 0) {
		fflush(0);
	    for (i = cons_connections_num - 1; i >= 0; i--) {
	      if (2==cons_connections[i]->init_state) {
		
		if (FD_ISSET (cons_connections[i]->socket, &socksetw)) {
		  ret4 = 
		    send (cons_connections[i]->socket, cons_connections[i]->sbuffer,
			  strlen(cons_connections[i]->sbuffer), MSG_DONTWAIT);
		  total_bytes_sent += ret4;
		  total_cons_bytes_sent += ret4;
		  cons_connections[i]->justwritten = ret4;
			this_sec_sent += ret4;

		  if (cons_connections[i]->justwritten > 0) {
		    cons_connections[i]->written = ret4;
				memmove(cons_connections[i]->sbuffer,cons_connections[i]->sbuffer+cons_connections[i]->justwritten,strlen(cons_connections[i]->sbuffer)-cons_connections[i]->justwritten+1);
		  } else {
		    tsock = cons_connections[i]->socket;
		    closesocket (tsock);
		    remove_cons_connection (tsock);
		    fprintf (system_log, "%s: remote console closed\n", logtime);
		  }
		}
}
	    }
	  }
	}
	
////////////////////////////////////////////////////////////////////////////////

	if (count_feed_connections ()) {
/* incoming feed */
	  FD_FSETALL (&sockset);
	  FD_FSETALL (&socksetw);
	  resettimeout (&tv);
	  ret = select (get_max_feed_socket () + 1, &sockset, &socksetw, NULL, BLOCK ? NULL : &tv);
	  if (ret > 0) {
	    for (i = feed_connections_num - 1; i >= 0; i--) {
	      if (1 == feed_connections[i]->init_state) {
		if (FD_ISSET (feed_connections[i]->socket, &socksetw)) {
		  ret4 = send (feed_connections[i]->socket, "OK\r\n\r\n", 6, MSG_DONTWAIT);
		  total_bytes_sent += ret4;
		  feed_connections[i]->justwritten = ret4;
		}
		if (feed_connections[i]->justwritten > 0) {
		  set_feed_connection_state (feed_connections[i]->socket, 2);
		} else {
		  tsock = feed_connections[i]->socket;
		  closesocket (tsock);
		  remove_feed_connection (tsock);
		  fprintf (system_log, "%s: remote feeder closed\n", logtime);
		}
	      } else {
		if (FD_ISSET (feed_connections[i]->socket, &sockset)) {
		  ret4 =
		    recv (feed_connections[i]->socket, feed_connections[i]->buffer,
			  bufsize_from_bitrate (feed_connections[i]->bitrate), MSG_DONTWAIT);
		  total_bytes_received += ret4;
		  total_feed_bytes_received += ret4;
		  feed_connections[i]->justreaded = ret4;

		  if (feed_connections[i]->justreaded > 0) {
		    feed_connections[i]->readed = ret4;
//                                      printf("\b");
//                                      printf(".");
//                                      fflush(0);
		    if (0 == feed_connections[i]->packets) {
		      if (!memcmp ("OggS", feed_connections[i]->buffer, 4)) {
//                  if (memfind ("OggS", feed_buf[i], feed_readed[i], 4)) {
			set_feed_connection_type (feed_connections[i]->socket, STREAM_OGG);
			fprintf (system_log, "%s: incoming stream is %s\n", logtime,
				 stream_types[STREAM_OGG]);
		      
} else if ((!memcmp ("ID3", feed_connections[i]->buffer, 3))
				  || ((255 == feed_connections[i]->buffer[0])
				      && ((feed_connections[i]->buffer[1] & 0xE0) == 0xE0))) {
			set_feed_connection_type (feed_connections[i]->socket, STREAM_MP3);
			fprintf (system_log, "%s: incoming stream is %s\n", logtime,
				 stream_types[STREAM_MP3]);
		      } else if ((!memcmp ("RIFF", feed_connections[i]->buffer, 4))&&(!memcmp ("AVI ", feed_connections[i]->buffer+8, 4))) {
			set_feed_connection_type (feed_connections[i]->socket, STREAM_AVI);
			fprintf (system_log, "%s: incoming stream is %s\n", logtime,
				 stream_types[STREAM_AVI]);
		      } else if (!memcmp ("moov", feed_connections[i]->buffer+4, 4)) {
			set_feed_connection_type (feed_connections[i]->socket, STREAM_MOV);
			fprintf (system_log, "%s: incoming stream is %s\n", logtime,
				 stream_types[STREAM_MOV]);
		      
			} else if (!memcmp ("\x30\x26\xB2\x75\x8E\x66\xCF\x11\xA6\xD9\x00\xAA\x00\x62\xCE\x6C", feed_connections[i]->buffer, 16)) {
			set_feed_connection_type (feed_connections[i]->socket, STREAM_ASF);
			fprintf (system_log, "%s: incoming stream is %s\n", logtime,
				 stream_types[STREAM_ASF]);
		      
			} else if (!memcmp (".RMF", feed_connections[i]->buffer, 4)) {
			set_feed_connection_type (feed_connections[i]->socket, STREAM_REAL);
			fprintf (system_log, "%s: incoming stream is %s\n", logtime,
				 stream_types[STREAM_REAL]);
		      
			} else {
/*			set_feed_connection_type (feed_connections[i]->socket, STREAM_UNKNOWN);
			fprintf (system_log, "%s: incoming stream is %s\n", logtime,
				 stream_types[STREAM_UNKNOWN]);*/
			set_feed_connection_type (feed_connections[i]->socket, STREAM_MP3);
			fprintf (system_log, "%s: incoming stream is %s, treating as %s\n", logtime,
				 stream_types[STREAM_UNKNOWN],stream_types[STREAM_MP3]);
		      }

		    }
		    if (STREAM_UNKNOWN == feed_connections[i]->stream_type) {
		      tsock = feed_connections[i]->socket;
		      feed_to_term (tsock, response_messages[RESPONSE_UNKNOWN_TYPE]);
		      fprintf (system_log, "%s: incoming feed denied: unknown type\n", logtime);
		    } else {
					headercheck=1;
		      if ((STREAM_OGG == feed_connections[i]->stream_type)
			  && (!memcmp ("OggS", feed_connections[i]->buffer, 4))) {
			headercheck=check_ogg_header (feed_connections[i]->socket);
		      } else if (STREAM_AVI == feed_connections[i]->stream_type) {
			if ((!memcmp ("RIFF", feed_connections[i]->buffer, 4))&&(!memcmp ("AVI ", feed_connections[i]->buffer+8, 4))) {
			  headercheck=check_avi_header (feed_connections[i]->socket);
			}
			check_avi_frame (feed_connections[i]->socket);
		      } else if (STREAM_MOV == feed_connections[i]->stream_type) {
			if (!memcmp ("moov", feed_connections[i]->buffer+4, 4)) {
			  headercheck=check_mov_header (feed_connections[i]->socket);
			}
//			check_mov_frame (feed_connections[i]->socket);
		      } else if (STREAM_ASF == feed_connections[i]->stream_type) {
			if (!memcmp ("\x30\x26\xB2\x75\x8E\x66\xCF\x11\xA6\xD9\x00\xAA\x00\x62\xCE\x6C", feed_connections[i]->buffer, 16)) {
			  headercheck=check_asf_header (feed_connections[i]->socket);
			}
			check_asf_frame (feed_connections[i]->socket);
		      } else if (STREAM_REAL == feed_connections[i]->stream_type) {
			if (!memcmp (".RMF", feed_connections[i]->buffer, 4)) {
			  headercheck=check_real_header (feed_connections[i]->socket);
			}
			check_real_frame (feed_connections[i]->socket);
		      }
					if ((headercheck)||(feed_connections[i]->media_header_len)){
						if (!headercheck){
		      		fprintf (system_log, "%s: broken header received on %s, using previous header\n", logtime,feed_connections[i]->mount);
						}
		      for (j = 0; j < recv_connections_num; j++) {
			if (i == recv_connections[j]->feed_conn) {
			  reset_recv_connection_written (j);
			}
		      }
		      feed_connections[i]->packets++;
		      feed_connections[i]->bytes += ret4;
		      this_sec_readed += ret4;
				}else{
		      tsock = feed_connections[i]->socket;
		      feed_to_term (tsock, response_messages[RESPONSE_BROKEN_HEADER]);
		      fprintf (system_log, "%s: incoming feed dropped: broken header\n", logtime);
				}
		    }
		  } else {
		    tsock = feed_connections[i]->socket;
		    closesocket (tsock);
		    remove_feed_connection (tsock);
		    fprintf (system_log, "%s: remote feeder closed\n", logtime);
		  }
		}
	      
}
	    }
	  } else {
	    reset_feed_connections_readed ();
	  }

////////////////////////////////////////////////////////////////////////////////

	  if (any_feed_connections_readed ()) {
/* outgoing streams */
	    FD_SETALL (&socksetw);
	    resettimeout (&tv);
	    ret = select (get_max_recv_socket () + 1, NULL, &socksetw, NULL, BLOCK ? NULL : &tv);
	    if (ret > 0) {
	      for (i = recv_connections_num - 1; i >= 0; i--) {
		if (FD_ISSET (recv_connections[i]->socket, &socksetw)) {
		  if (recv_connections[i]->init_state) {
//                              readed=BUFSIZE;
		    /* avi frame sync */
		    if ((0 == recv_connections[i]->packets)
			&& (feed_connections[recv_connections[i]->feed_conn]->skipfirst)) {
		      recv_connections[i]->written =
			feed_connections[recv_connections[i]->feed_conn]->skipfirst;
		    }
		    if (recv_connections[i]->written <
			feed_connections[recv_connections[i]->feed_conn]->readed) {
		      ret3 =
			send (recv_connections[i]->socket,
			      feed_connections[recv_connections[i]->feed_conn]->buffer +
			      recv_connections[i]->written,
			      feed_connections[recv_connections[i]->feed_conn]->readed -
			      recv_connections[i]->written, MSG_DONTWAIT);
		      total_bytes_sent += ret3;
		      total_feed_bytes_sent += ret3;
		      recv_connections[i]->justwritten = ret3;
		      recv_connections[i]->written += ret3;
		      recv_connections[i]->allwritten += ret3;
		      recv_connections[i]->packets++;
		      this_sec_sent += ret3;
		    }
		  } else {
		    if (header != NULL) {
		      g_free (header);
		    }
		    if (STREAM_OGG == feed_connections[recv_connections[i]->feed_conn]->stream_type) {
		      headerlen =
			strlen (feed_connections[recv_connections[i]->feed_conn]->header) +
			feed_connections[recv_connections[i]->feed_conn]->media_header_len;
		      header = g_malloc (headerlen);
		      header =
			strncpy (header, feed_connections[recv_connections[i]->feed_conn]->header,
				 strlen (feed_connections[recv_connections[i]->feed_conn]->header));
		      memcpy (header + strlen (feed_connections[recv_connections[i]->feed_conn]->header),
			      feed_connections[recv_connections[i]->feed_conn]->media_header,
			      feed_connections[recv_connections[i]->feed_conn]->media_header_len);
		    } else if (STREAM_AVI == feed_connections[recv_connections[i]->feed_conn]->stream_type) {
		      headerlen =
			strlen (feed_connections[recv_connections[i]->feed_conn]->header) +
			feed_connections[recv_connections[i]->feed_conn]->media_header_len;
		      header = g_malloc (headerlen);
		      header =
			strncpy (header, feed_connections[recv_connections[i]->feed_conn]->header,
				 strlen (feed_connections[recv_connections[i]->feed_conn]->header));
		      memcpy (header + strlen (feed_connections[recv_connections[i]->feed_conn]->header),
			      feed_connections[recv_connections[i]->feed_conn]->media_header,
			      feed_connections[recv_connections[i]->feed_conn]->media_header_len);
		    } else if (STREAM_MOV == feed_connections[recv_connections[i]->feed_conn]->stream_type) {
		      headerlen =
			strlen (feed_connections[recv_connections[i]->feed_conn]->header) +
			feed_connections[recv_connections[i]->feed_conn]->media_header_len;
		      header = g_malloc (headerlen);
		      header =
			strncpy (header, feed_connections[recv_connections[i]->feed_conn]->header,
				 strlen (feed_connections[recv_connections[i]->feed_conn]->header));
		      memcpy (header + strlen (feed_connections[recv_connections[i]->feed_conn]->header),
			      feed_connections[recv_connections[i]->feed_conn]->media_header,
			      feed_connections[recv_connections[i]->feed_conn]->media_header_len);
		    } else if (STREAM_ASF == feed_connections[recv_connections[i]->feed_conn]->stream_type) {
		      headerlen =
			strlen (feed_connections[recv_connections[i]->feed_conn]->header) +
			feed_connections[recv_connections[i]->feed_conn]->media_header_len;
		      header = g_malloc (headerlen);
		      header =
			strncpy (header, feed_connections[recv_connections[i]->feed_conn]->header,
				 strlen (feed_connections[recv_connections[i]->feed_conn]->header));
		      memcpy (header + strlen (feed_connections[recv_connections[i]->feed_conn]->header),
			      feed_connections[recv_connections[i]->feed_conn]->media_header,
			      feed_connections[recv_connections[i]->feed_conn]->media_header_len);
		    } else if (STREAM_REAL == feed_connections[recv_connections[i]->feed_conn]->stream_type) {
		      headerlen =
			strlen (feed_connections[recv_connections[i]->feed_conn]->header) +
			feed_connections[recv_connections[i]->feed_conn]->media_header_len;
		      header = g_malloc (headerlen);
		      header =
			strncpy (header, feed_connections[recv_connections[i]->feed_conn]->header,
				 strlen (feed_connections[recv_connections[i]->feed_conn]->header));
		      memcpy (header + strlen (feed_connections[recv_connections[i]->feed_conn]->header),
			      feed_connections[recv_connections[i]->feed_conn]->media_header,
			      feed_connections[recv_connections[i]->feed_conn]->media_header_len);
		    } else {
		      header = g_strdup (feed_connections[recv_connections[i]->feed_conn]->header);
		      headerlen = strlen (header);
		    }
		    ret3 =
		      send (recv_connections[i]->socket, header + recv_connections[i]->written,
			    headerlen - recv_connections[i]->written, MSG_DONTWAIT);
		    total_bytes_sent += ret3;
		    recv_connections[i]->justwritten = ret3;
		    recv_connections[i]->written += ret3;
		    recv_connections[i]->allwritten += ret3;
		    this_sec_sent += ret3;
		    g_free (header);
		    header = NULL;
		  }
		  if (recv_connections[i]->justwritten > 0) {
		    if (!recv_connections[i]->init_state) {
		      if (recv_connections[i]->written == headerlen) {
			set_recv_connection_state (recv_connections[i]->socket, 1);
			recv_connections[i]->written = 0;
		      }
		    }
		  } else {
		    tsock = recv_connections[i]->socket;
		    remove_recv_connection (tsock);
		    closesocket (tsock);
		    fprintf (system_log, "%s: remote receiver closed\n", logtime);
		  }
		}
	      }
	    }
	  }
	}
	
////////////////////////////////////////////////////////////////////////////////

	if (0 < term_connections_num) {
/* term connections */
	  FD_RSETALL (&socksetw);
	  resettimeout (&tv);
	  ret = select (get_max_term_socket () + 1, NULL, &socksetw, NULL, BLOCK ? NULL : &tv);
	  if (ret > 0) {
	    for (i = term_connections_num - 1; i >= 0; i--) {
	      if (FD_ISSET (term_connections[i]->socket, &socksetw)) {
		if (term_connections[i]->written < strlen (term_connections[i]->response_buffer)) {
		  ret3 =
		    send (term_connections[i]->socket,
			  term_connections[i]->response_buffer + term_connections[i]->written,
			  strlen (term_connections[i]->response_buffer) - term_connections[i]->written,
			  MSG_DONTWAIT);
		  total_bytes_sent += ret3;
		  term_connections[i]->written += ret3;
		  term_connections[i]->justwritten = ret3;
		  if (0 < term_connections[i]->justwritten) {
		    if (term_connections[i]->written >= strlen (term_connections[i]->response_buffer)) {
		      tsock = term_connections[i]->socket;
		      remove_term_connection (tsock);
		      closesocket (tsock);
		      fprintf (system_log, "%s: closing term socket\n", logtime);
		    }
		  } else {
		    tsock = term_connections[i]->socket;
		    remove_term_connection (tsock);
		    closesocket (tsock);
		    fprintf (system_log, "%s: remote socket closed\n", logtime);
		  }
		}
	      }
	    }
	  }
	}

////////////////////////////////////////////////////////////////////////////////

	usleep (2);
	loopc++;
	gettimeofday (&et, NULL);
	if (prevsec < et.tv_sec) {
	  allseconds += et.tv_sec - prevsec;
	  prevsec = et.tv_sec;
//print out status
	  log_day = allseconds / 60 / 60 / 24;
	  log_hour = (allseconds - (log_day * 60 * 60 * 24)) / 60 / 60;
	  log_min = (allseconds - (log_day * 60 * 60 * 24) - (log_hour * 60 * 60)) / 60;
	  if ((status_log = fopen (status_log_name2, "w+"))) {
	    fprintf (status_log, "Uptime: %d days, %d hours, %d minutes and %lu seconds\n", log_day, log_hour,
		     log_min, allseconds % 60);
	    fprintf (status_log,
		     "Connections: all: %lu, feed: %lu, recv: %lu; Active: temp: %d, feed: %d, recv %d\n",
		     total_connections, total_feed_connections, total_get_connections, temp_connections_num,
		     feed_connections_num, recv_connections_num);
	    fprintf (status_log, "Data: in all: %lu K, in feed: %lu K, out all: %lu K, out stream: %lu K\n",
		     total_bytes_received / 1024, total_feed_bytes_received / 1024, total_bytes_sent / 1024,
		     total_feed_bytes_sent / 1024);
	    fprintf (status_log, "Bandwidth: in: %lu kbps out: %lu kbps\n",
		     this_sec_readed * 8 / 1024, this_sec_sent * 8 / 1024);
	    fprintf (status_log, "Feeds:\n");
	    for (i = 0; i < feed_connections_num; i++) {
	      in_address.s_addr = htonl (feed_connections[i]->remote_ip);
	      fprintf (status_log, "mount: %s, bitrate: %d kbps, type: %s, ip: %s, readed: %lu K\n",
		       feed_connections[i]->mount, feed_connections[i]->bitrate,
		       stream_types[feed_connections[i]->stream_type], inet_ntoa (in_address),
		       feed_connections[i]->bytes / 1024);
	    }
	    fprintf (status_log, "Admins:\n");
	    for (i = 0; i < cons_connections_num; i++) {
	      in_address.s_addr = htonl (cons_connections[i]->remote_ip);
				log_day =  (allseconds-cons_connections[i]->sec_last) / 60 / 60 / 24;
				log_hour = ((allseconds-cons_connections[i]->sec_last) - (log_day * 60 * 60 * 24)) / 60 / 60;
				log_min = ((allseconds-cons_connections[i]->sec_last) - (log_day * 60 * 60 * 24) - (log_hour * 60 * 60)) / 60;
	      fprintf (status_log, "ip: %s, idle: %02d:%02d:%02lu, %dd\n",
		       inet_ntoa (in_address),log_hour,log_min,(allseconds-cons_connections[i]->sec_last)%60,log_day);
	    }
	    fprintf (status_log, "Clients:\n");
	    for (i = 0; i < recv_connections_num; i++) {
	      in_address.s_addr = htonl (recv_connections[i]->remote_ip);
	      fprintf (status_log, "ip: %s, mount: %s, player: %s, written: %lu K\n",
		       inet_ntoa (in_address), feed_connections[recv_connections[i]->feed_conn]->mount,
		       recv_connections[i]->useragent, recv_connections[i]->allwritten / 1024);
	    }
//                      fflush (status_log);
	    fclose (status_log);
	  }
	  fflush (system_log);
	  fflush (access_log);
	  formatlogtime ();
	  this_sec_readed = 0;
	  this_sec_sent = 0;

	  send_title_infos ();
	}
      }
      closeall ();
    } else {
      printf ("cannot listen!\n");
    }
  } else {
    printf ("cannot open one or more of the logfiles!\n");
  }
  return 0;
}
