/*
  gdbm-create.c
  
  Jonathan B. Leffert <jonathan@leffert.net>
  21 April 2000
  
  $Id: gdbm-create.c,v 1.23 2000/05/17 18:44:52 jonathan Exp $
  
  Copyright (c) 2000 Jonathan B. Leffert <jonathan@leffert.net>
  All rights reserved.
*/

#include "common.h"

/* prototypes */
GDBM_FILE create_gdbm_database(char *);
void parse_args(int *, char ***, char **);
void usage(char *);

/*
  function defs
*/

/*
  creates a GDBM database, but only if it doesn't already
  exist
*/
GDBM_FILE create_gdbm_database(char *name)
{
  return open_gdbm_database(name, GDBM_WRCREAT);
}

void parse_args(int *argc, char ***argv, char **dbName)
{
  char *options = "d:hv";
  int c;

  while ( (c = getopt(*argc, *argv, options)) != EOF )
    {
      switch (c)
	{
	case 'd':
	  /* database name */
	  *dbName = optarg;
	  break;
	case 'h':
	  /* request help */
	  usage((*argv)[0]);
	  exit(0);
	  break;
	case 'v':
	  /* version info */
	  printf("Version %f\n", VERSION);
	  exit(0);
	  break;
	case '?':
	  /* unknown option */
	  usage((*argv)[0]);
	  exit(1);
	  break;
	default:
	  /* default: do nothing */
	  break;
	}
    }
}

void usage(char *program_name)
{
  printf("Usage:\n");
  printf("\n");
  printf("%s [-hv] [-d DB]\n", program_name);
  printf("\n");
  printf("-d DB = Create database with specified name.\n");
  printf("        Must be supplied.\n");
  printf("-h    = This help message.\n");
  printf("-v    = Prints version information.\n");
  printf("\n");
  printf("This program creates the given database.\n");
  printf("\n");
  printf("Error Codes:\n");
  printf("  0 on success\n");
  printf("  1 on non-database failure\n");
  printf("  2 on database failure (e.g., database already exists)\n");
}

/*
  main
*/

int main(int argc, char **argv)
{
  char *name = NULL;
  GDBM_FILE db = NULL;
  int returnValue = 0;

  parse_args(&argc, &argv, &name);

  if ( name != NULL )
    {
      /* first try to open the database */
      db = open_gdbm_database_ro(name);
      if ( db != NULL )
	{
	  /* open suceeded, database already exists */
	  returnValue = 2;
	  /* close the database */
	  gdbm_close(db);
	}
      else
	{
	  db = create_gdbm_database(name);
	  
	  if ( db == NULL )
	    {
	      /* error occured */
	      perror(argv[0]);
	      exit(2);
	    }
	  else
	    {
	      /* close the database */
	      gdbm_close(db);
	    }
	}
    }
  else
    {
      /* happens if no -d flag given */
      usage(argv[0]);
      exit(1);
    }
  
  return returnValue;
}
