
/* Copyright (C) 2005-2008, Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/

#include <unistd.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <string.h>
#include <fcntl.h>
#include <limits.h>


#include "runtime.h"


#if HAVE_ERRNO_H
#include <errno.h>
#else
extern int errno;    /* In the old days, this was enough... */
#endif





static char yes[] = "YES", no[] = "NO", unknown[] = "UNKNOWN";



/* standard_desc()-- Return nonzero if the descriptor is a standard
 * descriptor inherited from the operating system. */

int standard_desc(OS_HANDLE fd) {

    return (fd == STDIN_FILENO || fd == STDOUT_FILENO || fd == STDERR_FILENO);
}



/* fd_length()-- Given a file descriptor, return the length of the
 * file.  This is zero for files that aren't seekable. */

off_t fd_length(int fd) {
struct stat statbuf;

   fstat(fd, &statbuf);
   return statbuf.st_size;
}



/* filename_length()-- Return the length of a file for a particular
 * filename. */

off_t filename_length(void) {
char path[PATH_LENGTH+1];
struct stat statbuf;

    if (unpack_filename(path, ioparm->file, ioparm->file_len))
	return 0;

    if (stat(path, &statbuf) < 0)
	return 0;

    return statbuf.st_size;
}



/* get_oserror()-- Get the most recent operating system error.  For
 * unix, this is determined by errno. */

char *get_oserror(void) {

    return strerror(errno);
}



/* get_oserrno()-- Get the operating system error number */

int get_oserrno(void) {

    return errno;
}



/* fix_fd()-- Given a file descriptor, make sure it is not one of the
 * standard descriptors, returning a non-standard descriptor.  If the
 * user specifies that system errors should go to standard output,
 * then closes standard output, we don't want the system errors to a
 * file that has been given file descriptor 1 or 0.  We want to send
 * the error to the invalid descriptor. */

static int fix_fd(int fd) {
int input, output, error;

    input = output = error = 0;

/* Unix allocates the lowest descriptors first, so a loop is not
 * required, but this order is. */

    if (fd == STDIN_FILENO)  { fd = dup(fd); input  = 1; }
    if (fd == STDOUT_FILENO) { fd = dup(fd); output = 1; }
    if (fd == STDERR_FILENO) { fd = dup(fd); error  = 1; }

    if (input)  close(STDIN_FILENO);
    if (output) close(STDOUT_FILENO);
    if (error)  close(STDERR_FILENO);

    return fd;
}


/* open_file()-- Open a file.  Returns zero and stores the descriptor
 * if everything goes OK. */

int open_file(unit_action action, unit_status status, OS_HANDLE *result,
	      char *path) { 
struct stat statbuf;
int mode, fd;

    if (unpack_filename(path, ioparm->file, ioparm->file_len)) {
	errno = ENOENT;   /* Fake an OS error */
	return -1;
    }

    mode = O_CREAT;

    switch(action) {
    case ACTION_READ:
	mode |= O_RDONLY;
	break;

    case ACTION_WRITE:
	mode |= O_WRONLY;
	break;

    case ACTION_READWRITE:
	mode |= O_RDWR;
	break;

    default:
	internal_error("open_file(): Bad action");
    }

    switch(status) {
    case STATUS_NEW:
	mode |= O_EXCL;
	break;

    case STATUS_OLD:  /* file must exist, so check for its existence */
	if (stat(path, &statbuf) < 0)
	    return -1;
	break;

    case STATUS_UNKNOWN:
    case STATUS_SCRATCH:
	mode |= O_CREAT;
	break;

    case STATUS_REPLACE:
	mode |= O_TRUNC;
	break;

    default:
	internal_error("open_file(): Bad status");
    }

    fd = open(path, mode,
	      S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH);

    if (fd >= 0)
	*result = fix_fd(fd);

    return fd < 0;
}



/* open_tempfile()-- Generate a temporary filename for a scratch file
 * and open it.  mkstemp() opens the file for reading and writing, but
 * the library mode prevents anything that is not allowed.  The
 * template is pointed to by ioparm->file, which is copied into the
 * unit structure and freed later. */

int open_tempfile(char *path, OS_HANDLE *result) {
char *tempdir;
int fd;

    tempdir = getenv("G95_TMPDIR");
    if (tempdir == NULL || tempdir[0] == '\0') tempdir = getenv("TEMP");
    if (tempdir == NULL || tempdir[0] == '\0') tempdir = getenv("TMP");
    if (tempdir == NULL || tempdir[0] == '\0') tempdir = DEFAULT_TEMPDIR;

    st_sprintf(path, "%s/g95tmpXXXXXX", tempdir);

    fd = mkstemp(path);

    if (fd >= 0)
	*result = fix_fd(fd);

    return fd < 0;
}



/* input_stream()-- Return a stream pointer to the default input stream.
 * Called on initialization. */

stream *input_stream(void) {

    return fd_to_stream(STDIN_FILENO, 0);
}



/* output_stream()-- Return a stream pointer to the default output stream.
 * Called on initialization. */

stream *output_stream(void) {

    return fd_to_stream(STDOUT_FILENO, 0);
}



/* error_stream()-- Return a stream pointer to the default error stream.
 * Called on initialization. */

stream *error_stream(void) {

    return fd_to_stream(STDERR_FILENO, 0);
}



/* compare_file_filename()-- Given a unit and a fortran string that is
 * a filename, figure out if the file is the same as the filename. */

int compare_file_filename(iounit_t *u, char *filename, int filename_len) {
char path[PATH_LENGTH+1];
struct stat st1, st2;

    if (unpack_filename(path, filename, filename_len))
	return 0;

    /* Can't be the same */

    /* If the filename doesn't exist, then there is no match with the
     * existing file. */

    if (stat(path, &st1) < 0)
	return 0;

    fstat(u->s->fd, &st2);

    return (st1.st_dev == st2.st_dev) && (st1.st_ino == st2.st_ino);
}



/* find_file0()-- Recursive work function for find_file() */

static iounit_t *find_file0(struct stat *target, iounit_t *u) {
struct stat s;
iounit_t *v;
int fd;

    if (u == NULL)
	return NULL;

    fd = u->s->fd;

    fstat(fd, &s);

    if (s.st_dev == target->st_dev && s.st_ino == target->st_ino)
	return u;

    v = find_file0(target, u->left);
    if (v != NULL)
	return v;

    v = find_file0(target, u->right);
    if (v != NULL)
	return v;

    return NULL;
}



/* find_file()-- Take the current filename and see if there is a unit
 * that has the file already open.  Returns a pointer to the unit if so. */

iounit_t *find_file(void) {
char path[PATH_LENGTH+1];
struct stat statbuf;

    if (unpack_filename(path, ioparm->file, ioparm->file_len)) 
	return NULL;

    if (stat(path, &statbuf) < 0)
	return NULL;

    return find_file0(&statbuf, globals.unit_root);
}


/* init_error_stream()-- Return a pointer to the error stream.  This
 * subroutine is called when the stream is needed, rather than at
 * initialization.  We want this to work even if memory has been
 * seriously corrupted. */

stream *init_error_stream(void) {
static stream error;

    memset(&error, '\0', sizeof(error));

    error.fd = options.use_stderr ? STDERR_FILENO : STDOUT_FILENO;

    error.unbuffered = 1;
    error.buffer = error.small_buffer;

    return &error;
}



/* terminal_device()-- See if the unit is associated with a terminal.
 * Unlike other files, terminal devices can be opened multiple times. */

int terminal_device(iounit_t *u) {
struct stat statbuf;

    fstat(u->s->fd, &statbuf);
    return S_ISCHR(statbuf.st_mode);
}



/* file_exists()-- Returns nonzero if the current filename exists on
 * the system */

int file_exists(void) {
char path[PATH_LENGTH+1];
struct stat statbuf;

    if (unpack_filename(path, ioparm->file, ioparm->file_len) ||
	stat(path, &statbuf) < 0)
	return 0;

    return 1;
}



/* sequential()-- Figure out if a file is openable in sequential mode */

static char *sequential(struct stat *statbuf) {

    if (S_ISREG(statbuf->st_mode) ||
	S_ISCHR(statbuf->st_mode) ||
	S_ISFIFO(statbuf->st_mode))
	return yes;

    if (S_ISDIR(statbuf->st_mode) ||
	S_ISBLK(statbuf->st_mode))
	return no;

    return unknown;
}



/* inquire_sequential()-- Given a fortran string, determine if the
 * file is suitable for sequential access.  Returns a C-style
 * string. */

char *inquire_sequential(char *fname, int fname_len) {
char path[PATH_LENGTH+1];
struct stat statbuf;

    if (fname == NULL ||
	unpack_filename(path, fname, fname_len) ||
	stat(path, &statbuf) < 0)
	return unknown;

    return sequential(&statbuf);
}


/* inquire_sequential_fd()-- Determine if an open file is suitable for
 * sequential access. */

char *inquire_sequential_fd(iounit_t *u) {
struct stat statbuf;

    if (u == NULL)
	return yes;

    if (u->flags.access == ACCESS_DIRECT)
	return no;

    if (fstat(u->s->fd, &statbuf) < 0)
	return unknown;

    return sequential(&statbuf);
}



/* direct()-- Determine if a file is openable in direct mode. */

static char *direct(struct stat *statbuf) {

    if (S_ISREG(statbuf->st_mode) ||
	S_ISBLK(statbuf->st_mode))
	return yes;

    if (S_ISDIR(statbuf->st_mode) ||
	S_ISCHR(statbuf->st_mode) ||
	S_ISFIFO(statbuf->st_mode))
	return no;

    return unknown;
}



/* inquire_direct()-- Given a fortran string, determine if the file is
 * suitable for direct access.  Returns a C-style string. */

char *inquire_direct(char *fname, int fname_len) {
char path[PATH_LENGTH+1];
struct stat statbuf;

    if (fname == NULL ||
	unpack_filename(path, fname, fname_len) ||
	stat(path, &statbuf) < 0)
	return unknown;

    return direct(&statbuf);
}



/* inquire_direct_fd()-- Given an open unit, see if it can be opened
 * in direct mode. */

char *inquire_direct_fd(iounit_t *u) {
struct stat statbuf;

    if (u == NULL || u->flags.access == ACCESS_SEQUENTIAL)
	return no;

    if (fstat(u->s->fd, &statbuf) < 0)
	return unknown;

    return direct(&statbuf);
}


/* formatted()-- Determine if a file can be opened in formattted form. */

static char *formatted(struct stat *statbuf) {

    if (S_ISREG(statbuf->st_mode) ||
	S_ISBLK(statbuf->st_mode) ||
	S_ISCHR(statbuf->st_mode) ||
	S_ISFIFO(statbuf->st_mode))
	return yes;

    if (S_ISDIR(statbuf->st_mode))
	return no;

    return unknown;
}



/* inquire_formatted()-- Given a fortran string, determine if the file
 * is suitable for formatted form.  Returns a C-style string. */

char *inquire_formatted(char *fname, int fname_len) {
char path[PATH_LENGTH+1];
struct stat statbuf;

    if (fname == NULL ||
	unpack_filename(path, fname, fname_len) ||
	stat(path, &statbuf) < 0)
	return unknown;

    return formatted(&statbuf);
}



/* inquire_formatted_fd()-- Determined if an open file can be opened
 * in formatted form. */

char *inquire_formatted_fd(iounit_t *u) {
struct stat statbuf;

    if (u == NULL)
	return yes;

    if (fstat(u->s->fd, &statbuf) < 0)
	return unknown;

    return formatted(&statbuf);
}



/* inquire_unformatted()-- Given a fortran string, determine if the file
 * is suitable for unformatted form.  Returns a C-style string. */

char *inquire_unformatted(char *fname, int fname_len) {

    return inquire_formatted(fname, fname_len);
}


char *inquire_unformatted_fd(iounit_t *u) {

    return inquire_formatted_fd(u);
}



/* inquire_access()-- Given a fortran string, determine if the file is
 * suitable for access. */

static char *inquire_access(char *fname, int fname_len, int mode) {
char path[PATH_LENGTH+1];

    if (fname == NULL ||
	unpack_filename(path, fname, fname_len) ||
	access(path, mode) < 0)
	return no;

    return yes;
}



static char *inquire_access_fd(iounit_t *u, int mode) {
int flags;
char *p;

    if (u != NULL && u->file_len != 0)
	return inquire_access(u->file, u->file_len, mode);

    flags = (u == NULL)
	? ACTION_READWRITE
	: u->flags.action;

    /* preopened unit */

    switch(flags) {
    case ACTION_READ:       p = ((mode & W_OK) == 0) ? yes : no; break;
    case ACTION_WRITE:      p = ((mode & R_OK) == 0) ? yes : no; break;
    case ACTION_READWRITE:  p = yes;      break;
    default:                p = unknown;  break;
    }

    return p;
}


/* inquire_read()-- Given a fortran string, determine if the file is
 * suitable for READ access. */

char *inquire_read(char *fname, int fname_len) {

    return inquire_access(fname, fname_len, R_OK);
}


char *inquire_read_fd(iounit_t *u) {

    return inquire_access_fd(u, R_OK);
}



/* inquire_write()-- Given a fortran string, determine if the file is
 * suitable for READ access. */

char *inquire_write(char *fname, int fname_len) {

    return inquire_access(fname, fname_len, W_OK);
}


char *inquire_write_fd(iounit_t *u) {

    return inquire_access_fd(u, W_OK);
}


/* inquire_readwrite()-- Given a fortran string, determine if the file is
 * suitable for read and write access. */

char *inquire_readwrite(char *fname, int fname_len) {

    return inquire_access(fname, fname_len, R_OK | W_OK);
}



char *inquire_readwrite_fd(iounit_t *u) {

    return inquire_access_fd(u, R_OK | W_OK);
}


/* default_action()-- Figure out a default action for a file that
 * we're about to open.  If the file doesn't exist, we look at the
 * directory. */

unit_action default_action(void) {
char *p, path[PATH_LENGTH+1];
int r, w;

    if (unpack_filename(path, ioparm->file, ioparm->file_len))
	goto doomed;

    if (access(path, F_OK)) {  /* Look at the directory */
	p = strchr(path, '\0');

	for(;;) {
	    p--;
	    if (p < path) {
		p++;
		*p++ = '.';
		break;
	    }

	    if (*p == '/')
		break;
	}

	*p = '\0';
    }

    r = !access(path, R_OK);
    w = !access(path, W_OK);

    if (r && w) return ACTION_READWRITE;
    if (r) return ACTION_READ;
    if (w) return ACTION_WRITE;

doomed:
    return ACTION_READWRITE;
}



/* sys_exit()-- Terminate the program with an exit code */

void sys_exit(int abend, int code) {

    if (abend && options.abort)
	abort();

    exit(code);
}



/* delete_file()-- Delete a file */

int delete_file(char *path) {

    return unlink(path);
}
