

#include "runtime.h"

#if HAVE_RESUME

#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <fcntl.h>
#include <limits.h>
#include <unistd.h>
#include <sys/mman.h>
#include <signal.h>
#include <errno.h>
#include <sys/stat.h>



typedef long elf_offset;


typedef struct {
    unsigned char e_ident[16];
    short         e_type;
    short         e_machine;
    int           e_version;
    void         *e_entry;
    elf_offset    e_phoff;
    elf_offset    e_shoff;
    int           e_flags;
    short         e_ehsize;
    short         e_phentsize;
    short         e_phnum;
    short         e_shentsize;
    short         e_shnum;
    short         e_shstrndx;
} header;


typedef struct {
    int         p_type;
    elf_offset  p_offset;
    void       *p_vaddr;
    void       *p_paddr;
    int         p_filesz;
    int         p_memsz;
    int         p_flags;
    int         p_align;
} pt_header;


#define PT_NULL    0
#define PT_LOAD    1

/* Flags for a loadable segment */

#define PF_X     1
#define PF_W     2
#define PF_R     4

#define ET_EXEC 2

#if 1
#define ELF_CLASS 1     /* 32 bits */
#else
#define ELF_CLASS 2     /* 64 bits */
#endif


#define ELF_DATA  1  /* little endian */
/* #define ELF_DATA 2 */ /* big endian */

#define ELF_VERSION 1

#define ELF_MACHINE 3   /* Intel i386 */


typedef enum {
    SEG_PRIVATE_MAP  = 0,
    SEG_PUBLIC_MAP   = 1,
    SEG_EXEC         = 2,
    SEG_DATA         = 3,
    SEG_HEAP         = 4,
    SEG_STACK        = 5,
    SEG_TLS          = 6,
    SEG_NULL         = 7
} segment_type;



/* Structure used by the *_thread_area calls.  This is specific to
 * linux 2.6, so we don't bother with system headers. */

typedef struct {
    unsigned int  entry_number;
    unsigned long base_addr;
    unsigned int  limit;

    unsigned int  seg_32bit:1;
    unsigned int  contents:2;
    unsigned int  read_exec_only:1;
    unsigned int  limit_in_pages:1;
    unsigned int  seg_not_present:1;
    unsigned int  useable:1;
} tls_info;




typedef struct {
    void *start, *end;
    int perm, offset, private;
    segment_type type;
    char name[PATH_MAX+1];
} mapline;


/* Register save structure.  The initialization forces these variables
 * into the initialized data segment. */

static void *regs_esp=NULL, *regs_ebp=NULL, *regs_eip=NULL;
static short regs_gs=0;

static volatile int restart_flag = 1;


/* System I/O calls.  When we get control from the kernel, the shared
 * C library isn't mapped, so regular system calls segfault. */

static int read0(int fd, void *buffer, int n) {
int rc;

    asm("mov $3, %%eax\n"
	"push %%ebx\n"
	"mov %1, %%ebx\n"
	"mov %2, %%ecx\n"
	"mov %3, %%edx\n"
	"int $0x80\n"
	"pop %%ebx\n"
	"mov %%eax, %0\n" : "=m" (rc) : "m" (fd), "m" (buffer), "m" (n) 
                          : "%eax", "%ecx", "%edx" );

    return (rc >= 0) ? rc : -1;
}



static int write0(int fd, void *buffer, int n) {
int rc;

    asm("mov $4, %%eax\n"
	"push %%ebx\n"
	"mov %1, %%ebx\n"
	"mov %2, %%ecx\n"
	"mov %3, %%edx\n"
	"int $0x80\n"
	"pop %%ebx\n"
	"mov %%eax, %0\n" : "=m" (rc) : "m" (fd), "m" (buffer), "m" (n) 
			  : "%eax", "%ecx", "%edx" );

    return (rc >= 0) ? rc : -1;
}



static int open0(char *name, int flags) {
int rc;

    asm("mov $5, %%eax\n"
	"push %%ebx\n"
	"mov %1, %%ebx\n"
	"mov %2, %%ecx\n"
	"mov $0, %%edx\n"
	"int $0x80\n"
	"pop %%ebx\n"
	"mov %%eax, %0\n" : "=m" (rc) : "m" (name), "m" (flags)
			  : "%eax", "%ecx", "%edx" );

    return (rc >= 0) ? rc : -1;
}



static void exit0(int rc) {

    asm("mov $1, %%eax\n"
	"mov %0, %%ebx\n"
	"int $0x80\n" : : "m" (rc) : "%eax");
}



static int close0(int fd) {
int rc;

    asm("mov $6, %%eax\n"
	"push %%ebx\n"
	"mov %1, %%ebx\n"
	"int $0x80\n"
	"pop %%ebx\n"
	"mov %%eax, %0\n" : "=m" (rc) : "m" (fd) : "%eax");

    return (rc >= 0) ? rc : -1;
}



static int lseek0(int fd, int offset, int whence) {
int rc;

    asm("mov $19, %%eax\n"
	"push %%ebx\n"
	"mov %1, %%ebx\n"
	"mov %2, %%ecx\n"
	"mov %3, %%edx\n"
	"int $0x80\n"
	"pop %%ebx\n"
	"mov %%eax, %0\n" : "=m" (rc) : "m" (fd), "m" (offset), "m" (whence)
			  : "%eax", "%ecx", "%edx" );

    return (rc >= 0) ? rc : -1;
}



static int brk0(void *end) {
int rc;

    asm("mov $45, %%eax\n"
	"push %%ebx\n"
	"mov %1, %%ebx\n"
	"int $0x80\n"
	"pop %%ebx\n"
	"mov %%eax, %0\n" : "=m" (rc) : "m" (end) : "%eax");

    return (rc >= 0) ? rc : -1;
}



static int mmap0(void *start, int length, int prot, int flags, int fd,
		 int offset) {
void *pointer;
int rc;

    pointer = &start;

    asm("mov $90, %%eax\n"
	"push %%ebx\n"
	"mov %1, %%ebx\n"
	"int $0x80\n"
	"pop %%ebx\n"
	"mov %%eax, %0\n" : "=m" (rc) : "m" (pointer) : "%eax");

    return (rc >= 0) ? rc : -1;
}



static int mprotect0(void *addr, int len, int prot) {
int rc;

    asm("mov $125, %%eax\n"
	"push %%ebx\n"
	"mov %1, %%ebx\n"
	"mov %2, %%ecx\n"
	"mov %3, %%edx\n"
	"int $0x80\n"
	"pop %%ebx\n"
	"mov %%eax, %0\n" : "=m" (rc) : "m" (addr), "m" (len), "m" (prot)
			  : "%eax", "%ecx", "%edx" );

    return (rc >= 0) ? rc : -1;
}



static int set_thread_area0(void *u) {
int rc;

    asm("mov $243, %%eax\n"
	"push %%ebx\n"
	"mov %1, %%ebx\n"
	"int $0x80\n"
	"pop %%ebx\n"
	"mov %%eax, %0\n" : "=m" (rc) : "m" (u) : "%eax");

    return (rc >= 0) ? rc : -1;
}



static int get_thread_area0(void *u) {
int rc;

    asm("mov $244, %%eax\n"
	"push %%ebx\n"
	"mov %1, %%ebx\n"
	"int $0x80\n"
	"pop %%ebx\n"
	"mov %%eax, %0\n" : "=m" (rc) : "m" (u) : "%eax");

    return (rc >= 0) ? rc : -1;
}



/* write_hex()-- Write a hexadecimal number */

static void write_hex(unsigned value) {
char *p, buffer[8];
int i, n;

    p = buffer;
    for(i=0; i<8; i++) {
	n = value >> 28;
	*p++ = (n < 10)
	    ? n + '0'
	    : n + 'A' - 10;

	value = value << 4;
    }

    write0(1, buffer, 8);
}



/* write_string()-- Write a string */

static void write_string(char *msg) {
int n;

    n = 0;
    while(msg[n] != '\0')
	n++;

    if (n > 0)
	write0(1, msg, n);
}



/* heap_failure()-- If the new process has a different heap in the
 * wrong direction, we can't restart.  Print a message explaining the
 * problem and exit gracefully. */

static void heap_failure(unsigned new, unsigned old) {

    write_string("\nLinux has chosen a heap for this process that starts "
		 "at 0x");
    write_hex(new);

    write_string(", but at\n0x");
    write_hex(old);

    write_string(" for the original process.  This makes it impossible to "
		 "resume.  If\nyou run the dumpfile again, you might have "
		 "better luck the next time.  To\nturn off this, ummm, "
		 " feature, run (as root)\n\n"
		 "    echo 0 >/proc/sys/kernel/randomize_va_space\n\n");

    exit0(96);
}



/* file_error()-- Report an error reopening a file. */

static void file_error(char *path, int flag) {
char buffer[200+PATH_LENGTH];

    if (flag == 1) {
	strcpy(buffer, "Error reopening ");
	strcat(buffer, path);
	strcat(buffer, ".  errno = ");
	strcat(buffer, int_to_a(errno));

    } else {
	strcpy(buffer, "Error seeking within ");
	strcat(buffer, path);
	strcat(buffer, ".  Different files?");
    }

    runtime_error(buffer);
}



/* restore_unit()-- Recursive function to reopen a unit */

static void restore_unit(iounit_t *u) {
char path[PATH_LENGTH+1];
int fd, flags;
stream *s;

    if (u == NULL)
	u = globals.unit_root;

    s = u->s;
    if (s->fd > 2) {
	switch(u->flags.action) {
	case ACTION_READ:       flags = O_RDONLY;  break;
	case ACTION_WRITE:      flags = O_WRONLY;  break;
	case ACTION_READWRITE:  flags = O_RDWR;    break;
	default:                flags = 0;         break;
	}

	memcpy(path, u->file, u->file_len);
	path[u->file_len] = '\0';

	fd = open0(path, flags);
	if (fd < 0)
	    file_error(path, 1);

	s->fd = fd;
	if (lseek0(fd, s->physical_offset, SEEK_SET) < 0)
	    file_error(path, 2);
    }

    if (u->left != NULL)
	restore_unit(u->left);

    if (u->right != NULL)
	restore_unit(u->right);
}



/* get_prot()-- Get an mmap() prot-mask from the analogous elf flags. */

static int get_prot(int elf_flags) {
int prot;

    prot = 0;

    if (elf_flags & PF_R)
	prot |= PROT_READ;

    if (elf_flags & PF_W)
	prot |= PROT_WRITE;

    if (elf_flags & PF_X)
	prot |= PROT_EXEC;

    return prot;
}



/* entry()-- Entry point for the restart. */

static void entry(void) {
char filename[PATH_MAX+10];
int i, m, fd, map_fd, mode, prot;
pt_header pt, stack;
tls_info thread;
unsigned esp;
header h;
void *b;

  /* If the current stack does not go up to the usual 0xC0000000, map
   * memory between TOS and 0xC0000000.  This works around a
   * randomized stack location. */

    asm("mov %%esp, %0\n" : : "m" (esp));

    i = (esp + 0x0FFF) & ~(0x0FFF);
    if (i != 0xC0000000)
	mmap0((void *) i, 0xC0000000 - i, PROT_READ | PROT_WRITE | PROT_EXEC,
	      MAP_PRIVATE | MAP_FIXED | MAP_ANONYMOUS, -1, 0);

    write0(2, "Restarting\n", 11);

    fd = open0("/proc/self/exe", O_RDONLY);

    read0(fd, &h, sizeof(h));

    for(i=0; i<h.e_phnum; i++) {
	write0(2, ".", 1);

	lseek0(fd, h.e_ehsize + i*sizeof(pt), SEEK_SET);
	read0(fd, &pt, sizeof(pt));
	lseek0(fd, pt.p_offset, SEEK_SET);

	if (pt.p_type != PT_NULL)
	    continue;

	switch(pt.p_memsz) {
	case SEG_TLS:
	    read0(fd, &thread, sizeof(thread));
	    set_thread_area0(&thread);

	    m = pt.p_filesz - sizeof(thread);

	    mmap0(pt.p_vaddr, m, PROT_READ | PROT_WRITE,
		  MAP_PRIVATE | MAP_FIXED | MAP_ANONYMOUS, -1, 0);

	    read0(fd, (void *) pt.p_vaddr, m);

	    mprotect0(pt.p_vaddr, m, get_prot(pt.p_flags));
	    break;

	case SEG_HEAP:
	    b = (void *) brk0(0);   /* Get start of brk area */
	    if (b > pt.p_vaddr)
		heap_failure((unsigned) b, (unsigned) pt.p_vaddr);

	    brk0(pt.p_vaddr + pt.p_filesz); /*Map pages and restore brk value*/
	    read0(fd, pt.p_vaddr, pt.p_filesz);
	    break;

	case SEG_STACK:
	    stack = pt;     /* Copy for now, restore later... */
	    break;

	    /* Stack segment: only load pages at and beyond the saved esp. */
	    m = regs_esp - pt.p_vaddr - 16;
	    if (m < 0)
		m = 0;

	    lseek0(fd, m, SEEK_CUR);
	    read0(fd, pt.p_vaddr+m, pt.p_filesz-m);
	    break;

	case SEG_PRIVATE_MAP:
	case SEG_PUBLIC_MAP:
	    prot = get_prot(pt.p_flags);

	    if (pt.p_memsz == SEG_PRIVATE_MAP) {  /* Anonymous map */
		mmap0(pt.p_vaddr, pt.p_filesz, PROT_READ | PROT_WRITE,
		      MAP_PRIVATE | MAP_FIXED | MAP_ANONYMOUS, -1, 0);

		read0(fd, pt.p_vaddr, pt.p_filesz);

		mprotect0(pt.p_vaddr, pt.p_filesz, prot);

	    } else {
		read0(fd, filename, PATH_MAX+1);

		if ((pt.p_flags & (PF_R | PF_W)) == (PF_R | PF_W))
		    mode = O_RDWR;

		else if ((pt.p_flags & PF_R) == PF_R)
		    mode = O_RDONLY;

		else if ((pt.p_flags & PF_W) == PF_W)
		    mode = O_WRONLY;

		else
		    mode = 0;  /* Ha, ha */

		map_fd = open0(filename, mode);

		mmap0(pt.p_vaddr, pt.p_filesz, prot, MAP_SHARED | MAP_FIXED,
		      map_fd, pt.p_align);

		close0(map_fd);
	    }

	default:
	    break;
	}
    }

    /* Move stack to a temporary area 3 pages below the ultimate
     * stack.  The regs_* variables should be accessible by now. */

    i = esp - (((unsigned) regs_esp) - 16384);

    asm("mov %0, %%eax\n"
	"sub %%eax, %%esp\n" 
	"sub %%eax, %%ebp\n"
	"mov %%eax, %0\n" : : "m" (i) : "%eax");

    /* Copy old variables to new frame. */

    memmove(&fd,    ((char *) &fd)    + i, sizeof(int));
    memmove(&stack, ((char *) &stack) + i, sizeof(stack));

    /* Stack segment: only load pages at and beyond the saved esp. */
    m = regs_esp - stack.p_vaddr - 16;
    if (m < 0)
	m = 0;

    lseek0(fd, stack.p_offset, SEEK_SET);
    lseek0(fd, m, SEEK_CUR);
    read0(fd, stack.p_vaddr + m, stack.p_filesz - m);

    close0(fd);
    restart_flag = 0;

    restore_unit(globals.unit_root);
    write0(2, "Jumping\n", 8);

    asm("add $16384, %%ebp\n"
	"add $16384, %%esp\n"
	"mov %0, %%gs\n"
	"jmp *%1\n"
	: : "m" (regs_gs), "m" (regs_eip));
}



/*******************************************************************/



/* writen()-- Write some bytes */

static int writen(int fd, void *buffer, int n) {
int i, m;

    i = n; 

    while(i>0) {
	m = write(fd, buffer, i);
	if (m < 0)
	    return -1;

	buffer += m;
	i -= m;
    }

    return n;
}



/* next_char()-- Returns the next character from a file descriptor.
 * Returns -1 for EOF. */

static int next_char(int fd) {
char c;

    return (read(fd, &c, 1) == 0)
	? -1
	: c;
}



/* parse_hex()-- Parse a hexadecimal value */

static int parse_hex(int fd, int *value) {
int i, m, c, v;

    v = 0;
    for(i=0; i<2*sizeof(void *); i++) {
	c = next_char(fd);
	if (c == -1)
	    return -1;

	if ('0' <= c && c <= '9')
	    m = c - '0';
	else if ('a' <= c && c <= 'f')
	    m = c - 'a' + 10;
	else
	    return -1;

	v = (v << 4) + m;
    }

    *value = v;
    return 0;
}


/* tls_segment()-- Determine if a segment is a thread-local storage
 * segment. */

static int tls_segment(void *start, void *end) {
tls_info thread;
int i;

    for(i=0; i<100; i++) {
	thread.entry_number = i;
	if (get_thread_area0(&thread) || thread.seg_not_present)
	    continue;

	if ((void *) thread.base_addr >= start &&
	    (void *) thread.base_addr < end)
	    return 1;
    }

    return 0;
}



/* parse_mapline()-- Parse a single map line.  Returns 0 if OK,
 * nonzero if eof/error has been reached.  The Sun guys are right.
 * The linux /proc sucks-- it's pretty pointless to have the kernel
 * convert its data structures into ascii, only to have to parse it
 * back again.  We return zero if we've read another line. */

static int parse_mapline(int fd, mapline *line) {
int c, m;
char *p;

    if (parse_hex(fd, &m) == -1)
	return 1;

    line->start = (void *) m;

    next_char(fd);

    if (parse_hex(fd, &m) == -1)
	return 1;

    line->end = (void *) m;

    next_char(fd);

    line->perm = 0;

    if (next_char(fd) == 'r')
	line->perm |= PF_R;

    if (next_char(fd) == 'w')
	line->perm |= PF_W;

    if (next_char(fd) == 'x')
	line->perm |= PF_X;

    line->private = (next_char(fd) == 'p');

    next_char(fd);

    if (parse_hex(fd, &line->offset) == -1)
	return 1;

    next_char(fd);
    while(next_char(fd) != ' ');    /* Skip device */

    do
	c = next_char(fd);
    while (c != ' ' && c != '\n');  /* Skip inode */

    if (c != '\n')
	do
	    c = next_char(fd);
	while(c == ' ');

    if (c == '\n')
	line->name[0] = '\0';

    else {
	p = line->name;
	*p++ = c;

	for(;;) {
	    c = next_char(fd);
	    if (c == '\n')
		break;

	    *p++ = c;
	}

	*p = '\0';

	if (line->name[0] == '[')
	    line->name[0] = '\0';
    }

    /* Figure out what kind of segment this is */

    if (line->start <= ((void *) &parse_mapline) &&
	((void *) &parse_mapline) < line->end)
	line->type = SEG_EXEC;

    else if (line->start <= sbrk(0) && sbrk(0) <= line->end)
	line->type = SEG_HEAP;

    else if (line->start <= ((void *) &m) && ((void *) &m) < line->end)
	line->type = SEG_STACK;

    else if (line->start <= ((void *) &globals) &&
	     ((void *) &globals) < line->end)
	line->type = SEG_DATA;

    else if (line->start <= ((void *) &restart_flag) &&
	     ((void *) &restart_flag) < line->end)
	line->type = SEG_DATA;

    else if (((unsigned long) line->start) >= 0xC0000000U)
	line->type = SEG_NULL;

    else if (tls_segment(line->start, line->end))
	line->type = SEG_TLS;

    else if (line->private)
	line->type = SEG_PRIVATE_MAP;

    else
	line->type = SEG_PUBLIC_MAP;

    return 0;
}



/* dump_file()-- Dumps the current process to an ELF executable.  We
 * return nonzero if we are resuming execution, or zero if we just did
 * a dump. */

static void dump_file(void) {
int i, j, segments, map_fd, dump_fd;
unsigned char *p;
tls_info thread;
mapline line;
pt_header pt;
char zero;
header h;

    asm("mov %%esp, %0\n"
	"mov %%ebp, %1\n"
	"movw %%gs, %2\n"
	: "=m" (regs_esp), "=m" (regs_ebp), "=m" (regs_gs) : );

    regs_eip = &&done;

    memset(&pt, '\0', sizeof(pt));
    memset(&h,  '\0', sizeof(h));

    dump_fd = open("dump", O_TRUNC | O_CREAT | O_RDWR, S_IRWXU);
    zero = '\0';

    p = &h.e_ident[0];
    *p++ = 0x7F;  *p++ = 'E';  *p++ = 'L';  *p++ = 'F';

    *p++ = ELF_CLASS;
    *p++ = ELF_DATA;
    *p++ = ELF_VERSION;

    h.e_type       = ET_EXEC;
    h.e_machine    = ELF_MACHINE;
    h.e_version    = ELF_VERSION;
    h.e_entry      = entry;
    h.e_phoff      = sizeof(header);
    h.e_shoff      = 0;            /* No section header */
    h.e_flags      = 0;
    h.e_ehsize     = sizeof(header);
    h.e_phentsize  = sizeof(pt_header);
    h.e_phnum      = 0;            /* To be determined */
    h.e_shentsize  = 40;
    h.e_shnum      = 0;
    h.e_shstrndx   = 0;

    writen(dump_fd, &h, sizeof(h));

    /* The structure of our elf file is real simple.  The elf header is
     * followed by the program headers.  Figure out how many segments we
     * have by counting the number of newlines, then reserve space for
     * the headers.  While we're writing segments, segment info is
     * appended to the end of the file, and we periodically seek back to
     * the correct program header to write them. */

    map_fd = open("/proc/self/maps", O_RDONLY);
    if (map_fd < 0)
	return;

    segments = 0;

    while(!parse_mapline(map_fd, &line)) {
	writen(dump_fd, &h, sizeof(h));
	segments++;
    }

    close(map_fd);
    map_fd = open("/proc/self/maps", O_RDONLY);

    /* Pass two, write the segments */

    for(i=0; i<segments; i++) {
	memset(&pt, '\0', sizeof(pt));
	pt.p_offset = lseek(dump_fd, 0, SEEK_CUR);

	parse_mapline(map_fd, &line);

	pt.p_vaddr  = pt.p_paddr = line.start;
	pt.p_filesz = pt.p_memsz = line.end - line.start;
	pt.p_flags  = line.perm;
	pt.p_align  = 4096;

	switch(line.type) {
	case SEG_EXEC:
	case SEG_DATA:
	    pt.p_type = PT_LOAD;

	    while((pt.p_offset & 0xfff) != 0) {
		/* Pad to a page boundary because the 'loadable' parts of the *
		 * disk image are in fact mmap()-ed. */
		write(dump_fd, &zero, 1);
		pt.p_offset++;
	    }

	    /* Text segment, initialized data and uninitialized data. */

	    writen(dump_fd, line.start, pt.p_filesz);
	    break;

	case SEG_HEAP:
	    pt.p_type = PT_NULL;
	    pt.p_memsz = SEG_HEAP;
	    pt.p_filesz = sbrk(0) - pt.p_vaddr; 
	    writen(dump_fd, line.start, pt.p_filesz);
	    break;

	case SEG_STACK:
	    pt.p_type = PT_NULL;
	    pt.p_memsz = SEG_STACK;
	    writen(dump_fd, line.start, pt.p_filesz);
	    break;

	case SEG_TLS:
	    pt.p_type = PT_NULL;
	    pt.p_memsz = SEG_TLS;
	    pt.p_align = 0;

	    for(j=0; j<100; j++) {
		thread.entry_number = j;
		if (get_thread_area0(&thread) == 0 &&
		    !thread.seg_not_present &&
		    (void *) thread.base_addr >= line.start &&
		    (void *) thread.base_addr < line.end) {
		    writen(dump_fd, &thread, sizeof(tls_info));
		    break;
		}
	    }

	    writen(dump_fd, line.start, pt.p_filesz);
	    pt.p_filesz += sizeof(tls_info);
	    break;

	case SEG_PRIVATE_MAP:
	case SEG_PUBLIC_MAP:
	    pt.p_type = PT_NULL;
	    pt.p_memsz = line.type;

	    if (line.private)
		writen(dump_fd, line.start, pt.p_filesz);

	    else {
		writen(dump_fd, line.name, strlen(line.name)+1);
		pt.p_align = line.offset;
	    }

	    break;

	default:
	    pt.p_type = PT_NULL;
	    pt.p_memsz = SEG_NULL;
	    break;
	}

	lseek(dump_fd, sizeof(header) + i*sizeof(pt_header), SEEK_SET);
	writen(dump_fd, &pt, sizeof(pt));

	lseek(dump_fd, 0, SEEK_END);
    }

    h.e_phnum = segments;

    lseek(dump_fd, 0, SEEK_SET);
    writen(dump_fd, &h, sizeof(h));

    close(map_fd);
    close(dump_fd);

    if (globals.checkpoint_msg)
	write(STDERR_FILENO, "Process dumped\n", 15);

done:
    return;
}



/* dump_signal()-- Signal handler for dumping the current state. */

void dump_signal(int signo) {
int option;

    option = 0;

    dump_file();

    if (!restart_flag) {        /* Resuming from a dumpfile */
	restart_flag = 1;
	init_fpu_trap();
 
    } else {   /* Dumping */
	switch(signo) {
	case SIGINT:    option = options.sigint;    break;
	case SIGQUIT:   option = options.sigquit;   break;
	case SIGHUP:    option = options.sighup;    break;
	case SIGALRM:
	    option = SIGDISP_DUMP; 
	    alarm(globals.alarm_value);
	    break;
	}

	if (option == SIGDISP_DUMP_QUIT)
	    _exit(0);
    }
}

#endif

