
/* Quad precision absolute value calculation
 *
 * Copyright (c) 2008, Andrew Vaught
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 *
 * * The name of Andrew Vaught may not be used to endorse or promote
 *   products derived from this software without specific prior
 *   written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE. */


#include "runtime.h"


static void unpacked_abs_z16(unpacked16 *result, unpacked16 *a, unpacked16 *b) {
unpacked16 *s, q, g;

    a->sign = 0;
    b->sign = 0;

    if (a->exp == EXP16_NAN &&
	(a->m[0] != 0 || a->m[1] != 0 || a->m[2] != 0 || a->m[3] != 0)) {
	    set_nan16(result);
	    return;
	}

    if (b->exp == EXP16_NAN &&
	(b->m[0] != 0 || b->m[1] != 0 || b->m[2] != 0 || b->m[3] != 0)) {
	    set_nan16(result);
	    return;
	}

    if (a->exp == EXP16_NAN || b->exp == EXP16_NAN) {
	result->m[0] = result->m[1] = result->m[2] = result->m[3] = 0;
	result->exp  = 0;
	result->sign = 0;
	return;
    }

    if (a->exp == 0 && a->m[0] == 0 && a->m[1] == 0 &&
	a->m[2] == 0 && a->m[3] == 0) {

	*result = *b;
	return;
    }

    if (b->exp == 0 && b->m[0] == 0 && b->m[1] == 0 &&
	b->m[2] == 0 && b->m[3] == 0) {

	*result = *a;
	return;
    }

    /* General case. */

    if (compare16(a, b) >= 0) {
	s = a;
	a = b;
	b = s;
    }

    /* a >= b */

    divide_unpacked(b, a, &q);
    multiply_unpacked(&q, &q, &q);

    g.exp = 16383;
    g.sign = 0;
    g.m[0] = 0x10000;
    g.m[1] = g.m[2] = g.m[3] = 0;

    add_unpacked(&g, &q, &g);

    unpacked_sqrt16(&g);

    multiply_unpacked(&g, a, result);
}



#define abs_z16 prefix(abs_z16)

#if QUAD_POINTER

void abs_z16(packed16 *result, packed16 *z) {
unpacked16 a, b, r;

    unpack_real_16(&z[0], a.m, &a.exp, &a.sign);
    unpack_real_16(&z[1], b.m, &b.exp, &b.sign);

    unpacked_abs_z16(&r, &a, &b);
    pack_real_16(result, r.m, &r.exp, &r.sign);
}

#else

packed16 abs_z16(packed16 *z) {
unpacked16 a, b, r;
packed16 result;

    unpack_real_16(&z[0], a.m, &a.exp, &a.sign);
    unpack_real_16(&z[1], b.m, &b.exp, &b.sign);

    unpacked_abs_z16(&r, &a, &b);
    pack_real_16(&result, r.m, &r.exp, &r.sign);

    return result;
}

#endif

