//
// flodo.c: simple network flow sampling tool
//
// i frequently want to know what the top few network flows are... and yet
// there didn't seem to be any tool which was as simple to use as "vmstat"
// or "iostat".  this tool fills that void for me.
//
// with the default options flodo will dump the top 20 flows every 5 seconds.
//
// flodo is quite basic at this point, understanding only ethernet frames,
// ipv4, tcp, and udp.
//
// flodo can't even do reverse DNS and doesn't bother mapping port numbers
// to services or display icmp or ethernet protocol names.
//

// 
// Copyright (c) 2004 Dean Gaudet <dean@arctic.org>
// 
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
// OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// OTHER DEALINGS IN THE SOFTWARE.
// 

// $Id: flodo.c,v 1.11 2004/11/08 04:05:14 dean Exp $

// _BSD_SOURCE gets all the nice packet header structures
#define _BSD_SOURCE 1
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <assert.h>
#include <signal.h>
#include <sys/time.h>
#include <errno.h>
#include <sysexits.h>
#include <pcap.h>
#if defined(__linux__) || defined(__FreeBSD__)
#include <net/ethernet.h>
#endif
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#if defined(__NetBSD__)
#include <net/if.h>
#include <net/if_ether.h>
#endif

#include "hash.h"
#include "local_mac.h"

static const char *progname;
static unsigned top_n = 20;
static pcap_t *handle;
static unsigned secs_per_sample = 5;
static unsigned group_layer = 4;
static struct bpf_program filter_program;
static enum {
	DIR_EITHER,
	DIR_IN,
	DIR_OUT
} direction = DIR_EITHER;
static int one_shot = 0;


static const char *print_ether(const uint8_t *addr)
{
	static char bufs[4][3*6];
	static unsigned cur_buf = 0;
	char *ret;

	ret = bufs[cur_buf];
	++cur_buf;
	if (cur_buf == 4) cur_buf = 0;
	sprintf(ret, "%02x:%02x:%02x:%02x:%02x:%02x",
		addr[0], addr[1], addr[2], addr[3], addr[4], addr[5]);
	return ret;
}


static const char *print_ipaddr(uint32_t addr)
{
	static char bufs[4][16];
	static unsigned cur_buf = 0;
	char *ret;

	ret = bufs[cur_buf];
	++cur_buf;
	if (cur_buf == 4) cur_buf = 0;
	addr = ntohl(addr);
	sprintf(ret, "%u.%u.%u.%u",
		(addr >> 24) & 0xff,
		(addr >> 16) & 0xff,
		(addr >> 8) & 0xff,
		addr & 0xff);
	return ret;
}


static const char *print_ipaddr_port(uint32_t addr, uint16_t port)
{
	static char bufs[4][22];
	static unsigned cur_buf = 0;
	char *ret;

	ret = bufs[cur_buf];
	++cur_buf;
	if (cur_buf == 4) cur_buf = 0;
	addr = ntohl(addr);
	sprintf(ret, "%u.%u.%u.%u.%u",
		(addr >> 24) & 0xff,
		(addr >> 16) & 0xff,
		(addr >> 8) & 0xff,
		addr & 0xff,
		ntohs(port));
	return ret;
}


typedef enum {
	ACCUM_UNKNOWN,
	ACCUM_ETHER,
	ACCUM_IP,
	ACCUM_TCP,
	ACCUM_UDP,
} accum_class_t;

typedef struct {
	accum_class_t class;
	int outbound;
	union {
		struct ether_header ether;
		struct {
			uint32_t saddr;
			uint32_t daddr;
			uint8_t protocol;
		} ip;
		struct {
			uint32_t saddr;
			uint32_t daddr;
			uint16_t sport;
			uint16_t dport;
		} tcp;
		struct {
			uint32_t saddr;
			uint32_t daddr;
			uint16_t sport;
			uint16_t dport;
		} udp;
	} x;
} accum_key_t;


static const char *print_accum_key(const accum_key_t *a)
{
	static char buf[256];
	const char *inout = a->outbound ? "out" : "in ";

	switch (a->class) {
	case ACCUM_UNKNOWN:
		snprintf(buf, sizeof(buf), "%s ?????", inout);
		break;

	case ACCUM_ETHER:
		snprintf(buf, sizeof(buf), "%s ether %s -> %s %04x",
			inout,
			print_ether(a->x.ether.ether_shost),
			print_ether(a->x.ether.ether_dhost),
			ntohs(a->x.ether.ether_type));
		break;

	case ACCUM_IP:
		snprintf(buf, sizeof(buf), "%s ip    %-15s -> %-15s %02x",
			inout,
			print_ipaddr(a->x.ip.saddr),
			print_ipaddr(a->x.ip.daddr),
			a->x.ip.protocol);
		break;

	case ACCUM_TCP:
		snprintf(buf, sizeof(buf), "%s tcp   %-21s -> %-21s",
			inout,
			print_ipaddr_port(a->x.tcp.saddr, a->x.tcp.sport),
			print_ipaddr_port(a->x.tcp.daddr, a->x.tcp.dport));
		break;

	case ACCUM_UDP:
		snprintf(buf, sizeof(buf), "%s udp   %-21s -> %-21s",
			inout,
			print_ipaddr_port(a->x.udp.saddr, a->x.udp.sport),
			print_ipaddr_port(a->x.udp.daddr, a->x.udp.dport));
		break;

	}
	return buf;
}


static int parse_ip(accum_key_t *a, const struct ip *ip, size_t len)
{
	struct tcphdr *tcp;
	struct udphdr *udp;

	if (len < sizeof(*ip)) {
		return -1;
	}
	if (group_layer > 3) {
		switch (ip->ip_p) {
		case IPPROTO_TCP:
			if (len >= sizeof(*ip) + sizeof(*tcp)) {
				tcp = (struct tcphdr *)((char *)ip + sizeof(*ip));
				a->class = ACCUM_TCP;
				a->x.tcp.saddr = *(uint32_t *)&ip->ip_src;
				a->x.tcp.daddr = *(uint32_t *)&ip->ip_dst;
				a->x.tcp.sport = tcp->th_sport;
				a->x.tcp.dport = tcp->th_dport;
				return 0;
			}
			break;
		case IPPROTO_UDP:
			if (len >= sizeof(*ip) + sizeof(*udp)) {
				udp = (struct udphdr *)((char *)ip + sizeof(*ip));
				a->class = ACCUM_UDP;
				a->x.udp.saddr = *(uint32_t *)&ip->ip_src;
				a->x.udp.daddr = *(uint32_t *)&ip->ip_dst;
				a->x.udp.sport = udp->uh_sport;
				a->x.udp.dport = udp->uh_dport;
				return 0;
			}
			break;
		}
	}
	a->class = ACCUM_IP;
	a->x.ip.saddr = *(uint32_t *)&ip->ip_src;
	a->x.ip.daddr = *(uint32_t *)&ip->ip_dst;
	a->x.ip.protocol = ip->ip_p;
	return 0;
}


typedef struct {
	uint64_t total_bytes;
	uint64_t total_packets;
} accum_value_t;

static hash_t *accum_hash;


static void pkt_handler(u_char *unused, const struct pcap_pkthdr *header, const u_char *packet)
{
	const struct ether_header *ether;
	accum_key_t a;
	accum_value_t *match;

	memset(&a, 0, sizeof(a));
	if (header->len < sizeof(*ether)) {
		a.class = ACCUM_UNKNOWN;
		goto parsed;
	}
	ether = (const struct ether_header *)packet;
	a.outbound = is_local_mac(ether->ether_shost);
	if (direction == DIR_IN && a.outbound) {
		return;
	}
	else if (direction == DIR_OUT && !a.outbound) {
		return;
	}
	if (group_layer > 2) {
		switch (ntohs(ether->ether_type)) {
		case ETHERTYPE_IP:
			if (parse_ip(&a,
				(const struct ip *)(packet + sizeof(*ether)),
				header->len - sizeof(*ether)) == 0)
				goto parsed;
			break;
		}
	}
	a.class = ACCUM_ETHER;
	a.x.ether = *ether;

parsed:
	match = hash_lookup(accum_hash, &a, sizeof(a));
	if (match == NULL) {
		match = malloc(sizeof(*match));
		memset(match, 0, sizeof(*match));
		hash_insert(accum_hash, &a, sizeof(a), match);
	}
	match->total_bytes += header->len;
	++match->total_packets;
}

	
static int accum_compare(const void *va, const void *vb)
{
	const hash_linear_t *a = va;
	accum_value_t *aval = (accum_value_t *)a->value;
	const hash_linear_t *b = vb;
	accum_value_t *bval = (accum_value_t *)b->value;

	if (aval->total_bytes != bval->total_bytes) {
		return (int64_t)bval->total_bytes - (int64_t)aval->total_bytes;
	}
	if (aval->total_packets != bval->total_packets) {
		return (int64_t)bval->total_packets - (int64_t)aval->total_packets;
	}
	return memcmp(a->key, b->key, sizeof(accum_key_t));
}


static void accum_dump(void)
{
	hash_linear_t *lin;
	size_t n_elts;
	size_t i;
	accum_value_t total_in;
	accum_value_t total_out;

	lin = hash_linearize(accum_hash);
	n_elts = hash_n_elts(accum_hash);
	memset(&total_in, 0, sizeof(total_in));
	memset(&total_out, 0, sizeof(total_out));
	for (i = 0; i < n_elts; ++i) {
		accum_value_t *val = lin[i].value;
		if (((accum_key_t *)lin[i].key)->outbound) {
			total_out.total_bytes += val->total_bytes;
			total_out.total_packets += val->total_packets;
		}
		else {
			total_in.total_bytes += val->total_bytes;
			total_in.total_packets += val->total_packets;
		}
	}
	qsort(lin, n_elts, sizeof(hash_linear_t), accum_compare);

	printf ("\n");
	printf("%9s %6s %3s %5s %s\n", "bytes/s", "pps", "dir", "proto", "details");
	if (direction != DIR_IN) {
		printf("%9.0f %6.0f out total\n",
			total_out.total_bytes / (double)secs_per_sample,
			total_out.total_packets / (double)secs_per_sample);
	}
	if (direction != DIR_OUT) {
		printf("%9.0f %6.0f in  total\n",
			total_in.total_bytes / (double)secs_per_sample,
			total_in.total_packets / (double)secs_per_sample);
	}
	n_elts = (n_elts > top_n) ? top_n : n_elts;
	for (i = 0; i < n_elts; ++i) {
		accum_value_t *val = lin[i].value;
		printf("%9.0f %6.0f %s\n",
			val->total_bytes / (double)secs_per_sample,
			val->total_packets / (double)secs_per_sample,
			print_accum_key(lin[i].key));
	}
	free(lin);
}


static void alrm_handler(int sig)
{
	pcap_breakloop(handle);
}


static void flodo(void)
{
	struct itimerval itv;
	struct sigaction sa;
	int rc;

	itv.it_interval.tv_usec = 0;
	itv.it_interval.tv_sec = secs_per_sample;
	itv.it_value = itv.it_interval;

        sa.sa_handler = alrm_handler;
        sigemptyset(&sa.sa_mask);
        sigaddset(&sa.sa_mask, SIGALRM);
        sa.sa_flags = 0;
        if (sigaction(SIGALRM, &sa, NULL)) {
                printf("sigaction error: %s\n", strerror(errno));
                exit(1);
        }
        if (setitimer(ITIMER_REAL, &itv, NULL) == -1) {
                printf("setitimer error: %s\n", strerror(errno));
                exit(1);
        }

	setvbuf(stdout, NULL, _IOFBF, BUFSIZ);
	for (;;) {
		accum_hash = hash_create(9);
		rc = pcap_loop(handle, -1, pkt_handler, NULL);
		if (rc != -2) break;
		accum_dump();
		fflush(stdout);
		if (one_shot) exit(EX_OK);
		hash_destroy_and_free_value(accum_hash);
	}
}


static void usage(void)
{
	printf("usage: %s [options] [libpcap packet filter]\n", progname);
	printf("options:\n");
	printf(" -1                   one-shot mode -- dump only one summary\n");
	printf(" -d direction         direction must be one of 'in', 'out', or 'either'\n");
	printf(" -g group_layer       group the packets at layer 2, 3 or 4\n");
	printf(" -i device            select capture device\n");
	printf(" -p                   enable promiscuous capture (opposite of tcpdump -p)\n");
	printf(" -s N_secs            dump totals every N seconds\n");
	printf(" -t top_N             display only the top N flows\n");
	printf("\nthe packet filtering language is the same as tcpdump(1)\n");
	exit(EX_USAGE);
}


static void pcap_die(const char *errbuf)
{
	fputs(errbuf, stderr);
	fputc('\n', stderr);
	exit(1);
}


static void parse_filter(char **argv, char *dev)
{
	size_t filter_len;
	char *filter_str;
	char *p;
	int i;
	bpf_u_int32 mask;
	bpf_u_int32 net;
	char errbuf[PCAP_ERRBUF_SIZE];

	// need the netmask for dum ipv4 broadcast filtering
	mask = 0;
	if (pcap_lookupnet(dev, &net, &mask, errbuf) == -1) {
		fprintf(stderr, "warning from pcap_lookupnet: %s\n", errbuf);
		// continue anyhow
	}

	// need to paste the arguments together with spaces
	filter_len = 0;
	for (i = optind; argv[i]; ++i) {
		filter_len += strlen(argv[i]) + 1;
	}
	filter_str = malloc(filter_len);
	p = filter_str;
	for (i = optind; argv[i]; ++i) {
		size_t len = strlen(argv[i]);
		memcpy(p, argv[i], len);
		p += len;
		*p++ = ' ';
	}
	p[-1] = '\0';

	if (pcap_compile(handle, &filter_program, filter_str, 1, mask) == -1) {
		fprintf(stderr, "error from pcap_compile: %s\n", pcap_geterr(handle));
		exit(1);
	}
	if (pcap_setfilter(handle, &filter_program) == -1) {
		fprintf(stderr, "error from pcap_setfilter: %s\n", pcap_geterr(handle));
		exit(1);
	}
	free(filter_str);
}



int main(int argc, char **argv)
{
	int c;
	char *dev = NULL;
	char errbuf[PCAP_ERRBUF_SIZE];
	int dl;
	int promiscuous = 0;

	progname = argv[0];

	while ((c = getopt(argc, argv, "1d:g:i:ps:t:")) != -1) {
		switch (c) {
		case '1':
			one_shot = 1;
			break;
		case 'd':
			if (strcasecmp(optarg, "in") == 0) {
				direction = DIR_IN;
			}
			else if (strcasecmp(optarg, "out") == 0) {
				direction = DIR_OUT;
			}
			else if (strcasecmp(optarg, "either") == 0) {
				direction = DIR_EITHER;
			}
			else {
				fprintf(stderr, "direction must be 'in', 'out' or 'either'\n");
				exit(EX_USAGE);
			}
			break;
		case 'g':
			group_layer = strtoul(optarg, 0, 0);
			if (group_layer < 2 || group_layer > 4) {
				fprintf(stderr, "group_layer must be 2, 3 or 4\n");
				exit(EX_USAGE);
			}
			break;
		case 'i':
			dev = optarg;
			break;
		case 'p':
			promiscuous = 1;
			break;
		case 's':
			secs_per_sample = strtoul(optarg, 0, 0);
			if (secs_per_sample < 1) {
				fprintf(stderr, "secs_per_sample must be positive\n");
				exit(EX_USAGE);
			}
			break;
		case 't':
			top_n = strtoul(optarg, 0, 0);
			if (top_n < 1) {
				fprintf(stderr, "top_n must be positive\n");
				exit(EX_USAGE);
			}
			break;
		default:
			usage();
		}
	}

	if (dev == NULL) {
		dev = pcap_lookupdev(errbuf);
		if (dev == NULL) pcap_die(errbuf);
	}

	local_mac_register();

	// the to_ms == 50 is necessary for systems which actually do
	// support the timeout ... choosing 50 seems like a good tradeoff
	// on measurement error vs. cpu utilisation.  if you're super
	// anal you could use to_ms == 1 ... but at 50 and default -s 5
	// you'd be looking at .05/5 = 1% error.
	handle = pcap_open_live(dev, 96, promiscuous, 50, errbuf);
	if (handle == NULL) pcap_die(errbuf);

	// verify it's a datalink we can handle...
	switch ((dl = pcap_datalink(handle))) {
	case DLT_EN10MB:
		break;
	default:
		fprintf(stderr, "device %s unknown datalink %s -- %s\n",
			dev, pcap_datalink_val_to_name(dl),
			pcap_datalink_val_to_description(dl));
		exit(1);
	}

	// set up a filter if one is given
	if (argv[optind] != NULL) {
		parse_filter(argv, dev);
	}

	flodo();

	return EX_OK;
}
