/*
paljompa jaxaa
*/
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <stdarg.h>
#include "th_config.h"
#include "th_util.h"

#define SET_MAX_BUFFERLEN	(4096)
#define SET_MAX_ITEMLEN		(64)


void th_config_error(char *fileName, int lineNum, const char *msgFmt, ...)
{
 va_list msgAp;
 fprintf(stderr, "CONFIG: Error in '%s', line #%i:\n", fileName, lineNum);
 
 va_start(msgAp, msgFmt);
 vfprintf(stderr, msgFmt, msgAp);
 va_end(msgAp);
}


t_config *th_config_new(void)
{
 t_config *pResult;
 
 /* Allocate memory for structure */
 pResult = (t_config *) calloc(1, sizeof(t_config));
 return pResult;
}


void th_config_free(t_config *pConfig)
{
 t_config_item *pCurr, *pNext;
 
 assert(pConfig);
 
 /* Go through the list of nodes */
 pCurr = pConfig->pItems;
 while (pCurr)
 	{
 	pNext = pCurr->pNext;

 	if (pCurr->itemData) free(pCurr->itemData);
 	free(pCurr);

 	pCurr = pNext;
 	}

 /* Free the config struct */ 	
 free(pConfig);
}


t_config_item *th_config_item_new(char *itemName, int itemType)
{
 t_config_item *pResult;

 /* Allocate memory for item */
 pResult = (t_config_item *) calloc(1, sizeof(t_config_item));
 if (pResult == NULL) return NULL;

 /* Set fields */
 th_strcpy(&pResult->itemName, itemName);
 pResult->itemType = itemType;

 /* Return result */
 return pResult;
}


void th_config_insert(t_config *pConfig, t_config_item *pNode)
{
 /* Insert it to the linked list */
 if (pConfig->pItems)
	{
	/* The first node's pPrev points to last node */
	pNode->pPrev = pConfig->pItems->pPrev;
	pConfig->pItems->pPrev->pNext = pNode;
	pConfig->pItems->pPrev = pNode;
	pNode->pNext = NULL;
	} else {
	pConfig->pItems = pNode;	/* First node */
	pNode->pPrev = pNode;
	pNode->pNext = NULL;		/* Next is NULL */
	}
}


t_config_item *th_config_set(t_config *pConfig, char *itemName, int itemType)
{
 t_config_item *pCurr, *pNode;
 assert(pConfig); 

 /* Check the itemlist if same item already exists */
 pNode = NULL;
 pCurr = pConfig->pItems;
 while (pCurr && !pNode)
 	{
	if ((strcasecmp(itemName, pCurr->itemName) == 0) &&
	    (pCurr->itemType == itemType))
		pNode = pCurr;
	
	pCurr = pCurr->pNext;
	}

 /* If not, create a new item */
 if (pNode == NULL)
 	{
	/* Allocate new node */
	pNode = th_config_item_new(itemName, itemType);
	th_config_insert(pConfig, pNode);
	}

 return pNode;
}


int th_config_read(char *fileName, t_config **pConfig)
{
 FILE *inFile;
 t_config_item *pItem;
 char inLine[SET_MAX_BUFFERLEN + 1],
      itemName[SET_MAX_ITEMLEN + 1],
      itemValue[SET_MAX_BUFFERLEN + 1];
 BOOL tmpBool;
 int itemType;
 int lineNum, linePos, i;

 assert(pConfig);

 /* Allocate memory for config, if required */
 if (*pConfig == NULL)
	*pConfig = th_config_new();

 if (*pConfig == NULL)
 	return -4;

 /* Open the file */
 if ((inFile = fopen(fileName, "ra")) == NULL)
	return -1;

 /* Parse */
 lineNum = 0;
 
 while (fgets(inLine, sizeof(inLine), inFile) != NULL)
 {
 lineNum++;
 linePos = 0;
 
 if ((strlen(inLine) > 1) && (inLine[0] != '#'))
 	{
 	/* Find start of item */
 	th_findnext(inLine, &linePos);

	/* Get item to tmpStr */
	i = 0;
	while (inLine[linePos] && !isspace(inLine[linePos]) && 
		(inLine[linePos] != '=') && (i < SET_MAX_ITEMLEN))
		itemName[i++] = inLine[linePos++];

	itemName[i++] = 0;

	/* Find assign */
	th_findnext(inLine, &linePos);
	if (inLine[linePos] == '=')
		{
		/* Get type of item */
		linePos++;
		th_findnext(inLine, &linePos);

		if (inLine[linePos] == '"')
			{
			/* String values are handled differently */
			itemType = ITEM_STRING;
			
			/* Get value */
			linePos++;
			i = 0;
			while (inLine[linePos] && (inLine[linePos] != '"') && (i < SET_MAX_BUFFERLEN))
				itemValue[i++] = inLine[linePos++];

			itemValue[i++] = 0;
			} else {
			/* Others are grabbed until next whitespace */
			if (isdigit(inLine[linePos]))
				itemType = ITEM_INT;
				else
				itemType = ITEM_BOOL;

			/* Get value */
			i = 0;
			while (inLine[linePos] && (!isspace(inLine[linePos])) && (i < SET_MAX_BUFFERLEN))
				itemValue[i++] = inLine[linePos++];

			itemValue[i++] = 0;
			}

		/* Check current list of items */
		pItem = th_config_set(*pConfig, itemName, itemType);

		if (pItem == NULL)
			{
			th_config_error(
				fileName, lineNum,
				"Error allocating memory for config item '%s' of type %i!\n",
				itemName, itemType);
			return -2;
			}

		/* Free old item's data, if necessary */
		if (pItem->itemData) free(pItem->itemData);		

		switch (itemType) {
		case ITEM_STRING:
			th_strcpy((char **) &pItem->itemData, itemValue);
			break;

		case ITEM_INT:
			pItem->itemData = malloc(sizeof(int));
			*((int *) pItem->itemData) = atoi(itemValue);
			break;
			
		case ITEM_BOOL:
			switch (itemValue[0]) {
			case 'y': case 'Y': case 't': case 'T': tmpBool = TRUE; break;
			case 'n': case 'N': case 'f': case 'F': tmpBool = FALSE; break;
			default:
				th_config_error(fileName, lineNum,
				"Invalid boolean value '%s' for '%s', assuming false.\n",
				itemValue, itemName);
				tmpBool = FALSE;
				break;
			}
			pItem->itemData = malloc(sizeof(BOOL));
			*((BOOL *) pItem->itemData) = tmpBool;
			
			break;
		
		default:
			th_config_error(fileName, lineNum,
			"Invalid item type (%i), internal error.\n", itemType);
			exit(3);
			break;
		}

		} /* if '=' */
 	} /* strlen() */
 }
 
 /* Close files */
 fclose(inFile);

 return 0;
}


t_config_item * th_config_get(t_config *pConfig, char *itemName, int itemType)
{
 t_config_item *pCurr, *pItem;
 assert(pConfig);

 pItem = NULL;
 pCurr = pConfig->pItems;
 while (pCurr && !pItem)
 	{
	if ((strcasecmp(pCurr->itemName, itemName) == 0) &&
	    (pCurr->itemType == itemType))
		pItem = pCurr;
	
	pCurr = pCurr->pNext;
	}

 return pItem;
}


int th_config_get_int(t_config *pConfig, char *itemName, int defValue)
{
 t_config_item *pItem;
 assert(pConfig);

 if ((pItem = th_config_get(pConfig, itemName, ITEM_INT)) != NULL)
 	return (*(int *) pItem->itemData);
 	else
 	return defValue;
}


char *th_config_get_str(t_config *pConfig, char *itemName, char *defValue)
{
 t_config_item *pItem;
 assert(pConfig);

 if ((pItem = th_config_get(pConfig, itemName, ITEM_STRING)) != NULL)
 	return ((char *) pItem->itemData);
 	else
 	return defValue;
}


BOOL th_config_get_bool(t_config *pConfig, char *itemName, BOOL defValue)
{
 t_config_item *pItem;
 assert(pConfig);

 if ((pItem = th_config_get(pConfig, itemName, ITEM_BOOL)) != NULL)
 	return (*(BOOL *) pItem->itemData);
 	else
 	return defValue;
}


