/* Notes *//*{{{C}}}*//*{{{*/
/*

This modules implements generic functions which work on a gap buffer of
character and meta-characters.  A gap buffer is a common way for to
store text in text editors and it works like this:

| text before gap |   ... gap ... | text after gap |

The start/end of the gap reflects the cursor position in the text.
Inserting is done by decreasing the gap, deleting by increasing it.  The
stored text consists of characters and fold marks.  The special
character '\0' quotes a following fold mark or nul character, using the
constants NUL, FOLDSTART and FOLDEND.  Special care must be taken not to
have more than one fold mark per line and to keep the fold structure 
when using the functions from this module.

My personal experience from hacking on the folding editor Origami is, 
that a fancy data structure (double-linked n-tree) is very efficient, as
even with a deeply folded 3 MB file there is no slowdown.  This can not 
be expected from using a gap buffer.  The reason why I use it never the 
less, is that a fancy data structure makes it nearly impossible to have 
correct and elegant primitives which work on the raw text.  In fact, 
a significant slow down can be felt from lots of node-traversing which 
happens when deleting bigger amounts of text.  This is the reason, why 
most versions of Origami enforce using folds for everything.  While this
seemed elegant in the beginning, after a longer time I have found that 
folds are no universal solution to everything.

As a result, this implementation is slower for basic operations (opening
and closing folds, screen display), faster for complex operations (block
operations, loading, saving) and tries to offer the best of both worlds
(traditional text editor and folding editor).

*/
/*}}}*/
/* #includes *//*{{{*/
#ifndef NO_POSIX_SOURCE
#undef  _POSIX_SOURCE
#define _POSIX_SOURCE   1
#undef  _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 2
#endif

#ifdef DMALLOC
#include "dmalloc.h"
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#ifdef OLD_REALLOC
#define realloc(s,l) myrealloc(s,l)
#endif
#include <unistd.h>

#include "buffer.h"
#include "misc.h"

#include <string.h>
/*}}}*/
/* #defines *//*{{{*/
#define DEF_BUFLEN 1024

#if 0
#define ASSERT_CONSISTENCY(b) \
assert(b!=(struct Buffer*)0); \
assert(b->buffer!=(char*)0); \
assert(b->gapstart>=b->buffer); \
assert(b->gapend<=b->buffer+b->size-1)
#else
#define ASSERT_CONSISTENCY(b)
#endif

#define BF_FORWARDOK(b) (b->gapend<b->buffer+b->size-1)

#define BF_FORWARD(b) \
switch (++b->curch,*(b->gapstart++)=*(++b->gapend)) \
{ \
  case '\0': \
  { \
    *(b->gapstart++)=*(++b->gapend); \
    break; \
  } \
  case '\n': \
  { \
    ++b->cury; \
    break; \
  } \
}

#define BF_BACKWARDOK(b) (b->gapstart>b->buffer)

#define BF_BACKWARD(b) \
do \
{ \
  *(b->gapend--)=*(--b->gapstart); \
  if (b->gapstart-1>=b->buffer && *(b->gapstart-1)=='\0') \
  { \
    *(b->gapend--)=*(--b->gapstart); \
  } \
  if (*(b->gapend+1)=='\n') --b->cury; \
  --b->curch; \
} while(0);
/*}}}*/

/* variables *//*{{{*/
struct Language language[LANGUAGES]=
{
  { "C", "/*", "*/" },
  { "script", "#", "" },
  { "mail", "# ", "" },
  { "X11", "!", "" },
  { "roff", "\\\"", "" },
  { "SGML", "<!--", "-->" },
  { "Occam", "--", "" },
  { "C++", "//", "" },
  { "TeX", "%", "" },
  { "Texinfo", "@c ", "" },
  { "Lisp", ";", "" },
  { "Pascal", "(*", "*)" },
  { "", "", "" },
  { (const char*)0, (const char*)0, (const char*)0 },
};
/*}}}*/

/* bf_alloc      -- allocate new buffer *//*{{{*/
struct Buffer *bf_alloc(void)
{
  struct Buffer *b;

  if ((b=malloc(sizeof(struct Buffer)))==(struct Buffer*)0) return b;
  b->size=DEF_BUFLEN;
  if ((b->buffer=malloc(DEF_BUFLEN+1))==(char*)0)
  {
    free(b);
    return (struct Buffer*)0;
  }
  *(b->buffer+DEF_BUFLEN)='\0';
  b->gapstart=b->buffer;
  b->gapend=b->buffer+b->size-1;
  b->cury=0;
  b->mode=0;
  b->name=(char*)0;
  b->curch=0;
  b->msz=0;
  b->refcnt=0;
  b->lang=&language[sizeof(language)/sizeof(language[0])-2];
  ASSERT_CONSISTENCY(b);
  return b;
}
/*}}}*/
/* bf_free       -- free buffer *//*{{{*/
void bf_free(struct Buffer *b)
{
  ASSERT_CONSISTENCY(b);
  if (b->name) free(b->name);
  free(b->buffer);
  b->buffer=b->gapstart=b->gapend=(char*)0;
  b->msz=0;
  free(b);
}
/*}}}*/
/* bf_empty      -- empty buffer */ /*{{{*/
int bf_empty(struct Buffer *b)
{
  int i;
  if (b->mode&READONLY) return 0;
  b->gapstart=b->buffer;
  b->gapend=b->buffer+b->size-1;
  b->cury=0;
  b->mode=0;
  b->curch=0;
  for (i=0; i<b->msz; ++i) b->mark[i]=0;
  b->lang=&language[sizeof(language)/sizeof(language[0])-2];
  ASSERT_CONSISTENCY(b);
  return 1;
}
/*}}}*/
/* bf_forward    -- move one character forward *//*{{{*/
int bf_forward(struct Buffer *b)
{
  ASSERT_CONSISTENCY(b);
  if (BF_FORWARDOK(b)) { BF_FORWARD(b); return 1; }
  else return 0;
}
/*}}}*/
/* bf_backward   -- move one character backward *//*{{{*/
int bf_backward(struct Buffer *b)
{
  ASSERT_CONSISTENCY(b);
  if (BF_BACKWARDOK(b)) { BF_BACKWARD(b); return 1; }
  else return 0;
}
/*}}}*/
/* bf_insert     -- insert one character *//*{{{*/
int bf_insert(struct Buffer *b, char c, char mark)
{
  int i;

  ASSERT_CONSISTENCY(b);
  if (b->mode&READONLY) return 0;
  if (b->gapstart>=b->gapend-2)
  /* increase gap *//*{{{*/
  {
    char *new_buffer;
    size_t new_size,pregap_length,postgap_length;

    new_size=b->size*2;
    if ((new_buffer=malloc(new_size+1))==(char*)0) return 0;
    *(new_buffer+new_size)='\0';
    pregap_length=b->gapstart-b->buffer;
    postgap_length=b->buffer+b->size-1-b->gapend;
    if (pregap_length!=(size_t)0)
    {
      memcpy(new_buffer,b->buffer,pregap_length);
    }
    if (postgap_length!=(size_t)0)
    {
      memcpy(new_buffer+new_size-postgap_length,b->gapend+1,postgap_length);
    }
    free(b->buffer);
    b->buffer=new_buffer;
    b->size=new_size;
    b->gapstart=b->buffer+pregap_length;
    b->gapend=b->buffer+b->size-1-postgap_length;
  }
  /*}}}*/
  for (i=0; i<b->msz; ++i) if (b->gapstart-b->buffer<=b->mark[i]) ++b->mark[i];
  *(b->gapstart++)=c;
  switch (c)
  {
    case '\0':
    {
      assert(mark!='\0');
      *(b->gapstart++)=mark;
      break;
    }
    case '\n':
    {
      ++b->cury;
      break;
    }
  }
  b->mode|=CHANGED;
  ++b->curch;
  return 1;
}
/*}}}*/
/* bf_delete     -- delete one character *//*{{{*/
int bf_delete(struct Buffer *b)
{
  int i;

  ASSERT_CONSISTENCY(b);
  if (b->mode&READONLY) return 0;
  if (b->gapstart==b->buffer) return 0;
  else
  {
    for (i=0; i<b->msz; ++i) if (b->gapstart-b->buffer<=b->mark[i]) --b->mark[i];
    --b->gapstart;
    if (b->gapstart-1>=b->buffer && *(b->gapstart-1)=='\0')
    {
      --b->gapstart;
    }
    else if (*(b->gapstart)=='\n')
    {
      --b->cury;
    }
    b->mode|=CHANGED;
    --b->curch;
    return 1;
  }
}
/*}}}*/
/* bf_lchar      -- return character before beginning of gap *//*{{{*/
int bf_lchar(struct Buffer *b, char *c, char *mark)
{
  ASSERT_CONSISTENCY(b);
  if (b->gapstart==b->buffer) return 0;
  if (b->gapstart-2>=b->buffer && *(b->gapstart-2)=='\0')
  {
    *c='\0';
    *mark=*(b->gapstart-1);
  }
  else *c=*(b->gapstart-1);
  return 1;
}
/*}}}*/
/* bf_rchar      -- return character after end of gap *//*{{{*/
int bf_rchar(struct Buffer *b, char *c, char *mark)
{
  ASSERT_CONSISTENCY(b);
  if (b->gapend==b->buffer+b->size-1) return 0;
  if ((*c=*(b->gapend+1))=='\0') *mark=*(b->gapend+2);
  return 1;
}
/*}}}*/
/* bf_name       -- set buffer name *//*{{{*/
int bf_name(struct Buffer *b, const char *name)
{
  char *n;

  if (b->name) free(b->name);
  if ((n=malloc(strlen(name)+1))==(char *)0) return 0;
  strcpy(n,name);
  b->name=n;
  b->mode|=CHANGED;
  b->mode&=~READONLY;
  return 1;
}
/*}}}*/
/* bf_lang       -- get/set buffer language *//*{{{*/
struct Language *bf_lang(struct Buffer *b, struct Language *l)
{
  if (l)
  {
    if (b->mode&READONLY) return (struct Language*)0;
    b->lang=l;
    b->mode|=CHANGED;
  }
  return b->lang;
}
/*}}}*/
/* bf_load       -- load buffer from file *//*{{{*/
int bf_load(struct Buffer *b, const char *cmd, const char *file, int filefd, const char *beg, const char *end)
{
  /* variables *//*{{{*/
  int fd[2],eof,readonly=0;
  char buf[1024];
  char *line,*linerun,*bufrun=(char*)0;
  size_t linesize,remaining,begsz,endsz;
  enum { LANGUAGE, FOLDS } search_for;
  char *opening=(char*)0,*closing=(char*)0;
  size_t openingsz=-1,closingsz=-1;
  pid_t pid=0;
  int status;
  /*}}}*/

  assert(b!=(struct Buffer*)0);
  assert(beg!=(const char*)0);
  assert(end!=(const char*)0);
  begsz=strlen(beg);
  endsz=strlen(end);
  search_for=LANGUAGE;
  linesize=(size_t)256;
  remaining=(size_t)0;
  if ((line=malloc(linesize))==(char*)0) return 0;
  linerun=line;
  if (cmd) /*{{{*/
  {
    pipe(fd);
    switch (pid=fork())
    {
      case 0:
      {
        close(0);
        close(1);
        close(fd[0]);
        open(file,O_RDONLY);
        dup(fd[1]);
        close(fd[1]);
        execlp(getshell(),getshell(),"-c",cmd,(char*)0);
        _exit(127);
        break;
      }
      case -1:
      {
        close(fd[0]);
        close(fd[1]);
        break;
      }
      default: close(fd[1]);
    }    
  }
  /*}}}*/
  else /*{{{*/
  {
    struct stat statbuf;

    if (file)
    {
      if ((fd[0]=open(file,O_RDONLY))==-1) return 0;
      if (access(file,W_OK)==-1 && errno==EACCES) readonly=1;
    }
    else { file="(stdin)"; fd[0]=filefd; }
    if (fstat(fd[0],&statbuf)==-1) return 0;
    if (S_ISDIR(statbuf.st_mode)) { errno=EISDIR; return 0; }
  }
  /*}}}*/
  eof=0;
  do
  {
    /* enlarge read buffer if needed *//*{{{*/
    if (linerun==line+linesize)
    {
      if ((line=realloc(line,linesize+256))==(char*)0)
      /* can not allocate more memory, close file and return failure *//*{{{*/
      {
        int oerrno;

        oerrno=errno;
        free(line);
        close(fd[0]);
        if (cmd) while (wait(&status)!=pid);
        errno=oerrno;
        return 0;
      }
      /*}}}*/
      else
      {
        linerun=line+linesize;
        linesize+=256;
      }    
    }
    /*}}}*/
    /* read new block if needed *//*{{{*/
    if (remaining==0 && (remaining=read(fd[0],bufrun=buf,sizeof(buf)))<=0) eof=1;
    /*}}}*/
    --remaining;
    if ((eof || (*linerun++=*bufrun++)=='\n') && linerun>line) /* process line */
    {
      if (search_for==LANGUAGE) /* search for magic mark *//*{{{*/
      {
        const char *found_beg,*found_end;

        if ((found_beg=mymemmem(line,linerun-line,beg,begsz))!=(char*)0 && (found_end=mymemmem(found_beg+begsz,linerun-found_beg-begsz,end,endsz))!=(char*)0)
        {
          struct Language *langrun;

          /* search for given language between begin/end */ /*{{{*/
          for (langrun=language; langrun->name!=(const char*)0; ++langrun)
          {
            if (strncmp(found_beg+begsz,langrun->name,strlen(langrun->name))==0)
            {
              /* variables */ /*{{{*/
              const char *found;
              /*}}}*/

              /* determine opening and closing fold marks */ /*{{{*/
              if (opening) free(opening);
              if (closing) free(closing);
              opening=malloc((openingsz=strlen(langrun->beg)+begsz+strlen(langrun->end))+1);
              strcpy(opening,langrun->beg);
              strcpy(opening+strlen(langrun->beg),beg);
              strncpy(opening+strlen(langrun->beg)+begsz,langrun->end,strlen(langrun->end));
              closing=malloc((closingsz=strlen(langrun->beg)+endsz+strlen(langrun->end))+1);
              strcpy(closing,langrun->beg);
              strcpy(closing+strlen(langrun->beg),end);
              strncpy(closing+strlen(langrun->beg)+endsz,langrun->end,strlen(langrun->end));
              /*}}}*/
              /* now check if this is also a opening fold */
              if ((found=mymemmem(found_end+endsz,linerun-found_end-endsz,opening,openingsz))!=(char*)0) /* good */ /*{{{*/
              {
                const char *s;

                for (s=line; s<found_beg-strlen(langrun->beg); ++s) bf_insert(b,*s,NUL);
                for (s=found_end+endsz+strlen(langrun->end); s<found; ++s) bf_insert(b,*s,NUL);
                bf_insert(b,'\0',FOLDSTART);
                for (s=found+openingsz; s<linerun; ++s) bf_insert(b,*s,NUL);
                search_for=FOLDS;
                b->lang=langrun;
                break;
              }
              /*}}}*/
            }
          }
          if (langrun->name==(const char*)0) /* not found, so this was just a regular line */ /*{{{*/
          {
            char *s;

            for (s=line; s<linerun; ++s) bf_insert(b,*s,NUL);
          }
          /*}}}*/
          /*}}}*/
        }
        else
        {
          const char *s;

          for (s=line; s<linerun; ++s) bf_insert(b,*s,NUL);    
        }
      }
      /*}}}*/
      else /* search for folds *//*{{{*/
      {
        const char *found,*s;
        
        if ((found=mymemmem(line,linerun-line,opening,openingsz))!=(char*)0)
        {
          for (s=line; s<found; ++s) bf_insert(b,*s,NUL);
          bf_insert(b,'\0',FOLDSTART);
          for (s=found+openingsz; s<linerun; ++s) bf_insert(b,*s,NUL);
        }
        else if ((found=mymemmem(line,linerun-line,closing,closingsz))!=(char*)0)
        {
          for (s=line; s<found; ++s) bf_insert(b,*s,NUL);
          bf_insert(b,'\0',FOLDEND);
          for (s=found+closingsz; s<linerun; ++s) bf_insert(b,*s,NUL);
        }
        else
        {
          for (s=line; s<linerun; ++s) bf_insert(b,*s,NUL);
        }
      }
      /*}}}*/
      linerun=line;
    }
  } while (!eof);
  close(fd[0]);
  if (cmd) while (wait(&status)!=pid);
  free(line);
  if (opening) free(opening);
  if (closing) free(closing);
  if (readonly) b->mode|=READONLY;
  return 1;
}
/*}}}*/
/* bf_save       -- save buffer into file *//*{{{*/
int bf_save(struct Buffer *b, const char *file, const char *beg, const char *end, unsigned int *chars)
{
  /* variables *//*{{{*/
  char *gapstart;
  char *bufp;
  char buf[1024];
  char c,mark;
  const char *writestr;
  int fd;
  char *magic,*begfold,*endfold;
  int first=1;
  /*}}}*/

  assert(file!=(const char*)0);
  assert(beg!=(const char*)0);
  assert(end!=(const char*)0);
  if (chars!=(unsigned int*)0) *chars=0;
  if ((begfold=malloc(strlen(b->lang ? b->lang->beg : "")+strlen(beg)+strlen(b->lang ? b->lang->end : "")+1))==(char*)0) return 0;
  strcpy(begfold,b->lang ? b->lang->beg : "");
  strcat(begfold,beg);
  if (b->lang) strcat(begfold,b->lang->end);
  if ((magic=malloc(strlen(b->lang ? b->lang->beg : "")+strlen(beg)+strlen(b->lang ? b->lang->name : "")+strlen(end)+strlen(b->lang ? b->lang->end : "")+strlen(begfold)))==(char*)0)
  {
    free(begfold);
    return 0;
  }
  strcpy(magic,b->lang ? b->lang->beg : "");
  strcat(magic,beg);
  if (b->lang) strcat(magic,b->lang->name);
  strcat(magic,end);
  if (b->lang) strcat(magic,b->lang->end);
  strcat(magic,begfold);
  if ((endfold=malloc(strlen(b->lang ? b->lang->beg : "")+strlen(end)+strlen(b->lang ? b->lang->end : "")))==(char*)0)
  {
    free(begfold);
    free(magic);
    return 0;
  }
  strcpy(endfold,b->lang ? b->lang->beg : "");
  strcat(endfold,end);
  if (b->lang) strcat(endfold,b->lang->end);
  if ((fd=open(file,O_WRONLY|O_CREAT|O_TRUNC,0666))==-1) return 0;
  gapstart=b->gapstart;
  writestr=(const char*)0;
  bufp=buf;
  bf_begin(b);
  while (bf_rchar(b,&c,&mark))
  {
    if (c=='\0') switch (mark)
    {
      case NUL: *bufp++='\0'; break;
      case FOLDSTART: writestr=(first ? magic : begfold); first=0; break;
      case FOLDEND: writestr=endfold; break;
      default: assert(0);
    }
    else *bufp++=c;
    do
    {
      if (writestr && *writestr) *bufp++=*writestr++;
      if (bufp==buf+sizeof(buf))
      {
        if (write(fd,buf,sizeof(buf))!=sizeof(buf))
        {
          int oerrno;
      
          oerrno=errno;
          close(fd);
          free(begfold);
          free(magic);
          free(endfold);
          errno=oerrno;
          return 0;
        }
        else if (chars!=(unsigned int*)0) *chars+=sizeof(buf);
        bufp=buf;
      }
    } while (writestr && *writestr);
    BF_FORWARD(b);
  }
  while (gapstart<b->gapstart) BF_BACKWARD(b);
  free(begfold);
  free(magic);
  free(endfold);
  if (bufp>buf)
  {
    if (write(fd,buf,bufp-buf)!=bufp-buf)
    {
      int oerrno;
      
      oerrno=errno;
      close(fd);
      errno=oerrno;
      return 0;
    }
    else if (chars!=(unsigned int*)0) *chars+=(bufp-buf);
  }
  if (close(fd)==-1) return 0;
  b->mode&=~CHANGED;
  return 1;
}
/*}}}*/
/* bf_write      -- write buffer without fold marks into file *//*{{{*/
int bf_write(struct Buffer *b, const char *file, unsigned int *chars)
{
  /* variables *//*{{{*/
  char *gapstart;
  char *bufp;
  char buf[1024];
  char c,mark;
  int fd;
  /*}}}*/

  assert(file!=(const char*)0);
  if (chars!=(unsigned int*)0) *chars=0;
  if ((fd=open(file,O_WRONLY|O_CREAT|O_TRUNC,0666))==-1) return 0;
  gapstart=b->gapstart;
  bufp=buf;
  bf_begin(b);
  while (bf_rchar(b,&c,&mark))
  {
    if (c=='\0') switch (mark)
    {
      case NUL: *bufp++='\0'; break;
      case FOLDSTART: break;
      case FOLDEND: break;
      default: assert(0);
    }
    else *bufp++=c;
    if (bufp==buf+sizeof(buf))
    {
      if (write(fd,buf,sizeof(buf))!=sizeof(buf))
      {
        int oerrno;
      
        oerrno=errno;
        close(fd);
        errno=oerrno;
        return 0;
      }
      else if (chars!=(unsigned int*)0) *chars+=sizeof(buf);
      bufp=buf;
    }
    BF_FORWARD(b);
  }
  while (gapstart<b->gapstart) BF_BACKWARD(b);
  if (bufp>buf)
  {
    if (write(fd,buf,bufp-buf)!=bufp-buf)
    {
      int oerrno;
      
      oerrno=errno;
      close(fd);
      errno=oerrno;
      return 0;
    }
    else if (chars!=(unsigned int*)0) *chars+=(bufp-buf);
  }
  if (close(fd)==-1) return 0;
  b->mode&=~CHANGED;
  return 1;
}
/*}}}*/
/* bf_linestart  -- go to start of current line *//*{{{*/
int bf_linestart(struct Buffer *b, int realstart)
{
  char c,mark;
  
  if (realstart)
  {
    if (b->gapstart==b->buffer) return 0;
    while (bf_lchar(b,&c,&mark) && c!='\n' && BF_BACKWARDOK(b)) BF_BACKWARD(b);
  }
  else
  {
    int blanks;

    blanks=0;
    if (bf_lchar(b,&c,&mark) && c!='\n')
    {
      if (bf_rchar(b,&c,&mark) && c!=' ') blanks=1;
      while (bf_lchar(b,&c,&mark) && c!='\n' && BF_BACKWARDOK(b))
      {
        BF_BACKWARD(b);
        if (c==' ' && blanks) ++blanks; else blanks=0;
      }
    }
    if (blanks==0) while (bf_rchar(b,&c,&mark) && c==' ') BF_FORWARD(b);
  }
  return 1;
}
/*}}}*/
/* bf_lineend    -- go to end of current line *//*{{{*/
int bf_lineend(struct Buffer *b, int realend)
{
  char c,mark;
  
  while (bf_rchar(b,&c,&mark) && c!='\n') BF_FORWARD(b);
  return 1;
}
/*}}}*/
/* bf_isfold     -- is this a fold? *//*{{{*/
int bf_isfold(struct Buffer *b, int fold, int where)
{
  char *gapstart,c,mark;

  gapstart=b->gapstart;
  if (where&LEFT)
  {
    while (bf_lchar(b,&c,&mark))
    {
      if (c=='\0' && mark&fold)
      {
        while (b->gapstart<gapstart) BF_FORWARD(b);
        return 1;
      }
      else if (c=='\n') break;
      if (BF_BACKWARDOK(b)) BF_BACKWARD(b);
    }
    while (b->gapstart<gapstart) BF_FORWARD(b);
  }
  if (where&RIGHT)
  {
    while (bf_rchar(b,&c,&mark))
    {
      if (c=='\0' && mark&fold)
      {
        while (b->gapstart>gapstart) BF_BACKWARD(b);
        return 1;
      }
      else if (c=='\n') break;
      BF_FORWARD(b);
    }   
    while (b->gapstart>gapstart) BF_BACKWARD(b);
  }
  return 0;
}
/*}}}*/
/* bf_foldstart  -- goto start of this fold *//*{{{*/
int bf_foldstart(struct Buffer *b, int level, unsigned int *linesback)
{
  char *gapstart,c,mark;
  int entrylevel;
  int firstline=level;
        
  entrylevel=level;
  gapstart=b->gapstart;
  if (linesback) *linesback=0;
  do
  {
    if (!BF_BACKWARDOK(b) || bf_lchar(b,&c,&mark)==0)
    {
      while (b->gapstart<gapstart) BF_FORWARD(b);
      return 0;
    }
    BF_BACKWARD(b);
    if (c=='\0' && mark==FOLDEND && !firstline) ++level;
    else if (c=='\0' && mark==FOLDSTART && !firstline) --level;
    else if (c=='\n')
    {
      firstline=0;
      if (level==entrylevel && linesback) ++*linesback;
    }
  } while (!(c=='\0' && mark==FOLDSTART && level==0));
  return 1;
}
/*}}}*/
/* bf_foldend    -- goto end of this fold *//*{{{*/
int bf_foldend(struct Buffer *b, int level, unsigned int *linesforward)
{
  char *gapstart,c,mark;
  int entrylevel;
        
  if (linesforward) *linesforward=0;
  entrylevel=level;
  gapstart=b->gapstart;
  if (!BF_FORWARDOK(b)) return 0;
  do
  {
    BF_FORWARD(b);
    if (bf_lchar(b,&c,&mark)==0)
    {
      while (b->gapstart>gapstart) BF_BACKWARD(b);
      return 0;
    }
    if (c=='\0' && mark==FOLDEND) --level;
    else if (c=='\0' && mark==FOLDSTART) ++level;
    else if (c=='\n' && level==entrylevel && linesforward) ++*linesforward;
    if (c=='\0' && mark==FOLDEND && level==0) return 1;
  } while (BF_FORWARDOK(b));
  return 1;
}
/*}}}*/
/* bf_begin      -- goto begin of buffer *//*{{{*/
void bf_begin(struct Buffer *b)
{
  while (BF_BACKWARDOK(b)) BF_BACKWARD(b);
}
/*}}}*/
/* bf_end        -- goto end of buffer *//*{{{*/
void bf_end(struct Buffer *b)
{
  while (BF_FORWARDOK(b)) BF_FORWARD(b);
}
/*}}}*/
/* bf_gotoline   -- goto line of buffer *//*{{{*/
int bf_gotoline(struct Buffer *b, int line)
{
  char *gapstart;

  gapstart=b->gapstart;
  bf_begin(b);
  while (b->cury!=line && bf_forward(b));
  if (b->cury==line) return 1;
  while (b->gapstart<gapstart) bf_forward(b);
  while (b->gapstart>gapstart) bf_backward(b);
  return 0;
}
/*}}}*/
/* bf_strfsearch -- goto line of forwards searched string *//*{{{*/
int bf_strfsearch(struct Buffer *b, const char *needle, size_t needlesize)
{
  char *found;

  if ((b->buffer+b->size-b->gapend-1)>0 && (found=mymemmem(b->gapend+1,b->buffer+b->size-b->gapend-1,needle,needlesize)))
  {
    while (b->gapend<found) BF_FORWARD(b);
    if (b->gapend==found) bf_backward(b);
    return 1;
  }
  else return 0;
}
/*}}}*/
/* bf_regfsearch -- goto line of forwards searched string *//*{{{*/
int bf_regfsearch(struct Buffer *b, const regex_t *needle)
{
  regmatch_t found;
  int forward;
  int failed;
  char c,mark;

  forward=0;
  failed=REG_NOMATCH;
  while ((failed=regexec(needle,b->gapend+1,1,&found,((b->gapstart==b->buffer || (bf_lchar(b,&c,&mark) && c=='\n')) ? 0 : REG_NOTBOL)|( b->gapend+strlen(b->gapend)<b->buffer+b->size ? REG_NOTEOL : 0 )))==REG_NOMATCH)
  {
    if (bf_rchar(b,&c,&mark))
    {
      if (c=='\0')
      {
        bf_forward(b);
        ++forward;
      }
      else do
      {
        bf_forward(b);
        ++forward;
      } while (bf_rchar(b,&c,&mark) && c!='\0');
    }
    else break;
  }
  if (failed==0)
  {
    while (found.rm_so>0)
    {
      bf_forward(b);
      --found.rm_so;
    }
    return 1;
  }
  else
  {
    while (forward) { bf_backward(b); --forward; }
    return 0;
  }
}
/*}}}*/
/* bf_strbsearch -- goto line of backwards searched string *//*{{{*/
int bf_strbsearch(struct Buffer *b, const char *needle, size_t needlesize)
{
  int back=0;

  while (memcmp(b->gapend+1,needle,needlesize))
  {
    if (BF_BACKWARDOK(b)) { ++back; BF_BACKWARD(b); }
    else { while (back>0) { --back; BF_FORWARD(b); } return 0; }
  }
  return 1;
}
/*}}}*/
/* bf_regbsearch -- goto line of backwards searched string *//*{{{*/
int bf_regbsearch(struct Buffer *b, const regex_t *needle)
{
  regmatch_t found;
  int backward;
  int failed;
  char c,mark;

  backward=0;
  failed=REG_NOMATCH;
  while ((failed=regexec(needle,b->gapend+1,1,&found,((b->gapstart==b->buffer || (bf_lchar(b,&c,&mark) && c=='\n')) ? 0 : REG_NOTBOL)|( b->gapend+strlen(b->gapend)<b->buffer+b->size ? REG_NOTEOL : 0 )))==REG_NOMATCH || found.rm_so>0)
  {
    if (bf_backward(b)) ++backward;
    else break;
  }
  if (failed==0)
  {
    return 1;
  }
  else
  {
    while (backward) { bf_forward(b); --backward; }
    return 0;
  }
}
/*}}}*/
/* bf_tomark     -- goto alternate mark *//*{{{*/
void bf_tomark(struct Buffer *b, int to)
{
  assert(to<b->msz);
  while (b->curch<b->mark[to] && BF_FORWARDOK(b)) BF_FORWARD(b);
  while (b->curch>b->mark[to] && BF_BACKWARDOK(b)) BF_BACKWARD(b);
}
/*}}}*/
/* bf_mark       -- set mark *//*{{{*/
int bf_mark(struct Buffer *b, int mark)
{
  if (mark<0)
  {
    for (mark=0; b->markused[mark] && mark<b->msz; ++mark);
    if (mark==b->msz)
    {
      assert(b->msz<BUFMARKS);
      ++b->msz;
    }
    b->markused[mark]=1;
  }
  b->mark[mark]=b->curch;
  return mark;
}
/*}}}*/
/* bf_unmark     -- unset mark */ /*{{{*/
void bf_unmark(struct Buffer *b, int mark)
{
  assert(mark>=0);
  assert(b->markused[mark]);
  b->markused[mark]=0;
}
/*}}}*/
