/*****************************************************************
 * fbconvolve.c: FBM Release 1.2 18-Apr-93 Michael Mauldin
 *
 * Copyright (C) 1989-1993 by Michael Mauldin.  Permission is granted
 * to use this file in whole or in part for any purpose, educational,
 * recreational or commercial, provided that this copyright notice
 * is retained unchanged.  This software is available to all free of
 * charge by anonymous FTP and in the UUNET archives.
 *
 * fbconvolve.c: Sharpen an image by using a Laplacian edge enhancement
 *
 * USAGE
 *	% fbconvolve [ flags ] arguments
 *
 * EDITLOG
 *	LastEditDate = Mon Jun 25 00:04:40 1990 - Michael Mauldin
 *	LastFileName = /usr2/mlm/src/misc/fbm/fbconvolve.c
 *
 * HISTORY
 * 18-Apr-93  Michael Mauldin (mlm) at Carnegie-Mellon University
 *	Created from fbsharp.c
 *****************************************************************/

# include <stdio.h>
# include <math.h>
# include "fbm.h"

# define USAGE "fbconvolve [ -<type> ] [ -c ] <n> <k00> <k01> <k02> ... < 8bit > 8bit"

#ifndef lint
static char *fbmid =
"$FBM fbconvolve.c <1.2> 18-Apr-93 (C) 1989-1993 by Michael Mauldin, source \
code available free from MLM@CS.CMU.EDU and from UUNET archives$";
#endif

main (argc, argv)
char *argv[];
{ int w, h, k, n, absval=1, sobel=0, sobel5=0;
  double **kernel;
  FBM input, output;
  int outtype = DEF_8BIT;
  register int i, j;

  /* Get the options */
  while (--argc > 0 && (*++argv)[0] == '-')
  { while (*++(*argv))
    { switch (**argv)
      { 
	case 'A':	outtype = FMT_ATK; break;
	case 'B':	outtype = FMT_FACE; break;
	case 'F':	outtype = FMT_FBM; break;
	case 'G':	outtype = FMT_GIF; break;
	case 'I':	outtype = FMT_IFF; break;
	case 'J':	outtype = FMT_JPEG; break;
	case 'L':	outtype = FMT_LEAF; break;
	case 'M':	outtype = FMT_MCP; break;
	case 'P':	outtype = FMT_PBM; break;
	case 'R':	outtype = FMT_RLE; break;
	case 'S':	outtype = FMT_SUN; break;
	case 'T':	outtype = FMT_TIFF; break;
	case 'X':	outtype = FMT_X11; break;
	case 'Z':	outtype = FMT_PCX; break;
	case 'c':	absval = 0; break;
	case 's':	sobel++; break;
	case '5':	sobel5++; break;
	default:        fprintf (stderr, "%s\n", USAGE);
                        exit (1);
      }
    }
  }

  /* Clear the memory pointers so alloc_fbm wont be confused */
  input.cm  = input.bm  = (unsigned char *) NULL;
  output.cm = output.bm = (unsigned char *) NULL;

  /* Read the image first */
  if (read_bitmap (&input, (char *) NULL))
  {
    if (input.hdr.bits != 8 || input.hdr.physbits != 8)
    { fprintf (stderr,
	       "Can't handle images with %d bits and %d physbits per pixel\n",
	       input.hdr.bits, input.hdr.physbits);
      exit (1);
    }

    /* Determine output height & width (oh*ow <= size) */
    h = input.hdr.rows;
    w = input.hdr.cols;
    k = input.hdr.planes;
  }
  else
  { exit (1);
  }

  /* Sobel 5x5 doesnt have any other arguments */
  if (sobel5)
  {
    fprintf (stderr,
	     "Sobel 5x5 \"%s\" [%dx%dx%d]\n",
	     input.hdr.title[0] ? input.hdr.title : "(untitled)",
	     w, h, k);

    /* Convolve the image */
    if (sobel_5x5_fbm (&input, &output) &&
        write_bitmap (&output, stdout, outtype))
    { exit (0); }

    exit (1);
  }

  /* Sobel doesnt have any other arguments */
  if (sobel)
  {
    fprintf (stderr,
	     "Sobel \"%s\" [%dx%dx%d]\n",
	     input.hdr.title[0] ? input.hdr.title : "(untitled)",
	     w, h, k);

    /* Convolve the image */
    if (sobel_fbm (&input, &output) &&
        write_bitmap (&output, stdout, outtype))
    { exit (0); }

    exit (1);
  }
 
  /* Otherwise, general convolution, first arg is size of kernel */
  if (argc < 1 || (n = atoi (argv[0])) < 1)
  { fprintf (stderr, "%s\n", USAGE);
    exit (1);
  }
  
  if (n*n !=  argc-1)
  { fprintf (stderr, "fbconvolve: Mismatch between size and number of elements in kernel.\n");
    fprintf (stderr, "	    You specified a %dx%d kernel which should have %d elements, but", n, n, n*n);
    fprintf (stderr, "	    you only gave %d values.\n", argc-1);
  }
  
  /* Now allocate the kernel array and reads the command line */
  argv++;

  kernel = (double **) malloc (n * sizeof (double *));
  for (j=0; j<n; j++)
  { kernel[j] = (double *) malloc (n * sizeof (double));
    
    for (i=0; i<n; i++)
    { if (index ("+-.1234567890", argv[0][0]))
      { kernel[j][i] = atof (*argv++); }
    }
  }
  
  fprintf (stderr,
	   "Convolve \"%s\" [%dx%dx%d], kernel\n",
	   input.hdr.title[0] ? input.hdr.title : "(untitled)",
	   w, h, k);

  /* Print the kernel */
  for (j=0; j<n; j++)
  { fprintf (stderr, "\t");
    for (i=0; i<n; i++)
    { fprintf (stderr, "% 6.2lf", kernel[j][i]); }
    fprintf (stderr, "\n");
  }
  
  /* Convolve the image */
  if (convolve_fbm (&input, &output, kernel, n, absval) &&
      write_bitmap (&output, stdout, outtype))
  { exit (0); }
  
  exit (1);
}
