static char rcsid[] = "@(#)$Id: tls.c,v 1.24 2001/06/17 17:15:11 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.24 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@ozone.FMI.FI>
 *****************************************************************************/

#include "elmtls.h"

DEBUG_VAR(Debug,__FILE__,"tls");

/* RAND_ functions require only 20 bytes of randomnes ... 
 * See ENTROPY_NEEDED on openssl library
 *
 * RNG state size is 1023 bytes so more need not read ...
 * See STATE_SIZE on openssl library
 */

#define RAND_BYTES  1024
#define RAND_EGD_B   10

static int init_done       = 0;    /* -1 == failure
                                       1 == OK
				   */

/* Manual page: That's the global context structure which is created
                 by a server or client once per program life-time.
*/
static SSL_CTX *ssl_pCtx = NULL;
static SSL_CTX *tls_pCtx = NULL;

static char raw_rand_file[SLEN] = { 0 }; /* Default set by RAND_load_file */
static char rand_file[SLEN] = { 0 };

static char raw_rand_egd[SLEN] = { 0 };
static char rand_egd[SLEN] = { 0 };

static int rand_init_done = 0;

static void rand_init P_((int *errors));
static void rand_init(errors)
     int *errors;
{
    int bytes = 0;

    /* If not set, use openSSL default ... */
    if (!raw_rand_file[0]) {
	if (RAND_file_name(rand_file,sizeof rand_file)) {	
	    strfcpy(raw_rand_file,rand_file, sizeof raw_rand_file);
	} else {
	    strfcpy(raw_rand_file,"none", sizeof raw_rand_file);
	    strfcpy(rand_file,"none", sizeof rand_file);
	}
    } else if (0 == strcmp(raw_rand_file,"none")) {
	strfcpy(raw_rand_file,"none", sizeof raw_rand_file);
	strfcpy(rand_file,"none", sizeof rand_file);
    } else {
	if (0 != expand_meta(rand_file, raw_rand_file, sizeof rand_file)) {
	    
	    lib_error(FRM("Failed to expand rand-file value %s"),
		      raw_rand_file);
	    (*errors)++;
	    
	    strfcpy(raw_rand_file,"none", sizeof raw_rand_file);
	    strfcpy(rand_file,"none", sizeof rand_file);
	}
    }

    if (!raw_rand_egd[0] ||
	0 == strcmp(raw_rand_egd,"none")) {
	strfcpy(raw_rand_egd,"none", sizeof raw_rand_egd);
	strfcpy(rand_egd,"none", sizeof rand_egd);	
    } else {
	if (0 != expand_meta(rand_egd, raw_rand_egd, sizeof rand_egd)) {
	    
	    lib_error(FRM("Failed to expand rand-egd value %s"),
		      raw_rand_egd);
	    (*errors)++;
	    
	    strfcpy(raw_rand_egd,"none", sizeof raw_rand_egd);
	    strfcpy(rand_egd,"none", sizeof rand_egd);
	}
    }
    
    if (0 != strcmp(rand_file,"none")) {
	int err,z;
	
	DPRINT(Debug,2,(&Debug,			    
			"tls: rand-file is %s\n",rand_file));
	
	
	if (0 < (z = RAND_load_file(rand_file,RAND_BYTES)))
	    bytes += z;
	
	/* Check if we can write random bytes back ... */
	err = can_open(rand_file,"w");
	if (0 != err) {
	    /* Error message? */
	    lib_error(FRM("TLS random file %s not writable: %s"),
		      raw_rand_file,error_description(err));
	    
	    (*errors)++;
	    
	    strfcpy(raw_rand_file,"none", sizeof raw_rand_file);
	    strfcpy(rand_file,"none", sizeof rand_file);
	}
    }

    if (0 != strcmp(rand_egd,"none")) {
	int z,l;
	
	DPRINT(Debug,2,(&Debug,			    
			"tls: rand-egd is %s\n",rand_egd));

	l = RAND_BYTES - bytes;
	if (l < RAND_EGD_B)
	    l = RAND_EGD_B;

	if (0 < (z = RAND_egd_bytes(rand_egd,l)))
	    bytes += z;
	else {
	    lib_error(FRM("EGD socket %s does not work"),
		      rand_egd);
	}
    }

    DPRINT(Debug,2,(&Debug,			    
		    "tls: Seeded PRNG with %d bytes\n",bytes));

    rand_init_done = 1;
}

#if ANSI_C
wants_rand_bits_f wants_rand_bits;
#endif
void wants_rand_bits(buf,size,entropy_bits)
     CONST char *buf;
     int size;
     int entropy_bits;
{
    double entropy_bytes = entropy_bits / 8.0;

    RAND_add(buf,size,entropy_bytes);
    DPRINT(Debug,10,(&Debug,"RAND_add()ed %d bytes, entropy %.2f bytes\n",
		     size,entropy_bytes));
}

int tls_init(int init_rand)
{
    int junk;

    if (init_done < 0)
	return 0;            /* Can't initialize */
    if (init_done > 0)
	return 1;            /* Already initialized */

    SSL_load_error_strings(); 
    SSL_library_init();

    /* Seeding with random ? */

    ssl_pCtx = SSL_CTX_new(SSLv23_client_method());
    if (!ssl_pCtx) {
	DPRINT(Debug,2,(&Debug,			    
			"tls: tls_init: SSL_CTX_new failed -- SLv23_client_method\n"));
    }
    
    tls_pCtx = SSL_CTX_new(TLSv1_client_method());
    if (!tls_pCtx) {
	DPRINT(Debug,2,(&Debug,			    
			"tls: tls_init: SSL_CTX_new failed -- tlsv1_client_method\n"));
    }
    
    if (!ssl_pCtx && !tls_pCtx) {
	DPRINT(Debug,2,(&Debug,			    
			"tls: tls_init: Both SSL_CTX_new failed\n"));
	init_done = -1;
	goto fail;
    }

#if 0
    n = SSL_CTX_use_certificate_file(pCtx,filename,SSL_FILETYPE_PEM);
    if (n > 0) ok;


    n = SSL_CTX_use_PrivateKey_file(pCtx,filename,SSL_FILETYPE_PEM);
    if (n > 0) ok;

#endif

    if (init_rand &&
	!rand_init_done)
	rand_init(&junk);

    init_done = 1;

 fail:
    return (init_done > 0);
}

#if ANSI_C
RC_post_init_f RC_post_init;
#endif
void RC_post_init(errors)
     int *errors;
{
    if (tls_init(0)) {
	rand_init(errors);
    }
}

struct shared_type_1 {      /* Type for TLS */
    struct stream_type    *type;

    BIO *pbioRead;
    BIO *pbioWrite;
    SSL *pSSL;
    
    int wants;
    enum tls_connection_state state;

    int noread;   /* do not signal read */

    struct Read_Buffer read_buffer;
    struct Write_Buffer write_buffer;
}; 

#define READ_BLOCK   1024

static int ReadFromSSL P_((SSL *ssl, struct Read_Buffer *buffer,
			   int wanted));
static int ReadFromSSL(ssl,buffer,wanted)
     SSL *ssl; 
     struct Read_Buffer *buffer;
     int wanted;
{
    int n;

    if (wanted > 0) {
	buffer -> read_buffer = safe_realloc(buffer -> read_buffer,
					     buffer -> read_len + wanted);
	n = SSL_read(ssl, buffer -> read_buffer + buffer -> read_len, 
		     wanted);
    } else {
	buffer -> read_buffer = safe_realloc(buffer -> read_buffer,
					     buffer -> read_len + READ_BLOCK);
	n = SSL_read(ssl, buffer -> read_buffer + buffer -> read_len, 
		     READ_BLOCK);
    }

    return n;
}

static int WriteToSSL P_((SSL *ssl, struct Write_Buffer *buffer));
static int WriteToSSL(ssl,buffer)
     SSL *ssl;
     struct Write_Buffer *buffer;
{
    int n = SSL_write(ssl,buffer->write_buffer, buffer->write_len);
    
    return n;
}

#ifdef ANSI_C
static ss_ReadFromStream  ss_ReadFromTLS;
#endif
static int ss_ReadFromTLS (ss,stack_idx,buffer,wanted)
     struct streamsched *ss;
     int stack_idx;
     struct Read_Buffer *buffer;
     int wanted;
{
    int n;
    int err_x;
    int p = 0;
    int ssl_error1;

    DPRINT(Debug,15,(&Debug,			    
		"tls: ss_ReadFromTLS: ss=%p, stack_idx=%d\n",
		ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].generic_1 ||
	ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_ReadFromTLS",
	      "Bad stack index or stack",0);

    n = ReadFromSSL(ss->this_stack[stack_idx].generic_1->pSSL,
		    buffer,wanted);

    if (n > 0) {
	ss->this_stack[stack_idx].generic_1->noread = 0;
	return n;
    }

    if (n == 0) {
	err_x = SSL_get_error(ss->this_stack[stack_idx].generic_1->pSSL,n);
	
	if (SSL_ERROR_ZERO_RETURN == err_x) {	    
	    ss->this_stack[stack_idx].generic_1->noread = 0;
	    /* Indicate EOF */
	    return 0;
	}
	
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_ReadFromTLS: SSL read returned %d, err_x=%d (not SSL_ERROR_ZERO_RETURN)\n",
			 n, err_x));
	
    } else {
	err_x = SSL_get_error(ss->this_stack[stack_idx].generic_1->pSSL,n);

	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_ReadFromTLS: SSL read returned %d (error), err_x=%d\n",
		    n, err_x));
	
    }
    
    switch(err_x) {
	int ssl_error;
    case SSL_ERROR_WANT_READ:
	ss->this_stack[stack_idx].generic_1->wants |= SS_read_act;
	ss->this_stack[stack_idx].generic_1->noread++;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... adding want read -> noread=%d\n",
			 ss->this_stack[stack_idx].generic_1->noread));
	break;
    case SSL_ERROR_WANT_WRITE:
	ss->this_stack[stack_idx].generic_1->wants |= SS_write_act;
	ss->this_stack[stack_idx].generic_1->noread++;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... adding want write -> noread=%d\n",
		    ss->this_stack[stack_idx].generic_1->noread));
	break;
    case SSL_ERROR_WANT_X509_LOOKUP:
	DPRINT(Debug,13,(&Debug,			    
			 "    ... WANTS X509 lOOKUP (SSL_CTX_set_client_cert_cb())\n"));
	break;
    case SSL_ERROR_SYSCALL:
	ssl_error = ERR_get_error();
	DPRINT(Debug,13,(&Debug,			    
			 "    ... SYSCALL ERROR (ssl_error=%d)\n",
		    ssl_error));
	if (0 == ssl_error) {
	    if (-1 == n) {
		if (ss->error_state) {
		    DPRINT(Debug,13,(&Debug,			    
				     "   ... error state: %s\n",
				     ss->error_state));
		} else {
		    int err1 = errno;
		    DPRINT(Debug,13,(&Debug,			    
				     "   ... read error %s (errno %d)\n",
				     error_description(err1),err1));
		    
		    if (err1 != EINTR && err1 != EAGAIN && 
			err1 != EWOULDBLOCK) {
			ss->error_state = strmcpy(ss->error_state,
						  error_description(err1));
		    }
		}
	    } else if (0 == n) {
		/* FIXME: Messages */
		ss->error_state = strmcpy(ss->error_state,
					  "Unexpected EOF when reading");
	    }
	} else {
	    char *s = ERR_error_string(ssl_error,NULL);
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... SSL error %s (code %d)\n",
			     s,ssl_error));
	    ss->error_state = strmcpy(ss->error_state,
				      s);
	}
	break;
    case SSL_ERROR_SSL:
	ssl_error = ERR_get_error();
	DPRINT(Debug,13,(&Debug,			    
			 "    ... SSL ERROR (ssl_error=%d)\n",
			 ssl_error));
	
	if (ssl_error != 0) {
	    char * s = ERR_error_string(ssl_error,NULL);
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... SSL error %s (code %d)\n",
			     s,ssl_error));
	    ss->error_state = strmcpy(ss->error_state,
				      s);
	}
    }

    /* Clear error queue */
    while (0 != (ssl_error1 = ERR_get_error())) {
	char *s = ERR_error_string(ssl_error1,NULL);
	if (!p) {
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... Rest of SSL error queue follows:\n"));
	    p = 1;
	}
	DPRINT(Debug,13,(&Debug,			    
			 "   ... SSL error %s (code %d)\n",
			 s,ssl_error1));

    }
    return -1;
}

#ifdef ANSI_C
static ss_WriteToStream ss_WriteToTLS;
#endif
static int ss_WriteToTLS(ss,stack_idx,buffer)
     struct streamsched *ss; 
     int stack_idx;
     struct Write_Buffer *buffer;
{
    int n;
    int err_x;
    int p = 0;
    int ssl_error1;

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_WriteToTLS: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].generic_1 ||
	ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_WriteToTLS",
	      "Bad stack index or stack",0);

    n = WriteToSSL(ss->this_stack[stack_idx].generic_1->pSSL,
		  buffer);

    if (n > 0)
	return n;

    err_x = SSL_get_error(ss->this_stack[stack_idx].generic_1->pSSL,n);

    DPRINT(Debug,13,(&Debug,			    
		     "tls: ss_WriteToTLS: SSL write returned %d, err_x=%d\n",
		     n, err_x));

    switch(err_x) {
	int ssl_error;
    case SSL_ERROR_ZERO_RETURN:
	DPRINT(Debug,13,(&Debug,			    
		    "    ... zero return????????\n"));	
	break;
    case SSL_ERROR_WANT_READ:
	ss->this_stack[stack_idx].generic_1->wants |= SS_read_act;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... adding want read\n"));
	break;
    case SSL_ERROR_WANT_WRITE:
	ss->this_stack[stack_idx].generic_1->wants |= SS_write_act;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... adding want write\n"));
	break;
    case SSL_ERROR_WANT_X509_LOOKUP:
	DPRINT(Debug,13,(&Debug,			    
			 "    ... WANTS X509 lOOKUP (SSL_CTX_set_client_cert_cb())\n"));
	break;
    case SSL_ERROR_SYSCALL:
	ssl_error = ERR_get_error();
	DPRINT(Debug,13,(&Debug,			    
			 "    ... SYSCALL ERROR (ssl_error=%d)\n",
			 ssl_error));
	if (0 == ssl_error) {
	    if (-1 == n) {
		if (ss->error_state) {
		    DPRINT(Debug,13,(&Debug,			    
				     "   ... error state: %s\n",
				     ss->error_state));
		} else {
		    int err1 = errno;
		    DPRINT(Debug,13,(&Debug,			    
				     "   ... write error %s (errno %d)\n",
				     error_description(err1),err1));
		    
		    if (err1 != EINTR && err1 != EAGAIN && 
			err1 != EWOULDBLOCK) {
			ss->error_state = strmcpy(ss->error_state,
						  error_description(err1));
		    }
		}
	    }
	} else {
	    char *s = ERR_error_string(ssl_error,NULL);
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... SSL error %s (code %d)\n",
			     s,ssl_error));
	    ss->error_state = strmcpy(ss->error_state,
				      s);
	}
	break;
    case SSL_ERROR_SSL:
	ssl_error = ERR_get_error();
	DPRINT(Debug,13,(&Debug,			    
			 "    ... SSL ERROR (ssl_error=%d)\n",
			 ssl_error));
	
	if (ssl_error != 0) {
	    char * s = ERR_error_string(ssl_error,NULL);
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... SSL error %s (code %d)\n",
			     s,ssl_error));
	    ss->error_state = strmcpy(ss->error_state,
				      s);
	}
    }

    /* Clear error queue */
    while (0 != (ssl_error1 = ERR_get_error())) {
	char *s = ERR_error_string(ssl_error1,NULL);
	if (!p) {
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... Rest of SSL error queue follows:\n"));
	    p = 1;
	}
	DPRINT(Debug,13,(&Debug,			    
			 "   ... SSL error %s (code %d)\n",
			 s,ssl_error1));
	
    }

    return -1;
}

#ifdef ANSI_C
static ss_FreeThis ss_FreeTLS;
#endif
static void ss_FreeTLS(ss,stack_idx)
     struct streamsched *ss;
     int stack_idx;
{
    DPRINT(Debug,13,(&Debug,			    
		     "tls: ss_FreeTLS: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].generic_1 ||
	ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_FreeTLS",
	      "Bad stack index or stack",0);
    
    /* Closing of SSL stream... */

    if (tls_connected == ss->this_stack[stack_idx].generic_1->state) {
	DPRINT(Debug,15,(&Debug,			    
			 "tls: ss_FreeTLS:  closing (shutdown)\n"));

	ss->this_stack[stack_idx].generic_1->state = tls_closing;

	while (tls_closing == ss->this_stack[stack_idx].generic_1->state) {
	    char * M = NULL;
	    
	    WaitStreamFor(ss,SS_shutdown_act);
	    
	    M =  RemoveStreamError(ss);
	    if (M) {
		lib_error(FRM("%s"),
			  M);
		free(M);
		break;
	    }
	}
    }

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_FreeTLS: SSL_free (%p):\n",
		     ss->this_stack[stack_idx].generic_1->pSSL));

    if (ss->this_stack[stack_idx].generic_1->pSSL)
	SSL_free(ss->this_stack[stack_idx].generic_1->pSSL);
    ss->this_stack[stack_idx].generic_1->pSSL      = NULL;

    /* These are freed by SSL_free */
    ss->this_stack[stack_idx].generic_1->pbioRead  = NULL;
    ss->this_stack[stack_idx].generic_1->pbioWrite = NULL;

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_FreeTLS: buffers\n"));

    free_Read_Buffer(&(ss->this_stack[stack_idx].generic_1->read_buffer));
    free_Write_Buffer(&(ss->this_stack[stack_idx].generic_1->write_buffer));

    ss->this_stack[stack_idx].generic_1->state = tls_closed;

    /* Write some bytes back to random generator */
    if (0 != strcmp(rand_file,"none")) {
	int err = can_open(rand_file,"w");

	if (0 != err) {
	    lib_error(FRM("TLS random file %s not writable: %s"),
		      raw_rand_file,error_description(err));
	    
	    strfcpy(raw_rand_file,"none", sizeof raw_rand_file);
	    strfcpy(rand_file,"none", sizeof rand_file);
	} else {
	    RAND_write_file(rand_file);
	    elm_chown (rand_file, userid, groupid);
	}
    }

    free(ss->this_stack[stack_idx].generic_1);
    ss->this_stack[stack_idx].generic_1 = NULL;

}

#ifdef ANSI_C
static ss_StreamAction ss_TLSReadAction;
#endif
static int ss_TLSReadAction(ss,stack_idx)
     struct streamsched *ss;
     int stack_idx;
{
    int ret = 1;

    int n;

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_TLSReadAction: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].generic_1 ||
	ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLSReadAction",
	      "Bad stack index or stack",0);
    
    if (stack_idx+1 >= ss->stack_len) 
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLSReadAction",
	      "No provider on stack!!!",0);
	

    if (ss->this_stack[stack_idx].generic_1->noread) {
	/* Reset for reconsideration */
	ss->this_stack[stack_idx].generic_1->noread = 0;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... noread=%d\n",
			 ss->this_stack[stack_idx].generic_1->noread));
    }

    n = (*(ss -> this_stack[stack_idx+1].TYPE))->
	read_from_it(ss,stack_idx+1,
		     &(ss->this_stack[stack_idx].generic_1->read_buffer),
		     -1);

    if (n < 0) {      
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_TLSReadAction: read error %s\n",
			 ss->error_state ? ss->error_state : "(none)"));
    }

    if (n > 0) {
	int x;
	ss->this_stack[stack_idx].generic_1->read_buffer.read_len += n;
	
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_TLSReadAction: Readed %d bytes (%d total)\n",
			 n,ss->this_stack[stack_idx].generic_1->
			 read_buffer.read_len));
	
	x = BIO_write(ss->this_stack[stack_idx].generic_1->pbioRead,
		      ss->this_stack[stack_idx].generic_1->
		      read_buffer.read_buffer,
		      ss->this_stack[stack_idx].generic_1->
		      read_buffer.read_len);
	
	if (x > 0) {
	    cut_line(& (ss->this_stack[stack_idx].generic_1->read_buffer),
		     x);
	} else {
	    DPRINT(Debug,13,(&Debug,			    
			     "... writing it to read BIO failed, ret=%d\n",
			     x));
	}
	
	if (ss->this_stack[stack_idx].generic_1->read_buffer.read_len >0) {
	    DPRINT(Debug,13,(&Debug,			    
			     "... %d chars LEFT, not written to read BIO\n",
			     ss->this_stack[stack_idx].generic_1->
			     read_buffer.read_len));
	}	
    }

    if (0 == n) {
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_TLSReadAction: EOF on read -- setting read BIOs EOF flag\n"));
	BIO_set_mem_eof_return(ss->this_stack[stack_idx].generic_1->pbioRead,
			       0);

	/* No LONGER wants read ... */
	ret = 0;
	ss->this_stack[stack_idx].generic_1->wants &= ~SS_read_act;
    }

    if (n < 0 &&
	ss->this_stack[stack_idx].generic_1->read_buffer.read_len == 0 &&
	ss->error_state) {

	/* No LONGER wants read ... */
	ret = 0;
	ss->this_stack[stack_idx].generic_1->wants &= ~SS_read_act;
    }

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_TLSReadAction=%d %s\n",
		     ret, ret == 0 ? "(disabling)" : ""));
    return ret;
}

#ifdef ANSI_C
static ss_StreamAction ss_TLSWriteAction;
#endif
static int ss_TLSWriteAction(ss,stack_idx)
     struct streamsched *ss;
     int stack_idx;
{
    char * s;
    int n,y;
    int ret = 0;
    int blocksize = 0;

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_TLSWriteAction: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].generic_1 ||
	ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLSWriteAction",
	      "Bad stack index or stack",0);
    
    if (stack_idx+1 >= ss->stack_len) 
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLSWriteAction",
	      "No provider on stack!!!",0);

    if (ss->this_stack[stack_idx].generic_1->noread) {
	/* Reset for reconsideration */
	ss->this_stack[stack_idx].generic_1->noread = 0;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... noread=%d\n",
			 ss->this_stack[stack_idx].generic_1->noread));
    }

    blocksize = BIO_pending(ss->this_stack[stack_idx].generic_1->pbioWrite);
    if (blocksize < 1)
	blocksize = READ_BLOCK;
    else {
	DPRINT(Debug,13,(&Debug,			    
			 "    (%d bytes available on write BIO for read)\n",
			 blocksize));
    }

    s = safe_malloc(blocksize);
 
    n = BIO_read(ss->this_stack[stack_idx].generic_1->pbioWrite,
		 s,blocksize);

    if (n < 0) {
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_TLSWriteAction: Failed to read from WRITE BIO, ret=%d\n",
			 n));

	free(s);
	s = NULL;

    } else {
	/* add_to_Write_Buffer will free s */

	add_to_Write_Buffer( &(ss->this_stack[stack_idx].generic_1->
			       write_buffer),
			     &s,n);
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_TLSWriteAction: Readed %d bytes from write BIO (%d total)\n",
			 n,ss->this_stack[stack_idx].generic_1->
			 write_buffer.write_len));
    }
		
    if (ss->this_stack[stack_idx].generic_1->write_buffer.write_len > 0) {
	int x = (*(ss -> this_stack[stack_idx+1].TYPE))->
	    write_to_it(ss,stack_idx+1,
			&(ss->this_stack[stack_idx].generic_1->write_buffer));

	if (x < 0) {      
	    DPRINT(Debug,13,(&Debug,			    
			     "tls: ss_TLSWriteAction: write error %s\n",
			     ss->error_state ? ss->error_state : "(none)"));
	}

	if (x > 0) {
	    cut_Write_Buffer(&(ss->this_stack[stack_idx].generic_1->
			       write_buffer),x);
	    DPRINT(Debug,13,(&Debug,			    
			     " ... written %d bytes (%d left)\n",
			     x,ss->this_stack[stack_idx].generic_1->
			     write_buffer.write_len));
	    ret = x;
	}	
    }

    /* If there is nothing read from BIO and write buffer flushed,
       then we not want longer write ... 
    */

    y = BIO_pending(ss->this_stack[stack_idx].generic_1->pbioWrite);
    if (y > 0) {
	DPRINT(Debug,13,(&Debug,			    
			 " ... %d bytes available on write BIO for read\n",
			 y));
    } else if (0 == ss->this_stack[stack_idx].generic_1->
	       write_buffer.write_len) {
	ss->this_stack[stack_idx].generic_1->wants &= ~SS_write_act;
	ret = 0;
    }

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_TLSWriteAction=%d %s\n",
		     ret, ret == 0 ? "(disabling)" : ""));
    return ret;
}

#ifdef ANSI_C
static ss_StreamSchedule ss_TLSSchedule;
#endif
static void ss_TLSSchedule(ss,stack_idx,previous_needs,
			   needs,provides)
     struct streamsched *ss;
     int stack_idx;
     int previous_needs;
     int *needs;
     int *provides;
{
    
    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_TLSSchedule: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLSSchedule",
	      "Bad stack index or stack",0);

    *provides = 0;
    *needs    = 0;

    if (!ss->this_stack[stack_idx].generic_1) {
	DPRINT(Debug,15,(&Debug,			    
			 "tls: ss_TLSSchedule:  closed (no data)\n"));
	return;
    }

    if (ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLSSchedule",
	      "Bad stack index or stack",0);

    if (tls_closed == ss->this_stack[stack_idx].generic_1->state) {
	DPRINT(Debug,15,(&Debug,			    
			 "tls: ss_TLSSchedule:  state = closed\n"));
	return;
    }

    if (ss->this_stack[stack_idx].generic_1->pbioWrite) {
	int y;
	y = BIO_pending(ss->this_stack[stack_idx].generic_1->pbioWrite);
	if (y > 0) {
	    DPRINT(Debug,13,(&Debug,			    
			"tls: ss_TLSSchedule: %d bytes available on write BIO for read\n",
			y));
	    ss->this_stack[stack_idx].generic_1->wants |= SS_write_act;
	}
    } else {
	DPRINT(Debug,13,(&Debug,			    
		    "tls: ss_TLSSchedule: NO Write BIO\n"));
    }

    if (ss->this_stack[stack_idx].generic_1->pbioRead) {
	if (BIO_should_read(ss->this_stack[stack_idx].generic_1->pbioRead)) {
	    DPRINT(Debug,13,(&Debug,			    
			     "tls: ss_TLSSchedule: read BIO needs read data\n"));
	    ss->this_stack[stack_idx].generic_1->wants |= SS_read_act;
	}
    } else {	
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_TLSSchedule: NO Read BIO\n"));	
    }

    *needs    = ss->this_stack[stack_idx].generic_1->wants;

    switch (ss->this_stack[stack_idx].generic_1->state) {
    case tls_connected:	
	if (ss->this_stack[stack_idx].generic_1->pSSL) {
	    int n;
	    if ((n = SSL_pending(ss->this_stack[stack_idx].generic_1->pSSL)) > 0) {
		DPRINT(Debug,13,(&Debug,			    
				 "tls: ss_TLSSchedule: %d bytes available for read\n",
				 n));
		*provides |= SS_read_act;
		/* Reset noread */
		ss->this_stack[stack_idx].generic_1->noread = 0;
	    }
	} else {
	    DPRINT(Debug,15,(&Debug,			    
			     "tls: ss_TLSSchedule: No SSL!\n"));
	}

	
	if (ss->this_stack[stack_idx].generic_1->pbioRead) {
	    int n;
	    n = BIO_pending(ss->this_stack[stack_idx].generic_1->pbioRead);
	    if (n > 0) {
		DPRINT(Debug,15,(&Debug,			    
				 "tls: ss_TLSSchedule: %d bytes available on read BIO for read\n",
				 n));
		/* Because SSL_read reads from read BIO, we can now try read from it */
		*provides |= SS_read_act;
		ss->this_stack[stack_idx].generic_1->noread = 0;
	    }
	} else {
	    DPRINT(Debug,15,(&Debug,			    
			     "tls: ss_TLSSchedule: No read BIO!\n"));
	}
	
	/* We always 'provide' what upstream wants (if connected)
	   .. after all there is buffering BIOs on between
	   .. but do not provide read after failure because
	      it may cause loop
	*/
	if (ss->this_stack[stack_idx].generic_1->noread > 0 &&
	    0 != (previous_needs & SS_read_act)) {
	    DPRINT(Debug,13,(&Debug,			    
			     "tls: ss_TLSSchedule: will not provide read, noread=%d\n",
			     ss->this_stack[stack_idx].generic_1->noread
			));
	    *provides |= previous_needs &
		(SS_write_act|SS_timeout_act);

	} else {
	    *provides |= previous_needs &
		(SS_read_act|SS_write_act|SS_timeout_act);
	}

	break;
    case tls_not_connected:
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_TLSSchedule: Hanshake not completed yet...\n"));
	*needs |= SS_setup_act;
	break;
    case tls_closing:
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_TLSSchedule: Stream is closing...\n"));
	*needs |= SS_shutdown_act;
	break;
    default:
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_TLSSchedule: Have TLS error\n"));
	*provides |= previous_needs &
	    (SS_read_act|SS_write_act|SS_timeout_act);
	break;
    }
}

static int ss_SetupTLSAction P_((struct streamsched *ss,
			      int stack_idx));
static int ss_SetupTLSAction(ss,stack_idx)
     struct streamsched *ss;
     int stack_idx;
{
    int n;
    int err_x;
    int ssl_error1;
    int p = 0;
    int ret =1;

    DPRINT(Debug,13,(&Debug,			    
		     "tls: ss_SetupTLSAction: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].generic_1 ||
	ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_SetupTLSAction",
	      "Bad stack index or stack",0);

    if (ss->this_stack[stack_idx].generic_1->noread) {
	/* Reset for reconsideration */
	ss->this_stack[stack_idx].generic_1->noread = 0;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... noread=%d\n",
			 ss->this_stack[stack_idx].generic_1->noread));
    }


    n = SSL_connect(ss->this_stack[stack_idx].generic_1->pSSL);
    
    if (n > 0) {
	ss->this_stack[stack_idx].generic_1->state = tls_connected;
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_SetupTLSAction: Setting connected -- SSL connect returned %d\n",
			 n));
	ret = 0;
	goto out;
    }
    err_x = SSL_get_error(ss->this_stack[stack_idx].generic_1->pSSL,n);
    
    DPRINT(Debug,13,(&Debug,			    
		     "tls: ss_SetupTLSAction: SSL connect returned %d, err_x=%d\n",
		     n, err_x));

    switch(err_x) {
	int ssl_error;
	int A,B;
    case SSL_ERROR_ZERO_RETURN:
	DPRINT(Debug,13,(&Debug,			    
			 "    ... zero return????????\n"));	
	break;
    case SSL_ERROR_WANT_READ:
	ss->this_stack[stack_idx].generic_1->wants |= SS_read_act;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... adding want read\n"));
	break;
    case SSL_ERROR_WANT_WRITE:
	ss->this_stack[stack_idx].generic_1->wants |= SS_write_act;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... adding want write\n"));
	break;
    case SSL_ERROR_WANT_X509_LOOKUP:
	DPRINT(Debug,13,(&Debug,			    
			 "    ... WANTS X509 lOOKUP (SSL_CTX_set_client_cert_cb())\n"));
	break;
    case SSL_ERROR_SYSCALL:
	ssl_error = ERR_get_error();
	A = ERR_GET_LIB(ssl_error);
	B = ERR_GET_REASON(ssl_error);
	DPRINT(Debug,13,(&Debug,			    
			 "    ... SYSCALL ERROR (ssl_error=%d: lib=%d, reason=%d)\n",
			 ssl_error,A,B));
	if (0 == ssl_error) {
	    if (-1 == n) {
		if (ss->error_state) {
		    DPRINT(Debug,13,(&Debug,			    
				     "   ... error state: %s\n",
				     ss->error_state));
		} else {
		    int err1 = errno;
		    DPRINT(Debug,13,(&Debug,			    
				     "   ... write error %s (errno %d)\n",
				     error_description(err1),err1));
		    
		    if (err1 != EINTR && err1 != EAGAIN && 
			err1 != EWOULDBLOCK) {
			ss->error_state = strmcpy(ss->error_state,
						  error_description(err1));
		    }
		}
	    }
	} else {
	    char *s = ERR_error_string(ssl_error,NULL);
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... SSL error %s (code %d)\n",
			     s,ssl_error));
	    ss->error_state = strmcpy(ss->error_state,
				      s);
	}
	ss->this_stack[stack_idx].generic_1->state = tls_error;
	DPRINT(Debug,13,(&Debug,			    
		    "tls: ss_SetupTLSAction: Setting error state\n"));
	break;
    case SSL_ERROR_SSL:
	ssl_error = ERR_get_error();
	A = ERR_GET_LIB(ssl_error);
	B = ERR_GET_REASON(ssl_error);
	DPRINT(Debug,13,(&Debug,			    
			 "    ... SSL ERROR (ssl_error=%d: lib=%d, reason=%d)\n",
			 ssl_error,A,B));
	
	if (ssl_error != 0) {
	    char * s = ERR_error_string(ssl_error,NULL);
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... SSL error %s (code %d)\n",
			     s,ssl_error));
	    ss->error_state = strmcpy(ss->error_state,
				      s);
	}

	ss->this_stack[stack_idx].generic_1->state = tls_error;
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_SetupTLSAction: Setting error state\n"));
    }

    /* Clear error queue */
    while (0 != (ssl_error1 = ERR_get_error())) {
	char *s = ERR_error_string(ssl_error1,NULL);
	if (!p) {
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... Rest of SSL error queue follows:\n"));
	    p = 1;
	}
	DPRINT(Debug,13,(&Debug,			    
			 "   ... SSL error %s (code %d)\n",
			 s,ssl_error1));
	
    }

 out:
    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_SetupTLSAction=%d %s\n",
		     ret, ret == 0 ? "(disabling?)" : ""));

    return ret;
}

static int ss_ShutdownTLSAction P_((struct streamsched *ss,
			      int stack_idx));
static int ss_ShutdownTLSAction(ss,stack_idx)
     struct streamsched *ss;
     int stack_idx;
{
    int n;
    int err_x;
    int ssl_error1;
    int p = 0;
    int ret =1;

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_ShutdownTLSAction: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].generic_1 ||
	ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_shutdownTLSAction",
	      "Bad stack index or stack",0);


    n = SSL_shutdown(ss->this_stack[stack_idx].generic_1->pSSL);
    
    if (n > 0) {
	ss->this_stack[stack_idx].generic_1->state = tls_closed;
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_ShutdownTLSAction: Setting closed -- SSL shutdown returned %d\n",
			 n));
	ret = 0;
	goto out;
    }
    err_x = SSL_get_error(ss->this_stack[stack_idx].generic_1->pSSL,n);
    
    DPRINT(Debug,13,(&Debug,			    
		     "tls: ss_ShutdownTLSAction: SSL shutdown returned %d, err_x=%d\n",
		     n, err_x));

    switch(err_x) {
	int ssl_error;
	int A,B;
    case SSL_ERROR_ZERO_RETURN:
	DPRINT(Debug,13,(&Debug,			    
			 "    ... zero return????????\n"));	
	break;
    case SSL_ERROR_WANT_READ:
	ss->this_stack[stack_idx].generic_1->wants |= SS_read_act;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... adding want read\n"));
	break;
    case SSL_ERROR_WANT_WRITE:
	ss->this_stack[stack_idx].generic_1->wants |= SS_write_act;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... adding want write\n"));
	break;
    case SSL_ERROR_WANT_X509_LOOKUP:
	DPRINT(Debug,13,(&Debug,			    
			 "    ... WANTS X509 lOOKUP (SSL_CTX_set_client_cert_cb())\n"));
	break;
    case SSL_ERROR_SYSCALL:
	ssl_error = ERR_get_error();
	A = ERR_GET_LIB(ssl_error);
	B = ERR_GET_REASON(ssl_error);
	DPRINT(Debug,13,(&Debug,			    
			 "    ... SYSCALL ERROR (ssl_error=%d: lib=%d, reason=%d)\n",
			 ssl_error,A,B));
	if (0 == ssl_error) {
	    if (-1 == n) {
		if (ss->error_state) {
		    DPRINT(Debug,13,(&Debug,			    
				     "   ... error state: %s\n",
				     ss->error_state));
		} else {
		    int err1 = errno;
		    DPRINT(Debug,13,(&Debug,			    
				     "   ... write error %s (errno %d)\n",
				     error_description(err1),err1));
		    
		    if (err1 != EINTR && err1 != EAGAIN && 
			err1 != EWOULDBLOCK) {
			ss->error_state = strmcpy(ss->error_state,
						  error_description(err1));
		    }
		}
	    }
	} else {
	    char *s = ERR_error_string(ssl_error,NULL);
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... SSL error %s (code %d)\n",
			     s,ssl_error));
	    ss->error_state = strmcpy(ss->error_state,
				      s);
	}
	ss->this_stack[stack_idx].generic_1->state = tls_error;
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_ShutdownTLSAction: Setting error state\n"));
	break;
    case SSL_ERROR_SSL:
	ssl_error = ERR_get_error();
	A = ERR_GET_LIB(ssl_error);
	B = ERR_GET_REASON(ssl_error);
	DPRINT(Debug,13,(&Debug,			    
			 "    ... SSL ERROR (ssl_error=%d: lib=%d, reason=%d)\n",
			 ssl_error,A,B));
	
	if (ssl_error != 0) {
	    char * s = ERR_error_string(ssl_error,NULL);
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... SSL error %s (code %d)\n",
			     s,ssl_error));
	    ss->error_state = strmcpy(ss->error_state,
				      s);
	}

	ss->this_stack[stack_idx].generic_1->state = tls_error;
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_ShutdownTLSAction: Setting error state\n"));
    }

    /* Clear error queue */
    while (0 != (ssl_error1 = ERR_get_error())) {
	char *s = ERR_error_string(ssl_error1,NULL);
	if (!p) {
	    DPRINT(Debug,13,(&Debug,			    
			"   ... Rest of SSL error queue follows:\n"));
	    p = 1;
	}
	DPRINT(Debug,13,(&Debug,			    
		    "   ... SSL error %s (code %d)\n",
		    s,ssl_error1));
	
    }

 out:
    DPRINT(Debug,13,(&Debug,			    
		     "tls: ss_ShutdownTLSAction=%d %s\n",
		     ret, ret == 0 ? "(disabling?)" : ""));

    return ret;
}


#ifdef ANSI_C
static ss_StreamInfo ss_TLSInfo;
#endif
static void ss_TLSInfo(ss,stack_idx,function,int_val,str_val)
     struct streamsched *ss; 
     int stack_idx;
     enum SS_info function;
     int *int_val; 
     char **str_val;
{    
    int x;

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_TLSInfo: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].generic_1 ||
	ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLSInfo",
	      "Bad stack index or stack",0);

    switch (function) {
    case SS_ssf:
	*int_val = SSL_get_cipher_bits(ss->this_stack[stack_idx].generic_1->
				       pSSL,&x);
	
	DPRINT(Debug,10,(&Debug,			    
			 "tls: ss_TLSInfo: cipher_bits=%d (aka ssf), alg_bits=%d\n",
			 *int_val,x));
	break;
    }    

}

struct stream_type TLS_STREAM = {
    "TLS",
    ss_ReadFromTLS,
    ss_WriteToTLS,
    ss_FreeTLS,
    ss_TLSReadAction,
    ss_TLSWriteAction,
    ss_TLSSchedule,
    ss_SetupTLSAction,
    ss_ShutdownTLSAction,
    ss_TLSInfo
};

union stream_types create_TLS_stream(int tls)
{
    union stream_types ret;
    struct shared_type_1 * S = safe_malloc(sizeof (struct shared_type_1));

    /* bzero is defined on hdrs/defs.h */
    bzero((void *)S,sizeof (struct shared_type_1));

    S->type = &TLS_STREAM;

    S->pbioRead  = NULL;
    S->pbioWrite = NULL;
    S->pSSL      = NULL;
    S->wants     = 0;
    S->noread    = 1;
    S->state     = tls_not_connected;

    zero_Read_Buffer( &(S->read_buffer));
    zero_Write_Buffer( &(S->write_buffer));


    

    if (tls) {
	if (!tls_pCtx) {
	    DPRINT(Debug,4,(&Debug,			    
			    "tls: create_TLS_stream: tls not available\n"));
	    goto failure;
	}
	S->pSSL = SSL_new(tls_pCtx);
    } else {
	if (!ssl_pCtx) {
	    DPRINT(Debug,4,(&Debug,			    
			    "tls: create_TLS_stream: ssl not available\n"));
	    goto failure;
	}
	S->pSSL = SSL_new(ssl_pCtx);
    }

    if (!S->pSSL) {
	DPRINT(Debug,3,(&Debug,			    
		   "tls: create_TLS_stream: SSL_new failed (%s)\n",
		   tls ? "TLS" : "SSL"));
	
	goto failure;
    }

    DPRINT(Debug,15,(&Debug,			    
		     "tls: create_TLS_stream: SSL_new=%p:\n",
		     S->pSSL));


    S->pbioRead = BIO_new(BIO_s_mem());

    if (!S->pbioRead) {
	DPRINT(Debug,3,(&Debug,			    
			"tls: create_TLS_stream: BIO_new failed\n"));
	       
	goto failure;
    }
    BIO_set_mem_eof_return(S->pbioRead,-1);   /* empty not EOF */

    S->pbioWrite = BIO_new(BIO_s_mem());

    if (!S->pbioWrite) {
	DPRINT(Debug,3,(&Debug,			    
			"tls: create_TLS_stream: BIO_new failed\n"));
	
	BIO_free(S->pbioRead);

	goto failure;
    }

    /* After this SSL_free() also frees BIOs */
    SSL_set_bio(S->pSSL,S->pbioRead,S->pbioWrite);


    ret.generic_1 = S;

    return ret;

 failure:

    if (S->pSSL)
	SSL_free(S->pSSL);
    S->pSSL = NULL;
    
    S->pbioRead  = NULL;
    S->pbioWrite = NULL;

    free(S);
    
    ret.TYPE = NULL;
    return ret;
}

static struct provides_shared_ST STREAM_LIST[] = {
    { &TLS_STREAM },
};

enum tls_connection_state ss_TLS_state(ss,stack_idx)
     struct streamsched *ss;
     int stack_idx;
{
    DPRINT(Debug,15,(&Debug,			    
		     "tls: tls_connection_state: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack)
	panic("TLS PANIC",__FILE__,__LINE__,"tls_connection_state",
	      "Bad stack index or stack",0);

    if (!ss->this_stack[stack_idx].generic_1)
	return tls_closed;

    if (ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"tls_connection_state",
	      "Bad stack index or stack",0);
    
    return ss->this_stack[stack_idx].generic_1->state;
}

#ifdef ANSI_C
provides_shared_ST_f provides_shared_ST;
#endif
struct provides_shared_ST * provides_shared_ST(count,s_size)
     int *count;
     size_t *s_size;
{
    *s_size = sizeof (STREAM_LIST[0]);

    *count = (sizeof STREAM_LIST) / *s_size;
    return &(STREAM_LIST[0]);
}

#ifdef ANSI_C
static SE_option_parse   tls_parse_on_option;
#endif
static int tls_parse_on_option(X,name,value)
     struct SE_option *X;
     const char       *name;   /* without prefix */
     const char       * value;
{
    int ret = 1;

    if (X->type != &tls_options ||
	!X->value) {
	panic("TLS PANIC",__FILE__,__LINE__,"tls_parse_on_option",
	      "Bad option storage",0);
    }

    if (0 == istrcmp(name,"starttls-version")) {
	if (!value) {
	    lib_error(FRM("TLS option %s:%s requires value"),
		      X->prefix,name);
	    ret = 0;
	} else if (0 == istrcmp("ssl",value))
	    X->value->v_starttls = v_ssl;
	else if (0 == istrcmp("tls",value))
	    X->value->v_starttls = v_tls;
	else if (0 == istrcmp("none",value))
	    X->value->v_starttls = tls_none;
	else {
	    /* Error message ? */
	    lib_error(FRM("Bad TLS option %s:%s value %s"),
		      X->prefix,name,value);
	    ret = 0;
	}
    } else {
	/* Error message ? */
	lib_error(FRM("Bad TLS option %s:%s"),
		  X->prefix,name);
	ret = 0;
    }
	
    return ret;
}

static SE_option_values  tls_give_values;
/* return: malloced string -- caller must free */
static  char * tls_give_values(X,prefix)
     struct     SE_option *X;
     CONST char *prefix;
{
    char * ret;
    char * Y = "???";

    if (X->type != &tls_options ||
	!X->value) {
	panic("TLS PANIC",__FILE__,__LINE__,"tls_give_values",
	      "Bad option storage",0);
    }

    switch (X->value->v_starttls) {
    case tls_none: Y = "none"; break;
    case v_ssl:    Y = "ssl"; break;
    case v_tls:    Y = "tls"; break;
    }
    
    ret = elm_message(FRM("%s:starttls-version=%s"),
		      prefix,Y);
    
    return ret;
}

static SE_option_zero    tls_zero_options;
static void tls_zero_options(X)
     struct     SE_option *X;
{
    X->value = safe_malloc(sizeof (struct SE_option_value));

    bzero((void *)X->value, sizeof (*(X->value)));

    X->value->v_starttls = v_tls;  /* Default: Use TLS */
}

static SE_option_free    tls_free_options;
static void tls_free_options(X)
     struct     SE_option *X;
{
    if (X->value) {
	free(X->value);
	X->value = NULL;
    }
}

struct SE_option_type tls_options = {
    tls_parse_on_option,
    tls_give_values,
    tls_zero_options,
    tls_free_options
};

static struct provides_shared_SEOT OPTION_LIST[] = {
    &tls_options
};

#if ANSI_C
provides_shared_SEOT_f provides_shared_SEOT;
#endif
struct provides_shared_SEOT * provides_shared_SEOT(count,s_size)
     int *count; 
     size_t *s_size;
{
    *s_size = sizeof (OPTION_LIST[0]);

    *count = (sizeof OPTION_LIST) / *s_size;
    return &(OPTION_LIST[0]);
}

#include "../../hdrs/rc_imp.h"
#include "../../hdrs/save_opts.h"


static ZZZ_SAVE_TYPE save_info_data[] = {
    { "rand-egd",    -1L, ZZZ_DT_STR(raw_rand_egd),
      sizeof raw_rand_egd, NULL },
    { "rand-file",   -1L, ZZZ_DT_STR(raw_rand_file),
      sizeof raw_rand_file, NULL },

};

#if ANSI_C
provides_RC_options_f provides_RC_options;
#endif
struct rc_save_info_rec * provides_RC_options(count,s_size)
     int *count; 
     size_t *s_size;
{
    *s_size = sizeof (save_info_data[0]);

    *count = (sizeof save_info_data) / *s_size;
    return (struct rc_save_info_rec *) save_info_data;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 * End:
 */



