static char rcsid[] = "@(#)$Id: mime_parse.c,v 1.25 2001/06/08 17:54:01 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.25 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@ozone.FMI.FI>
 *
 *  Initially written by: Michael Elkins <elkins@aero.org>, 1995
 *****************************************************************************/

#include "headers.h"
#include "melib.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"mime");

/* As ordered in mime.h */
char *mime_types[] = {
	"*unknown*",
	"application",
	"audio",
	"image",
	"message",
	"multipart",
	"text",
	"video",
	/* Non standard main types */
	"x-world",        /* x-word/x-vmrl */
	"model",          /* draft-nelson-model-mail-ext-02.txt */
        NULL
};

void
mime_destroy (ptr)
     mime_t *ptr;
{
    mime_t *tmp;

    DPRINT(Debug,15,(&Debug,
		    "mime_destroy(%p) --> BEGIN\n",ptr));
    
    if (ptr && ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"mime_destroy",
		   "Bad magic number");
    
    while (ptr) {
	tmp = ptr;
	ptr = ptr->next;
	
	if (tmp->magic != MIME_magic)
	    mime_panic(__FILE__,__LINE__,"mime_destroy",
		       "Bad magic number (next -chain)");
	
	if (tmp->description)
	    free_string (&(tmp->description));
	if (tmp->type_opts)
	    free (tmp->type_opts);
	if (tmp->disposition_opts)
	    free (tmp->disposition_opts);
	
	if (tmp->parts)
	    mime_destroy (tmp->parts);
	
	if (tmp->unlink)
	    unlink (tmp->pathname);

	if (tmp->pathname)
	    free (tmp->pathname);
	tmp->pathname = NULL;

	if (tmp->dispname)
	    free_string(& (tmp->dispname));

	tmp -> magic = 0;

	free (tmp);
  }
    DPRINT(Debug,15,(&Debug,
		     "mime_destroy(...) <-- END\n"));
    return;
}

void mime_t_clear (mt)
     mime_t *mt;
{
    DPRINT(Debug,15,(&Debug,
		     "mime_t_clear(%p) --> BEGIN\n",mt));

    if (mt->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"mime_t_clear",
		   "Bad magic number");
    
    mt->flags = mt->offset = mt->begin_offset = 0;
    mt->length = -1;
    mt->encoding = ENCODING_7BIT;
    mt->unlink = 0;
    mt->type = MIME_TYPE_TEXT;
    mt->disposition = DISP_INLINE;
    mt->notplain = 0;
    
    if (mt->parts)
	mime_destroy (mt->parts);
    if (mt->next)
	mime_destroy (mt->next);

    mt->next = mt->prev = mt->parts = NULL;

    if (mt->description)
	free_string (&(mt->description));
    if (mt->type_opts)
	free (mt->type_opts);
    mt->type_opts = NULL;
    if (mt->disposition_opts)
	free (mt->disposition_opts);
    mt->disposition_opts = NULL;
    
    if (mt->pathname)
	free (mt->pathname);
    mt->pathname = NULL;
    if (mt->dispname)
	free_string(& (mt->dispname));
    
    strfcpy (mt->subtype, "plain", sizeof mt->subtype);
    
    DPRINT(Debug,15,(&Debug,
		     "mime_t_clear(%p) <-- END\n",mt));
    return;
}

void
mime_get_disposition (str, mt)
     char *str;
     mime_t *mt;
{
  char *c, tmp[VERY_LONG_STRING];

  DPRINT(Debug,9,(&Debug,
		  "mime_get_disposition(): str=\"%s\"\n", str));
  
  if (mt->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"mime_get_disposition",
	       "Bad magic number");

  /* Don't harm "str" */
  strfcpy (tmp, str, sizeof(tmp));
  
  rfc822_reap_comments (tmp, NULL, 0);
  
  /* Look for the options field */
  if ((c = strchr (tmp, ';')) != NULL) {
    char *d = c;
    while (d > tmp && whitespace(*(d-1)))
      d--;
    *d = '\0';
    c++;
    while (*c && whitespace(*c))
      c++;
    mt->disposition_opts = strmcpy (mt->disposition_opts, c);
  }
  else {
    char *d = tmp + strlen(tmp);
    while (d > tmp && whitespace(*(d-1)))
      d--;
    *d = '\0';
    if (mt->disposition_opts) {
      free (mt->disposition_opts);
      mt->disposition_opts = NULL;
    }
  }
  
  /* All that's left now is the main disposition */
  c = tmp;
  while (*c && whitespace(*c))
    c++;
  /* No Content-Disposition -header     -> DISP_INLINE
   *    Content-Disposition: inline     -> DISP_INLINE
   *    Content-Disposition: attachment -> DISP_ATTACH
   *    Content-Disposition: {unknown}  -> DISP_ATTACH
   * See RFC 1806 (Experimental protocol) for details.
   */   
  if (istrcmp (c, "inline") != 0)
    mt->disposition = DISP_ATTACH;
  else
    mt->disposition = DISP_INLINE;
     
  DPRINT(Debug,9,(&Debug,
		  "mime_get_disposition(): disposition=\"%s\", disposition_opts=\"%s\"\n",
		  DISPOSITION(mt->disposition), NONULL(mt->disposition_opts)));
}

int mime_check_type (str)
     char *str;
{
  int i;
  for (i = 0; mime_types[i] != NULL; i++) {
    if (istrcmp(str,mime_types[i]) == 0)
      return i;
  }
  return MIME_TYPE_UNKNOWN;
}

void
mime_get_content (str, mt)
     char *str;
     mime_t *mt;
{
    char *c, tmp[VERY_LONG_STRING];
    
    DPRINT(Debug,9,(&Debug,
		    "mime_get_content(): str=\"%s\"\n", str));
    
    if (mt->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"mime_get_content",
		   "Bad magic number");
    
    /* Don't harm "str" */
    strfcpy (tmp, str, sizeof(tmp));
    
    rfc822_reap_comments (tmp, NULL, 0);
    
    /* Look for the options field */
    if ((c = strchr (tmp, ';')) != NULL) {
	char *d = c;
	while (d > tmp && whitespace(*(d-1)))
	    d--;
	*d = '\0';
	c++;
	while (*c && whitespace(*c))
	    c++;
	mt->type_opts = strmcpy (mt->type_opts, c);
    }
    else {
	char *d = tmp + strlen(tmp);
	while (d > tmp && whitespace(*(d-1)))
	    d--;
	*d = '\0';
	if (mt->type_opts) {
	    free (mt->type_opts);
	    mt->type_opts = NULL;
	}
    }
    
    mt->subtype[0] = '\0';
    /* Get the subtype */
    if ((c = strchr (tmp, '/')) != NULL) {
	char *d = c;
	while (d > tmp && whitespace(*(d-1)))
	    d--;
	*d = '\0';
	
	c++;
	while (*c && whitespace(*c))
	    c++;
	strfcpy (mt->subtype, c, sizeof(mt->subtype));
    }
    
    /* All that's left now is the main type */
    c = tmp;
    while (*c && whitespace(*c))
	c++;
    mt->type = mime_check_type (c);
    
    /* Mark MESSAGE/RFC822 so we can do special handling later */
    if (mt->type == MIME_TYPE_MESSAGE && istrcmp (mt->subtype, "rfc822") == 0)
	mt->flags |= MIME_RFC822;
    else if (mt->type == MIME_TYPE_MULTIPART) {
	if (istrcmp (mt->subtype, "mixed") == 0)
	    mt->flags |= MIME_MIXED;
	if (istrcmp (mt->subtype, "report") == 0)
	    mt->flags |= MIME_MIXED;
	else if (istrcmp (mt->subtype, "digest") == 0)
	    mt->flags |= MIME_DIGEST;
	else if (istrcmp (mt->subtype, "alternative") == 0)
	    mt->flags |= MIME_ALTERNATIVE;
	else if (istrcmp (mt->subtype, "signed") == 0)
	    mt->flags |= MIME_SIGNED;
	else if (istrcmp (mt->subtype, "encrypted") == 0)
	    mt->flags |= MIME_ENCRYPTED;
    }

    DPRINT(Debug,9,(&Debug,
		    "mime_get_content(): type=\"%s\", subtype=\"%s\", opts=\"%s\"\n",
		    mime_types[mt->type], mt->subtype, NONULL(mt->type_opts)));
    
  return;
}

void mime_get_boundary (boundary, opts, size)
     char *opts, *boundary;
     int size;
{
  if (!mime_get_param ("boundary", boundary, opts, size)) {
    lib_error(CATGETS(elm_msg_cat, MeSet, MeParseNoBoundary,
		      "'boundary' paramater is missing from Multipart -type!"));
  }

  return;
}

int class_charset(charset_value) 
     char * charset_value;
{
    /* Return > 0  (2)  if charset is displayable with display_charset
     * Return < 0  (-2) if charset needs translating to display_charset
     */
    int ret = 0;
    charset_t charset_value_p = MIME_name_to_charset(charset_value,0);
    
    /* Check agaist display charset */
    if (0 == istrcmp(charset_value,"US-ASCII") && 
	charset_ok_p(display_charset))
	ret = 2; 
    /* If display_charset can show us-ascii? */
    else if (display_charset->MIME_name &&
	     0 == istrcmp(charset_value,display_charset->MIME_name))
	ret = 2;
    /* if charset is subset of display */
    else if (charset_value_p && charset_superset_of(display_charset,
						    charset_value_p))
	ret = 2;
    else if (charset_value_p && 
	     (CS_printable & charset_properties(charset_value_p)))
	ret = -2;  /* convert (filter?) to display charset */
    else
	ret = 0;
    
    DPRINT(Debug,9,(&Debug,
		    "class_charset: charset_value=\"%s\"\n",
		    charset_value));
    DPRINT(Debug,9,(&Debug,
		    "             : display_charset=\"%s\"\n",
		    display_charset->MIME_name ? 
		    display_charset->MIME_name :
	      "<no MIME name>"));
    DPRINT(Debug,9,(&Debug,
		    "class_charset=%d\n",ret));

    return ret;
}

int mime_get_charset (charset_value, opts, display_charset)
     CONST char *opts; 
     charset_t *charset_value;
     charset_t * display_charset; /* vector */
{  
    /* Return > 0  (2)  if charset is displayable with display_charset
     * Return < 0  (-2) if charset needs translating to display_charset
     */
    int ret = 0,j;
    char buffer[80];

    if (!mime_get_param("charset",buffer,opts,sizeof buffer)) 
	strfcpy(buffer,"US-ASCII", sizeof buffer); 
    /* Default charset if nothing specified */

    *charset_value = MIME_name_to_charset(buffer,1);

    if (!opts) {
	DPRINT(Debug,9,(&Debug,
			"mime_get_charset: opts=NULL\n"));
    } else {
	DPRINT(Debug,9,(&Debug,
			"mime_get_charset: opts=\"%s\"\n",opts));
    }
    DPRINT(Debug,9,(&Debug,
		    "                : charset_value=%p '%s' (type=%p)\n",
		    *charset_value,
		    (*charset_value)->MIME_name,
		    (*charset_value)->charset_type));
    
    for (j = 0; display_charset[j]; j++) {
	DPRINT(Debug,9,(&Debug,
			"            [%d] : display_charset=%p '%s' (type=%p)\n",
			j,display_charset[j],
			display_charset[j]->MIME_name ? 
			display_charset[j]->MIME_name :
			"<no MIME name>",
			display_charset[j]->charset_type));
	
	
	if (display_charset[j]->MIME_name &&
	    0 == istrcmp(buffer,display_charset[j]->MIME_name)) {
	    ret = 2; /* If display charset              */
	    break;
	} else if (charset_superset_of(display_charset[j],*charset_value)) {
	    ret = 2; /* if charset is subset of display */
	    break;
	} else if (0 == istrcmp(buffer,"US-ASCII") && 
		   charset_ok_p(display_charset[j])) {
	    ret = 2; /* If display_charset can show us-ascii? */
	    break;
	}
    }
    if ( 0 == ret) {
	if (CS_printable & charset_properties(*charset_value))
	    ret = -2;  /* convert (filter?) to display charset */
	else
	    ret = 0;
    }
    DPRINT(Debug,9,(&Debug,
		    "mime_get_charset=%d\n",ret));

    return ret;
}

void
mime_t_zero (ptr)
     mime_t *ptr;
{
  /* This routine should be called whenever a new "mime_t" is created.  It
   * makes sure that the pointers inside are correctly initialized to NULL
   * so that we don't end up calling free() on an uninitialized pointer.
   */
    DPRINT(Debug,15,(&Debug,
		     "mime_t_zero(%p)\n",ptr));

  ptr->next = ptr->parts = NULL;
  ptr->type_opts = ptr->disposition_opts = 
      ptr->pathname = NULL;
  ptr->dispname = NULL;
  ptr->description = NULL;

  ptr->magic = MIME_magic;
  

  ptr->flags = ptr->offset = ptr->begin_offset = 0;
  ptr->length = -1;
  ptr->encoding = ENCODING_7BIT;
  ptr->unlink = 0;
  ptr->type = MIME_TYPE_TEXT;
  ptr->disposition = DISP_INLINE;
  ptr->notplain = 0;
  strfcpy (ptr->subtype, "plain", sizeof ptr->subtype);
}

void mime_t_copy(trg, src)
     mime_t *trg, *src;
{
  /* This routines make copy of mime_t structure ... */

    DPRINT(Debug,15,(&Debug,
		     "mime_t_copy(%p,%p) --> BEGIN\n",trg,src));
  
  if (trg->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"mime_t_copy",
	       "Bad magic number (trg)");

  if (src->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"mime_t_copy",
	       "Bad magic number (src)");

  mime_t_clear(trg);

  trg->flags  =            src->flags;
  trg->offset =            src->offset; 
  trg->begin_offset  =     src->begin_offset; 
  trg->length =            src->length;
  trg->encoding =          src->encoding;
  trg->unlink = 0;         /* Don't unlink ! */
  trg->type =              src->type;
  trg->disposition =       src->disposition;
  trg->notplain =          src->notplain;

  if (src->parts) {
      trg->parts = mime_t_alloc();
      mime_t_copy(trg->parts,src->parts);
  }

  if (src->next) {
      trg->next = mime_t_alloc();
      mime_t_copy(trg->next,src->next);
  }

  if (src->description) {
      trg->description = dup_string(src->description);
  }

  if (src->type_opts) {
      trg->type_opts = strmcpy(trg->type_opts, src->type_opts);
  }

  if (src->disposition_opts) {
      trg->disposition_opts = strmcpy(trg->disposition_opts, 
				      src->disposition_opts);
  }

  strfcpy(trg->subtype,src->subtype, sizeof trg->subtype);

  DPRINT(Debug,15,(&Debug,
		   "mime_t_copy(%p,%p) <-- END\n",trg,src));
}

mime_t *
mime_t_alloc ()
{
  mime_t *ptr;

  DPRINT(Debug,15,(&Debug,
		   "mime_t_alloc()     --> BEGIN\n"));

  ptr = (mime_t *) safe_malloc (sizeof (mime_t));
  /* Make sure to clear the pointers initially so that later we know when
   * to reclaim memory in mime_t_clear().
   */
  mime_t_zero (ptr);
  mime_t_clear (ptr);

    DPRINT(Debug,15,(&Debug,
		     "mime_t_alloc() = %p <-- END\n",ptr));
    return ptr;
}

void parse_mime_headers1 (ptr,headers,part_offset,body_offset,opts, 
			  hdr_charset)
     mime_t *ptr;
     header_list_ptr headers;
     long part_offset;
     long body_offset;
     int opts;
     charset_t hdr_charset;
{
    header_list_ptr this_header;

    /* set some defaults */
    ptr->encoding = ENCODING_7BIT;
    if (opts & MIME_DIGEST) {
	ptr->type = MIME_TYPE_MESSAGE;
	strfcpy (ptr->subtype, "rfc822", sizeof ptr->subtype);
    } else {
	ptr->type = MIME_TYPE_TEXT;
	strfcpy (ptr->subtype, "plain", sizeof ptr->subtype);
    }
    ptr->disposition = DISP_INLINE;
    ptr->description = NULL;
    
    ptr->begin_offset = part_offset;
    ptr->offset       = body_offset;
    ptr->length = -1;
    
    if (NULL != (this_header = 
		 locate_header_by_name(headers,"Content-Type")) &&
	NULL != this_header->body) {
	mime_get_content (this_header->body, ptr);
	if (this_header->next_this_header &&
	    show_header_errors) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeParseErrorContent,
			"PARSE ERROR: Several Content-Type headers!"));
	}
    } else {
	DPRINT(Debug,9,(&Debug,
			"parse_mime_headers1: No Content-Type -header\n"));
    }

    if (NULL != (this_header = 
		 locate_header_by_name(headers,"Content-Disposition")) &&
	NULL != this_header->body) {
	mime_get_disposition (this_header->body, ptr);
	if (this_header->next_this_header &&
	    show_header_errors) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeParseErrorDisposition,
			      "PARSE ERROR: Several Content-Disposition headers!"));
	}
    } else {
	DPRINT(Debug,9,(&Debug,
			"parse_mime_headers1: No Content-Disposition -header\n"));
    }

    if (NULL != (this_header = 
		 locate_header_by_name(headers,"Content-Transfer-Encoding")) &&
	NULL != this_header->body) {
	char  * value = this_header->body, *c;
	
	/* This removes comments from buffer this_header->body */
	
	rfc822_reap_comments (value, NULL, 0);
	c = value;
	while (*c && isspace((unsigned char) *c))
	    c++;
	ptr->encoding = check_encoding (c);
	
	if (this_header->next_this_header &&
	    show_header_errors) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeParseErrorTransfer,
			      "PARSE ERROR: Several Content-Transfer-Encoding headers!"));
	}
    } else {
	DPRINT(Debug,9,(&Debug,
			"parse_mime_headers1: No Content-Transfer-Encoding -header\n"));
    }
    
    if (NULL != (this_header = 
		 locate_header_by_name(headers,"Content-Description"))  &&
	NULL != this_header->body) {
	char * c = this_header->body;

	while (*c && whitespace(*c))
	    c++;
	ptr->description = NULL;
	ptr->description = hdr_to_string(HDR_TEXT,c,
					 hdr_charset,
					 1);
					 	
	if (this_header->next_this_header &&
	    show_header_errors) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeParseErrorDescription,
			      "PARSE ERROR: Several Content-Description headers!"));
	}
    } else {
	DPRINT(Debug,9,(&Debug,
			"parse_mime_headers1: No Content-Description -header\n"));
    }

    /* mime_get_content don't set this if this is set as defualt value
     * because of MIME_DIGEST -flag (in opt)
     */
    if (ptr->type == MIME_TYPE_MESSAGE && 
	istrcmp (ptr->subtype, "rfc822") == 0)
	ptr->flags |= MIME_RFC822;
    
    if (mime_notplain(ptr)) {
	ptr->notplain = TRUE;
	DPRINT(Debug,9,(&Debug,
			"parse_mime_headers1: 'Not plain'\n"));
    }

}

mime_t * parse_mime_headers(headers,part_offset,body_offset,opts,
			    hdr_charset)
     header_list_ptr headers;
     long part_offset;
     long body_offset;
     int opts;
     charset_t hdr_charset;
{
  mime_t *ptr;

  DPRINT(Debug,9,(&Debug,
		  "parse_mime_headers(): part_offset=%ld, body_offset=%ld, opts=%d\n",
		  part_offset,body_offset,opts));

  ptr = mime_t_alloc ();

  parse_mime_headers1(ptr,headers,part_offset,body_offset,opts,hdr_charset);


  DPRINT(Debug,10,(&Debug,
		   "parse_mime_headers- type=%s/%s; flags=%d;\n",
		   mime_types[ptr->type], ptr->subtype, ptr->flags));
  DPRINT(Debug,10,(&Debug,
		   "parse_mime_headers- begin=%ld, offset=%ld, length=%ld\n",
		   ptr->begin_offset,ptr->offset,ptr->length));

  DPRINT(Debug,9,(&Debug,
		  "parse_mime_headers=%p <-- END\n",(void *)ptr));

  return ptr;
}

mime_t * mime_read_header (fp, opts, defcharset)
     FILE *fp;
     int opts;
     charset_t defcharset;
{
  mime_t *ptr;
  header_list_ptr headers = NULL;
  long part_offset;
  long body_offset;


  DPRINT(Debug,19,(&Debug,
		   "mime_read_header: opts=%d --> START\n",opts));

  part_offset = ftell (fp);
  headers = file_read_headers(fp,0);
  body_offset = ftell(fp);

  ptr = parse_mime_headers(headers,part_offset,body_offset,opts,
			   defcharset);

  delete_headers(&headers);

  return ptr;
}

mime_t * multipart_parse (fp, length, boundary, opts, defcharset)
     FILE *fp;
     int length, opts;
     char *boundary;
     charset_t defcharset;
{
  int blen, len,last_pos;
  long end_offset, debug_pos;
  char buf[VERY_LONG_STRING], subbound[STRING];
  mime_t *ptr = NULL, *tmp, *ret = NULL;

  DPRINT(Debug,9,(&Debug,
		  "multipart_parse --> length=%d, boundary=%s\n",
		  length, boundary));

  blen = strlen (boundary);
  end_offset = ftell (fp) + length;
  last_pos = ftell(fp);

  while ((debug_pos = ftell (fp)) < end_offset) {
    if ((len = mail_gets (buf, VERY_LONG_STRING, fp)) == 0)
      break;

    DPRINT(Debug,49,(&Debug,
		     "multipart_parse: Readed %d bytes from %ld; last_pos=%d\n",
		     len,debug_pos,last_pos));


    if (buf[0] == '-' && buf[1] == '-' &&
        strncmp (buf + 2, boundary, blen) == 0) {

      /* Save the length of the previous part */
      if (ptr) {
        ptr->length = last_pos - ptr->offset;
	DPRINT(Debug,9,(&Debug,
			"multipart_parse: fixing length=%ld\n",
			(long) ptr->length));
	DPRINT(Debug,49,(&Debug,
			 "              : last_pos=%ld, offset=%ld\n",
			 (long) last_pos, (long) ptr->offset));
      }
      /* Check for the end boundary. */
      if (buf[blen+2] == '-' && buf[blen+3] == '-')
        break;
      
      tmp = mime_read_header (fp, opts, defcharset);

      DPRINT(Debug,9,(&Debug,
		      "multipart_parse: (reading) content-type=%s/%s; flags=%d\n",
		      mime_types[tmp->type], tmp->subtype, tmp->flags));

      
      if (ret == NULL)
        ptr = ret = tmp;
      else {
        ptr->next = tmp;
        ptr = ptr->next;
      }
#if 0
      if (ptr->length >= 0) {
	/* If the length of this part is known, skip ahead to the next
	 * part.  If the length is not known, we don't have to worry
	 * about it because the algorithm will search for the next
	 * boundary...
	 */
        fseek (fp, (long) (ptr->length), SEEK_CUR);
	continue;
      }
#endif
    }
    if (ptr && ptr->length < 0) { /* mark position before CR LF */
      int pos = ftell(fp);
      if (len > 1 && buf[len-2] == '\r' && buf[len-1] == '\n')
	last_pos = pos -2;
      else if (len > 0 && buf[len-1] == '\n')
	last_pos = pos -1;
    }
  }

  if (ptr && ptr->length != last_pos - ptr->offset) {
    ptr->length = last_pos - ptr->offset;
    DPRINT(Debug,9,(&Debug,
		    "multipart_parse: fixing length=%ld (corrupted?)\n",
		    (long) ptr->length));

    lib_error(CATGETS(elm_msg_cat, MeSet, MeParseMultipartCorrupted,
		      "Seems that multipart structure was corrupted."));
  }

  /* Now that we know what this message consists of, see if any of the
   * parts contain data that needs to be parsed. */

  for (tmp = ret; tmp != NULL; tmp = tmp->next) {

    if (tmp->magic != MIME_magic)
      mime_panic(__FILE__,__LINE__,"multipart_parse",
		 "Bad magic number (next -chain)");

    DPRINT(Debug,9,(&Debug,
		    "multipart_parse: (parsing) content-type=%s/%s; flags=%d\n",
		    mime_types[tmp->type],tmp->subtype,tmp->flags));

    if (tmp->flags & MIME_RFC822) {
	DPRINT(Debug,9,(&Debug,
			"multipart_parse- (parsing) RFC822\n"));
      fseek (fp, tmp->offset, SEEK_SET);
      tmp->parts = rfc822_parse (fp, tmp->length, defcharset);
    }
    else if (tmp->type == MIME_TYPE_MULTIPART) {
      fseek (fp, tmp->offset, SEEK_SET);
      mime_get_boundary (subbound, tmp->type_opts, sizeof (subbound));
      DPRINT(Debug,9,(&Debug,
		      "multipart_parse- (parsing) MULTIPART; boundary=%s\n",
		      subbound));
      tmp->parts = multipart_parse (fp, tmp->length, subbound, tmp->flags,
				    defcharset);
    }
  }

  /* Make sure to leave the stream at the end of the data since the
   * calling function might be assuming this.  */
  fseek (fp, end_offset, SEEK_SET);

  DPRINT(Debug,9,(&Debug,
		  "multipart_parse <-- DONE\n"));

  return ret;
}

mime_t *
rfc822_parse (fp, len, defcharset)
     FILE *fp;
     int len;
     charset_t defcharset;
{
    /* Called to read MESSAGE/RFC822 data.  First reads the header of the
     * message for MIME information, then (when necessary) calls other
     * functions to determine the content of MULTIPART or MESSAGE/RFC822
     * data contained.
     */
    mime_t *ret = NULL;
    long part_offset = ftell (fp);
    long body_offset;
    long end_offset  =  part_offset + len;
    header_list_ptr headers = NULL, mime_version, content_type;
    int pre_mime_content_type = 0;
    
    DPRINT(Debug,9,(&Debug,
		    "rfc822_parse --> len=%d\n",len));
    
    headers     = file_read_headers(fp,0);
    body_offset = ftell(fp);
    
    DPRINT(Debug,9,(&Debug,
		    "rfc822_parse- part_offset=%ld, body_offset=%ld, end_offset=%ld\n",
		    part_offset, body_offset, end_offset));
    
    if (!locate_header_by_name(headers,"From") &&
	!locate_header_by_name(headers,"Subject") &&
	!locate_header_by_name(headers,"To") &&
	!locate_header_by_name(headers,"CC")) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MeParseRFC822Corrupted,
			  "Seems that message/rfc822 data was corrupted."));
    }
    
    mime_version = locate_header_by_name(headers,"MIME-Version");
    content_type = locate_header_by_name(headers,"Content-Type");
    
    /* FIXME? update defcharset? */

    if (content_type && content_type ->body) {
	ret = mime_t_alloc();
	ret->begin_offset = part_offset;
	ret->offset       = body_offset;
	ret->length = -1;
	
	pre_mime_content_type = is_pre_mime_content_type(ret,
							 content_type->body);
    }
    
    if (mime_version && pre_mime_content_type) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MeParseRFC822PreWarn,
			  "Warning: message/rfc822 data with MIME-Version and pre-mime Content-type"));
    } 
    
    if (mime_version || 
	(!pre_mime_content_type && req_mime_bodyencoding)) {
	if (ret) mime_destroy(ret);
	ret = parse_mime_headers(headers,part_offset,body_offset,MIME_MIXED,
				 defcharset);  
    } else if (!ret) {
	ret = mime_t_alloc();
	ret->begin_offset = part_offset;
	ret->offset       = body_offset;
	ret->length = -1;
	
    }

    DPRINT(Debug,9,(&Debug,
		    "rfc822_parse: content-type=%s/%s; flags=%d\n",
		    mime_types[ret->type], ret->subtype,ret->flags));

    if (ret->length < 0) { 
	ret->length = end_offset - body_offset;
	DPRINT(Debug,9,(&Debug,
			"rfc822_parse: fixing length=%ld\n",
			(long) ret->length));
    }
    
    if (ret->type == MIME_TYPE_MULTIPART) {
	char boundary[STRING];
	
	mime_get_boundary (boundary, ret->type_opts, STRING);
	DPRINT(Debug,9,(&Debug,
			"rfc822_parse- (parsing) MULTIPART; boundary=%s\n",
			boundary));
	
	ret->parts = multipart_parse (fp, ret->length, boundary, ret->flags,
				      defcharset);
    }
    else if (ret->flags & MIME_RFC822) {
	DPRINT(Debug,9,(&Debug,
			"rfc822_parse- (parsing) RFC822\n"));
	ret->parts = rfc822_parse (fp, ret->length, defcharset);
    }
  
    delete_headers(&headers);
    
    /* Make sure the leave the stream at the end of the data! */
    fseek (fp, end_offset, SEEK_SET);

    DPRINT(Debug,9,(&Debug,
		    "rfc822_parse <-- DONE\n"));
    
    return ret;
}

void
mime_warnings(hdr) 
     struct header_rec *hdr;
{

  if (hdr->status & PRE_MIME_CONTENT) {
    lib_error(CATGETS(elm_msg_cat, MeSet, MeParsePreMime,
		      "Error: MIME-message has pre-MIME content-type!"));
  }
  if (hdr->status & MIME_UNSUPPORTED) {
    lib_error(CATGETS(elm_msg_cat, MeSet, MeParseUnsupportedMime,
		      "Warning: Unsupported MIME-Version!"));
  }
}

void
attach_parse (hdr, fp)
     struct header_rec *hdr;
     FILE *fp;
{
    int parsing = 0;
    /* This routine checks to see if the multipart messages specified by
     * "hdr" has been parsed for its subparts, and if not, calls the routine
     * to do so.
     */
    
    char boundary[STRING];
    
    if (hdr->mime_rec.magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"attach_parse",
		   "Bad magic number (mime_rec)");
    
    /* Copy value */
    if (hdr -> content_length >= 0)
	hdr->mime_rec.length = hdr -> content_length;
    
    if (hdr->mime_rec.begin_offset <= 0) {
	
	fseek(fp,hdr->offset,SEEK_SET);
	hdr->mime_rec.begin_offset = hdr->offset;
	/* Skip mailbox's separator lines ... */
	
	DPRINT(Debug,9,(&Debug,
			"attach_parse: scanning begin_offset: %ld\n",
			(long) hdr->mime_rec.begin_offset));
	
	hdr->mime_rec.begin_offset = skip_envelope(hdr,fp);
	
	DPRINT(Debug,9,(&Debug,
			"attach_parse: begin_offset=%ld\n",
			(long) hdr->mime_rec.begin_offset));
	
	if (hdr->mime_rec.begin_offset < 0) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeParseMailError,
			      "Can't parse mail..."));
	    return;
	}
    }

    if (hdr->mime_rec.type == MIME_TYPE_MULTIPART) {
	if (hdr->mime_rec.parts == NULL) {
	    mime_get_boundary (boundary, hdr->mime_rec.type_opts, STRING);
	    if (0 != fseek (fp, hdr->mime_rec.offset, SEEK_SET)) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeParseFailedSeek,
				  "Failed to seek beginning body..."));
	    } else {
		int tmp;
		parsing = 1;
		lib_transient(CATGETS(elm_msg_cat, MeSet, MeParsingMime,
				      "Parsing MIME structure..."));
		hdr->mime_rec.parts = 
		    multipart_parse (fp, hdr->content_length, boundary, 
				     hdr->mime_rec.flags, 
				     hdr->header_charset);
		/* Reconsider it */
		tmp = mime_notplain(&(hdr->mime_rec));
		if (tmp != hdr->mime_rec.notplain) {
		    hdr->mime_rec.notplain = tmp;
		    if (!tmp) {
			lib_error(CATGETS(elm_msg_cat, MeSet, MeParsingMimeNoMetamail,
					  "Parsing MIME structure... metamail not needed"));
			parsing = 2;
		    }
		}
	    }
	}
    } else if (hdr->mime_rec.flags & MIME_RFC822) {
	if (hdr->mime_rec.parts == NULL) {
	    if (0 != fseek (fp, hdr->mime_rec.offset, SEEK_SET)) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeParseFailedSeek,
				  "Failed to seek beginning body..."));
	    } else {
		int tmp;
		parsing = 1;
		lib_transient(CATGETS(elm_msg_cat, MeSet, MeParsingMime,
				      "Parsing MIME structure..."));
		hdr->mime_rec.parts = rfc822_parse (fp, hdr->content_length,
						    hdr->header_charset);
		/* Reconsider it */
		tmp = mime_notplain(&(hdr->mime_rec));
		if (tmp != hdr->mime_rec.notplain) {
		    hdr->mime_rec.notplain = tmp;
		    if (!tmp) {
			lib_error(CATGETS(elm_msg_cat, MeSet, MeParsingMimeNoMetamail,
					  "Parsing MIME structure... metamail not needed"));
			parsing = 2;
		    }
		}	
	    }
	}
    }
    
    if (parsing) {
	if (2 == parsing) {
	    if (sleepmsg > 0) {
#if POLL_METHOD
		wait_for_timeout ((sleepmsg+2)/3);
#else
		sleep ((sleepmsg+2)/3);
#endif
	    }
	}
    }
    return;
}

int is_text_type (primary_type, subtype, enc)
     char *primary_type, *subtype;
     int enc;
{
    /* encoding rules are different for text and no text 
     * for text we must do \n -> \r\n before base64 encoding */
    
    /* Current MIME Draft Standard says that part is text
     * (ie. line oriented) when it is Text/ -type -- otherwise no.
     *
     * new draft for Mime standards allows also other parts ot be text.
     * It says that content-transfer-encodings: 7bit and 8bit are only
     * allowed for line orienteed types */
    
    /* Actually Message/ and Multipart/ can't be newer encoded directly */
    if ( istrcmp(primary_type,"Message") == 0 ||
	 istrcmp(primary_type,"Multipart") == 0)
	return -1;
    
    if (istrcmp(primary_type,"Text") == 0) 
	return 1;               /* Subtypes of text are always text (ie.
				 * line oriented). */
    
    if (istrcmp(primary_type,"Application") == 0 &&
	istrcmp(subtype,"X-ELM-encode") == 0)
	return 1;            /*  That is text ... */
    
    if (enc == ENCODING_NONE || enc == ENCODING_7BIT || enc == ENCODING_8BIT)
	return 1;             /* It is text */
    
    if (enc == ENCODING_BINARY)
	return 0;            /* It is probably binary (or very long lines) */

    if (istrcmp(primary_type,"Application") == 0 &&
	istrcmp(subtype,"Postscript") == 0)
	return 1;            /*  Postscript is often text */
  
    return 0;              /* Default: It is binary */
}

int is_pre_mime_content_type (ptr,content_type)
     mime_t *ptr;
     char *content_type;
{
    int result;
    char *cptr = strpbrk(content_type,"/;()");

    DPRINT(Debug,10,(&Debug,
		     "is_pre_mime_content_type(): content_type=%s\n",
		     content_type));
    
    if (!cptr || ';' == *cptr) {
	char *ptr2;
	char *tmp = content_type;
	while (whitespace(*tmp))
	    tmp++;
	
	if (cptr)
	    *cptr = '\0';
	
	ptr2 = strpbrk(tmp," \t");
	if (ptr2)
	    *ptr2 = '\0';
	
	if (istrcmp(tmp,"text")!=0) {
	    char buf[STRING];
	    ptr -> notplain = TRUE;
	    
	    /* Put some 'intelligent' value */
	    ptr ->type = MIME_TYPE_APPLICATION;
	    elm_sfprintf(buf,sizeof buf,
			 FRM("X-RFC1049-%.30s"),tmp);
	    strfcpy(ptr->subtype,buf,sizeof(ptr->subtype));
	} else {
	    ptr -> notplain = FALSE;
	    ptr ->type = MIME_TYPE_TEXT;
	    strfcpy(ptr->subtype,"plain",sizeof(ptr->subtype));
	}
	result = 1;
    } else 
	result = 0;
    
    DPRINT(Debug,10,(&Debug,
		     "is_pre_mime_content_type=%d\n",result));
    
    return result;
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 * End:
 */
