/*
   ELF: a general ELF tool

   Author : Samy Al Bahra
   Website: http://www.kerneled.com
   E-mail : samy@kerneled.com

   License: GPL (read COPYING for details)
   Copyright(C) 2002-2003 Samy Al Bahra
*/


#include <sehdr.h>
#include <define.h>

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

int is_elf(const char *executable){
    FILE *binary;
    char *buffer;
    
    binary=fopen(executable, "r");
    if(binary==NULL){
	fprintf(stderr, "ERROR: could not open file: %s\n", strerror(errno));
	return 1;
    }
    
    buffer=(char *)malloc(SELFMAG+1);
    if(buffer==NULL){
	fprintf(stderr, "ERROR: could not allocate memory for magic number\n");
	return 1;
    }
    memset(buffer, 0, SELFMAG+1);
    
    if(fread(buffer, 1, SELFMAG, binary)!=SELFMAG){
	fprintf(stderr, "ERROR: could not read data from executable\n");
	return 1;
    }
    
    if(strcmp(ELFMAG, buffer)==0){
	free(buffer);
	fclose(binary);
	return 0;
    }
    else{
	free(buffer);
	fclose(binary);
	return 1;
    }
}


struct elf32_ehdr *get_ehdr(char *executable){
    struct elf32_ehdr *ehdr;
    FILE *binary;
    char *buffer;
    int size;
    
    binary=fopen(executable, "r");
    if(binary==NULL){
	fprintf(stderr, "ERROR: could not open file: %s\n", strerror(errno));
	return 0;
    }
    
    size=sizeof(struct elf32_ehdr);
    buffer=(char *)malloc(size);
    if(buffer==NULL){
	fprintf(stderr, "ERROR: could not initialize memory for elf header\n");
	return 0;
    }
    
    if(fread(buffer, 1, size, binary)!=size){
	fprintf(stderr, "ERROR: could not read elf header from executable\n");
	return 0;
    }

    ehdr=(struct elf32_ehdr *)buffer;
    
    switch(ehdr->e_ident[EI_OSABI]){
	case ELFOSABI_SYSV:
	    ehdr->se_abi="UNIX System V";
	    break;
	case ELFOSABI_HPUX:
	    ehdr->se_abi="HP-UX";
	    break;
	case ELFOSABI_NETBSD:
	    ehdr->se_abi="NetBSD";
	    break;
	case ELFOSABI_LINUX:
	    ehdr->se_abi="GNU/Linux";
	    break;
	case ELFOSABI_HURD:
	    ehdr->se_abi="GNU/Hurd";
	    break;
	case ELFOSABI_SOLARIS:
	    ehdr->se_abi="Solaris";
	    break;
	case ELFOSABI_AIX:
	    ehdr->se_abi="AIX";
	    break;
	case ELFOSABI_IRIX:
	    ehdr->se_abi="IRIX";
	    break;
	case ELFOSABI_FREEBSD:
	    ehdr->se_abi="FreeBSD";
	    break;
	case ELFOSABI_TRU64:
	    ehdr->se_abi="Tru64 UNIX";
	    break;
	case ELFOSABI_MODESTO:
	    ehdr->se_abi="Novell Modesto";
	    break;
	case ELFOSABI_OPENBSD:
	    ehdr->se_abi="OpenBSD";
	    break;
	case ELFOSABI_ARM:
	    ehdr->se_abi="ARM";
	    break;
	case ELFOSABI_STANDALONE:
	    ehdr->se_abi="Standalone (embedded) application";
	    break;
	default:
	    ehdr->se_abi="unknown ABI";
    }
    
    switch(ehdr->e_ident[EI_CLASS]){
	case ELFCLASSNONE:
	    ehdr->se_class="invalid class";
	    break;
	case ELFCLASS32:
	    ehdr->se_class="32-bit objects";
	    break;
	case ELFCLASS64:
	    ehdr->se_class="64-bit objects";
	    break;
	case ELFCLASSNUM:
	    ehdr->se_class="numerous";
	    break;
	default:
	    ehdr->se_class="unknown";
    }
    
    switch(ehdr->e_ident[EI_DATA]){
	case ELFDATANONE:
	    ehdr->se_data="invalid data encoding";
	    break;
	case ELFDATA2LSB:
	    ehdr->se_data="2s complement, little endian";
	    break;
	case ELFDATA2MSB:
	    ehdr->se_data="2s complement, big endian";
	    break;
	case ELFDATANUM:
	    ehdr->se_data="numerous";
	    break;
	default:
	    ehdr->se_data="unknown";
    } 
    
    switch(ehdr->e_type){
	case ET_NONE:
	    ehdr->se_type="no file type";
	    break;
	case ET_REL:
	    ehdr->se_type="relocatable file";
	    break;
	case ET_EXEC:
	    ehdr->se_type="executable file";
	    break;
	case ET_DYN:
	    ehdr->se_type="shared object file";
	    break;
	case ET_CORE:
	    ehdr->se_type="core file";
	    break;
	case ET_NUM:
	    ehdr->se_type="number of defined types";
	    break;
	case ET_LOOS:
	    ehdr->se_type="OS specific range start";
	    break;
	case ET_HIOS:
	    ehdr->se_type="OS specific range end";
	    break;
	case ET_LOPROC:
	    ehdr->se_type="processor specific range start";
	    break;
	case ET_HIPROC:
	    ehdr->se_type="processor specific range end";
	    break;
	default:
	    ehdr->se_type="unknown elf file";
    }
    
    switch(ehdr->e_machine){
	case EM_NONE:
	    ehdr->se_machine="no machine";
	    break;
	case EM_M32:
	    ehdr->se_machine="AT&T WE 32100";
	    break;
	case EM_SPARC:
	    ehdr->se_machine="Sun Sparc";
	    break;
	case EM_386:
	    ehdr->se_machine="Intel 80386";
	    break;
	case EM_68K:
	    ehdr->se_machine="Motorola m68k family";
	    break;
	case EM_88K:
	    ehdr->se_machine="Motorola m88k family";
	    break;
	case EM_860:
	    ehdr->se_machine="Intel 80860";
	    break;
	case EM_MIPS:
	    ehdr->se_machine="MIPS R3000 (big endian)";
	    break;
	case EM_S370:
	    ehdr->se_machine="IBM System/370";
	    break;
	case EM_MIPS_RS3_LE:
	    ehdr->se_machine="MIPS R3000 (little endian)";
	    break;
	case EM_PARISC:
	    ehdr->se_machine="HP PA-RISC";
	    break;
	case EM_VPP500:
	    ehdr->se_machine="Fujitsu VPP500";
	    break;
	case EM_SPARC32PLUS:
	    ehdr->se_machine="Sun Sparc version 8 plus";
	    break;
	case EM_960:
	    ehdr->se_machine="Intel 80960";
	    break;
	case EM_PPC:
	    ehdr->se_machine="PowerPC";
	    break;
	case EM_PPC64:
	    ehdr->se_machine="PowerPC (64-bit)";
	    break;
	case EM_S390:
	    ehdr->se_machine="IBM S390";
	    break;
	case EM_V800:
	    ehdr->se_machine="NEC V800 series";
	    break;
	case EM_FR20:
	    ehdr->se_machine="Fujitsu FR20";
	    break;
	case EM_RH32:
	    ehdr->se_machine="TRW RH-32";
	    break;
	case EM_RCE:
	    ehdr->se_machine="Motorola RCE";
	    break;
	case EM_ARM:
	    ehdr->se_machine="ARM";
	    break;
	case EM_FAKE_ALPHA:
	    ehdr->se_machine="Digital Alpha";
	    break;
	case EM_SH:
	    ehdr->se_machine="Hitachi SH";
	    break;
	case EM_SPARCV9:
	    ehdr->se_machine="Sun Sparc version 9 (64 bit)";
	    break;
	case EM_TRICORE:
	    ehdr->se_machine="Siemens Tricore";
	    break;
	case EM_ARC:
	    ehdr->se_machine="Argonaut RISC Core";
	    break;
	case EM_H8_300:
	    ehdr->se_machine="Hitachi H8/300";
	    break;
	case EM_H8_300H:
	    ehdr->se_machine="Hitachi H8/300H";
	    break;
	case EM_H8S:
	    ehdr->se_machine="Hitachi H8S";
	    break;
	case EM_H8_500:
	    ehdr->se_machine="Hitachi H8/500";
	    break;
	case EM_IA_64:
	    ehdr->se_machine="Intel Merced";
	    break;
	case EM_MIPS_X:
	    ehdr->se_machine="Stanford MIPS-X";
	    break;
	case EM_COLDFIRE:
	    ehdr->se_machine="Motorola Coldfire";
	    break;
	case EM_68HC12:
	    ehdr->se_machine="Motorola M68HC12";
	    break;
	case EM_MMA:
	    ehdr->se_machine="Fujitsu MMA Multimedia Accelerator";
	    break;
	case EM_PCP:
	    ehdr->se_machine="Siemens PCP";
	    break;
	case EM_NCPU:
	    ehdr->se_machine="Sony nCPU embedded RISC";
	    break;
	case EM_NDR1:
	    ehdr->se_machine="Denso NDR1";
	    break;
	case EM_STARCORE:
	    ehdr->se_machine="Motorola Start*Core";
	    break;
	case EM_ME16:
	    ehdr->se_machine="Toyota ME16";
	    break;
	case EM_ST100:
	    ehdr->se_machine="STMicroelectronic ST100";
	    break;
	case EM_TINYJ:
	    ehdr->se_machine="Advanced Logic Corp. TinyJ";
	    break;
	case EM_X86_64:
	    ehdr->se_machine="AMD x86 (64 bit) architecture";
	    break;
	case EM_PDSP:
	    ehdr->se_machine="Sony DSP Processor";
	    break;
	case EM_FX66:
	    ehdr->se_machine="Siemens FX66 Microcontroller";
	    break;
	case EM_ST9PLUS:
	    ehdr->se_machine="STMicroelectronics ST9+ 8/16 mc";
	    break;
	case EM_ST7:
	    ehdr->se_machine="STmicroelectronics ST7 8 bit mc";
	    break;
	case EM_68HC16:
	    ehdr->se_machine="Motorola MC68HC16 microcontroller";
	    break;
	case EM_68HC11:
	    ehdr->se_machine="Motorola MC68HC11 microcontroller";
	    break;
	case EM_68HC08:
	    ehdr->se_machine="Motorola MC68HC08 microcontroller";
	    break;
	case EM_68HC05:
	    ehdr->se_machine="Motorola MC68HC05 microcontroller";
	    break;
	case EM_SVX:
	    ehdr->se_machine="Silicon Graphics SVx";
	    break;
	case EM_AT19:
	    ehdr->se_machine="STMicroelectronics ST19 8 bit mc";
	    break;
	case EM_VAX:
	    ehdr->se_machine="Digital VAX";
	    break;
	case EM_CRIS:
	    ehdr->se_machine="Axis Communications 32-bit embedded processor";
	    break;
	case EM_JAVELIN:
	    ehdr->se_machine="Infineon Technologies 32-bit embedded processor";
	    break;
	case EM_FIREPATH:
	    ehdr->se_machine="Element 14 64-bit DSP processor";
	    break;
	case EM_ZSP:
	    ehdr->se_machine="LSI Logic 16-bit DSP processor";
	    break;
	case EM_MMIX:
	    ehdr->se_machine="Donald Knuth's educational 64-bit processor";
	    break;
	case EM_HUANY:
	    ehdr->se_machine="Harvard University machine-independent object files";
	    break;
	case EM_PRISM:
	    ehdr->se_machine="SiTera Prism";
	    break;
	case EM_AVR:
	    ehdr->se_machine="Atmel AVR 8-bit microcontroller";
	    break;
	case EM_FR30:
	    ehdr->se_machine="Fujitsu FR30";
	    break;
	case EM_D10V:
	    ehdr->se_machine="Mitsubishi D10V";
	    break;
	case EM_D30V:
	    ehdr->se_machine="Mitsubishi D30V";
	    break;
	case EM_V850:
	    ehdr->se_machine="NEC v850";
	    break;
	case EM_M32R:
	    ehdr->se_machine="Mitsubishi M32R";
	    break;
	case EM_MN10300:
	    ehdr->se_machine="Matsushita MN10300";
	    break;
	case EM_MN10200:
	    ehdr->se_machine="Matsushita MN10200";
	    break;
	case EM_PJ:
	    ehdr->se_machine="picoJava";
	    break;
	case EM_OPENRISC:
	    ehdr->se_machine="OpenRISC 32-bit embedded processor";
	    break;
	case EM_ARC_A5:
	    ehdr->se_machine="ARC Cores Tangent-A5";
	    break;
	case EM_XTENSA:
	    ehdr->se_machine="Tensilica Xtensa Architecture";
	    break;
	case EM_ALPHA:
	    ehdr->se_machine="Alpha";
	    break;
	case EM_NUM:
	    ehdr->se_machine="number of defined types";
	    break;
	default:
	    ehdr->se_machine="unknown machine";
    }
    
    switch(ehdr->e_version){
	case EV_NONE:
	    ehdr->se_version="invalid ELF version";
	    break;
	case EV_CURRENT:
	    ehdr->se_version="current ELF version";
	    break;
	case EV_NUM:
	    ehdr->se_version="numerous";
	    break;
	default:
	    ehdr->se_version="unknown";
    }
    
    fclose(binary);
    return ehdr;
}


int write_ehdr(const char *executable, struct elf32_ehdr *ehdr){
    FILE *fp;

    if(!(fp=fopen(executable, "r+"))){
        fprintf(stderr, "ERROR: could not open file, %s\n", strerror(errno));
	return 1;
    }

    if(fseek(fp, 0, SEEK_SET)){
        fprintf(stderr, "ERROR: could not rewind to beginning of file, %s\n", strerror(errno));
        return 1;
    }

    fwrite(ehdr, (sizeof(ehdr)-(sizeof(char *)*6)), 1, fp);
    fclose(fp);
}
