/****************************************************************************
*                                                                            
*  ADSP21K INTERFACE LIBRARY                                                 
*  -------------------------                                                 
*  coff21k.h : COFF interface routines (common object file format)           
*                                                                            
*  PROVIDED BY:                                                              
*  ------------                                                              
*  BittWare Research Systems, Inc.                                           
*  33 North Main Street                                                      
*  Concord, NH  03301                                                        
*  Ph: 800-848-0435 (US/Canada)                                              
*  Ph: 603-226-0404                                                          
*  Fax: 603-226-6667                                                         
*  BBS: 603-226-6668 (3-12-24/N/8/1)                                         
*  E-mail: support@bittware.com                                              
*                                                                            
*    Version   Changes                                                       
*    --------  -------------------------------------------------------       
*     1.0       6/22/92 - Initial coding                                     
*                                                                            
*****************************************************************************/

#ifndef _COFF21K_H
#define _COFF21K_H

/********************************************************
*             Common Object File Format
*    __________________________________
*    |          File Header            |
*    |_________________________________|
*    |        Optional Information     |
*    |_________________________________|
*    |        Section 1 Header         |
*    |_________________________________|
*    |             ...                 |
*    |_________________________________|
*    |        Section n Header         |
*    |_________________________________|
*    |       Raw Data for Section 1    |
*    |_________________________________|
*    |     Relocation Info for Sect. 1 |
*    |_________________________________|
*    |             ...                 |
*    |_________________________________|
*    |       Raw Data for Section n    |
*    |_________________________________|
*    |     Relocation Info for Sect. n |
*    |_________________________________|
*    |     Line Numbers for Section 1  |
*    |      ( not use at this time )   |
*    |_________________________________|
*    |             ...                 |
*    |_________________________________|
*    |     Line Numbers for Section n  |
*    |      ( not use at this time )   |
*    |_________________________________|
*    |         Symbol Table            |
*    |_________________________________|
*    |         String Table            |
*    |_________________________________|
*/

/* This is the object file header */
struct filehdr {
     unsigned short f_magic;  /* magic number */
     unsigned short f_nscns;  /* number of sections */
     long      f_timdat; /* time & date stamp */
     long      f_symptr; /* file pointer to symtab */
     long      f_nsyms;  /* number of symtab entries */
     unsigned short f_opthdr; /* sizeof(optional hdr) */
     unsigned short f_flags;  /* flags */
};
#define COFF_HDR       struct filehdr
#define COFF_HDR_SIZE  sizeof(COFF_HDR)

#ifndef DSP32C
  #define DSP21K_COFF_MAGIC 0x521C
  #define ISCOFF(x) ((x)==DSP21K_COFF_MAGIC)

#else
  /* FOR DSP32c COMPAT */
  #define    DSP32MAGICOLD 0x0160     /* previous magic number for DSP32 */
  #define    DSP32CMAGIC   0x0162     /* magic number for DSP32C */
  #define    DSP32MAGIC    0x0164     /* magic number for DSP32 */

  #define ISCOFF(x) ( ((x)==DSP21K_COFF_MAGIC) || \
		      ((x)==DSP32MAGICOLD)     || \
		      ((x)==DSP32CMAGIC)       || \
		      ((x)==DSP32MAGIC)         )
#endif

/*
 *   Bits for f_flags:
 *
 *  F_RELFLG   relocation info stripped from file
 *  F_EXEC     file is executable  (i.e. no unresolved
 *        externel references)
 *  F_LNO line nunbers stripped from file
 *  F_LSYS     local symbols stripped from file
 *  F_MINMAL   this is a minimal object file (".m") output of fextract
 *  F_UPDATE   this is a fully bound update file, output of ogen
 *  F_SWABD    this file has had its bytes swabbed (in names)
 *  F_AR16WR   this file has the byte ordering of an AR16WR machine
 *                 (e.g. 11/70)
 *  F_AR32WR   this file has the byte ordering of an AR32WR machine
 *                 (e.g. vax, I386 )
 *  F_AR32W    this file has the byte ordering of an AR32W machine
 *                 (e.g. 3b, sun3, sun4, SPARC, 680xxx, maxi)
 *  F_PATCH    file contains "patch" list in optional header
 *  F_NODF     (minimal file only) no decision functions for
 *             replaced functions
 */

#define  F_RELFLG   0000001
#define  F_EXEC     0000002
#define  F_LNNO     0000004
#define  F_LSYMS    0000010
#define  F_MINMAL   0000020
#define  F_UPDATE   0000040
#define  F_SWABD    0000100
#define  F_AR16WR   0000200
#define  F_AR32WR   0000400
#define  F_AR32W    0001000
#define  F_PATCH    0002000
#define  F_NODF     0002000

#define SEC_NAME_LEN 8

/* The section header */
typedef   struct {
     char      s_name[SEC_NAME_LEN];     /* section name */
     long      s_paddr;  /* physical address */
     long      s_vaddr;  /* virtual address */
     long      s_size;        /* section size */
     long      s_scnptr; /* file ptr to raw data for section */
     long      s_relptr; /* file ptr to relocation */
     long      s_lnnoptr;     /* file ptr to line numbers */
     unsigned short s_nreloc; /* number of relocation entries */
     unsigned short s_nlnno;  /* number of line number entries */
     long      s_flags;  /* flags */
} SCNHDR;

#define SCNHSZ    sizeof(SCNHDR)

/*
 * The low 2 bytes of s_flags is used as a section "type"
 */

#define STYP_REG    0x00 /* "regular" section: allocated, relocated, loaded */
#define STYP_DSECT  0x01 /* "dummy" section: not allocated, relocated, not loaded */
#define STYP_NOLOAD 0x02 /* "noload" section: allocated, relocated, not loaded */
#define STYP_GROUP  0x04 /* "grouped" section: formed of input sections */
#define STYP_PAD    0x08 /* "padding" section: not allocated, not relocated, loaded */
#define STYP_COPY   0x10 /* "copy" section:
			    for decision function used
			    by field update;  not
			    allocated, not relocated,
			    loaded;  reloc & lineno
			    entries processed normally */
#define STYP_INFO   0x200 /* comment section : not allocated not relocated, not loaded */
#define STYP_OVER   0x400 /* overlay section : relocated not allocated or loaded */
#define STYP_TEXT   0x20  /* section contains text only */
#define STYP_DATA   0x40  /* section contains data only */
#define STYP_BSS    0x80  /* section contains bss only */

/*
 *  In a minimal file or an update file, a new function
 *  (as compared with a replaced function) is indicated by S_NEWFCN
 */

#define S_NEWFCN  0x100
#define SYMNMLEN 8
#define SYMESZ   18      /* size of a symbol table entry */

typedef struct
{
     union
     {
	  char      _n_name[SYMNMLEN];  /* old COFF version */
	  struct
	  {
	       long _n_zeroes;     /* new == 0 */
	       long _n_offset;     /* offset into string table */
	  } _n_n;
	  char      *_n_nptr[2];   /* allows for overlaying */
     } _n;
     long           n_value;       /* value of symbol */
     short          n_scnum;       /* section number */
     unsigned short n_type;        /* type and derived type */
     char           n_sclass;      /* storage class */
     char           n_numaux;      /* number of aux. entries */
} SYMTBL;

#define n_name      _n._n_name
#define n_nptr      _n._n_nptr[1]
#define n_zeroes    _n._n_n._n_zeroes
#define n_offset    _n._n_n._n_offset

struct linebuf
{
     union
     {
	  long l_symndx;      /* symtbl indes of func name */
	  long l_paddr;       /* paddr of line number */
     } l_addr;
     unsigned short l_lnno;   /* line number */
};

#define LINEBUF struct linebuf
#define LINESZ 6

#define MAX_BUF_SIZE ((unsigned int) 64000L) /* max buffer size */
#define SEC_BUF_SIZE 300*30   /* "chunk" used when reading section data */
			      /* mult. of 6 and 5 (code/data byte width) */
#define MAX_LBL_LEN 26        /* maximum label length displayed */
#define MESS_LEN 120          /* maximum length of messages */

/* storage class mnemonics */
#define C_EFCN          -1      /* physical end of a function */
#define C_NULL          0       /* -- */
#define C_AUTO          1       /* automatic variable */
#define C_EXT           2       /* external symbol */
#define C_STAT          3       /* static */
#define C_REG           4       /* register variable */
#define C_EXTDEF        5       /* external definition */
#define C_LABEL         6       /* label */
#define C_ULABEL        7       /* undefined label */
#define C_MOS           8       /* member of structure */
#define C_ARG           9       /* function argument */
#define C_STRTAG        10      /* structure tag */
#define C_MOU           11      /* member of union */
#define C_UNTAG         12      /* union tag */
#define C_TPDEF         13      /* type definition */
#define C_USTATIC       14      /* uninitialized static */
#define C_ENTAG         15      /* unumeration tag */
#define C_MOE           16      /* member of enumeration */
#define C_REGPARM       17      /* register parameter */
#define C_FIELD         18      /* bit field */
#define C_BLOCK         100     /* beginning and end of block */
#define C_FCN           101     /* beginning and end of block */
#define C_EOS           102     /* end of structure */
#define C_FILE          103     /* filename */
#define C_LINE          104     /* used only by utility programs */
#define C_ALIAS         105     /* duplicate tag */
#define C_HIDDEN        106     /* like static, used to avoid name conflicts */

int upload (FILE *fp, int baud, int verbose);
unsigned long sym_lookup (FILE *fp, char *sym);

#endif /* _COFF21K_H */
