/***********************************************************
        Copyright 1991,1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[] = "$Id: TargetDB_Read.c,v 2.0 1995/02/13 20:48:42 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 * Author: Sohan C. Ramakrishna Pillai
 */

#include "depotlib.h"

#include "util.h"
#include "DepotErrorCodes.h"
#include "File.h"
#include "TargetDB.h"


static unsigned short TargetDB_ReadUShort();
static unsigned short TargetDB_ReadOctalUShort();
static unsigned int TargetDB_ReadUnsigned();
static int TargetDB_ReadChar();
static char *TargetDB_ReadString();
static TARGETSOURCELIST *TargetSourceList_Read();
static TARGETSOURCE *TargetSource_Read();
static TARGETDB *TargetDB_Tree_Read();
static TARGETDB *TargetDB_ReadUncompressed();



static unsigned short
TargetDB_ReadUShort(fp)
     FILE *fp;
{
  register int ch;
  unsigned short value;

  ch = getc(fp);
  while ((ch == ' ') | (ch == '\n'))
    ch = getc(fp);

  value = 0;
  if (ch == EOF) {
    FatalError(E_BADTARGETDBFILE, "Premature end to targetdatabase file.\n");
  } else {
    value = 0;
    while ((PROGRAM_ErrorNo == E_NULL)
	   && (ch >= '0') && (ch <= '9')) {
      value *= 10;
      value += (ch - '0');
      ch = getc(fp);
    }
  }

  return (PROGRAM_ErrorNo == E_NULL) ? value : 0;
}

static unsigned int
TargetDB_ReadUnsigned(fp)
     FILE *fp;
{
  register int ch;
  unsigned int value;

  ch = getc(fp);
  while ((ch == ' ') | (ch == '\n'))
    ch = getc(fp);

  value = 0;
  if (ch == EOF) {
    FatalError(E_BADTARGETDBFILE, "Premature end to targetdatabase file.\n");
  } else {
    value = 0;
    while ((PROGRAM_ErrorNo == E_NULL)
	   && (ch >= '0') && (ch <= '9')) {
      value *= 10;
      value += (ch - '0');
      ch = getc(fp);
    }
  }

  return (PROGRAM_ErrorNo == E_NULL) ? value : 0;
}


static unsigned short
TargetDB_ReadOctalUShort(fp)
     FILE *fp;
{
  register int ch;
  unsigned short value;

  ch = getc(fp);
  while ((ch == ' ') | (ch == '\n'))
    ch = getc(fp);

  value = 0;
  if (ch == EOF) {
    FatalError(E_BADTARGETDBFILE, "Premature end to targetdatabase file.\n");
  } else {
    value = 0;
    while ((PROGRAM_ErrorNo == E_NULL)
	   && (ch >= '0') && (ch <= '7')) {
      value *= 8;
      value += (ch - '0');
      ch = getc(fp);
    }
  }

  return (PROGRAM_ErrorNo == E_NULL) ? value : 0;
}


static int 
TargetDB_ReadChar(fp)
     FILE *fp;
{
  int ch;

  ch = getc(fp);

  return ch;
}

static char *
TargetDB_ReadString(fp)
     FILE *fp;
{
  register char *cp;
  register int i;
  register int ch;
  char *buffer;
  unsigned buffersize;
  unsigned increment = 100;

  ch = getc(fp);
  while ((ch == ' ') | (ch == '\n'))
    ch = getc(fp);

  if (ch == EOF) {
    FatalError(E_BADTARGETDBFILE, "Premature end to targetdatabase file.\n");
  } else {
    buffersize = increment;
    buffer = (char *) emalloc(buffersize * sizeof(char));
    cp = buffer;
    i = 0;
    while ((PROGRAM_ErrorNo == E_NULL)
	   && (ch != EOF) && (ch != ' ') && (ch != '\n')) {
      if (ch == '\\') {
	ch = getc(fp);
	if (ch == EOF) {
	  FatalError(E_BADTARGETDBFILE,
		     "Premature end to targetdatabase file.\n");
	}
      }
      if (PROGRAM_ErrorNo == E_NULL) {
	*cp++ = ch;
	i++;
	ch = getc(fp);
	if (i == buffersize) {	/* get more space for string */
	  buffersize += increment;
	  buffer = (char *) erealloc((void *) buffer,
				     buffersize * sizeof(char));
	}
      }
    }
    if (PROGRAM_ErrorNo == E_NULL)
      *cp = '\0';
  }

  return (PROGRAM_ErrorNo == E_NULL) ? buffer : NULL;
}



static TARGETSOURCE *
TargetSource_Read(fp, nodename, sourcepathset)
     FILE *fp;
     char *nodename;
     STRINGSET *sourcepathset;
{
  char buffer[MAXPATHLEN + 1];
  Boolean SecondarySourceExists;
  TARGETSOURCE *sourcep;
  int sourceindex;
  Boolean childofsource;

  sourcep = (TARGETSOURCE *) emalloc(sizeof(TARGETSOURCE));
  if (PROGRAM_ErrorNo == E_NULL) {
    sourceindex = efgetint(fp);
    if (sourceindex < 0) {
      TARGETSOURCE_Path(sourcep) = TargetDB_ReadString(fp);
    } else {
      if (StringArray_Size(sourcepathset) <= sourceindex) {
	FatalError(E_BADRANGE,
		   "Index %d > %d (expected size) for node %s at %s in the target database.\n",
		   sourceindex, StringArray_Size(sourcepathset), nodename, TARGETSOURCE_Path(sourcep));
      }
      if (PROGRAM_ErrorNo == E_NULL) {
	(void) strcpy(buffer, STRINGSET_String(sourcepathset, sourceindex));
	childofsource = TargetDB_ReadUShort(fp);
	if (childofsource) {
	  if (String_Comparator(buffer, "/") == 0) {
	    (void) strcpy(buffer, nodename);
	  } else {
	    (void) strcat(buffer, "/");
	    (void) strcat(buffer, nodename);
	  }
	}
	TARGETSOURCE_Path(sourcep) = String(buffer, strlen(buffer));
      }
    }
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    TARGETSOURCE_UpdateSpec(sourcep) = efgetunsigned(fp);
    TARGETSOURCE_OldUpdateSpec(sourcep) = TARGETSOURCE_UpdateSpec(sourcep);
  }
  if (PROGRAM_ErrorNo == E_NULL)
    TARGETSOURCE_CollectionId(sourcep) = efgetint(fp);
  if (PROGRAM_ErrorNo == E_NULL)
    SecondarySourceExists = TargetDB_ReadUShort(fp);
  if (PROGRAM_ErrorNo == E_NULL) {
    if (SecondarySourceExists)
      TARGETSOURCE_SecondarySource(sourcep) =
	TargetSource_Read(fp, nodename, sourcepathset);
    else
      TARGETSOURCE_SecondarySource(sourcep) = NULL;
  }
  if (PROGRAM_ErrorNo == E_NULL)
    TARGETSOURCE_Status(sourcep) = S_NULL;

  return (PROGRAM_ErrorNo == E_NULL) ? sourcep : NULL;
}


static TARGETSOURCELIST *
TargetSourceList_Read(fp, nodename, sourcepathset)
     FILE *fp;
     char *nodename;
     STRINGSET *sourcepathset;
{
  register int i;
  unsigned int nsources;
  TARGETSOURCELIST *sourcelist;

  nsources = TargetDB_ReadUnsigned(fp);
  if (PROGRAM_ErrorNo == E_NULL) {
    if (nsources > 0) {
      sourcelist = (TARGETSOURCELIST *) emalloc(sizeof(TARGETSOURCELIST));
      if (PROGRAM_ErrorNo == E_NULL)
	TARGETSOURCELIST_NSources(sourcelist) = nsources;
      if (PROGRAM_ErrorNo == E_NULL)
	TARGETSOURCELIST_SourceArray(sourcelist) =
	  (TARGETSOURCE **) emalloc(nsources * sizeof(TARGETSOURCE *));
      for (i = 0; ((i < nsources) && (PROGRAM_ErrorNo == E_NULL)); i++) {
	TARGETSOURCELIST_Source(sourcelist, i) =
	  TargetSource_Read(fp, nodename, sourcepathset);
      }
    } else
      sourcelist = NULL;
  }
  return (PROGRAM_ErrorNo == E_NULL) ? sourcelist : NULL;
}


static TARGETDB *
TargetDB_Tree_Read(fp, sourcepathset)
     FILE *fp;
     STRINGSET *sourcepathset;
{
  register TARGETDB **tp;
  register int i;
  char *entryname;
  TARGETDBENTRY *thisentryp;

  entryname = TargetDB_ReadString(fp);
  if ((entryname != NULL) && (PROGRAM_ErrorNo == E_NULL))
    thisentryp = (TARGETDBENTRY *) emalloc(sizeof(TARGETDBENTRY));
  if (PROGRAM_ErrorNo == E_NULL)
    TARGETDB_Name(thisentryp) = entryname;
  if (PROGRAM_ErrorNo == E_NULL) {
    TARGETDB_UpdateSpec(thisentryp) = efgetunsigned(fp);
    TARGETDB_OldUpdateSpec(thisentryp) = TARGETDB_UpdateSpec(thisentryp);
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    if (TARGETDB_UpdateSpec(thisentryp)
	& (U_OWNER | U_GROUP | U_MODE | U_SETUID | U_SETGID)) {
      TARGETDB_FileStatus(thisentryp) =
	(FILESTAT *) emalloc(sizeof(FILESTAT));
      TARGETDB_OldFileStatus(thisentryp) =
	(FILESTAT *) emalloc(sizeof(FILESTAT));
      if ((PROGRAM_ErrorNo == E_NULL)
	  && (TARGETDB_UpdateSpec(thisentryp) & (U_OWNER | U_SETUID))) {
	FILESTAT_Uid(TARGETDB_FileStatus(thisentryp)) =
	  FILESTAT_Uid(TARGETDB_OldFileStatus(thisentryp)) =
	  (uid_t)TargetDB_ReadUnsigned(fp);
      }
      if ((PROGRAM_ErrorNo == E_NULL)
	  && (TARGETDB_UpdateSpec(thisentryp) & (U_GROUP | U_SETGID))) {
	FILESTAT_Gid(TARGETDB_FileStatus(thisentryp)) =
	  FILESTAT_Gid(TARGETDB_OldFileStatus(thisentryp)) =
	  (gid_t)TargetDB_ReadUnsigned(fp);
      }
      if ((PROGRAM_ErrorNo == E_NULL)
	  && (TARGETDB_UpdateSpec(thisentryp) & U_MODE)) {
	FILESTAT_Mode(TARGETDB_FileStatus(thisentryp)) =
	  FILESTAT_Mode(TARGETDB_OldFileStatus(thisentryp)) =
	  TargetDB_ReadOctalUShort(fp);
      }
    } else {
      TARGETDB_FileStatus(thisentryp) = NULL;
    }
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    TARGETDB_SourceList(thisentryp) =
      TargetSourceList_Read(fp, TARGETDB_Name(thisentryp), sourcepathset);
  }
  if (PROGRAM_ErrorNo == E_NULL)
    TARGETDB_ChildCount(thisentryp) = TargetDB_ReadUnsigned(fp);
  if (PROGRAM_ErrorNo == E_NULL) {
    if (TARGETDB_ChildCount(thisentryp) <= 0) {
      TARGETDB_Children(thisentryp) = NULL;
    } else {
      STRINGSET *childsourcepathset = NULL;

      childsourcepathset =
	TargetSourceList_PathSet(TARGETDB_SourceList(thisentryp));
      TARGETDB_Children(thisentryp) =
	(TARGETDBENTRY **) emalloc((unsigned) TARGETDB_ChildCount(thisentryp) *
				   sizeof(TARGETDBENTRY *));

      for (i = 0, tp = TARGETDB_Children(thisentryp);
	   ((PROGRAM_ErrorNo == E_NULL)
	    && (i < TARGETDB_ChildCount(thisentryp)));
	   i++, tp++) {
	*tp = TargetDB_Tree_Read(fp, childsourcepathset);
      }
      if (childsourcepathset != NULL)
	StringArray_Free(childsourcepathset);
    }
  }
  if (PROGRAM_ErrorNo == E_NULL)
    TARGETDB_Status(thisentryp) = S_NULL;

  return (PROGRAM_ErrorNo == E_NULL) ? thisentryp : NULL;
}



static TARGETDB *
TargetDB_ReadUncompressed(fp)
     FILE *fp;
{
  STRINGSET *sourcepathset = NULL;
  int sourcepathsetsize;
  char *newstring;

  TARGETDB *TargetDB;

  sourcepathset = NULL;
  sourcepathsetsize = efgetint(fp);
  while ((PROGRAM_ErrorNo == E_NULL) && (sourcepathsetsize-- > 0)) {
    /* 
     * since we know this is a sorted set which was written out,
     * we append instead of add
     */
    newstring = efgetstring(fp, '\\');
    sourcepathset = StringArray_AppendString(sourcepathset, newstring);
    String_Free(newstring);
  }

  if (PROGRAM_ErrorNo == E_NULL)
    TargetDB = TargetDB_Tree_Read(fp, sourcepathset);

  if (PROGRAM_ErrorNo != E_NULL) {
    FatalError(E_TARGETDBREADFAILED,
	       "Attempt to read target database file failed.\n");
  }
  if (sourcepathset != NULL)
    StringArray_Free(sourcepathset);

  return (PROGRAM_ErrorNo == E_NULL) ? TargetDB : NULL;
}



TARGETDB *
TargetDB_Read(fp, compressflag)
     FILE *fp;
     Boolean compressflag;
{
  int pipefd[2];
  FILE *pipeinfp;
  int pid;
  int fd_targetdb;
  long pos_targetdb;
#ifdef HAVE_UNION_WAIT
  union wait ExecStatus;
#else
  int ExecStatus;
#endif

  TARGETDB *TargetDB;
  char *TargetDBVersionString;

  TargetDBVersionString = TargetDB_ReadString(fp);
  if (strcmp(TargetDBVersionString, TARGETDBVERSIONSTRING) == 0) {
    (void) free(TargetDBVersionString);
  } else {
    FatalError(E_TARGETDBVERSION,
	       "Attempt to read %s, expected %s\n",
	       TargetDBVersionString, TARGETDBVERSIONSTRING);
  }

  if (PROGRAM_ErrorNo == E_NULL) {
    if (!compressflag) {
      TargetDB = TargetDB_ReadUncompressed(fp);
    } else {
      fd_targetdb = fileno(fp);
      pos_targetdb = ftell(fp);
      (void) lseek(fd_targetdb, (off_t) pos_targetdb, SEEK_SET);
      /* open a pipe to uncompress the stuff */
      if (pipe(pipefd) < 0) {
	FatalError(E_PIPEFAILED,
		   "Could not open pipe to write target database : %s.\n",
		   strerror(errno));
      }
      /* fork and uncompress in child, read in parent */
      if (PROGRAM_ErrorNo == E_NULL) {
	pid = vfork();
	if (pid < 0) {
	  FatalError(E_VFORKFAILED,
		     "Could not vfork to uncompress target database: %s.\n",
		     strerror(errno));
	} else if (pid > 0) {
	  close(pipefd[1]);
	  pipeinfp = efdopen(pipefd[0], "r");
	  if (PROGRAM_ErrorNo == E_NULL) {
	    TargetDB = TargetDB_ReadUncompressed(pipeinfp);
	  }
	  fclose(pipeinfp);
	  close(pipefd[0]);
	} else {		/* pid == 0, child */
	  STRINGARRAY *ex = decompress_command();

	  if (dup2(pipefd[1], STDOUT_FILENO) < 0) {
	    FatalError(E_DUPFAILED, "dup2 failed in mapping stdout to pipe: %s\n",
		       strerror(errno));
	  }
	  if (dup2(fd_targetdb, STDIN_FILENO) < 0) {
	    FatalError(E_DUPFAILED, "dup2 failed in mapping stdout to pipe: %s\n",
		       strerror(errno));
	  }
	  close(pipefd[0]);
	  execv(STRINGARRAY_String(ex,0), STRINGARRAY_Values(ex));
	  FatalError(E_EXECFAILED, "Unable to exec %s to compress targetdb: %s\n",
		     STRINGARRAY_String(ex,0), strerror(errno));
	  _exit(-1);
	}
	wait(&ExecStatus);
	if (WEXITSTATUS(ExecStatus) != 0) {
	  FatalError(E_DECOMPRESSFAILED,
		     "Attempt to decompress target database failed.\n");
	}
      }
    }
  }
  return (PROGRAM_ErrorNo == E_NULL) ? TargetDB : NULL;
}

/* $Source: /afs/andrew.cmu.edu/system/src/local/depot2/018/src/lib/TargetDB/RCS/TargetDB_Read.c,v $ */
