#include "lamarc.h"

/***************************************************************************
 *  BETA                                                                   *
 *  version 1.50. (c) Copyright 1986, 1991, 1992 by the University of      *
 *  Washington and Joseph Felsenstein.  Written by Joseph Felsenstein,     *
 *  Mary K. Kuhner and Jon A. Yamato, with some additional grunt work by   *
 *  Sean T. Lamont.  Permission is granted to copy and use this program    *
 *   provided no fee is charged for it and provided that this copyright    *
 *  notice is not removed.                                                 *
 *                                                                         *
 ***************************************************************************/

/* This program implements a Hastings-Metropolis Monte Carlo
  Maximum Likelihood sampling method for phylogenetic trees
  without recombination. */

/* Time, 'tyme', in this tree is measured from the tips to the root.
   I.e. the tips are at tyme '0', and the root node has the largest
   value for 'tyme'. */

/* Only long chains included in joint estimates */

/* Modified to accept multiple loci. */

FILE *infile, *outfile, *treefile, *bestree, *seedfile, *simlog,
	    *parmfile, *thetafile, *intree;
long numsp, numnodes, sites, rootnum, categs, apps, inseed, col,
	    slidenum, numtrees, numsets, cycle, chaintype, totchains;
boolean  freqsfrom, ctgry, watt, printdata, usertree,
	       progress, treeprint, interleaved, ibmpc, ansi, auto_;
tree curtree;
double xi, xv, ttratio, locus_ttratio, freqa, freqc, freqg, freqt, freqr,
	      freqy, clearseed, freqar, freqcy, freqgr, freqty, fracchange,
	      lambda, probsum, theta0, branch0;
longer		seed;
sequence	y;
long		*numout;
long		*category,*weight;
double		*rate;
double		*probcat;
contribarr	*contribution;
node		**freenodes;
node		**slidenodes;
long            increm[2], steps[2], chains[2];
rlrec		*alogf;
char            gch;

/* the following are used in site aliasing (makesiteptr) */
long *siteptr;

/* the following are for reading in parameters (readparmfile) */
char *booltokens[NUMBOOL] = {"interleaved","printdata","progress",
                          "print-trees","freqs-from-data","categories",
                          "watterson", "usertree", "autocorrelation"},
     *numbertokens[NUMNUMBER] = {"ttratio","short-chains",
                          "short-steps","short-inc","long-chains",
                          "long-inc","long-steps"};

/* the following are for coal */
float		***sum;
double		**theti, **lntheti, *fixed, *locuslike, **savethetai;
long            numfix;

/* "Nearly global" variables for maketree: */

double	 	sumweightrat, oldlikelihood, watttheta;
double	 	*weightrat;
valrec   	*tbl;
boolean		accept_slide;
long		slid, slacc, indecks, chaintype;

void openfile(FILE **fp, char *filename, char *mode, char *application,
   char *perm)
{
  FILE *of;
  char file[100];

  strcpy(file,filename);
  while (1){
    of = fopen(file,mode);
    if (of)
      break;
    else {
      switch (*mode){
      case 'r':
        fprintf(stdout,"%s:  can't read %s\n",application,file);
        file[0] = '\0';
        while (file[0] =='\0'){
          fprintf(stdout,"Please enter a new filename>");
          gets(file);
          }
        break;
      case 'w':
        fprintf(stdout,"%s: can't write %s\n",application,file);
        file[0] = '\0';
        while (file[0] =='\0'){
          fprintf(stdout,"Please enter a new filename>");
          gets(file);
          }
        break;
      }
    }
  }
  *fp=of;
  if (perm != NULL)
    strcpy(perm,file);
} /* openfile */

double randum(long *seed)
/* random number generator -- slow but machine independent */
{
  long i, j, k, sum;
  longer mult, newseed;
  double x;

  mult[0] = 13;
  mult[1] = 24;
  mult[2] = 22;
  mult[3] = 6;
  for (i = 0; i <= 5; i++)
    newseed[i] = 0;
  for (i = 0; i <= 5; i++) {
    sum = newseed[i];
    k = i;
    if (i > 3)
      k = 3;
    for (j = 0; j <= k; j++)
      sum += mult[j] * seed[i - j];
    newseed[i] = sum;
    for (j = i; j <= 4; j++) {
      newseed[j + 1] += newseed[j] / 64;
      newseed[j] &= 63;
    }
  }
  memcpy(seed, newseed, sizeof(longer));
  seed[5] &= 3;
  x = 0.0;
  for (i = 0; i <= 5; i++)
    x = x / 64.0 + seed[i];
  x /= 4.0;
  return x;
}  /* randum */

void printtymelist()
/* prints the entire tymelist */
{
  long i;
  tlist *t;
  long limit;

  t = curtree.tymelist;
  fprintf(ERRFILE,"TYMELIST BEGINS\n");
  while (t != NULL) {
    fprintf(ERRFILE,"%3ld age%8.6f branches %3ld--",
           t->eventnode->number, t->age, t->numbranch);
    limit = t->numbranch;
    for (i = 0; i < limit; i++)
      fprintf(ERRFILE," %3ld", t->branchlist[i]->number);
    fprintf(ERRFILE,"\n");
    t = t->succ;
  }
  fprintf(ERRFILE,"TYMELIST ENDS\n");
}  /* printtymelist */

double lengthof(node *p)
/* returns the length of the branch "rootward" of the passed node */
{  /* lengthof */
  return fabs(p->tyme - p->back->tyme);
}  /* lengthof */

void findtop(node **p)
/* findtop returns the first 'top' it finds, for a given node the _same_
   'top' will always be found */
{
  while (!(*p)->top)
    *p = (*p)->next;
}  /* findtop */

void VarMalloc(node *p, boolean allokate)  
/* callocs or frees the 'x' [basewise Lnlikelihood] field of a node */
{
   long i;

   if (allokate) {
      if (p->x == NULL) {
         p->x = (phenotype)calloc(1,sites*sizeof(ratelike));
         p->x[0] = (ratelike)calloc(1,sites*categs*sizeof(sitelike));
         for(i = 1; i < sites; i++) 
            p->x[i] = p->x[0] + i * categs;
      }
   }
   else {
      if (p->x != NULL) {
         free(p->x[0]);
         free(p->x);
      }
      p->x = NULL;
   }
} /* VarMalloc */

/* "newnode" & "freenode" are paired memory managers for tree nodes.
   They maintain a linked list of unused nodes which should be faster
   to use than "calloc" & "free" (at least for recombination) */
void newnode(node **p)
{
  long i;

  i = 0;
  while (i < numnodes + 3) {   
    if (freenodes[i] != NULL) {
      *p = freenodes[i];
      freenodes[i] = NULL;
      return;
    }
    i++;
  }
  fprintf(ERRFILE,"newnode failed!\n");
  exit(-1);
}  /* newnode */

void freenode(node *p)
{
  long i;

  i = 0;
  while (i < numnodes + 3) {
     if (freenodes[i] == NULL) {
        freenodes[i] = p;
        return;
     }
     i++;
  }
  fprintf(ERRFILE,"freenode failed!\n");
  exit(-1);
}  /* freenode */
/* END of treenode allocation functions */

void newtymenode(tlist **t)
{
  *t = (tlist *)calloc(1,sizeof(tlist));
  (*t)->branchlist = (node **)calloc(1,numsp*sizeof(node *));
  /* WARNING recombination! */
}  /* newtymenode */

void freetymenode(tlist *t)
{
  free(t->branchlist);
  free(t);
}  /* freetymenode*/

void freetymelist(tlist *t)
{
   if (t->succ == NULL) freetymenode(t);
   else {
      t = t->succ;
      freetymelist(t);
   }
} /* freetymelist */

void hookup(node *p, node *q)
{
  p->back = q;
  q->back = p;
}  /* hookup */

void atr(node *p) 
/* "atr" prints out a text representation of a tree.  Pass 
   curtree.root->back for normal results */
{
  if (p->back == curtree.root) {
     fprintf(ERRFILE,"next node is root\n");
     fprintf(ERRFILE,"Node %4ld length %12.6f tyme %12.6f",
             p->back->number, lengthof(p), p->back->tyme);
     fprintf(ERRFILE," --> %4ld\n",p->number);
  }
  fprintf(ERRFILE,"Node %4ld length %12.6f tyme %12.6f -->",
         p->number, lengthof(p), p->tyme);
  if (p->top && p->back->top) fprintf(ERRFILE,"TWO TOPS HERE!!!!");
  if (!p->tip) {
     if (!p->next->top) fprintf(ERRFILE,"%4ld",p->next->back->number);
     if (!p->next->next->top) fprintf(ERRFILE,"%4ld",
         p->next->next->back->number);
     fprintf(ERRFILE,"\n");
     if (!p->next->top) atr(p->next->back);
     if (!p->next->next->top)
          atr(p->next->next->back);
  }
  else fprintf(ERRFILE,"\n");
} /* atr */

/* The next set of functions [zerocollis/onecollis/twocollis] compute
   the chance that there are 0/1/2 coalescences [respectively]
   in an interval of length "length", with "numl" active lineages, and
   "numother" inactive lineages */
double zerocollis(long numl, long numother, double length)
{

  return exp(-(numl * (numl - 1) + numl * numother * 2) * (length / theta0));
}  /* zerocollis */


double onecollis(long numl, long numother, double length)
{
  double expon1, expon2;

  expon1 = -((numl - 1) * numother * 2 + (numl - 1) * (numl - 2)) *
	   (length / theta0);
  expon2 = -(numl * numother * 2 + numl * (numl - 1)) * (length / theta0);

  return (numl * (numl - 1.0) / (numother * 2 + (numl - 1) * 2) *
	  (exp(expon1) - exp(expon2)));
}  /* onecollis */


double twocollis(long numother, double length)
/* For this case "numl" is assumed to be equal to 3 */
{
  double expon1, expon2, expon3;

  expon1 = numother * -2 * (length / theta0);
  expon2 = -(numother * 4 + 2.0) * (length / theta0);
  expon3 = -(numother * 6 + 6.0) * (length / theta0);

  return (6.0 / (numother + 1) *
	  (1.0 / (numother * 4 + 6) * (exp(expon1) - exp(expon3)) -
	   1.0 / (numother * 2 + 4) * (exp(expon2) - exp(expon3))));
}  /* twocollis */
/* End of coalescence functions */

void gettymenode(tlist **t, long target)
/* Return a pointer to the tymelist entry whose 'eventnode' has
   the number of 'target'. */
{
  boolean found;

  if (target == curtree.root->number) {
    *t = NULL;
    return;
  }
  *t = curtree.tymelist;
  if (curtree.nodep[target - 1]->tip)
    return;
  found = false;
  while (*t != NULL && !found) {
    if ((*t)->eventnode->number == target)
      found = true;
    else
      *t = (*t)->succ;
  }
  if (!found) {
    fprintf(ERRFILE,"In gettymenode, failed to find node%12ld\n", target);
    fprintf(ERRFILE,"CATASTROPHIC ERROR\n");
    exit(-1);
  }
}  /* gettymenode */

void gettyme(tlist **t, node *p, node *daughter1, node *daughter2, 
   node *ans)
/* Return a pointer to the tymelist entry which encompasses the time
   into which you wish to place node "p".
   tipwards/upper bound: "daughter1" and  "daughter2"
   rootward/lower bound: "ans" */
{
  boolean found, done;
  tlist *b1, *b2, *before, *after;

  /* first establish a tipward bound on the search */
  before = curtree.tymelist;
  found = false;
  done = false;
  gettymenode(&b1, daughter1->number);
  gettymenode(&b2, daughter2->number);
  while (!done) {
     if ((before == b1) || (before == b2))
        if ((found) || (b1 == b2))
           done = true;
        else
           found = true;
     if (!done) before = before->succ;
     }
  /* now establish a rootward bound on the search */
  gettymenode(&after, ans->number);
  /* begin the search at the tipward bound */
  *t = before;
  found = false;
  while (*t != after && !found) {
    if ((*t)->age >= p->tyme)
      found = true;
    else
      *t = (*t)->succ;
  }
  if (!found)
    *t = (*t)->prev;
  /* prime^.tyme is tied with after, so goes directly in front of it */
}  /* gettyme */

void inserttymelist(node *prime)
/* inserts 2 entries into the tymelist: 
   "prime" and "primans" [prime->back]. */
{
  tlist *t, *temp;
  node *parent, *q, *primans, *d3, *d[2];
  long i, j;

  newtymenode(&t);
  /* find daughters and parents */
  /* this complicated mess is needed because prime must be correctly
  bounded, not by primans (which is not yet in the tymelist), but by
  the parent of primans */
  q = prime;
  j = 1;
  for (i = 1; i <= 3; i++) {
    if (q->top) {
      primans = q->back;
      if (primans->next->top) {
	parent = primans->next->back;
	d3 = primans->next->next->back;
      } else {
	parent = primans->next->next->back;
	d3 = primans->next->back;
      }
    } else {
      d[j - 1] = q->back;
      j++;
    }
    q = q->next;
  }
  /* insert prime */
  t->eventnode = prime;
  gettyme(&temp, prime, d[0], d[1], parent);
  t->succ = temp->succ;
  t->prev = temp;
  if (temp->succ != NULL)
    temp->succ->prev = t;
  temp->succ = t;
  if (t->succ != NULL)
    t->age = t->succ->eventnode->tyme;
  else
    t->age = t->prev->age;
  t->prev->age = t->eventnode->tyme;
  /* insert primans */
  newtymenode(&t);
  t->eventnode = primans;
  gettyme(&temp, primans, prime, d3, parent);
  t->succ = temp->succ;
  t->prev = temp;
  if (temp->succ != NULL)
    temp->succ->prev = t;
  temp->succ = t;
  if (t->succ != NULL)
    t->age = t->succ->eventnode->tyme;
  else
    t->age = t->prev->age;
  t->prev->age = t->eventnode->tyme;
}  /* inserttymelist */

void subtymelist(node *ndonor, node *nrecip)
/* takes out 2 entries from the tymelist:
   "ndonor" and "nrecip" [which must be tipward/above ndonor] */
{
  long i, j, limit;
  tlist *d, *r, *t;
  node *badbranch, *p;
  boolean found;

  i = 0;
  
  gettymenode(&r, nrecip->number);
  gettymenode(&d, ndonor->number);
  p = nrecip;
  for (j = 1; j <= 3; j++) {
    p = p->next;
    if (p->back->number == ndonor->number)
      badbranch = p;
  }

  t = r;

  while (t != d) {
    limit = t->numbranch;
    for (i = 1; i <= limit; i++) {
      if (t->branchlist[i - 1] == badbranch) {
	j = i;
	t->numbranch--;
      }
    }
    for (i = j; i <= t->numbranch; i++)
      t->branchlist[i - 1] = t->branchlist[i];
    t = t->succ;
  }
  p = ndonor;
  findtop(&p);
  badbranch = p;
  found = true;
  while (t != NULL && found) {
    found = false;
    for (i = 1; i <= t->numbranch; i++) {
      if (t->branchlist[i - 1] == badbranch) {
	j = i;
	t->numbranch--;
	found = true;
      }
    }
    if (found) {
      for (i = j; i <= t->numbranch; i++)
	t->branchlist[i - 1] = t->branchlist[i];
    }
    t = t->succ;
  }
  r->prev->succ = r->succ;
  r->succ->prev = r->prev;
  r->prev->age = r->age;
  freetymenode(r);
  d->prev->succ = d->succ;
  if (d->succ != NULL)
    d->succ->prev = d->prev;
  d->prev->age = d->age;
  freetymenode(d);
}  /* subtymelist */

void ltov(node *p)
/* ltov recalculates the proper "v" value of a branch, from
   the tymes at either end of the branch */
{
  p->v = 1.0 - exp(-(lengthof(p) / fracchange));
  p->back->v = p->v;
}  /* ltov */

void getnums()
{
  /* input number of sequences, number of sites */
  fprintf(outfile, "\n");
  fscanf(infile, "%ld%ld", &numsp, &sites);
    fprintf(outfile, "%4ld Sequences, %4ld Sites\n", numsp, sites);
  numnodes = numsp * 2 - 1;   /*number of nodes in tree, excluding root*/
  rootnum = numnodes + 3;
  freenodes = (node **)calloc(1,2*sizeof(node *));
  /* number of internal nodes in tree is numsp-1 */
  slidenodes = (node **)calloc(1,(numsp-1)*sizeof(node *));
}  /* getnums */

/* boolcheck(), booleancheck(), numbercheck(), and readparmfile() 
   are used in reading the parameter file "parmfile" */
long boolcheck(char ch)
{
   ch = toupper(ch);
   if (ch == 'F') return 0;
   if (ch == 'T') return 1;
   return -1;
} /* boolcheck */

boolean booleancheck(char *var, char *value)
{
   long i, j, check;

   check = boolcheck(value[0]);
   if(check == -1) return false;

   for(i = 0; i < NUMBOOL; i++) {
      if(!strcmp(var,booltokens[i])) {
         if(i == 0) interleaved = (boolean)(check);
         if(i == 1) printdata = (boolean)(check);
         if(i == 2) progress = (boolean)(check);
         if(i == 3) treeprint = (boolean)(check);
         if(i == 4) {
            freqsfrom = (boolean)(check);
            if(!freqsfrom) {
               strtok(value,":");
               freqa = (double)atof(strtok(NULL,";"));
               freqc = (double)atof(strtok(NULL,";"));
               freqg = (double)atof(strtok(NULL,";"));
               freqt = (double)atof(strtok(NULL,";"));
            }
         }
         if(i == 5) {
            ctgry = (boolean)(check);
            if(ctgry) {
               strtok(value,":");
               categs = (long)atof(strtok(NULL,";"));
               rate = (double *)realloc(rate,categs*sizeof(double));
               probcat = (double *)realloc(probcat,categs*sizeof(double));
               for(j = 0; j < categs; j++) {
                  rate[j] = (double)atof(strtok(NULL,";"));
                  probcat[j] = (double)atof(strtok(NULL,";"));
               }
            }
         }
         if(i == 6) {
            watt = (boolean)(check);
            if (!watt) {
               strtok(value,":");
               theta0 = (double)atof(strtok(NULL,";"));
            }
         }
         if(i == 7) usertree = (boolean)(check);
         if(i == 8) {
            auto_ = (boolean)(check);
            if (auto_) {
               strtok(value,":");
               lambda = 1.0 / (double)atof(strtok(NULL,";"));
            }
         }
         return true;
      }
   }
   return false;
} /* booleancheck */

boolean numbercheck(char *var, char *value)
{
   long i;

   for(i = 0; i < NUMNUMBER; i++) {
      if(!strcmp(var,numbertokens[i])) {
         if(i == 0) locus_ttratio = atof(value);
         if(i == 1) chains[0] = atof(value);
         if(i == 2) steps[0] = atof(value);
         if(i == 3) increm[0] = atof(value);
         if(i == 4) chains[1] = atof(value);
         if(i == 5) increm[1] = atof(value);
         if(i == 6) steps[1] = atof(value);
         return true;
      }
   }
   return false;
} /* numbercheck */

void readparmfile()
{
   char fileline[LINESIZE],parmvar[LINESIZE],varvalue[LINESIZE];

   parmfile = fopen("parmfile","r");

   if(parmfile) {
      while(fgets(fileline, LINESIZE, parmfile) != NULL) {
         if(fileline[0] == '#') continue;
         if(!strncmp(fileline,"end",3)) break;
         strcpy(parmvar,strtok(fileline,"="));
         strcpy(varvalue,strtok(NULL,"\n"));
         /* now to process... */
         if(!booleancheck(parmvar,varvalue))
            if(!numbercheck(parmvar,varvalue)) {
               fprintf(ERRFILE,
                  "Inappropiate entry in parmfile: %s\n", fileline);
               exit(-1);
            }
      }
   } else
      if (!menu) {
         fprintf(simlog,"Parameter file (parmfile) missing\n");
         exit(-1);
      }
} /* readparmfile */
/* END parameter file read */

void getoptions()
/* interactively set options using a very basic menu */
{
  boolean done, done1, done2;
  char ch;
  long i, j;
  char input[LINESIZE];

  rate    = (double *)calloc(1,sizeof(double));
  probcat = (double *)calloc(1,sizeof(double));

  /* first some multiple rate-categories code stuff */
  ctgry = false;
  rate[0] = 1.0;
  probcat[0] = 1.0;
  categs = 1;
  lambda = 1.0;
  auto_ = false;  /* false if categs == 1 */
  /* end categories code stuff */

  /* default initializations */
  interleaved = false;
  printdata = false;
  progress = true;
  treeprint = false;
  locus_ttratio = 2.0;
  freqsfrom = true;
  watt = false;
  usertree = true;
  theta0 = 1.0;
  chains[0] = 10;
  increm[0] = 20;
  steps[0] = 200;
  chains[1] = 2;
  increm[1] = 20;
  steps[1] = 20000;
  /* end defaults */

  readparmfile();
  fscanf(infile,"%ld",&numsets);
  fprintf(outfile, "\nHastings-Metropolis Monte Carlo ML");
  fprintf(outfile, " method, version 1.3\n\n");
  if (!menu) {
    fscanf(seedfile, "%ld%*[^\n]", &inseed);
    getc(seedfile);
  } else {
    seedfile = fopen("seedfile","r");
    if (seedfile) {
      fscanf(seedfile, "%ld%*[^\n]", &inseed);
      getc(seedfile);
    }
    else {
      printf("Random number seed (must be odd)?\n");
      scanf("%ld%*[^\n]", &inseed);
      getchar();
    }
  }
  for (i = 0; i <= 5; i++)
    seed[i] = 0;
  i = 0;
  do {
    seed[i] = inseed & 63;
    inseed /= 64;
    i++;
  } while (inseed != 0);
  if (!menu)
    return;
  putchar('\n');
  do {
    printf("\n%s", ansi ? "\033[2J\033[H" : "\n");
    printf("Hastings-Metropolis Markov Chain Monte Carlo");
    printf(" method, version 1.3\n\n");
    printf("INPUT/OUTPUT FORMATS\n");
    printf("  I          Input sequences interleaved?  %s\n",
           interleaved ? "Yes" : "No, sequential");
    printf("  E        Echo the data at start of run?  %s\n",
           printdata ? "Yes" : "No");
    printf("  P Print indications of progress of run?  %s\n",
           progress ? "Yes" : "No");
    printf("  G                Print out genealogies?  %s\n",
           treeprint ? "Yes" : "No");
    printf("MODEL PARAMETERS\n");
    printf("  T        Transition/transversion ratio:");
    printf("  %8.4f\n",locus_ttratio);
    printf("  F       Use empirical base frequencies?  %s\n",
	   freqsfrom ? "Yes" : "No");
    printf("  C   One category of substitution rates?");
    if (!ctgry || categs == 1)
      printf("  Yes\n");
    else {
      printf("  %ld categories\n", categs);
      printf("  R   Rates at adjacent sites correlated?");
      if (!auto_)
	printf("  No, they are independent\n");
      else
	printf("  Yes, mean block length =%6.1f\n", 1.0 / lambda);
    }
    printf("  W      Use Watterson estimate of theta?");
    if (watt)
      printf("  Yes\n");
    else
      printf("  No, initial theta = %6.4f\n", theta0);
    printf("  U      Use user tree in file \"intree\" ?  %s\n",
           usertree ? "Yes" : "No");
    printf("SEARCH STRATEGY\n");
    printf("  S        Number of short chains to run?  %6ld\n", chains[0]);
    if (chains[0] > 0) {
       printf("  1             Short sampling increment?  %6ld\n",
	   increm[0]);
       printf("  2   Number of steps along short chains?  %6ld\n",
           steps[0]);
    }
    printf("  L         Number of long chains to run?  %6ld\n", chains[1]);
    if (chains[1] > 0) {
       printf("  3              Long sampling increment?  %6ld\n",
	   increm[1]);
       printf("  4    Number of steps along long chains?  %6ld\n",
           steps[1]);
    }
    printf("\n");
    printf("Are these settings correct? (type Y or the letter for one to change)\n");
    gets(input);
    ch = input[0];
    ch = toupper(ch);
    done = (ch == 'Y');
    if (!done) {
      ch = toupper(ch);
      if (strchr("FGTIE1234CWUSLPR",ch) != NULL){
	switch (ch) {

	case 'S':
	  do {
	    printf("How many Short Chains?\n");
            gets(input);
            chains[0] = atoi(input);
	    if (chains[0] < 0)
	      printf("Must be non-negative\n");
	  } while (chains[0] < 0);
	  break;

	case 'L':
	  do {
	    printf("How many Long Chains?\n");
            gets(input);
            chains[1] = atoi(input);
	    if (chains[1] < 0)
	      printf("Must be non-negative\n");
	  } while (chains[1] < 0);
	  break;

	case 'C':
	  ctgry = !ctgry;
	  if (!ctgry)
	    auto_ = false;
	  if (ctgry) {
	    do {
	      printf("Number of categories ?");
              gets(input);
              categs = atoi(input);
	    } while (categs < 1);
	    free(rate);
	    free(probcat);
	    printf("Rate for each category? (use a space to");
	    printf(" separate)\n");
	    rate    = (double *)calloc(1,categs*sizeof(double));
	    probcat = (double *)calloc(1,categs*sizeof(double));
	    for (j = 0; j < categs; j++)
	      scanf("%lf*[^\n]", &rate[j]);

	    getchar();
	    do {
	      printf("Probability for each category?");
	      printf(" (use a space to separate)\n");
	      for (j = 0; j < categs; j++)
		scanf("%lf", &probcat[j]);
	      scanf("%*[^\n]");
	      getchar();
	      done2 = true;
	      probsum = 0.0;
	      for (j = 0; j < categs; j++)
		probsum += probcat[j];
	      if (fabs(1.0 - probsum) > 0.001) {
		done2 = false;
		printf("Probabilities must add up to");
		printf(" 1.0, plus or minus 0.001.\n");
	      }
	    } while (!done2);
	  }
	  break;

	case 'R':
	  auto_ = !auto_;
	  if (auto_) {
	    do {
	      printf("Mean block length of sites having the same ");
	      printf("rate (greater than 1)?\n");
	      scanf("%lf%*[^\n]", &lambda);
	      getchar();
	    } while (lambda <= 1.0);
	    lambda = 1.0 / lambda;
	  }
	  break;

	case 'F':
	  freqsfrom = !freqsfrom;
	  if (!freqsfrom) {
	    printf("Base frequencies for A, C, G, T/U (use blanks to separate)?\n");
	    scanf("%lf%lf%lf%lf%", &freqa, &freqc, &freqg, &freqt);
            scanf("%*[^\n]");
	  }
	  break;

	case 'T':
	  do {
	    printf("Transition/transversion ratio?\n");
            gets(input);
            locus_ttratio = atof(input);
	  } while (locus_ttratio < 0.0);
	  break;

	case 'I':
	  interleaved = !interleaved;
	  break;

	case 'W':
	  watt = !watt;
	  if (!watt) {
	    do {
	      printf("Initial theta estimate?\n");
              gets(input);
              theta0 = atof(input);
	    } while (theta0 <= 0.0);
	  }
	  break;

        case 'U':
          usertree = !usertree;
          break;

	case 'E':
	  printdata = !printdata;
	  break;

	case 'P':
	  progress = !progress;
	  break;

	case 'G':
	  treeprint = !treeprint;
	  break;

	case '1':
	  done1 = false;
	  while (!done1) {
	    printf("How often to sample trees?\n");
            gets(input);
            increm[0] = atoi(input);
	    if (increm[0] > 0)
	      done1 = true;
	    else
	      printf("Must be positive\n");
	  }
	  break;

	case '2':
	  done1 = false;
	  while (!done1) {
	    printf("How many short steps?\n");
            gets(input);
            steps[0] = atoi(input);
	    if (steps[0] > 0)
	      done1 = true;
	    else
	      printf("Must be a positive integer\n");
	  }
	  break;

	case '3':
	  done1 = false;
	  while (!done1) {
	    printf("How often to sample trees?\n");
            gets(input);
            increm[1] = atoi(input);
	    if (increm[1] > 0)
	      done1 = true;
	    else
	      printf("Must be positive\n");
	  }
	  break;

	case '4':
	  done1 = false;
	  while (!done1) {
	    printf("How many long steps?\n");
            gets(input);
            steps[1] = atoi(input);
	    if (steps[1] > 0)
	      done1 = true;
	    else
	      printf("Must be a positive integer\n");
	  }
	  break;

       default:
          fprintf(stderr,"Impossible option %c detected!\n",ch);
          break;     

       }
      } else
	printf("Not a possible option!\n");
    }
  } while (!done);
}  /* getoptions */

void coalinit()
/* coalinit calculates the "fixed" values to be used in computing
   theta estimates */
{
  double b, b1, bnum, bdenom;

  b = MINTHETA;
  b1 = exp(log(10.0) * (long)floor(log(b) / log(10.0) + 0.5));
  bdenom = (long)floor(1 / b1 + 0.5);
  bnum = (long)floor(b / b1 + 0.5);
  fixed = (double *)calloc(1,sizeof(double));
  numfix = 0;
  do {
    numfix++;
    fixed = (double *)realloc(fixed,numfix*sizeof(double));
    fixed[numfix - 1] = bnum/bdenom;
    if (fabs(bnum - 2.0) < epsilon)
      bnum = (long)floor(2.5 * bnum + 0.5);
    else
      bnum *= 2.0;
    if (fabs(bnum - 10.0) < epsilon) {
      bnum = 1.0;
      bdenom /= 10.0;
    }
  } while (bnum / bdenom <= MAXTHETA * (1.0 + epsilon));
}  /* coalinit */

void firstinit()
/* initialization for things that are recorded over multiple loci */
{
  long i, j;

  numtrees = MAX(steps[0]/increm[0],steps[1]/increm[1]);
  totchains = chains[0] + chains[1];
  numout = (long *)calloc(1,2 * sizeof(long));

  sum = (float ***)calloc(1,numsets * sizeof(float **));
  sum[0] = (float **)calloc(1,numsets*(1+chains[1]) * sizeof(float*));
  for (i = 1; i < numsets; i++)
    sum[i] = sum[0] + i*(1+chains[1]);
  sum[0][0] = (float *)calloc(1,numsets*(1+chains[1])*numtrees * sizeof(float));
  for (i = 0; i < numsets; i++)
     for(j = 0; j < (1+chains[1]); j++)
        sum[i][j] = sum[0][0] + i*(1+chains[1])*numtrees + j*numtrees;
  coalinit();
  theti = (double **)calloc(1,numsets * sizeof(double *));
  lntheti = (double **)calloc(1,numsets * sizeof(double *));
  savethetai = (double **)calloc(1,numsets * sizeof(double *));
  lntheti[0] = (double *)
     calloc(1,numsets*(totchains+numfix+1) * sizeof(double));
  theti[0] = (double *)
     calloc(1,numsets*(totchains+numfix+1) * sizeof(double));
  savethetai[0] = (double *)calloc(1,(numsets*totchains) * sizeof(double));
  for (i = 1; i < numsets; i++) {
     theti[i] = theti[0] + i*(totchains+numfix+1);
     lntheti[i] = lntheti[0] + i*(totchains+numfix+1);
     savethetai[i] = savethetai[0] + i*totchains;
  }
  locuslike = (double *)calloc(1,numsets+numfix * sizeof(double)); 
}  /* firstinit */

void locusinit()
/* initialization of things that are specific to one locus */ 
{
  getnums();
  if ((increm[0] < 0) || (increm[1] < 0)) {
     fprintf(ERRFILE,"Error in input sampling increment");
     fprintf(ERRFILE," increment set to 10\n");
     if (increm[0] < 0)
        increm[0] = 10;
     if (increm[1] < 0)
        increm[1] = 10;
  }
} /* locusinit */

void inputoptions()
{
  char ch;
  long i, extranum;

  category = (long *)calloc(1,sites*sizeof(long));
  weight   = (long *)calloc(1,sites*sizeof(long));

  for (i = 0; i < sites; i++)
    category[i] = 1,
    weight[i] = 1;
  extranum = 0;
  while (!(eoln(infile))) {
    ch = getc(infile);
    if (ch == '\n')
      ch = ' ';
    ch = isupper(ch) ? ch : toupper(ch);
    if (ch == 'C')
      extranum++;
    else if (ch != ' ') {
      printf("BAD OPTION CHARACTER: %c\n", ch);
      exit(-1);
    }
  }
  fscanf(infile, "%*[^\n]");
  getc(infile);
  for (i = 1; i <= extranum; i++) {
    ch = getc(infile);
    if (ch == '\n')
      ch = ' ';
    ch = isupper(ch) ? ch : toupper(ch);
    if (ch != 'W'){
      printf("ERROR: INCORRECT AUXILIARY OPTIONS LINE WHICH STARTS WITH %c\n",
	     ch);
      exit(-1);
    }
  }
  if (categs <= 1)
    return;
  fprintf(outfile, "\nSite category   Rate of change  Probability\n");
  for (i = 1; i <= categs; i++)
    fprintf(outfile, "%12ld%13.3f%13.3f\n", i, rate[i - 1], probcat[i - 1]);
  putc('\n', outfile);
}  /* inputoptions */

void getbasefreqs()
{
  double aa, bb;

  putc('\n', outfile);
  if (freqsfrom)
    fprintf(outfile, "Empirical ");
  fprintf(outfile, "Base Frequencies:\n\n");
  fprintf(outfile, "   A    %10.5f\n", freqa);
  fprintf(outfile, "   C    %10.5f\n", freqc);
  fprintf(outfile, "   G    %10.5f\n", freqg);
  fprintf(outfile, "  T(U)  %10.5f\n", freqt);
  freqr = freqa + freqg;
  freqy = freqc + freqt;
  freqar = freqa / freqr;
  freqcy = freqc / freqy;
  freqgr = freqg / freqr;
  freqty = freqt / freqy;
  fprintf(outfile, "Transition/transversion ratio = %10.6f\n", locus_ttratio);
  aa = locus_ttratio * freqr * freqy - freqa * freqg - freqc * freqt;
  bb = freqa * freqgr + freqc * freqty;
  xi = aa / (aa + bb);
  xv = 1.0 - xi;
  ttratio = xi / xv;
  if (xi <= 0.0) {
    printf("WARNING: This transition/transversion ratio\n");
    printf("is impossible with these base frequencies!\n");
    xi = 3.0 / 5;
    xv = 2.0 / 5;
    fprintf(outfile, " Transition/transversion parameter reset\n\n");
  }
  fprintf(outfile, "(Transition/transversion parameter = %10.6f)\n",
	  xi / xv);
  fracchange = xi * (2 * freqa * freqgr + 2 * freqc * freqty) +
      xv * (1.0 - freqa * freqa - freqc * freqc - freqg * freqg - freqt * freqt);
}  /* getbasefreqs */

void setuptree()
{
  long i, j;
  node *p, *q;

  curtree.nodep = (node **)calloc(1,(numnodes + 3)*sizeof(node *));
  alogf = (rlrec *)calloc(1,sizeof(rlrec));
  alogf->val = (double *)calloc(1,sites*sizeof(double));
  newtymenode(&curtree.tymelist);
  for (i = 0; i < 2; i++)
    freenodes[i] = NULL;

  /* make the tips first */
  for (i = 0; i < numsp; i++) {
    curtree.nodep[i] = (node *)calloc(1,sizeof(node));
    curtree.nodep[i]->tip = true;
    curtree.nodep[i]->number = i + 1;
    VarMalloc(curtree.nodep[i],true);
  }

  /* now make the interior nodes and the freenodes, but not the root */
  for (i = numsp; i < numnodes+2; i++) {
    q = NULL;
    for (j = 0; j < 3; j++) {
      p = (node *)calloc(1,sizeof(node));
      if (p == NULL) {
        fprintf(ERRFILE,"tree setup fails, allocate more space\n");
        exit(-1);
      }
      p->number = i + 1;
      p->tip = false;
     /* initialize the following pointers to NULL
        space will be allocated as appropiate in procedure
        orient() */
      p->x = NULL;
     /* end NULL assignments */
      p->next = q;
      q = p;
    }
    p->next->next->next = p; /* close up the chain into a loop */
    curtree.nodep[i] = p;
    if (i >= numnodes)
      freenodes[i - numnodes] = p;
      /* do memory allocation for initial freenodes now, orient
         only covers nodes in initial tree */
      p->top = true;
      VarMalloc(p,true);
  }

  /* now make the root */
  curtree.nodep[rootnum - 1] = (node *)calloc(1,sizeof(node));
  curtree.nodep[rootnum - 1]->tip = true;
  curtree.nodep[rootnum - 1]->number = rootnum;
  VarMalloc(curtree.nodep[rootnum - 1],true);
  strncpy(curtree.nodep[rootnum-1]->nayme,"ROOT",4);
  /* guarantee that the root node contributes nothing to the likelihood
     of a tree (since its supposed to be at the end of a theoretically
     infinite root branch) */
  for (i = 0; i < sites; i++) {
    for (j = 0; j < categs; j++) {
      curtree.nodep[rootnum - 1]->x[i][j][0] = 1.0;
      curtree.nodep[rootnum - 1]->x[i][j][(long)C - (long)A] = 1.0;
      curtree.nodep[rootnum - 1]->x[i][j][(long)G - (long)A] = 1.0;
      curtree.nodep[rootnum - 1]->x[i][j][(long)T - (long)A] = 1.0;
    }
  }
  curtree.likelihood = NEGMAX;
}  /* setuptree */

void freetree()
/* we do not free the following arrays:
      sum, theti, lntheti, fixed, numout  */
{
   long i;
   node *p;

   free(alogf->val);
   free(alogf);
   freetymelist(curtree.tymelist);
   /* free the tips */
   for(i = 0; i < numsp; i++) {
      VarMalloc(curtree.nodep[i],false);
      free(curtree.nodep[i]);
   }
   /* free internal nodes including slidenodes */
   for(i = numsp; i < numnodes + 2; i++) {
      p = curtree.nodep[i];
      VarMalloc(p,false);
      VarMalloc(p->next,false);
      VarMalloc(p->next->next,false);
      free(p->next->next);
      free(p->next);
      free(p);
   }
   free(slidenodes);
   free(freenodes);
   /* free the root node */
   VarMalloc(curtree.nodep[rootnum-1],false);
   free(curtree.nodep[rootnum-1]);
   /* free the tree */
   free(curtree.nodep);
   /* free the sequences */
   free(y[0]);
   free(y);
   free(siteptr);
   /* free the working arrays */
   free(weightrat);
   free(tbl);
   free(contribution[0]);
   free(contribution);
} /* freetree */

void getdata()
{
  /* read sequences */
  long i, j, k, l, basesread, basesnew;
  char ch;
  boolean allread, done;

  y = (char **)calloc(1,numsp*sizeof(char *)); /* alloc memory for sequence. */
  y[0] = (char *)calloc(1,numsp*sites*sizeof(char));
  for (i=1;i<numsp;i++)
     y[i] = y[0] + i*sites;
  
  setuptree();

  putc('\n', outfile);
  j = nmlngth + (sites + (sites - 1) / 10) / 2 - 5;
  if (j < nmlngth - 1)
    j = nmlngth - 1;
  if (j > 37)
    j = 37;
  if (printdata) {
    fprintf(outfile, "Name");
    for (i = 1; i <= j; i++)
      putc(' ', outfile);
    fprintf(outfile, "Sequences\n");
    fprintf(outfile, "----");
    for (i = 1; i <= j; i++)
      putc(' ', outfile);
    fprintf(outfile, "---------\n\n");
  }
  basesread = 0;
  allread = false;
  while (!(allread)) {
    allread = true;
    if (eoln(infile)) {
      fscanf(infile, "%*[^\n]");
      getc(infile);
    }
    i = 1;
    while (i <= numsp) {
      if (interleaved && basesread == 0 || !interleaved) {
	for (j = 0; j < nmlngth; j++) {
	  curtree.nodep[i - 1]->nayme[j] = getc(infile);
	  if (curtree.nodep[i - 1]->nayme[j] == '\n')
	    curtree.nodep[i - 1]->nayme[j] = ' ';
	  if (eof(infile) || eoln(infile)){
	    printf("ERROR: END-OF-LINE OR END-OF-FILE IN THE MIDDLE OF A SPECIES NAME\n");
	    exit(-1);
	  }
	}
      }
      if (interleaved)
	j = basesread;
      else
	j = 0;
      done = false;
      while (((!done) & (!(eoln(infile) | eof(infile))))) {
	if (interleaved)
	  done = true;
	while (((j < sites) & (!(eoln(infile) | eof(infile))))) {
	  ch = getc(infile);
	  if (ch == '\n')
	    ch = ' ';
	  if (ch == ' ' || ch >= '0' && ch <= '9')
	    continue;
	  ch = isupper(ch) ? ch : toupper(ch);
	  if (!(int)strchr("ABCDGHKMNRSTUVWXY?O-.",ch)){
	    printf("ERROR: BAD BASE:%c AT POSITION%5ld OF SPECIES %3ld\n",
		   ch, j, i);
	    exit(-1);
	  }
	  j++;
	  if (ch == '.')
	    ch = y[0][j - 1];
	  y[i - 1][j - 1] = ch;
	}
	if (interleaved)
	  continue;
	if (j < sites) {
	  fscanf(infile, "%*[^\n]");
	  getc(infile);
	} else if (j == sites)
	  done = true;
      }
      if (interleaved && i == 1)
	basesnew = j;
      fscanf(infile, "%*[^\n]");
      getc(infile);
      if ((interleaved && j != basesnew) || (!interleaved && j != sites)){
	printf("ERROR: SEQUENCES OUT OF ALIGNMENT\n");
	exit(-1);}
      i++;
    }
    if (interleaved) {
      basesread = basesnew;
      allread = (basesread == sites);
    } else
      allread = (i > numsp);
  }
  if (!printdata)
    return;
  for (i = 1; i <= ((sites - 1) / 60 + 1); i++) {
    for (j = 1; j <= numsp; j++) {
      for (k = 0; k < nmlngth; k++)
	putc(curtree.nodep[j - 1]->nayme[k], outfile);
      fprintf(outfile, "   ");
      l = i * 60;
      if (l > sites)
	l = sites;
      for (k = (i - 1) * 60 + 1; k <= l; k++) {
	if (j > 1 && y[j - 1][k - 1] == y[0][k - 1])
	  ch = '.';
	else
	  ch = y[j - 1][k - 1];
	putc(ch, outfile);
	if (k % 10 == 0 && k % 60 != 0)
	  putc(' ', outfile);
      }
      putc('\n', outfile);
    }
    putc('\n', outfile);
  }
}  /* getdata */

void makevalues()
{
  /* set up fractional likelihoods at tips */
  long i, k, l;
  base b;

  for (k = 0; k < sites; k++) {
    for (i = 0; i < numsp; i++) {
      for (l = 0; l < categs; l++) {
	for (b = A; (long)b <= (long)T; b = (base)((long)b + 1))
	  curtree.nodep[i]->x[k][l][(long)b - (long)A] = 0.0;
	switch (y[i][k]) {

	case 'A':
	  curtree.nodep[i]->x[k][l][0] = 1.0;
	  break;

	case 'C':
	  curtree.nodep[i]->x[k][l][(long)C - (long)A] = 1.0;
	  break;

	case 'G':
	  curtree.nodep[i]->x[k][l][(long)G - (long)A] = 1.0;
	  break;

	case 'T':
	  curtree.nodep[i]->x[k][l][(long)T - (long)A] = 1.0;
	  break;

	case 'U':
	  curtree.nodep[i]->x[k][l][(long)T - (long)A] = 1.0;
	  break;

	case 'M':
	  curtree.nodep[i]->x[k][l][0] = 1.0;
	  curtree.nodep[i]->x[k][l][(long)C - (long)A] = 1.0;
	  break;

	case 'R':
	  curtree.nodep[i]->x[k][l][0] = 1.0;
	  curtree.nodep[i]->x[k][l][(long)G - (long)A] = 1.0;
	  break;

	case 'W':
	  curtree.nodep[i]->x[k][l][0] = 1.0;
	  curtree.nodep[i]->x[k][l][(long)T - (long)A] = 1.0;
	  break;

	case 'S':
	  curtree.nodep[i]->x[k][l][(long)C - (long)A] = 1.0;
	  curtree.nodep[i]->x[k][l][(long)G - (long)A] = 1.0;
	  break;

	case 'Y':
	  curtree.nodep[i]->x[k][l][(long)C - (long)A] = 1.0;
	  curtree.nodep[i]->x[k][l][(long)T - (long)A] = 1.0;
	  break;

	case 'K':
	  curtree.nodep[i]->x[k][l][(long)G - (long)A] = 1.0;
	  curtree.nodep[i]->x[k][l][(long)T - (long)A] = 1.0;
	  break;

	case 'B':
	  curtree.nodep[i]->x[k][l][(long)C - (long)A] = 1.0;
	  curtree.nodep[i]->x[k][l][(long)G - (long)A] = 1.0;
	  curtree.nodep[i]->x[k][l][(long)T - (long)A] = 1.0;
	  break;

	case 'D':
	  curtree.nodep[i]->x[k][l][0] = 1.0;
	  curtree.nodep[i]->x[k][l][(long)G - (long)A] = 1.0;
	  curtree.nodep[i]->x[k][l][(long)T - (long)A] = 1.0;
	  break;

	case 'H':
	  curtree.nodep[i]->x[k][l][0] = 1.0;
	  curtree.nodep[i]->x[k][l][(long)C - (long)A] = 1.0;
	  curtree.nodep[i]->x[k][l][(long)T - (long)A] = 1.0;
	  break;

	case 'V':
	  curtree.nodep[i]->x[k][l][0] = 1.0;
	  curtree.nodep[i]->x[k][l][(long)C - (long)A] = 1.0;
	  curtree.nodep[i]->x[k][l][(long)G - (long)A] = 1.0;
	  break;

	case 'N':
	  for (b = A; (long)b <= (long)T; b = (base)((long)b + 1))
	    curtree.nodep[i]->x[k][l][(long)b - (long)A] = 1.0;
	  break;

	case 'X':
	  for (b = A; (long)b <= (long)T; b = (base)((long)b + 1))
	    curtree.nodep[i]->x[k][l][(long)b - (long)A] = 1.0;
	  break;

	case '?':
	  for (b = A; (long)b <= (long)T; b = (base)((long)b + 1))
	    curtree.nodep[i]->x[k][l][(long)b - (long)A] = 1.0;
	  break;

	case 'O':
	  for (b = A; (long)b <= (long)T; b = (base)((long)b + 1))
	    curtree.nodep[i]->x[k][l][(long)b - (long)A] = 1.0;
	  break;

	case '-':
	  for (b = A; (long)b <= (long)T; b = (base)((long)b + 1))
	    curtree.nodep[i]->x[k][l][(long)b - (long)A] = 1.0;
	  break;
	}
      }
    }
  }
}  /* makevalues */

boolean sitecompare(long site1, long site2)
{
   long i;

   for(i = 0; i < numsp; i++)
      if(y[i][site1] != y[i][site2])
         return false;

   return true;
} /* sitecompare */

void makesiteptr()
/* create the siteptr array: -1 means do a likelihood calculation
   for this site; a number >= 0 means use the site with that number
   for x (likelihood) values */
{
   long whichsite, i;
   boolean found;

   siteptr = (long *)calloc(1,sites*sizeof(long));

   siteptr[0] = -1;

   for(whichsite = 1; whichsite < sites; whichsite++) {
      found = false;
      for(i = whichsite - 1; i >= 0; i--) {
         if(sitecompare(i,whichsite)) {
            siteptr[whichsite] = i;
            found = true;
            break;
         }
      }
      if (found) continue;
      siteptr[whichsite] = -1;
   }
} /* makesiteptr */

void empiricalfreqs()
{
  /* Get empirical base frequencies from the data */
  long i, j, k;
  double sum, suma, sumc, sumg, sumt, w;
 
  freqa = 0.25;
  freqc = 0.25;
  freqg = 0.25;
  freqt = 0.25;
  for (k = 1; k <= 8; k++) {
    suma = 0.0;
    sumc = 0.0;
    sumg = 0.0;
    sumt = 0.0;
     for (i = 0; i < numsp; i++) {
      for (j = 0; j < sites; j++) {
	w = weight[j];
	sum = freqa * curtree.nodep[i]->x[j][0][0];
	sum += freqc * curtree.nodep[i]->x[j][0][(long)C - (long)A];
	sum += freqg * curtree.nodep[i]->x[j][0][(long)G - (long)A];
	sum += freqt * curtree.nodep[i]->x[j][0][(long)T - (long)A];
	suma += w * freqa * curtree.nodep[i]->x[j][0][0] / sum;
	sumc += w * freqc * curtree.nodep[i]->x[j][0][(long)C - (long)A] / sum;
	sumg += w * freqg * curtree.nodep[i]->x[j][0][(long)G - (long)A] / sum;
	sumt += w * freqt * curtree.nodep[i]->x[j][0][(long)T - (long)A] / sum;
      }
    }
    sum = suma + sumc + sumg + sumt;
    freqa = suma / sum;
    freqc = sumc / sum;
    freqg = sumg / sum;
    freqt = sumt / sum;
  }
}  /* empiricalfreqs */

void getinput()
{
  /* reads the input data */
  inputoptions();
  if (!freqsfrom)
    getbasefreqs();
  getdata();
  makesiteptr();
  makevalues();
  if (freqsfrom) {
    empiricalfreqs();
    getbasefreqs();
  }
}  /* getinput */

static double watterson()
{
  /* estimate theta using method of Watterson */
  long i, j, kn;
  boolean varies;
  double watter;

  kn = 0;
  for (i = 0; i < sites; i++) {
    varies = false;
    for (j = 1; j < numsp; j++) {
      if (y[j][i] != y[0][i])
	varies = true;
    }
    if (varies)
      kn++;
  }
  watter = 0.0;
  if (kn > 0) {
    for (i = 1; i < numsp; i++)
      watter += 1.0 / i;
    watter = kn / (sites * watter);
    return watter;
  }
  fprintf(outfile, "Warning:  There are no variable sites");
  fprintf(outfile, " in this data set.\n\n");
  if (menu) printf("Warning:  There are no variable sites in this data set.\n");
  else {
     fprintf(simlog, "Warning:  There are no variable sites");
     fprintf(simlog, " in this data set.\n\n");
  }
  exit(-1);
}  /* watterson */

void orient(node *p)
{
  tlist *t, *u;

  t = curtree.tymelist;

  if (p->tip) {
    p->top = true;
    p->tyme = 0.0;
    t->eventnode = p;
    t->branchlist[p->number - 1] = p;
    return;
  }

  p->top = true;
  curtree.nodep[p->number-1] = p;  /* insure that curtree.nodep points
                                    to nodes with info */

 /* since p is a top nodelet, it needs to actually store
    likelihood information, x is a NULL pointer
    in all other non-tip nodelets */
  VarMalloc(p,true);

  p->next->top = false;
  p->next->next->top = false;

  orient(p->next->back);
  orient(p->next->next->back);
  p->tyme = p->next->length + p->next->back->tyme;
  p->next->tyme = p->tyme;
  p->next->next->tyme = p->tyme;
  if (p->number == curtree.root->back->number) {
    p->back->top = false;
    p->back->tyme = rootlength;
  }
  newtymenode(&u);
  u->eventnode = p;
  while (t != NULL) {
    if (u->eventnode->tyme < t->eventnode->tyme) {
      u->prev = t->prev;
      t->prev = u;
      u->succ = t;
      u->prev->succ = u;
      break;
    }
    if (t->succ != NULL)
      t = t->succ;
    else {
      t->succ = u;
      u->prev = t;
      u->succ = NULL;
      break;
    }
  }
}  /* orient */

void finishsetup(node *p)
{
  if (p->tip) {
    ltov(p);
    return;
  }
  ltov(p);
  finishsetup(p->next->back);
  finishsetup(p->next->next->back);
  return;
} /* finishsetup */

void initbranchlist()
{
  tlist *t;
  node *p, *q;
  long i, j, k, n;

  t = curtree.tymelist;
  n = numsp;
  t->numbranch = n;
  t->age = t->succ->eventnode->tyme;
  t = t->succ;
  for (i = 0; i < (numnodes - numsp); i++) {
    /* for each interior node, do...assumes at least 3 tips */
    n--;
    t->numbranch = n;
    if (n == 1)
      t->age = t->eventnode->tyme + rootlength;
    else
      t->age = t->succ->eventnode->tyme;
    p = t->eventnode->next->back;
    q = t->eventnode->next->next->back;
    k = 0;
    for (j = 0; j < t->prev->numbranch ; j++) {
      /* for the number of branches above the coalescent node, do...*/
      if (t->prev->branchlist[j] != p && t->prev->branchlist[j] != q) {
	t->branchlist[k] = t->prev->branchlist[j];
	k++;
      }
    }
    t->branchlist[t->numbranch - 1] = t->eventnode;
    t = t->succ;
  }
  /* initialize the slidelist, assume that curtree.nodep[numsp] through
     curtree.nodep[numnodes-1] point to all the interior nodes of the
     initial tree (one node will be curtree.root->back and so ineligible
     to be slid) */
  i = 0;
  slidenum = 0;
  for (j = numsp; j < numnodes; j++) {
    if (!(curtree.nodep[j]->back->number == rootnum)) {
      slidenodes[i] = curtree.nodep[j];
      i++;
      slidenum++;
    }
  }
}  /* initbranchlist */

void inittable()
{
  long i;
  tbl = (valrec *)calloc(1,categs*sizeof(valrec));
  /* Define a lookup table. Precompute values and store them in a table */
  for (i = 0; i < categs; i++) {
    tbl[i].rat_xi = rate[i] * xi;
    tbl[i].rat_xv = rate[i] * xv;
  }
}  /* inittable */

void initweightrat()
{
  long i;
  weightrat = (double *)calloc(1,sites*sizeof(double));
  sumweightrat = 0.0;
  for (i = 0; i < sites; i++) {
    weightrat[i] = weight[i] * rate[category[i] - 1];
    sumweightrat += weightrat[i];
  }
}  /* initweightrat */

void treeout(node *p, long s, FILE **usefile)
{
  /* write out file with representation of final tree */
  long i, n, w;
  char c;
  double x;

  if (p->tip) {
    n = 0;
    for (i = 1; i <= nmlngth; i++) {
      if (p->nayme[i - 1] != ' ')
	n = i;
    }
    for (i = 0; i < n; i++) {
      c = p->nayme[i];
      if (c == ' ')
	c = '_';
      putc(c, *usefile);
    }
    col += n;
  } else {
    putc('(', *usefile);
    col++;
    treeout(p->next->back, s, usefile);
    putc(',', *usefile);
    col++;
    if (col > 45) {
      putc('\n', *usefile);
      col = 0;
    }
    treeout(p->next->next->back, s, usefile);
    putc(')', *usefile);
    col++;
  }
  if (p->v >= 1.0)
    x = -1.0;
  else
    x = lengthof(p);
  if (x > 0.0)
    w = (long)(0.4343 * log(x));
  else if (x == 0.0)
    w = 0;
  else
    w = (long)(0.4343 * log(-x)) + 1;
  if (w < 0)
    w = 0;
  if (p == curtree.root->back)
    putc(';', *usefile);
  else {
    fprintf(*usefile, ":%*.5f", (int)(w + 7), x);
    col += w + 8;
  }
}  /* treeout */

double evaluate(tree *tr, boolean first)
{
  double sum, sum2, sumc, sumterm, lterm;
  contribarr like, nulike, term, clai;
  long i, j, k;
  node *p;
  sitelike x1;

  like   = (double *)calloc(1,categs*sizeof(double));
  nulike = (double *)calloc(1,categs*sizeof(double));
  term   = (double *)calloc(1,categs*sizeof(double));
  clai   = (double *)calloc(1,categs*sizeof(double));

  sum = 0.0;
  p = tr->root->back;

  for (i = 0; i < sites; i++) {
    for (j = 0; j < categs; j++) {
       memcpy(x1, p->x[i][j], sizeof(sitelike));
       term[j] = freqa * x1[0] + freqc * x1[(long)C - (long)A] +
  	      freqg * x1[(long)G - (long)A] + freqt * x1[(long)T - (long)A];
       if(term[j] == 0) {
          fprintf(ERRFILE,"Encountered tree incompatible with data\n");
          if(first) {
             fprintf(ERRFILE,"starting tree needs to be legal\n");
             exit(-1);
          }
          curtree.likelihood = NEGMAX;
          return(-1);
       }
    }
    sumterm = 0.0;
    for (j = 0; j < categs; j++)
      sumterm += probcat[j] * term[j];
    lterm = log(sumterm);
    for (j = 0; j < categs; j++)
      clai[j] = term[j] / sumterm;
    memcpy(contribution[i], clai, categs*sizeof(double));
    if (!auto_)
      alogf->val[i] = lterm;
    sum += weight[i] * lterm;
  }
  for (j = 0; j < categs; j++)
    like[j] = 1.0;
  for (i = 0; i < sites; i++) {
    sumc = 0.0;
    for (k = 1; k <= categs; k++)
      sumc += probcat[k - 1] * like[k - 1];
    sumc *= lambda;
    memcpy(clai, contribution[i], categs*sizeof(double));
    for (j = 0; j < categs; j++)
      nulike[j] = ((1.0 - lambda) * like[j] + sumc) * clai[j];
    memcpy(like, nulike, categs*sizeof(double));
  }
  sum2 = 0.0;
  for (i = 0; i < categs; i++)
    sum2 += probcat[i] * like[i];
  sum += log(sum2);
  curtree.likelihood = sum;
  free(like);
  free(nulike);
  free(term);
  free(clai);
  return sum;
}  /* evaluate */

void nuview(node *p)
{
  long i, j;
  double w1, w2, lw1, lw2, yy1, yy2, ww1zz1, vv1zz1, ww2zz2, vv2zz2,
	 vv1zz1_sumr1, vv2zz2_sumr2, vv1zz1_sumy1, vv2zz2_sumy2, sum1, sum2,
	 sumr1, sumr2, sumy1, sumy2;
  node *q, *r;
  sitelike xx1, xx2, xx3;

  q = p->next->back;
  r = p->next->next->back;

  w1 = 1.0 - q->v;
  w2 = 1.0 - r->v;
  if (w1 > 0.0) {
    lw1 = log(w1);
    for (i = 0; i < categs; i++) {
      tbl[i].ww1 = exp(tbl[i].rat_xi * lw1);
      tbl[i].zz1 = exp(tbl[i].rat_xv * lw1);
      tbl[i].ww1zz1 = tbl[i].ww1 * tbl[i].zz1;
      tbl[i].vv1zz1 = (1.0 - tbl[i].ww1) * tbl[i].zz1;
    }
  }
  if (w2 > 0.0) {
    lw2 = log(w2);
    for (i = 0; i < categs; i++) {
      tbl[i].ww2 = exp(tbl[i].rat_xi * lw2);
      tbl[i].zz2 = exp(tbl[i].rat_xv * lw2);
      tbl[i].ww2zz2 = tbl[i].ww2 * tbl[i].zz2;
      tbl[i].vv2zz2 = (1.0 - tbl[i].ww2) * tbl[i].zz2;
    }
  }
  for (i = 0; i < sites; i++) {
    for (j = 0; j < categs; j++) {
       if(siteptr[i] == -1) { /* if we need to calculate this site */
          if (w1 <= 0.0) {
             ww1zz1 = 0.0;
             vv1zz1 = 0.0;
             yy1 = 1.0;
          } else {
             ww1zz1 = tbl[j].ww1zz1;
             vv1zz1 = tbl[j].vv1zz1;
             yy1 = 1.0 - tbl[j].zz1;
          }
          if (w2 <= 0.0) {
             ww2zz2 = 0.0;
             vv2zz2 = 0.0;
             yy2 = 1.0;
          } else {
             ww2zz2 = tbl[j].ww2zz2;
             vv2zz2 = tbl[j].vv2zz2;
             yy2 = 1.0 - tbl[j].zz2;
          }
          memcpy(xx1, q->x[i][j], sizeof(sitelike));
          memcpy(xx2, r->x[i][j], sizeof(sitelike));
          sum1 = yy1 * (freqa * xx1[0] + freqc * xx1[(long)C - (long)A] +
    	    freqg * xx1[(long)G - (long)A] + freqt * xx1[(long)T - (long)A]);
          sum2 = yy2 * (freqa * xx2[0] + freqc * xx2[(long)C - (long)A] +
    	    freqg * xx2[(long)G - (long)A] + freqt * xx2[(long)T - (long)A]);
          sumr1 = freqar * xx1[0] + freqgr * xx1[(long)G - (long)A];
          sumr2 = freqar * xx2[0] + freqgr * xx2[(long)G - (long)A];
          sumy1 = freqcy * xx1[(long)C - (long)A] + 
                  freqty * xx1[(long)T - (long)A];
          sumy2 = freqcy * xx2[(long)C - (long)A] +
                  freqty * xx2[(long)T - (long)A];
          vv1zz1_sumr1 = vv1zz1 * sumr1;
          vv2zz2_sumr2 = vv2zz2 * sumr2;
          vv1zz1_sumy1 = vv1zz1 * sumy1;
          vv2zz2_sumy2 = vv2zz2 * sumy2;
          xx3[0] = (sum1 + ww1zz1 * xx1[0] + vv1zz1_sumr1) *
                   (sum2 + ww2zz2 * xx2[0] + vv2zz2_sumr2);
          xx3[(long)C - (long)A] =
      	(sum1 + ww1zz1 * xx1[(long)C - (long)A] + vv1zz1_sumy1) *
      	(sum2 + ww2zz2 * xx2[(long)C - (long)A] + vv2zz2_sumy2);
            xx3[(long)G - (long)A] =
      	(sum1 + ww1zz1 * xx1[(long)G - (long)A] + vv1zz1_sumr1) *
      	(sum2 + ww2zz2 * xx2[(long)G - (long)A] + vv2zz2_sumr2);
            xx3[(long)T - (long)A] =
      	(sum1 + ww1zz1 * xx1[(long)T - (long)A] + vv1zz1_sumy1) *
      	(sum2 + ww2zz2 * xx2[(long)T - (long)A] + vv2zz2_sumy2);
            memcpy(p->x[i][j], xx3, sizeof(sitelike));
       }
       else {
      /* this site is just like site #(siteptr[i]), use its values */
          memcpy(p->x[i][j], p->x[siteptr[i]][j], sizeof(sitelike));
       }
    }
  }
}  /* nuview */

void update(node *p)
{
  if (!p->tip)
    nuview(p);
}  /* update */

void smooth(node *p)
{
  if (!p->tip) {
    if (!p->next->top)
      smooth(p->next->back);
    if (!p->next->next->top)
      smooth(p->next->next->back);
  }
  update(p);
}  /* smooth */

void localsmooth(node *p)
{
  if (p->number != curtree.root->number) {
     findtop(&p);
     nuview(p);
  }
  if (p->number != curtree.root->number)
     localsmooth(p->back);
} /* localsmooth */

boolean testratio()
{
  /* decide to accept or not */
  double test, x;

  if(curtree.likelihood == NEGMAX)
     return false;
  test = exp(curtree.likelihood - oldlikelihood);
  if (test >= 1.0)
    return true;
  else {
    x = randum(seed);
    if (test >= x)
      return true;
    else
      return false;
  }
}  /* testratio */

void seekch(char c) /* use only in reading file intree! */
{
  if (gch == c)
    return;
  do {
    if (eoln(intree)) {
      fscanf(intree, "%*[^\n]");
      getc(intree);
    }
    gch = getc(intree);
    if (gch == '\n')
      gch = ' ';
  } while (gch != c);
}  /* seekch */

void getch(char *c) /* use only in reading file intree! */
{
  /* get next nonblank character */
  do {
    if (eoln(intree)) {
      fscanf(intree, "%*[^\n]");
      getc(intree);
    }
    *c = getc(intree);
    if (*c == '\n')
      *c = ' ';
  } while (*c == ' ');
}  /* getch */

void processlength(node *p)
{
  long digit;
  double valyew, divisor;
  boolean pointread;
 
  pointread = false;
  valyew = 0.0;
  divisor = 1.0;
  getch(&gch);
  digit = gch - '0';
  while (((unsigned long)digit <= 9) || gch == '.'){
    if (gch == '.')
      pointread = true;
    else {
      valyew = valyew * 10.0 + digit;
      if (pointread)
	divisor *= 10.0;
    }
    getch(&gch);
    digit = gch - '0';
  }
  p->length = valyew / divisor;
  p->back->length = p->length;
}  /* processlength */

void addelement(node *p, long *nextnode)
{
  node *q;
  long i, n;
  boolean found;
  char str[nmlngth];

  getch(&gch);
  if (gch == '(') {
    (*nextnode)++;
    q = curtree.nodep[(*nextnode) - 1];
    hookup(p, q);
    addelement(q->next,nextnode);
    seekch(',');
    addelement(q->next->next, nextnode);
    seekch(')');
    getch(&gch);
  } else {
    for (i = 0; i < nmlngth; i++)
      str[i] = ' ';
    n = 1;
    do {
      if (gch == '_')
	gch = ' ';
      str[n - 1] = gch;
      if (eoln(intree)) {
	fscanf(intree, "%*[^\n]");
	getc(intree);
      }
      gch = getc(intree);
      if (gch == '\n')
	gch = ' ';
      n++;
    } while (gch != ':' && gch != ',' && gch != ')' && n <= nmlngth);
    n = 1;
    do {
      found = true;
      for (i = 0; i < nmlngth; i++)
	found = (found && str[i] == curtree.nodep[n - 1]->nayme[i]);
      if (!found)
	n++;
    } while (!(n > numsp || found));
    if (n > numsp) {
      printf("Cannot find sequence: ");
      for (i = 0; i < nmlngth; i++)
	putchar(str[i]);
      putchar('\n');
    }
    hookup(curtree.nodep[n - 1], p);
  }
  if (gch == ':')
    processlength(p);
}  /* addelement */

void treeread()
{
  long nextnode;
  node *p;

  curtree.root = curtree.nodep[rootnum - 1];
  getch(&gch);
  if (gch == '(') {
    nextnode = numsp + 1;
    p = curtree.nodep[nextnode - 1];
    addelement(p, &nextnode);
    seekch(',');
    addelement(p->next, &nextnode);
    hookup(p->next->next, curtree.nodep[rootnum - 1]);
    p->next->next->length = rootlength;
    curtree.nodep[rootnum - 1]->length = p->next->next->length;
    ltov(curtree.nodep[rootnum - 1]);
  }
  fscanf(intree, "%*[^\n]");
  getc(intree);
}  /* treeread */

void treevaluate()
{
  double dummy;

  smooth(curtree.root->back);
  smooth(curtree.root);
  dummy = evaluate(&curtree,true);
}  /* treevaluate */

void localevaluate(node *p, node *pansdaught)
/* routine assumes that p points to the only 'top' nodelet
   in node 'p' */
{
  double dummy;

  /* first update all daughters and p itself */
  if (!p->next->back->tip)
     nuview(p->next->back);
  if (!p->next->next->back->tip)
     nuview(p->next->next->back);
  nuview(p);
  if (!pansdaught->tip)
      nuview(pansdaught);
  /* now update the rest of the tree */
  localsmooth(p->back);
  dummy = evaluate (&curtree,false);
} /* localevaluate */

void copynode(node *source, node *target)
/* copies source node to target node */
{
  long i, j;
  
  for (i = 1; i <= 3; i++) {
    /* NEVER! target->next := source->next; */
    target->back = source->back;
    target->tip = source->tip;
    /* but NOT target->number := source->number; */
    if (source->x != NULL) {
       VarMalloc(target,true);
       for (j = 0; j < sites; j++) {
          memcpy(target->x[j], source->x[j], categs*sizeof(sitelike));
       }
    }
    else
       VarMalloc(target,false);
    memcpy(target->nayme, source->nayme, sizeof(naym));
    target->top = source->top;
    target->v = source->v;
    target->tyme = source->tyme;
    target->length = source->length;
    source = source->next;
    target = target->next;
  }
}  /* copynode */

/* joinnode and constructree are used for constructing a rather bad
   starting tree if the user doesn't provide one */
void joinnode(float length, node *p, node *q)
{
   hookup(p,q);
   p->length = length;
   q->length = length;
   ltov(p);
} /* joinnode */

void constructtree(long numtips, float branchlength)
{
   long i, j, nextnode;
   float height;
   node *p, *q;
 
   curtree.root = curtree.nodep[rootnum - 1];
   nextnode = numsp;
   p = curtree.root;
   q = curtree.nodep[nextnode];
 
   p->back = q;
   q->back = p;
   p->length = rootlength;
   q->length = rootlength;
   ltov(p);
 
   height = (numtips - 1) * branchlength;
   p->tyme = rootlength + height;
   for (i = 0; i < numtips - 1; i++) {
      p = curtree.nodep[i];
      q = curtree.nodep[nextnode]->next;
      joinnode(height,p,q);
      q = q->next;
      if (i != numtips-2) {
         nextnode++;
         p = curtree.nodep[nextnode];
         joinnode(branchlength,p,q);
         height -= branchlength;
      }
      else {
         p = curtree.nodep[numtips - 1];
         joinnode(height,p,q);
      }
      for (j = 0; j < 3; j++)
         q->tyme = height;
   }
} /* constructtree */
/* End bad starting tree construction */

void updateslide(node *p, boolean wanted)
/* pass me FALSE only if sure that the node is invalid */
{
  boolean valid, found;
  node *q;
  long j, k;

  valid = true;
  q = p;
  if (!wanted)
    valid = false;
  else {
    if (q->tip)
      valid = false;
    else {
      findtop(&q);
      if (q->back->tip)
	valid = false;
    }
  }
  found = false;
  j = 1;
  while (!found && j <= slidenum) {
    if (slidenodes[j - 1]->number == p->number) {
      found = true;
      k = j;
    }
    j++;
  }
  if (valid && !found) {
    slidenum++;
    slidenodes[slidenum - 1] = p;
  }
  if (valid || !found)
    return;
  while (k < slidenum) {
    slidenodes[k - 1] = slidenodes[k];
    k++;
  }
  slidenum--;
}  /* updateslide */

void rebuildbranch()
{
  tlist *t;
  node *p;
  long i, k;
  boolean done;

  t = curtree.tymelist->succ;
  done = false;
  do {
    if (t->succ == NULL)
      done = true;
    p = t->eventnode;
    k = 1;
    findtop(&p);
    for (i = 0; i < t->prev->numbranch; i++) {
      if (t->prev->branchlist[i] != p->next->back &&
	  t->prev->branchlist[i] != p->next->next->back) {
	t->branchlist[k - 1] = t->prev->branchlist[i];
	k++;
      }
    }
    t->numbranch = t->prev->numbranch - 1;
    t->branchlist[t->numbranch - 1] = p;
    t = t->succ;
  } while (!done);
}  /* rebuildbranch */

void setlength(long numl, long numother, double tstart, double tlength,
   node *p)
{
  double x, e1;

  e1 = (numl - 1.0) * 2 + numother * 2;
  x = -(theta0 / e1) * log(1 - randum(seed) * (1 - exp(-(e1 * tlength / theta0))));
  if ((unsigned)x > tlength)
     fprintf(ERRFILE,"disaster in setlength\n");
  p->tyme = tstart + x;
}  /* setlength */

void setlength2(long numother, double tstart, double tlength, 
   node *p, node *q)
{
  long i;
  double x, xmin, xmax, r, xnew, e1, e2, norm;

  r = randum(seed);
  e1 = exp(numother * -2 * tlength / theta0);
  e2 = exp(-((numother * 2 + 2) * tlength / theta0));
  norm = -3 * e1 / ((numother + 1) * twocollis(numother, tlength));
  xmin = 0.0;
  xmax = tlength;
  for (i = 1; i <= 20; i++) {
    x = (xmax + xmin) / 2.0;
    xnew = norm *
	(1.0 / (numother * 2 + 3) * (exp(-((numother * 4 + 6) * x / theta0)) - 1) -
	 e2 / (numother + 2) * (exp(-((numother * 2 + 4) * x / theta0)) - 1));
    if (xnew > r)
      xmax = x;
    else
      xmin = x;
  }
  if ((unsigned)x > tlength)
     fprintf(ERRFILE,"disaster in setlength2\n");
  p->tyme = tstart + x;
  setlength(2L, numother, p->tyme, tlength - x, q);
}  /* setlength2 */

void updatebranch(node *oldans, node *oldp, node *prime)
{
  subtymelist(oldans, oldp);
  inserttymelist(prime);
  rebuildbranch();
}  /* updatebranch */

long counttymelist(tlist *first, tlist *last)
{
   long count;
   tlist *t;

   count = 0;
   for (t = first; t != last; t = t->succ)
      count++;
   return(count+1);
} /* counttymelist */

void slide()
{
/* Local variables for slide: */

  node *prime, *oldp, *oldans, *primans, *pansdaught, *ans, *p,
     *oldbr[3], *newbr[3];
  long i, j, k, leftout, cline, numother, numintervals;
  double chance, tlength, normalizer, *coll2[2], *coll3[3], c[3];
  tlist *t, *tstart, *tend, *tplus;
  boolean done, skipped;

  do { /* There exists a chance that randum returns a 1 */
     i = (long)(randum(seed) * slidenum) + 1;
  } while (i == slidenum + 1);
  oldp = slidenodes[i - 1];
  findtop(&oldp);
  oldans = oldp->back;
  /* copy old nodes to new */
  newnode(&prime);
  copynode(oldp, prime);
  newnode(&primans);
  copynode(oldans, primans);
  hookup(prime, primans);
  /* name and connect nodes */
  oldbr[0] = prime->next->back;
  oldbr[1] = prime->next->next->back;
  if (!primans->next->top) {
    pansdaught = primans->next;
    oldbr[2] = primans->next->back;
    ans = primans->next->next->back;
  } else {
    pansdaught = primans->next->next;
    oldbr[2] = primans->next->next->back;
    ans = primans->next->back;
  }
  /* tymelist sort the three branches' tips in arbitrary order */
  j = 1;
  for (i = 0; i <= 2; i++) {
    if (oldbr[i]->tip) {
      newbr[j - 1] = oldbr[i];
      j++;
    }
  }
  /* remainder of tree */
  if (j <= 3) {
    t = curtree.tymelist->succ;
    done = false;
    while (!done) {
      for (i = 0; i <= 2; i++) {
	if (oldbr[i]->number == t->eventnode->number) {
	  newbr[j - 1] = oldbr[i];
	  j++;
	  if (j > 3)
	    done = true;
	}
      }
      t = t->succ;
      if (t == NULL) {
	printf("ERROR IN TYMESORT\n");
        exit(-1);
      }
    }
  }
  gettymenode(&tplus, newbr[2]->number);
  skipped = false; /* needed for frees in zero length case */
  /* zero length branches are a special case */
  if (newbr[1]->tyme == ans->tyme) {
    prime->tyme = newbr[1]->tyme;
    primans->tyme = newbr[1]->tyme;
    skipped = true;
  } else {
    /* initialize probability arrays for state (ie. 1,2 or 3 branches
       present) */
    gettymenode(&tstart, newbr[1]->number);
    gettymenode(&tend, ans->number);
    numintervals = counttymelist(tstart,tend);
    coll2[0] = (double *)calloc(1,numintervals * sizeof(double));
    coll2[1] = (double *)calloc(1,numintervals * sizeof(double));
    coll3[0] = (double *)calloc(1,numintervals * sizeof(double));
    coll3[1] = (double *)calloc(1,numintervals * sizeof(double));
    coll3[2] = (double *)calloc(1,numintervals * sizeof(double));
    t = tstart;
    i = 0;
    cline = 1;
    numother = tstart->numbranch - 2;
    /* initialize 2-array */
    coll2[0][i] = 0.0;
    coll2[1][i] = 1.0;
    /* fill up 2-array */
    while (t != tplus) {
      i++;
      tlength = t->age - t->eventnode->tyme;
      if ((numother * 6 + 6) * tlength / theta0 > 20.0)
	tlength = 20 * theta0 / (numother * 6 + 6);
      coll2[0][i] = coll2[0][i - 1] * zerocollis(1L, numother, tlength)
                  + coll2[1][i - 1] * onecollis(2L, numother, tlength);
      coll2[1][i] = coll2[1][i - 1] * zerocollis(2L, numother, tlength);
      normalizer = coll2[0][i] + coll2[1][i];
      coll2[0][i] /= normalizer;
      coll2[1][i] /= normalizer;
      if (normalizer == 0.0) {
         fprintf(ERRFILE,"Encountered machine precision limits!\n");
         exit(-1);
      }
      t = t->succ;
      if (t->eventnode->number != oldp->number &&
	  t->eventnode->number != oldans->number)
	numother--;
    }
    if (newbr[2]->tyme >= ans->tyme) {
      /* case 2 zero length */
      primans->tyme = tplus->eventnode->tyme;
    } else {
      /* initialize 3-array */
      j = 0;
      numother--;
      coll3[0][j] = 0.0;
      coll3[1][j] = coll2[0][i];
      coll3[2][j] = coll2[1][i];
      /* fill up 3-array */
      done = false;
      while (!done) {
	j++;
	tlength = t->age - t->eventnode->tyme;
	if ((numother * 6 + 6) * tlength / theta0 > 20.0)
	  tlength = 20 * theta0 / (numother * 6 + 6);
	coll3[0][j] = coll3[0][j - 1] * zerocollis(1L, numother, tlength) +
		      coll3[1][j - 1] * onecollis(2L, numother, tlength) +
		      coll3[2][j - 1] * twocollis(numother, tlength);
	coll3[1][j] = coll3[1][j - 1] * zerocollis(2L, numother, tlength) +
		      coll3[2][j - 1] * onecollis(3L, numother, tlength);
	coll3[2][j] = coll3[2][j - 1] * zerocollis(3L, numother, tlength);
        normalizer = coll3[0][j] + coll3[1][j] + coll3[2][j];
        coll3[0][j] /= normalizer;
        coll3[1][j] /= normalizer;
        coll3[2][j] /= normalizer;
        if (normalizer == 0.0) {
           fprintf(ERRFILE,"Encountered machine precision limits!\n");
           exit(-1);
        }
	if (t->succ == tend) {
	  done = true;
	  break;
	}
	t = t->succ;
	if (t->eventnode->number != oldp->number &&
	    t->eventnode->number != oldans->number)
	  numother--;
      }
      /* now find out when prime and primans collide */
      k = j;
      while (cline != 3 && k != 0 && t != NULL) {
	tlength = t->age - t->eventnode->tyme;
	if ((numother * 6 + 6) * tlength / theta0 > 20.0)
	  tlength = 20 * theta0 / (numother * 6 + 6);
	chance = randum(seed);
	if (cline == 1) {
          c[0] = coll3[0][k-1] * zerocollis(1L, numother, tlength);
          c[1] = coll3[1][k-1] * onecollis(2L, numother, tlength);
          c[2] = coll3[2][k-1] * twocollis(numother, tlength);
          normalizer = c[0] + c[1] + c[2];
          c[0] /= normalizer;
          c[1] /= normalizer;
          if (chance > c[0]) {
            chance -= c[0];
            if (chance > c[1]) { /* two collisions */
              cline += 2;
	      setlength2(numother, t->eventnode->tyme, tlength, prime,
			 primans);
            } else { /* one collision */
              cline++;
	      setlength(2L, numother, t->eventnode->tyme, tlength, primans);
            }
          }
	} else {  /* cline must equal 2 */
          c[0] = coll3[1][k-1] * zerocollis(2L, numother, tlength);
          c[1] = coll3[2][k-1] * onecollis(3L, numother, tlength);
          normalizer = c[0] + c[1];
          c[0] /= normalizer;
	  if (chance > c[0]) {
	    cline++;
	    setlength(3L, numother, t->eventnode->tyme, tlength, prime);
	  }
	}
	if (t != NULL) {
	  if (t->eventnode->number != oldp->number &&
	      t->eventnode->number != oldans->number)
	    numother++;
	}
	k--;
	t = t->prev;
      }
      cline--;   /* A lineage dies! */
      numother++;
      if (cline == 0) {
	printf("ERROR in slide, no lineages left!");
	printf("  cline = 0, too few uncollisions\n");
	printf(" in loop %12ld\n", indecks);
        exit(-1);
      }
    }
    k = i;
    t = tplus->prev;
    while (cline == 1 && k != 0 && t != NULL) {
      tlength = t->age - t->eventnode->tyme;
      if ((numother * 6 + 6) * tlength / theta0 > 20.0)
	tlength = 20 * theta0 / (numother * 6 + 6);
      chance = randum(seed);
      c[0] = coll2[0][k-1] * zerocollis(1L, numother, tlength);
      c[1] = coll2[1][k-1] * onecollis(2L, numother, tlength);
      normalizer = c[0] + c[1];
      c[0] /= normalizer;
      if (chance > c[0]) {
	cline++;
	setlength(2L, numother, t->eventnode->tyme, tlength,prime);
      } else {
	k--;
	t = t->prev;
      }
      if (t->eventnode->number != oldp->number &&
	  t->eventnode->number != oldans->number)
	numother++;
    }
  }
  /* set up the phylogeny */
  if (prime->tyme < tplus->eventnode->tyme) {
    /* case 1 */
    hookup(newbr[0], prime->next);
    hookup(newbr[1], prime->next->next);
    hookup(newbr[2], pansdaught);
  } else {
    /* case 2 */
    leftout = (long)(randum(seed) * 3.0) + 1;
    p = prime->next;
    for (i = 1; i <= 3; i++) {
      if (i != leftout) {
	hookup(newbr[i - 1], p);
	p = p->next;
      } else
	hookup(newbr[i - 1], pansdaught);
    }
  }
  if (primans->next == pansdaught)
    hookup(primans->next->next, ans);
  else
    hookup(primans->next, ans);
  prime->next->tyme = prime->tyme;
  prime->next->next->tyme = prime->tyme;
  primans->next->tyme = primans->tyme;
  primans->next->next->tyme = primans->tyme;
  for (i = 0; i <= 2; i++)
    ltov(newbr[i]);
  ltov(prime);
  ltov(ans);
  /* get acceptance ratio */
  if (newbr[1]->tyme == ans->tyme)
     accept_slide = true;
  else {
     localevaluate(prime,pansdaught->back);
     accept_slide = testratio();
  }
  slid++;
  if (!skipped) {
     free(coll2[0]);
     free(coll2[1]);
     free(coll3[0]);
     free(coll3[1]);
     free(coll3[2]);
  }
  if (accept_slide) {
    slacc++;
    updatebranch(oldans,oldp,prime);
    updateslide(oldp, false);
    freenode(oldp);
    updateslide(oldans, false);
    freenode(oldans);
    updateslide(primans, true);
    updateslide(prime, true);
    return;
  }
  hookup(oldbr[0], oldp->next);
  hookup(oldbr[1], oldp->next->next);
  if (!oldans->next->top) {
    hookup(oldbr[2], oldans->next);
    hookup(oldans->next->next, ans);
  } else {
    hookup(oldbr[2], oldans->next->next);
    hookup(oldans->next, ans);
  }
  p = oldp;
  for (i = 1; i <= 2; i++) {
    p = p->next;
    p->back->v = p->v;
  }
  p = oldans;
  for (i = 1; i <= 2; i++) {
    p = p->next;
    p->back->v = p->v;
  }
  localevaluate(oldp,oldbr[2]);
  freenode(prime);
  freenode(primans);
       
  curtree.likelihood = oldlikelihood;
}  /* slide */

void scoretree(long chain)
{
  tlist *t;
  long refchain;
   
  refchain = REF_CHAIN(chain);

  sum[cycle][refchain][numout[chaintype] - 1] = 0.0;
  t = curtree.tymelist;
  while (t != NULL) {
    sum[cycle][refchain][numout[chaintype] - 1] +=
      t->numbranch * (t->numbranch - 1) * (t->age - t->eventnode->tyme);
    t = t->succ;
  }
  if (sum[cycle][refchain][numout[chaintype] - 1] == 0.0) {
     fprintf(ERRFILE,"WARNING:  Tree has become length zero\n");
  }
}  /* scoretree */

void thetaval(double thgiven, long chain, double *likel)
/* thetaval:  Ln(likelihood) of a chain, "chain", at a given theta,
   "thgiven". */
/* EQN:  k = # of active lineages within an interval
         inter_lngth = length of an interval
         thgiven = theta of interest
         thchain = theta under which the chain was run
         numinter = # of intervals in tree
         numtree = # of trees in chain

         finterval = sum_over_intervals[(k * (k - 1)) * inter_lngth]

                                exp(-finterval/thgiven)
         ftree = sum_over_trees -----------------------
                                exp(-finterval/thchain)

         likelihood = ((thchain/thgiven) ** numinter) * (ftree/numtree)

         Note: thetaval returns Ln(likelihood)!!!!!
*/
{
  double th, minsum, summ, naught;
  long j, refchain;

  refchain = REF_CHAIN(chain);
  naught = theti[cycle][chain];
  th = 1 / thgiven - 1 / naught;
  if (th == 0) {
    (*likel) = 0.0;
    return;
  }
  minsum = (double)(sum[cycle][refchain][0]) * th; /* Mr. Crashie. */

  for (j = 1; j < numout[chaintype]; j++) {
    if ((double)(sum[cycle][refchain][j]) * th < minsum)
      minsum = (double)(sum[cycle][refchain][j]) * th;
  }

  summ = 0.0;
  for (j = 0; j < numout[chaintype]; j++) {
    if (minsum - (double)(sum[cycle][refchain][j]) * th > EXPMIN)
      summ += exp(minsum - (double)(sum[cycle][refchain][j]) * th);
  }
  summ /= numout[chaintype];
  (*likel) = (numsp - 1) * (log(naught) - log(thgiven)) - minsum + log(summ); 
}  /* thetaval */

/* functions to be used in Newton-Raphson iteration of single chain
   point estimate */
double fnx (long chain, double theval, long numints, long *fxplus,
            boolean *fxzero)
/* this is the first derivative of the likelihood of a single chain.
   Note that it returns an answer in log form. */
/* EQN:  k = # of active lineages within an interval
         inter_lngth = length of an interval
         thgiven = theta of interest
         thchain = theta under which the chain was run
         numints = # of intervals in tree

         finterval = sum_over_intervals [(k * (k - 1)) * inter_lngth]

         likelihood = sum_over_trees [(finterval/thgiven - numints) *
                      exp(-finterval * (1.0/thgiven - 1.0/thchain))]

         Note: fnx returns Ln(likelihood)!!!!!
*/
{
  long i, refchain, *sign;
  double other, chaintheta, *temp, maxtemp, result;

  temp = (double *)calloc(1,numout[chaintype]*sizeof(double));
  sign = (long *)calloc(1,numout[chaintype]*sizeof(long));
  chaintheta = theti[cycle][chain];
  refchain = REF_CHAIN(chain);

  maxtemp = NEGMAX;
  for(i = 0; i < numout[chaintype]; i++) {
     other = ((double)sum[cycle][refchain][i])/theval - numints;
     if (other > 0) sign[i] = 1;
     else if (other < 0) sign[i] = -1;
          else sign[i] = 0;
     temp[i] = log(fabs(other)) +
        (-(double)sum[cycle][refchain][i])*(1.0/theval - 1.0/chaintheta);
     if (temp[i] > maxtemp) maxtemp = temp[i];
  }

  result = 0.0;
  for(i = 0; i < numout[chaintype]; i++) {
     if (temp[i] - maxtemp > EXPMIN)
        result += sign[i] * exp(temp[i] - maxtemp);
  }

  *fxzero = false;
  if (result == 0.0) {
     *fxzero = true;
     return(0.0);
  }

  *fxplus = 1;
  if (result < 0.0) *fxplus = -1;
  if (result != 0.0) result = log(fabs(result)) + maxtemp;

  free(sign);
  free(temp);
  return result;
} /* fnx */

double dfnx (long chain, double theval, long numints, long *dfxplus,
             boolean *dfxzero)
/* this is the second derivative of the likelihood of a single chain.
   Note that it returns an answer in log form. */
/* EQN:  k = # of active lineages within an interval
         inter_lngth = length of an interval
         thgiven = theta of interest
         thchain = theta under which the chain was run
         numints = # of intervals in tree

         finterval = sum_over_intervals [(k * (k - 1)) * inter_lngth]

         f1x = finterval/thgiven
         f2x = finterval/(thgiven**2)

         likelihood = sum_over_trees [
                      exp(-finterval * (1.0/thgiven - 1.0/thchain)) *
             (f2x*(-2*numints-2) + (numints*(numints+1))/thgiven + f1x*f2x) 
             ]

         Note: dfnx returns Ln(likelihood)!!!!!
*/
{
  long i, refchain, *sign;
  double f1x, f2x, other, chaintheta, *temp, result, maxtemp;

  temp = (double *)calloc(1,numout[chaintype]*sizeof(double));
  sign = (long *)calloc(1,numout[chaintype]*sizeof(long));
  chaintheta = theti[cycle][chain];
  refchain = REF_CHAIN(chain);

  maxtemp = NEGMAX;
  for(i = 0; i < numout[chaintype]; i++) {
     f1x = ((double)sum[cycle][refchain][i])/theval;
     f2x = ((double)sum[cycle][refchain][i])/(theval*theval);
     other = (f2x*(-2*numints-2) + (numints*(numints+1))/theval + f1x*f2x);
     if (other > 0) sign[i] = 1;
     else if (other < 0) sign[i] = -1;
          else sign[i] = 0;
     temp[i] = (-(double)sum[cycle][refchain][i])*(1.0/theval - 1.0/chaintheta) +
        log(fabs(other));
     if (temp[i] > maxtemp) maxtemp = temp[i];
  }

  result = 0.0;
  for(i = 0; i < numout[chaintype]; i++) {
     if (temp[i] - maxtemp > EXPMIN)
        result += sign[i] * exp(temp[i] - maxtemp);
  }

  *dfxzero = false;
  if (result == 0.0) {
     *dfxzero = true;
     return(0.0);
  }

  *dfxplus = 1;
  if (result < 0.0) *dfxplus = -1;
  if (result != 0.0) result = log(fabs(result)) + maxtemp;

  free(sign);
  free(temp);
  return result;
} /* dfnx */

void thetapoint(long chain, boolean chend, boolean rend)
/* solve for maximum of theta-likelihood curve using Newton-Raphson
   iteration */
/* EQN:  SUMs are over all intervals in the tree.
         k = # of active (coalesceable) lineages within an interval.
         f1(x) = SUM[k*(k-1)*t/x]
         f2(x) = SUM[k*(k-1)*t/x**2]
         f3(x) = SUM[-k*(k-1)*t/(1/x-1/chaintheta)]
         f4(x) = f2(x)*(-2*#intervals-2) + (#intervals**2+#intervals)/x +
                 f2(x)*f1(x)

                 exp[f3(oldtheta)] * (f1(oldtheta) - #intervals) 
       change =  -----------------------------------------------
                     ABS[exp[f3(oldtheta)] * f4(oldtheta)]

   newtheta = oldtheta + change 

   Both numerator and denominator of the above are summed over all trees.

   Note that this is not the standard Newton-Raphson iteration.  The
   ratio is being added rather than subtracted, and the absolute value
   of the second derivative is used.  These changes should ensure that
   the function is always moving towards a relative maximum, rather than
   a minimum.
*/
{
  int numloop; /* type "int" because of library function demands! */
  long i, numintervals, fxplus, dfxplus;
  double theta, newtheta, oldlike, newlike, fx, dfx, change;
  boolean fxzero, dfxzero;

  /* point estimate of theta */
  theta = watttheta;
  numintervals = numsp - 1; /* WARNING--Wrong for recombination */
  thetaval(theta,chain,&oldlike);
  i = 0;

  /* solve by modified Newton Raphson */
  while (1) {
     fx = fnx(chain,theta,numintervals,&fxplus,&fxzero);
     dfx = dfnx(chain,theta,numintervals,&dfxplus,&dfxzero); 

     if (fxzero) /* found a maximum! at theta! */
        break;
     if (dfxzero) {
        theta += epsilon;
        fx = fnx(chain,theta,numintervals,&fxplus,&fxzero);
        dfx = dfnx(chain,theta,numintervals,&dfxplus,&dfxzero);
     }
     if (dfxplus < 0) change = fxplus * exp(fx - dfx);
     else change = fxplus * theta/2.0;

     newtheta = theta + change;

    /* now deal with negative or zero theta values */
     numloop = 0;
     while (newtheta <= 0) {
        numloop++;
        newtheta = theta + ldexp(change,-numloop);
     }
     thetaval(newtheta,chain,&newlike);
     if(newlike < oldlike) {
    /* in case we overshoot the maximum, don't jump so far...*/
        numloop = 0;
        while(1) {
           numloop++;
           newtheta = theta + ldexp(change,-numloop);
           if (newtheta <= 0) continue;
           thetaval(newtheta,chain,&newlike);
           if(newlike >= oldlike) break;
        }
     }

     oldlike = newlike;
     if(fabs(newtheta - theta) < epsilon) {
        theta = newtheta;
        break;
     }
     theta = newtheta;
     i++;
  }


  if (!chend) {
    if (thetaout) fprintf(thetafile, 
      "within chain %ld point: %10.7f log likelihood %10.7f\n",
      chain+1, theta, oldlike);
    return;
  }
  if (theta > 0) {
    theta0 = theta;
    theti[cycle][chain+1] = theta;
    lntheti[cycle][chain+1] = log(theta);
    if (thetaout) fprintf(thetafile, 
      "chain %ld point: %10.7f log likelihood: %10.7f\n",
      chain+1, theta, oldlike);
    if (rend) { 
      fprintf(outfile,
        "Single chain point estimate of theta (from final chain)=%12.8f\n",
        theta);
    }
    return;
  }
  /* the estimate failed */
  theti[cycle][chain+1] = theti[cycle][chain];
  lntheti[cycle][chain+1] = lntheti[cycle][chain];
  fprintf(ERRFILE,"WARNING, point estimate of theta failed!\n");
  fprintf(ERRFILE,"using previous iteration theta estimate\n");
  if (rend) 
    fprintf(outfile,"Single chain point estimate of theta failed\n");
}  /* thetapoint */

double interp(double theta, long firstlong, long lastlong, 
   double *lthetai, long locus) 
/* This is Elizabeth's equation for combining estimates for
   a particular 'theta'   
   In other words, this is the Ln(likelihood) of a given "theta";
   over a set of chains, the last of which is "chain"; with
   a pre-computed set of Ln(likelihoods) for each chain contained
   in "lthetai" */
/* EQN: k = # of active (coalesceable) lineages within an interval.
        inter_lngth = length of an interval
        thgiven = theta of interest
        thchain = theta under which the chain was run
        numints = # of intervals in tree
        numtree = # of trees in chain
        plike = provisional likelihood of a chain

        f1(x) = sum_over_intervals [(k*(k-1) * inter_lngth)/x]

        numer = ((1/thgiven)**numints) * exp(-f1(thgiven))
        denom = sum_over_chains [numtree * exp(-f1(thchain)) / 
                                 (plike * (thchain**numints))]

        likelihood = sum_over_all_trees [numer / denom]

        Note: dfnx returns Ln(likelihood)!!!!!
*/
{
  long i, j, refchain, dataset, trii, numchains;
  double numer, denom, lntheta, **num, bigsum, maxdenom, maxnum, *tempdenom;

  numchains = lastlong + 1;
  num = (double **)calloc(1,numchains * sizeof (double *));
  num[firstlong] = (double *)calloc(1,chains[1]*numtrees*sizeof(double));
  for (i = firstlong + 1 ; i <= lastlong; i++)
     num[i] = num[firstlong] + (i-firstlong)*numtrees;
  tempdenom = (double *)calloc(1,numchains * sizeof(double));

  bigsum = 0.0;
  lntheta = log(theta);
  maxnum = NEGMAX;
  for (dataset = firstlong; dataset <= lastlong; dataset++) {
    refchain = REF_CHAIN(dataset);
    for (trii = 0; trii < numout[1]; trii++) {
      numer = (1 - numsp) * lntheta - 
              (double)(sum[locus][refchain][trii]) / theta;
      denom = 0.0;
      maxdenom = NEGMAX;
      for (j = firstlong; j <= lastlong; j++) {
	tempdenom[j] =  log((double)numout[1]) - 
            (double)(sum[locus][refchain][trii]) / theti[locus][j] -
            lthetai[j] + (1 - numsp) * lntheti[locus][j];
	if (tempdenom[j] > maxdenom) maxdenom = tempdenom[j];
      }
      for (j = firstlong; j <= lastlong; j++)
        if (tempdenom[j] - maxdenom > EXPMIN)
	   denom += exp(tempdenom[j] - maxdenom);
      num[dataset][trii] = numer - log(denom) - maxdenom;
      if (num[dataset][trii] > maxnum) maxnum = num[dataset][trii];
    }
  }
  for (dataset = firstlong; dataset <= lastlong; dataset++) {
    for (trii = 0; trii < numout[1]; trii++) {
      if (num[dataset][trii] - maxnum > EXPMIN)
	bigsum += exp(num[dataset][trii] - maxnum);
    }
  }
  free(num[firstlong]);
  free(num);
  free(tempdenom);
  return (log(bigsum) + maxnum);
  /* we're just trying to sum up num, honest */
}  /* interp */

/* functions to be used in Newton-Raphson iteration of combined chain
   point estimate */
double combined_fnx (double theval, long numints, double **treewt,
   long firstlong, long lastlong, long *fxplus, boolean *fxzero,
   long lowcus)
/* this is the first derivative of the combined likelihood of several
   chains.  Note that it returns an answer in log form. */
/* EQN:  k = # of active lineages within an interval
         inter_lngth = length of an interval
         thgiven = theta of interest
         thchain = theta under which the chain was run
         numints = # of intervals in tree
         plike = provisional likelihood of a chain

         kk = sum_over_intervals (k*(k - 1) * inter_lngth)

         chainlike = sum_over_trees [
                     numtree * exp(-kk/thchain) /
                     (plike * (thchain**numints)) *
                     exp(-kk/thgiven) * (kk/thgiven - numints)
                     ]

         likelihood = product_over_chains [chainlike]

         Note: combined_fnx returns Ln(likelihood)!!!!!
*/
{
  long i, j, refchain;
  double **temp, kk, other, maxtemp, result, **sign;

  temp = (double **)calloc(1, totchains * sizeof(double *));
  temp[firstlong] = (double *)calloc(1,chains[1]*numtrees*sizeof(double));
  sign = (double **)calloc(1,totchains * sizeof(double *));
  sign[firstlong] = (double *)calloc(1,chains[1]*numtrees*sizeof(double));
  for (i = firstlong + 1; i <= lastlong; i++) {
     temp[i] = temp[firstlong] + (i-firstlong)*numtrees;
     sign[i] = sign[firstlong] + (i-firstlong)*numtrees;
  }

  maxtemp = NEGMAX;
  for(i = firstlong; i <= lastlong; i++) {
     refchain = REF_CHAIN(i);
     for(j = 0; j < numout[1]; j++) {
        kk = (double)sum[lowcus][refchain][j];
        other = kk/theval - numints;
        if (other > 0) sign[i][j] = 1.0;
        else if (other < 0) sign[i][j] = -1.0;
        else sign[i][j] = 0.0;
        temp[i][j] = treewt[i][j] - kk/theval + log(fabs(other));
        if (temp[i][j] > maxtemp) maxtemp = temp[i][j];
     }
  }

  result = 0.0;
  for(i = firstlong; i <= lastlong; i++)
     for(j = 0; j < numout[1]; j++) {
        if (temp[i][j] - maxtemp > EXPMIN)
           result += sign[i][j] * exp(temp[i][j] - maxtemp);
     }

  *fxzero = false;
  if (result == 0.0) {
     *fxzero = true;
     return(0.0);
  }

  *fxplus = 1;
  if (result < 0) *fxplus = -1;
  else if (result > 0) result = log(fabs(result)) + maxtemp;

  free(temp[firstlong]);
  free(temp);
  free(sign[firstlong]);
  free(sign);

  return result;
} /* combined_fnx */

double combined_dfnx (double theval, long numints, double **treewt,
   long firstlong, long lastlong, long *dfxplus, boolean *dfxzero,
   long lowcus)
/* this is the second derivative of the combined likelihood of several
   chains.  Note that it returns an answer in log form. */
/* EQN:  k = # of active lineages within an interval
         inter_lngth = length of an interval
         thgiven = theta of interest
         thchain = theta under which the chain was run
         numints = # of intervals in tree
         plike = provisional likelihood of a chain

         kk = sum_over_intervals (k*(k - 1) * inter_lngth)
         f1x = kk/thgiven
         f2x = kk/(thgiven**2)

         chainlike = sum_over_trees [
                     numtree * exp(-kk/thchain) /
                     (plike * (thchain**numints)) *
                     exp(-kk/thgiven) * 
                     (f2x*(-2*numints-2) + (numints*(numints+1))/thgiven +
                     f1x*f2x)
                     ]

         likelihood = product_over_chains [chainlike]

         Note: combined_dfnx returns Ln(likelihood)!!!!!
*/
{
  long i, j, refchain;
  double f1x, f2x, kk, other, **temp, maxtemp, result, **sign;

  temp = (double **)calloc(1, totchains * sizeof(double *));
  temp[firstlong] = (double *)calloc(1,chains[1]*numtrees*sizeof(double));
  sign = (double **)calloc(1,totchains * sizeof(double *));
  sign[firstlong] = (double *)calloc(1,chains[1]*numtrees*sizeof(double));
  for (i = firstlong + 1; i <= lastlong; i++) {
     temp[i] = temp[firstlong] + (i-firstlong)*numtrees;
     sign[i] = sign[firstlong] + (i-firstlong)*numtrees;
  }

  maxtemp = NEGMAX;
  for(i = firstlong; i <= lastlong; i++) {
     refchain = REF_CHAIN(i);
     for(j = 0; j < numout[1]; j++) {
        kk = (double)sum[lowcus][refchain][j];
        f1x = kk/theval;
        f2x = kk/(theval*theval);
        other = f2x*(-2*numints-2) + (numints*(numints+1))/theval + f1x*f2x;
        if (other > 0) sign[i][j] = 1.0;
        else if (other < 0) sign[i][j] = -1.0;
             else sign[i][j] = 0.0;
        temp[i][j] = treewt[i][j] - kk/theval + log(fabs(other));
        if (temp[i][j] > maxtemp) maxtemp = temp[i][j];
     }
  }

  result = 0.0;
  for(i = firstlong; i <= lastlong; i++)
     for(j = 0; j < numout[1]; j++) {
        if (temp[i][j] - maxtemp > EXPMIN)
           result += sign[i][j] * exp(temp[i][j] - maxtemp);
     }

  *dfxzero = false;
  if (result == 0.0) {
     *dfxzero = true;
     return(0.0);
  }

  /* take the absolute value of result so that Newton-Raphson will always
     be moving uphill */
  *dfxplus = 1;
  if (result < 0) *dfxplus = -1;
  else if (result > 0) result = log(fabs(result)) + maxtemp;

  free(temp[firstlong]);
  free(temp);
  free(sign[firstlong]);
  free(sign);

  return result;
} /* combined_dfnx */

void thetapoint2(long firstlong, long lastlong, double *lthetai,
   boolean rend)
/* solve for maximum of theta-combined likelihood curve using Newton-Raphson
   iteration */
/* EQN:

                 combined_fnx
       change =  -------------
                 combined_dfnx 

   newtheta = oldtheta + change 

   Note that this is not the standard Newton-Raphson iteration.  The
   ratio is being added rather than subtracted, and the absolute value
   of the second derivative is used.  These changes should ensure that
   the function is always moving towards a relative maximum, rather than
   a minimum.
*/
{
  int numloop; /* type "int" because of library function demands! */
  double newtheta, theta, oldlike, newlike, fx, dfx, bestlike, besttheta,
     **treewt, *tempwt, maxwt, change;
  long	dataset, trii, i, chosen, numchains, numintervals, fxplus,
     dfxplus, refchain;
  boolean fxzero, dfxzero;

  numchains = lastlong + 1; /* chains count from zero */
  numintervals = numsp - 1; /* WARNING--Wrong for recombination */

  treewt = (double **)calloc(1,totchains * sizeof(double *));
  treewt[firstlong] = (double *)calloc(1,chains[1]*numtrees*sizeof(double));
  for (i=firstlong+1; i<=lastlong; i++)
     treewt[i] = treewt[firstlong] + (i-firstlong)*numtrees;
  tempwt = (double *)calloc(1,numchains * sizeof(double));
  
  /* first calculate the weighting factor for each tree */
  for (dataset = firstlong; dataset <= lastlong; dataset++) {
    refchain = REF_CHAIN(dataset);
    for (trii = 0; trii < numout[1]; trii++) {
      maxwt = NEGMAX;
      treewt[dataset][trii] = 0.0;
      for (i = firstlong; i <= lastlong; i++) {
	tempwt[i] =  log((double)numout[1]) - 
	  (double)(sum[cycle][refchain][trii]) / theti[cycle][i] -
	  lthetai[i] + (1 - numsp) * lntheti[cycle][i];
	if (tempwt[i] > maxwt) maxwt = tempwt[i];
      }
      for (i = firstlong; i <= lastlong; i++)
        if (tempwt[i] - maxwt > EXPMIN)
	   treewt[dataset][trii] += exp(tempwt[i] - maxwt);
      treewt[dataset][trii] = -(log(treewt[dataset][trii]) + maxwt);
    }
  }

  /* try the estimate once per chain to find the best of possible
   multiple maxima */
  besttheta = NEGMAX;
  bestlike = NEGMAX;
  for (chosen = firstlong; chosen <= lastlong; chosen++) {
    theta = theti[cycle][chosen];
    oldlike = interp(theta,firstlong,lastlong,lthetai,cycle);
    i = 0;

    while (1) {
       fx = combined_fnx(theta,numintervals,treewt,firstlong,lastlong,
          &fxplus,&fxzero,cycle);
       dfx = combined_dfnx(theta,numintervals,treewt,firstlong,lastlong,
          &dfxplus,&dfxzero,cycle);
  
       if (fxzero) {/* found a maximum! at theta! */
          newlike = oldlike;
          break;
       }
       if (dfxzero) {
          theta += epsilon;
          fx = combined_fnx(theta,numintervals,treewt,firstlong,lastlong,
             &fxplus,&fxzero,cycle);
          dfx = combined_dfnx(theta,numintervals,treewt,firstlong,lastlong,
             &dfxplus,&dfxzero,cycle);
       }
       if (dfxplus < 0) change = fxplus * exp(fx - dfx);
       else change = fxplus * theta/2.0;

       newtheta = theta + change;

      /* now deal with negative or zero theta values */
       numloop = 1;
       while (newtheta <= 0) {
          numloop++;
          newtheta = theta + ldexp(change,-numloop);
       }
       newlike = interp(newtheta,firstlong,lastlong,lthetai,cycle);
       if(newlike < oldlike) {
      /* in case we overshoot the maximum, don't jump so far...*/
          numloop = 1;
          while(1) {
             numloop++;
             newtheta = theta + ldexp(change,-numloop);
             if (newtheta <= 0) continue;
             newlike = interp(newtheta,firstlong,lastlong,lthetai,cycle);
             if(newlike >= oldlike) break;
          }
       }
  
       oldlike = newlike;
       if(fabs(newtheta - theta) < epsilon) {
          theta = newtheta;
          break;
       }
       theta = newtheta;
       i++;
    }

    if (newlike > bestlike) {
       besttheta = theta;
       bestlike = oldlike;
    }
  }
  if (rend) {
    theti[cycle][numchains] = besttheta;
    lntheti[cycle][numchains] = log(besttheta);
    fprintf(outfile, 
      "Combined point estimate of theta = %12.8f, lnL = %12.8f\n", besttheta,
       bestlike);
    if (thetaout) fprintf(thetafile, "  final combined: %12.8f\n", besttheta);
  } else
    if (thetaout) fprintf(thetafile, " chain %ld combined: %12.8f\n",
      numchains, besttheta);
  free(treewt[firstlong]);
  free(treewt);
  free(tempwt);
}  /* thetapoint2 */

double locus_fnx(double theta, long numintervals, double ***treewt,
   long firstlong, long lastlong, boolean *return_zero, long *return_plus)
{
long lowcus, dfxplus;
double *fntheta, *dfntheta, temp1, temp2;
boolean dfxzero;

fntheta = (double *)calloc(1,numsets * sizeof(double));
dfntheta = (double *)calloc(1,numsets * sizeof(double));

temp1 = 0.0;
temp2 = 0.0;

for(lowcus = 0; lowcus < numsets; lowcus++) {
   fntheta[lowcus] = 
      interp(theta,firstlong,lastlong,savethetai[lowcus],lowcus);
   dfntheta[lowcus] = 
      combined_fnx(theta,numintervals,treewt[lowcus],firstlong,
      lastlong,&dfxplus,&dfxzero,lowcus);
   temp1 += fntheta[lowcus];
   if ((dfntheta[lowcus]-fntheta[lowcus]) > EXPMIN)
      temp2 += dfxplus * exp(dfntheta[lowcus]-fntheta[lowcus]);
}

*return_plus = 1;
if (temp2 < 0) *return_plus = -1;

*return_zero = false;
if (temp2 == 0.0) *return_zero = true;

temp2 = log(fabs(temp2));

free(fntheta);
free(dfntheta);
return (temp1 + temp2);

} /* locus_fnx */

double locus_dfnx(double theta, long numintervals, double ***treewt,
   long firstlong, long lastlong, boolean *return_zero, long *return_plus)
{
long lowcus, dfxplus, ddfxplus;
double *fntheta, *dfntheta, *idfntheta, *ddfntheta, temp1, temp2, 
   temp3, temp4;
boolean dfxzero, ddfxzero;

fntheta = (double *)calloc(1,numsets * sizeof(double));
dfntheta = (double *)calloc(1,numsets * sizeof(double));
idfntheta = (double *)calloc(1,numsets * sizeof(double));
ddfntheta = (double *)calloc(1,numsets * sizeof(double));

temp1 = 0.0;
temp2 = 0.0;
temp3 = 0.0;
temp4 = 0.0;

for(lowcus = 0; lowcus < numsets; lowcus++) {
   fntheta[lowcus] = 
      interp(theta,firstlong,lastlong,savethetai[lowcus],lowcus);
   dfntheta[lowcus] = 
      combined_fnx(theta,numintervals,treewt[lowcus],firstlong,
      lastlong,&dfxplus,&dfxzero,lowcus);
/* this is from the generalized derivative of an inverse function */
   idfntheta[lowcus] =
      -1.0/(fntheta[lowcus]*fntheta[lowcus]) * dfntheta[lowcus];
   ddfntheta[lowcus] = 
      combined_dfnx(theta,numintervals,treewt[lowcus],firstlong,
      lastlong,&ddfxplus,&ddfxzero,lowcus);
   temp1 += fntheta[lowcus];
   if ((dfntheta[lowcus]-fntheta[lowcus]) > EXPMIN)
      temp2 += dfxplus * exp(dfntheta[lowcus]-fntheta[lowcus]);
   if ((ddfntheta[lowcus]-fntheta[lowcus]) > EXPMIN)
      temp3 += ddfxplus * exp(ddfntheta[lowcus]-fntheta[lowcus]);
   /* the next ratio is obligatorily negative on working out */
   if ((dfntheta[lowcus]-idfntheta[lowcus]) > EXPMIN)
      temp4 -= exp(dfntheta[lowcus]-idfntheta[lowcus]);
}

temp2 *= temp2;
temp2 = temp2 + temp3 + temp4;

*return_plus = 1;
if (temp2 < 0) *return_plus = -1;

*return_zero = false;
if (temp2 == 0.0) *return_zero = true;

temp2 = log(fabs(temp2));

free(fntheta);
free(dfntheta);
free(idfntheta);
free(ddfntheta);
return (temp1 + temp2);

} /* locus_dfnx */

void weightcalc(double **treewt, long firstlong, long lastlong, long
   lowcus, double *lthetai)
{
long i, dataset, trii, refchain;
double maxwt, *tempwt;

tempwt = (double *)calloc(1,(totchains + 1) * sizeof(double));

for (dataset = firstlong; dataset <= lastlong; dataset++) {
   refchain = REF_CHAIN(dataset);
   for (trii = 0; trii < numout[1]; trii++) {
      maxwt = NEGMAX;
      treewt[dataset][trii] = 0.0;
      for (i = firstlong; i <= lastlong; i++) {
         tempwt[i] =  log((double)numout[1]) - 
         (double)(sum[lowcus][refchain][trii]) / theti[lowcus][i] -
         lthetai[i] + (1 - numsp) * lntheti[lowcus][i];
         if (tempwt[i] > maxwt) maxwt = tempwt[i];
      }
      for (i = firstlong; i <= lastlong; i++)
        if (tempwt[i] - maxwt > EXPMIN)
           treewt[dataset][trii] += exp(tempwt[i] - maxwt);
      treewt[dataset][trii] = -(log(treewt[dataset][trii]) + maxwt);
   }
}

free(tempwt);
} /* weightcalc */

double sum_interp(double theta, long firstlong, long lastlong,
   double **lthetai)
{
long lowcus;
double temp;

temp = 0.0;
for (lowcus = 0; lowcus < numsets; lowcus++)
   temp += interp(theta,firstlong,lastlong,lthetai[lowcus],lowcus);

return(temp);

} /* sum_interp */

void locuspoints()
{
int numloop; /* type "int" because of library function demands! */
long i, j, lowcus, dataset, refchain, firstlong, lastlong, numintervals,
   fxplus, dfxplus, chosen;
double ***treewt, newtheta, theta, oldlike, newlike, besttheta, 
   bestlike, fx, dfx, change;
boolean fxzero, dfxzero;

firstlong = chains[0];
lastlong = chains[0] + chains[1] - 1;
numintervals = numsp - 1; /* WARNING: wrong for recombination!!! */

treewt = (double ***)calloc(1,numsets * sizeof(double **));
treewt[0] = (double **)calloc(1,numsets*totchains * sizeof(double *));
for(i = 1; i < numsets; i++)
   treewt[i] = treewt[0] + i*totchains;
treewt[0][0] = (double *)calloc(1,numsets*totchains*numtrees*sizeof(double));
for(i = 0; i < numsets; i++)
   for(j = 0; j < totchains; j++)
      treewt[i][j] = treewt[0][0] + i*totchains*numtrees + j*numtrees;

/* first calculate the weighting factor for each tree */
for (lowcus = 0; lowcus < numsets; lowcus++)
   weightcalc(treewt[lowcus],firstlong,lastlong,lowcus,savethetai[lowcus]);
   
besttheta = NEGMAX;
bestlike = NEGMAX;

for (chosen = 0; chosen < numsets; chosen++) {
  theta = theti[chosen][totchains];
  oldlike = sum_interp(theta,firstlong,lastlong,savethetai); 
  i = 0;

  while (1) {
     fx =
     locus_fnx(theta,numintervals,treewt,firstlong,lastlong,&fxzero,&fxplus);
     dfx = 
     locus_dfnx(theta,numintervals,treewt,firstlong,lastlong,&dfxzero,&dfxplus);

     if (fxzero) {/* found a maximum! at theta! */
        newlike = oldlike;
        break;
     }
     if (dfxzero) {
        theta += epsilon;
        fx =
    locus_fnx(theta,numintervals,treewt,firstlong,lastlong,&fxzero,&fxplus);
        dfx = 
  locus_dfnx(theta,numintervals,treewt,firstlong,lastlong,&dfxzero,&dfxplus);
     }
     if (dfxplus < 0) change = fxplus * exp(fx - dfx);
     else change = fxplus * theta/2.0;

     newtheta = theta + change;

    /* now deal with negative or zero theta values */
     numloop = 1;
     while (newtheta <= 0) {
        numloop++;
        newtheta = theta + ldexp(change,-numloop);
     }
     newlike = sum_interp(newtheta,firstlong,lastlong,savethetai);
     if(newlike < oldlike) {
    /* in case we overshoot the maximum, don't jump so far...*/
        numloop = 1;
        while(1) {
           numloop++;
           newtheta = theta + ldexp(change,-numloop);
           if (newtheta <= 0) continue;
           newlike = sum_interp(newtheta,firstlong,lastlong,savethetai);
           if(newlike >= oldlike) break;
        }
     }

     oldlike = newlike;
     if(fabs(newtheta - theta) < epsilon) {
        theta = newtheta;
        break;
     }
     theta = newtheta;
     i++;
  }

  if (newlike > bestlike) {
     besttheta = theta;
     bestlike = oldlike;
  }
}

fprintf(outfile,"Point estimate using %ld loci = %12.9f,",
        numsets,besttheta);
fprintf(outfile,"   with lnL = %12.9f\n",bestlike);

free(treewt[0][0]);
free(treewt[0]);
free(treewt);

} /* locuspoints */
   
void coaliter(long lastlong, boolean chend, boolean rend)
{
/* Local variables for coaliter: */
  long i, j, p, firstlong, numchains, extravalues, *sorted;
  double *lthetai, *newth;
  boolean found, alldone;

  /* allocate local arrays */
  firstlong = chains[0];
  numchains = lastlong + 1;
  lthetai   = (double *)calloc(1,(numchains + numfix+1) * sizeof(double));
  sorted = (long *)calloc(1,(numchains + numfix+1)*sizeof(long));
  newth  = (double *)calloc(1,numchains*sizeof(double));
  
  /* evaluate long chains, using data from last */
  for (i = firstlong; i <= lastlong; i++) 
    thetaval(theti[cycle][i], lastlong, &lthetai[i]);

  /* solve for lthetai by Thompson-Geyer joint estimate method */
  for (i = 0; i < iters; i++) {  /* iterate until pleased */
    alldone = true;
    for (j = firstlong; j <= lastlong; j++) {
      newth[j] = interp(theti[cycle][j],firstlong,lastlong,lthetai,cycle);
      if (fabs(newth[j]-lthetai[j]) > epsilon) alldone = false;
    }
    for (j = firstlong; j <= lastlong; j++)
      lthetai[j] = newth[j];
    if (alldone) break;
  } 
  /* point estimates */
  thetapoint(lastlong, chend, rend);
  thetapoint2(firstlong, lastlong, lthetai, rend);
  /* sorted table of theti */
  if (rend) {
    fprintf(outfile,
	    "\n There were %3ld short runs; each producing %8.1f trees\n",
	    chains[0], (double)steps[0] / increm[0]);
    fprintf(outfile,
	    " There were %3ld long runs; each producing %8.1f trees\n",
	    chains[1], (double)steps[1] / increm[1]);
    fprintf(outfile, "\n   Theta       LnL\n");
    fprintf(outfile, "   -----       ---\n");

    lthetai[numchains] = 
       interp(theti[cycle][numchains],firstlong,lastlong,lthetai,cycle);
    locuslike[cycle] = lthetai[numchains];
    for(i = firstlong; i <= lastlong; i++)
       savethetai[cycle][i] = lthetai[i];
    extravalues = 0;
    for (i = 0; i < numfix; i++) {
      found = false;
      for (j = firstlong; j <= lastlong; j++) {
	if (fixed[i] == theti[cycle][j]) {
	  found = true;
        }
      }
      if (!found) {
	extravalues++;
	theti[cycle][numchains + extravalues] = fixed[i];
        lntheti[cycle][numchains + extravalues] = log(fixed[i]);
        lthetai[numchains + extravalues] = 
           interp(theti[cycle][numchains+extravalues],
                  firstlong,lastlong,lthetai,cycle);
        }
      }
    for (i = firstlong; i <= lastlong + extravalues+1; i++) {
      sorted[i] = i;
      }
    for (i = firstlong; i <= lastlong + extravalues+1; i++) {
      for (j = firstlong + 1; j <= lastlong + extravalues+1; j++) {
	if (theti[cycle][sorted[j - 1]] > theti[cycle][sorted[j]]) {
	  p = sorted[j - 1];
	  sorted[j - 1] = sorted[j];
	  sorted[j] = p;
	}
      }
    }
    for (i = firstlong; i <= lastlong + extravalues+1; i++) {
      fprintf(outfile, "%12.8f  %12.8f\n",theti[cycle][sorted[i]],
	      lthetai[sorted[i]]);
      }
    putc('\n', outfile);
  }
  free(lthetai);
  free(newth);
  free(sorted);
}  /* coaliter */

void liketable()
/* the multi-locus LnLike curve constructor. */
{
  long i, j, temp, firstlong, lastlong, numpoints, *sorted;
  double *printth;

  /* allocate local arrays */
  firstlong = chains[0];
  lastlong = totchains - 1;
  numpoints = numsets + numfix;
  sorted = (long *)calloc(1,numpoints * sizeof(long));
  printth = (double *)calloc(1,numpoints * sizeof(double));

  fprintf(outfile,"Combined likelihood over all loci\n\n");
  for(i = 0; i < numsets; i++) printth[i] = theti[i][totchains];
  for(i = 0; i < numfix; i++) printth[i+numsets] = fixed[i];
  
  /* first calculate point estimate */
  locuspoints();

  /* first calculated summed Lnlike for each locus' final estimate of
     theta */
  for (i = 0; i < numsets; i++)
     for(j = 0; j < numsets; j++)
        if (i != j)
           locuslike[i] += interp(theti[i][totchains],firstlong,
              lastlong,savethetai[j],j);
  /* now calculate each "fixed" point's Lnlike for each chain, and
     sum that up */
  for(i = 0; i < numfix; i++)
     locuslike[numsets + i] = sum_interp(fixed[i],firstlong,lastlong,
        savethetai);

  /* sorted table of theti */
    for (i = 0; i < numpoints; i++) sorted[i] = i;

    for (i = 0; i < numpoints; i++)
      for (j = 1; j < numpoints; j++)
	if (printth[sorted[j - 1]] > printth[sorted[j]]) {
	  temp = sorted[j - 1];
	  sorted[j - 1] = sorted[j];
	  sorted[j] = temp;
	}

    fprintf(outfile, " There were %ld loci examined\n\n", numsets);
    fprintf(outfile, "   Theta       LnL\n");
    fprintf(outfile, "   -----       ---\n");

    for (i = 0; i < numpoints; i++) {
      fprintf(outfile, "%12.8f  %12.8f\n",printth[sorted[i]],
	      locuslike[sorted[i]]);
      }
    putc('\n', outfile);

  free(printth);
  free(sorted);

} /* liketable */

void maketree()
{
  long 		incrprog, tottrees, i, metout, thetout, progout;
  double 	bestlike;
  boolean 	chainend, runend;
  static char   *chainlit[] = {"Short","Long"};

  contribution = (contribarr *)calloc(1,sites*sizeof(contribarr));
  contribution[0] = (double *)calloc(1,sites*categs*sizeof(double));
  for (i=1;i<sites;i++)
     contribution[i] = contribution[0] + i*categs;

  inittable();
  initweightrat();
  getc(infile);
  fprintf(outfile, "Watterson estimate of theta is %12.8f\n", watttheta);
  if (usertree)
     treeread();
  else {
     branch0 = watttheta/numsp; 
     constructtree(numsp, branch0);
  }
  orient(curtree.root->back);
  finishsetup(curtree.root->back);
  initbranchlist();
  treevaluate();
  bestlike = NEGMAX;
  theti[cycle][0] = theta0;
  lntheti[cycle][0] = log(theta0);
  tottrees = 0;
  runend = false;
  /* We're going to start sampling thetas with tree 10, and resample after
     10 more trees have been outputted. */
  thetout = 10;
  /**********************************/
  /* Begin Hastings-Metropolis loop */
  /**********************************/
  for (apps = 0; apps < totchains; apps++) {
    if (apps >= chains[0]) chaintype = 1;
    else chaintype = 0;
    if (progress) {
      printf("%s chain %ld ",chainlit[chaintype],apps + 1);
      fflush(stdout);
    }
    metout = increm[chaintype] - 1;
    incrprog = (long)(steps[chaintype] / 10.0);
    progout = incrprog - 1;
    numout[chaintype] = 0;
    slacc = 0;
    slid = 0;
    chainend = false;
    for (indecks=0; indecks < steps[chaintype]; indecks++) {
      oldlikelihood = curtree.likelihood;
      col = 0; /* column number, used in treeout */
      slide();
      if (indecks == steps[chaintype] - 1) { /* end of chain? */
        chainend = true;
        if (apps == totchains - 1) /* end of run? */
          runend = true;
      }
      if (curtree.likelihood > bestlike) {
        if (onebestree) {
           fclose(bestree);
           bestree = fopen("bestree","w+");
	   fprintf(bestree, "Chain #%2ld (%s) Step:%8ld\n",apps+1, 
             chainlit[chaintype], indecks+1);
	   treeout(curtree.root->back, 1L, &bestree);
	   fprintf(bestree, " [%12.10f]\n", curtree.likelihood);
	   bestlike = curtree.likelihood;
        }
        else {
	   fprintf(bestree, "Chain #%2ld (%s) Step:%8ld\n",apps+1, 
             chainlit[chaintype], indecks+1);
	   treeout(curtree.root->back, 1L, &bestree);
	   fprintf(bestree, " [%12.10f]\n", curtree.likelihood);
	   bestlike = curtree.likelihood;
        }
      }
      if (indecks == metout) {
	numout[chaintype]++;
	tottrees++;
	scoretree(apps);
	metout += increm[chaintype];
        if(treeprint) treeout(curtree.root->back,1L,&treefile);
      }
      if (tottrees == thetout && thetaout) {
	thetout += 10;
        if (!chainend) {
          if (chaintype == 0 || apps == chains[0])  /* no joint estimate */
            thetapoint(apps,chainend,runend);
          else coaliter(apps,chainend,runend);
        }
      }
      if (progress && indecks == progout) {
	printf(".");
        fflush(stdout);
	progout += incrprog;
      }
    }
    if (chaintype == 0 || apps == chains[0]) 
      thetapoint(apps, chainend, runend);
    else coaliter(apps,chainend,runend);
    if (progress) printf("\nAccepted %ld/%ld rearrangements\n",slacc,slid);
  }
  if(slacc == 0) {
     fprintf(outfile,"WARNING--no proposed trees ever accepted\n");
     fprintf(ERRFILE,"WARNING--no proposed trees ever accepted\n");
  }
}  /* maketree */


long main(int argc, char *argv[])
{  /* Coalesce */
  long i;
  /* Open various filenames. */

  openfile(&infile,INFILE,"r",argv[0],NULL);
  if (!menu) 
      {
      openfile(&simlog,"simlog","w",argv[0],NULL);
      openfile(&seedfile,"seedfile","r",argv[0],NULL);
      }
  if (thetaout) openfile(&thetafile,"thetafile","w",argv[0],NULL);
  openfile(&outfile,OUTFILE,"w",argv[0],NULL);

  ibmpc = IBMCRT;
  ansi = ANSICRT;
  getoptions();
  for (i = 1; i <= 1000; i++)
    clearseed = randum(seed);
  if (usertree)
    openfile(&intree,INTREE,"r",argv[0],NULL);
  if (treeprint)   
    openfile(&treefile,TREEFILE,"w",argv[0],NULL);      
  openfile(&bestree,"bestree","w",argv[0],NULL);      
  firstinit();
  for (cycle = 0; cycle < numsets; cycle++) {
     if (progress) printf("Locus %ld\n",cycle+1);
     fprintf(outfile, "Locus %ld\n",cycle+1);
     locusinit();
     getinput();
     watttheta = watterson();
     if (watt)
       theta0 = watttheta;
     maketree();
     freetree();
  }
  if (numsets>1) liketable();
  FClose(infile);
  FClose(outfile);
  FClose(treefile);
  FClose(bestree);
  FClose(seedfile);
  FClose(simlog);
  FClose(parmfile);
  FClose(thetafile);
  exit(0);
}  /* Coalesce */

int eof(FILE *f)
{
    register int ch;

    if (feof(f))
        return 1;
    if (f == stdin)
        return 0;
    ch = getc(f);
    if (ch == EOF)
        return 1;
    ungetc(ch, f);
    return 0;
} /* eof */

int eoln(FILE *f)
{
  register int ch;
  
  ch = getc(f);
  if (ch == EOF)
    return 1;
  ungetc(ch, f);
  return (ch == '\n');
} /* eoln */

void memerror()
{
printf("Error allocating memory\n");
exit(-1);
} /* memerror */

MALLOCRETURN *mymalloc(long x)
{
  MALLOCRETURN *mem;
  mem = (MALLOCRETURN *)calloc(1,x);
  if (!mem)
    memerror();
  else
    return (MALLOCRETURN *)mem;
} /* MALLOCRETURN */
