/*
    Copyright 2004-2005 Brian Smith (brian@smittyware.com)
    This file is part of CM2GPX.

    CM2GPX is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    CM2GPX is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with CM2GPX; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "common.h"
#include "pdbreader.h"
#include "gpxwriter.h"
#include "getopt.h"

static string sInputPath, sOutputPath;
static int bToStdout, bShowVer, bQuietMode, bFoundFilt, bBkmkFilt;
static int bFudgeTZ, bExactMatch, bKeepName, bCategInType;
static string sCategFilt, sDefSymbol, sDefType;

string GetDefaultOutputPath(string sPath, string sName)
{
	int nSlash;
	string sOutPath;

	nSlash = sPath.rfind(PATH_SEP);
	if (nSlash != string::npos)
		sOutPath = sPath.substr(0, nSlash+1);

	sOutPath += sName + "-Out.gpx";
	return sOutPath;
}

int ParseCommandLine(int argc, char **argv)
{
	int errflg = 0;
	extern int optind;
	extern char *optarg;
	int c;

	bToStdout = 0;
	bShowVer = 0;
	bQuietMode = 0;
	bFoundFilt = 0;
	bBkmkFilt = 0;
	bFudgeTZ = 0;
	bExactMatch = 0;
	bKeepName = 0;
	bCategInType = 0;

	sInputPath.erase();
	sOutputPath.erase();
	sCategFilt.erase();
	sDefSymbol = "Waypoint";
	sDefType = "Waypoint";

	if (argc < 2)
		return 0;

	while (1)
	{
		c = getopt(argc, argv, "bcC:efno:qs:t:Tvz");
		if (c == -1)
			break;

		switch (c)
		{
		case 'b':	bBkmkFilt = 1; break;
		case 'c':	bToStdout = 1; break;
		case 'C':	sCategFilt = optarg; break;
		case 'e':	bExactMatch = 1; break;
		case 'f':	bFoundFilt = 1; break;
		case 'n':	bKeepName = 1; break;
		case 'o':	sOutputPath = optarg; break;
		case 'q':	bQuietMode = 1; break;
		case 's':	sDefSymbol = optarg; break;
		case 't':	sDefType = optarg; break;
		case 'T':	bCategInType = 1; break;
		case 'v':	bShowVer = 1; break;
		case 'z':	bFudgeTZ = 1; break;
		case '?':	errflg = 1; break;
		}
	}

	if (bShowVer)
		return 1;
	if ((optind >= argc) || errflg)
		return 0;

	// Output to stdout implies quiet mode (no corrupted output)
	if (bToStdout)
		bQuietMode = 1;

	sInputPath = argv[optind];
	return 1;
}

int PrintVersion()
{
	printf(PACKAGE_STRING
		" -- Copyright (C) 2004-2005 Brian Smith\n");
	return 0;
}

int PrintUsage(char *szExe)
{
	printf(
"Usage: %s [-b] [-c] [-C category_list] [-e] [-f] [-n] [-o output_file]\n"
	"\t[-q] [-s symbol] [-t wpt_type] [-T] [-v] [-z] input_file\n",
		szExe);
	return 2;
}

int CheckPDBStructSize()
{
	int err = 0;

	if (sizeof(PDBHeader) != 80)
		err = 1;
	if (sizeof(PDBRecordEntry) != 8)
		err = 1;

	if (err)
	{
		printf("PDB structure sizes incorrect!  "
			"Conversion aborted.\n");
		printf("Header size = %d (must be 80)\n",
			sizeof(PDBHeader));
		printf("Record entry size = %d (must be 8)\n",
			sizeof(PDBRecordEntry));
	}

	return !err;
}

int main(int argc, char **argv)
{
	if (!CheckPDBStructSize())
		return 1;

	if (!ParseCommandLine(argc, argv))
	{
		PrintVersion();
		return PrintUsage(argv[0]);
	}
	if (bShowVer)
		return PrintVersion();

	CPDBReader reader;
	reader.m_bFoundFilt = bFoundFilt;
	reader.m_bBkmkFilt = bBkmkFilt;
	reader.m_sCategFilt = sCategFilt;
	reader.m_bExactMatch = bExactMatch;
	if (!reader.ReadFile(sInputPath, bQuietMode))
	{
		printf("Error reading PDB file\n");
		return 1;
	}

	if (!bQuietMode)
	{
		int n = reader.m_entries.size();
		printf("%d record%s read from database %s\n", n,
			(n == 1) ? "" : "s", reader.m_sDBName.c_str());
	}

	if (sOutputPath.empty())
	{
		sOutputPath = GetDefaultOutputPath(sInputPath,
			reader.m_sDBName);
	}

	if (!bQuietMode)
		printf("Writing GPX file to %s\n", sOutputPath.c_str());

	CGPXWriter writer;
	writer.m_bToStdout = bToStdout;
	writer.m_bFudgeTZ = bFudgeTZ;
	writer.m_bKeepName = bKeepName;
	writer.m_bCategInType = bCategInType;
	writer.m_sDefSymbol = sDefSymbol;
	writer.m_sDefType = sDefType;
	
	if (!writer.WriteGPX(sOutputPath, reader.m_entries))
	{
		printf("Error writing GPX file.\n");
		return 1;
	}

	return 0;
}

