/*
 * CHEST, chess analyst.  For Copyright notice read file "COPYRIGHT".
 *
 * $Source: /home/heiner/ca/chest/RCS/output.c,v $
 * $Id: output.c,v 3.16 1999/10/17 22:08:55 heiner Exp $
 *
 *	Print moves and the like
 */

#include "types.h"
#include "board.h"
#include <stdio.h>
#include "lang.h"
#include "mgsubr.h"		/* mv_ck_attr() */
#include "move.h"		/* move_execute() */
#include "move_gen.h"		/* detect mate moves */
#include "output.h"

static int	out__lang	= DFT_LANG;


    Eximpl void
oflush(void)
{
    (void) fflush(stdout);
}


    Eximpl void
set_out_lang( int lang )
{
    if( is_lang(lang) ) {
	out__lang = lang;
    }
}

/*===========================================================================*/
/*
 * Helper functions
 */

    Eximpl char
chr_colour( Colour c )
{
    return lang_chr_colour(out__lang, c);
}

    Eximpl char
chr_Colour( Colour c )
{
    return lang_chr_Colour(out__lang, c);
}

    Eximpl char
chr_figure( Figure f )
{
    return lang_chr_figure(out__lang, f);
}

    Eximpl char
chr_Figure( Figure f )
{
    return lang_chr_Figure(out__lang, f);
}


    Eximpl int
chr_PosCol( Position pos )
{
    if( ! no_pos(pos) ) {
	switch( COL(pos) ) {
	 case 0:	return 'a';
	 case 1:	return 'b';
	 case 2:	return 'c';
	 case 3:	return 'd';
	 case 4:	return 'e';
	 case 5:	return 'f';
	 case 6:	return 'g';
	 case 7:	return 'h';
	}
    }
    return '?';
}


    Eximpl int
chr_PosLin( Position pos )
{
    if( ! no_pos(pos) ) {
	switch( LIN(pos) ) {
	 case 0:	return '1';
	 case 1:	return '2';
	 case 2:	return '3';
	 case 3:	return '4';
	 case 4:	return '5';
	 case 5:	return '6';
	 case 6:	return '7';
	 case 7:	return '8';
	}
    }
    return '?';
}

/*===========================================================================*/
/*
 * High level output (move conversion)
 */

#define AMBIG_UNIQ	04	/* no ambiguity present */
#define AMBIG_LINOK	02	/* line   would disambiguate */
#define AMBIG_COLOK	01	/* column would disambiguate */

    static int			/* ambiguity flags */
mvc_ambig_flags(
    register Xconst Board*	bp,
    register const Field*	fp,
    register Xconst Field*	tp,
    int				fpos )
{
    /*
     * Search for other (not fp->f_idx) of the same colour & figure,
     * which have a direct attack into tp.
     */
    register rPieceSet		atts;
    register int		idx;
    register int		zeroes;
    register int		aflags;
    register rPosition		xpos;

    aflags = AMBIG_UNIQ | AMBIG_LINOK | AMBIG_COLOK;
    atts   = F_DATT(tp) & ~SET1(fp->f_idx) & COLOUR_MASK(fp->f_c);
    for( idx=0 ; atts ; ++idx, atts>>=1 ) {
	while( ! (atts & 01) ) {
	    idx += (zeroes = MIN_LOW_ZERO(atts));
	    atts >>= zeroes;
	}
	xpos = bp->b_piece[idx];
	if( !no_pos(xpos) && (bp->b_f[xpos].f_f == fp->f_f) ) {
	    aflags &= ~AMBIG_UNIQ;		/* need to disambiguate */
	    if( COL(xpos) == COL(fpos) ) {
		aflags &= ~AMBIG_COLOK;		/* column not unique */
	    }
	    if( LIN(xpos) == LIN(fpos) ) {
		aflags &= ~AMBIG_LINOK;		/* line not unique */
	    }
	}
    }
    return aflags;
}


#if 0
/*
 * cnv_Mov()
 *	In the context of a board, create a readable string for the
 *	specified Move, using standard short notation.
 *	This is the shortest way to print a Move correctly.
 *	Mainly the figure type and destination field is given.
 *	- normal move:		Te8	Te8+
 *	- normal beat:		Te8:	Te8:+
 *	- B move:		e4	e4+
 *	- B beat:		ed5	ed5+
 *	- promotion move:	e8D	e8D+
 *	- promotion beat:	de8D	de8D+
 *	- e.p. beat:		de5ep	de5ep+
 *	- short castling:	O-O	O-O+
 *	- long castling:	O-O-0	O-O-O+
 *	Such a notation is considered ambiguous if there is more than one
 *	pseudo-legal (!!) move with that figure type and destination.
 *	Hence, there cannot be ambigous pawn moves or castlings.
 *	Only normal moves and beats may be ambigous.
 *	In most cases this can be resolved by specifying the source column:
 *	- column extended move	Tee8	Tee8+
 *	- column extended beat	Tee8:	Tee8:+
 *	Or, the source line may be unique:
 *	- line extended move	T8e8	T8e8+
 *	- line extended beat	T8e8:	T8e8:+
 *	By theory it is possible to have multiple queens for which neither
 *	is unique.  Complete source is needed.  In this case the beat
 *	indicator ':' is left away, here, to be limited to 6 chars:
 *	- total extended move	De6d5	De6d5+
 *	- total extended beat	De6d5	De6d5+
 *	All interesting other pseudo-legal moves can easily be found
 *	via the direct attacks in the target field.
 */
    Eximpl const char*		/* points to static result data */
cnv_Mov( 
    register Xconst Board*	bp,
    register const Move*	mp)
{
    static char			buf[6+1];
    register char*		p;
    int				fpos;
    int				tpos;
    register rFigure		fig;
    register const Field*	fp;
    register Xconst Field*	tp;
    int				aflags;

    p = buf;
    if( no_pos(mp->m_from) ) {		/* denotes "no move" */
	*p++ = '-'; *p++ = '-';
	goto out;
    }
    fp = &(bp->b_f[fpos = mp->m_from]);
    tp = &(bp->b_f[tpos = mp->m_to  ]);
    fig = fp->f_f;
    if( fig == bauer ) {		/* some B is moving */
	if( COL(fpos) == COL(tpos) ) {	/* same column: move */
	    *p++ = chr_PosCol(tpos);
	    *p++ = chr_PosLin(tpos);
	    if( mp->m_fig != bauer ) {	/* promotion */
		*p++ = chr_Figure(mp->m_fig);
	    }
	}else {				/* different column: beat */
	    *p++ = chr_PosCol(fpos);
	    *p++ = chr_PosCol(tpos);
	    *p++ = chr_PosLin(tpos);
	    if( tp->f_c == empty ) {	/* e.p. */
		*p++ = 'e';
		*p++ = 'p';
	    }else if( mp->m_fig != bauer ) {
		*p++ = chr_Figure(mp->m_fig);
	    }
	}
    }else {				/* not a B move */
	if( fig == koenig ) {
	    switch( tpos - fpos ) {
	     case MK_DELTA(-2,0):	/* long castling */
		*p++ = 'O'; *p++ = '-';
		/*FALLTHROUGH*/
	     case MK_DELTA( 2,0):	/* short castling */
		*p++ = 'O'; *p++ = '-'; *p++ = 'O';
		goto chk;
	    }
	}
	*p++ = chr_Figure((Figure)fig);
	aflags = mvc_ambig_flags(bp, fp, tp, fpos);
	switch( aflags ) {
	 case AMBIG_COLOK | AMBIG_LINOK:	/* both can be used */
	 case AMBIG_COLOK:			/* column is unique */
	    *p++ = chr_PosCol(fpos);
	    break;
	 case 00:				/* need complete source */
	    *p++ = chr_PosCol(fpos);
	    /*FALLTHROUGH*/
	 case AMBIG_LINOK:			/* line is unique */
	    *p++ = chr_PosLin(fpos);
	    /*FALLTHROUGH*/
	 default:				/* not ambiguous */
	    break;
	}
	*p++ = chr_PosCol(tpos);
	*p++ = chr_PosLin(tpos);
	if( aflags && (tp->f_c != empty) ) {
	    *p++ = ':';
	}
    }
chk:;					/* append a '+' for checking move */
    if( mp->m_attr & MA__CK ) {
	*p++ = '+';
    }
out:;
    *p = '\0';				/* terminate the string */
    return buf;
}
#endif


/*
 * In order to be SAN compatible, we invent a lot of options, and use
 * a general function with flags.
 * Longer than 6 chars can appear with:
 *	Qe6xd5+
 *	exd8=Q+
 *	exd6ep+
 * Fields:
 *	 1 piece type		0..1
 *	 2 source col		0..1
 *	 3 source row		0..1
 *	 4 capture infix	0..1
 *	 5 target col		1
 *	 6 target row		1
 *	 7 capture suffix	0..1
 *	 8 e.p. suffix		0..2
 *	 9 promotion indicator	0..1
 *	10 promotion piece	0..1
 *	11 check indicator	0..1
 * or
 *	 0 castling		3..5
 *	11 check indicator	0..1
 * Check indicator is computed, first (needed always).
 */
#define MVCO_CAPT_X	(1<< 0)	/* use "x" (not ":") */
#define MVCO_CAPT_INFIX	(1<< 1)	/* capture indicator is infix (not postfix) */
#define MVCO_CAPT_PAWN	(1<< 2)	/* capture indicator even for pawns */
#define MVCO_MATE	(1<< 3)	/* check mate indicator "#" (not "+") */
#define MVCO_CHK_SELF	(1<< 4)	/* do not rely on check-bits in move */
#define MVCO_PROM_EQ	(1<< 5)	/* promotion indicator "=" (not "") */
#define MVCO_LEN_7	(1<< 6)	/* allow extended length 7 (not 6) */
#define MVCO_EP_NONE	(1<< 7)	/* e.p. indicator "" (not "ep") */
#define MVCO_EP_CAPT	(1<< 8)	/* e.p. capture indicator "*" (not ":") */
#define MVCO_L6D_EP	(1<< 9)	/* for length 6 drop e.p. indicator */
#define MVCO_L6D_PROM	(1<<10)	/* for length 6 drop promotion indicator */
#define MVCO_L6D_CAPT	(1<<11)	/* for length 6 drop capture indicator */
#define MVCO_L6D_CHK	(1<<12)	/* for length 6 drop check indicator */
#define MVCO_LANG_ENG	(1<<13)	/* use English (not output language) */

#define MVCO_SAN	(  MVCO_CAPT_X		\
			 | MVCO_CAPT_INFIX	\
			 | MVCO_CAPT_PAWN	\
			 | MVCO_CHK_SELF/*FFS*/	\
			 | MVCO_MATE		\
			 | MVCO_PROM_EQ		\
			 | MVCO_LEN_7		\
			 | MVCO_L6D_EP		\
			 | MVCO_L6D_PROM	\
			 | MVCO_L6D_CAPT	\
			 | MVCO_L6D_CHK		\
			 | MVCO_LANG_ENG	\
			)

#define MVCO_SAN6	(  MVCO_SAN & ~MVCO_LEN_7 )

#define MVCO_STD	(  MVCO_L6D_EP		\
			 | MVCO_L6D_PROM	\
			 | MVCO_L6D_CAPT	\
			 | MVCO_L6D_CHK		\
			)

#define MVFX_FIG	 0	/* move conversion field index for fig */
#define MVFX_SRC_COL	 1
#define MVFX_SRC_LIN	 2
#define MVFX_CAPT_INF	 3
#define MVFX_DST_COL	 4
#define MVFX_DST_LIN	 5
#define MVFX_CAPT_SUF	 6
#define MVFX_EP		 7
#define MVFX_EP2	 8
#define MVFX_PROM_IND	 9
#define MVFX_PROM_FIG	10
#define MVFX_CHECK	11
#define MVFX_COUNT	12	/* #(move conversion fields) */

typedef struct MvConv	MvConv;

struct MvConv
{
    char	fld[ MVFX_COUNT ];
};

    static void
mvc_clear( register MvConv* p )
{
    register int	x;

    for( x=0 ; x < MVFX_COUNT ; ++x ) {
	p->fld[x] = 0;
    }
}

    static void
mvc_string( register MvConv* p, register const char* str )
{
    register int	x;

    for( x=0 ; x < MVFX_COUNT ; ++x ) {
	if( ! str[x] ) break;
	p->fld[x] = str[x];
    }
}

    static char
mvc_Fig( Figure f, int flags )
{
    if( flags & MVCO_LANG_ENG ) {
	return lang_chr_Figure(LANG_ENGLISH, f);
    }
    return chr_Figure(f);
}

    static int				/* copied length */
mvc_copy( register const MvConv* ip, register char* op )
{
    register int	ix;
    register int	ox = 0;

    for( ix=0 ; ix < MVFX_COUNT ; ++ix ) {
	if( ip->fld[ix] ) {
	    op[ox++] = ip->fld[ix];
	}
    }
    return ox;
}

    static char*		/* echo buf */
mvc_copy_term( MvConv* p, char* buf, int flags )
{
    int		len;
    int		oklen;

    oklen = ((flags & MVCO_LEN_7) ? 7 : 6);
    while( (len = mvc_copy(p, buf)) > oklen ) {
	if( (flags & MVCO_L6D_EP) && (p->fld[MVFX_EP] || p->fld[MVFX_EP2]) ) {
	    p->fld[MVFX_EP ] = 0;
	    p->fld[MVFX_EP2] = 0;
	    continue;
	}
#define MV_redu(flag, inx)	if( (flags & (flag)) && p->fld[inx] ) { \
				    p->fld[inx] = 0; continue;		 \
				}
	MV_redu( MVCO_L6D_PROM, MVFX_PROM_IND )
	MV_redu( MVCO_L6D_CAPT, MVFX_CAPT_INF )
	MV_redu( MVCO_L6D_CAPT, MVFX_CAPT_SUF )
	MV_redu( MVCO_L6D_CHK , MVFX_CHECK    )
#undef MV_redu
	len = oklen;		/* effectively forces truncation */
	break;
    }
    buf[len] = 0;		/* terminate output buffer */
    return buf;
}

    const Bool
mvc_is_mate( Board* bp, const Move* mp )
{
    /*
     * We know already, that it is a check move.
     * We have no directly applicable function.
     * Naively, we execute the move, call the move generator,
     * and look whether any legal move is generated.
     */
    int		fsave;		/* caller not expecting move tracing */
    Bool	ismate;

    fsave = f_mvtrace; f_mvtrace = 0;

    move_execute(bp, mp);
    ismate = ! move_gen(bp, (Movelist*)0);
    move_undo(bp);

    f_mvtrace = fsave;
    return ismate;
}

    const char*			/* --> static result data */
mvc_conv( 
    register Board*		bp,
    register const Move*	mp,
    int				flags)		/* MVCO_* */
{
    static char			buf[ MVFX_COUNT + 1 ];	/* result buffer */
    MvConv			cnv;
    char			ci;		/* capture indicator */
    int				fpos;
    int				tpos;
    register rFigure		fig;
    register const Field*	fp;
    register Xconst Field*	tp;

    mvc_clear(&cnv);
    ci = 0;
    if( no_pos(mp->m_from) ) {			/* denotes "no move" */
	mvc_string(&cnv, "--");
	goto cpy;
    }
    fp = &(bp->b_f[fpos = mp->m_from]);
    tp = &(bp->b_f[tpos = mp->m_to  ]);
    fig = fp->f_f;
    if( fig == bauer ) {			/* some B is moving */
	cnv.fld[MVFX_DST_COL] = chr_PosCol(tpos);
	cnv.fld[MVFX_DST_LIN] = chr_PosLin(tpos);
	if( COL(fpos) != COL(tpos) ) {		/* different column: beat */
	    cnv.fld[MVFX_SRC_COL] = chr_PosCol(fpos);
	    if( flags & MVCO_CAPT_PAWN ) {
		ci = ((flags & MVCO_CAPT_X) ? 'x' : ':');
	    }
	    if( (tp->f_c == empty) && (mp->m_fig == bauer) ) {	/* e.p. */
		if( ! (flags & MVCO_EP_NONE) ) {
		    cnv.fld[MVFX_EP ] = 'e';
		    cnv.fld[MVFX_EP2] = 'p';
		}
		if( flags & MVCO_EP_CAPT ) ci = '*';
	    }
	}
	if( mp->m_fig != bauer ) {		/* promotion */
	    cnv.fld[MVFX_PROM_FIG] = mvc_Fig(mp->m_fig, flags);
	    cnv.fld[MVFX_PROM_IND] = ((flags & MVCO_PROM_EQ) ? '=' : 0);
	}
    }else {					/* not a B move */
	if( fig == koenig ) {
	    switch( tpos - fpos ) {
	     case MK_DELTA(-2,0):		/* long castling */
		mvc_string(&cnv, "O-O-O");
		goto chk;
	     case MK_DELTA( 2,0):		/* short castling */
		mvc_string(&cnv, "O-O");
		goto chk;
	    }
	}
	cnv.fld[MVFX_FIG    ] = mvc_Fig((Figure)fig, flags);
	cnv.fld[MVFX_DST_COL] = chr_PosCol(tpos);
	cnv.fld[MVFX_DST_LIN] = chr_PosLin(tpos);
	switch( mvc_ambig_flags(bp, fp, tp, fpos) ) {
	 case AMBIG_COLOK | AMBIG_LINOK:	/* both can be used */
	 case AMBIG_COLOK:			/* column is unique */
	    cnv.fld[MVFX_SRC_COL] = chr_PosCol(fpos);
	    break;
	 case 00:				/* need complete source */
	    cnv.fld[MVFX_SRC_COL] = chr_PosCol(fpos);
	    /*FALLTHROUGH*/
	 case AMBIG_LINOK:			/* line is unique */
	    cnv.fld[MVFX_SRC_LIN] = chr_PosLin(fpos);
	    /*FALLTHROUGH*/
	 default:				/* not ambiguous */
	    break;
	}
	if( tp->f_c != empty ) {
	    ci = ((flags & MVCO_CAPT_X) ? 'x' : ':');
	}
    }
    if( ci ) {
	cnv.fld[(flags&MVCO_CAPT_INFIX) ? MVFX_CAPT_INF : MVFX_CAPT_SUF] = ci;
    }
chk:;		/* generate check indicator ... */
    {
	    Move	mov;
	mov = *mp;
	if( flags & MVCO_CHK_SELF ) {
	    mv_ck_attr(bp, &mov);
	}
	if( mov.m_attr & MA__CK ) {
	    cnv.fld[MVFX_CHECK] = '+';
	    if( (flags & MVCO_MATE) && mvc_is_mate(bp, &mov) ) {
		cnv.fld[MVFX_CHECK] = '#';
	    }
	}
    }
cpy:;
    /*
     * Copy together, check permissable length and terminate buffer.
     */
    return mvc_copy_term(&cnv, buf, flags);
}

    Eximpl const char*		/* points to static result data */
mvc_L6( Board*	bp, const Move*	mp)
{
    int	flags = ((out__lang == LANG_ENGLISH) ? MVCO_SAN : MVCO_STD);
    return mvc_conv(bp, mp, flags & ~MVCO_LEN_7);
}

    Eximpl const char*		/* points to static result data */
mvc_L7( Board*	bp, const Move*	mp)
{
    int	flags = ((out__lang == LANG_ENGLISH) ? MVCO_SAN : MVCO_STD);
    return mvc_conv(bp, mp, flags |  MVCO_LEN_7);
}

    Eximpl const char*		/* points to static result data */
mvc_SAN( Board*	bp, const Move*	mp)
{
    return mvc_conv(bp, mp, MVCO_SAN);
}

/*===========================================================================*/
/*
 * Medium level output (just show data present)
 */

/*
 * Print one character representing a figure type.
 */
    Eximpl void
put_figure( Figure f )
{
    char	c;

    c = chr_Figure(f);
    if( c == '?' ) {
	printf("(%d)", f);
    }else {
	printf("%c", c);
    }
}


/*
 * Print the algebraic notation of a position.
 */
    Eximpl void
put_position( Position pos )
{
    printf("%c%c", chr_PosCol(pos), chr_PosLin(pos));
}


/*
 * put_piece()
 *	Print readable figure and position of a piece (given by its index).
 *	Returns the number of printed pieces (0 or 1).
 */
    static int			/* success */
put_piece( const Board* bp, int idx )
{
    Position	pos;

    if( (idx < 0) || (idx >= 2*MAX_PIECE) ) {
	pos = NO_POS;
    }else {
	pos = bp->b_piece[idx];
    }
    if( no_pos(pos) ) {
	return 0;
    }
    put_figure(bp->b_f[pos].f_f);
    put_position(pos);
    return 1;
}


/*
 * put_control()
 *	Print the control position of the board.
 */
    Eximpl void
put_control( register const Board* bp )
{
    register int	 idx;
    register int	 imax;
    register rColour	 c;
    register int	 cnt;

    c = white;
    do {
	cnt = 0;
	printf("%c: ", chr_Colour(c));
	idx = COLOUR_IDX(c);
	imax = idx + bp->b_max_piece[c];
	for( ; idx < imax ; ++idx ) {
	    printf(" ");
	    cnt += put_piece(bp, idx);
	}
	printf(" (%d)\n", cnt);
    }while( (c = opp_colour(c)) != white );
}


    static char
chr_fen_fig( Colour c, Figure f, int lang )
{
    return ((c == white) ? lang_chr_Figure : lang_chr_figure)(lang, f);
}

    static void
put_fen_fig( Colour c, Figure f, int lang )
{
    printf("%c", chr_fen_fig(c, f, lang));
}


    Eximpl char*		/* --> NUL behind filled */
app_fen_board( register const Board* bp, register char* buf, int lang )
{
    register int		col;
    register int		row;
    register int		ecnt;
    register const Field*	fp;

    ecnt = 0;
    for( row=7 ; row>=0 ; --row ) {
	if( row < 7 ) {
	    *buf++ = '/';
	}
	for( col=0 ; col<8 ; ++col ) {
	    fp = &(bp->b_f[ MK_POS(col,row) ]);
	    if( fp->f_c == empty ) {
		++ecnt;
	    }else {
		if( ecnt ) {
		    *buf++ = "012345678"[ecnt]; ecnt = 0;	/*FFS*/
		}
		*buf++ = chr_fen_fig(fp->f_c, fp->f_f, lang);
	    }
	}
	if( ecnt ) {
	    *buf++ = "012345678"[ecnt]; ecnt = 0;		/*FFS*/
	}
    }
    *buf = 0;
    return buf;
}

    Eximpl char*		/* --> NUL behind filled */
app_fen_tom( const Board* bp, char* buf, int lang )
{
    *buf++ = lang_chr_colour(lang, bp->b_tomove);
    *buf = 0;
    return buf;
}

    Eximpl char*		/* --> NUL behind filled */
app_fen_castle( const Board* bp, char* buf, int lang )
{
    if( (bp->b_castle[white] | bp->b_castle[black]) & (castle00 | castle000) ) {
# define AppCa(c,m,f)	\
		if( bp->b_castle[c] & (m) ) *buf++ = chr_fen_fig(c, f, lang);
	AppCa(white, castle00 , koenig);
	AppCa(white, castle000, dame  );
	AppCa(black, castle00 , koenig);
	AppCa(black, castle000, dame  );
# undef AppCa
    }else {
	*buf++ = '-';
    }
    *buf = 0;
    return buf;
}

    Eximpl char*		/* --> NUL behind filled */
app_fen_ep( const Board* bp, char* buf, int lang )	/*ARGSUSED*/
{
    if( ! no_pos(bp->b_ep) ) {
	    Position	tpos;
	tpos = bp->b_ep - bau_mov[bp->b_tomove];
	*buf++ = chr_PosCol(tpos);
	*buf++ = chr_PosLin(tpos);
    }else {
	*buf++ = '-';
    }
    *buf = 0;
    return buf;
}


    Eximpl void
put_fen( const Board* bp, int lang, const char* pref )
{
    char		buf[200];
    register char*	p;

    if( pref && *pref ) {
	printf("%s ", pref);
    }
    p = buf;
    p = app_fen_board (bp, p, lang); *p++ = ' ';
    p = app_fen_tom   (bp, p, lang); *p++ = ' ';
    p = app_fen_castle(bp, p, lang); *p++ = ' ';
    p = app_fen_ep    (bp, p, lang);
    printf("%s\n", buf);
}

    Eximpl void
put_fen_eng( const Board* bp )
{
    put_fen(bp, LANG_ENGLISH, "FEN:");
}


/*
 * Print a move in the context of a board.
 * FFS: B is printed.
 * put_move()
 *	Print with a terminating newline.
 * put__move()
 *	Print without a terminating newline.
 */
    Eximpl void
put__move( const Board* bp, const Move* mp )
{
    Figure	 f;

    if( no_pos(mp->m_from) ) {		/* denotes "no move" */
	printf("(no move)");
	return;
    }
    f = bp->b_f[mp->m_from].f_f;
    if( f == koenig ) {
	switch( mp->m_to - mp->m_from ) {
	 case MK_DELTA( 2,0):
	    printf("  O - O  ");
	    return;
	 case MK_DELTA(-2,0):
	    printf("  O-O-O  ");
	    return;
	}
    }
    put_figure(f);
    put_position(mp->m_from);
    if( bp->b_f[mp->m_to].f_c == empty ) {
	if( f == bauer ) {
	    switch( mp->m_to - mp->m_from ) {
	     case MK_DELTA(-1,-1):
	     case MK_DELTA(-1, 1):
	     case MK_DELTA( 1,-1):
	     case MK_DELTA( 1, 1):
		printf(" * ");			/* beat e.p. */
		break;
	     default:
		printf(" - ");			/* move */
		break;
	    }
	}else {
	    printf(" - ");			/* move */
	}
    }else {
	printf(" : ");				/* beat */
    }
    put_position(mp->m_to);
						/* indicate promotion: */
    if( mp->m_fig != f ) {
	put_figure(mp->m_fig);
    }else {
	printf(" ");
    }
}


    Eximpl void
put_move( const Board* bp, const Move* mp )
{
    put__move(bp, mp);
    printf("\n");
}


/*
 * Print a list of moves in the context of a board.
 */
    Eximpl void
put_list( const Board* bp, const Movelist* lp )
{
    const Move*	mp;

    formoves( lp, mp ) {
	printf("\t");
	put_move(bp, mp);
    }
}


    Eximpl void
put_packed( register const PackedBoard* pkp, register const char* pref )
{
    register int	n;
    register int	v;
    register int	col;
    register int	lin;
    register char	c;

    for( lin=7 ; lin>=0 ; --lin ) {
	if( pref ) {
	    printf("%s", pref);
	}
	for( col=0 ; col<8 ; ++col ) {
	    n = MK_POS64(col,lin);
	    v = ULSHR(*PB_64_P(pkp, n), PB_64_shift(n)) & 0x0f;
	    switch( v ) {
	     default:	c = '?'; break;
	     case 0:	c = '-'; break;
#define CASE_w(f)	case PB_cf_val(white,f): c = chr_Figure(f); break;
#define CASE_b(f)	case PB_cf_val(black,f): c = chr_figure(f); break;
#define CASE_wb(f)	CASE_w(f) CASE_b(f)
	     CASE_wb(bauer)
	     CASE_wb(springer)
	     CASE_wb(laeufer)
	     CASE_wb(turm)
	     CASE_wb(dame)
	     CASE_wb(koenig)
#undef  CASE_wb
#undef  CASE_w
#undef  CASE_b
	    }
	    printf("%c ", c);
	}
	printf("\n");
    }
}

/*===========================================================================*/

    Eximpl void
tit_put_line( const char* linp )
{
    if( linp ) {
	printf("#\t%s\n", linp);
    }
}


#define MAX_TIT_LINES	10
#define MAX_TIT_BUFLEN	500

static int	tit_full	= 0;
static char	tit_buf[MAX_TIT_BUFLEN];
static char*	tit_ptr		= tit_buf;	/* free pointer */
static char*	tit_line[MAX_TIT_LINES];	/* -> tit_buf[] */
static Bool	tit_trunc	= FALSE;	/* truncation occurred */

#define		tit_buf_end	(tit_buf + sizeof(tit_buf))


    Eximpl void
tit_clear(void)
{
    tit_full  = 0;
    tit_ptr   = tit_buf;
    tit_trunc = FALSE;
}


    Eximpl void
tit_add_line( const char* linp )
{
    if( tit_full >= MAX_TIT_LINES ) {
	goto trunc;
    }
    if( (tit_buf_end - tit_ptr) <= 1 ) {
	goto trunc;
    }
    tit_line[tit_full++] = tit_ptr;	/* remember start */
    while( *linp && ((tit_buf_end - tit_ptr) >= 1) ) {
	*tit_ptr++ = *linp++;
    }
    *tit_ptr++ = '\0';		/* terminate string */
    if( ! *linp ) {
	return;			/* completely saved */
    }
trunc:;
    tit_trunc = TRUE;		/* could not store any more */
}


    Eximpl void
put_tit_lines(void)
{
    if( tit_full ) {
	    int		i;
	for( i=0 ; i<tit_full ; ++i ) {
	    tit_put_line(tit_line[i]);
	}
	if( tit_trunc ) {
	    printf("# ...\n");
	}
	printf("\n");
    }
}
