/*
 * CHEST, chess analyst.  For Copyright notice read file "COPYRIGHT".
 *
 * $Source: /home/heiner/ca/chest/RCS/move_gen.c,v $
 * $Id: move_gen.c,v 3.9 1999/07/20 21:05:54 heiner Exp $
 *
 *	generate list of legal moves
 */

#include "bsd.h"
#include "types.h"
#include "board.h"
#include "mgsubr.h"
#include "output.h"
#include "stats.h"
#include <stdio.h>
#include "move_gen.h"


#if PROD_LEV
# define MOVG_STATS	0
#else	/* !PROD_LEV */
# ifndef MOVG_STATS
#  define MOVG_STATS	1		/* CF: local statistics */
# endif
#endif	/* !PROD_LEV */

#undef  THIS_STATS
#define THIS_STATS	MOVG_STATS
#include "statsupp.h"

#define C_COLOUR(c)	chr_Colour(c)

#if MOVG_STATS

typedef struct MgStats	MgStats;

struct MgStats
{
    Counter	ms_inchk[3][2];		/* [count][tomove] (always) */
				/* level >= 2: */
    Counter	ms_fmv[8];		/* [fig] */
    Counter	ms_figmv[3][2][8];	/* [count][tomove][fig] */
    Counter	ms_ixfipc[3][32][8];	/* [count][idx][fig] mg saw piece */
    Counter	ms_ixfimv[3][32][8];	/* [count][idx][fig] made moves */
};

static MgStats	sc_mg;
static MgStats	sc_mg_ala;

#endif	/* MOVG_STATS */


#if MOVG_STATS
    static void
show_fvec( register const Counter* cntp )
{
    register rFigure	f;

    for( f=0 ; f<MAX_FIGURES ; ++f ) {		/* 6*12 = 72 chars */
	show_sc(0,10, cntp[f]); printf(" %c", chr_Figure((Figure)f));
    }
}

    static void
show_div_fvec( 
    register const Counter*	sump,
    register const Counter*	cntp)
{
    register rFigure	f;

    for( f=0 ; f<MAX_FIGURES ; ++f ) {		/* 6*12 = 72 chars */
	if( cntp[f] ) {
	    printf("%10.2f %c",
			sump[f] / (double)cntp[f],
			chr_Figure((Figure)f));
	}else {
	    printf("%10s %c", "-", chr_Figure((Figure)f));
	}
    }
}

    static void
show_nz_fvec(
    const char*			hdr,
    register const Counter*	cntp)
{
    register rFigure	f;

    for( f=0 ; f<MAX_FIGURES ; ++f ) {
	if( cntp[f] ) {
	    printf("%s", hdr); show_fvec(cntp); printf("\n");
	    break;
	}
    }
}

    static void
show_nz_div_fvec(
    const char*			hdr,
    register const Counter*	sump,
    register const Counter*	cntp)
{
    register rFigure	f;

    for( f=0 ; f<MAX_FIGURES ; ++f ) {
	if( cntp[f] ) {
	    printf("%s", hdr); show_div_fvec(sump, cntp); printf("\n");
	    break;
	}
    }
}
#endif


#if MOVG_STATS
    static void
movg_stats(
    register MgStats*		msp,
    register const char*	hdr)
{
    register int	i;
    register Counter	sum;
    register rColour	c;
    static char*	numstrs[3] ={ "none", "one", "double" };

    if( hdr && (f_stats >= 1) ) {
	for( c=0 ; c<2 ; ++c ) {
	    sum = 0;
	    for( i=0 ; i<3 ; ++i ) {
		sum += msp->ms_inchk[i][c];
	    }
	    if( sum ) {
		printf("%2s", hdr);
		show_sc(1,9, sum);
		printf(" %c; inchk:", C_COLOUR(c));
		for( i=0 ; i<3 ; ++i ) {
		    show_sc(1,9, msp->ms_inchk[i][c]);
		    printf(" %s", numstrs[i]);
		}
		printf("\n");
	    }
	}
	/* FFS: tell avg move list length */
	if( f_stats >= 2 ) {
		char	hbuf[80];
	    sprintf(hbuf, "%2.2s   :", hdr); show_nz_fvec(hbuf, msp->ms_fmv);
	    for( c=0 ; c<2 ; ++c ) {
		for( i=0 ; i<3 ; ++i ) {
		    sprintf(hbuf, "%2.2s %d%c:", hdr, i, C_COLOUR(c));
		    show_nz_fvec(hbuf, msp->ms_figmv[i][c]);
		}
	    }
	    /*
	     * From ms_ixfipc and ms_ixfimv we currently want to know,
	     * how often is a certain color/figure occured, and how
	     * many moves were generated in average.
	     * This for check/not in check separately.
	     */
	    {
		    Counter		pc[3][2][8];	/* [cnt][color][fig] */
		    Counter		mv[3][2][8];	/* [cnt][color][fig] */
		    Counter		tpc[2][8];	/*      [color][fig] */
		    Counter		tmv[2][8];	/*      [color][fig] */
		    register rFigure	f;
		    register uint32	pcs;
		    register int	idx;
		for( c=0 ; c<2 ; ++c ) {
		    for( f=0 ; f<MAX_FIGURES ; ++f ) {
			tpc[c][f] = 0;
			tmv[c][f] = 0;
			for( i=0 ; i<3 ; ++i ) {
			    pcs = 0; sum = 0;
			    for( idx=0 ; idx<32 ; ++idx ) {
				if( IDX_COLOUR(idx) != c ) continue;
				pcs += msp->ms_ixfipc[i][idx][f];
				sum += msp->ms_ixfimv[i][idx][f];
			    }
			    pc[i][c][f] = pcs;
			    mv[i][c][f] = sum;
			    tpc[c][f] += pcs;
			    tmv[c][f] += sum;
			}
		    }
		}
		for( c=0 ; c<2 ; ++c ) {
		    for( i=0 ; i<3 ; ++i ) {
			sprintf(hbuf, "%2.2s %d%c p", hdr, i, C_COLOUR(c));
			show_nz_fvec(hbuf, pc[i][c]);
		    }
		}
		for( c=0 ; c<2 ; ++c ) {
		    for( i=0 ; i<3 ; ++i ) {
			sprintf(hbuf, "%2.2s %d%c m", hdr, i, C_COLOUR(c));
			show_nz_div_fvec(hbuf, mv[i][c], pc[i][c]);
		    }
		}
		for( c=0 ; c<2 ; ++c ) {
		    sprintf(hbuf, "%2.2s  %c m", hdr, C_COLOUR(c));
		    show_nz_div_fvec(hbuf, tmv[c], tpc[c]);
		}
	    }
	}
    }
							/* reinitialize: */
    bzero((char*)msp, sizeof(*msp));
}
#endif	/* MOVG_STATS */


    Eximpl void
movegen_stats( Bool print )		/*ARGSUSED*/
{
    SC_(    movg_stats(&sc_mg    , (print ? "mg" : (char*)0));
	    movg_stats(&sc_mg_ala, (print ? "mG" : (char*)0)); )
}


#if MOVG_STATS
    static void
stat_list(
    register const Board*	bp,
    register const Movelist*	lp,
    register Counter		fmv[8],
    register Counter		figmv[8],
    register Counter		ixfipc[32][8],
    register Counter		ixfimv[32][8])
{
    {
	    register Move*	mp;
	    register int	f;
	formoves(lp, mp) {
	    f = bp->b_f[mp->m_from].f_f;
	    fmv  [f] += 1;
	    figmv[f] += 1;
	    ixfimv[mp->m_idx][f] += 1;
	}
    }
    {
	    register int	ifig;
	    register int	minifig;
	    register rPosition	pos;
	minifig = COLOUR_IDX(bp->b_tomove);
	for( ifig = minifig + bp->b_max_piece[bp->b_tomove] - 1
	   ; ifig >= minifig
	   ; --ifig ) {
	    pos = bp->b_piece[ifig];
	    if( ! no_pos(pos) ) {
		ixfipc[ifig][bp->b_f[pos].f_f] += 1;
	    }
	}
    }
}
#endif


/*
 * Generate all legal moves for the specified Board.
 * Return, whether any legal move found.
 * When the Movelist is not specified, just generate the result value.
 */
    Eximpl Bool
move_gen(
    register Xconst Board*	bp,
    register Movelist*		lp)
{
    register Xconst Field*	tp;
    register int		tpos;
    register int		delta;
    register int		i;
    register int		imax;
    register rPieceSet		myatts;
    Position			pos;
    Xconst Field*		p;
    Move			m;
    const Field*		tp2;
    int				delta2;
    Colour			self;
    Colour			enemy;
    Position			kpos;
    int8			ifig;
    int8			minifig;
    int				j;
    int				ckdir;
    int				pindir;
    Position			ckpos;
    PieceSet			iatts;
    PieceSet			atts;
    PieceSet			enemymask;
    int				nchecks;
    int8			checkdir[2];	/* LTD checks from own king */

    if( lp ) {
	clear_list(lp);
    }
    m.m_attr  = 0;
    m.m_value = 0;
    self  = bp->b_tomove;
    enemy = opp_colour(self);
    kpos = K_POS(bp, self);
			/* Analyse the checking situation of our king: */
    nchecks = 0;
    enemymask = COLOUR_MASK(enemy);
    iatts = F_IATT(&(bp->b_f[kpos])) & enemymask;
    atts  = F_DATT(&(bp->b_f[kpos])) & enemymask;
    if( atts ) {
	i = 0;
	do {
	    while( ! (atts & 01) ) {
		j = MIN_LOW_ZERO(atts);
		i += j; atts >>= j;
	    }
	    ckpos = bp->b_piece[i];
	    ckdir = att_dir(kpos, ckpos);	/* save this for later */
	    switch( bp->b_f[ckpos].f_f ) {
	     case laeufer:
	     case turm:
	     case dame:
		checkdir[nchecks] = ckdir;
		break;
	     default:
		checkdir[nchecks] = NO_DIR;
		break;
	    }
	    ++nchecks;
	    ++i; atts >>= 1;
	}while( atts );
    }
    scinc(sc_mg.ms_inchk[nchecks][self]);
					/* decide type of move enumeration */
    if( nchecks == 0 ) {
	/*
	 * We are not in check. Thus our pieces can move relatively freely,
	 * as they are not forced to defend against a check.
	 * Thus enumerate the move possibilities by the capable figures:
	 */
	minifig = COLOUR_IDX(self);
	for( ifig = minifig + bp->b_max_piece[self] - 1
	   ; ifig >= minifig
	   ; --ifig ) {
	    pos = bp->b_piece[ifig];
	    if( no_pos(pos) )
		continue;		/* skip empty slot (beaten piece) */
	    p = &(bp->b_f[pos]);
	    m.m_from = pos;
	    m.m_idx  = ifig;
					/* Analyse the pinning of this piece: */
	    pindir = NO_DIR;
	    if( (atts = (iatts & F_DATT(p))) ) {
		i = att_dir(pos, kpos);
		if( dam_dir(i) ) {
		    delta = dam_mov[i];
		    if( (pos + delta) != kpos ) {
			if( atts & F_IATT(&(p[delta])) ) {
			    pindir = i;
			}
		    }else {
			/* Piece to be checked is near to its king.
			 * Therefore there is only one f_iatt,
			 * which alone is not sufficient for
			 * check of direction.
			 * If T-dir, we are already sure to be pinned.
			 */
			if( trm_dir(i) ) {
			    pindir = i;
			}else {			/* L-dir, search: */
			    tp = p;
			    do {
				tp -= delta;
			    }while( tp->f_c == empty );
			    if( (tp->f_c == enemy)
			     && (atts & SET1(tp->f_idx)) ) {
				pindir = i;
			    }
			}
		    }
		}
	    }
	    switch( p->f_f ) {
	     case bauer:
		/* Try single and double step: */
		delta = bau_mov[self];
		if( no_dir(pindir)
		 || (delta == dam_mov[pindir])
		 || (delta == dam_mov[opp_dir(pindir)]) ) {
					/* Not pinned or staying in pinning */
		    tpos = pos;
		    for( i=0 ; i<2 ; ++i ) {
			if( i && (LIN64(p->f_pos64) != BAS_LIN(self)) )
			    break;
			tpos += delta;
			if( bp->b_f[tpos].f_c != empty )
			    break;
			if( ! lp )
			    return TRUE;
			m.m_to = tpos;
			if( LIN64(p->f_pos64) == BAS_LIN(enemy) ) {
			    m.m_fig = dame    ; app_move(&m, lp);
			    m.m_fig = springer; app_move(&m, lp);
			    m.m_fig = turm    ; app_move(&m, lp);
			    m.m_fig = laeufer ; app_move(&m, lp);
			}else {
			    m.m_fig = bauer   ; app_move(&m, lp);
			}
		    }
		}
		/* Try to beat in both directions: */
		for( i=0 ; i<2 ; ++i ) {
		    delta = (i ? bau_right : bau_left)[self];
		    if( ! no_dir(pindir)
		     && (delta != dam_mov[pindir])
		     && (delta != dam_mov[opp_dir(pindir)]) ) {
			continue;
		    }
					/* Not pinned or staying in pinning */
		    tpos = pos + delta;
		    tp = &(bp->b_f[tpos]);
		    if( (tp->f_c == border) || (tp->f_c == self) )
			continue;
		    if( tp->f_c == empty ) {		/* try e.p.: */
			if( bp->b_ep == (tpos - bau_mov[self]) ) {
			    if( ! ep_legal(bp, pos, tpos, kpos) )
				continue;
			    if( ! lp )
				return TRUE;
			    m.m_to  = tpos;
			    m.m_fig = bauer;
			    app_move(&m, lp);
			}
		    }else {			/* try normal beat: */
			if( ! lp )
			    return TRUE;
			m.m_to = tpos;
			if( LIN64(p->f_pos64) == BAS_LIN(enemy) ) {
			    m.m_fig = dame    ; app_move(&m, lp);
			    m.m_fig = springer; app_move(&m, lp);
			    m.m_fig = turm    ; app_move(&m, lp);
			    m.m_fig = laeufer ; app_move(&m, lp);
			}else {
			    m.m_fig = bauer   ; app_move(&m, lp);
			}
		    }
		}
		break;
	     case springer:
		if( ! no_dir(pindir) )
		    break;			/* pinning always left by S */
		if( lp ) {
			register Move*	mp;
		    mp = lp->l_free;
		    for( i=0 ; i<8 ; ++i ) {
			tp = &(bp->b_f[ tpos = pos + spr_mov[i] ]);
			if( (tp->f_c == border) || (tp->f_c == self) )
			    continue;
			mp->m_from  = pos;
			mp->m_to    = tpos;
			mp->m_fig   = springer;
			mp->m_idx   = ifig;
			mp->m_attr  = 0;
			mp->m_value = 0;
			++mp;
		    }
		    lp->l_free = mp;
		}else {						/* ! lp */
		    for( i=0 ; i<8 ; ++i ) {
			tp = &(bp->b_f[ tpos = pos + spr_mov[i] ]);
			if( (tp->f_c == border) || (tp->f_c == self) )
			    continue;
			return TRUE;
		    }
		}
		break;
	     case laeufer:
		if( no_dir(pindir) ) {
		    i    = MIN_L_DIR;
		    imax = MAX_L_DIR;
		}else if( lfr_dir(pindir) ) {
		    i    = pindir & ~1;
		    imax = i + 2;
		}else {
		    break;
		}
		goto ltd;
	     case turm:
		if( no_dir(pindir) ) {
		    i    = MIN_T_DIR;
		    imax = MAX_T_DIR;
		}else if( trm_dir(pindir) ) {
		    i    = pindir & ~1;
		    imax = i + 2;
		}else {
		    break;
		}
		goto ltd;
	     case dame:
		if( no_dir(pindir) ) {
		    i    = MIN_D_DIR;
		    imax = MAX_D_DIR;
		}else {
		    i    = pindir & ~1;
		    imax = i + 2;
		}
	ltd:
		/* When pinned, we here already have restricted
		 * the directions to pindir and its complement.
		 */
		if( lp ) {
			register Move*	mp;
		    mp = lp->l_free;
		    for( ; i<imax ; ++i ) {
			tp = &(bp->b_f[tpos = pos]);
			delta = dam_mov[i];
			do {
			    tp += delta;
			    if( (tp->f_c == border) || (tp->f_c == self) )
				break;
			    tpos += delta;
			    mp->m_from  = pos;
			    mp->m_to    = tpos;
			    mp->m_fig   = p->f_f;
			    mp->m_idx   = ifig;
			    mp->m_attr  = 0;
			    mp->m_value = 0;
			    ++mp;
			}while( tp->f_c == empty );
		    }
		    lp->l_free = mp;
		}else {						/* ! lp */
		    for( ; i<imax ; ++i ) {
			tp = &(bp->b_f[pos+dam_mov[i]]);
			if( (tp->f_c == border) || (tp->f_c == self) )
			    continue;
			return TRUE;
		    }
		}
		break;
	     case koenig:		/* is handled below */
		break;
	     default:
		panic("move_gen: figure %d", p->f_f);
		break;
	    }
	}
    }else if( nchecks == 1 ) {
	/* Try to generate moves which avoid the existing check by construction:
	 * Either beat the checker or block it.
	 */
	PieceSet	selfmask;

	selfmask = COLOUR_MASK(self);
	tpos = kpos;
	tp = &(bp->b_f[tpos]);
	if( dam_dir(ckdir) ) {
	    delta = dam_mov[ckdir];
	}else {
	    delta = spr_mov[ckdir - MIN_S_DIR];
	}
	do {
	    tpos += delta;
	    tp   += delta;
					/* Try to find moves by own attacks: */
	    myatts = F_DATT(tp) & selfmask;
	    if( tp->f_c == empty ) {
		/* Fake B moves as if it were attacks: */
		p = tp - bau_mov[self];
		if( (p->f_c == self) && (p->f_f == bauer) ) {
		    myatts |= SET1(p->f_idx);
		}else if( p->f_c == empty ) {
		    p -= bau_mov[self];
		    if( (p->f_c == self) && (p->f_f == bauer)
		     && (LIN64(p->f_pos64) == BAS_LIN(self)) ) {
			myatts |= SET1(p->f_idx);
		    }
		}
	    }
	    for( i = 0 ; myatts ; ++i, myatts >>= 1 ) {
		while( ! (myatts & 01) ) {
		    j = MIN_LOW_ZERO(myatts);
		    i += j; myatts >>= j;
		}
		pos = bp->b_piece[i];
		p = &(bp->b_f[pos]);
		switch( p->f_f ) {
		 case bauer:		/* must not beat to empty field */
		    if( (tp->f_c == empty) && lfr_dir(att_dir(pos, tpos)) )
			continue;
		    break;
		 case koenig:
		    continue;		/* Handle below */
		 default:
		    break;
		}
		/* This piece could move and avoid the check.
		 * Check, whether it is pinned.
		 * If so, the move cannot stay within the pinning.
		 */
		if( (atts = (iatts & F_DATT(p))) ) {
		    j = att_dir(pos, kpos);
		    if( dam_dir(j) ) {
			delta2 = dam_mov[j];
			if( (pos + delta2) != kpos ) {
			    if( atts & F_IATT(&(p[delta2])) ) {
				continue;
			    }
			}else {
			    /* Piece to be checked is near to its king.
			     * Therefore there is only one f_iatt,
			     * which alone is not sufficient for
			     * check of direction. So search:
			     */
			    tp2 = p;
			    do {
				tp2 -= delta2;
			    }while( tp2->f_c == empty );
			    if( (tp2->f_c == enemy)
			     && (atts & SET1(tp2->f_idx)) ) {
				continue;
			    }
			}
		    }
		}
		/* Is a legal move: */
		if( ! lp )
		    return TRUE;
		if( (p->f_f == bauer)
		 && (LIN64(p->f_pos64) == BAS_LIN(enemy)) ) {
		    m.m_from = pos;
		    m.m_to   = tpos;
		    m.m_idx  = i;
		    m.m_fig  = dame    ; app_move(&m, lp);
		    m.m_fig  = springer; app_move(&m, lp);
		    m.m_fig  = turm    ; app_move(&m, lp);
		    m.m_fig  = laeufer ; app_move(&m, lp);
		}else {
			register Move*	mp;
		    mp = lp->l_free;
		    mp->m_from  = pos;
		    mp->m_to    = tpos;
		    mp->m_fig   = p->f_f;
		    mp->m_idx   = i;
		    mp->m_attr  = 0;
		    mp->m_value = 0;
		    lp->l_free = ++mp;
		}
	    }
	}while( tp->f_c == empty );
	/* Now tpos/tp points to the checker.
	 * Not yet generated are e.p. moves.
	 * When the double B step uncovers an indirect check,
	 * it cannot be removed by beating e.p.
	 * Thus e.p. is legal here only if beating a B which
	 * is the checker itself.
	 */
	if( tpos == bp->b_ep ) {
	    tpos += bau_mov[self];
	    tp += bau_mov[self];
	    for( i = 0 ; i < 2 ; ++i ) {
		delta = (i ? bau_left : bau_right)[self];
		p = tp - delta;
		pos = tpos - delta;
		if( (p->f_c == self) && (p->f_f == bauer) ) {
					/* May beat e.p. when not pinned */
		    if( (iatts & F_DATT(p)) && dam_dir(att_dir(pos, kpos)) ) {
			/*
			 * Direct attack one T-step away from indirect at K
			 * <=> pinned (on T-dir, thus moving on L-dir illegal)
			 */
			continue;
		    }
		    if( ! lp )
			return TRUE;
		    m.m_from = pos;
		    m.m_to   = tpos;
		    m.m_idx  = p->f_idx;
		    m.m_fig  = bauer;
		    app_move(&m, lp);
		}
	    }
	}
    }
						/* Now add the king moves: */
    {
	    register Move*	mp;
	if( lp )
	    mp = lp->l_free;
	p = &(bp->b_f[kpos]);
	for( i=MIN_D_DIR ; i<MAX_D_DIR ; ++i ) {
	    delta = dam_mov[i];
	    tp = p + delta;
	    if( (tp->f_c == border) || (tp->f_c == self) )
		continue;
	    if( F_DATT(tp) & enemymask )
		continue;
	    for( j=0 ; j<nchecks ; ++j ) {
		ckdir = checkdir[j];
		if( ! no_dir(ckdir) ) {	/* is LTD check direction (from king) */
		    if( i == opp_dir(ckdir) )
			goto next_dir;
		}
	    }
	    if( ! lp )
		return TRUE;
	    mp->m_from  = kpos;
	    mp->m_to    = kpos + delta;
	    mp->m_fig   = koenig;
	    mp->m_idx   = p->f_idx;
	    mp->m_attr  = 0;
	    mp->m_value = 0;
	    ++mp;
next_dir:   ;
	}
	if( lp )
	    lp->l_free = mp;
    }
							/* Try to castle: */
    if( ! lp )
	return FALSE;			/* no castling without legal K moves */
    if( (nchecks == 0) && bp->b_castle[self] ) {
	mg_app_castle(bp, lp);
    }
    if( list_length(lp) > MAX_MOVES ) {
	panic("Movelist overflow (%d > %d)", (int)list_length(lp), MAX_MOVES);
    }
    mg_link(lp);
    SC_(if( f_stats >= 2 ) {
	    stat_list(bp, lp,   sc_mg.ms_fmv,
				sc_mg.ms_figmv[nchecks][self],
				sc_mg.ms_ixfipc[nchecks],
				sc_mg.ms_ixfimv[nchecks]
	    );
	}
    )
    return ! empty_list(lp);
}


/*
 * Generate all legal moves for the specified Board,
 * provided they are useful for a mate attacker.
 * Return, whether any such legal move found.
 *
 * Special operation is performed only, when the attacker has only
 * one more piece than his king left, and the movelist is really to be filled.
 *
 * The idea here is, that an attacker with a bare king cannot win.
 * Hence, if the last piece is seen to become or be left beatable,
 * that cannot be a good move for the attacker.
 *    Also, when considering attacks to the last piece of the attacker,
 * legality of the enemy (defender) move cannot be hindered by pinnings.
 * Hence, attack bits of the defender within the attacker always indicate
 * a legal beat, which is fatal, here.
 */
    Eximpl Bool
ala_move_gen(
    register Xconst Board*	bp,
    register Movelist*		lp)
{
    register Xconst Field*	tp;
    register int		tpos;
    register int		delta;
    register int		i;
    register int		imax;
    Position			pos;
    Position			pos1;
    Xconst Field*		p;
    Move			m;
    const Field*		tp2;
    int				delta2;
    Colour			self;
    Colour			enemy;
    Position			kpos;
    int8			ifig;
    int8			minifig;
    int				j;
    int				ckdir;
    int				pindir;
    Position			ckpos;
    PieceSet			iatts;
    register rPieceSet		atts;
    PieceSet			enemymask;
    PieceSet			nkenemymask;	/* except K */
    PieceSet			kenemymask;	/* only K */
    PieceSet			selfmask;
    PieceSet			ifigmask;	/* only ifig */
    PieceSet			nifigselfmask;	/* except ifig */
    int				nchecks;
    int8			checkdir[2];	/* LTD checks from own king */
    int8			checkinx[2];
    Position			mustnear;

    self = bp->b_tomove;
    if( !lp || (bp->b_cur_piece[self] != 2) ) {
	return move_gen(bp, lp);
    }
    /*
     * Well, this is the special situation.
     * There is just one more piece than the king itself.
     * Also, the move list is guaranteed.
     */
    clear_list(lp);
    m.m_attr  = 0;
    m.m_value = 0;
    enemy = opp_colour(self);
    kpos = K_POS(bp, self);
			/* Find that single other piece we have: */
    minifig = COLOUR_IDX(self);
    ifig = minifig + bp->b_max_piece[self];
    do {
	--ifig;
	if( ifig < minifig ) {
	    panic("ala_mov_gen: can't find piece");
	}
				/* skip empty slot (beaten piece) and K */
	pos = bp->b_piece[ifig];
    }while( no_pos(pos) || (pos == kpos) );
    pos1 = pos;			/* here it is */
    selfmask = COLOUR_MASK(self);
    ifigmask = SET1(ifig);
    nifigselfmask = selfmask & ~ifigmask;	/* effectively self-K */
			/* Analyse the checking situation of our king: */
    nchecks = 0;
    enemymask = COLOUR_MASK(enemy);
    nkenemymask = enemymask & ~BOTH_KINGS;
    kenemymask  = enemymask &  BOTH_KINGS;
    iatts = F_IATT(&(bp->b_f[kpos])) & enemymask;
    atts  = F_DATT(&(bp->b_f[kpos])) & enemymask;
    if( atts ) {
	i = 0;
	do {
	    while( ! (atts & 01) ) {
		j = MIN_LOW_ZERO(atts);
		i += j; atts >>= j;
	    }
	    ckpos = bp->b_piece[i];
	    ckdir = att_dir(kpos, ckpos);	/* save this for later */
	    switch( bp->b_f[ckpos].f_f ) {
	     case laeufer:
	     case turm:
	     case dame:
		checkdir[nchecks] = ckdir;
		checkinx[nchecks] = i;
		break;
	     default:
		checkdir[nchecks] = NO_DIR;
		checkinx[nchecks] = -1;
		break;
	    }
	    ++nchecks;
	    ++i; atts >>= 1;
	}while( atts );
    }
    scinc(sc_mg_ala.ms_inchk[nchecks][self]);
					/* decide type of move enumeration */
    if( nchecks == 0 ) {
	/*
	 * We are not in check. Thus our piece can move relatively freely,
	 * as it is not forced to defend against a check.
	 * Thus enumerate the move possibilities by the capable figures:
	 */
	p = &(bp->b_f[ pos = pos1 ]);
	m.m_from = pos;
	m.m_idx  = ifig;
					/* Analyse the pinning of this piece: */
	pindir = NO_DIR;
	if( (atts = (iatts & F_DATT(p))) ) {
	    i = att_dir(pos, kpos);
	    if( dam_dir(i) ) {
		delta = dam_mov[i];
		if( (pos + delta) != kpos ) {
		    if( atts & F_IATT(&(p[delta])) ) {
			pindir = i;
		    }
		}else {
		    /* Piece to be checked is near to its king.
		     * Therefore there is only one f_iatt,
		     * which alone is not sufficient for
		     * check of direction.
		     * If T-dir, we are already sure to be pinned.
		     */
		    if( trm_dir(i) ) {
			pindir = i;
		    }else {			/* L-dir, search: */
			tp = p;
			do {
			    tp -= delta;
			}while( tp->f_c == empty );
			if( (tp->f_c == enemy)
			 && (atts & SET1(tp->f_idx)) ) {
			    pindir = i;
			}
		    }
		}
	    }
	}
	/*
	 * If this piece is pinned, its only chance is to beat the pinner,
	 * as staying in the pinning line cannot avoid him to beat us.
	 * But in no way there can be a K move which avoids that currently
	 * pinned piece to be beaten.  In that case -> end_kall;
	 */
	switch( p->f_f ) {
	 case bauer:
	    /*
	     * Try single and double step, provided we are not pinned:
	     */
	    if( no_dir(pindir) ) {		/* Not pinned */
		delta = bau_mov[self];
		tpos = pos;
		for( i=0 ; i<2 ; ++i ) {
		    if( i && (LIN64(p->f_pos64) != BAS_LIN(self)) )
			break;
		    tpos += delta;
		    tp = &(bp->b_f[tpos]);
		    if( tp->f_c != empty )
			break;
		    if( ! (F_DATT(tp) & nkenemymask) ) {
			m.m_to = tpos;
			if( LIN64(p->f_pos64) == BAS_LIN(enemy) ) {
			    m.m_fig = dame    ; app_move(&m, lp);
			    m.m_fig = springer; app_move(&m, lp);
			    m.m_fig = turm    ; app_move(&m, lp);
			    m.m_fig = laeufer ; app_move(&m, lp);
			}else {
			    m.m_fig = bauer   ; app_move(&m, lp);
			}
		    }
		}
	    }
					/* Try to beat in both directions: */
	    for( i=0 ; i<2 ; ++i ) {
		delta = (i ? bau_right : bau_left)[self];
		if( ! no_dir(pindir)
		 && (delta != dam_mov[pindir])
		 && (delta != dam_mov[opp_dir(pindir)]) ) {
		    continue;
		}
					/* Not pinned or staying in pinning */
		tpos = pos + delta;
		tp = &(bp->b_f[tpos]);
		if( (tp->f_c == border) || (tp->f_c == self) )
		    continue;
		if( tp->f_c == empty ) {		/* try e.p.: */
		    if( bp->b_ep == (tpos - bau_mov[self]) ) {
			if( ! ep_legal(bp, pos, tpos, kpos) )
			    continue;
			m.m_to  = tpos;
			m.m_fig = bauer;
			app_move(&m, lp);
		    }
		}else {			/* try normal beat: */
		    if( ! (F_DATT(tp) & nkenemymask) ) {
			m.m_to = tpos;
			if( LIN64(p->f_pos64) == BAS_LIN(enemy) ) {
			    m.m_fig = dame    ; app_move(&m, lp);
			    m.m_fig = springer; app_move(&m, lp);
			    m.m_fig = turm    ; app_move(&m, lp);
			    m.m_fig = laeufer ; app_move(&m, lp);
			}else {
			    m.m_fig = bauer   ; app_move(&m, lp);
			}
		    }
		}
	    }
	    if( ! no_dir(pindir) ) {
		goto end_kall;		/* also no K move */
	    }
	    break;
	 case springer:
	    if( ! no_dir(pindir) ) {	/* pinning always left by S */
		goto end_kall;		/* also no K move */
	    }
	    {
		    register Move*	mp;
		mp = lp->l_free;
		for( i=0 ; i<8 ; ++i ) {
		    tp = &(bp->b_f[ tpos = pos + spr_mov[i] ]);
		    if( (tp->f_c == border) || (tp->f_c == self) )
			continue;
		    atts = F_DATT(tp);
		    if( ! (atts & nkenemymask)
		     && (!(atts & kenemymask) || (atts & nifigselfmask)) ) {
			mp->m_from  = pos;
			mp->m_to    = tpos;
			mp->m_fig   = springer;
			mp->m_idx   = ifig;
			mp->m_attr  = 0;
			mp->m_value = 0;
			++mp;
		    }
		}
		lp->l_free = mp;
	    }
	    break;
	 case laeufer:
	    if( ! no_dir(pindir) ) {
		if( lfr_dir(pindir) ) {
		    goto ltd_pinned;
		}
		goto end_kall;
	    }
	    i    = MIN_L_DIR;
	    imax = MAX_L_DIR;
	    goto ltd;
	 case turm:
	    if( ! no_dir(pindir) ) {
		if( trm_dir(pindir) ) {
		    goto ltd_pinned;
		}
		goto end_kall;
	    }
	    i    = MIN_T_DIR;
	    imax = MAX_T_DIR;
	    goto ltd;
	 case dame:
	    if( ! no_dir(pindir) ) {
		goto ltd_pinned;
	    }
	    i    = MIN_D_DIR;
	    imax = MAX_D_DIR;
    ltd:			/* When pinned, we don't come here.  */
	    {
		    register Move*	mp;
		mp = lp->l_free;
		for( ; i<imax ; ++i ) {
		    tpos = pos; tp = p;
		    delta = dam_mov[i];
#if 1
		    /*
		     * We must not run away from an already existing
		     * attack into the currently indirect shadow.
		     */
		    if( (atts  = F_DATT(tp) & enemymask)
		     && (atts &= F_IATT(&(tp[delta]))) ) {
			do {
			    tp -= delta;
			}while( tp->f_c == empty );
			if( (tp->f_c == enemy) && (atts & SET1(tp->f_idx)) ) {
			    continue;	/* this direction is useless */
			}
			tp = p;		/* restores old value */
		    }
#endif
		    do {
			tp += delta;
			if( (tp->f_c == border) || (tp->f_c == self) )
			    break;
			tpos += delta;
			atts = F_DATT(tp);
			if( atts & nkenemymask )
			    continue;
			if( (atts & kenemymask) && ! (atts & nifigselfmask) ) {
			    continue;
			}
			mp->m_from  = pos;
			mp->m_to    = tpos;
			mp->m_fig   = p->f_f;
			mp->m_idx   = ifig;
			mp->m_attr  = 0;
			mp->m_value = 0;
			++mp;
		    }while( tp->f_c == empty );
		}
		lp->l_free = mp;
	    }
	    break;
ltd_pinned: ;
	    /*
	     * We are pinned & thus forced to beat the pinner.  Search him:
	     */
	    i = opp_dir(pindir);
	    tp = &(bp->b_f[ tpos = pos ]);
	    delta = dam_mov[i];
	    do {
		tp += delta;
		tpos += delta;
	    }while( tp->f_c == empty );
	    /*
	     * He must not be guarded by non-K (would imply beat).
	     * If he is guarded by his K, that would also imply fatal beat,
	     * ar our K is guaranteed to be too far away.
	     */
	    if( ! (F_DATT(tp) & enemymask) ) {
		    register Move*	mp;
		mp = lp->l_free;
		mp->m_from  = pos;
		mp->m_to    = tpos;
		mp->m_fig   = p->f_f;
		mp->m_idx   = ifig;
		mp->m_attr  = 0;
		mp->m_value = 0;
		lp->l_free = ++mp;
	    }
	    goto end_kall;	/* pinned piece implies no K-moves */
	 case koenig:		/* must not occur here */
	 default:
	    panic("ala_move_gen: figure %d", p->f_f);
	    break;
	}
    }else if( nchecks == 1 ) {
	/*
	 * Try to generate moves which avoid the existing check by construction:
	 * Either beat the checker or block it.
	 *    Oops, wait, blocking implies that the blocker can be beaten,
	 * what is considered fatal, here.  Hence we must beat the checker.
	 * But then, when he is guarded by a non-K, we must not do that,
	 * as a back-beat cannot be avoided.
	 */
	tp = &(bp->b_f[ tpos = ckpos ]);	/* he says check to us */
					/* Try to find moves by own attacks: */
	atts = F_DATT(tp);
	if( atts & ifigmask ) {		/* ifig attacks to checking enemy */
	    p = &(bp->b_f[ pos = pos1 ]);
	    if(   ! (atts & nkenemymask)	/* ! non-K */
	       && (   ! (atts & kenemymask)
		   ||   (atts & nifigselfmask)
		  )
	      ) {
		/*
		 * This piece could move (beat the checker) and avoid the check.
		 * Check, whether it is pinned.  If so, the move cannot
		 * stay within the pinning, and is illegal.
		 * Also, the pinned piece implies K moves to be hopeless.
		 */
		if( (atts = (iatts & F_DATT(p))) ) {
		    j = att_dir(pos, kpos);
		    if( dam_dir(j) ) {
			delta2 = dam_mov[j];
			if( (pos + delta2) != kpos ) {
			    if( atts & F_IATT(&(p[delta2])) ) {
				goto end_kall;
			    }
			}else {
			    /* Piece to be checked is near to its king.
			     * Therefore there is only one f_iatt,
			     * which alone is not sufficient for
			     * check of direction. So search:
			     */
			    tp2 = p;
			    do {
				tp2 -= delta2;
			    }while( tp2->f_c == empty );
			    if( (tp2->f_c == enemy)
			     && (atts & SET1(tp2->f_idx)) ) {
				goto end_kall;
			    }
			}
		    }
		}
						/* Is a legal move: */
		if( (p->f_f == bauer)
		 && (LIN64(p->f_pos64) == BAS_LIN(enemy)) ) {
		    m.m_from = pos;
		    m.m_to   = tpos;
		    m.m_idx  = ifig;
		    m.m_fig  = dame    ; app_move(&m, lp);
		    m.m_fig  = springer; app_move(&m, lp);
		    m.m_fig  = turm    ; app_move(&m, lp);
		    m.m_fig  = laeufer ; app_move(&m, lp);
		}else {
			register Move*	mp;
		    mp = lp->l_free;
		    mp->m_from  = pos;
		    mp->m_to    = tpos;
		    mp->m_fig   = p->f_f;
		    mp->m_idx   = ifig;
		    mp->m_attr  = 0;
		    mp->m_value = 0;
		    lp->l_free = ++mp;
		}
	    }
	}else {		/* our piece has no dir att in the checker */
	    /* Now tpos/tp points to the checker.
	     * Not yet generated are e.p. moves.
	     * When the double B step uncovers an indirect check,
	     * it cannot be removed by beating e.p.
	     * Thus e.p. is legal here only if beating a B which
	     * is the checker itself.
	     */
	    if( tpos == bp->b_ep ) {
		p = &(bp->b_f[ pos = pos1 ]);
		if( p->f_f == bauer ) {
		    tpos += bau_mov[self];
		    delta = tpos - pos;
		    if( (delta == bau_left[self])
		     || (delta == bau_right[self]) ) {
			/*
			 * May beat e.p. when not pinned.
			 * Direct attack one T-step away from indirect at K
			 * <=> pinned (on T-dir, thus moving on L-dir illegal)
			 */
			if( ! (iatts & F_DATT(p))
			 || ! dam_dir(att_dir(pos, kpos)) ) {
			    m.m_from = pos;
			    m.m_to   = tpos;
			    m.m_idx  = p->f_idx;
			    m.m_fig  = bauer;
			    app_move(&m, lp);
			}
		    }
		}
	    }
	}
    }
						/* Now add the king moves: */
    /*
     * When there is a direct non-K attack of the enemy already in our
     * other piece, the moving K is forced to beat that piece.
     *
     * When there is a K attack already to the last piece,
     * we are forced to guard it with our K.
     *
     * The K move here also must not open an attack to our last piece,
     * except it beats that one also.  This is scanned via check info.
     */
    imax = NO_DIR;				/* note forced K-step dir */
    tp = &(bp->b_f[pos1]);			/* our last piece */
    mustnear = ((F_DATT(tp) & kenemymask) ? pos1 : NO_POS);
    atts = F_DATT(tp) & nkenemymask;		/* bad enemies attack him */
    if( atts ) {
	if( min2elems(atts) ) {
	    goto end_knorm;			/* can't beat more than one */
	}
	i = 0;		/* Search `i', the piece index, our K must beat */
	while( ! (atts & 01) ) {
	    j = MIN_LOW_ZERO(atts);
	    i += j; atts >>= j;
	}
	tpos = bp->b_piece[i];			/* K must beat there */
	imax = att_dir(kpos, tpos);		/* K must go this dir */
	if( !dam_dir(imax) || ((kpos + dam_mov[imax]) != tpos) ) {
	    goto end_knorm;			/* K cannot beat */
	}
    }
    if( nchecks ) {
	/*
	 * One of the at most two checkers (iff LTD there is a checkdir[]
	 * from our K) may indirectly attack our last piece.  If then we
	 * do not beat him, we are lost, as we must escape from that line.
	 */
	if( dam_dir(j = att_dir(pos1, kpos)) ) {
	    for( i=0 ; i<nchecks ; ++i ) {
		if( checkdir[i] == j ) {
		    /*
		     * This LTD checker, our K and our piece are in a line.
		     */
		    tp = &(bp->b_f[pos1]);
		    if( F_IATT(tp) & SET1(checkinx[i]) ) {
			/*
			 * Hence, the indir att would become direct,
			 * and our K must beat into dir `j'.
			 */
			if( !no_dir(imax)	/* already forced different */
			 || (bp->b_f[kpos + dam_mov[j]].f_c != enemy) ) {
			    goto end_kall;	/* cannot beat */
			}
			imax = j;
		    }
		    break;	/* anyhow, only one checker has such dir */
		}
	    }
	}
    }
    if( no_dir(imax) ) {	/* no direction forced, scan all */
	i    = MIN_D_DIR;
	imax = MAX_D_DIR;	/* exclusive */
    }else {			/* force direction `imax' */
	i    = imax;
	imax = i + 1;		/* exclusive */
    }
    {
	    register Move*	mp;
	mp = lp->l_free;
	p = &(bp->b_f[kpos]);
	do {
	    delta = dam_mov[i];
	    tp = p + delta;
	    if( (tp->f_c == border) || (tp->f_c == self) )
		continue;
	    if( F_DATT(tp) & enemymask )
		continue;
	    tpos = kpos + delta;
	    if( ! no_pos(mustnear) ) {
		if( ! dam_dir(j = att_dir(tpos, mustnear))
		 || ((tpos + dam_mov[j]) != mustnear) ) {
		    continue;
		}
	    }
	    for( j=0 ; j<nchecks ; ++j ) {
		ckdir = checkdir[j];
		if( ! no_dir(ckdir) ) {	/* is LTD check direction (from king) */
		    if( i == opp_dir(ckdir) )
			goto next_dir;
		}
	    }
	    mp->m_from  = kpos;
	    mp->m_to    = tpos;
	    mp->m_fig   = koenig;
	    mp->m_idx   = p->f_idx;
	    mp->m_attr  = 0;
	    mp->m_value = 0;
	    ++mp;
next_dir:   ;
	}while( ++i < imax );
	lp->l_free = mp;
    }
end_knorm:;
							/* Try to castle: */
    if( (nchecks == 0) && bp->b_castle[self] ) {
	mg_app_castle(bp, lp);
    }
end_kall:;
    if( list_length(lp) > MAX_MOVES ) {
	panic("Movelist overflow (%d > %d)", (int)list_length(lp), MAX_MOVES);
    }
    mg_link(lp);
    SC_(if( f_stats >= 2 ) {
	    stat_list(bp, lp,   sc_mg_ala.ms_fmv,
				sc_mg_ala.ms_figmv[nchecks][self],
				sc_mg_ala.ms_ixfipc[nchecks],
				sc_mg_ala.ms_ixfimv[nchecks]
	    );
	}
    )
    return ! empty_list(lp);
}
