// distribution boxbackup-0.11.1 (svn version: 2821_2827)
// Box Backup, http://www.boxbackup.org/
// 
// Copyright (c) 2003-2010, Ben Summers and contributors.
// All rights reserved.
// 
// Note that this project uses mixed licensing. Any file with this license
// attached, or where the code LICENSE-DUAL appears on the first line, falls
// under this license. See the file COPYING.txt for more information.
// 
// This file is dual licensed. You may use and distribute it providing that you
// comply EITHER with the terms of the BSD license, OR the GPL license. It is
// not necessary to comply with both licenses, only one.
// 
// The BSD license option follows:
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
// 
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//  
// 3. Neither the name of the Box Backup nor the names of its contributors may
//    be used to endorse or promote products derived from this software without
//    specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
// DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
// ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// 
// [http://en.wikipedia.org/wiki/BSD_licenses#3-clause_license_.28.22New_BSD_License.22.29]
// 
// The GPL license option follows:
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
// 
// [http://www.gnu.org/licenses/old-licenses/gpl-2.0.html#SEC4]
// --------------------------------------------------------------------------
//
// File
//		Name:    UnixUser.cpp
//		Purpose: Interface for managing the UNIX user of the current process
//		Created: 21/1/04
//
// --------------------------------------------------------------------------

#include "Box.h"

#ifdef HAVE_PWD_H
	#include <pwd.h>
#endif

#ifdef HAVE_UNISTD_H
	#include <unistd.h>
#endif

#include "UnixUser.h"
#include "CommonException.h"

#include "MemLeakFindOn.h"


// --------------------------------------------------------------------------
//
// Function
//		Name:    UnixUser::UnixUser(const char *)
//		Purpose: Constructor, initialises to info of given username
//		Created: 21/1/04
//
// --------------------------------------------------------------------------
UnixUser::UnixUser(const char *Username)
	: mUID(0),
	  mGID(0),
	  mRevertOnDestruction(false)
{
	// Get password info
	struct passwd *pwd = ::getpwnam(Username);
	if(pwd == 0)
	{
		THROW_EXCEPTION(CommonException, CouldNotLookUpUsername)
	}
	
	// Store UID and GID
	mUID = pwd->pw_uid;
	mGID = pwd->pw_gid;
}


// --------------------------------------------------------------------------
//
// Function
//		Name:    UnixUser::UnixUser(uid_t, gid_t)
//		Purpose: Construct from given UNIX user ID and group ID
//		Created: 15/3/04
//
// --------------------------------------------------------------------------
UnixUser::UnixUser(uid_t UID, gid_t GID)
	: mUID(UID),
	  mGID(GID),
	  mRevertOnDestruction(false)
{
}


// --------------------------------------------------------------------------
//
// Function
//		Name:    UnixUser::~UnixUser()
//		Purpose: Destructor -- reverts to previous user if the change wasn't perminant
//		Created: 21/1/04
//
// --------------------------------------------------------------------------
UnixUser::~UnixUser()
{
	if(mRevertOnDestruction)
	{
		// Revert to "real" user and group id of the process
		if(::setegid(::getgid()) != 0 || ::seteuid(::getuid()) != 0)
		{
			THROW_EXCEPTION(CommonException, CouldNotRestoreProcessUser)
		}
	}
}


// --------------------------------------------------------------------------
//
// Function
//		Name:    UnixUser::ChangeProcessUser(bool)
//		Purpose: Change the process user and group ID to the user. If Temporary == true
//				 the process username will be changed back when the object is destructed.
//		Created: 21/1/04
//
// --------------------------------------------------------------------------
void UnixUser::ChangeProcessUser(bool Temporary)
{
	if(Temporary)
	{
		// Change temporarily (change effective only)
		if(::setegid(mGID) != 0 || ::seteuid(mUID) != 0)
		{
			THROW_EXCEPTION(CommonException, CouldNotChangeProcessUser)
		}
		
		// Mark for change on destruction
		mRevertOnDestruction = true;
	}
	else
	{
		// Change permanently (change all UIDs and GIDs)
		if(::setgid(mGID) != 0 || ::setuid(mUID) != 0)
		{
			THROW_EXCEPTION(CommonException, CouldNotChangeProcessUser)
		}
	}
}




