/*
 * Copyright (c) 2001 Tommy Bohlin <tommy@gatespace.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* jeteye.c
 */

#include <irda.h>

/**********************************************************************
 * Constants
 **********************************************************************/

static const char id_jet[]="jeteye";

/**********************************************************************
 * Data structures
 **********************************************************************/

typedef struct JetEye {
  SerialDevice sd;
  SerialPort* sp;
} JetEye;

/**********************************************************************
 * Methods
 **********************************************************************/

static int jetSetSpeed(SerialDevice* sd, int speed)
{
  JetEye* jet=(JetEye*)sd;
  SerialPort* sp=jet->sp;

  switch (speed) {
  case 19200:
    sp->setLine(sp,LINE_DTR);
    break;
  case 115200:
    sp->setLine(sp,LINE_DTR|LINE_RTS);
    break;
  default: /*9600*/
    sp->setLine(sp,LINE_RTS);
    break;
  }
  sp->setSpeed(sp,speed);
  return 0;
}

static int jetGetSpeedMask(SerialDevice* sd)
{
  JetEye* jet=(JetEye*)sd;

  return jet->sp->getSpeedMask(jet->sp)&
    (SPEED_9600|SPEED_19200|SPEED_115200);
}

static int jetGetMinTurnaroundMask(SerialDevice* sd)
{
  return MIN_TA_10ms;
}

static int jetGetChar(SerialDevice* sd)
{
  JetEye* jet=(JetEye*)sd;

  return jet->sp->getChar(jet->sp);
}

static void jetPutChar(SerialDevice* sd, int c)
{
  JetEye* jet=(JetEye*)sd;

  jet->sp->putChar(jet->sp,c);
}

static void jetClose(SerialDevice* sd)
{
  JetEye* jet=(JetEye*)sd;
  SerialPort* sp=jet->sp;

  sp->setLine(sp,0);
  sp->handle=0;
  sp->status=0;
  freeMem(jet);
}

static void jetStatus(SerialPort* sp, int event)
{
  JetEye* jet=(JetEye*)sp->handle;

  if(jet->sd.status) jet->sd.status(&jet->sd,event);
}

/**********************************************************************
 * External interface
 **********************************************************************/

SerialDevice* createJetEyeDevice(SerialPort* sp)
{
  JetEye* jet=allocMem(id_jet,sizeof(JetEye));

  jet->sd.close=jetClose;
  jet->sd.setSpeed=jetSetSpeed;
  jet->sd.getSpeedMask=jetGetSpeedMask;
  jet->sd.getMinTurnaroundMask=jetGetMinTurnaroundMask;
  jet->sd.getChar=jetGetChar;
  jet->sd.putChar=jetPutChar;
  jet->sd.handle=0;
  jet->sd.status=0;
  jet->sp=sp;

  sp->handle=jet;
  sp->status=jetStatus;

  return &jet->sd;
}
