/*
 * Copyright (c) 1995-2004 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL").
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#define __NO_VERSION__

#include <nnpfs/nnpfs_locl.h>
#include <nnpfs/nnpfs_dev.h>
#include <nnpfs/nnpfs_syscalls.h>
#include <nnpfs/nnnpfs.h>

#ifdef LINUX2_5
#include <linux/statfs.h>
#endif

#ifdef RCSID
RCSID("$Id: nnpfs_vfsops.c,v 1.104 2006/04/04 07:06:24 tol Exp $");
#endif

struct nnpfs nnpfs[NNNPFS];

static void nnpfs_read_inode(struct inode *inode);
static void nnpfs_delete_inode(struct inode *inode);
static void nnpfs_put_super(struct super_block *sb);
static void nnpfs_put_inode(struct inode *inode);
static void nnpfs_write_super(struct super_block * sb);

#ifdef LINUX2_5
static int nnpfs_statfs(struct super_block *sb, struct kstatfs *buf);
#else
static int nnpfs_statfs(struct super_block *sb, struct statfs *buf);
#endif

static struct super_operations nnpfs_sops = { 
    read_inode		: nnpfs_read_inode,
    put_inode		: nnpfs_put_inode,
    delete_inode	: nnpfs_delete_inode,
    put_super		: nnpfs_put_super,
    write_super		: nnpfs_write_super,
    statfs		: nnpfs_statfs,
};

int
nnpfs_fetch_xnode(struct inode *i)
{
    struct super_block *sb;
    struct nnpfs *nnpfsp;
    int error = 0;
    struct nnpfs_message_getroot msg;

    NNPFSDEB(XDEBVFOPS, ("nnpfs_fetch_xnode: inode %p\n", i));

    sb = i->i_sb;
    nnpfsp = VFS_TO_NNPFS(sb);

    if (sb->s_root->d_inode != i) {
	NNPFSDEB(XDEBVFOPS, ("nnpfs_fetch_xnode: Not root node\n"));
	return -ENODEV;
    }

    while (nnpfsp->root == NULL && error == 0) {

	msg.header.opcode = NNPFS_MSG_GETROOT;
	/*
	 * Mounting should done by root, so get the root node with
	 * root's priviliges (usually none, and none is needed).
	 */
	
	msg.cred.uid = 0;
	msg.cred.pag = 0;
	
	error = nnpfs_message_rpc(nnpfsp, &msg.header, sizeof(msg));
	NNPFSDEB(XDEBVFOPS,
		 ("nnpfs_fetch_xnode nnpfs_message_rpc error = %d\n", error));
	if (error == 0)
	    error = ((struct nnpfs_message_wakeup *) &msg)->error;
	NNPFSDEB(XDEBVFOPS,
		 ("nnpfs_fetch_xnode nnpfs_message_wakeup error = %d\n", error));
    }
    if (error) {
	NNPFSDEB(XDEBVFOPS, ("nnpfs_fetch_xnode failed: %d\n", error));
	return error;
    }
    
    return 0;
}

/*
 * create a `fake' root inode for `sb'
 */

static struct inode *
make_root_vnode(struct nnpfs *nnpfsp)
{
    struct nnpfs_node *xn;
    struct nnpfs_msg_node node;
    struct inode *inode;

    memset (&node, 0, sizeof(node));

    node.handle.a = node.handle.b = node.handle.c = node.handle.d = 0;
    node.anonrights = NNPFS_RIGHT_R;
    XA_SET_MODE(&node.attr, S_IFDIR | 0777);
    XA_SET_NLINK(&node.attr, 100);
    XA_SET_SIZE(&node.attr, 0);
    XA_SET_UID(&node.attr, 0);
    XA_SET_GID(&node.attr, 0);
    XA_SET_ATIME(&node.attr, 0);
    XA_SET_MTIME(&node.attr, 0);
    XA_SET_CTIME(&node.attr, 0);
    XA_SET_FILEID(&node.attr, 0);
    XA_SET_TYPE(&node.attr, NNPFS_FILE_DIR);

    new_nnpfs_node (nnpfsp, &node, NULL, &xn);
    inode = XNODE_TO_VNODE(xn);

    down(&nnpfsp->inactive_sem);	
    clear_nnpfs_node(xn);
    nnpfs_free_node(xn);
    up(&nnpfsp->inactive_sem);	

    inode->i_op = &nnpfs_dir_inode_operations;
    inode->i_fop = &nnpfs_dir_operations;

    return inode;
}

/*
 * create a root dcache entry for `sb'
 */

static void
make_root (struct super_block *sb)
{
    struct nnpfs *nnpfsp = VFS_TO_NNPFS(sb);
    struct dentry *dp;
    dp = d_alloc_root(make_root_vnode(nnpfsp));
    nnpfs_d_init(dp);
    sb->s_root = dp;
    nnpfsp->status |= NNPFS_ROOTINSTALLED;
}

struct super_block *
nnpfs_read_super (struct super_block * sb, void * data,
		  int silent)
{
    int minordevice=0;
    struct dentry *ddev;
    
    NNPFSDEB(XDEBVFOPS, ("nnpfs_read_super starting\n"));
    NNPFSDEB(XDEBVFOPS, ("nnpfs_read_super: sb: %p data: %p silent: %d\n",
			 sb, data, silent));
    NNPFSDEB(XDEBVFOPS, ("nnpfs_read_super: %d:%d\n",
			 (int) MAJOR(sb->s_dev),
			 (int) MINOR(sb->s_dev)));
    
    NNPFSDEB(XDEBVFOPS, ("nnpfs_read_super: begin setting variables\n"));

    if (data != NULL) {
	struct nameidata nd;
	int error = 0;
	
#ifdef LINUX2_5
	error = path_lookup(data, 0, &nd);
#else
	if (path_init (data, LOOKUP_POSITIVE, &nd))
	    error = path_walk (data, &nd);
#endif
	if (error)
	    ddev = ERR_PTR(error);
	else
	    ddev = nd.dentry;

	if (!IS_ERR(ddev)) {
	    minordevice = MINOR(ddev->d_inode->i_rdev);
	    dput (ddev);

	    if (minordevice >= NNNPFS) {
		return NULL;
	    }
	}
    }

    nnpfs[minordevice].status |= NNPFS_MOUNTED;
    nnpfs[minordevice].sb = sb;
    nnpfs[minordevice].root = 0;
    VFS_SET_NNPFS(sb, &nnpfs[minordevice]);
    
    sb->s_op = &nnpfs_sops;
    make_root(sb);
    sb->s_blocksize = 1024;
    sb->s_blocksize_bits = 10;
    sb->s_maxbytes = (1ULL<<63) - 1;

    NNPFSDEB(XDEBVFOPS, ("nnpfs_read_super: returning\n"));

    return sb;
}

#ifdef LINUX2_5
static int
nnpfs_fill_super (struct super_block *sb, void *data, int silent)
{
    struct super_block *ret = nnpfs_read_super(sb, data, silent);
    if (ret == NULL)
        return -1; /* XXX */
    return 0;
}

struct super_block *
nnpfs_get_sb (struct file_system_type *fs_type,
	      int flags, const char *dev_name, void *data) 
{
    return get_sb_nodev(fs_type, flags, data, nnpfs_fill_super);
}
#endif

static void
nnpfs_write_super(struct super_block * sb)
{
    sb->s_dirt = 0;
}

static void
nnpfs_put_super(struct super_block *sb)
{
    struct nnpfs *nnpfsp = VFS_TO_NNPFS(sb);

    NNPFSDEB(XDEBVFOPS, ("nnpfs_put_super starting\n"));
    nnpfsp->status &= ~NNPFS_MOUNTED;
    nnpfsp->status &= ~NNPFS_ROOTINSTALLED;
    sb->s_dev = 0;
    nnpfsp->root = NULL;
    NNPFSDEB(XDEBVFOPS, ("nnpfs_put_super exiting\n"));
}

static void
nnpfs_read_inode(struct inode *inode)
{
    NNPFSDEB(XDEBVFOPS,("nnpfs_read_inode starting inode: %p\n",inode));
    NNPFSDEB(XDEBVFOPS,("nnpfs_read_inode inode->i_ino: %ld\n",inode->i_ino));
    inode->i_blksize = 1024;
    inode->i_mode = 0;
    inode->i_op = NULL;
    NNPFSDEB(XDEBVFOPS,("nnpfs_read_inode exiting\n"));
}
 
/*
 * Called `inode' is de-referenced, ie when iput is called.
 */

static void
nnpfs_put_inode(struct inode *inode)
{
    NNPFSDEB(XDEBVFOPS,("nnpfs_put_inode: inode: %p count: %d aliases:",
			inode, nnpfs_icount(inode)));
    nnpfs_print_aliases(inode);

    if (nnpfs_icount(inode) == 1) {
	struct nnpfs_node *xn = VNODE_TO_XNODE(inode);

	inode->i_nlink = 0;

	if (xn) {
	    if (xn->flags & NNPFS_STALE) {
		struct nnpfs *nnpfsp = NNPFS_FROM_XNODE(xn);
		nnpfs_force_invalid_xnode (xn);

		down(&nnpfsp->inactive_sem);	
		nnpfs_queue_inactive(xn);
		up(&nnpfsp->inactive_sem);	
	    }
	    xn->flags &= ~NNPFS_STALE;
	}
    }
}

/*
 * `inode' is about to die.  add it to the list of nodes to be inactivated.
 * this is not done immediately to avoid allocating memory in a function
 * that's called when memory is scarce.
 */

static void
nnpfs_delete_inode(struct inode *inode)
{
    struct nnpfs_node *xn = VNODE_TO_XNODE(inode);

    NNPFSDEB(XDEBNODE,
	     ("nnpfs_delete_inode inode = 0x%p i_ino:0x%lx i_count:%u\n",
	      inode, inode->i_ino, nnpfs_icount(inode)));
    
    if (xn != NULL) {
	struct nnpfs *nnpfsp = NNPFS_FROM_XNODE(xn);

	down(&nnpfsp->inactive_sem);	

	clear_nnpfs_node (xn);
	nnpfs_queue_inactive(xn);

	up(&nnpfsp->inactive_sem);	
    }
    clear_inode(inode);
}

void
vattr2nnpfs_attr(const struct iattr *iattr, struct nnpfs_attr *attr)
{
    XA_CLEAR(attr);
    if (iattr->ia_valid & ATTR_MODE)
	XA_SET_MODE(attr,iattr->ia_mode);
    if (iattr->ia_valid & ATTR_UID)
	XA_SET_UID(attr,iattr->ia_uid);
    if (iattr->ia_valid & ATTR_GID)
	XA_SET_GID(attr,iattr->ia_gid);
    if (iattr->ia_valid & ATTR_ATIME)
	XA_SET_ATIME(attr, NNPFS_GET_TIME_SEC(iattr->ia_atime));
    if (iattr->ia_valid & ATTR_MTIME)
	XA_SET_MTIME(attr, NNPFS_GET_TIME_SEC(iattr->ia_mtime));
    if (iattr->ia_valid & ATTR_CTIME)
	XA_SET_CTIME(attr, NNPFS_GET_TIME_SEC(iattr->ia_ctime));
    if (iattr->ia_valid & ATTR_SIZE)
	XA_SET_SIZE(attr,iattr->ia_size);
}

#ifdef LINUX2_5
static int nnpfs_statfs(struct super_block *sb, struct kstatfs *buf)
{
    struct kstatfs tmp;
#else
static int nnpfs_statfs(struct super_block *sb, struct statfs *buf)
{
    struct statfs tmp;
#endif
    
    tmp.f_type    = 0x47114711;
    tmp.f_bsize   = sb->s_blocksize;
    tmp.f_blocks  = 1024*1024*2;
    tmp.f_bfree   = 1024*1024*2-100;
    tmp.f_bavail  = 1024*1024*2-50;
    tmp.f_files   = 1024*1024;
    tmp.f_ffree   = 1024*1024-100;
    tmp.f_namelen = NAME_MAX;
    *buf = tmp;
    return 0;
}
