/*
 * Copyright (c) 1995 - 2002, 2004 - 2005 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * NNPFS operations.
 */

#include <nnpfs/nnpfs_locl.h>
#include <nnpfs/nnpfs_message.h>
#include <nnpfs/nnpfs_common.h>
#include <nnpfs/nnpfs_fs.h>
#include <nnpfs/nnpfs_dev.h>
#include <nnpfs/nnpfs_deb.h>
#include <nnpfs/nnpfs_syscalls.h>
#include <nnpfs/nnpfs_vnodeops.h>

RCSID("$Id: nnpfs_vnodeops-common.c,v 1.113 2005/11/21 11:25:40 tol Exp $");

static void
nnpfs_handle_stale(struct nnpfs_node *xn)
{
#if __APPLE__
    struct vnode *vp = XNODE_TO_VNODE(xn);
#endif

    if ((xn->flags & NNPFS_STALE) == 0)
	return;

#if 0
    if (UBCISVALID(vp) && !ubc_isinuse(vp, 1)) {
	xn->flags &= ~NNPFS_STALE;
	ubc_setsize(vp, 0);
	NNPFS_TOKEN_CLEAR(xn, ~0,
			NNPFS_OPEN_MASK | NNPFS_ATTR_MASK |
			NNPFS_DATA_MASK | NNPFS_LOCK_MASK);
    }
#endif
}

int
nnpfs_open_valid(struct vnode *vp, nnpfs_vfs_context ctx, u_int tok)
{
    struct nnpfs *nnpfsp = NNPFS_FROM_VNODE(vp);
    struct nnpfs_node *xn = VNODE_TO_XNODE(vp);
    struct ucred *cred = nnpfs_vfs_context_ucred(ctx);
    int error = 0;

    NNPFSDEB(XDEBVFOPS, ("nnpfs_open_valid\n"));

    nnpfs_handle_stale(xn);

    do {
	if (!NNPFS_TOKEN_GOT(xn, tok)) {
	    struct nnpfs_message_open msg;

	    msg.header.opcode = NNPFS_MSG_OPEN;
	    msg.cred.uid = cred->cr_uid;
	    msg.cred.pag = nnpfs_get_pag(cred);
	    msg.handle = xn->handle;
	    msg.tokens = tok;

	    error = nnpfs_message_rpc(nnpfsp->fd, &msg.header,
				      sizeof(msg), nnpfs_vfs_context_proc(ctx));
	    if (error == 0)
		error = NNPFS_MSG_WAKEUP_ERROR(&msg);

	} else {
	    goto done;
	}
    } while (error == 0);

done:
    NNPFSDEB(XDEBVFOPS, ("nnpfs_open_valid: error = %d\n", error));

    return error;
}

int
nnpfs_attr_valid(struct vnode *vp, struct ucred *cred, d_thread_t *p,
		 u_int tok)
{
    struct nnpfs *nnpfsp = NNPFS_FROM_VNODE(vp);
    struct nnpfs_node *xn = VNODE_TO_XNODE(vp);
    int error = 0;
    nnpfs_pag_t pag = nnpfs_get_pag(cred);

    do {
	if (!NNPFS_TOKEN_GOT(xn, tok) || !nnpfs_has_pag(xn, pag)) {
	    struct nnpfs_message_getattr msg;

	    msg.header.opcode = NNPFS_MSG_GETATTR;
	    msg.cred.uid = cred->cr_uid;
	    msg.cred.pag = pag;
	    msg.handle = xn->handle;
	    error = nnpfs_message_rpc(nnpfsp->fd, &msg.header, sizeof(msg), p);
	    if (error == 0)
		error = NNPFS_MSG_WAKEUP_ERROR(&msg);
	} else {
	    goto done;
	}
    } while (error == 0);

done:
    return error;
}

int
nnpfs_data_valid(struct vnode *vp, nnpfs_cred *cred,
		 u_int tok, off_t want_offset)
{
    struct nnpfs *nnpfsp = NNPFS_FROM_VNODE(vp);
    struct nnpfs_node *xn = VNODE_TO_XNODE(vp);
    int error = 0;
    off_t offset;
    struct nnpfs_message_getdata msg;

    do {
	offset = want_offset;
	if (NNPFS_TOKEN_GOT(xn, tok|NNPFS_ATTR_R) && offset > nnpfs_vattr_get_size(&xn->attr)) {
	    offset = nnpfs_vattr_get_size(&xn->attr);
	}
    
	NNPFSDEB(XDEBVNOPS, ("nnpfs_data_valid: offset: want %ld has %ld, "
			   "tokens: want %lx has %lx length: %ld\n",
			   (long) offset, (long) xn->offset,
			   (long) tok, (long) xn->tokens,
			   (long) nnpfs_vattr_get_size(&xn->attr)));

	if (NNPFS_TOKEN_GOT(xn, tok)) {
	    if (offset <= xn->offset || xn->attr.va_type == VDIR) {
		break;
	    }
	}

	msg.header.opcode = NNPFS_MSG_GETDATA;
	msg.cred = *cred;
	msg.handle = xn->handle;
	msg.tokens = tok;
	msg.offset = offset;
	
	error = nnpfs_message_rpc(nnpfsp->fd, &msg.header, sizeof(msg),
				  nnpfs_curproc());
	if (error == 0)
	    error = NNPFS_MSG_WAKEUP_ERROR(&msg);
	
    } while (error == 0);

    return error;
}

int
nnpfs_open_common(struct vnode *vp,
		  int mode,
		  nnpfs_vfs_context ctx)
{
    struct nnpfs_node *xn = VNODE_TO_XNODE(vp);
    struct ucred *cred = nnpfs_vfs_context_ucred(ctx);
    int ret;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_open\n"));

    if (mode & FWRITE) {
	ret = nnpfs_open_valid(vp, ctx, NNPFS_OPEN_NW);
#ifdef __APPLE__
	xn->writers++;
#endif	
    } else {
	ret = nnpfs_open_valid(vp, ctx, NNPFS_OPEN_NR);
    }

    /* always update the read cred */

    if (mode & FWRITE)
	nnpfs_setcred(&xn->wr_cred, cred);
    nnpfs_setcred(&xn->rd_cred, cred);

    return ret;
}

static int
do_fsync(struct nnpfs *nnpfsp,
         struct nnpfs_node *xn,
         struct ucred *cred,
         nnpfs_cred *ncred,
         d_thread_t *p,
         u_int flag)
{
    struct nnpfs_message_putdata msg;
    int error;

    msg.header.opcode = NNPFS_MSG_PUTDATA;
    msg.handle = xn->handle;
    vattr2nnpfs_attr(&xn->attr, &msg.attr);
    msg.flag   = flag;

    if (ncred != NULL)
	msg.cred = *ncred;
    else
	nnpfs_setcred(&msg.cred, cred);

    error = nnpfs_message_rpc(nnpfsp->fd, &msg.header, sizeof(msg), p);
    if (error == 0)
	error = NNPFS_MSG_WAKEUP_ERROR(&msg);

    if (error == 0)
	xn->flags &= ~NNPFS_DATA_DIRTY;

    return error;
}

int
nnpfs_fsync_common(struct vnode *vp, struct ucred *cred, nnpfs_cred *ncred,
		   int waitfor, d_thread_t *proc)
{
    struct nnpfs *nnpfsp = NNPFS_FROM_VNODE(vp);
    struct nnpfs_node *xn = VNODE_TO_XNODE(vp);
    int error = 0;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_fsync: %lx\n", (unsigned long)vp));

    /*
     * It seems that fsync is sometimes called after reclaiming a node.
     * In that case we just look happy.
     */

    if (xn == NULL) {
	printf("NNPFS PANIC WARNING! nnpfs_fsync called after reclaiming!\n");
	return 0;
    }
    
    nnpfs_pushdirty(vp);

    if (xn->flags & NNPFS_DATA_DIRTY) {
#ifdef FSYNC_RECLAIM
	/* writing back the data from this vnode failed */
	if (waitfor & FSYNC_RECLAIM) {
	    printf("nnpfs_fsync: data lost, failed to write back\n");
	    xn->flags &= ~NNPFS_DATA_DIRTY;
	    return 0;
	}
#endif    
	error = do_fsync(nnpfsp, xn, cred, ncred, proc, NNPFS_WRITE | NNPFS_FSYNC);
    }

    return error;
}

int
nnpfs_close_common(struct vnode *vp, int fflag,
		   d_thread_t *proc, struct ucred *cred)
{
    struct nnpfs *nnpfsp = NNPFS_FROM_VNODE(vp);
    struct nnpfs_node *xn = VNODE_TO_XNODE(vp);
    int error = 0;
    
    NNPFSDEB(XDEBVNOPS,
	     ("nnpfs_close cred = %lx, fflag = %x, xn->flags = %x\n",
	      (unsigned long)cred, fflag, xn->flags));
    
    if (nnpfs_vnode_isreg(vp))
        nnpfs_pushdirty(vp);
    
    if (fflag & FWRITE && xn->flags & NNPFS_DATA_DIRTY) {
	error = do_fsync(nnpfsp, xn, cred, NULL, proc, NNPFS_WRITE);
	
#ifdef __APPLE__
	if (--xn->writers < 0)
	    panic("xn -ve writers");
#endif	
    }
    
    return error;
}

off_t
nnpfs_uio_end_length (struct uio *uio)
{
#if defined(DIAGNOSTIC) && !defined(__APPLE__)
    size_t sz = 0;
    int i;

    for (i = 0; i < uio->uio_iovcnt; i++)
	sz += uio->uio_iov[i].iov_len;
    if (sz != uio->uio_resid)
	panic("nnpfs_uio_end_length");
#endif
    return nnpfs_uio_offset(uio) + nnpfs_uio_resid(uio);
}


int
nnpfs_read_common(struct vnode *vp, struct uio *uio, int ioflag, struct ucred *cred)
{
    int error = 0;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_read\n"));

    nnpfs_setcred(&(VNODE_TO_XNODE(vp))->rd_cred, cred); 

    error = nnpfs_data_valid(vp, &(VNODE_TO_XNODE(vp))->rd_cred,
			     NNPFS_DATA_R, nnpfs_uio_end_length(uio));

    if (error == 0) {
	struct vnode *t = DATA_FROM_VNODE(vp);

#if defined(__APPLE__)
	nnpfs_vop_read(t, uio, ioflag, NULL, error);
#else
	nnpfs_vfs_readlock(t, nnpfs_uio_to_proc(uio));
	nnpfs_vop_read(t, uio, ioflag, cred, error);
	nnpfs_vfs_unlock(t, nnpfs_uio_to_proc(uio));
#endif
    }

    NNPFSDEB(XDEBVNOPS, ("nnpfs_read offset: %lu resid: %lu\n",
			 (unsigned long)nnpfs_uio_offset(uio),
			 (unsigned long)nnpfs_uio_resid(uio)));
    NNPFSDEB(XDEBVNOPS, ("nnpfs_read error: %d\n", error));

    return error;
}

int
nnpfs_write_common(struct vnode *vp, struct uio *uiop, int ioflag,
		   nnpfs_vfs_context ctx)
{
    struct ucred *cred = nnpfs_vfs_context_ucred(ctx);
    struct nnpfs_node *xn = VNODE_TO_XNODE(vp);
    int error = 0;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_write\n"));

    nnpfs_setcred(&xn->wr_cred, cred); 

    error = nnpfs_data_valid(vp, &xn->wr_cred, NNPFS_DATA_W,
			     nnpfs_vattr_get_size(&xn->attr));

    if (error == 0) {
	struct vnode *t = DATA_FROM_XNODE(xn);
	struct nnpfs_vfs_vattr sub_attr;
	int error2 = 0;
 
#if defined(__APPLE__)
	nnpfs_vop_write(t, uiop, ioflag, ctx, error);

	VATTR_INIT(&sub_attr);
	VATTR_WANTED(&sub_attr, va_data_size);
	VATTR_WANTED(&sub_attr, va_total_alloc);
	VATTR_WANTED(&sub_attr, va_modify_time);
#else
	nnpfs_vfs_writelock(t, nnpfs_vfs_context_proc(ctx));
	nnpfs_vop_write(t, uiop, ioflag, cred, error);
#endif

	VNODE_TO_XNODE(vp)->flags |= NNPFS_DATA_DIRTY;
	nnpfs_vop_getattr(t, &sub_attr, ctx, error2);
	if (error2 == 0) {
	    nnpfs_vattr_set_size(&xn->attr, nnpfs_vattr_get_size(&sub_attr));
	    nnpfs_vattr_set_bytes(&xn->attr, nnpfs_vattr_get_size(&sub_attr));
	    nnpfs_vattr_set_mtime(&xn->attr,
				  nnpfs_vattr_get_mtime_sec(&sub_attr),
				  nnpfs_vattr_get_mtime_nsec(&sub_attr));

            nnpfs_set_vp_size(vp, nnpfs_vattr_get_size(&sub_attr));
	    xn->offset = nnpfs_vattr_get_size(&sub_attr);
	}
	nnpfs_vfs_unlock(t, nnpfs_vfs_context_proc(ctx));
    }

    return error;
}

int
nnpfs_getattr_common(struct vnode *vp, struct nnpfs_vfs_vattr *vap,
		     struct ucred *cred, d_thread_t *p)
{
    int error = 0;

    struct nnpfs_node *xn = VNODE_TO_XNODE(vp);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_getattr\n"));

    error = nnpfs_attr_valid(vp, cred, p, NNPFS_ATTR_R);
    if (error == 0)
	*vap = xn->attr;

    return error;
}

static int
setattr_is_noop(struct nnpfs_node *xn, struct nnpfs_vfs_vattr *vap)
{
#ifdef __APPLE__
    
#define CHECK_NNPFSATTR(A) (!VATTR_IS_ACTIVE(vap, A) || vap->A == xn->attr.A)
    if (CHECK_NNPFSATTR(va_mode) &&
	CHECK_NNPFSATTR(va_nlink) &&
	CHECK_NNPFSATTR(va_data_size) &&
	CHECK_NNPFSATTR(va_uid) &&
	CHECK_NNPFSATTR(va_gid) &&
	CHECK_NNPFSATTR(va_fileid) && /* we ignore va_type */
	(!VATTR_IS_ACTIVE(vap, va_modify_time)
	 || vap->va_modify_time.tv_sec == xn->attr.va_modify_time.tv_sec))
	return 1;
#undef CHECK_NNPFSATTR

#else

#define CHECK_NNPFSATTR(A, cast) (vap->A == cast VNOVAL || vap->A == xn->attr.A)
    if (CHECK_NNPFSATTR(va_mode,(mode_t)) &&
	CHECK_NNPFSATTR(va_nlink,(short)) &&
	CHECK_NNPFSATTR(va_size,(va_size_t)) &&
	CHECK_NNPFSATTR(va_uid,(uid_t)) &&
	CHECK_NNPFSATTR(va_gid,(gid_t)) &&
	CHECK_NNPFSATTR(va_mtime.tv_sec,(time_t)) &&
	CHECK_NNPFSATTR(va_fileid,(long)) &&
	CHECK_NNPFSATTR(va_type,(enum vtype)))
	return 1;
#undef CHECK_NNPFSATTR

#endif /* ! __APPLE__ */

    return 0;
}

int
nnpfs_setattr_common(struct vnode *vp, struct nnpfs_vfs_vattr *vap,
		     struct ucred *cred, d_thread_t *p)
{
    struct nnpfs *nnpfsp = NNPFS_FROM_VNODE(vp);
    struct nnpfs_node *xn = VNODE_TO_XNODE(vp);
    int error = 0;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_setattr\n"));

    if (setattr_is_noop(xn, vap))
	return 0; /* nothing to do */

    if (NNPFS_TOKEN_GOT(xn, NNPFS_ATTR_W)) {
	/* Update attributes and mark them dirty. */
	VNODE_TO_XNODE(vp)->flags |= NNPFS_ATTR_DIRTY;
	error = EINVAL;		       /* XXX not yet implemented */
	goto done;
    } else {
	struct nnpfs_message_putattr msg;

	msg.header.opcode = NNPFS_MSG_PUTATTR;
	nnpfs_setcred(&msg.cred, cred);
	msg.handle = xn->handle;
	vattr2nnpfs_attr(vap, &msg.attr);

	if (NNPFS_TOKEN_GOT(xn, NNPFS_DATA_R)) {
	    if (nnpfs_vnode_isreg(vp)) {
		if (nnpfs_vattr_size_isactive(vap))
		    XA_SET_SIZE(&msg.attr, nnpfs_vattr_get_size(vap));
		else
		    XA_SET_SIZE(&msg.attr, nnpfs_vattr_get_size(&xn->attr));
	    }

	    if (nnpfs_vattr_mtime_isactive(vap))
		XA_SET_MTIME(&msg.attr, nnpfs_vattr_get_mtime_sec(vap));
	    else
		XA_SET_MTIME(&msg.attr, nnpfs_vattr_get_mtime_sec(&xn->attr));
	}
	
	error = nnpfs_message_rpc(nnpfsp->fd, &msg.header, sizeof(msg), p);
	if (error == 0)
	    error = NNPFS_MSG_WAKEUP_ERROR(&msg);
    }

done:
    NNPFS_VN_KNOTE(vp, NOTE_ATTRIB);

    return error;
}

static int
check_rights (nnpfs_rights rights, int mode, int dirp)
{
    int error = 0;

#ifdef __APPLE__
    /* VNOP_ACCESS passes a kauth action instead of ordinary unix mode */

    if (dirp) {
	if (mode & (KAUTH_VNODE_LIST_DIRECTORY | KAUTH_VNODE_SEARCH))
	    if ((rights & NNPFS_RIGHT_X) == 0)
		error = EACCES;
	if (mode & KAUTH_VNODE_DELETE_CHILD)
	    if ((rights & NNPFS_RIGHT_AD) == 0)
		error = EACCES;
	if (mode & (KAUTH_VNODE_ADD_FILE | KAUTH_VNODE_ADD_SUBDIRECTORY))
	    if ((rights & NNPFS_RIGHT_AI) == 0)
		error = EACCES;

	/* XXX can't check KAUTH_VNODE_DELETE, have no info */

    } else {
	if (mode & (KAUTH_VNODE_READ_DATA | KAUTH_VNODE_READ_ATTRIBUTES))
	    if ((rights & NNPFS_RIGHT_AR) == 0)
		error = EACCES;
	if (mode & (KAUTH_VNODE_WRITE_DATA | KAUTH_VNODE_APPEND_DATA | KAUTH_VNODE_WRITE_ATTRIBUTES | KAUTH_VNODE_TAKE_OWNERSHIP))
	    if ((rights & NNPFS_RIGHT_W) == 0)
		error = EACCES;
	if (mode & KAUTH_VNODE_EXECUTE)
	    if ((rights & NNPFS_RIGHT_X) == 0)
		error = EACCES;
	if (mode & KAUTH_VNODE_DELETE)
	    if ((rights & NNPFS_RIGHT_AD) == 0)
		error = EACCES;
    }
    
#if 0
KAUTH_VNODE_READ_EXTATTRIBUTES
KAUTH_VNODE_WRITE_EXTATTRIBUTES
KAUTH_VNODE_READ_SECURITY
KAUTH_VNODE_WRITE_SECURITY
KAUTH_VNODE_SYNCHRONIZE notused
KAUTH_VNODE_LINKTARGET like insert, but for target???
KAUTH_VNODE_CHECKIMMUTABLE always ok
KAUTH_VNODE_ACCESS (advisory)
KAUTH_VNODE_NOIMMUTABLE ?
#endif

#else /* !__APPLE__ */

    if (mode & VREAD)
	if ((rights & NNPFS_RIGHT_R) == 0)
	    error = EACCES;
    if (mode & VWRITE)
	if ((rights & NNPFS_RIGHT_W) == 0)
	    error = EACCES;
    if (mode & VEXEC)
	if ((rights & NNPFS_RIGHT_X) == 0)
	    error = EACCES;

#endif /* !__APPLE__ */

    return error;
}

int
nnpfs_access_common(struct vnode *vp, int mode, struct ucred *cred,
		    d_thread_t *p)
{
    int dirp = nnpfs_vnode_isdir(vp);

    int error = 0;
    nnpfs_pag_t pag = nnpfs_get_pag(cred);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_access mode = 0%o\n", mode));

    error = nnpfs_attr_valid(vp, cred, p, NNPFS_ATTR_R);
    if (error == 0) {
	struct nnpfs_node *xn = VNODE_TO_XNODE(vp);
	int i;

	error = check_rights(xn->anonrights, mode, dirp);

	if (error == 0)
	    goto done;

	NNPFSDEB(XDEBVNOPS, ("nnpfs_access anonaccess failed\n"));

	error = EACCES;		/* default to EACCES if pag isn't in xn->id */

	for (i = 0; i < NNPFS_MAXRIGHTS; i++)
	    if (xn->id[i] == pag) {
		error = check_rights(xn->rights[i], mode, dirp);
		break;
	    }
    }

done:
    NNPFSDEB(XDEBVNOPS, ("nnpfs_access(0%o) = %d\n", mode, error));

    return error;
}

int
nnpfs_lookup_common(struct vnode *dvp, 
		    nnpfs_componentname *cnp, 
		    struct vnode **vpp,
		    nnpfs_vfs_context ctx)
{
    struct nnpfs_message_getnode msg;
    struct ucred *cred = nnpfs_vfs_context_ucred(ctx);
    d_thread_t *p  = nnpfs_vfs_context_proc(ctx);
    struct nnpfs *nnpfsp = NNPFS_FROM_VNODE(dvp);
    struct nnpfs_node *d = VNODE_TO_XNODE(dvp);
    int error = 0;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_lookup_common: enter\n"));

    *vpp = NULL;

    if (cnp->cn_namelen >= NNPFS_MAX_NAME)
	return ENAMETOOLONG;
	
    if (!nnpfs_vnode_isdir(dvp))
	return ENOTDIR;

    if (cnp->cn_namelen == 1 && cnp->cn_nameptr[0] == '.') {
	*vpp = dvp;
#ifdef __APPLE__
	nnpfs_do_vget(*vpp, 0 /* XXX flag */, NULL /*proc */);
#else
	nnpfs_vref(*vpp);
#endif
	return 0;
    }
    
    do {
	nnpfs_lookup_access(dvp, ctx, p, error);
	if (error != 0)
	    goto done;

	NNPFSDEB(XDEBVNOPS, ("nnpfs_lookup_common: dvp = %lx\n",
			   (unsigned long) dvp));
	NNPFSDEB(XDEBVNOPS, ("nnpfs_lookup_common: cnp = %lx, "
			   "cnp->cn_nameiop = %d\n", 
			   (unsigned long) cnp, (int)cnp->cn_nameiop));
	

	error = nnpfs_dnlc_lookup(dvp, cnp, vpp);
	if (error == 0) {

	    /*
	     * Doesn't quite work.
	     */

#if 0
	    if ((cnp->cn_nameiop == CREATE || cnp->cn_nameiop == RENAME)
		&& (cnp->cn_flags & ISLASTCN)) {
		error = EJUSTRETURN;
		goto done;
	    }
#endif

	    msg.header.opcode = NNPFS_MSG_GETNODE;
	    nnpfs_setcred(&msg.cred, cred);
	    msg.parent_handle = d->handle;
	    memcpy(msg.name, cnp->cn_nameptr, cnp->cn_namelen);
	    msg.name[cnp->cn_namelen] = '\0';
	    error = nnpfs_message_rpc(nnpfsp->fd, &msg.header, sizeof(msg), p);
	    if (error == 0)
		error = NNPFS_MSG_WAKEUP_ERROR(&msg);
	    if(error == ENOENT && cnp->cn_nameiop != CREATE) {
		NNPFSDEB(XDEBVNOPS, ("nnpfs_lookup: neg cache %lx (%s, %ld)\n",
				   (unsigned long)dvp,
				   cnp->cn_nameptr, cnp->cn_namelen));
		nnpfs_dnlc_enter (dvp, cnp, NULL);
	    }
	} else if (error == -1) {
	    error = 0;
	    goto done;
	}
    } while (error == 0);

 done:
    NNPFS_VN_KNOTE(dvp, NOTE_WRITE);
    
    NNPFSDEB(XDEBVNOPS, ("nnpfs_lookup_common: return %d, vn %p\n",
			 error, *vpp));
    return error;
}

int
nnpfs_create_common(struct vnode *dvp,
		    const char *name,
		    struct nnpfs_vfs_vattr *vap, 
		    struct ucred *cred,
		    d_thread_t *p)
{
    struct nnpfs *nnpfsp = NNPFS_FROM_VNODE(dvp);
    struct nnpfs_node *xn = VNODE_TO_XNODE(dvp);
    int error = 0;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_create: (%lx, %s)\n",
		       (unsigned long)dvp, name));
    {
	struct nnpfs_message_create msg;

	msg.header.opcode = NNPFS_MSG_CREATE;
	msg.parent_handle = xn->handle;
	if (strlcpy(msg.name, name, sizeof(msg.name)) >= NNPFS_MAX_NAME)
	    return ENAMETOOLONG;
	vattr2nnpfs_attr(vap, &msg.attr);

	msg.mode = 0;		       /* XXX - mode */
	nnpfs_setcred(&msg.cred, cred);

#ifdef __APPLE__
	/* needed for subsequent writes to succeed with n */
	msg.attr.valid &= ~XA_V_UID; 
#endif

	error = nnpfs_message_rpc(nnpfsp->fd, &msg.header, sizeof(msg), p);
	if (error == 0)
	    error = NNPFS_MSG_WAKEUP_ERROR(&msg);
    }

#if 0
    if (error == EEXIST)
	error = 0;
#endif

    return error;
}

int
nnpfs_remove_common(struct vnode *dvp,
		    struct vnode *vp,
		    const char *name,
		    struct ucred *cred,
		    d_thread_t *p)
{
    struct nnpfs *nnpfsp  = NNPFS_FROM_VNODE(dvp);
    struct nnpfs_node *xn = VNODE_TO_XNODE(dvp);
    struct nnpfs_message_remove msg;
    int error;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_remove: %s\n", name));

    msg.header.opcode = NNPFS_MSG_REMOVE;
    msg.parent_handle = xn->handle;
    msg.cred.uid = cred->cr_uid;
    msg.cred.pag = nnpfs_get_pag(cred);
    
    if (strlcpy(msg.name, name, sizeof(msg.name)) >= NNPFS_MAX_NAME)
	error = ENAMETOOLONG;
    else
	error = nnpfs_message_rpc(nnpfsp->fd, &msg.header, sizeof(msg), p);
    if (error == 0)
	error = NNPFS_MSG_WAKEUP_ERROR(&msg);

    if (error == 0) {
	nnpfs_dnlc_purge (vp);
	NNPFS_VN_KNOTE(vp, NOTE_DELETE);
	NNPFS_VN_KNOTE(dvp, NOTE_WRITE);
    }

    return error;
}

int
nnpfs_rename_common(struct vnode *fdvp, 
		    struct vnode *fvp,
		    const char *fname,
		    struct vnode *tdvp,
		    struct vnode *tvp,
		    const char *tname,
		    struct ucred *cred,
		    d_thread_t *p)
{
    struct nnpfs *nnpfsp = NNPFS_FROM_VNODE(fdvp);
    int error;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_rename: %s %s\n", fname, tname));

#if 0
    if ((fvp->v_mount != tdvp->v_mount)
	|| (tvp && (fvp->v_mount != tvp->v_mount))) {
	return  EXDEV;
    }
#endif

    {
	struct nnpfs_message_rename msg;

	msg.header.opcode = NNPFS_MSG_RENAME;
	msg.old_parent_handle = VNODE_TO_XNODE(fdvp)->handle;
	if (strlcpy(msg.old_name, fname, sizeof(msg.old_name)) >= NNPFS_MAX_NAME)
	    return ENAMETOOLONG;
	msg.new_parent_handle = VNODE_TO_XNODE(tdvp)->handle;
	if (strlcpy(msg.new_name, tname, sizeof(msg.new_name)) >= NNPFS_MAX_NAME)
	    return ENAMETOOLONG;
	msg.cred.uid = cred->cr_uid;
	msg.cred.pag = nnpfs_get_pag(cred);
	error = nnpfs_message_rpc(nnpfsp->fd, &msg.header, sizeof(msg), p);
	if (error == 0)
	    error = ((struct nnpfs_message_wakeup *) &msg)->error;

    }

    NNPFS_VN_KNOTE(fdvp, NOTE_WRITE);
    NNPFS_VN_KNOTE(fvp, NOTE_WRITE);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_rename: error = %d\n", error));

    return error;
}

int
nnpfs_mkdir_common(struct vnode *dvp, 
		   const char *name,
		   struct nnpfs_vfs_vattr *vap, 
		   struct ucred *cred,
		   d_thread_t *p)
{
    struct nnpfs *nnpfsp = NNPFS_FROM_VNODE(dvp);
    struct nnpfs_node *xn = VNODE_TO_XNODE(dvp);
    int error = 0;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_mkdir: %s\n", name));
    {
	struct nnpfs_message_mkdir msg;

	msg.header.opcode = NNPFS_MSG_MKDIR;
	msg.parent_handle = xn->handle;
	if (strlcpy(msg.name, name, sizeof(msg.name)) >= NNPFS_MAX_NAME)
	    return ENAMETOOLONG;
	vattr2nnpfs_attr(vap, &msg.attr);
	nnpfs_setcred(&msg.cred, cred);
	error = nnpfs_message_rpc(nnpfsp->fd, &msg.header, sizeof(msg), p);
	if (error == 0)
	    error = NNPFS_MSG_WAKEUP_ERROR(&msg);
    }

    if (error == 0)
	NNPFS_VN_KNOTE(dvp, NOTE_WRITE | NOTE_LINK);

    return error;
}

int
nnpfs_rmdir_common(struct vnode *dvp,
		   struct vnode *vp,
		   const char *name,
		   struct ucred *cred,
		   d_thread_t *p)
{
    struct nnpfs *nnpfsp  = NNPFS_FROM_VNODE(dvp);
    struct nnpfs_node *xn = VNODE_TO_XNODE(dvp);
    struct nnpfs_message_rmdir msg;
    int error;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_rmdir: %s\n", name));

    msg.header.opcode = NNPFS_MSG_RMDIR;
    msg.parent_handle = xn->handle;
    msg.cred.uid = cred->cr_uid;
    msg.cred.pag = nnpfs_get_pag(cred);
    if (strlcpy(msg.name, name, sizeof(msg.name)) >= NNPFS_MAX_NAME)
	error = ENAMETOOLONG;
    else
	error = nnpfs_message_rpc(nnpfsp->fd, &msg.header, sizeof(msg), p);
    if (error == 0)
	error = ((struct nnpfs_message_wakeup *) &msg)->error;

    if (error == 0)
	nnpfs_dnlc_purge (vp);

    NNPFS_VN_KNOTE(dvp, NOTE_WRITE | NOTE_LINK);
    NNPFS_VN_KNOTE(vp, NOTE_DELETE);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_rmdir error: %d\n", error));

    return error;
}

int
nnpfs_readdir_common(struct vnode *vp, 
		     struct uio *uiop, 
		     int *eofflag,
		     nnpfs_vfs_context ctx)
{
    int error = 0;
    nnpfs_cred cred;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_readdir\n"));

    if(eofflag)
	*eofflag = 0;

    nnpfs_setcred(&cred, nnpfs_vfs_context_ucred(ctx));
    error = nnpfs_data_valid(vp, &cred, NNPFS_DATA_R,
			     nnpfs_uio_end_length(uiop));
    if (error == 0) {
	struct vnode *t = DATA_FROM_VNODE(vp);

	nnpfs_vfs_readlock(t, nnpfs_uio_to_proc(uiop));
	nnpfs_vop_read(t, uiop, 0, NULL, error);
	if (eofflag) {
	    struct nnpfs_vfs_vattr t_attr;
	    int error2;

#ifdef __APPLE__
	    VATTR_INIT(&t_attr);
	    VATTR_WANTED(&t_attr, va_data_size);
#endif
	    nnpfs_vop_getattr(t, &t_attr, ctx, error2);  /* XXX check bitmask */
	    if (error2 == 0)
	      *eofflag = nnpfs_vattr_get_size(&t_attr) <= nnpfs_uio_offset(uiop);
	}
	nnpfs_vfs_unlock(t, nnpfs_uio_to_proc(uiop));
    }
    return error;
}

int
nnpfs_link_common(struct vnode *dvp, 
		  struct vnode *vp, 
		  const char *name,
		  struct ucred *cred,
		  d_thread_t *p)
{
    struct nnpfs *nnpfsp = NNPFS_FROM_VNODE(dvp);
    struct nnpfs_node *xn = VNODE_TO_XNODE(dvp);
    struct nnpfs_node *xn2 = VNODE_TO_XNODE(vp);
    struct nnpfs_message_link msg;
    int error = 0;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_link: %s\n", name));
    
    msg.header.opcode = NNPFS_MSG_LINK;
    msg.parent_handle = xn->handle;
    msg.from_handle   = xn2->handle;
    if (strlcpy(msg.name, name, sizeof(msg.name)) >= NNPFS_MAX_NAME)
	return ENAMETOOLONG;
    msg.cred.uid = cred->cr_uid;
    msg.cred.pag = nnpfs_get_pag(cred);

    error = nnpfs_message_rpc(nnpfsp->fd, &msg.header, sizeof(msg), p);
    if (error == 0)
	error = NNPFS_MSG_WAKEUP_ERROR(&msg);
    
    NNPFS_VN_KNOTE(vp, NOTE_LINK);

    return error;
}

int
nnpfs_symlink_common(struct vnode *dvp,
		     struct vnode **vpp,
		     nnpfs_componentname *cnp,
		     struct nnpfs_vfs_vattr *vap,
		     char *target,
		     nnpfs_vfs_context ctx)
{
    struct nnpfs *nnpfsp = NNPFS_FROM_VNODE(dvp);
    struct nnpfs_node *xn = VNODE_TO_XNODE(dvp);
    struct ucred *cred = nnpfs_vfs_context_ucred(ctx);
    struct nnpfs_message_symlink msg;
    const char *name = cnp->cn_nameptr;
    int error = 0;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_symlink: %s\n", name));

    msg.header.opcode = NNPFS_MSG_SYMLINK;
    msg.parent_handle = xn->handle;
    vattr2nnpfs_attr(vap, &msg.attr);
    msg.cred.uid = cred->cr_uid;
    msg.cred.pag = nnpfs_get_pag(cred);
    if (strlcpy (msg.contents, target, sizeof(msg.contents)) >= NNPFS_MAX_SYMLINK_CONTENT) {
	error = ENAMETOOLONG;
	goto done;
    }
    if (strlcpy(msg.name, name, sizeof(msg.name)) >= NNPFS_MAX_NAME) {
	error = ENAMETOOLONG;
	goto done;
    }
    error = nnpfs_message_rpc(nnpfsp->fd, &msg.header, sizeof(msg),
			      nnpfs_vfs_context_proc(ctx));
    if (error == 0)
	error = NNPFS_MSG_WAKEUP_ERROR(&msg);

 done:
    NNPFS_VN_KNOTE(dvp, NOTE_WRITE);

    return error;
}

int
nnpfs_readlink_common(struct vnode *vp, struct uio *uiop, nnpfs_vfs_context ctx)
{
    int error = 0;
    nnpfs_cred cred;
    d_thread_t *proc = nnpfs_vfs_context_proc(ctx);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_readlink\n"));

    if (!nnpfs_vnode_islnk(vp)) {
	return EINVAL;
    }

    nnpfs_setcred(&cred, nnpfs_vfs_context_ucred(ctx));
    error = nnpfs_data_valid(vp, &cred, NNPFS_DATA_R, nnpfs_uio_end_length(uiop));
    if (error == 0) {
	struct vnode *t = DATA_FROM_VNODE(vp);

#if defined(__APPLE__)
	nnpfs_vop_read(t, uiop, 0, ctx, error);
#else
	nnpfs_vfs_readlock(t, proc);
	nnpfs_vop_read(t, uiop, 0, nnpfs_vfs_context_ucred(ctx), error);
	nnpfs_vfs_unlock(t, proc);
#endif
    }
    return error;
}

int
nnpfs_inactive_common(struct vnode *vp, d_thread_t *proc)
{
    int error;
    struct nnpfs_node *xn = VNODE_TO_XNODE(vp);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_inactive, %lx\n",
			 (unsigned long)vp));

    /*
     * This seems rather bogus, but sometimes we get an already
     * cleaned node to be made inactive.  Just ignoring it seems safe.
     */

    if (xn == NULL) {
	NNPFSDEB(XDEBVNOPS, ("nnpfs_inactive: clean node\n"));
	return 0;
    }

    /* xn->wr_cred not set -> NOCRED */

    if (nnpfs_vnode_isreg(vp))
	nnpfs_pushdirty(vp);

    error = nnpfs_fsync_common(vp, NULL, &xn->wr_cred, 0, proc);
    if (error) {
	printf ("nnpfs_inactive: failed writing back data: %d\n", error);
	xn->flags &= ~NNPFS_DATA_DIRTY;
    }

    /* If this node is no longer valid, recycle immediately. */
    if (!NNPFS_TOKEN_GOT(xn, NNPFS_ATTR_R | NNPFS_ATTR_W)
	|| (xn->flags & NNPFS_STALE) == NNPFS_STALE)
    {
	nnpfs_vfs_unlock(vp, proc);
        NNPFSDEB(XDEBVNOPS, ("nnpfs_inactive: vrecycle\n"));
        nnpfs_vrecycle(vp, 0, proc);
    } else {
	nnpfs_vfs_unlock(vp, proc);
	xn->flags &= ~NNPFS_STALE;
    }

    NNPFSDEB(XDEBVNOPS, ("return: nnpfs_inactive\n"));

    return 0;
}

int
nnpfs_reclaim_common(struct vnode *vp)
{
    struct nnpfs *nnpfsp = NNPFS_FROM_VNODE(vp);
    struct nnpfs_node *xn = VNODE_TO_XNODE(vp);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_reclaim: %lx\n",
			 (unsigned long)vp));

    nnpfs_dev_lock(nnpfsp);

    xn->flags |= NNPFS_LIMBO;

    NNPFS_TOKEN_CLEAR(xn,
		      ~0,
		      NNPFS_OPEN_MASK | NNPFS_ATTR_MASK |
		      NNPFS_DATA_MASK | NNPFS_LOCK_MASK);
    /* Release, data if we still have it. */
    if (DATA_FROM_XNODE(xn) != NULL) {
        nnpfs_vletgo(DATA_FROM_XNODE(xn));
	DATA_FROM_XNODE(xn) = NULL;
    }

    nnpfs_dnlc_purge(vp);

    NNPQUEUE_INSERT_HEAD(&nnpfsp->freehead, xn, nn_free);

    if (nnpfsp->status & CHANNEL_OPENED) {
	struct nnpfs_message_inactivenode msg;

	nnpfs_dev_unlock(nnpfsp);

	msg.header.opcode = NNPFS_MSG_INACTIVENODE;
	msg.handle = xn->handle;
	msg.flag   = NNPFS_NOREFS | NNPFS_DELETE;
	nnpfs_message_send(nnpfsp->fd, &msg.header, sizeof(msg));
    } else {
	nnpfs_free_node(nnpfsp, xn);
	nnpfs_dev_unlock(nnpfsp);
    }

#if 0
	vnode_removefsref(vp);
	cache_purge(vp);
	vnode_clearfsnode(vp);
#endif

    return 0;
}

/*
 *
 */

#if 0

int
nnpfs_advlock_common(struct vnode *dvp, 
		   int locktype,
		   unsigned long lockid, /* XXX this good ? */
		   struct ucred *cred)
{
    struct nnpfs *nnpfsp = NNPFS_FROM_VNODE(dvp);
    struct nnpfs_node *xn = VNODE_TO_XNODE(dvp);
    int error = 0;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_advlock\n"));
    {
	struct nnpfs_message_advlock msg;

	msg.header.opcode = NNPFS_MSG_ADVLOCK;
	msg.handle = xn->handle;
	msg.locktype = locktype;
	msg.lockid = lockid;

	nnpfs_setcred(&msg.cred, cred);
	error = nnpfs_message_rpc(nnpfsp->fd, &msg.header, sizeof(msg), 
				  nnpfs_vfs_context_proc(ctx));
	if (error == 0)
	    error = NNPFS_MSG_WAKEUP_ERROR(&msg);
    }

    if (error == 0) {
	
	/* sleep until woken */

    } else {

	/* die */
    }

    return error;
}

#endif

/*
 *
 */

void
nnpfs_printnode_common (struct vnode *vp)
{
    struct nnpfs_node *xn = VNODE_TO_XNODE(vp);

    printf ("xnode: fid: %d.%d.%d.%d\n", 
	    xn->handle.a, xn->handle.b, xn->handle.c, xn->handle.d);
    printf ("\tattr: %svalid\n", 
	    NNPFS_TOKEN_GOT(xn, NNPFS_ATTR_VALID) ? "": "in");
    printf ("\tdata: %svalid\n", 
	    NNPFS_TOKEN_GOT(xn, NNPFS_DATA_VALID) ? "": "in");
    printf ("\tflags: 0x%x\n", xn->flags);
    printf ("\toffset: %lu\n", (unsigned long)xn->offset);
}
