/*
 * Copyright (c) 1995 - 2006 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <nnpfs/nnpfs_locl.h>
#include <nnpfs/nnpfs_deb.h>
#include <nnpfs/nnpfs_fs.h>
#include <nnpfs/nnpfs_message.h>
#include <nnpfs/nnpfs_msg_locl.h>
#include <nnpfs/nnpfs_syscalls.h>
#include <nnpfs/nnpfs_vfsops.h>
#include <nnpfs/nnpfs_vnodeops.h>
#include <nnpfs/nnpfs_dev.h>

RCSID("$Id: nnpfs_message.c,v 1.102 2006/06/08 12:01:38 tol Exp $");

int
nnpfs_message_installroot(int fd,
			struct nnpfs_message_installroot * message,
			u_int size,
			d_thread_t *p)
{
    int error = 0;

    NNPFSDEB(XDEBMSG, ("nnpfs_message_installroot (%d,%d,%d,%d)\n",
		     message->node.handle.a,
		     message->node.handle.b,
		     message->node.handle.c,
		     message->node.handle.d));

    if (nnpfs_dev[fd].root != NULL) {
	printf("NNPFS PANIC nnpfs_message_installroot: called again!\n");
	error = EBUSY;
    } else {
	error = nnpfs_new_node(&nnpfs_dev[fd],
			       &message->node,
			       NULL,
			       &nnpfs_dev[fd].root,
			       p,
			       1 /* mark as root */);
    }

    NNPFSDEB(XDEBMSG, ("installroot returning %d\n", error));

    return error;
}

int
nnpfs_message_installnode(int fd,
			struct nnpfs_message_installnode * message,
			u_int size,
			d_thread_t *p)
{
    int error = 0;
    struct nnpfs_node *n, *dp;

    NNPFSDEB(XDEBMSG, ("nnpfs_message_installnode (%d,%d,%d,%d)\n",
		     message->node.handle.a,
		     message->node.handle.b,
		     message->node.handle.c,
		     message->node.handle.d));

retry:
    error = nnpfs_node_find(&nnpfs_dev[fd], &message->parent_handle, &dp);
    if (error) {
	if (error == EISDIR)
	    NNPFSDEB(XDEBMSG, ("installnode: parent node deleted\n"));
	else if (error == ENOENT)
	    printf("NNPFS PANIC WARNING! nnpfs_message_installnode: no parent\n");
	
	return error;
    }

    if (dp) {
	struct vnode *t_vnode = XNODE_TO_VNODE(dp);

	NNPFSDEB(XDEBMSG, ("nnpfs_message_installnode: t_vnode = %lx\n",
			   (unsigned long)t_vnode));

	if (nnpfs_do_vget(t_vnode, 0 /* LK_SHARED */, p))
		goto retry;

	error = nnpfs_new_node(&nnpfs_dev[fd], &message->node, NULL, &n, p, 0);
	if (error) {
	    nnpfs_vletgo(t_vnode);
	    return error;
	}

	nnpfs_dnlc_enter_name(t_vnode,
			      message->name,
			      XNODE_TO_VNODE(n));
	nnpfs_vletgo(XNODE_TO_VNODE(n));
	nnpfs_vletgo(t_vnode);
    } else {
	printf("NNPFS PANIC WARNING! nnpfs_message_installnode: no node\n");
	error = ENOENT;
    }

    NNPFSDEB(XDEBMSG, ("return: nnpfs_message_installnode: %d\n", error));

    return error;
}

int
nnpfs_message_installattr(int fd,
			struct nnpfs_message_installattr * message,
			u_int size,
			d_thread_t *p)
{
    int error = 0;
    struct nnpfs_node *t;

    NNPFSDEB(XDEBMSG, ("nnpfs_message_installattr (%d,%d,%d,%d) \n",
		       message->node.handle.a,
		       message->node.handle.b,
		       message->node.handle.c,
		       message->node.handle.d));

    error = nnpfs_node_find(&nnpfs_dev[fd], &message->node.handle, &t);
    if (error) {
	if (error == EISDIR) {
	    NNPFSDEB(XDEBMSG, ("nnpfs_message_installattr: node deleted\n"));
	} else if (error == ENOENT) {
	    NNPFSDEB(XDEBMSG, ("nnpfs_message_installattr: no such node\n"));
	    error = 0;
	}
	return error;
    }

    t->tokens = message->node.tokens;
    if ((t->tokens & NNPFS_DATA_MASK) && DATA_FROM_XNODE(t) == NULL) {
	printf("nnpfs_message_installattr: tokens and no data "
	       "(%d,%d,%d,%d) \n",
	       message->node.handle.a,
	       message->node.handle.b,
	       message->node.handle.c,
	       message->node.handle.d);
	t->tokens &= ~NNPFS_DATA_MASK;
    }
    nnpfs_attr2vattr(&message->node.attr, &t->attr, 0);
    if ((t->flags & NNPFS_VMOPEN) == 0)
	nnpfs_set_vp_size(XNODE_TO_VNODE(t), nnpfs_vattr_get_size(&t->attr));
    bcopy(message->node.id, t->id, sizeof(t->id));
    bcopy(message->node.rights, t->rights, sizeof(t->rights));
    t->anonrights = message->node.anonrights;
    
    return 0;
}

int
nnpfs_message_installdata(int fd,
			struct nnpfs_message_installdata * message,
			u_int size,
			d_thread_t *p)
{
    struct nnpfs_node *t;
    struct vnode *t_vnode, *vp;
    struct nnpfs_fhandle_t *fh;
    int error = 0;

    NNPFSDEB(XDEBMSG, ("nnpfs_message_installdata (%d,%d,%d,%d)\n",
		       message->node.handle.a,
		       message->node.handle.b,
		       message->node.handle.c,
		       message->node.handle.d));

 retry:
    error = nnpfs_node_find(&nnpfs_dev[fd], &message->node.handle, &t);
    if (error) {
	if (error == ENOENT) {
	    printf("NNPFS PANIC WARNING! nnpfs_message_installdata failed\n");
	    printf("Reason: No node to install the data into!\n");
	} else if (error == EISDIR) {
	    NNPFSDEB(XDEBMSG, ("installdata: node deleted\n"));
	}
	return error;
    }

    fh = (struct nnpfs_fhandle_t *)&message->cache_handle;
    t_vnode = XNODE_TO_VNODE(t);

    message->cache_name[sizeof(message->cache_name)-1] = '\0';
    NNPFSDEB(XDEBMSG, ("cache_name = '%s'\n", message->cache_name));

    if (nnpfs_do_vget(t_vnode, 0 /* LK_SHARED */, p))
	goto retry;

    if (message->flag & NNPFS_ID_HANDLE_VALID) {
	error = nnpfs_fhlookup (p, fh, &vp);
    } else {
	error = EINVAL;
    }
    if (error != 0) {
#ifdef __APPLE__ /* XXX */
	error = vnode_lookup(message->cache_name, 0 /* flags */, &vp, NULL /* ctx */);
#else
	struct nameidata nd;
	struct nameidata *ndp = &nd;

	NNPFSDEB(XDEBMSG,
		 ("nnpfs_message_installdata: fhlookup failed: %d, "
		  "opening by name\n", error));

	NDINIT(ndp, LOOKUP, FOLLOW | NNPFS_LOCKLEAF, UIO_SYSSPACE,
	       message->cache_name, p);
	error = namei(ndp);
	vp = ndp->ni_vp;
#endif
    }

    if (error == 0) {
	nnpfs_vfs_unlock(vp, p);

	if (DATA_FROM_XNODE(t))
	    nnpfs_vletgo(DATA_FROM_XNODE(t));
	DATA_FROM_XNODE(t) = vp;

	NNPFSDEB(XDEBMSG, ("nnpfs_message_installdata: t = %lx;"
			   " tokens = %x\n",
			   (unsigned long)t, message->node.tokens));

	t->tokens = message->node.tokens;
#if 0
	if ((t->tokens & NNPFS_DATA_MASK) == 0) {
	    printf("nnpfs_message_installdata: data but no tokens "
		   "(%d,%d,%d,%d) \n",
		   message->node.handle.a,
		   message->node.handle.b,
		   message->node.handle.c,
		   message->node.handle.d);
	    /* t->tokens |= NNPFS_DATA_R; */
	}
#endif
	nnpfs_attr2vattr(&message->node.attr, &t->attr, 1);
	if ((t->flags & NNPFS_VMOPEN) == 0)
	  nnpfs_set_vp_size(XNODE_TO_VNODE(t), nnpfs_vattr_get_size(&t->attr));
	if (nnpfs_vnode_isdir(XNODE_TO_VNODE(t))
	    && (message->flag & NNPFS_ID_INVALID_DNLC))
	    nnpfs_dnlc_purge (XNODE_TO_VNODE(t));
	bcopy(message->node.id, t->id, sizeof(t->id));
	bcopy(message->node.rights, t->rights, sizeof(t->rights));
	t->anonrights = message->node.anonrights;
	t->offset = message->offset;
#if 0
	if (message->flag & NNPFS_ID_AFSDIR)
	    t->flags |= NNPFS_AFSDIR;
#endif
    } else {
	printf("NNPFS PANIC WARNING! nnpfs_message_installdata failed!\n");
	printf("Reason: lookup failed on cache file '%s', error = %d\n",
	       message->cache_name, error);
    }
    nnpfs_vletgo(t_vnode);

    return error;
}

int
nnpfs_message_invalidnode(int fd,
			  struct nnpfs_message_invalidnode * message,
			  u_int size,
			  d_thread_t *p)
{
    struct nnpfs_node *t;
    struct vnode *vp;
    int error;

    NNPFSDEB(XDEBMSG, ("nnpfs_message_invalidnode (%d,%d,%d,%d)\n",
		       message->handle.a,
		       message->handle.b,
		       message->handle.c,
		       message->handle.d));

    error = nnpfs_node_find(&nnpfs_dev[fd], &message->handle, &t);
    if (error) {
	if (error == ENOENT)
	    NNPFSDEB(XDEBMSG, ("nnpfs_message_invalidnode: no such node\n"));
	else if (error == EISDIR)
	    NNPFSDEB(XDEBMSG, ("invalidnode: node deleted\n"));
	
	return error;
    }

    vp = XNODE_TO_VNODE(t);

    /* If open for writing, return immediately. Last close:er wins! */
#ifdef __APPLE__
    if (t->writers > 0)
	return 0;
#else
    if (vp->v_writecount >= 1)
	return 0;
#endif

#if defined(__FreeBSD__) || defined(__DragonFly__)
    {
	vm_object_t obj = vp->v_object;

	if (obj != NULL
	    && (obj->ref_count != 0
#ifdef OBJ_MIGHTBEDIRTY
		|| (obj->flags & OBJ_MIGHTBEDIRTY) != 0
#endif
		))
	    return 0;

    }
#endif /* __FreeBSD__ || __DragonFly__ */

    /* If node is in use, mark as stale */
    if (nnpfs_vnode_isinuse(vp, 0) && !nnpfs_vnode_isdir(vp)) {
	t->flags |= NNPFS_STALE;
	return 0;
    }
    
    /* printf("nnpfs_invalid: not used\n"); */
    NNPFS_TOKEN_CLEAR(t, ~0,
		      NNPFS_OPEN_MASK | NNPFS_ATTR_MASK |
		      NNPFS_DATA_MASK | NNPFS_LOCK_MASK);

    if (DATA_FROM_XNODE(t)) {
	nnpfs_vletgo(DATA_FROM_XNODE(t));
	DATA_FROM_XNODE(t) = (struct vnode *) NULL;
    }
    /* Dir changed, must invalidate DNLC. */
    if (nnpfs_vnode_isdir(vp))
	nnpfs_dnlc_purge(vp);
    if (!nnpfs_vnode_isinuse(vp, 0)) {
	NNPFSDEB(XDEBVNOPS, ("nnpfs_message_invalidnode: vrecycle\n"));
	nnpfs_vrecycle(vp, 0, p);
    }

    return 0;
}

int
nnpfs_message_updatefid(int fd,
		      struct nnpfs_message_updatefid * message,
		      u_int size,
		      d_thread_t *p)
{
    int error = 0;

    NNPFSDEB(XDEBMSG, ("nnpfs_message_updatefid (%d,%d,%d,%d) (%d,%d,%d,%d)\n",
		       message->old_handle.a,
		       message->old_handle.b,
		       message->old_handle.c,
		       message->old_handle.d,
		       message->new_handle.a,
		       message->new_handle.b,
		       message->new_handle.c,
		       message->new_handle.d));

    error = nnpfs_update_handle(&nnpfs_dev[fd], 
				&message->old_handle,
				&message->new_handle);
    if (error)
	printf ("NNPFS PANIC WARNING! nnpfs_message_updatefid: %d\n", error);
    return error;
}

/*
 * Try to clean out nodes for the userland daemon
 */

#ifdef __APPLE__
static void
gc_vnode (struct vnode *vp, d_thread_t *p)
{
    /* This node is on the freelist */
    if (!nnpfs_vnode_isinuse(vp, 0)) {
	NNPFSDEB(XDEBMSG, ("nnpfs_message_gc: success\n"));
	
	vnode_recycle(vp);
    } else {
	NNPFSDEB(XDEBMSG, ("nnpfs_message_gc: used\n"));
    }
}
#else /* !__APPLE__ */

static void
gc_vnode (struct vnode *vp, d_thread_t *p)
{
#ifdef __DragonFly__
    lwkt_tokref vlock;
    lwkt_gettoken(&vlock, vp->v_interlock);
#else
    nnpfs_interlock_lock(&vp->v_interlock);
#endif
    
    /* This node is on the freelist */
    if (vp->v_usecount <= 0) {
#if __FreeBSD__ || __DragonFly__
	vm_object_t obj;

	obj = vp->v_object;

	if (obj != NULL
	    && (obj->ref_count != 0
#ifdef OBJ_MIGHTBEDIRTY
		|| (obj->flags & OBJ_MIGHTBEDIRTY) != 0
#endif
		)) {
#ifdef __DragonFly__
            lwkt_reltoken(&vlock);
#else
	    nnpfs_interlock_unlock(&vp->v_interlock);
#endif
	    return;
	}
#endif /* __FreeBSD__ || __DragonFly__*/
	
	if (vp->v_usecount < 0 || vp->v_writecount != 0) {
		    nnpfs_vprint("vrele: bad ref count", vp);
		    panic("vrele: ref cnt");
	}
	
	NNPFSDEB(XDEBMSG, ("nnpfs_message_gc: success\n"));
	
#ifdef __DragonFly__
        vgonel (vp, &vlock, p);
#elif HAVE_KERNEL_VGONEL
	vgonel (vp, p);
#else /* !have vgonel nor is dragonfly */
	nnpfs_interlock_unlock(&vp->v_interlock);
	vgone (vp);
#endif

    } else {
#ifdef __DragonFly__
        lwkt_reltoken(&vlock);
#else
	nnpfs_interlock_unlock(&vp->v_interlock);
#endif
	NNPFSDEB(XDEBMSG, ("nnpfs_message_gc: used\n"));
    }
}
#endif /* !__APPLE__ */

int
nnpfs_message_gc_nodes(int fd,
		       struct nnpfs_message_gc_nodes *message,
		       u_int size,
		       d_thread_t *p)
{
    struct nnpfs_node *node;
    int i;

    NNPFSDEB(XDEBMSG, ("nnpfs_message_gc\n"));

    for (i = 0; i < message->len; i++) {
	int error = nnpfs_node_find(&nnpfs_dev[fd], &message->handle[i], &node);
	if (error) {
	    if (error == ENOENT)
		NNPFSDEB(XDEBMSG, ("PANIC gc_nodes node not found\n"));
	    else if (error == EISDIR)
		NNPFSDEB(XDEBMSG, ("gcnode: node deleted\n"));
	    continue;
	}
	gc_vnode(XNODE_TO_VNODE(node), p);
    }

    return 0;
}


/*
 * Probe what version of nnpfs this support
 */

int
nnpfs_message_version(int fd,
		    struct nnpfs_message_version *message,
		    u_int size,
		    d_thread_t *p)
{
    struct nnpfs_message_wakeup msg;
    int ret;

    ret = NNPFS_VERSION;

    msg.header.opcode = NNPFS_MSG_WAKEUP;
    msg.sleepers_sequence_num = message->header.sequence_num;
    msg.error = ret;

    return nnpfs_message_send(fd, 
			      (struct nnpfs_message_header *) &msg,
			      sizeof(msg));
}

/*
 *
 */

int
nnpfs_message_delete_node(int fd,
			  struct nnpfs_message_delete_node *message,
			  u_int size,
			  d_thread_t *p)
{
    struct nnpfs_node *node;
    struct nnpfs *nnpfsp = &nnpfs_dev[fd];
    int ret;

    NNPFSDEB(XDEBMSG, ("nnpfs_message_delete_node\n"));

    nnpfs_dev_lock(nnpfsp);

    ret = nnpfs_node_find_locked(nnpfsp, &message->handle, &node);
    if (ret == ENOENT) {
	printf("nnpfs_message_delete_node: no such node\n");
	nnpfs_dev_unlock(nnpfsp);
	return ENOENT;
    }
    NNPFSDEB(XDEBMSG, ("delete_node: %p\n", node));
    NNPFSDEB(XDEBMSG, ("delete_node: flags 0x%x\n", node->flags));
    if (node->flags & NNPFS_LIMBO) {
	NNPFSDEB(XDEBMSG, ("delete_node: free node\n"));
	nnpfs_free_node(&nnpfs_dev[fd], node);
    } else {
	NNPFSDEB(XDEBMSG, ("delete_node: not deleted"));
    }
    nnpfs_dev_unlock(nnpfsp);

    return 0;
}
