/******************************************************************************
This file is part of the argtable library. It contains a simple example program
showing how to use the argtable library to implement multiple alternate command
line usages.

Copyright (C) 1998,1999 Stewart Heitmann (Stewart.Heitmann@tip.csiro.au)

The argtable library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

This software is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
USA.


How it works
------------
This program implements three alternate argument usages, namely,
   (1) progname circle <radius>
   (2) progname triangle <base> <height>
   (3) progname rectangle <width> <height>
which calculate the area of a circle, triangle and rectangle respectively.
Each of the three alternatives are implemented as a single argument table.
The literal arguments (circle, triangle, or rectangle) are a convenient way to
identify which of the three argument usages is being used.
The program works by trying each of the argument tables in turn until either
a successful match is made, or all three tables have failed. This approach is
viable because the literal arguments make the usages unique and thus mutually
exclusive.
Should all the tables fail to scan the arguments, the program makes no attempt
to guess which of the usages the user was trying to apply. Rather, it just
prints the error messages from each of the failed attempts, and lets the user
decide which of the three messages is appropriate.   
******************************************************************************/

#include "argtable.h"

double radius, base, height;

int main(int argc, char**argv)
  {
  /* first argument table, "progname circle <radius>" */
  arg_rec argtable1[] =
     {  
     /*-tag--name-------type-----storage--default--descrip --*/
     { NULL, "circle",  arg_lit,  NULL,     NULL,  NULL },
     { NULL, "<radius>",arg_dbl,  &radius,  NULL,  "\t radius of the circle" },
     };
  const size_t narg1 = sizeof(argtable1)/sizeof(arg_rec);

  /* second argument table, "progname triangle <base> <height>" */
  arg_rec argtable2[] =
     {  
     /*-tag--name--------type----storage--default--descrip --*/
     { NULL, "triangle", arg_lit,  NULL,    NULL,  NULL },
     { NULL, "<base>",   arg_dbl,  &base,   NULL,  "\t base of the triangle" },
     { NULL, "<height>", arg_dbl,  &height, NULL,  "\t height of the triangle or rectangle" },
     };
  const size_t narg2 = sizeof(argtable2)/sizeof(arg_rec);

  /* third argument table, "progname rectangle <base> <height>" */
  arg_rec argtable3[] =
     { /*-tag--name--------type----storage--default--descrip --*/
     { NULL,  "rectangle",arg_lit,  NULL,    NULL,  NULL },
     { NULL,  "<width>",  arg_dbl,  &base,   NULL,  "\t width of the rectangle" },
     { NULL,  "<height>", arg_dbl,  &height, NULL,  NULL },
     };
  const size_t narg3 = sizeof(argtable3)/sizeof(arg_rec);

  int chosen_table;

  /*-- process the command line args --*/
  if (argc==1)
     {
     /*-- no command line arguments, so display program usage and exit. --*/
     printf("This program calculates the area of (1) a circle, or (2) a triangle or (3) a rectangle.\n");
     printf("Usages:\n");
     printf("   (1) %s %s\n", argv[0], arg_syntax(argtable1,narg1));
     printf("   (2) %s %s\n", argv[0], arg_syntax(argtable2,narg2));
     printf("   (3) %s %s\n", argv[0], arg_syntax(argtable3,narg3));
     printf("where:\n");
     printf(arg_glossary(argtable1,narg1,"   "));
     printf(arg_glossary(argtable2,narg2,"   "));
     printf(arg_glossary(argtable3,narg3,"   "));
     return 0;
     }
  else
     {
     /*-- scan command line arguments from argv[] as per each argtablex[] --*/
     char cmdline[200];
     char errmsg1[200], errmark1[200]; 
     char errmsg2[200], errmark2[200]; 
     char errmsg3[200], errmark3[200]; 

     if (arg_scanargv(argc,argv,argtable1,narg1,cmdline,errmsg1,errmark1))
        chosen_table = 1;
     else if (arg_scanargv(argc,argv,argtable2,narg2,NULL,errmsg2,errmark2))
        chosen_table = 2;
     else if (arg_scanargv(argc,argv,argtable3,narg3,NULL,errmsg3,errmark3))
        chosen_table = 3;
     else
	{
	/*-- scan errors occurred on all tables, print error messages and exit --*/
        printf("ERROR: %s\n", cmdline);
        printf("   (1) %s %s\n", errmark1, errmsg1);
        printf("or (2) %s %s\n", errmark2, errmsg2);
        printf("or (3) %s %s\n", errmark3, errmsg3);
        return 1;
	}
     }

  /*-- act according to which argument table was satisfied --*/
  switch(chosen_table)
    {
    case 1:
       printf("area of circle = %f\n", radius*radius*3.14159);
       break;
    case 2:
       printf("area of triangle = %f\n", 0.5*base*height);
       break;
    case 3:
       printf("area of rectangle = %f\n", base*height);
       break;
    default:
       return 1;
    }
 
  return 0; 
  }


