/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000-2004 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * This module contains the routines which process the runtime
 * configuration file ~/.aldrc
 *
 * $Id: rc.c,v 1.1.1.1 2004/04/26 00:40:41 pa33 Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>

#include "alddefs.h"
#include "command.h"
#include "rc.h"
#include "set.h"
#include "version.h"

/*
 * libString includes
 */
#include "Strn.h"

/*
initRC()
  Initialize a runtime commands workspace

Return: pointer to new workspace
*/

struct rcWorkspace *
initRC()

{
  struct rcWorkspace *ws;
  char *homepath;

  homepath = getenv("HOME");

  if (!homepath)
  {
    fprintf(stderr,
            "initRC: warning: cannot determine location of configuration file\n");
    return (0);
  }

  ws = (struct rcWorkspace *) malloc(sizeof(struct rcWorkspace));
  if (!ws)
  {
    fprintf(stderr, "initRC: malloc failed: %s\n", strerror(errno));
    return (0);
  }

  sprintf(ws->filename, "%s/%s", homepath, RC_FILENAME);

  return (ws);
} /* initRC() */

/*
termRC()
  Terminate a run commands workspace

Inputs: ws - run commands workspace
*/

void
termRC(struct rcWorkspace *ws)

{
  if (!ws)
    return;

  free(ws);
} /* termRC() */

/*
readRC()
  Read and process run commands file

Inputs: ws - pointer to ald workspace

Return: 1 upon success
        0 upon failure
*/

int
readRC(struct aldWorkspace *ws)

{
  FILE *fp;
  char buf[MAXLINE],
       tmpbuf[MAXLINE];
  struct Command *cptr;
  unsigned int flags;
  int ac;
  char **av;

  if (!ws->rcWorkspace_p)
    return (0);

  fp = fopen(ws->rcWorkspace_p->filename, "r");
  if (!fp)
  {
    if (errno != ENOENT)
    {
      fprintf(stderr,
              "readRC: unable to open %s: %s\n",
              ws->rcWorkspace_p->filename,
              strerror(errno));
    }

    return (0);
  }

  while (fgets(buf, MAXLINE, fp))
  {
    if (*buf == '#')
      continue;  /* comment */

    ac = SplitBuffer(buf, &av);
    if (ac <= 1)
      continue;

    if (!Strcasecmp(av[0], "set"))
    {
      cptr = FindSetCommand(av[1], &flags);
      if (!cptr)
        continue;

      /*
       * Call the appropriate set function
       */
      (*cptr->funcptr)(ws, ac, av, 0, tmpbuf);
    }

    free(av);
  } /* while (fgets(buf, MAXLINE, fp)) */

  fclose(fp);

  return (1);
} /* readRC() */

/*
writeRC()
  Write settings to configuration file

Inputs: ws - ald workspace

Return: 1 upon success
        0 upon failure
*/

int
writeRC(struct aldWorkspace *ws)

{
  FILE *fp;

  if (!ws->rcWorkspace_p)
    return (0);

  fp = fopen(ws->rcWorkspace_p->filename, "w");
  if (!fp)
  {
    fprintf(stderr,
            "writeRC: unable to open %s: %s\n",
            ws->rcWorkspace_p->filename,
            strerror(errno));
    return (0);
  }

  fprintf(fp,
          "# Generated by ald %s. Do not edit.\n",
          aVersion);
  fprintf(fp,
          "set disasm-show-syms %s\n",
          IsSetDisasmShowSyms(ws) ? "on" : "off");
  fprintf(fp,
          "set pause-print %s\n",
          ws->printWorkspace_p->PausePrint ? "on" : "off");
  fprintf(fp,
          "set prompt \"%s\"\n",
          ws->commandWorkspace_p->CmdPrompt);
  fprintf(fp,
          "set step-display-regs %s\n",
          IsSetStepDisplayRegs(ws) ? "on" : "off");
  fprintf(fp,
          "set step-display-fpregs %s\n",
          IsSetStepDisplayFpRegs(ws) ? "on" : "off");
  fprintf(fp,
          "set step-display-mmxregs %s\n",
          IsSetStepDisplayMmxRegs(ws) ? "on" : "off");

  fclose(fp);

  return (1);
} /* writeRC() */
