#include "X.h"
#include "xf86.h"
#include "xf86Priv.h"

#include "xf86_OSlib.h"
#include "xf86OSpriv.h"

#include <sys/ioctl.h>
#include <sys/vm.h>
#include <machine/vm.h>

#define DEV_VIDEO "/dev/video"

static Bool useDevVideo = FALSE;
static int devVideoFd= -1;

static void checkDevVideo(Bool warn);
static pointer mapVidMem(int, unsigned long, unsigned long, int);
static void unmapVidMem(int, pointer, unsigned long);
static void *mapmem(int fd, off_t base, size_t size, int readonly);

#define VMUNKNOWN	-1
#define VMNO		0
#define VMYES		1

static int vmmode = VMUNKNOWN;

void
xf86OSInitVidMem(VidMemInfoPtr pVidMem)
{
	checkDevVideo(TRUE);
	pVidMem->linearSupported = useDevVideo;
	pVidMem->mapMem = mapVidMem;
	pVidMem->unmapMem = unmapVidMem;

	pVidMem->initialised = TRUE;
}

/*
 * Check if /dev/video can be mmap'd.  If it can't print a warning when
 * "warn" is TRUE.
 */
static void
checkDevVideo(Bool warn)
{
	static Bool devVideoChecked = FALSE;
	int fd;
	pointer base;

	if (devVideoChecked)
	    return;
	devVideoChecked = TRUE;

	if ((fd = open(DEV_VIDEO, O_RDWR)) >= 0)
	{
		devVideoFd = fd;
		useDevVideo = TRUE;
		ErrorF("checkDevVideo: devVideoFd = %d\n", fd);
		return;
	}
	FatalError("checkDevVideo: unable to open '%s': %s",
		DEV_VIDEO, strerror(errno));
}

static pointer
mapVidMem(int ScreenNum, unsigned long Base, unsigned long Size, int flags)
{
	pointer base;

	checkDevVideo(FALSE);

	if (useDevVideo)
	{
	    if (devVideoFd < 0) 
	    {
		FatalError("xf86MapVidMem: failed to open %s (%s)",
			   DEV_VIDEO, strerror(errno));
	    }
	    base = mapmem(devVideoFd, Base, Size,  !!(flags & VIDMEM_READONLY));
	    if (base == NULL)
	    {
		struct stat st;
		int t_errno;

		t_errno= errno;

		ErrorF("mapmem failed on fd %d\n", devVideoFd);
		errno= t_errno;
		FatalError("%s: could not map %s [s=%lx,a=%lx] (%s)",
			   "xf86MapVidMem", DEV_VIDEO, Size, Base, 
			   strerror(errno));
	    }
	    return(base);
	}
		
	FatalError("mapVidMem: !useDevVideo?");
}

static void
unmapVidMem(int ScreenNum, pointer Base, unsigned long Size)
{
	int r;
	char *buf, **bpp;
	struct mapreq mapreq;
	struct mapreqvm vmmapreq;

	if(vmmode == VMUNKNOWN) {
		FatalError("unmapVidMem: no mapping done yet? vmmode unset");
	}

    if(vmmode == VMNO) {
	mapreq.base= Base;
	mapreq.size= Size;
	r= ioctl(devVideoFd, MIOCUNMAP, &mapreq);
	if (r == -1)
		FatalError("MIOCUNMAP failed: %s", strerror(errno));

	bpp= (char **)Base;
	buf= bpp[-1];

#if 0
	ErrorF("unmapVidMem: freeing 0x%p\n", buf);
#endif
	free(buf);
    }
    else {
	memset(&vmmapreq, 0, sizeof(vmmapreq));
	vmmapreq.flags = 0;
	vmmapreq.vaddr = (void *) Base;
	vmmapreq.size = Size;
	r= ioctl(devVideoFd, TIOCUNMAPMEM, &vmmapreq);
	if (r < 0) {
		FatalError("TIOCUNMAPMEM failed: %s", strerror(errno));
	}
    }
}

static void *mapmem(int fd, off_t base, size_t size, int readonly)
{
	size_t bufsize;
	int o, r, t_errno;
	char *abuf, *buf, **bpp;
	struct mapreq mapreq;
	int withvm = 0;

	/* Assume base and size are multiples of the page size */
	if (base % I386_PAGE_SIZE)
	{
		FatalError(
			"mapmem: base (0x%x) not a multiple of the page size",
			base);
	}
	if (size % I386_PAGE_SIZE)
	{
		FatalError(
			"mapmem: size (0x%x) not a multiple of the page size",
			size);
	}

#ifdef TIOCMAPMEM
	if(vmmode != VMNO) {
		struct mapreqvm mapreqvm;
		memset(&mapreqvm, 0, sizeof(mapreqvm));
		mapreqvm.flags = 0;
		mapreqvm.phys_offset = base;
		mapreqvm.size = size;
		mapreqvm.readonly = readonly;
		mapreqvm.vaddr_ret = NULL;
	
		if((r = ioctl(fd, TIOCMAPMEM, &mapreqvm)) < 0) {
			ErrorF("TIOCMAPMEM failed: %d\n", t_errno);
		} else {
			buf = mapreqvm.vaddr_ret;
			if(buf == NULL)
				ErrorF("TIOCMAPMEM returned NULL\n");
			else
				vmmode = VMYES;
		}
	}
#endif

    if(vmmode != VMYES) {
	/* Add a page for alignment and room a point to the start of the
	 * malloced buffer.
	 */
	bufsize= I386_PAGE_SIZE+size + sizeof(abuf);
	abuf= malloc(bufsize);
	if (abuf == NULL)
		FatalError("mapmem: unable to allocate %u bytes", bufsize);
#if 0
	ErrorF("mapmem: got buffer size 0x%x at %p\n", bufsize, abuf);
#endif
	buf= abuf + sizeof(abuf);
	o= (unsigned)buf % I386_PAGE_SIZE;
	if (o)
		buf= (char *)buf + (I386_PAGE_SIZE-o);

	mapreq.base= buf;
	mapreq.size= size;
	mapreq.offset= base;
	mapreq.readonly= readonly;
	r= ioctl(fd, MIOCMAP, &mapreq);
	if (r == -1)
	{
		t_errno= errno;
		free(abuf);
		buf= NULL;
		ErrorF("MIOCMAP failed: %d\n", t_errno);
		errno= t_errno;
	}
	if (buf)
	{
		bpp= (char **)buf;
		bpp[-1]= abuf;
		vmmode = VMNO;
	}
    }

	return buf;
}
