#ifndef _GraphConfig_hh_
#define _GraphConfig_hh_

const int ScreenDepth = ColorDepth;
const int ScreenSizeX = 800;
const int ScreenSizeY = 600;

#if (ColorDepth==8)
#define Color8
#endif
#if (ColorDepth==16)
#define Color16
#endif
#if (ColorDepth==24)
#define Color24
#endif
#if (ColorDepth==32)
#define Color32
#endif

class Pixel_t {
  private:
    friend Pixel_t RGB(int, int, int);
#ifdef Color8
    unsigned char data;
#endif
#ifdef Color16
    unsigned short data16;
#endif
#ifdef Color24
    unsigned char b;
    unsigned char g;
    unsigned char r;
#endif
#ifdef Color32
    int data32;
#endif
  public:
    inline bool operator==(const Pixel_t& d) const;
    inline void operator&=(const Pixel_t& d);
    inline void operator|=(const Pixel_t& d);
    inline void AlphaBlend(const Pixel_t& d);
    inline void AlphaBlendDarker(const Pixel_t& d);
    inline bool IsZero() const;
    inline bool NotZero() const;
    inline int RPixel() const;
    inline int GPixel() const;
    inline int BPixel() const;
    inline int GetColor() const;
    inline void SetColor(int color);
    inline const int Data() const;
};

inline bool Pixel_t::IsZero() const
{
#ifdef Color32
  return (!data32);
#endif
#ifdef Color24
  return (!(r || g || b));
#endif
#ifdef Color16
  return (!data16);
#endif
#ifdef Color8
  return (!data);
#endif
}

inline bool Pixel_t::NotZero() const
{
#ifdef Color32
  return (data32);
#endif
#ifdef Color24
  return (r || g || b);
#endif
#ifdef Color16
  return (data16);
#endif
#ifdef Color8
  return (data);
#endif
}

inline bool Pixel_t::operator==(const Pixel_t& d) const
{
#ifdef Color32
  return (data32 == d.data32);
#endif
#ifdef Color24
  return ((d.r == r) && (d.g == g) && (d.b == b));
#endif
#ifdef Color16
  return (data16 == d.data16);
#endif
#ifdef Color8
  return (data == d.data);
#endif
}

inline void Pixel_t::operator&=(const Pixel_t& d)
{
#ifdef Color32
  data32 &= d.data32;
#endif
#ifdef Color24
  r &= d.r;
  g &= d.g;
  b &= d.b;
#endif
#ifdef Color16
  data16 &= d.data16;
#endif
#ifdef Color8
  data &= d.data;
#endif
}

inline void Pixel_t::operator|=(const Pixel_t& d)
{
#ifdef Color32
  data32 |= d.data32;
#endif
#ifdef Color24
  r |= d.r;
  g |= d.g;
  b |= d.b;
#endif
#ifdef Color16
  data16 |= d.data16;
#endif
#ifdef Color8
  data |= d.data;
#endif
}

// conversion TRUEColor => Pixel_t
inline Pixel_t RGB(int r, int g, int b)
{
  Pixel_t q;
#ifdef Color32
  q.data32 = (r << 16) | (g << 8) | b;
#endif
#ifdef Color24
  q.r = r;
  q.g = g;
  q.b = b;
#endif
#ifdef Color16
  q.data16 = ((r & 0xf8) << 8) | ((g & 0xfc) << 3) | (b >> 3);
#endif
#ifdef Color8
  q.data = (r & 0xe0) | ((g >> 3) & 0x1e) | (b >> 6);
#endif
  return q;
}

inline void Pixel_t::AlphaBlendDarker(const Pixel_t& d)
{
  if (d.NotZero()) {
#ifdef Color32
    data32 &= 0x00f8f8f8;
    data32 >>= 3;
#endif
#ifdef Color24
    r >>= 3;
    g >>= 3;
    b >>= 3;
#endif
#ifdef Color16
    data16 &= 0xe79c;
    data16 >>= 2;
#endif
#ifdef Color8
    data &= 0xda;
    data >>= 1;    
#endif
    this->operator|=(d);
  }
}

inline void Pixel_t::AlphaBlend(const Pixel_t& d)
{
  if (d.NotZero()) {
#ifdef Color32
    data32 &= 0x00fcfcfc;
    data32 >>= 2;
#endif
#ifdef Color24
    r >>= 2;
    g >>= 2;
    b >>= 2;
#endif
#ifdef Color16
    data16 &= 0xf7dc; 
    data16 >>= 1;
#endif
#ifdef Color8
    data &= 0xda;
    data >>= 1;    
#endif
    this->operator|=(d);
  }
}

inline int Pixel_t::RPixel() const
{
#ifdef Color32
   return (data32 >> 16) & 0xff;
#endif
#ifdef Color24
   return r;
#endif
#ifdef Color16
  return ((data16 & 0xf800) >> 8);
#endif
#ifdef Color8
   return (data & 0xe0);
#endif
}

inline int Pixel_t::GPixel() const
{
#ifdef Color32
  return (data32 >> 8) & 0xff;
#endif
#ifdef Color24
  return g;
#endif
#ifdef Color16
  return ((data16 & 0x07e0) >> 3);
#endif
#ifdef Color8
  return ((data & 0x1c) << 3);
#endif
}

inline int Pixel_t::BPixel() const
{
#ifdef Color32
  return (data32 & 0xff);
#endif
#ifdef Color24
  return b;
#endif
#ifdef Color16
  return ((data16 & 0x001f) << 3);
#endif
#ifdef Color8
  return ((data & 0x03) << 6);
#endif
}

inline int Pixel_t::GetColor() const
{
#ifdef Color32
  return data32;
#endif
#ifdef Color24
  return (r << 16) | (g << 8) | b;
#endif
#ifdef Color16
  return data16;
#endif
#ifdef Color8
  return data;
#endif
}

inline void Pixel_t::SetColor(int color)
{
#ifdef Color32
  data32 = color;
#endif
#ifdef Color24
  r = (color >> 16);
  g = (color >> 8);
  b = color;
#endif
#ifdef Color16
  data16 = color;
#endif
#ifdef Color8
  data = color;
#endif
}

inline const int Pixel_t::Data() const
{
  return *((const int *) this);
}

const int MaxColorValue = 0xff;

#ifdef Color8
const int MinColorValue = 0x40;
const int MinGColorValue = 0x20; // green has 3 bits
#endif
#ifdef Color16
const int MinColorValue = 8;
const int MinGColorValue = 4;
#endif
#ifdef Color24
const int MinColorValue = 1;
const int MinGColorValue = 1;
#endif
#ifdef Color32
const int MinColorValue = 1;
const int MinGColorValue = 1;
#endif

const Pixel_t Pixel_t_Zero = RGB(0, 0, 0);
const Pixel_t Pixel_t_White = RGB(MaxColorValue, MaxColorValue, MaxColorValue);
const Pixel_t Pixel_t_Black = RGB(0, MinGColorValue, 0);
const Pixel_t Pixel_t_HiRed = RGB(0xc0, 0, 0);
const Pixel_t Pixel_t_HiGreen = RGB(0, 0xc0, 0);
const Pixel_t Pixel_t_HiBlue = RGB(0, 0, 0xc0);
const Pixel_t Pixel_t_Red = RGB(0x80, 0, 0);
const Pixel_t Pixel_t_Green = RGB(0, 0x80, 0);
const Pixel_t Pixel_t_Blue = RGB(0, 0, 0x80);
const Pixel_t Pixel_t_DarkRed = RGB(0x40, 0, 0);
const Pixel_t Pixel_t_DarkGreen = RGB(0, 0x40, 0);
const Pixel_t Pixel_t_DarkBlue = RGB(0, 0, 0x40);
const Pixel_t Pixel_t_Yellow = RGB(0xff, 0xff, 20);

#endif






