#ifndef _NetMessage_hh_
#define _NetMessage_hh_

#include <string.h>
#include "Compat.hh"

typedef int StdInteger;
typedef bool StdBool;
typedef char StdChar;

enum MsgType_t2 {
  mtInitConnection,
  mtStartGame,
  mtStartRound,
  mtPlayerSetup,
  mtQuit,
  mtOptions,
  mtEndRound,
  mtEndGame,
  mtBreak,
  mtSetGameFPS,
  mtPlayerDead,
  mtStep,
  mtPlayerSetPos,
  mtChat
};

typedef StdInteger MsgType_t;
const MsgType_t MsgType_t_count = 14;

class PacketHeader {
  public:
    static const StdInteger MagicConst = 0x37349423;
    StdInteger Magic;
    StdInteger Number;
    StdBool ResendReply;
    StdInteger Size;
};

class NetMessage {
  protected:
      NetMessage(MsgType_t _type, StdInteger _size)
	: size(_size), type(_type) { };
  public:
    StdInteger size; // size of this message, size of this header is included
    const MsgType_t type; // type of message
};

class nmInitConnection : public NetMessage {
  public:
    // port is send only from server to client as reply
    const StdInteger Version;
    const StdBool accept;
    const StdInteger port;
    const StdInteger ScreenSizeX;
    const StdInteger ScreenSizeY;
    static const StdInteger CurrentVersion = 1;
    nmInitConnection(StdBool _accept, StdInteger _port, StdInteger X, StdInteger Y)
      : NetMessage(mtInitConnection, sizeof(nmInitConnection)),
    Version(CurrentVersion), accept(_accept), port(_port), ScreenSizeX(X), ScreenSizeY(Y)
      { };
};

class nmStartGame : public NetMessage {
  // as request from client to server
  // from server to client, after all clients sent StartGame
  public:
    const StdInteger rounds;
    nmStartGame(StdInteger _rounds)
      : NetMessage(mtStartGame, sizeof(nmStartGame)), rounds(_rounds) { };
};

class nmStartRound : public NetMessage {
  public:
    nmStartRound() : NetMessage(mtStartRound, sizeof(nmStartRound)) { };
  // from client to server as request
  // from server to client, after all clients sent StartRound
};

class nmPlayerSetup : public NetMessage {
  // both ways
  public:
    static const int NameSize = 10;    
    const StdInteger PlayerNumber;
    const StdInteger Type;
    const StdInteger FillerMainType;
    const StdInteger FillerFirstOption;
    const StdInteger FillerSecondOption;
    StdChar Name[NameSize];  
    nmPlayerSetup(const StdInteger _PlayerNumber, const StdInteger _Type, const StdInteger _FillerMainType,
		  const StdInteger _FillerFirstOption, const StdInteger _FillerSecondOption, const StdChar _Name[NameSize])
      : NetMessage(mtPlayerSetup, sizeof(nmPlayerSetup)),
    PlayerNumber(_PlayerNumber), Type(_Type), FillerMainType(_FillerMainType),
    FillerFirstOption(_FillerFirstOption), FillerSecondOption(_FillerSecondOption)
      { strcpy(Name, _Name);};
};

class nmQuit : public NetMessage {
  public:
    // sends client to server before exiting
    nmQuit() : NetMessage(mtQuit, sizeof(nmQuit)) { };
};

class nmOptions : public NetMessage {
  public:
    const StdBool Surround;
    const StdInteger Speed;
    const StdInteger WormStepSize;
    const StdInteger WormTurnSize;
    const StdInteger Rounds;
    nmOptions(const StdBool _Surround, const StdInteger _Speed,
	      const StdInteger _WormStepSize, const StdInteger _WormTurnSize,
	      const StdInteger _Rounds)
      : NetMessage(mtOptions, sizeof(nmOptions)),
    Surround(_Surround), Speed(_Speed),
    WormStepSize(_WormStepSize), WormTurnSize(_WormTurnSize),
    Rounds(_Rounds) { };
};

class nmEndRound : public NetMessage {
  public:
    // sends client when is selected End round
    nmEndRound() : NetMessage(mtEndRound, sizeof(nmEndRound)) { };
};

class nmEndGame : public NetMessage {
  public:
    // sends client when is selected End game
    nmEndGame() : NetMessage(mtEndGame, sizeof(nmEndGame)) { };
};

class nmBreak : public NetMessage {
  public:
    // sends client when opens menu
    nmBreak() : NetMessage(mtBreak, sizeof(nmBreak)) { };
};

class nmSetGameFPS: public NetMessage {
  public:
    // sends client when when game speed is too fast/slow
    // sends server to other clients
    const StdInteger wantFPS;
    nmSetGameFPS(StdInteger _wantFPS)
      : NetMessage(mtSetGameFPS, sizeof(nmSetGameFPS)), wantFPS(_wantFPS) { };
};

class nmPlayerDead : public NetMessage {
  public:
    // sends client when any local player is dead
    // sends server to other clients
    const StdInteger PlayerNumber;
    const StdInteger KillerNumber;
    const StdInteger AddPoints; // == new players->score-1
    nmPlayerDead(StdInteger _PlayerNumber, const StdInteger _KillerNumber,
		 StdInteger _AddPoints)
      : NetMessage(mtPlayerDead, sizeof(nmPlayerDead)), PlayerNumber(_PlayerNumber),
      KillerNumber(_KillerNumber), AddPoints(_AddPoints) { };							
};

class nmStepItem {
  public:
    static const StdInteger DirLeft = 0;
    static const StdInteger DirUnpress = 1;
    static const StdInteger DirRight = 2; // must be the same numbers as in Worm.hh!!!
    StdInteger PlayerNumber;
    StdInteger Dir;
    void Set(StdInteger _PlayerNumber, StdInteger _Dir)
      { PlayerNumber = _PlayerNumber; Dir = _Dir; };
};

class nmStep : public NetMessage {
  public:
    static const int MAXArraySize = 150;
    // in fact array of (size-sizeof(NetMessage))/sizeof(nmStepItem)) elements
    // see GetCount()
    nmStepItem array[MAXArraySize]; // can't use const!!!
    nmStep(int Count) : NetMessage(mtStep, sizeof(NetMessage)+Count*sizeof(nmStepItem)) { };
    // sends client about its players to server
    // sends server to clients
    inline void Resize(int newCount);
    int GetCount() { return (size-sizeof(NetMessage))/sizeof(nmStepItem); };
};

inline void nmStep::Resize(int newCount)
{
  size = sizeof(NetMessage)+newCount*sizeof(nmStepItem);
}

class nmPlayerSetPos : public NetMessage {
  public:
    nmPlayerSetPos(StdInteger _PlayerNum, StdInteger _X, StdInteger _Y,
		   StdInteger _Angle)
      : NetMessage(mtPlayerSetPos, sizeof(nmPlayerSetPos)),
      PlayerNum(_PlayerNum), X(_X), Y(_Y), Angle(_Angle) { };
    // sends client about its players to server
    // sends server to clients
    const StdInteger PlayerNum;
    const StdInteger X;
    const StdInteger Y;
    const StdInteger Angle;
};

class nmChat : public NetMessage {
  public:
    static const int MsgSize = 60;
    char Text[MsgSize];
    nmChat(const char * _Text)
      : NetMessage(mtChat, sizeof(nmChat))
	{ strcpy(Text, _Text); };
};

#endif
