#include "NetConnector.hh"
#include "NetCTree.hh"
#include "Debug.hh"
#include "Utils.hh"
#include "NetworkTimings.hh"
#include <time.h>
#include <sys/time.h>
#include <unistd.h>

#define NetCD(x)
//#define NetCD(x) x
//#define DEBGTCD(x) x
#define DEBGTCD(x)

NetConnector::NetConnector(bool _IsServer, Connection * _connection,
			   int _ScreenSizeX, int _ScreenSizeY)
    : connection(_connection), tree(new NetCTree),
  out_buffer(new char[buffersize]), in_buffer(new char[buffersize]),
  save_out_buffer1(new char[buffersize]),
  cached_in_buffer(new char[buffersize]),
  bufpos(bufstart), save_bufsize1(0), cached_bufsize(0),
  PacketCounter(0), AlreadyReceived(FALSE), AlreadySent(FALSE), WaitCounter(0),
  IsServer(_IsServer), ScreenSizeX(_ScreenSizeX), ScreenSizeY(_ScreenSizeY),
  status(0)
{
}

NetConnector:: ~NetConnector()
{
  Disconnect();
  delete tree;
  delete out_buffer;
  delete in_buffer;
  delete save_out_buffer1;
  delete cached_in_buffer;
  delete connection;
}

void NetConnector::Reset()
{
  PacketCounter = 0;
  AlreadyReceived = FALSE;
  AlreadySent = FALSE;
}

void NetConnector::DeleteMessages()
{
  tree->DeleteMessages();
}

void NetConnector::SendMsg(NetMessage * msg)
{
  int size = msg->size;
  if ((bufpos+size) >= buffersize)
    return; // EEK!
  // there should be change order of bytes...
  memcpy(out_buffer + bufpos, msg, size);
  bufpos += size;
}

NetMessage * NetConnector::GetMsg(MsgType_t type)
{
  return tree->GetMsg(type);
}

bool NetConnector::ReceiveMessages(bool Wait)
{
  if (!NetworkGame())
    return TRUE;
  WaitCounter = 0;
  int ret;
  if (!AlreadyReceived) {
    for (;;) {
      // not cached in cached_in_buffer
      ret = connection->Receive(in_buffer, buffersize, Wait);
      if (ret<=0) {
	if (!Wait)
	  return FALSE;
	WaitCounter++;
	if (WaitCounter < NetConnectorRecvTimeout)
	  continue;
	if (!AlreadySent) // we didn't send yet => can't resend :)
	  return FALSE;
	// Ugh - can't receive message => send again
	ResendMessages();
	return FALSE;
      }
      PacketHeader * ph = (PacketHeader *) ((void *) in_buffer);
      DEBUGP("data packet: " << ph->Size << " " << ph->Number);    
      if ((ret<int(sizeof(PacketHeader))) || (ph->Magic != ph->MagicConst)
	  || (ph->Size != ret)) {
	Error("received bad data packet");
	continue; // ignore this packet
      }
      int n = ph->Number;
      // now decode ph->Number
      if (n == (PacketCounter-1)) {
	// previous packet => send previous cached packet
	if (ph->ResendReply) {
	  if (NetworkShowWarnings)
	    Print("got previous resend reply msg (ignoring)");
	  continue;
	}
	DEBGTCD(timeval x);
	DEBGTCD(gettimeofday(&x, 0));
	DEBGTCD(PrintErr << "resend prev: " << x.tv_usec << nl);
	if (NetworkShowWarnings)
	  PrintErr << "sending again previous msg " << PacketCounter << nl;
	PacketHeader * ph2 = (PacketHeader *) ((void *) save_out_buffer1);
	ph2->ResendReply = TRUE;
	if (!connection->Send(save_out_buffer1, save_bufsize1))
	  Error("can't send cached (previous) message");
	ph2->ResendReply = FALSE;
	continue;
      }
      if (n < PacketCounter) {
	PrintErr << "warning: received old packet " << n << " (now " << PacketCounter << ") (ignoring)" << nl;
	continue;
      }
      if (n == (PacketCounter+1)) {
	// next packet => cache and send again
	// exchange cached_in_buffer and in_buffer
	char * tmp = in_buffer;
	in_buffer = cached_in_buffer;
	cached_in_buffer = tmp;
	cached_bufsize = ret;
	// send current message again (need current frame, not next now!)
	if (NetworkShowWarnings)
	  PrintErr << " got next packet (expecting " << PacketCounter << "), sending again" << nl;
	if (!connection->Send(save_out_buffer1, save_bufsize1))
	  Error("can't send cached (current) message");
	continue;
      }
      if (n != PacketCounter) {
	PrintErr << "warning: unexpected message " << n << " (ignoring)" << nl;
	continue;
      }
      // YES! we got the right packet!
      if (cached_bufsize)
	// set flag about next message
	AlreadyReceived = TRUE;
      break;
    } // for(;;)
    NetCD(PrintErr << "got msg " << PacketCounter << nl);
  } else { // AlreadyReceived
    if (NetworkShowWarnings)
      PrintErr << "using cached msg " << PacketCounter << nl;
    // exchange cached_in_buffer and in_buffer, set ret to cached_bufsize
    char * tmp = cached_in_buffer;
    cached_in_buffer = in_buffer;
    in_buffer = tmp;
    ret = cached_bufsize;
    AlreadyReceived = FALSE;
  }
  // OK
  DEBUGP("received messages " << PacketCounter);
  ProcessMessages();
  if (IsServer)
    PacketCounter++;
  return TRUE;
}

void NetConnector::ResendMessages()
{
  DEBGTCD(timeval x);
  DEBGTCD(gettimeofday(&x, 0));
  DEBGTCD(PrintErr << "resend: " << x.tv_usec << nl);
  if (NetworkShowWarnings)
    PrintErr << "timeout - sending again message " << PacketCounter << nl;
  if (!connection->Send(save_out_buffer1, save_bufsize1))
    Error("can't resend current message");
}

bool NetConnector::ReceiveMessagesWaitMore()
{
  int t = NetConnectorRecvTries;
  while (t--)
    if (ReceiveMessages(TRUE))
      return TRUE;
  return FALSE;
}

bool NetConnector::SendMessages()
{
  if (!NetworkGame()) {
    bufpos = bufstart;
    return TRUE;
  }
  NetCD(PrintErr << "sending messages " << PacketCounter << nl);
  DEBGTCD(timeval x);
  DEBGTCD(gettimeofday(&x, 0));
  DEBGTCD(PrintErr << "send: " << x.tv_usec << nl);
  PacketHeader * ph = (PacketHeader *) ((void *) out_buffer);
  ph->Magic = ph->MagicConst;
  ph->Number = PacketCounter;
  ph->Size = bufpos;
  ph->ResendReply = FALSE;
  DEBUGP("sending messages" << PacketCounter);
  for (int q=0; q<NetConnectorSendTries; q++)
    if (connection->Send(out_buffer, bufpos)) {
      // exchange buffers out <=> save1
      char * tmp = out_buffer;
      out_buffer = save_out_buffer1;
      save_out_buffer1 = tmp;
      save_bufsize1 = bufpos;
      bufpos = bufstart;
      WaitCounter = 0;
      AlreadySent = TRUE;
      if (!IsServer)
	PacketCounter++;
      return TRUE;
    }
  return FALSE;
}

void NetConnector::ProcessMessages()
{
  PacketHeader * ph = (PacketHeader *) ((void *) in_buffer);
  int savesize = ph->Size-sizeof(PacketHeader);
  if (savesize)
    tree->PutMessages(ph+1, savesize); // EEK! ph+1
}

static char status_text[nmChat::MsgSize];

bool NetConnector::Connect()
{
  // Only for client!!!
  if (IsServer)
    return FALSE;
  SendMessages(); // flush old buffer
  bufpos = bufstart;
  bool ret = connection->Connect();
  if (!ret)
    return FALSE;
  nmInitConnection msg(FALSE, FALSE, ScreenSizeX, ScreenSizeY);
  for (int i=0; i<NetConnectorInitTries; i++) {
    Reset();
    DeleteMessages();
    SendMsg(&msg);
    if (SendMessages()) {
      for (int j=0; j<NetConnectorInitTries; j++)
	if (ReceiveMessagesWaitMore())
	  break;
      nmInitConnection * ret_msg = (nmInitConnection *) GetMsg(mtInitConnection);
      if (ret_msg) {
	if (ret_msg->Version != nmInitConnection::CurrentVersion) {
	  PrintErr << "server refused connection - incompatible versions of client/server "
	    "(client version: " << nmInitConnection::CurrentVersion << ", server "
	      "version " << ret_msg->Version << nl;
	  status = "Server refused connection - incompatible versions";
	  connection->Disconnect();
	  return FALSE;
	}
	if (ret_msg->accept) {
	  status = "Connected to server";
	  Print(status);
	  connection->SetPeerPort(ret_msg->port);
	  // success!!!
	  Reset();
	  if (connection->Connect() && SendMessages() &&
	      ReceiveMessagesWaitMore() && SendMessages())
	    return TRUE;
	  status = "Bad network connection to server";
	  Error(status);
	  continue;
	}
	if (ret_msg->ScreenSizeX) {
	  char text[100]; 
	  sprintf(text,
		   "Server refused connection: need screen size %ix%i",
		   ret_msg->ScreenSizeX, ret_msg->ScreenSizeY);
	  strncpy(status_text, text, nmChat::MsgSize-1);
	  status = status_text;
	  PrintErr << "error: server refused connection - invalid screen size"
	    << ScreenSizeX << "x" << ScreenSizeY << nl;
	  PrintErr << "other clients use " << ret_msg->ScreenSizeX <<
	    "x" << ret_msg->ScreenSizeY << " screen" << nl;
	} else {
	  status = "Server refused connection (too many clients)!";
	  Error(status);
	}
	connection->Disconnect();
	return FALSE;
      } else {
	status = "Bad response from server";
	Error(status);
      }
    }
  }
  status = "Initial connection to server timed out";
  Error(status);
  connection->Disconnect();
  return FALSE;
}

bool NetConnector::Disconnect()
{
  if (!NetworkGame())
    return TRUE;
  nmQuit nm;
  SendMsg(&nm);
  SendMessages();
  connection->Disconnect();
  if (!IsServer) {
    status = "Disconnected from server";
    Print(status);
  }
  return TRUE;
}
