#include "Fillers.hh"
#include "PaintInterface/TargaPicture.hh"
#include "GraphConfig.hh"
#include <math.h>
#include "Compat.hh"

static const int ColorDiv = 4;
static const int ColorStep = (MaxColorValue / (ColorDiv-1))-1;
static const int ColorsTotal = ColorDiv*ColorDiv*ColorDiv;

static Pixel_t GetCodedColor(int Number)
{
  while (Number<0)
    Number += ColorsTotal;
  Number %= ColorsTotal;
  if (!Number) // can't use color RGB(0, 0, 0)
    return RGB(MinColorValue, MinColorValue, MinColorValue); 
  int r = (Number%ColorDiv)*ColorStep;
  Number /= ColorDiv;
  int g = (Number%ColorDiv)*ColorStep;
  Number /= ColorDiv;
  int b = (Number%ColorDiv)*ColorStep;
  return RGB(r, g, b);
}

FlatFiller::FlatFiller(int _color=0)
    : Filler(ftFlat), color(_color)
{
  CreateImage();
}

void FlatFiller::CreateImage()
{
  Pixel_t c = GetCodedColor(color);
  Pixel_t im[ImageSize*ImageSize];
  // fill background
  for (int x=0; x<ImageSize; x++)
    for (int y=0; y<ImageSize; y++)
      im[x+y*ImageSize] = c;
  TileBuffer(im, ImageSize, ImageSize);  
}

CircleFiller::CircleFiller(int _circle_color=0, int _bg_color=0)
    : TwoOptFiller(ftCircle, _circle_color, _bg_color)
{
  CreateImage();
}

void CircleFiller::CreateImage()
{
  const int R = 32;
  Pixel_t im[R*R];
  // fill background
  Pixel_t bg_c = GetCodedColor(opt1);
  Pixel_t circle_c = GetCodedColor(opt2);
  for (int x=0; x<R; x++)
    for (int y=0; y<R; y++)
      im[x+R*y] = bg_c;
  // paint circle with size == R/2
  for (int r=1; r<(R/3); r++)
    for(int j=0; j<4*R; j++) {
      double ang = double(j)/R/2*M_PI;
      int x = (int) floor((cos(ang)*r)+R/2);
      int y = (int) floor((sin(ang)*r)+R/2);
      im[x+R*y] = circle_c;
    } 
  TileBuffer(im, R, R);
}
    
CheckerFiller::CheckerFiller(int _white=0, int _black=0)
    : TwoOptFiller(ftChecker, _white, _black)
{
  CreateImage();
}

void CheckerFiller::CreateImage()
{
  const int R = 32;
  Pixel_t im[R*R];
  // fill background
  Pixel_t w_c = GetCodedColor(opt1);
  Pixel_t b_c = GetCodedColor(opt2);
  for (int x=0; x<R; x++)
    for (int y=0; y<R; y++) {
      int i = (x%(R/4)<(R/8)) ? 0 : 1;
      if (y%(R/4)<(R/8))
	i = !i;
      im[x+R*y] = i ? w_c : b_c;
    }
  TileBuffer(im, R, R);
}

PictureFiller::PictureFiller(int _NumPictures, Picture ** _pictures, int _ActualPicture=0)
    : Filler(ftPicture), pictures(_pictures), NumPictures(_NumPictures), ActualPicture(_ActualPicture)
{
  CreateImage();
}

void PictureFiller::CreateImage()
{
  while (ActualPicture<0)
    ActualPicture += NumPictures;
  ActualPicture %= NumPictures;
  Picture * p = pictures[ActualPicture];
  TileBuffer(p->Data(), p->Width(), p->Height());
}

