#!perl

#
# test_command.t
# Copyright (C) 1997-2007 by John Heidemann
# $Id$
#
# This program is distributed under terms of the GNU general
# public license, version 2.  See the file COPYING
# in $dblibdir for details.
#

=head1 NAME

test_command.t - test fsdb commands

=head1 SYNOPSIS

test_command.t [-uv] [TEST/file.cmd]

=head1 OPTIONS

=over

=item B[-d]
Enable debugging output.

=item B[-v]
Enable verbose output.

=item B<--help>
Show help.

=item B<--man>
Show full manual.

=back

=cut

use Test::More;
use Pod::Usage;
use Getopt::Long;
use File::Copy qw(mv cp);

Getopt::Long::Configure ("bundling");
pod2usage(2) if ($#ARGV >= 0 && $ARGV[0] eq '-?');
#my(@orig_argv) = @ARGV;
my($prog) = $0;
my $debug = undef;
my $verbose = undef;
my $update = undef;
&GetOptions('d|debug+' => \$debug,   
 	'help' => sub { pod2usage(1); },
	'man' => sub { pod2usage(-verbose => 2); },
	'u|update!' => \$update,
         'v|verbose+' => \$verbose) or &pod2usage(2);

my $src_root = ".";
chdir $src_root or die "$0: cannot chdir $src_root\n";
foreach (qw(README Makefile.PL Makefile)) {
    die "test_command.t: must be run from the Fsdb source root directory.\n\t(can't find $_)\n"
	if (! -f $_);
};

#
# now all releative to src root:
#
my $test_dir = "TEST";
my $scripts_dir = "blib/script";
my $lib_dir = "blib/lib";

my $env_cmd = '';
if (defined($ENV{PERLLIB})) {
    # xxx: not portable separator
    $ENV{PERLLIB} = $lib_dir . ":" . $ENV{PERLLIB};
    $env_cmd .= "PERLLIB=$lib_dir:\$PERLLIB ";
} else {
    $ENV{PERLLIB} = $lib_dir;
    $env_cmd .= "PERLLIB=$lib_dir ";
};
$ENV{PATH} = "$scripts_dir:" . $ENV{PATH};
$env_cmd .= "PATH=$scripts_dir:\$PATH ";

#
# what to run?
#
my @TESTS = @ARGV;
if ($#TESTS == -1) {
    @TESTS = glob "$test_dir/*.cmd";
};

plan tests => ($#TESTS + 1);

foreach (@TESTS) {
    ok(run_test($_), $_);
};

exit 0;

sub parse_cmd_file {
    my($cmd_file) = @_;
    my %opts;
    open(CMD, "<$cmd_file") or die "$0: cannot read $cmd_file\n";
    while (<CMD>) {
	chomp;
	next if (/^\s*\#/);
	next if (/^\s*$/);
	my($key, $value) = /([^=]+)=(.*)$/;
	if (!defined($key) || !defined($value)) {
	    warn "confusion on cmd_file $cmd_file, line: $_\n";
	    next;
	};
	$value =~ s/^'(.*)'$/$1/;  # only support single quotes, allowing doubles to pass through to shell
	$opts{$key} = $value;
    };
    close CMD;
    return \%opts;
};

sub diff_output {
    my($cmd_base, $optref, $out, $trial, $alternative_p) = @_;
    if (! -e $out) {
	diag "    test $cmd_base is missing output $out\n";
	return undef;
    };
    system($optref->{cmp} . " $out $cmd_base.trial >$cmd_base.diff");
    if (-s "$cmd_base.diff") {
	open(DIFF, "<$cmd_base.diff") or die "cannot open $cmd_base.diff\n";
	my(@diff) = <DIFF>;
	close DIFF;
	if ($alternative_p ne 'altout') {
	    diag "    test $cmd_base failed, delta:\n" . join('', @diff);
	};
	return undef;
    } else {
	unlink("$cmd_base.diff");
	unlink("$cmd_base.trial");
    };
}


sub run_test {
    my($cmd_file) = @_;

    my $cmd_base = $cmd_file;
    $cmd_base =~ s/\.cmd$//;

    my $optref = parse_cmd_file($cmd_file);

    my $prog_path = '';
    if ($optref->{prog} =~ /^(perl|sh)$/) {
        $prog_path = $optref->{prog};
    }  else {
        $prog_path = "$scripts_dir/" . $optref->{prog};
    };

    my $in;
    if (!defined($optref->{in})) {
	$in = " < $cmd_base.in";
    } elsif ($optref->{in} eq '') {
	$in = '';
    } else {
	$in = " < " . $optref->{in};
    };
    my $out = (defined($optref->{out}) ? $optref->{in} : "$cmd_base.out");
    my $run_cmd = $prog_path . " " . (defined($optref->{args}) ? $optref->{args} : '') ." $in";
    $run_cmd .= $optref->{cmd_tail} if (defined($optref->{cmd_tail}));
    print "$env_cmd $run_cmd\n" if ($verbose);

    if (defined($optref->{enabled}) && !$optref->{enabled}) {
	diag "    test $cmd_file skipped (disabled in .cmd)\n";
	return 1;
    };
    if (defined($optref->{requires})) {
	# check for required modules:
	eval "use $optref->{requires};";
	if ($@ ne '') {
	    diag "   test $run_cmd skipped because of missing module $optref->{requires}\n";
	    return 1;
	};
    };

    if (!open(RUN, "$run_cmd 2>&1 |")) {
	diag "   failed to run $run_cmd\n";
	return undef;
    };
    # Icky.  Hack around some ithreads warnings that are hard to suppress.
    my $suppress_warnings_regexp = undef;
    if ($optref->{suppress_warnings}) {
	$suppress_warnings_regexp = '';
	my $this_perl_version = sprintf("%vd", $^V);
	foreach (split(/;/, $optref->{suppress_warnings})) {
	    my($version, $warning) = (/^([.0-9]+):(.*)$/);
	    die "test_command.t: bad suppress warning entry: $_\n"
		if (!defined($version) || !defined($warning));
	    $version = quotemeta($version);
	    if ($this_perl_version =~ /^$version/) {
		$suppress_warnings_regexp .= (quotemeta($warning) . "|")
	    };
	};
	$suppress_warnings_regexp =~ s/\|$//;
	$suppress_warnings_regexp = undef if ($suppress_warnings_regexp eq '');
    };
    open(OUT, ">$cmd_base.trial") or die "$0: cannot write $cmd_base.trial\n";
    while (<RUN>) {
	chomp;
	# normalize floating point numbers
	s/([ \t])\.([0-9efgEFG])/$10.$2/g;
	s/^\./0./;
	if (defined($suppress_warnings_regexp)) {
	    if (/^($suppress_warnings_regexp)/) {
	        # print "skipping: $_\n";
		next;
	    };
	};
	print OUT "$_\n";
    };
    if (!close RUN) {
        if ($? == -1) {
	    diag "failed to execute command: $!\n";
	    return undef;
	} elsif ($? & 127) {
	    diag "program " . $optref->{prog} . " received signal...very bad! $!\n\t$run_cmd";
	    return undef;
	} else {
	    my $exit_code = ($? >> 8);
	    if (defined($optref->{expected_exit_code})) {
		my $expected_result = undef;
		$expected_result = 1 if (!$expected_result && $optref->{expected_exit_code} eq 'fail' && $exit_code != 0);
		$expected_result = 1 if (!$expected_result && $optref->{expected_exit_code} ne 'fail' && $exit_code == $optref->{expected_exit_code});
		if (!$expected_result) {
		    diag "test $cmd_file exited with unexpected exit code $exit_code (should be " . $optref->{expected_exit_code} . ")\n\t$run_cmd\n";
		    return undef;
		    # fall through
		};
	    } else {
		# fall through... got EXPECTED non-zero exit code
	    };
	};
    } else {
	# check for sucessful programs that maybe should have failed
	if (defined($optref->{expected_exit_code}) &&
		'0' ne $optref->{expected_exit_code}) {
	    diag "program " . $optref->{prog} . " exited successfully when expected exit code was " . $optref->{expected_exit_code} . "\n\t$run_cmd";
	    return undef;
	};
    };
    close OUT;

    if ($update) {
	print "	updating saved output $out\n";
	mv("$out", "$out-") if (-f "$out");
	cp("$cmd_base.trial", "$out") or die "copy failed: $!";
    };

    #
    # finally do the compare
    #
    # start with alternative output
    my($out_ok) = diff_output($cmd_base, $optref, $out, "$cmd_base.trial", 'altout');
    if (!$out_ok && $optref->{altout} eq 'true') {
	$out_ok = diff_output($cmd_base, $optref, "$cmd_base.altout", "$cmd_base.trial", 'out');
    };
    return undef if (!$out_ok);

    system($optref->{cleanup}) if (defined($optref->{cleanup}));

    1;
}

