// EXchess source code, (c) Daniel C. Homan  1997-2000
// Released under the GNU public license, see file license.txt

/*
  Functions dealing with the standard hash table and the pawn hash tables.
  The standard hash table is divided into two pieces.  The lower half uses
  a "smart" replace strategy, and the upper half uses an always replace
  strategy.
  The pawn hash is very much like the standard hash - but simpler.  Only
  the hash key and score is stored.  The pawn table is always replace.
*/

#include "define.h"
#include "chess.h"
#include "funct.h"
#include "const.h"
#include "hash.h"

/* hash table variables */
unsigned long TAB_SIZE = 262144, PAWN_SIZE = 65536;  // hash and pawn hash sizes in entries
hash_rec *hash_table;             // pointer to start of hash table
pawn_rec *pawn_table;             // pointer to start of pawn table
h_code wtm, btm, hval[13][64];    // hash code variables
h_code castle_code[16], ep_code[64], stage_code[4];

int h_id = 1;  // hash code id flag for old-current searches

extern int max_ply;
extern int THRESHOLD;  // THRESHOLD for extensions

/* Function to set the hash table size */
void set_hash_size(int Mbytes)
{
  unsigned long Tab_entry_size = sizeof(hash_rec);
  unsigned long Pawn_entry_size = sizeof(pawn_rec);
  unsigned long Max_tab_size, Max_pawn_size;

  Max_pawn_size = Mbytes*1000000/(4*Pawn_entry_size);

  PAWN_SIZE = 2;
  while(2*PAWN_SIZE < Max_pawn_size) { PAWN_SIZE *= 2; }

  Max_tab_size = (Mbytes*1000000 - PAWN_SIZE*Pawn_entry_size)/Tab_entry_size;

  TAB_SIZE = 2;
  while(2*TAB_SIZE < Max_tab_size) { TAB_SIZE *= 2; }

  close_hash();
  open_hash();
}

/* Function with allocates space for the hash tables and generates
   the hash codes */
void open_hash()
{
 start_code();
 hash_table = new hash_rec[TAB_SIZE];
 pawn_table = new pawn_rec[PAWN_SIZE];
}

/* function to close the hash table */
void close_hash()
{
 delete [] hash_table;
 delete [] pawn_table;
}

/* function to stuff a hash entry into the hash table */
void put_hash(h_code h_key, int score, int alpha, int beta,
                            int depth, move hmove, int mate_ext)
{
 hash_rec *h, *r; int flag; 

 // is this an upper bound, lower bound, or exact score?
 if (score >= beta) { flag = FLAG_B; score = beta; }
 else if (score <= alpha) 
   { flag = FLAG_A; score = alpha; /* hmove.t = NOMOVE; */ }
 else { flag = FLAG_P; }

 // adjust any mate score to be a bound
 if(score >= 32000) { 
  if(flag == FLAG_B || flag == FLAG_P) {
   score = 32000; flag = FLAG_B; 
  } else return;
 } else if(score <= -32000) { 
  if(flag == FLAG_A || flag == FLAG_P) {
   score = -32000; flag = FLAG_A; 
  } else return;
 }

 // find location in table
 h = hash_table + (h_key.address&(TAB_SIZE-1));

 if((h_key.address&(TAB_SIZE-1))&1) h--;

 // use "smart" replace strategy if possible
 if ((h->id != h_id) || (h->depth < depth)
     || (h->depth == depth &&
          ((flag == FLAG_A && h->flag == FLAG_A && h->score > score) || 
           (flag == FLAG_B && h->flag == FLAG_B && h->score < score) || 
           (flag == FLAG_P))))
   {
     if((h->key != h_key.key || (h_key.address>>16) != h->address) && h->id == h_id) 
       { r = h + 1; (*r) = (*h); }  // copy entry to upper half of table
     h->key = h_key.key;
     h->address = (h_key.address>>16);
     h->score = short(score);
     h->flag = flag;
     h->depth = depth;
     h->hmove = hmove;
     h->id = h_id;
     h->mate_ext = mate_ext;
   }
      // or use always replace in the upper half of table
 else if(h_key.key != h->key || (h_key.address>>16) != h->address)   
   {
     h++;           // move to upper half of table
     h->key = h_key.key;
     h->address = (h_key.address>>16);
     h->score = short(score);
     h->flag = flag;
     h->depth = depth;
     h->hmove = hmove;
     h->id = h_id;
     h->mate_ext = mate_ext;
   }
}

/* function to find and return a hash entry */
int get_hash(h_code h_key, int alpha, int beta, int depth, int *mate_ext, int *null_hash, move *gmove)
{
 hash_rec *h; int hscore;

 // find the right location in the table
 h = hash_table + (h_key.address&(TAB_SIZE-1));

 if((h_key.address&(TAB_SIZE-1))&1) h--;

 // check lower part of table first then upper part
 if (h->key != h_key.key || h->address != (h_key.address>>16))
   { h++; if (h->key != h_key.key || h->address != (h_key.address>>16)) 
           { gmove->t = NOMOVE;  return HASH_MISS; } }

 // set null-move switch
 if(h->depth >= depth-8 && (h->flag==FLAG_A || h->flag==FLAG_P)
        && h->score < beta) (*null_hash) = 0;

 hscore = h->score;
 (*mate_ext) = h->mate_ext;
 (*gmove) = h->hmove;
   
 // return hash score if possible 
 if (h->depth < depth) return HASH_MISS; 
 if (h->flag==FLAG_A) {
   if(hscore <= alpha) return hscore;
 }
 if (h->flag==FLAG_B) {
   if(hscore >= beta) return hscore;
 }
 if (h->flag==FLAG_P) return hscore;
 return HASH_MISS;
}

/* function to retrieve the hash move from the table */
int get_move(h_code h_key)
{
 hash_rec *h;
 h = hash_table + (h_key.address&(TAB_SIZE-1));

 if((h_key.address&(TAB_SIZE-1))&1) h--;

 // if the hit was in the upper half of the table, go there
 // check lower part of table first then upper part
 if (h->key != h_key.key || h->address != (h_key.address>>16))
   { h++; if (h->key != h_key.key || h->address != (h_key.address>>16)) 
             return NOMOVE; }

 return h->hmove.t;
}

/* function to retrieve the hash move from the table */
int put_move(h_code h_key, int putmove)
{
 hash_rec *h;
 h = hash_table + (h_key.address&(TAB_SIZE-1));

 if((h_key.address&(TAB_SIZE-1))&1) h--;

 h->key = h_key.key;
 h->address = (h_key.address>>16);
 h->hmove.t = putmove;
 h->score = HASH_MISS;      // will cause score to be ignored by get_hash()

 return 0;
}

/* generate the hash code for a given position */
// also calculate material score and fill piecelists
h_code gen_code(position *p)
{
  int i;
  h_code temp_rec = { 0, 0 };
  p->material = 0;
  p->pieces[0] = 0; 
  p->pieces[1] = 0;

  for(i=0; i<=KING; i++) {
    p->plist[0][i][0] = 0;
    p->plist[1][i][0] = 0;
  }

  for(i = 0; i < 64; i++) {
   or(temp_rec, hval[ID(p->sq[i])][i]);
   if(p->sq[i].side == p->wtm) p->material += value[p->sq[i].type];
   else p->material -= value[p->sq[i].type];
   if(p->sq[i].type > PAWN) p->pieces[p->sq[i].side]++;
   if(p->sq[i].side != -1) {
    p->plist[p->sq[i].side][p->sq[i].type][0]++;
    p->plist[p->sq[i].side][p->sq[i].type][p->plist[p->sq[i].side][p->sq[i].type][0]] = i;
   } 
  }

  if(p->wtm) { or(temp_rec, wtm); }
  else { or(temp_rec, btm); }

  or(temp_rec, castle_code[p->castle]);
  or(temp_rec, ep_code[p->ep]);

  return temp_rec;
}

/* function to generate hash codes for all pieces/positions */
void start_code()
{
  long seed1 = -1712913483;
  int i, j, k;

  for (i = 0; i < 64; i++)
   { hval[0][i].key = 0; hval[0][i].address = 0; }
  for (j = 1; j < 13; j++)
   {
     for (i = 0; i < 64; i++)
      {
       hval[j][i].key = 0;
       hval[j][i].address = 0;
       for(k = 0; k < 32; k++)
        {
         hval[j][i].key = hval[j][i].key | (ibit(&seed1) << k);
         hval[j][i].address = hval[j][i].address | (ibit(&seed1) << k);
        }
      }
   }
   wtm.address = 0; wtm.key = 0;
   btm.address = 0; btm.key = 0;
   for(k = 0; k < 32; k++) {
    wtm.key = wtm.key | (ibit(&seed1) << k);
    wtm.address = wtm.address | (ibit(&seed1) << k);
    btm.key = btm.key | (ibit(&seed1) << k);
    btm.address = btm.address | (ibit(&seed1) << k);
   }

   for(i = 0; i < 16; i++) {
    castle_code[i].key = 0;
    castle_code[i].address = 0;
    for(k = 0; k < 32; k++) {
     castle_code[i].key = castle_code[i].key | (ibit(&seed1) << k);
     castle_code[i].address = castle_code[i].address | (ibit(&seed1) << k);
    }
   }

   for(i = 0; i < 64; i++) {
    ep_code[i].key = 0;
    ep_code[i].address = 0;
    for(k = 0; k < 32; k++) {
     ep_code[i].key = ep_code[i].key | (ibit(&seed1) << k);
     ep_code[i].address = ep_code[i].address | (ibit(&seed1) << k);
    }
   }

   for(i = 0; i < 4; i++) {
    stage_code[i].key = 0;
    stage_code[i].address = 0;
    for(k = 0; k < 32; k++) {
     stage_code[i].key = stage_code[i].key | (ibit(&seed1) << k);
     stage_code[i].address = stage_code[i].address | (ibit(&seed1) << k);
    }
   }


}






