// EXchess source code, (c) Daniel C. Homan  1997-2000
// Released under the GNU public license, see file license.txt

/* Execute move function */

#include "define.h"
#include "chess.h"
#include "const.h"
#include "funct.h"
#include "hash.h"
#include "iostream.h"

/* Hash code variables */
extern hash_rec *hash_table;    // pointer to hash_table
extern pawn_rec *pawn_table;    // pointer to pawn_table
extern h_code wtm, btm, hval[13][64];
extern h_code castle_code[16], ep_code[64];

/* Simple tables for quick in_check? tests */
extern int check_table[64][64];
extern int rook_check_table[64][64];
extern int bishop_check_table[64][64];
extern int knight_check_table[64][64];

/* Function to do a simple look to see if we are in check */
inline int simple_check(position *p, int side, int move_sq)
{
  int ksq = p->plist[side][KING][1];  // king-square

  if(rook_check_table[move_sq][ksq])
    return hor_attacks(ksq, p, side^1, 1);
  else 
    return dia_attacks(ksq, p, side^1, 1);
}

// Function to execute the move.
// If the move is legal, the function returns a 1, otherwise 0.
// Note that the move is made, regardless - so proper precautions
// need to be taken, if the move might need to be undone
int exec_move(position *p, move emove, int ply)
{
  register int pi;
  // if this is a castle, check that it is legal then
  // move the rook, the king move is given below
  if(emove.b.type&2) {
       if(p->check) return 0;
       switch(emove.b.to) {
	 /* white kingside castle */
           case 6:
              if(p->sq[5].type || p->sq[6].type ||
                  attacks(5,p,p->wtm^1,1))
                { return 0; }
              p->sq[5] = p->sq[7];
              p->sq[7] = empty;
              or(p->hcode, hval[WROOK][5]);
              or(p->hcode, hval[WROOK][7]);
              p->has_castled[1] = 1;
              /* update piece list */ 
              for(pi=1;pi<=p->plist[p->wtm][ROOK][0];pi++)
                if(p->plist[p->wtm][ROOK][pi] == 7) {
                   p->plist[p->wtm][ROOK][pi] = 5;
                   break;
                }
              break;
	 /* white queenside castle */  
           case 2:
              if(p->sq[1].type || p->sq[2].type || p->sq[3].type ||
                  attacks(3,p,p->wtm^1,1))
                { return 0; }
              p->sq[3] = p->sq[0];
              p->sq[0] = empty;
              or(p->hcode, hval[WROOK][3]);
              or(p->hcode, hval[WROOK][0]);
              p->has_castled[1] = 1;
              /* update piece list */
              for(pi=1;pi<=p->plist[p->wtm][ROOK][0];pi++)
                if(p->plist[p->wtm][ROOK][pi] == 0) {
                   p->plist[p->wtm][ROOK][pi] = 3;
                   break;
                }
              break;
	 /* black kingside castle */  
           case 62:
              if(p->sq[61].type || p->sq[62].type ||
                  attacks(61,p,p->wtm^1,1))
                { return 0; }
              p->sq[61] = p->sq[63];
              p->sq[63] = empty;
              or(p->hcode, hval[BROOK][61]);
              or(p->hcode, hval[BROOK][63]);
              p->has_castled[0] = 1;
              /* update piece list */
              for(pi=1;pi<=p->plist[p->wtm][ROOK][0];pi++)
                if(p->plist[p->wtm][ROOK][pi] == 63) {
                   p->plist[p->wtm][ROOK][pi] = 61;
                   break;
                }
              break;
	 /* black queenside castle */
           case 58:
              if(p->sq[57].type || p->sq[58].type || p->sq[59].type ||
                  attacks(59,p,p->wtm^1,1))
                { return 0; }
              p->sq[59] = p->sq[56];
              p->sq[56] = empty;
              or(p->hcode, hval[BROOK][59]);
              or(p->hcode, hval[BROOK][56]);
              p->has_castled[0] = 1;
              /* update piece list */
              for(pi=1;pi<=p->plist[p->wtm][ROOK][0];pi++)
                if(p->plist[p->wtm][ROOK][pi] == 56) {
                   p->plist[p->wtm][ROOK][pi] = 59;
                   break;
                }
              break;
           }
  }

  // update piece list for moving piece
  for(pi=1;pi<=p->plist[p->wtm][p->sq[emove.b.from].type][0];pi++)
   if(p->plist[p->wtm][p->sq[emove.b.from].type][pi] == emove.b.from) {
      p->plist[p->wtm][p->sq[emove.b.from].type][pi] = emove.b.to;
      break;
   }

  if(p->sq[emove.b.to].type) {
   // Remove hashcode for the target square 
   or(p->hcode, hval[ID(p->sq[emove.b.to])][emove.b.to]);   
   // Remove piece from piece list
   for(pi=1;pi<=p->plist[p->wtm^1][p->sq[emove.b.to].type][0];pi++)
    if(p->plist[p->wtm^1][p->sq[emove.b.to].type][pi] == emove.b.to) {
       p->plist[p->wtm^1][p->sq[emove.b.to].type][pi] =
       p->plist[p->wtm^1][p->sq[emove.b.to].type][p->plist[p->wtm^1][p->sq[emove.b.to].type][0]];
       p->plist[p->wtm^1][p->sq[emove.b.to].type][0]--;
       break;
    }
   // adjust material score   
   p->material += value[p->sq[emove.b.to].type];
   // adjust total piece count
   if(p->sq[emove.b.to].type > PAWN) p->pieces[p->wtm^1]--;
  }

  // Move the new piece to the target square
  p->sq[emove.b.to] = p->sq[emove.b.from];
  // Update the hash code to reflect the move
  or(p->hcode, hval[ID(p->sq[emove.b.from])][emove.b.from]);
  or(p->hcode, hval[ID(p->sq[emove.b.from])][emove.b.to]);
  // Original square is now empty
  p->sq[emove.b.from] = empty;

  // if move is en-passant, finish it
  if(emove.b.type&EP) {
    if(p->wtm) {
      p->sq[emove.b.to-8] = empty;
      or(p->hcode, hval[BPAWN][emove.b.to-8]);
      // Update piece lists 
      for(pi=1;pi<=p->plist[p->wtm^1][PAWN][0];pi++)
        if(p->plist[BLACK][PAWN][pi] == emove.b.to-8) {
           p->plist[BLACK][PAWN][pi] = 
           p->plist[BLACK][PAWN][p->plist[p->wtm^1][PAWN][0]];
	   p->plist[BLACK][PAWN][0]--;
           break;
        }
    } else {
      p->sq[emove.b.to+8] = empty;
      or(p->hcode, hval[WPAWN][emove.b.to+8]);
      // Update piece lists
      for(pi=1;pi<=p->plist[p->wtm^1][PAWN][0];pi++)
       if(p->plist[WHITE][PAWN][pi] == emove.b.to+8) {
          p->plist[WHITE][PAWN][pi] = 
          p->plist[WHITE][PAWN][p->plist[p->wtm^1][PAWN][0]];
	  p->plist[WHITE][PAWN][0]--;
          break;
       }
    }
    p->material += value[PAWN];
  }

  // if we are in check, move isn't legal
  // return 0
 if(!p->check) {
  if(p->check || emove.b.type&EP || p->sq[emove.b.to].type == KING)
  {
    if(attacks(p->plist[p->wtm][KING][1], p, p->wtm^1, 1)) return 0;
  } else if(check_table[emove.b.from][p->plist[p->wtm][KING][1]]) {
    if(simple_check(p, p->wtm, emove.b.from)) return 0;
  }
 }
  // if the move is a promotion, promote it
  if(emove.b.type&PROMOTE) {
    // Remove the pawn from the hash code
    or(p->hcode, hval[ID(p->sq[emove.b.to])][emove.b.to]);
    // Change the piece type to the promoted piece
    p->sq[emove.b.to].type = emove.b.promote;
    // Add the new piece to the piece lists
    p->plist[p->wtm][emove.b.promote][0]++;
    p->plist[p->wtm][emove.b.promote][p->plist[p->wtm][emove.b.promote][0]]=emove.b.to;
    // Remove the pawn from the piece lists
    for(pi=1;pi<=p->plist[p->wtm][PAWN][0];pi++)
      if(p->plist[p->wtm][PAWN][pi] == emove.b.to) {
         p->plist[p->wtm][PAWN][pi] = 
         p->plist[p->wtm][PAWN][p->plist[p->wtm][PAWN][0]];
	 p->plist[p->wtm][PAWN][0]--;
         break;
      }
    // adjust material score
    p->material += value[emove.b.promote] - value[PAWN];
    // add piece to hash code 
    or(p->hcode, hval[ID(p->sq[emove.b.to])][emove.b.to]);
    // adjust total piece count
    p->pieces[p->wtm]++;
  }

  // update position characteristics
  p->wtm = p->wtm^1;
  p->material = -p->material;
  or(p->hcode, btm); or(p->hcode, wtm);

  // undo hash code for en-passant and castling status
  or(p->hcode, ep_code[p->ep]);
  or(p->hcode, castle_code[p->castle]);

  // if move is a pawn push 2 spaces, set en passant flag
  if((emove.b.type&PAWN_PUSH2) && 
     ((FILE(emove.b.to) < 7 && p->sq[emove.b.to+1].type == PAWN && p->sq[emove.b.to+1].side == p->wtm)
      || (FILE(emove.b.to) && p->sq[emove.b.to-1].type == PAWN && p->sq[emove.b.to-1].side == p->wtm)))
   { p->ep = (emove.b.from+emove.b.to)>>1; } else { p->ep = 0; }
  // if move is not a capture or a pawn move, increase fifty count
  if(emove.b.type&CAPTURE || emove.b.type&PAWN_PUSH2 || emove.b.type&PAWN_PUSH)
   { p->fifty = 0; } else { p->fifty++; }
  // update castling status
  p->castle = p->castle&castle_mask[emove.b.from];
  p->castle = p->castle&castle_mask[emove.b.to];
  // put this move in as the last move
  p->last = emove;

  // update hash code for en-passant and castling status
  or(p->hcode, ep_code[p->ep]);
  or(p->hcode, castle_code[p->castle]);

  // check whether other side is placed in check
  int ptype = p->sq[emove.b.to].type; 
  int ksq = p->plist[p->wtm][KING][1];

  p->check = 0;

  if((ptype == BISHOP || ptype == QUEEN)
      && bishop_check_table[emove.b.to][ksq])
    p->check = dia_attacks(ksq, p, p->wtm^1, 1);
  else if((ptype == ROOK || ptype == QUEEN)
         && rook_check_table[emove.b.to][ksq])
    p->check = hor_attacks(ksq, p, p->wtm^1, 1);
  else if(ptype == PAWN && bishop_check_table[emove.b.to][ksq]) {
    if(p->wtm && (emove.b.to == ksq+9 || emove.b.to == ksq+7))
      { p->check = 1; return 1; }
    else if(p->wtm^1 && (emove.b.to == ksq-9 || emove.b.to == ksq-7))
      { p->check = 1; return 1; }
  }
  if(check_table[emove.b.from][ksq] && p->check^1) 
    p->check = simple_check(p, p->wtm, emove.b.from);
  if(emove.b.type&EP && p->check^1)
    p->check = attacks(ksq, p, p->wtm^1, 1);
  if(emove.b.type&CASTLE && p->check^1)
    p->check = hor_attacks(ksq, p, p->wtm^1, 1);
  if(ptype == KNIGHT && knight_check_table[emove.b.to][ksq]
      && p->check^1) p->check = 1;

  return 1;
}

// Function to make the move in the q-search
//  - much simpler
int qexec_move(position *p, move emove, int ply)
{
  register int pi;

  // update piece lists
  for(pi=1;pi<=p->plist[p->wtm][p->sq[emove.b.from].type][0];pi++)
    if(p->plist[p->wtm][p->sq[emove.b.from].type][pi] == emove.b.from) {
       p->plist[p->wtm][p->sq[emove.b.from].type][pi] = emove.b.to;
       break;
    }
  for(pi=1;pi<=p->plist[p->wtm^1][p->sq[emove.b.to].type][0];pi++)
    if(p->plist[p->wtm^1][p->sq[emove.b.to].type][pi] == emove.b.to) {
       p->plist[p->wtm^1][p->sq[emove.b.to].type][pi] = 
       p->plist[p->wtm^1][p->sq[emove.b.to].type][p->plist[p->wtm^1][p->sq[emove.b.to].type][0]];
       p->plist[p->wtm^1][p->sq[emove.b.to].type][0]--;
       break;
    }

  // make the move - updating hash code as well
  p->material += value[p->sq[emove.b.to].type];
  if(p->sq[emove.b.to].type > PAWN) p->pieces[p->wtm^1]--;
  p->sq[emove.b.to] = p->sq[emove.b.from];
  p->sq[emove.b.from] = empty;

  // if move is en-passant, finish it
  if(emove.b.type&EP) {
    if(p->wtm) {
      p->sq[emove.b.to-8] = empty;
      for(pi=1;pi<=p->plist[p->wtm^1][PAWN][0];pi++)
        if(p->plist[p->wtm^1][PAWN][pi] == emove.b.to-8) {
           p->plist[p->wtm^1][PAWN][pi] = 
           p->plist[p->wtm^1][PAWN][p->plist[p->wtm^1][PAWN][0]];
	   p->plist[p->wtm^1][PAWN][0]--;
           break;
        }
    } else {
      p->sq[emove.b.to+8] = empty;
      for(pi=1;pi<=p->plist[p->wtm^1][PAWN][0];pi++)
        if(p->plist[p->wtm^1][PAWN][pi] == emove.b.to+8) {
           p->plist[p->wtm^1][PAWN][pi] = 
           p->plist[p->wtm^1][PAWN][p->plist[p->wtm^1][PAWN][0]];
	   p->plist[p->wtm^1][PAWN][0]--;
           break;
        }
    }
    p->material += value[PAWN];
  }

  // if the move is a promotion, promote it
  if(emove.b.type&PROMOTE) {
    p->sq[emove.b.to].type = emove.b.promote;
    p->material += value[emove.b.promote] - value[PAWN];
    p->pieces[p->wtm]++;
    p->plist[p->wtm][emove.b.promote][0]++;
    p->plist[p->wtm][emove.b.promote][p->plist[p->wtm][emove.b.promote][0]]=emove.b.to;
    for(pi=1;pi<=p->plist[p->wtm][PAWN][0];pi++)
      if(p->plist[p->wtm][PAWN][pi] == emove.b.to) {
         p->plist[p->wtm][PAWN][pi] = 
         p->plist[p->wtm][PAWN][p->plist[p->wtm][PAWN][0]];
	 p->plist[p->wtm][PAWN][0]--;
         break;
      }
  }

  // update position characteristics
  p->wtm = p->wtm^1;
  p->material = -p->material;
  p->ep = 0;
  p->check = 0;
  p->fifty = 0;
  p->last = emove;

  return 1;
}








