package ShiftJIS::CP932::MapUTF;

use strict;
use vars qw($VERSION $PACKAGE @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);
use vars qw(%CP932_UTF16 %CP932_UTF8 %UTF16_CP932 %UTF8_CP932);

use Carp;
require Exporter;
@ISA = qw(Exporter);
@EXPORT    = qw(cp932_to_utf8 cp932_to_utf16 utf8_to_cp932 utf16_to_cp932);
@EXPORT_OK = qw(toUTF8 fromUTF8 toUTF16 fromUTF16);
%EXPORT_TAGS = ('all' => [ @EXPORT, @EXPORT_OK] );

$VERSION = '0.07';
$PACKAGE = 'ShiftJIS::CP932::MapUTF'; # __PACKAGE__

my $Uchar = 
     '(?:[\x00-\x7F]|[\xC2-\xDF][\x80-\xBF]|'
   . '\xE0[\xA0-\xBF][\x80-\xBF]|'
   . '[\xE1-\xEF][\x80-\xBF][\x80-\xBF]|'
   . '\xF0[\x90-\xBF][\x80-\xBF][\x80-\xBF]|'
   . '[\xF1-\xF3][\x80-\xBF][\x80-\xBF][\x80-\xBF]|'
   . '\xF4[\x80-\x8F][\x80-\xBF][\x80-\xBF])';

my $Schar = '(?:[\x81-\x9F\xE0-\xFC][\x00-\xFF]|[\x00-\xFF])';

my $Hchar = '(?:[\x00-\xFF][\x00-\xD7\xE0-\xFF]|'
   . '[\x00-\xFF][\xD8-\xDB][\x00-\xFF][\xDC-\xDF])';

sub toUTF8 ($)
{
  my $u = shift;

  croak("$PACKAGE: out of range utf-8 in toUTF8")
    if $u < 0 || 0x10FFFF < $u;

  croak("$PACKAGE: unpaired surrogate in toUTF8")
    if 0xD800 < $u && $u < 0xDFFF;

  return
    $u <  0x0080 ? chr($u) : 
    $u <  0x0800 ?
 pack("CC",
  ( ($u >>  6)         | 0xc0),
  ( ($u        & 0x3f) | 0x80)
 ) :
    $u < 0x10000 ?
 pack("CCC", 
  ( ($u >> 12)         | 0xe0),
  ((($u >>  6) & 0x3f) | 0x80),
  ( ($u        & 0x3f) | 0x80)
 ) :
 pack("CCCC",
  ( ($u >> 18)         | 0xf0),
  ((($u >> 12) & 0x3f) | 0x80),
  ((($u >>  6) & 0x3f) | 0x80),
  ( ($u        & 0x3f) | 0x80)
 );
}

sub fromUTF8 ($)
{
  my @c = unpack 'C*', shift;
  return @c == 1 ? $c[0] : @c == 2
    ? ((($c[0] & 0x1F)<<6)|($c[1] & 0x3F))
    : @c == 3
      ? ((($c[0] & 0x0F)<<12)|(($c[1] & 0x3F)<< 6)|($c[2] & 0x3F))
      : @c == 4
        ? ((($c[0] & 0x07)<<18)|(($c[1] & 0x3F)<<12)|
          ( ($c[2] & 0x3F)<< 6)|( $c[3] & 0x3F))
        : croak("$PACKAGE: invalid utf-8 in fromUTF8")
}

sub toUTF16 ($)
{
  my $n = shift;
  croak("$PACKAGE: out of range utf-16 in toUTF16")
    if $n < 0 || 0x10FFFF < $n;

  croak("$PACKAGE: unpaired surrogate in toUTF16")
    if 0xD800 < $n && $n < 0xDFFF;

  return pack('v', $n) if $n < 0x10000;
  $n -= 0x10000;
  pack('vv', (($n >> 10) | 0xD800), (($n & 0x3FF) | 0xDC00));
}

sub fromUTF16 ($)
{
  my $h = shift;
  return unpack('v', $h) if length $h == 2;
  croak("$PACKAGE: invalid utf-16 in fromUTF16") if length $h != 4;

  my($high,$low) = unpack("v*",$h);

  croak("$PACKAGE: invalid utf-16 in fromUTF16")
    unless 0xD800 <= $high && $high <= 0xDBFF
        && 0xDC00 <= $low  && $low  <= 0xDFFF;

  $high &= 0x3FF;
  $low  &= 0x3FF;
  return 0x10000 + ($high << 10 | $low);
}

foreach(qw/
00-0000
01-0001
02-0002
03-0003
04-0004
05-0005
06-0006
07-0007
08-0008
09-0009
0A-000A
0B-000B
0C-000C
0D-000D
0E-000E
0F-000F
10-0010
11-0011
12-0012
13-0013
14-0014
15-0015
16-0016
17-0017
18-0018
19-0019
1A-001A
1B-001B
1C-001C
1D-001D
1E-001E
1F-001F
20-0020
21-0021
22-0022
23-0023
24-0024
25-0025
26-0026
27-0027
28-0028
29-0029
2A-002A
2B-002B
2C-002C
2D-002D
2E-002E
2F-002F
30-0030
31-0031
32-0032
33-0033
34-0034
35-0035
36-0036
37-0037
38-0038
39-0039
3A-003A
3B-003B
3C-003C
3D-003D
3E-003E
3F-003F
40-0040
41-0041
42-0042
43-0043
44-0044
45-0045
46-0046
47-0047
48-0048
49-0049
4A-004A
4B-004B
4C-004C
4D-004D
4E-004E
4F-004F
50-0050
51-0051
52-0052
53-0053
54-0054
55-0055
56-0056
57-0057
58-0058
59-0059
5A-005A
5B-005B
5C-005C
5D-005D
5E-005E
5F-005F
60-0060
61-0061
62-0062
63-0063
64-0064
65-0065
66-0066
67-0067
68-0068
69-0069
6A-006A
6B-006B
6C-006C
6D-006D
6E-006E
6F-006F
70-0070
71-0071
72-0072
73-0073
74-0074
75-0075
76-0076
77-0077
78-0078
79-0079
7A-007A
7B-007B
7C-007C
7D-007D
7E-007E
7F-007F
A1-FF61
A2-FF62
A3-FF63
A4-FF64
A5-FF65
A6-FF66
A7-FF67
A8-FF68
A9-FF69
AA-FF6A
AB-FF6B
AC-FF6C
AD-FF6D
AE-FF6E
AF-FF6F
B0-FF70
B1-FF71
B2-FF72
B3-FF73
B4-FF74
B5-FF75
B6-FF76
B7-FF77
B8-FF78
B9-FF79
BA-FF7A
BB-FF7B
BC-FF7C
BD-FF7D
BE-FF7E
BF-FF7F
C0-FF80
C1-FF81
C2-FF82
C3-FF83
C4-FF84
C5-FF85
C6-FF86
C7-FF87
C8-FF88
C9-FF89
CA-FF8A
CB-FF8B
CC-FF8C
CD-FF8D
CE-FF8E
CF-FF8F
D0-FF90
D1-FF91
D2-FF92
D3-FF93
D4-FF94
D5-FF95
D6-FF96
D7-FF97
D8-FF98
D9-FF99
DA-FF9A
DB-FF9B
DC-FF9C
DD-FF9D
DE-FF9E
DF-FF9F
8140-3000
8141-3001
8142-3002
8143-FF0C
8144-FF0E
8145-30FB
8146-FF1A
8147-FF1B
8148-FF1F
8149-FF01
814A-309B
814B-309C
814C-00B4
814D-FF40
814E-00A8
814F-FF3E
8150-FFE3
8151-FF3F
8152-30FD
8153-30FE
8154-309D
8155-309E
8156-3003
8157-4EDD
8158-3005
8159-3006
815A-3007
815B-30FC
815C-2015
815D-2010
815E-FF0F
815F-FF3C
8160-FF5E
8161-2225
8162-FF5C
8163-2026
8164-2025
8165-2018
8166-2019
8167-201C
8168-201D
8169-FF08
816A-FF09
816B-3014
816C-3015
816D-FF3B
816E-FF3D
816F-FF5B
8170-FF5D
8171-3008
8172-3009
8173-300A
8174-300B
8175-300C
8176-300D
8177-300E
8178-300F
8179-3010
817A-3011
817B-FF0B
817C-FF0D
817D-00B1
817E-00D7
8180-00F7
8181-FF1D
8182-2260
8183-FF1C
8184-FF1E
8185-2266
8186-2267
8187-221E
8188-2234
8189-2642
818A-2640
818B-00B0
818C-2032
818D-2033
818E-2103
818F-FFE5
8190-FF04
8191-FFE0
8192-FFE1
8193-FF05
8194-FF03
8195-FF06
8196-FF0A
8197-FF20
8198-00A7
8199-2606
819A-2605
819B-25CB
819C-25CF
819D-25CE
819E-25C7
819F-25C6
81A0-25A1
81A1-25A0
81A2-25B3
81A3-25B2
81A4-25BD
81A5-25BC
81A6-203B
81A7-3012
81A8-2192
81A9-2190
81AA-2191
81AB-2193
81AC-3013
81B8-2208
81B9-220B
81BA-2286
81BB-2287
81BC-2282
81BD-2283
81BE-222A-t
81BF-2229-t
81C8-2227
81C9-2228
81CA-FFE2-t
81CB-21D2
81CC-21D4
81CD-2200
81CE-2203
81DA-2220-t
81DB-22A5-t
81DC-2312
81DD-2202
81DE-2207
81DF-2261-t
81E0-2252-t
81E1-226A
81E2-226B
81E3-221A-t
81E4-223D
81E5-221D
81E6-2235-t
81E7-222B-t
81E8-222C
81F0-212B
81F1-2030
81F2-266F
81F3-266D
81F4-266A
81F5-2020
81F6-2021
81F7-00B6
81FC-25EF
824F-FF10
8250-FF11
8251-FF12
8252-FF13
8253-FF14
8254-FF15
8255-FF16
8256-FF17
8257-FF18
8258-FF19
8260-FF21
8261-FF22
8262-FF23
8263-FF24
8264-FF25
8265-FF26
8266-FF27
8267-FF28
8268-FF29
8269-FF2A
826A-FF2B
826B-FF2C
826C-FF2D
826D-FF2E
826E-FF2F
826F-FF30
8270-FF31
8271-FF32
8272-FF33
8273-FF34
8274-FF35
8275-FF36
8276-FF37
8277-FF38
8278-FF39
8279-FF3A
8281-FF41
8282-FF42
8283-FF43
8284-FF44
8285-FF45
8286-FF46
8287-FF47
8288-FF48
8289-FF49
828A-FF4A
828B-FF4B
828C-FF4C
828D-FF4D
828E-FF4E
828F-FF4F
8290-FF50
8291-FF51
8292-FF52
8293-FF53
8294-FF54
8295-FF55
8296-FF56
8297-FF57
8298-FF58
8299-FF59
829A-FF5A
829F-3041
82A0-3042
82A1-3043
82A2-3044
82A3-3045
82A4-3046
82A5-3047
82A6-3048
82A7-3049
82A8-304A
82A9-304B
82AA-304C
82AB-304D
82AC-304E
82AD-304F
82AE-3050
82AF-3051
82B0-3052
82B1-3053
82B2-3054
82B3-3055
82B4-3056
82B5-3057
82B6-3058
82B7-3059
82B8-305A
82B9-305B
82BA-305C
82BB-305D
82BC-305E
82BD-305F
82BE-3060
82BF-3061
82C0-3062
82C1-3063
82C2-3064
82C3-3065
82C4-3066
82C5-3067
82C6-3068
82C7-3069
82C8-306A
82C9-306B
82CA-306C
82CB-306D
82CC-306E
82CD-306F
82CE-3070
82CF-3071
82D0-3072
82D1-3073
82D2-3074
82D3-3075
82D4-3076
82D5-3077
82D6-3078
82D7-3079
82D8-307A
82D9-307B
82DA-307C
82DB-307D
82DC-307E
82DD-307F
82DE-3080
82DF-3081
82E0-3082
82E1-3083
82E2-3084
82E3-3085
82E4-3086
82E5-3087
82E6-3088
82E7-3089
82E8-308A
82E9-308B
82EA-308C
82EB-308D
82EC-308E
82ED-308F
82EE-3090
82EF-3091
82F0-3092
82F1-3093
8340-30A1
8341-30A2
8342-30A3
8343-30A4
8344-30A5
8345-30A6
8346-30A7
8347-30A8
8348-30A9
8349-30AA
834A-30AB
834B-30AC
834C-30AD
834D-30AE
834E-30AF
834F-30B0
8350-30B1
8351-30B2
8352-30B3
8353-30B4
8354-30B5
8355-30B6
8356-30B7
8357-30B8
8358-30B9
8359-30BA
835A-30BB
835B-30BC
835C-30BD
835D-30BE
835E-30BF
835F-30C0
8360-30C1
8361-30C2
8362-30C3
8363-30C4
8364-30C5
8365-30C6
8366-30C7
8367-30C8
8368-30C9
8369-30CA
836A-30CB
836B-30CC
836C-30CD
836D-30CE
836E-30CF
836F-30D0
8370-30D1
8371-30D2
8372-30D3
8373-30D4
8374-30D5
8375-30D6
8376-30D7
8377-30D8
8378-30D9
8379-30DA
837A-30DB
837B-30DC
837C-30DD
837D-30DE
837E-30DF
8380-30E0
8381-30E1
8382-30E2
8383-30E3
8384-30E4
8385-30E5
8386-30E6
8387-30E7
8388-30E8
8389-30E9
838A-30EA
838B-30EB
838C-30EC
838D-30ED
838E-30EE
838F-30EF
8390-30F0
8391-30F1
8392-30F2
8393-30F3
8394-30F4
8395-30F5
8396-30F6
839F-0391
83A0-0392
83A1-0393
83A2-0394
83A3-0395
83A4-0396
83A5-0397
83A6-0398
83A7-0399
83A8-039A
83A9-039B
83AA-039C
83AB-039D
83AC-039E
83AD-039F
83AE-03A0
83AF-03A1
83B0-03A3
83B1-03A4
83B2-03A5
83B3-03A6
83B4-03A7
83B5-03A8
83B6-03A9
83BF-03B1
83C0-03B2
83C1-03B3
83C2-03B4
83C3-03B5
83C4-03B6
83C5-03B7
83C6-03B8
83C7-03B9
83C8-03BA
83C9-03BB
83CA-03BC
83CB-03BD
83CC-03BE
83CD-03BF
83CE-03C0
83CF-03C1
83D0-03C3
83D1-03C4
83D2-03C5
83D3-03C6
83D4-03C7
83D5-03C8
83D6-03C9
8440-0410
8441-0411
8442-0412
8443-0413
8444-0414
8445-0415
8446-0401
8447-0416
8448-0417
8449-0418
844A-0419
844B-041A
844C-041B
844D-041C
844E-041D
844F-041E
8450-041F
8451-0420
8452-0421
8453-0422
8454-0423
8455-0424
8456-0425
8457-0426
8458-0427
8459-0428
845A-0429
845B-042A
845C-042B
845D-042C
845E-042D
845F-042E
8460-042F
8470-0430
8471-0431
8472-0432
8473-0433
8474-0434
8475-0435
8476-0451
8477-0436
8478-0437
8479-0438
847A-0439
847B-043A
847C-043B
847D-043C
847E-043D
8480-043E
8481-043F
8482-0440
8483-0441
8484-0442
8485-0443
8486-0444
8487-0445
8488-0446
8489-0447
848A-0448
848B-0449
848C-044A
848D-044B
848E-044C
848F-044D
8490-044E
8491-044F
849F-2500
84A0-2502
84A1-250C
84A2-2510
84A3-2518
84A4-2514
84A5-251C
84A6-252C
84A7-2524
84A8-2534
84A9-253C
84AA-2501
84AB-2503
84AC-250F
84AD-2513
84AE-251B
84AF-2517
84B0-2523
84B1-2533
84B2-252B
84B3-253B
84B4-254B
84B5-2520
84B6-252F
84B7-2528
84B8-2537
84B9-253F
84BA-251D
84BB-2530
84BC-2525
84BD-2538
84BE-2542
8740-2460
8741-2461
8742-2462
8743-2463
8744-2464
8745-2465
8746-2466
8747-2467
8748-2468
8749-2469
874A-246A
874B-246B
874C-246C
874D-246D
874E-246E
874F-246F
8750-2470
8751-2471
8752-2472
8753-2473
8754-2160-t
8755-2161-t
8756-2162-t
8757-2163-t
8758-2164-t
8759-2165-t
875A-2166-t
875B-2167-t
875C-2168-t
875D-2169-t
875F-3349
8760-3314
8761-3322
8762-334D
8763-3318
8764-3327
8765-3303
8766-3336
8767-3351
8768-3357
8769-330D
876A-3326
876B-3323
876C-332B
876D-334A
876E-333B
876F-339C
8770-339D
8771-339E
8772-338E
8773-338F
8774-33C4
8775-33A1
877E-337B
8780-301D
8781-301F
8782-2116-t
8783-33CD
8784-2121-t
8785-32A4
8786-32A5
8787-32A6
8788-32A7
8789-32A8
878A-3231-t
878B-3232
878C-3239
878D-337E
878E-337D
878F-337C
8790-2252-f
8791-2261-f
8792-222B-f
8793-222E
8794-2211
8795-221A-f
8796-22A5-f
8797-2220-f
8798-221F
8799-22BF
879A-2235-f
879B-2229-f
879C-222A-f
889F-4E9C
88A0-5516
88A1-5A03
88A2-963F
88A3-54C0
88A4-611B
88A5-6328
88A6-59F6
88A7-9022
88A8-8475
88A9-831C
88AA-7A50
88AB-60AA
88AC-63E1
88AD-6E25
88AE-65ED
88AF-8466
88B0-82A6
88B1-9BF5
88B2-6893
88B3-5727
88B4-65A1
88B5-6271
88B6-5B9B
88B7-59D0
88B8-867B
88B9-98F4
88BA-7D62
88BB-7DBE
88BC-9B8E
88BD-6216
88BE-7C9F
88BF-88B7
88C0-5B89
88C1-5EB5
88C2-6309
88C3-6697
88C4-6848
88C5-95C7
88C6-978D
88C7-674F
88C8-4EE5
88C9-4F0A
88CA-4F4D
88CB-4F9D
88CC-5049
88CD-56F2
88CE-5937
88CF-59D4
88D0-5A01
88D1-5C09
88D2-60DF
88D3-610F
88D4-6170
88D5-6613
88D6-6905
88D7-70BA
88D8-754F
88D9-7570
88DA-79FB
88DB-7DAD
88DC-7DEF
88DD-80C3
88DE-840E
88DF-8863
88E0-8B02
88E1-9055
88E2-907A
88E3-533B
88E4-4E95
88E5-4EA5
88E6-57DF
88E7-80B2
88E8-90C1
88E9-78EF
88EA-4E00
88EB-58F1
88EC-6EA2
88ED-9038
88EE-7A32
88EF-8328
88F0-828B
88F1-9C2F
88F2-5141
88F3-5370
88F4-54BD
88F5-54E1
88F6-56E0
88F7-59FB
88F8-5F15
88F9-98F2
88FA-6DEB
88FB-80E4
88FC-852D
8940-9662
8941-9670
8942-96A0
8943-97FB
8944-540B
8945-53F3
8946-5B87
8947-70CF
8948-7FBD
8949-8FC2
894A-96E8
894B-536F
894C-9D5C
894D-7ABA
894E-4E11
894F-7893
8950-81FC
8951-6E26
8952-5618
8953-5504
8954-6B1D
8955-851A
8956-9C3B
8957-59E5
8958-53A9
8959-6D66
895A-74DC
895B-958F
895C-5642
895D-4E91
895E-904B
895F-96F2
8960-834F
8961-990C
8962-53E1
8963-55B6
8964-5B30
8965-5F71
8966-6620
8967-66F3
8968-6804
8969-6C38
896A-6CF3
896B-6D29
896C-745B
896D-76C8
896E-7A4E
896F-9834
8970-82F1
8971-885B
8972-8A60
8973-92ED
8974-6DB2
8975-75AB
8976-76CA
8977-99C5
8978-60A6
8979-8B01
897A-8D8A
897B-95B2
897C-698E
897D-53AD
897E-5186
8980-5712
8981-5830
8982-5944
8983-5BB4
8984-5EF6
8985-6028
8986-63A9
8987-63F4
8988-6CBF
8989-6F14
898A-708E
898B-7114
898C-7159
898D-71D5
898E-733F
898F-7E01
8990-8276
8991-82D1
8992-8597
8993-9060
8994-925B
8995-9D1B
8996-5869
8997-65BC
8998-6C5A
8999-7525
899A-51F9
899B-592E
899C-5965
899D-5F80
899E-5FDC
899F-62BC
89A0-65FA
89A1-6A2A
89A2-6B27
89A3-6BB4
89A4-738B
89A5-7FC1
89A6-8956
89A7-9D2C
89A8-9D0E
89A9-9EC4
89AA-5CA1
89AB-6C96
89AC-837B
89AD-5104
89AE-5C4B
89AF-61B6
89B0-81C6
89B1-6876
89B2-7261
89B3-4E59
89B4-4FFA
89B5-5378
89B6-6069
89B7-6E29
89B8-7A4F
89B9-97F3
89BA-4E0B
89BB-5316
89BC-4EEE
89BD-4F55
89BE-4F3D
89BF-4FA1
89C0-4F73
89C1-52A0
89C2-53EF
89C3-5609
89C4-590F
89C5-5AC1
89C6-5BB6
89C7-5BE1
89C8-79D1
89C9-6687
89CA-679C
89CB-67B6
89CC-6B4C
89CD-6CB3
89CE-706B
89CF-73C2
89D0-798D
89D1-79BE
89D2-7A3C
89D3-7B87
89D4-82B1
89D5-82DB
89D6-8304
89D7-8377
89D8-83EF
89D9-83D3
89DA-8766
89DB-8AB2
89DC-5629
89DD-8CA8
89DE-8FE6
89DF-904E
89E0-971E
89E1-868A
89E2-4FC4
89E3-5CE8
89E4-6211
89E5-7259
89E6-753B
89E7-81E5
89E8-82BD
89E9-86FE
89EA-8CC0
89EB-96C5
89EC-9913
89ED-99D5
89EE-4ECB
89EF-4F1A
89F0-89E3
89F1-56DE
89F2-584A
89F3-58CA
89F4-5EFB
89F5-5FEB
89F6-602A
89F7-6094
89F8-6062
89F9-61D0
89FA-6212
89FB-62D0
89FC-6539
8A40-9B41
8A41-6666
8A42-68B0
8A43-6D77
8A44-7070
8A45-754C
8A46-7686
8A47-7D75
8A48-82A5
8A49-87F9
8A4A-958B
8A4B-968E
8A4C-8C9D
8A4D-51F1
8A4E-52BE
8A4F-5916
8A50-54B3
8A51-5BB3
8A52-5D16
8A53-6168
8A54-6982
8A55-6DAF
8A56-788D
8A57-84CB
8A58-8857
8A59-8A72
8A5A-93A7
8A5B-9AB8
8A5C-6D6C
8A5D-99A8
8A5E-86D9
8A5F-57A3
8A60-67FF
8A61-86CE
8A62-920E
8A63-5283
8A64-5687
8A65-5404
8A66-5ED3
8A67-62E1
8A68-64B9
8A69-683C
8A6A-6838
8A6B-6BBB
8A6C-7372
8A6D-78BA
8A6E-7A6B
8A6F-899A
8A70-89D2
8A71-8D6B
8A72-8F03
8A73-90ED
8A74-95A3
8A75-9694
8A76-9769
8A77-5B66
8A78-5CB3
8A79-697D
8A7A-984D
8A7B-984E
8A7C-639B
8A7D-7B20
8A7E-6A2B
8A80-6A7F
8A81-68B6
8A82-9C0D
8A83-6F5F
8A84-5272
8A85-559D
8A86-6070
8A87-62EC
8A88-6D3B
8A89-6E07
8A8A-6ED1
8A8B-845B
8A8C-8910
8A8D-8F44
8A8E-4E14
8A8F-9C39
8A90-53F6
8A91-691B
8A92-6A3A
8A93-9784
8A94-682A
8A95-515C
8A96-7AC3
8A97-84B2
8A98-91DC
8A99-938C
8A9A-565B
8A9B-9D28
8A9C-6822
8A9D-8305
8A9E-8431
8A9F-7CA5
8AA0-5208
8AA1-82C5
8AA2-74E6
8AA3-4E7E
8AA4-4F83
8AA5-51A0
8AA6-5BD2
8AA7-520A
8AA8-52D8
8AA9-52E7
8AAA-5DFB
8AAB-559A
8AAC-582A
8AAD-59E6
8AAE-5B8C
8AAF-5B98
8AB0-5BDB
8AB1-5E72
8AB2-5E79
8AB3-60A3
8AB4-611F
8AB5-6163
8AB6-61BE
8AB7-63DB
8AB8-6562
8AB9-67D1
8ABA-6853
8ABB-68FA
8ABC-6B3E
8ABD-6B53
8ABE-6C57
8ABF-6F22
8AC0-6F97
8AC1-6F45
8AC2-74B0
8AC3-7518
8AC4-76E3
8AC5-770B
8AC6-7AFF
8AC7-7BA1
8AC8-7C21
8AC9-7DE9
8ACA-7F36
8ACB-7FF0
8ACC-809D
8ACD-8266
8ACE-839E
8ACF-89B3
8AD0-8ACC
8AD1-8CAB
8AD2-9084
8AD3-9451
8AD4-9593
8AD5-9591
8AD6-95A2
8AD7-9665
8AD8-97D3
8AD9-9928
8ADA-8218
8ADB-4E38
8ADC-542B
8ADD-5CB8
8ADE-5DCC
8ADF-73A9
8AE0-764C
8AE1-773C
8AE2-5CA9
8AE3-7FEB
8AE4-8D0B
8AE5-96C1
8AE6-9811
8AE7-9854
8AE8-9858
8AE9-4F01
8AEA-4F0E
8AEB-5371
8AEC-559C
8AED-5668
8AEE-57FA
8AEF-5947
8AF0-5B09
8AF1-5BC4
8AF2-5C90
8AF3-5E0C
8AF4-5E7E
8AF5-5FCC
8AF6-63EE
8AF7-673A
8AF8-65D7
8AF9-65E2
8AFA-671F
8AFB-68CB
8AFC-68C4
8B40-6A5F
8B41-5E30
8B42-6BC5
8B43-6C17
8B44-6C7D
8B45-757F
8B46-7948
8B47-5B63
8B48-7A00
8B49-7D00
8B4A-5FBD
8B4B-898F
8B4C-8A18
8B4D-8CB4
8B4E-8D77
8B4F-8ECC
8B50-8F1D
8B51-98E2
8B52-9A0E
8B53-9B3C
8B54-4E80
8B55-507D
8B56-5100
8B57-5993
8B58-5B9C
8B59-622F
8B5A-6280
8B5B-64EC
8B5C-6B3A
8B5D-72A0
8B5E-7591
8B5F-7947
8B60-7FA9
8B61-87FB
8B62-8ABC
8B63-8B70
8B64-63AC
8B65-83CA
8B66-97A0
8B67-5409
8B68-5403
8B69-55AB
8B6A-6854
8B6B-6A58
8B6C-8A70
8B6D-7827
8B6E-6775
8B6F-9ECD
8B70-5374
8B71-5BA2
8B72-811A
8B73-8650
8B74-9006
8B75-4E18
8B76-4E45
8B77-4EC7
8B78-4F11
8B79-53CA
8B7A-5438
8B7B-5BAE
8B7C-5F13
8B7D-6025
8B7E-6551
8B80-673D
8B81-6C42
8B82-6C72
8B83-6CE3
8B84-7078
8B85-7403
8B86-7A76
8B87-7AAE
8B88-7B08
8B89-7D1A
8B8A-7CFE
8B8B-7D66
8B8C-65E7
8B8D-725B
8B8E-53BB
8B8F-5C45
8B90-5DE8
8B91-62D2
8B92-62E0
8B93-6319
8B94-6E20
8B95-865A
8B96-8A31
8B97-8DDD
8B98-92F8
8B99-6F01
8B9A-79A6
8B9B-9B5A
8B9C-4EA8
8B9D-4EAB
8B9E-4EAC
8B9F-4F9B
8BA0-4FA0
8BA1-50D1
8BA2-5147
8BA3-7AF6
8BA4-5171
8BA5-51F6
8BA6-5354
8BA7-5321
8BA8-537F
8BA9-53EB
8BAA-55AC
8BAB-5883
8BAC-5CE1
8BAD-5F37
8BAE-5F4A
8BAF-602F
8BB0-6050
8BB1-606D
8BB2-631F
8BB3-6559
8BB4-6A4B
8BB5-6CC1
8BB6-72C2
8BB7-72ED
8BB8-77EF
8BB9-80F8
8BBA-8105
8BBB-8208
8BBC-854E
8BBD-90F7
8BBE-93E1
8BBF-97FF
8BC0-9957
8BC1-9A5A
8BC2-4EF0
8BC3-51DD
8BC4-5C2D
8BC5-6681
8BC6-696D
8BC7-5C40
8BC8-66F2
8BC9-6975
8BCA-7389
8BCB-6850
8BCC-7C81
8BCD-50C5
8BCE-52E4
8BCF-5747
8BD0-5DFE
8BD1-9326
8BD2-65A4
8BD3-6B23
8BD4-6B3D
8BD5-7434
8BD6-7981
8BD7-79BD
8BD8-7B4B
8BD9-7DCA
8BDA-82B9
8BDB-83CC
8BDC-887F
8BDD-895F
8BDE-8B39
8BDF-8FD1
8BE0-91D1
8BE1-541F
8BE2-9280
8BE3-4E5D
8BE4-5036
8BE5-53E5
8BE6-533A
8BE7-72D7
8BE8-7396
8BE9-77E9
8BEA-82E6
8BEB-8EAF
8BEC-99C6
8BED-99C8
8BEE-99D2
8BEF-5177
8BF0-611A
8BF1-865E
8BF2-55B0
8BF3-7A7A
8BF4-5076
8BF5-5BD3
8BF6-9047
8BF7-9685
8BF8-4E32
8BF9-6ADB
8BFA-91E7
8BFB-5C51
8BFC-5C48
8C40-6398
8C41-7A9F
8C42-6C93
8C43-9774
8C44-8F61
8C45-7AAA
8C46-718A
8C47-9688
8C48-7C82
8C49-6817
8C4A-7E70
8C4B-6851
8C4C-936C
8C4D-52F2
8C4E-541B
8C4F-85AB
8C50-8A13
8C51-7FA4
8C52-8ECD
8C53-90E1
8C54-5366
8C55-8888
8C56-7941
8C57-4FC2
8C58-50BE
8C59-5211
8C5A-5144
8C5B-5553
8C5C-572D
8C5D-73EA
8C5E-578B
8C5F-5951
8C60-5F62
8C61-5F84
8C62-6075
8C63-6176
8C64-6167
8C65-61A9
8C66-63B2
8C67-643A
8C68-656C
8C69-666F
8C6A-6842
8C6B-6E13
8C6C-7566
8C6D-7A3D
8C6E-7CFB
8C6F-7D4C
8C70-7D99
8C71-7E4B
8C72-7F6B
8C73-830E
8C74-834A
8C75-86CD
8C76-8A08
8C77-8A63
8C78-8B66
8C79-8EFD
8C7A-981A
8C7B-9D8F
8C7C-82B8
8C7D-8FCE
8C7E-9BE8
8C80-5287
8C81-621F
8C82-6483
8C83-6FC0
8C84-9699
8C85-6841
8C86-5091
8C87-6B20
8C88-6C7A
8C89-6F54
8C8A-7A74
8C8B-7D50
8C8C-8840
8C8D-8A23
8C8E-6708
8C8F-4EF6
8C90-5039
8C91-5026
8C92-5065
8C93-517C
8C94-5238
8C95-5263
8C96-55A7
8C97-570F
8C98-5805
8C99-5ACC
8C9A-5EFA
8C9B-61B2
8C9C-61F8
8C9D-62F3
8C9E-6372
8C9F-691C
8CA0-6A29
8CA1-727D
8CA2-72AC
8CA3-732E
8CA4-7814
8CA5-786F
8CA6-7D79
8CA7-770C
8CA8-80A9
8CA9-898B
8CAA-8B19
8CAB-8CE2
8CAC-8ED2
8CAD-9063
8CAE-9375
8CAF-967A
8CB0-9855
8CB1-9A13
8CB2-9E78
8CB3-5143
8CB4-539F
8CB5-53B3
8CB6-5E7B
8CB7-5F26
8CB8-6E1B
8CB9-6E90
8CBA-7384
8CBB-73FE
8CBC-7D43
8CBD-8237
8CBE-8A00
8CBF-8AFA
8CC0-9650
8CC1-4E4E
8CC2-500B
8CC3-53E4
8CC4-547C
8CC5-56FA
8CC6-59D1
8CC7-5B64
8CC8-5DF1
8CC9-5EAB
8CCA-5F27
8CCB-6238
8CCC-6545
8CCD-67AF
8CCE-6E56
8CCF-72D0
8CD0-7CCA
8CD1-88B4
8CD2-80A1
8CD3-80E1
8CD4-83F0
8CD5-864E
8CD6-8A87
8CD7-8DE8
8CD8-9237
8CD9-96C7
8CDA-9867
8CDB-9F13
8CDC-4E94
8CDD-4E92
8CDE-4F0D
8CDF-5348
8CE0-5449
8CE1-543E
8CE2-5A2F
8CE3-5F8C
8CE4-5FA1
8CE5-609F
8CE6-68A7
8CE7-6A8E
8CE8-745A
8CE9-7881
8CEA-8A9E
8CEB-8AA4
8CEC-8B77
8CED-9190
8CEE-4E5E
8CEF-9BC9
8CF0-4EA4
8CF1-4F7C
8CF2-4FAF
8CF3-5019
8CF4-5016
8CF5-5149
8CF6-516C
8CF7-529F
8CF8-52B9
8CF9-52FE
8CFA-539A
8CFB-53E3
8CFC-5411
8D40-540E
8D41-5589
8D42-5751
8D43-57A2
8D44-597D
8D45-5B54
8D46-5B5D
8D47-5B8F
8D48-5DE5
8D49-5DE7
8D4A-5DF7
8D4B-5E78
8D4C-5E83
8D4D-5E9A
8D4E-5EB7
8D4F-5F18
8D50-6052
8D51-614C
8D52-6297
8D53-62D8
8D54-63A7
8D55-653B
8D56-6602
8D57-6643
8D58-66F4
8D59-676D
8D5A-6821
8D5B-6897
8D5C-69CB
8D5D-6C5F
8D5E-6D2A
8D5F-6D69
8D60-6E2F
8D61-6E9D
8D62-7532
8D63-7687
8D64-786C
8D65-7A3F
8D66-7CE0
8D67-7D05
8D68-7D18
8D69-7D5E
8D6A-7DB1
8D6B-8015
8D6C-8003
8D6D-80AF
8D6E-80B1
8D6F-8154
8D70-818F
8D71-822A
8D72-8352
8D73-884C
8D74-8861
8D75-8B1B
8D76-8CA2
8D77-8CFC
8D78-90CA
8D79-9175
8D7A-9271
8D7B-783F
8D7C-92FC
8D7D-95A4
8D7E-964D
8D80-9805
8D81-9999
8D82-9AD8
8D83-9D3B
8D84-525B
8D85-52AB
8D86-53F7
8D87-5408
8D88-58D5
8D89-62F7
8D8A-6FE0
8D8B-8C6A
8D8C-8F5F
8D8D-9EB9
8D8E-514B
8D8F-523B
8D90-544A
8D91-56FD
8D92-7A40
8D93-9177
8D94-9D60
8D95-9ED2
8D96-7344
8D97-6F09
8D98-8170
8D99-7511
8D9A-5FFD
8D9B-60DA
8D9C-9AA8
8D9D-72DB
8D9E-8FBC
8D9F-6B64
8DA0-9803
8DA1-4ECA
8DA2-56F0
8DA3-5764
8DA4-58BE
8DA5-5A5A
8DA6-6068
8DA7-61C7
8DA8-660F
8DA9-6606
8DAA-6839
8DAB-68B1
8DAC-6DF7
8DAD-75D5
8DAE-7D3A
8DAF-826E
8DB0-9B42
8DB1-4E9B
8DB2-4F50
8DB3-53C9
8DB4-5506
8DB5-5D6F
8DB6-5DE6
8DB7-5DEE
8DB8-67FB
8DB9-6C99
8DBA-7473
8DBB-7802
8DBC-8A50
8DBD-9396
8DBE-88DF
8DBF-5750
8DC0-5EA7
8DC1-632B
8DC2-50B5
8DC3-50AC
8DC4-518D
8DC5-6700
8DC6-54C9
8DC7-585E
8DC8-59BB
8DC9-5BB0
8DCA-5F69
8DCB-624D
8DCC-63A1
8DCD-683D
8DCE-6B73
8DCF-6E08
8DD0-707D
8DD1-91C7
8DD2-7280
8DD3-7815
8DD4-7826
8DD5-796D
8DD6-658E
8DD7-7D30
8DD8-83DC
8DD9-88C1
8DDA-8F09
8DDB-969B
8DDC-5264
8DDD-5728
8DDE-6750
8DDF-7F6A
8DE0-8CA1
8DE1-51B4
8DE2-5742
8DE3-962A
8DE4-583A
8DE5-698A
8DE6-80B4
8DE7-54B2
8DE8-5D0E
8DE9-57FC
8DEA-7895
8DEB-9DFA
8DEC-4F5C
8DED-524A
8DEE-548B
8DEF-643E
8DF0-6628
8DF1-6714
8DF2-67F5
8DF3-7A84
8DF4-7B56
8DF5-7D22
8DF6-932F
8DF7-685C
8DF8-9BAD
8DF9-7B39
8DFA-5319
8DFB-518A
8DFC-5237
8E40-5BDF
8E41-62F6
8E42-64AE
8E43-64E6
8E44-672D
8E45-6BBA
8E46-85A9
8E47-96D1
8E48-7690
8E49-9BD6
8E4A-634C
8E4B-9306
8E4C-9BAB
8E4D-76BF
8E4E-6652
8E4F-4E09
8E50-5098
8E51-53C2
8E52-5C71
8E53-60E8
8E54-6492
8E55-6563
8E56-685F
8E57-71E6
8E58-73CA
8E59-7523
8E5A-7B97
8E5B-7E82
8E5C-8695
8E5D-8B83
8E5E-8CDB
8E5F-9178
8E60-9910
8E61-65AC
8E62-66AB
8E63-6B8B
8E64-4ED5
8E65-4ED4
8E66-4F3A
8E67-4F7F
8E68-523A
8E69-53F8
8E6A-53F2
8E6B-55E3
8E6C-56DB
8E6D-58EB
8E6E-59CB
8E6F-59C9
8E70-59FF
8E71-5B50
8E72-5C4D
8E73-5E02
8E74-5E2B
8E75-5FD7
8E76-601D
8E77-6307
8E78-652F
8E79-5B5C
8E7A-65AF
8E7B-65BD
8E7C-65E8
8E7D-679D
8E7E-6B62
8E80-6B7B
8E81-6C0F
8E82-7345
8E83-7949
8E84-79C1
8E85-7CF8
8E86-7D19
8E87-7D2B
8E88-80A2
8E89-8102
8E8A-81F3
8E8B-8996
8E8C-8A5E
8E8D-8A69
8E8E-8A66
8E8F-8A8C
8E90-8AEE
8E91-8CC7
8E92-8CDC
8E93-96CC
8E94-98FC
8E95-6B6F
8E96-4E8B
8E97-4F3C
8E98-4F8D
8E99-5150
8E9A-5B57
8E9B-5BFA
8E9C-6148
8E9D-6301
8E9E-6642
8E9F-6B21
8EA0-6ECB
8EA1-6CBB
8EA2-723E
8EA3-74BD
8EA4-75D4
8EA5-78C1
8EA6-793A
8EA7-800C
8EA8-8033
8EA9-81EA
8EAA-8494
8EAB-8F9E
8EAC-6C50
8EAD-9E7F
8EAE-5F0F
8EAF-8B58
8EB0-9D2B
8EB1-7AFA
8EB2-8EF8
8EB3-5B8D
8EB4-96EB
8EB5-4E03
8EB6-53F1
8EB7-57F7
8EB8-5931
8EB9-5AC9
8EBA-5BA4
8EBB-6089
8EBC-6E7F
8EBD-6F06
8EBE-75BE
8EBF-8CEA
8EC0-5B9F
8EC1-8500
8EC2-7BE0
8EC3-5072
8EC4-67F4
8EC5-829D
8EC6-5C61
8EC7-854A
8EC8-7E1E
8EC9-820E
8ECA-5199
8ECB-5C04
8ECC-6368
8ECD-8D66
8ECE-659C
8ECF-716E
8ED0-793E
8ED1-7D17
8ED2-8005
8ED3-8B1D
8ED4-8ECA
8ED5-906E
8ED6-86C7
8ED7-90AA
8ED8-501F
8ED9-52FA
8EDA-5C3A
8EDB-6753
8EDC-707C
8EDD-7235
8EDE-914C
8EDF-91C8
8EE0-932B
8EE1-82E5
8EE2-5BC2
8EE3-5F31
8EE4-60F9
8EE5-4E3B
8EE6-53D6
8EE7-5B88
8EE8-624B
8EE9-6731
8EEA-6B8A
8EEB-72E9
8EEC-73E0
8EED-7A2E
8EEE-816B
8EEF-8DA3
8EF0-9152
8EF1-9996
8EF2-5112
8EF3-53D7
8EF4-546A
8EF5-5BFF
8EF6-6388
8EF7-6A39
8EF8-7DAC
8EF9-9700
8EFA-56DA
8EFB-53CE
8EFC-5468
8F40-5B97
8F41-5C31
8F42-5DDE
8F43-4FEE
8F44-6101
8F45-62FE
8F46-6D32
8F47-79C0
8F48-79CB
8F49-7D42
8F4A-7E4D
8F4B-7FD2
8F4C-81ED
8F4D-821F
8F4E-8490
8F4F-8846
8F50-8972
8F51-8B90
8F52-8E74
8F53-8F2F
8F54-9031
8F55-914B
8F56-916C
8F57-96C6
8F58-919C
8F59-4EC0
8F5A-4F4F
8F5B-5145
8F5C-5341
8F5D-5F93
8F5E-620E
8F5F-67D4
8F60-6C41
8F61-6E0B
8F62-7363
8F63-7E26
8F64-91CD
8F65-9283
8F66-53D4
8F67-5919
8F68-5BBF
8F69-6DD1
8F6A-795D
8F6B-7E2E
8F6C-7C9B
8F6D-587E
8F6E-719F
8F6F-51FA
8F70-8853
8F71-8FF0
8F72-4FCA
8F73-5CFB
8F74-6625
8F75-77AC
8F76-7AE3
8F77-821C
8F78-99FF
8F79-51C6
8F7A-5FAA
8F7B-65EC
8F7C-696F
8F7D-6B89
8F7E-6DF3
8F80-6E96
8F81-6F64
8F82-76FE
8F83-7D14
8F84-5DE1
8F85-9075
8F86-9187
8F87-9806
8F88-51E6
8F89-521D
8F8A-6240
8F8B-6691
8F8C-66D9
8F8D-6E1A
8F8E-5EB6
8F8F-7DD2
8F90-7F72
8F91-66F8
8F92-85AF
8F93-85F7
8F94-8AF8
8F95-52A9
8F96-53D9
8F97-5973
8F98-5E8F
8F99-5F90
8F9A-6055
8F9B-92E4
8F9C-9664
8F9D-50B7
8F9E-511F
8F9F-52DD
8FA0-5320
8FA1-5347
8FA2-53EC
8FA3-54E8
8FA4-5546
8FA5-5531
8FA6-5617
8FA7-5968
8FA8-59BE
8FA9-5A3C
8FAA-5BB5
8FAB-5C06
8FAC-5C0F
8FAD-5C11
8FAE-5C1A
8FAF-5E84
8FB0-5E8A
8FB1-5EE0
8FB2-5F70
8FB3-627F
8FB4-6284
8FB5-62DB
8FB6-638C
8FB7-6377
8FB8-6607
8FB9-660C
8FBA-662D
8FBB-6676
8FBC-677E
8FBD-68A2
8FBE-6A1F
8FBF-6A35
8FC0-6CBC
8FC1-6D88
8FC2-6E09
8FC3-6E58
8FC4-713C
8FC5-7126
8FC6-7167
8FC7-75C7
8FC8-7701
8FC9-785D
8FCA-7901
8FCB-7965
8FCC-79F0
8FCD-7AE0
8FCE-7B11
8FCF-7CA7
8FD0-7D39
8FD1-8096
8FD2-83D6
8FD3-848B
8FD4-8549
8FD5-885D
8FD6-88F3
8FD7-8A1F
8FD8-8A3C
8FD9-8A54
8FDA-8A73
8FDB-8C61
8FDC-8CDE
8FDD-91A4
8FDE-9266
8FDF-937E
8FE0-9418
8FE1-969C
8FE2-9798
8FE3-4E0A
8FE4-4E08
8FE5-4E1E
8FE6-4E57
8FE7-5197
8FE8-5270
8FE9-57CE
8FEA-5834
8FEB-58CC
8FEC-5B22
8FED-5E38
8FEE-60C5
8FEF-64FE
8FF0-6761
8FF1-6756
8FF2-6D44
8FF3-72B6
8FF4-7573
8FF5-7A63
8FF6-84B8
8FF7-8B72
8FF8-91B8
8FF9-9320
8FFA-5631
8FFB-57F4
8FFC-98FE
9040-62ED
9041-690D
9042-6B96
9043-71ED
9044-7E54
9045-8077
9046-8272
9047-89E6
9048-98DF
9049-8755
904A-8FB1
904B-5C3B
904C-4F38
904D-4FE1
904E-4FB5
904F-5507
9050-5A20
9051-5BDD
9052-5BE9
9053-5FC3
9054-614E
9055-632F
9056-65B0
9057-664B
9058-68EE
9059-699B
905A-6D78
905B-6DF1
905C-7533
905D-75B9
905E-771F
905F-795E
9060-79E6
9061-7D33
9062-81E3
9063-82AF
9064-85AA
9065-89AA
9066-8A3A
9067-8EAB
9068-8F9B
9069-9032
906A-91DD
906B-9707
906C-4EBA
906D-4EC1
906E-5203
906F-5875
9070-58EC
9071-5C0B
9072-751A
9073-5C3D
9074-814E
9075-8A0A
9076-8FC5
9077-9663
9078-976D
9079-7B25
907A-8ACF
907B-9808
907C-9162
907D-56F3
907E-53A8
9080-9017
9081-5439
9082-5782
9083-5E25
9084-63A8
9085-6C34
9086-708A
9087-7761
9088-7C8B
9089-7FE0
908A-8870
908B-9042
908C-9154
908D-9310
908E-9318
908F-968F
9090-745E
9091-9AC4
9092-5D07
9093-5D69
9094-6570
9095-67A2
9096-8DA8
9097-96DB
9098-636E
9099-6749
909A-6919
909B-83C5
909C-9817
909D-96C0
909E-88FE
909F-6F84
90A0-647A
90A1-5BF8
90A2-4E16
90A3-702C
90A4-755D
90A5-662F
90A6-51C4
90A7-5236
90A8-52E2
90A9-59D3
90AA-5F81
90AB-6027
90AC-6210
90AD-653F
90AE-6574
90AF-661F
90B0-6674
90B1-68F2
90B2-6816
90B3-6B63
90B4-6E05
90B5-7272
90B6-751F
90B7-76DB
90B8-7CBE
90B9-8056
90BA-58F0
90BB-88FD
90BC-897F
90BD-8AA0
90BE-8A93
90BF-8ACB
90C0-901D
90C1-9192
90C2-9752
90C3-9759
90C4-6589
90C5-7A0E
90C6-8106
90C7-96BB
90C8-5E2D
90C9-60DC
90CA-621A
90CB-65A5
90CC-6614
90CD-6790
90CE-77F3
90CF-7A4D
90D0-7C4D
90D1-7E3E
90D2-810A
90D3-8CAC
90D4-8D64
90D5-8DE1
90D6-8E5F
90D7-78A9
90D8-5207
90D9-62D9
90DA-63A5
90DB-6442
90DC-6298
90DD-8A2D
90DE-7A83
90DF-7BC0
90E0-8AAC
90E1-96EA
90E2-7D76
90E3-820C
90E4-8749
90E5-4ED9
90E6-5148
90E7-5343
90E8-5360
90E9-5BA3
90EA-5C02
90EB-5C16
90EC-5DDD
90ED-6226
90EE-6247
90EF-64B0
90F0-6813
90F1-6834
90F2-6CC9
90F3-6D45
90F4-6D17
90F5-67D3
90F6-6F5C
90F7-714E
90F8-717D
90F9-65CB
90FA-7A7F
90FB-7BAD
90FC-7DDA
9140-7E4A
9141-7FA8
9142-817A
9143-821B
9144-8239
9145-85A6
9146-8A6E
9147-8CCE
9148-8DF5
9149-9078
914A-9077
914B-92AD
914C-9291
914D-9583
914E-9BAE
914F-524D
9150-5584
9151-6F38
9152-7136
9153-5168
9154-7985
9155-7E55
9156-81B3
9157-7CCE
9158-564C
9159-5851
915A-5CA8
915B-63AA
915C-66FE
915D-66FD
915E-695A
915F-72D9
9160-758F
9161-758E
9162-790E
9163-7956
9164-79DF
9165-7C97
9166-7D20
9167-7D44
9168-8607
9169-8A34
916A-963B
916B-9061
916C-9F20
916D-50E7
916E-5275
916F-53CC
9170-53E2
9171-5009
9172-55AA
9173-58EE
9174-594F
9175-723D
9176-5B8B
9177-5C64
9178-531D
9179-60E3
917A-60F3
917B-635C
917C-6383
917D-633F
917E-63BB
9180-64CD
9181-65E9
9182-66F9
9183-5DE3
9184-69CD
9185-69FD
9186-6F15
9187-71E5
9188-4E89
9189-75E9
918A-76F8
918B-7A93
918C-7CDF
918D-7DCF
918E-7D9C
918F-8061
9190-8349
9191-8358
9192-846C
9193-84BC
9194-85FB
9195-88C5
9196-8D70
9197-9001
9198-906D
9199-9397
919A-971C
919B-9A12
919C-50CF
919D-5897
919E-618E
919F-81D3
91A0-8535
91A1-8D08
91A2-9020
91A3-4FC3
91A4-5074
91A5-5247
91A6-5373
91A7-606F
91A8-6349
91A9-675F
91AA-6E2C
91AB-8DB3
91AC-901F
91AD-4FD7
91AE-5C5E
91AF-8CCA
91B0-65CF
91B1-7D9A
91B2-5352
91B3-8896
91B4-5176
91B5-63C3
91B6-5B58
91B7-5B6B
91B8-5C0A
91B9-640D
91BA-6751
91BB-905C
91BC-4ED6
91BD-591A
91BE-592A
91BF-6C70
91C0-8A51
91C1-553E
91C2-5815
91C3-59A5
91C4-60F0
91C5-6253
91C6-67C1
91C7-8235
91C8-6955
91C9-9640
91CA-99C4
91CB-9A28
91CC-4F53
91CD-5806
91CE-5BFE
91CF-8010
91D0-5CB1
91D1-5E2F
91D2-5F85
91D3-6020
91D4-614B
91D5-6234
91D6-66FF
91D7-6CF0
91D8-6EDE
91D9-80CE
91DA-817F
91DB-82D4
91DC-888B
91DD-8CB8
91DE-9000
91DF-902E
91E0-968A
91E1-9EDB
91E2-9BDB
91E3-4EE3
91E4-53F0
91E5-5927
91E6-7B2C
91E7-918D
91E8-984C
91E9-9DF9
91EA-6EDD
91EB-7027
91EC-5353
91ED-5544
91EE-5B85
91EF-6258
91F0-629E
91F1-62D3
91F2-6CA2
91F3-6FEF
91F4-7422
91F5-8A17
91F6-9438
91F7-6FC1
91F8-8AFE
91F9-8338
91FA-51E7
91FB-86F8
91FC-53EA
9240-53E9
9241-4F46
9242-9054
9243-8FB0
9244-596A
9245-8131
9246-5DFD
9247-7AEA
9248-8FBF
9249-68DA
924A-8C37
924B-72F8
924C-9C48
924D-6A3D
924E-8AB0
924F-4E39
9250-5358
9251-5606
9252-5766
9253-62C5
9254-63A2
9255-65E6
9256-6B4E
9257-6DE1
9258-6E5B
9259-70AD
925A-77ED
925B-7AEF
925C-7BAA
925D-7DBB
925E-803D
925F-80C6
9260-86CB
9261-8A95
9262-935B
9263-56E3
9264-58C7
9265-5F3E
9266-65AD
9267-6696
9268-6A80
9269-6BB5
926A-7537
926B-8AC7
926C-5024
926D-77E5
926E-5730
926F-5F1B
9270-6065
9271-667A
9272-6C60
9273-75F4
9274-7A1A
9275-7F6E
9276-81F4
9277-8718
9278-9045
9279-99B3
927A-7BC9
927B-755C
927C-7AF9
927D-7B51
927E-84C4
9280-9010
9281-79E9
9282-7A92
9283-8336
9284-5AE1
9285-7740
9286-4E2D
9287-4EF2
9288-5B99
9289-5FE0
928A-62BD
928B-663C
928C-67F1
928D-6CE8
928E-866B
928F-8877
9290-8A3B
9291-914E
9292-92F3
9293-99D0
9294-6A17
9295-7026
9296-732A
9297-82E7
9298-8457
9299-8CAF
929A-4E01
929B-5146
929C-51CB
929D-558B
929E-5BF5
929F-5E16
92A0-5E33
92A1-5E81
92A2-5F14
92A3-5F35
92A4-5F6B
92A5-5FB4
92A6-61F2
92A7-6311
92A8-66A2
92A9-671D
92AA-6F6E
92AB-7252
92AC-753A
92AD-773A
92AE-8074
92AF-8139
92B0-8178
92B1-8776
92B2-8ABF
92B3-8ADC
92B4-8D85
92B5-8DF3
92B6-929A
92B7-9577
92B8-9802
92B9-9CE5
92BA-52C5
92BB-6357
92BC-76F4
92BD-6715
92BE-6C88
92BF-73CD
92C0-8CC3
92C1-93AE
92C2-9673
92C3-6D25
92C4-589C
92C5-690E
92C6-69CC
92C7-8FFD
92C8-939A
92C9-75DB
92CA-901A
92CB-585A
92CC-6802
92CD-63B4
92CE-69FB
92CF-4F43
92D0-6F2C
92D1-67D8
92D2-8FBB
92D3-8526
92D4-7DB4
92D5-9354
92D6-693F
92D7-6F70
92D8-576A
92D9-58F7
92DA-5B2C
92DB-7D2C
92DC-722A
92DD-540A
92DE-91E3
92DF-9DB4
92E0-4EAD
92E1-4F4E
92E2-505C
92E3-5075
92E4-5243
92E5-8C9E
92E6-5448
92E7-5824
92E8-5B9A
92E9-5E1D
92EA-5E95
92EB-5EAD
92EC-5EF7
92ED-5F1F
92EE-608C
92EF-62B5
92F0-633A
92F1-63D0
92F2-68AF
92F3-6C40
92F4-7887
92F5-798E
92F6-7A0B
92F7-7DE0
92F8-8247
92F9-8A02
92FA-8AE6
92FB-8E44
92FC-9013
9340-90B8
9341-912D
9342-91D8
9343-9F0E
9344-6CE5
9345-6458
9346-64E2
9347-6575
9348-6EF4
9349-7684
934A-7B1B
934B-9069
934C-93D1
934D-6EBA
934E-54F2
934F-5FB9
9350-64A4
9351-8F4D
9352-8FED
9353-9244
9354-5178
9355-586B
9356-5929
9357-5C55
9358-5E97
9359-6DFB
935A-7E8F
935B-751C
935C-8CBC
935D-8EE2
935E-985B
935F-70B9
9360-4F1D
9361-6BBF
9362-6FB1
9363-7530
9364-96FB
9365-514E
9366-5410
9367-5835
9368-5857
9369-59AC
936A-5C60
936B-5F92
936C-6597
936D-675C
936E-6E21
936F-767B
9370-83DF
9371-8CED
9372-9014
9373-90FD
9374-934D
9375-7825
9376-783A
9377-52AA
9378-5EA6
9379-571F
937A-5974
937B-6012
937C-5012
937D-515A
937E-51AC
9380-51CD
9381-5200
9382-5510
9383-5854
9384-5858
9385-5957
9386-5B95
9387-5CF6
9388-5D8B
9389-60BC
938A-6295
938B-642D
938C-6771
938D-6843
938E-68BC
938F-68DF
9390-76D7
9391-6DD8
9392-6E6F
9393-6D9B
9394-706F
9395-71C8
9396-5F53
9397-75D8
9398-7977
9399-7B49
939A-7B54
939B-7B52
939C-7CD6
939D-7D71
939E-5230
939F-8463
93A0-8569
93A1-85E4
93A2-8A0E
93A3-8B04
93A4-8C46
93A5-8E0F
93A6-9003
93A7-900F
93A8-9419
93A9-9676
93AA-982D
93AB-9A30
93AC-95D8
93AD-50CD
93AE-52D5
93AF-540C
93B0-5802
93B1-5C0E
93B2-61A7
93B3-649E
93B4-6D1E
93B5-77B3
93B6-7AE5
93B7-80F4
93B8-8404
93B9-9053
93BA-9285
93BB-5CE0
93BC-9D07
93BD-533F
93BE-5F97
93BF-5FB3
93C0-6D9C
93C1-7279
93C2-7763
93C3-79BF
93C4-7BE4
93C5-6BD2
93C6-72EC
93C7-8AAD
93C8-6803
93C9-6A61
93CA-51F8
93CB-7A81
93CC-6934
93CD-5C4A
93CE-9CF6
93CF-82EB
93D0-5BC5
93D1-9149
93D2-701E
93D3-5678
93D4-5C6F
93D5-60C7
93D6-6566
93D7-6C8C
93D8-8C5A
93D9-9041
93DA-9813
93DB-5451
93DC-66C7
93DD-920D
93DE-5948
93DF-90A3
93E0-5185
93E1-4E4D
93E2-51EA
93E3-8599
93E4-8B0E
93E5-7058
93E6-637A
93E7-934B
93E8-6962
93E9-99B4
93EA-7E04
93EB-7577
93EC-5357
93ED-6960
93EE-8EDF
93EF-96E3
93F0-6C5D
93F1-4E8C
93F2-5C3C
93F3-5F10
93F4-8FE9
93F5-5302
93F6-8CD1
93F7-8089
93F8-8679
93F9-5EFF
93FA-65E5
93FB-4E73
93FC-5165
9440-5982
9441-5C3F
9442-97EE
9443-4EFB
9444-598A
9445-5FCD
9446-8A8D
9447-6FE1
9448-79B0
9449-7962
944A-5BE7
944B-8471
944C-732B
944D-71B1
944E-5E74
944F-5FF5
9450-637B
9451-649A
9452-71C3
9453-7C98
9454-4E43
9455-5EFC
9456-4E4B
9457-57DC
9458-56A2
9459-60A9
945A-6FC3
945B-7D0D
945C-80FD
945D-8133
945E-81BF
945F-8FB2
9460-8997
9461-86A4
9462-5DF4
9463-628A
9464-64AD
9465-8987
9466-6777
9467-6CE2
9468-6D3E
9469-7436
946A-7834
946B-5A46
946C-7F75
946D-82AD
946E-99AC
946F-4FF3
9470-5EC3
9471-62DD
9472-6392
9473-6557
9474-676F
9475-76C3
9476-724C
9477-80CC
9478-80BA
9479-8F29
947A-914D
947B-500D
947C-57F9
947D-5A92
947E-6885
9480-6973
9481-7164
9482-72FD
9483-8CB7
9484-58F2
9485-8CE0
9486-966A
9487-9019
9488-877F
9489-79E4
948A-77E7
948B-8429
948C-4F2F
948D-5265
948E-535A
948F-62CD
9490-67CF
9491-6CCA
9492-767D
9493-7B94
9494-7C95
9495-8236
9496-8584
9497-8FEB
9498-66DD
9499-6F20
949A-7206
949B-7E1B
949C-83AB
949D-99C1
949E-9EA6
949F-51FD
94A0-7BB1
94A1-7872
94A2-7BB8
94A3-8087
94A4-7B48
94A5-6AE8
94A6-5E61
94A7-808C
94A8-7551
94A9-7560
94AA-516B
94AB-9262
94AC-6E8C
94AD-767A
94AE-9197
94AF-9AEA
94B0-4F10
94B1-7F70
94B2-629C
94B3-7B4F
94B4-95A5
94B5-9CE9
94B6-567A
94B7-5859
94B8-86E4
94B9-96BC
94BA-4F34
94BB-5224
94BC-534A
94BD-53CD
94BE-53DB
94BF-5E06
94C0-642C
94C1-6591
94C2-677F
94C3-6C3E
94C4-6C4E
94C5-7248
94C6-72AF
94C7-73ED
94C8-7554
94C9-7E41
94CA-822C
94CB-85E9
94CC-8CA9
94CD-7BC4
94CE-91C6
94CF-7169
94D0-9812
94D1-98EF
94D2-633D
94D3-6669
94D4-756A
94D5-76E4
94D6-78D0
94D7-8543
94D8-86EE
94D9-532A
94DA-5351
94DB-5426
94DC-5983
94DD-5E87
94DE-5F7C
94DF-60B2
94E0-6249
94E1-6279
94E2-62AB
94E3-6590
94E4-6BD4
94E5-6CCC
94E6-75B2
94E7-76AE
94E8-7891
94E9-79D8
94EA-7DCB
94EB-7F77
94EC-80A5
94ED-88AB
94EE-8AB9
94EF-8CBB
94F0-907F
94F1-975E
94F2-98DB
94F3-6A0B
94F4-7C38
94F5-5099
94F6-5C3E
94F7-5FAE
94F8-6787
94F9-6BD8
94FA-7435
94FB-7709
94FC-7F8E
9540-9F3B
9541-67CA
9542-7A17
9543-5339
9544-758B
9545-9AED
9546-5F66
9547-819D
9548-83F1
9549-8098
954A-5F3C
954B-5FC5
954C-7562
954D-7B46
954E-903C
954F-6867
9550-59EB
9551-5A9B
9552-7D10
9553-767E
9554-8B2C
9555-4FF5
9556-5F6A
9557-6A19
9558-6C37
9559-6F02
955A-74E2
955B-7968
955C-8868
955D-8A55
955E-8C79
955F-5EDF
9560-63CF
9561-75C5
9562-79D2
9563-82D7
9564-9328
9565-92F2
9566-849C
9567-86ED
9568-9C2D
9569-54C1
956A-5F6C
956B-658C
956C-6D5C
956D-7015
956E-8CA7
956F-8CD3
9570-983B
9571-654F
9572-74F6
9573-4E0D
9574-4ED8
9575-57E0
9576-592B
9577-5A66
9578-5BCC
9579-51A8
957A-5E03
957B-5E9C
957C-6016
957D-6276
957E-6577
9580-65A7
9581-666E
9582-6D6E
9583-7236
9584-7B26
9585-8150
9586-819A
9587-8299
9588-8B5C
9589-8CA0
958A-8CE6
958B-8D74
958C-961C
958D-9644
958E-4FAE
958F-64AB
9590-6B66
9591-821E
9592-8461
9593-856A
9594-90E8
9595-5C01
9596-6953
9597-98A8
9598-847A
9599-8557
959A-4F0F
959B-526F
959C-5FA9
959D-5E45
959E-670D
959F-798F
95A0-8179
95A1-8907
95A2-8986
95A3-6DF5
95A4-5F17
95A5-6255
95A6-6CB8
95A7-4ECF
95A8-7269
95A9-9B92
95AA-5206
95AB-543B
95AC-5674
95AD-58B3
95AE-61A4
95AF-626E
95B0-711A
95B1-596E
95B2-7C89
95B3-7CDE
95B4-7D1B
95B5-96F0
95B6-6587
95B7-805E
95B8-4E19
95B9-4F75
95BA-5175
95BB-5840
95BC-5E63
95BD-5E73
95BE-5F0A
95BF-67C4
95C0-4E26
95C1-853D
95C2-9589
95C3-965B
95C4-7C73
95C5-9801
95C6-50FB
95C7-58C1
95C8-7656
95C9-78A7
95CA-5225
95CB-77A5
95CC-8511
95CD-7B86
95CE-504F
95CF-5909
95D0-7247
95D1-7BC7
95D2-7DE8
95D3-8FBA
95D4-8FD4
95D5-904D
95D6-4FBF
95D7-52C9
95D8-5A29
95D9-5F01
95DA-97AD
95DB-4FDD
95DC-8217
95DD-92EA
95DE-5703
95DF-6355
95E0-6B69
95E1-752B
95E2-88DC
95E3-8F14
95E4-7A42
95E5-52DF
95E6-5893
95E7-6155
95E8-620A
95E9-66AE
95EA-6BCD
95EB-7C3F
95EC-83E9
95ED-5023
95EE-4FF8
95EF-5305
95F0-5446
95F1-5831
95F2-5949
95F3-5B9D
95F4-5CF0
95F5-5CEF
95F6-5D29
95F7-5E96
95F8-62B1
95F9-6367
95FA-653E
95FB-65B9
95FC-670B
9640-6CD5
9641-6CE1
9642-70F9
9643-7832
9644-7E2B
9645-80DE
9646-82B3
9647-840C
9648-84EC
9649-8702
964A-8912
964B-8A2A
964C-8C4A
964D-90A6
964E-92D2
964F-98FD
9650-9CF3
9651-9D6C
9652-4E4F
9653-4EA1
9654-508D
9655-5256
9656-574A
9657-59A8
9658-5E3D
9659-5FD8
965A-5FD9
965B-623F
965C-66B4
965D-671B
965E-67D0
965F-68D2
9660-5192
9661-7D21
9662-80AA
9663-81A8
9664-8B00
9665-8C8C
9666-8CBF
9667-927E
9668-9632
9669-5420
966A-982C
966B-5317
966C-50D5
966D-535C
966E-58A8
966F-64B2
9670-6734
9671-7267
9672-7766
9673-7A46
9674-91E6
9675-52C3
9676-6CA1
9677-6B86
9678-5800
9679-5E4C
967A-5954
967B-672C
967C-7FFB
967D-51E1
967E-76C6
9680-6469
9681-78E8
9682-9B54
9683-9EBB
9684-57CB
9685-59B9
9686-6627
9687-679A
9688-6BCE
9689-54E9
968A-69D9
968B-5E55
968C-819C
968D-6795
968E-9BAA
968F-67FE
9690-9C52
9691-685D
9692-4EA6
9693-4FE3
9694-53C8
9695-62B9
9696-672B
9697-6CAB
9698-8FC4
9699-4FAD
969A-7E6D
969B-9EBF
969C-4E07
969D-6162
969E-6E80
969F-6F2B
96A0-8513
96A1-5473
96A2-672A
96A3-9B45
96A4-5DF3
96A5-7B95
96A6-5CAC
96A7-5BC6
96A8-871C
96A9-6E4A
96AA-84D1
96AB-7A14
96AC-8108
96AD-5999
96AE-7C8D
96AF-6C11
96B0-7720
96B1-52D9
96B2-5922
96B3-7121
96B4-725F
96B5-77DB
96B6-9727
96B7-9D61
96B8-690B
96B9-5A7F
96BA-5A18
96BB-51A5
96BC-540D
96BD-547D
96BE-660E
96BF-76DF
96C0-8FF7
96C1-9298
96C2-9CF4
96C3-59EA
96C4-725D
96C5-6EC5
96C6-514D
96C7-68C9
96C8-7DBF
96C9-7DEC
96CA-9762
96CB-9EBA
96CC-6478
96CD-6A21
96CE-8302
96CF-5984
96D0-5B5F
96D1-6BDB
96D2-731B
96D3-76F2
96D4-7DB2
96D5-8017
96D6-8499
96D7-5132
96D8-6728
96D9-9ED9
96DA-76EE
96DB-6762
96DC-52FF
96DD-9905
96DE-5C24
96DF-623B
96E0-7C7E
96E1-8CB0
96E2-554F
96E3-60B6
96E4-7D0B
96E5-9580
96E6-5301
96E7-4E5F
96E8-51B6
96E9-591C
96EA-723A
96EB-8036
96EC-91CE
96ED-5F25
96EE-77E2
96EF-5384
96F0-5F79
96F1-7D04
96F2-85AC
96F3-8A33
96F4-8E8D
96F5-9756
96F6-67F3
96F7-85AE
96F8-9453
96F9-6109
96FA-6108
96FB-6CB9
96FC-7652
9740-8AED
9741-8F38
9742-552F
9743-4F51
9744-512A
9745-52C7
9746-53CB
9747-5BA5
9748-5E7D
9749-60A0
974A-6182
974B-63D6
974C-6709
974D-67DA
974E-6E67
974F-6D8C
9750-7336
9751-7337
9752-7531
9753-7950
9754-88D5
9755-8A98
9756-904A
9757-9091
9758-90F5
9759-96C4
975A-878D
975B-5915
975C-4E88
975D-4F59
975E-4E0E
975F-8A89
9760-8F3F
9761-9810
9762-50AD
9763-5E7C
9764-5996
9765-5BB9
9766-5EB8
9767-63DA
9768-63FA
9769-64C1
976A-66DC
976B-694A
976C-69D8
976D-6D0B
976E-6EB6
976F-7194
9770-7528
9771-7AAF
9772-7F8A
9773-8000
9774-8449
9775-84C9
9776-8981
9777-8B21
9778-8E0A
9779-9065
977A-967D
977B-990A
977C-617E
977D-6291
977E-6B32
9780-6C83
9781-6D74
9782-7FCC
9783-7FFC
9784-6DC0
9785-7F85
9786-87BA
9787-88F8
9788-6765
9789-83B1
978A-983C
978B-96F7
978C-6D1B
978D-7D61
978E-843D
978F-916A
9790-4E71
9791-5375
9792-5D50
9793-6B04
9794-6FEB
9795-85CD
9796-862D
9797-89A7
9798-5229
9799-540F
979A-5C65
979B-674E
979C-68A8
979D-7406
979E-7483
979F-75E2
97A0-88CF
97A1-88E1
97A2-91CC
97A3-96E2
97A4-9678
97A5-5F8B
97A6-7387
97A7-7ACB
97A8-844E
97A9-63A0
97AA-7565
97AB-5289
97AC-6D41
97AD-6E9C
97AE-7409
97AF-7559
97B0-786B
97B1-7C92
97B2-9686
97B3-7ADC
97B4-9F8D
97B5-4FB6
97B6-616E
97B7-65C5
97B8-865C
97B9-4E86
97BA-4EAE
97BB-50DA
97BC-4E21
97BD-51CC
97BE-5BEE
97BF-6599
97C0-6881
97C1-6DBC
97C2-731F
97C3-7642
97C4-77AD
97C5-7A1C
97C6-7CE7
97C7-826F
97C8-8AD2
97C9-907C
97CA-91CF
97CB-9675
97CC-9818
97CD-529B
97CE-7DD1
97CF-502B
97D0-5398
97D1-6797
97D2-6DCB
97D3-71D0
97D4-7433
97D5-81E8
97D6-8F2A
97D7-96A3
97D8-9C57
97D9-9E9F
97DA-7460
97DB-5841
97DC-6D99
97DD-7D2F
97DE-985E
97DF-4EE4
97E0-4F36
97E1-4F8B
97E2-51B7
97E3-52B1
97E4-5DBA
97E5-601C
97E6-73B2
97E7-793C
97E8-82D3
97E9-9234
97EA-96B7
97EB-96F6
97EC-970A
97ED-9E97
97EE-9F62
97EF-66A6
97F0-6B74
97F1-5217
97F2-52A3
97F3-70C8
97F4-88C2
97F5-5EC9
97F6-604B
97F7-6190
97F8-6F23
97F9-7149
97FA-7C3E
97FB-7DF4
97FC-806F
9840-84EE
9841-9023
9842-932C
9843-5442
9844-9B6F
9845-6AD3
9846-7089
9847-8CC2
9848-8DEF
9849-9732
984A-52B4
984B-5A41
984C-5ECA
984D-5F04
984E-6717
984F-697C
9850-6994
9851-6D6A
9852-6F0F
9853-7262
9854-72FC
9855-7BED
9856-8001
9857-807E
9858-874B
9859-90CE
985A-516D
985B-9E93
985C-7984
985D-808B
985E-9332
985F-8AD6
9860-502D
9861-548C
9862-8A71
9863-6B6A
9864-8CC4
9865-8107
9866-60D1
9867-67A0
9868-9DF2
9869-4E99
986A-4E98
986B-9C10
986C-8A6B
986D-85C1
986E-8568
986F-6900
9870-6E7E
9871-7897
9872-8155
989F-5F0C
98A0-4E10
98A1-4E15
98A2-4E2A
98A3-4E31
98A4-4E36
98A5-4E3C
98A6-4E3F
98A7-4E42
98A8-4E56
98A9-4E58
98AA-4E82
98AB-4E85
98AC-8C6B
98AD-4E8A
98AE-8212
98AF-5F0D
98B0-4E8E
98B1-4E9E
98B2-4E9F
98B3-4EA0
98B4-4EA2
98B5-4EB0
98B6-4EB3
98B7-4EB6
98B8-4ECE
98B9-4ECD
98BA-4EC4
98BB-4EC6
98BC-4EC2
98BD-4ED7
98BE-4EDE
98BF-4EED
98C0-4EDF
98C1-4EF7
98C2-4F09
98C3-4F5A
98C4-4F30
98C5-4F5B
98C6-4F5D
98C7-4F57
98C8-4F47
98C9-4F76
98CA-4F88
98CB-4F8F
98CC-4F98
98CD-4F7B
98CE-4F69
98CF-4F70
98D0-4F91
98D1-4F6F
98D2-4F86
98D3-4F96
98D4-5118
98D5-4FD4
98D6-4FDF
98D7-4FCE
98D8-4FD8
98D9-4FDB
98DA-4FD1
98DB-4FDA
98DC-4FD0
98DD-4FE4
98DE-4FE5
98DF-501A
98E0-5028
98E1-5014
98E2-502A
98E3-5025
98E4-5005
98E5-4F1C
98E6-4FF6
98E7-5021
98E8-5029
98E9-502C
98EA-4FFE
98EB-4FEF
98EC-5011
98ED-5006
98EE-5043
98EF-5047
98F0-6703
98F1-5055
98F2-5050
98F3-5048
98F4-505A
98F5-5056
98F6-506C
98F7-5078
98F8-5080
98F9-509A
98FA-5085
98FB-50B4
98FC-50B2
9940-50C9
9941-50CA
9942-50B3
9943-50C2
9944-50D6
9945-50DE
9946-50E5
9947-50ED
9948-50E3
9949-50EE
994A-50F9
994B-50F5
994C-5109
994D-5101
994E-5102
994F-5116
9950-5115
9951-5114
9952-511A
9953-5121
9954-513A
9955-5137
9956-513C
9957-513B
9958-513F
9959-5140
995A-5152
995B-514C
995C-5154
995D-5162
995E-7AF8
995F-5169
9960-516A
9961-516E
9962-5180
9963-5182
9964-56D8
9965-518C
9966-5189
9967-518F
9968-5191
9969-5193
996A-5195
996B-5196
996C-51A4
996D-51A6
996E-51A2
996F-51A9
9970-51AA
9971-51AB
9972-51B3
9973-51B1
9974-51B2
9975-51B0
9976-51B5
9977-51BD
9978-51C5
9979-51C9
997A-51DB
997B-51E0
997C-8655
997D-51E9
997E-51ED
9980-51F0
9981-51F5
9982-51FE
9983-5204
9984-520B
9985-5214
9986-520E
9987-5227
9988-522A
9989-522E
998A-5233
998B-5239
998C-524F
998D-5244
998E-524B
998F-524C
9990-525E
9991-5254
9992-526A
9993-5274
9994-5269
9995-5273
9996-527F
9997-527D
9998-528D
9999-5294
999A-5292
999B-5271
999C-5288
999D-5291
999E-8FA8
999F-8FA7
99A0-52AC
99A1-52AD
99A2-52BC
99A3-52B5
99A4-52C1
99A5-52CD
99A6-52D7
99A7-52DE
99A8-52E3
99A9-52E6
99AA-98ED
99AB-52E0
99AC-52F3
99AD-52F5
99AE-52F8
99AF-52F9
99B0-5306
99B1-5308
99B2-7538
99B3-530D
99B4-5310
99B5-530F
99B6-5315
99B7-531A
99B8-5323
99B9-532F
99BA-5331
99BB-5333
99BC-5338
99BD-5340
99BE-5346
99BF-5345
99C0-4E17
99C1-5349
99C2-534D
99C3-51D6
99C4-535E
99C5-5369
99C6-536E
99C7-5918
99C8-537B
99C9-5377
99CA-5382
99CB-5396
99CC-53A0
99CD-53A6
99CE-53A5
99CF-53AE
99D0-53B0
99D1-53B6
99D2-53C3
99D3-7C12
99D4-96D9
99D5-53DF
99D6-66FC
99D7-71EE
99D8-53EE
99D9-53E8
99DA-53ED
99DB-53FA
99DC-5401
99DD-543D
99DE-5440
99DF-542C
99E0-542D
99E1-543C
99E2-542E
99E3-5436
99E4-5429
99E5-541D
99E6-544E
99E7-548F
99E8-5475
99E9-548E
99EA-545F
99EB-5471
99EC-5477
99ED-5470
99EE-5492
99EF-547B
99F0-5480
99F1-5476
99F2-5484
99F3-5490
99F4-5486
99F5-54C7
99F6-54A2
99F7-54B8
99F8-54A5
99F9-54AC
99FA-54C4
99FB-54C8
99FC-54A8
9A40-54AB
9A41-54C2
9A42-54A4
9A43-54BE
9A44-54BC
9A45-54D8
9A46-54E5
9A47-54E6
9A48-550F
9A49-5514
9A4A-54FD
9A4B-54EE
9A4C-54ED
9A4D-54FA
9A4E-54E2
9A4F-5539
9A50-5540
9A51-5563
9A52-554C
9A53-552E
9A54-555C
9A55-5545
9A56-5556
9A57-5557
9A58-5538
9A59-5533
9A5A-555D
9A5B-5599
9A5C-5580
9A5D-54AF
9A5E-558A
9A5F-559F
9A60-557B
9A61-557E
9A62-5598
9A63-559E
9A64-55AE
9A65-557C
9A66-5583
9A67-55A9
9A68-5587
9A69-55A8
9A6A-55DA
9A6B-55C5
9A6C-55DF
9A6D-55C4
9A6E-55DC
9A6F-55E4
9A70-55D4
9A71-5614
9A72-55F7
9A73-5616
9A74-55FE
9A75-55FD
9A76-561B
9A77-55F9
9A78-564E
9A79-5650
9A7A-71DF
9A7B-5634
9A7C-5636
9A7D-5632
9A7E-5638
9A80-566B
9A81-5664
9A82-562F
9A83-566C
9A84-566A
9A85-5686
9A86-5680
9A87-568A
9A88-56A0
9A89-5694
9A8A-568F
9A8B-56A5
9A8C-56AE
9A8D-56B6
9A8E-56B4
9A8F-56C2
9A90-56BC
9A91-56C1
9A92-56C3
9A93-56C0
9A94-56C8
9A95-56CE
9A96-56D1
9A97-56D3
9A98-56D7
9A99-56EE
9A9A-56F9
9A9B-5700
9A9C-56FF
9A9D-5704
9A9E-5709
9A9F-5708
9AA0-570B
9AA1-570D
9AA2-5713
9AA3-5718
9AA4-5716
9AA5-55C7
9AA6-571C
9AA7-5726
9AA8-5737
9AA9-5738
9AAA-574E
9AAB-573B
9AAC-5740
9AAD-574F
9AAE-5769
9AAF-57C0
9AB0-5788
9AB1-5761
9AB2-577F
9AB3-5789
9AB4-5793
9AB5-57A0
9AB6-57B3
9AB7-57A4
9AB8-57AA
9AB9-57B0
9ABA-57C3
9ABB-57C6
9ABC-57D4
9ABD-57D2
9ABE-57D3
9ABF-580A
9AC0-57D6
9AC1-57E3
9AC2-580B
9AC3-5819
9AC4-581D
9AC5-5872
9AC6-5821
9AC7-5862
9AC8-584B
9AC9-5870
9ACA-6BC0
9ACB-5852
9ACC-583D
9ACD-5879
9ACE-5885
9ACF-58B9
9AD0-589F
9AD1-58AB
9AD2-58BA
9AD3-58DE
9AD4-58BB
9AD5-58B8
9AD6-58AE
9AD7-58C5
9AD8-58D3
9AD9-58D1
9ADA-58D7
9ADB-58D9
9ADC-58D8
9ADD-58E5
9ADE-58DC
9ADF-58E4
9AE0-58DF
9AE1-58EF
9AE2-58FA
9AE3-58F9
9AE4-58FB
9AE5-58FC
9AE6-58FD
9AE7-5902
9AE8-590A
9AE9-5910
9AEA-591B
9AEB-68A6
9AEC-5925
9AED-592C
9AEE-592D
9AEF-5932
9AF0-5938
9AF1-593E
9AF2-7AD2
9AF3-5955
9AF4-5950
9AF5-594E
9AF6-595A
9AF7-5958
9AF8-5962
9AF9-5960
9AFA-5967
9AFB-596C
9AFC-5969
9B40-5978
9B41-5981
9B42-599D
9B43-4F5E
9B44-4FAB
9B45-59A3
9B46-59B2
9B47-59C6
9B48-59E8
9B49-59DC
9B4A-598D
9B4B-59D9
9B4C-59DA
9B4D-5A25
9B4E-5A1F
9B4F-5A11
9B50-5A1C
9B51-5A09
9B52-5A1A
9B53-5A40
9B54-5A6C
9B55-5A49
9B56-5A35
9B57-5A36
9B58-5A62
9B59-5A6A
9B5A-5A9A
9B5B-5ABC
9B5C-5ABE
9B5D-5ACB
9B5E-5AC2
9B5F-5ABD
9B60-5AE3
9B61-5AD7
9B62-5AE6
9B63-5AE9
9B64-5AD6
9B65-5AFA
9B66-5AFB
9B67-5B0C
9B68-5B0B
9B69-5B16
9B6A-5B32
9B6B-5AD0
9B6C-5B2A
9B6D-5B36
9B6E-5B3E
9B6F-5B43
9B70-5B45
9B71-5B40
9B72-5B51
9B73-5B55
9B74-5B5A
9B75-5B5B
9B76-5B65
9B77-5B69
9B78-5B70
9B79-5B73
9B7A-5B75
9B7B-5B78
9B7C-6588
9B7D-5B7A
9B7E-5B80
9B80-5B83
9B81-5BA6
9B82-5BB8
9B83-5BC3
9B84-5BC7
9B85-5BC9
9B86-5BD4
9B87-5BD0
9B88-5BE4
9B89-5BE6
9B8A-5BE2
9B8B-5BDE
9B8C-5BE5
9B8D-5BEB
9B8E-5BF0
9B8F-5BF6
9B90-5BF3
9B91-5C05
9B92-5C07
9B93-5C08
9B94-5C0D
9B95-5C13
9B96-5C20
9B97-5C22
9B98-5C28
9B99-5C38
9B9A-5C39
9B9B-5C41
9B9C-5C46
9B9D-5C4E
9B9E-5C53
9B9F-5C50
9BA0-5C4F
9BA1-5B71
9BA2-5C6C
9BA3-5C6E
9BA4-4E62
9BA5-5C76
9BA6-5C79
9BA7-5C8C
9BA8-5C91
9BA9-5C94
9BAA-599B
9BAB-5CAB
9BAC-5CBB
9BAD-5CB6
9BAE-5CBC
9BAF-5CB7
9BB0-5CC5
9BB1-5CBE
9BB2-5CC7
9BB3-5CD9
9BB4-5CE9
9BB5-5CFD
9BB6-5CFA
9BB7-5CED
9BB8-5D8C
9BB9-5CEA
9BBA-5D0B
9BBB-5D15
9BBC-5D17
9BBD-5D5C
9BBE-5D1F
9BBF-5D1B
9BC0-5D11
9BC1-5D14
9BC2-5D22
9BC3-5D1A
9BC4-5D19
9BC5-5D18
9BC6-5D4C
9BC7-5D52
9BC8-5D4E
9BC9-5D4B
9BCA-5D6C
9BCB-5D73
9BCC-5D76
9BCD-5D87
9BCE-5D84
9BCF-5D82
9BD0-5DA2
9BD1-5D9D
9BD2-5DAC
9BD3-5DAE
9BD4-5DBD
9BD5-5D90
9BD6-5DB7
9BD7-5DBC
9BD8-5DC9
9BD9-5DCD
9BDA-5DD3
9BDB-5DD2
9BDC-5DD6
9BDD-5DDB
9BDE-5DEB
9BDF-5DF2
9BE0-5DF5
9BE1-5E0B
9BE2-5E1A
9BE3-5E19
9BE4-5E11
9BE5-5E1B
9BE6-5E36
9BE7-5E37
9BE8-5E44
9BE9-5E43
9BEA-5E40
9BEB-5E4E
9BEC-5E57
9BED-5E54
9BEE-5E5F
9BEF-5E62
9BF0-5E64
9BF1-5E47
9BF2-5E75
9BF3-5E76
9BF4-5E7A
9BF5-9EBC
9BF6-5E7F
9BF7-5EA0
9BF8-5EC1
9BF9-5EC2
9BFA-5EC8
9BFB-5ED0
9BFC-5ECF
9C40-5ED6
9C41-5EE3
9C42-5EDD
9C43-5EDA
9C44-5EDB
9C45-5EE2
9C46-5EE1
9C47-5EE8
9C48-5EE9
9C49-5EEC
9C4A-5EF1
9C4B-5EF3
9C4C-5EF0
9C4D-5EF4
9C4E-5EF8
9C4F-5EFE
9C50-5F03
9C51-5F09
9C52-5F5D
9C53-5F5C
9C54-5F0B
9C55-5F11
9C56-5F16
9C57-5F29
9C58-5F2D
9C59-5F38
9C5A-5F41
9C5B-5F48
9C5C-5F4C
9C5D-5F4E
9C5E-5F2F
9C5F-5F51
9C60-5F56
9C61-5F57
9C62-5F59
9C63-5F61
9C64-5F6D
9C65-5F73
9C66-5F77
9C67-5F83
9C68-5F82
9C69-5F7F
9C6A-5F8A
9C6B-5F88
9C6C-5F91
9C6D-5F87
9C6E-5F9E
9C6F-5F99
9C70-5F98
9C71-5FA0
9C72-5FA8
9C73-5FAD
9C74-5FBC
9C75-5FD6
9C76-5FFB
9C77-5FE4
9C78-5FF8
9C79-5FF1
9C7A-5FDD
9C7B-60B3
9C7C-5FFF
9C7D-6021
9C7E-6060
9C80-6019
9C81-6010
9C82-6029
9C83-600E
9C84-6031
9C85-601B
9C86-6015
9C87-602B
9C88-6026
9C89-600F
9C8A-603A
9C8B-605A
9C8C-6041
9C8D-606A
9C8E-6077
9C8F-605F
9C90-604A
9C91-6046
9C92-604D
9C93-6063
9C94-6043
9C95-6064
9C96-6042
9C97-606C
9C98-606B
9C99-6059
9C9A-6081
9C9B-608D
9C9C-60E7
9C9D-6083
9C9E-609A
9C9F-6084
9CA0-609B
9CA1-6096
9CA2-6097
9CA3-6092
9CA4-60A7
9CA5-608B
9CA6-60E1
9CA7-60B8
9CA8-60E0
9CA9-60D3
9CAA-60B4
9CAB-5FF0
9CAC-60BD
9CAD-60C6
9CAE-60B5
9CAF-60D8
9CB0-614D
9CB1-6115
9CB2-6106
9CB3-60F6
9CB4-60F7
9CB5-6100
9CB6-60F4
9CB7-60FA
9CB8-6103
9CB9-6121
9CBA-60FB
9CBB-60F1
9CBC-610D
9CBD-610E
9CBE-6147
9CBF-613E
9CC0-6128
9CC1-6127
9CC2-614A
9CC3-613F
9CC4-613C
9CC5-612C
9CC6-6134
9CC7-613D
9CC8-6142
9CC9-6144
9CCA-6173
9CCB-6177
9CCC-6158
9CCD-6159
9CCE-615A
9CCF-616B
9CD0-6174
9CD1-616F
9CD2-6165
9CD3-6171
9CD4-615F
9CD5-615D
9CD6-6153
9CD7-6175
9CD8-6199
9CD9-6196
9CDA-6187
9CDB-61AC
9CDC-6194
9CDD-619A
9CDE-618A
9CDF-6191
9CE0-61AB
9CE1-61AE
9CE2-61CC
9CE3-61CA
9CE4-61C9
9CE5-61F7
9CE6-61C8
9CE7-61C3
9CE8-61C6
9CE9-61BA
9CEA-61CB
9CEB-7F79
9CEC-61CD
9CED-61E6
9CEE-61E3
9CEF-61F6
9CF0-61FA
9CF1-61F4
9CF2-61FF
9CF3-61FD
9CF4-61FC
9CF5-61FE
9CF6-6200
9CF7-6208
9CF8-6209
9CF9-620D
9CFA-620C
9CFB-6214
9CFC-621B
9D40-621E
9D41-6221
9D42-622A
9D43-622E
9D44-6230
9D45-6232
9D46-6233
9D47-6241
9D48-624E
9D49-625E
9D4A-6263
9D4B-625B
9D4C-6260
9D4D-6268
9D4E-627C
9D4F-6282
9D50-6289
9D51-627E
9D52-6292
9D53-6293
9D54-6296
9D55-62D4
9D56-6283
9D57-6294
9D58-62D7
9D59-62D1
9D5A-62BB
9D5B-62CF
9D5C-62FF
9D5D-62C6
9D5E-64D4
9D5F-62C8
9D60-62DC
9D61-62CC
9D62-62CA
9D63-62C2
9D64-62C7
9D65-629B
9D66-62C9
9D67-630C
9D68-62EE
9D69-62F1
9D6A-6327
9D6B-6302
9D6C-6308
9D6D-62EF
9D6E-62F5
9D6F-6350
9D70-633E
9D71-634D
9D72-641C
9D73-634F
9D74-6396
9D75-638E
9D76-6380
9D77-63AB
9D78-6376
9D79-63A3
9D7A-638F
9D7B-6389
9D7C-639F
9D7D-63B5
9D7E-636B
9D80-6369
9D81-63BE
9D82-63E9
9D83-63C0
9D84-63C6
9D85-63E3
9D86-63C9
9D87-63D2
9D88-63F6
9D89-63C4
9D8A-6416
9D8B-6434
9D8C-6406
9D8D-6413
9D8E-6426
9D8F-6436
9D90-651D
9D91-6417
9D92-6428
9D93-640F
9D94-6467
9D95-646F
9D96-6476
9D97-644E
9D98-652A
9D99-6495
9D9A-6493
9D9B-64A5
9D9C-64A9
9D9D-6488
9D9E-64BC
9D9F-64DA
9DA0-64D2
9DA1-64C5
9DA2-64C7
9DA3-64BB
9DA4-64D8
9DA5-64C2
9DA6-64F1
9DA7-64E7
9DA8-8209
9DA9-64E0
9DAA-64E1
9DAB-62AC
9DAC-64E3
9DAD-64EF
9DAE-652C
9DAF-64F6
9DB0-64F4
9DB1-64F2
9DB2-64FA
9DB3-6500
9DB4-64FD
9DB5-6518
9DB6-651C
9DB7-6505
9DB8-6524
9DB9-6523
9DBA-652B
9DBB-6534
9DBC-6535
9DBD-6537
9DBE-6536
9DBF-6538
9DC0-754B
9DC1-6548
9DC2-6556
9DC3-6555
9DC4-654D
9DC5-6558
9DC6-655E
9DC7-655D
9DC8-6572
9DC9-6578
9DCA-6582
9DCB-6583
9DCC-8B8A
9DCD-659B
9DCE-659F
9DCF-65AB
9DD0-65B7
9DD1-65C3
9DD2-65C6
9DD3-65C1
9DD4-65C4
9DD5-65CC
9DD6-65D2
9DD7-65DB
9DD8-65D9
9DD9-65E0
9DDA-65E1
9DDB-65F1
9DDC-6772
9DDD-660A
9DDE-6603
9DDF-65FB
9DE0-6773
9DE1-6635
9DE2-6636
9DE3-6634
9DE4-661C
9DE5-664F
9DE6-6644
9DE7-6649
9DE8-6641
9DE9-665E
9DEA-665D
9DEB-6664
9DEC-6667
9DED-6668
9DEE-665F
9DEF-6662
9DF0-6670
9DF1-6683
9DF2-6688
9DF3-668E
9DF4-6689
9DF5-6684
9DF6-6698
9DF7-669D
9DF8-66C1
9DF9-66B9
9DFA-66C9
9DFB-66BE
9DFC-66BC
9E40-66C4
9E41-66B8
9E42-66D6
9E43-66DA
9E44-66E0
9E45-663F
9E46-66E6
9E47-66E9
9E48-66F0
9E49-66F5
9E4A-66F7
9E4B-670F
9E4C-6716
9E4D-671E
9E4E-6726
9E4F-6727
9E50-9738
9E51-672E
9E52-673F
9E53-6736
9E54-6741
9E55-6738
9E56-6737
9E57-6746
9E58-675E
9E59-6760
9E5A-6759
9E5B-6763
9E5C-6764
9E5D-6789
9E5E-6770
9E5F-67A9
9E60-677C
9E61-676A
9E62-678C
9E63-678B
9E64-67A6
9E65-67A1
9E66-6785
9E67-67B7
9E68-67EF
9E69-67B4
9E6A-67EC
9E6B-67B3
9E6C-67E9
9E6D-67B8
9E6E-67E4
9E6F-67DE
9E70-67DD
9E71-67E2
9E72-67EE
9E73-67B9
9E74-67CE
9E75-67C6
9E76-67E7
9E77-6A9C
9E78-681E
9E79-6846
9E7A-6829
9E7B-6840
9E7C-684D
9E7D-6832
9E7E-684E
9E80-68B3
9E81-682B
9E82-6859
9E83-6863
9E84-6877
9E85-687F
9E86-689F
9E87-688F
9E88-68AD
9E89-6894
9E8A-689D
9E8B-689B
9E8C-6883
9E8D-6AAE
9E8E-68B9
9E8F-6874
9E90-68B5
9E91-68A0
9E92-68BA
9E93-690F
9E94-688D
9E95-687E
9E96-6901
9E97-68CA
9E98-6908
9E99-68D8
9E9A-6922
9E9B-6926
9E9C-68E1
9E9D-690C
9E9E-68CD
9E9F-68D4
9EA0-68E7
9EA1-68D5
9EA2-6936
9EA3-6912
9EA4-6904
9EA5-68D7
9EA6-68E3
9EA7-6925
9EA8-68F9
9EA9-68E0
9EAA-68EF
9EAB-6928
9EAC-692A
9EAD-691A
9EAE-6923
9EAF-6921
9EB0-68C6
9EB1-6979
9EB2-6977
9EB3-695C
9EB4-6978
9EB5-696B
9EB6-6954
9EB7-697E
9EB8-696E
9EB9-6939
9EBA-6974
9EBB-693D
9EBC-6959
9EBD-6930
9EBE-6961
9EBF-695E
9EC0-695D
9EC1-6981
9EC2-696A
9EC3-69B2
9EC4-69AE
9EC5-69D0
9EC6-69BF
9EC7-69C1
9EC8-69D3
9EC9-69BE
9ECA-69CE
9ECB-5BE8
9ECC-69CA
9ECD-69DD
9ECE-69BB
9ECF-69C3
9ED0-69A7
9ED1-6A2E
9ED2-6991
9ED3-69A0
9ED4-699C
9ED5-6995
9ED6-69B4
9ED7-69DE
9ED8-69E8
9ED9-6A02
9EDA-6A1B
9EDB-69FF
9EDC-6B0A
9EDD-69F9
9EDE-69F2
9EDF-69E7
9EE0-6A05
9EE1-69B1
9EE2-6A1E
9EE3-69ED
9EE4-6A14
9EE5-69EB
9EE6-6A0A
9EE7-6A12
9EE8-6AC1
9EE9-6A23
9EEA-6A13
9EEB-6A44
9EEC-6A0C
9EED-6A72
9EEE-6A36
9EEF-6A78
9EF0-6A47
9EF1-6A62
9EF2-6A59
9EF3-6A66
9EF4-6A48
9EF5-6A38
9EF6-6A22
9EF7-6A90
9EF8-6A8D
9EF9-6AA0
9EFA-6A84
9EFB-6AA2
9EFC-6AA3
9F40-6A97
9F41-8617
9F42-6ABB
9F43-6AC3
9F44-6AC2
9F45-6AB8
9F46-6AB3
9F47-6AAC
9F48-6ADE
9F49-6AD1
9F4A-6ADF
9F4B-6AAA
9F4C-6ADA
9F4D-6AEA
9F4E-6AFB
9F4F-6B05
9F50-8616
9F51-6AFA
9F52-6B12
9F53-6B16
9F54-9B31
9F55-6B1F
9F56-6B38
9F57-6B37
9F58-76DC
9F59-6B39
9F5A-98EE
9F5B-6B47
9F5C-6B43
9F5D-6B49
9F5E-6B50
9F5F-6B59
9F60-6B54
9F61-6B5B
9F62-6B5F
9F63-6B61
9F64-6B78
9F65-6B79
9F66-6B7F
9F67-6B80
9F68-6B84
9F69-6B83
9F6A-6B8D
9F6B-6B98
9F6C-6B95
9F6D-6B9E
9F6E-6BA4
9F6F-6BAA
9F70-6BAB
9F71-6BAF
9F72-6BB2
9F73-6BB1
9F74-6BB3
9F75-6BB7
9F76-6BBC
9F77-6BC6
9F78-6BCB
9F79-6BD3
9F7A-6BDF
9F7B-6BEC
9F7C-6BEB
9F7D-6BF3
9F7E-6BEF
9F80-9EBE
9F81-6C08
9F82-6C13
9F83-6C14
9F84-6C1B
9F85-6C24
9F86-6C23
9F87-6C5E
9F88-6C55
9F89-6C62
9F8A-6C6A
9F8B-6C82
9F8C-6C8D
9F8D-6C9A
9F8E-6C81
9F8F-6C9B
9F90-6C7E
9F91-6C68
9F92-6C73
9F93-6C92
9F94-6C90
9F95-6CC4
9F96-6CF1
9F97-6CD3
9F98-6CBD
9F99-6CD7
9F9A-6CC5
9F9B-6CDD
9F9C-6CAE
9F9D-6CB1
9F9E-6CBE
9F9F-6CBA
9FA0-6CDB
9FA1-6CEF
9FA2-6CD9
9FA3-6CEA
9FA4-6D1F
9FA5-884D
9FA6-6D36
9FA7-6D2B
9FA8-6D3D
9FA9-6D38
9FAA-6D19
9FAB-6D35
9FAC-6D33
9FAD-6D12
9FAE-6D0C
9FAF-6D63
9FB0-6D93
9FB1-6D64
9FB2-6D5A
9FB3-6D79
9FB4-6D59
9FB5-6D8E
9FB6-6D95
9FB7-6FE4
9FB8-6D85
9FB9-6DF9
9FBA-6E15
9FBB-6E0A
9FBC-6DB5
9FBD-6DC7
9FBE-6DE6
9FBF-6DB8
9FC0-6DC6
9FC1-6DEC
9FC2-6DDE
9FC3-6DCC
9FC4-6DE8
9FC5-6DD2
9FC6-6DC5
9FC7-6DFA
9FC8-6DD9
9FC9-6DE4
9FCA-6DD5
9FCB-6DEA
9FCC-6DEE
9FCD-6E2D
9FCE-6E6E
9FCF-6E2E
9FD0-6E19
9FD1-6E72
9FD2-6E5F
9FD3-6E3E
9FD4-6E23
9FD5-6E6B
9FD6-6E2B
9FD7-6E76
9FD8-6E4D
9FD9-6E1F
9FDA-6E43
9FDB-6E3A
9FDC-6E4E
9FDD-6E24
9FDE-6EFF
9FDF-6E1D
9FE0-6E38
9FE1-6E82
9FE2-6EAA
9FE3-6E98
9FE4-6EC9
9FE5-6EB7
9FE6-6ED3
9FE7-6EBD
9FE8-6EAF
9FE9-6EC4
9FEA-6EB2
9FEB-6ED4
9FEC-6ED5
9FED-6E8F
9FEE-6EA5
9FEF-6EC2
9FF0-6E9F
9FF1-6F41
9FF2-6F11
9FF3-704C
9FF4-6EEC
9FF5-6EF8
9FF6-6EFE
9FF7-6F3F
9FF8-6EF2
9FF9-6F31
9FFA-6EEF
9FFB-6F32
9FFC-6ECC
E040-6F3E
E041-6F13
E042-6EF7
E043-6F86
E044-6F7A
E045-6F78
E046-6F81
E047-6F80
E048-6F6F
E049-6F5B
E04A-6FF3
E04B-6F6D
E04C-6F82
E04D-6F7C
E04E-6F58
E04F-6F8E
E050-6F91
E051-6FC2
E052-6F66
E053-6FB3
E054-6FA3
E055-6FA1
E056-6FA4
E057-6FB9
E058-6FC6
E059-6FAA
E05A-6FDF
E05B-6FD5
E05C-6FEC
E05D-6FD4
E05E-6FD8
E05F-6FF1
E060-6FEE
E061-6FDB
E062-7009
E063-700B
E064-6FFA
E065-7011
E066-7001
E067-700F
E068-6FFE
E069-701B
E06A-701A
E06B-6F74
E06C-701D
E06D-7018
E06E-701F
E06F-7030
E070-703E
E071-7032
E072-7051
E073-7063
E074-7099
E075-7092
E076-70AF
E077-70F1
E078-70AC
E079-70B8
E07A-70B3
E07B-70AE
E07C-70DF
E07D-70CB
E07E-70DD
E080-70D9
E081-7109
E082-70FD
E083-711C
E084-7119
E085-7165
E086-7155
E087-7188
E088-7166
E089-7162
E08A-714C
E08B-7156
E08C-716C
E08D-718F
E08E-71FB
E08F-7184
E090-7195
E091-71A8
E092-71AC
E093-71D7
E094-71B9
E095-71BE
E096-71D2
E097-71C9
E098-71D4
E099-71CE
E09A-71E0
E09B-71EC
E09C-71E7
E09D-71F5
E09E-71FC
E09F-71F9
E0A0-71FF
E0A1-720D
E0A2-7210
E0A3-721B
E0A4-7228
E0A5-722D
E0A6-722C
E0A7-7230
E0A8-7232
E0A9-723B
E0AA-723C
E0AB-723F
E0AC-7240
E0AD-7246
E0AE-724B
E0AF-7258
E0B0-7274
E0B1-727E
E0B2-7282
E0B3-7281
E0B4-7287
E0B5-7292
E0B6-7296
E0B7-72A2
E0B8-72A7
E0B9-72B9
E0BA-72B2
E0BB-72C3
E0BC-72C6
E0BD-72C4
E0BE-72CE
E0BF-72D2
E0C0-72E2
E0C1-72E0
E0C2-72E1
E0C3-72F9
E0C4-72F7
E0C5-500F
E0C6-7317
E0C7-730A
E0C8-731C
E0C9-7316
E0CA-731D
E0CB-7334
E0CC-732F
E0CD-7329
E0CE-7325
E0CF-733E
E0D0-734E
E0D1-734F
E0D2-9ED8
E0D3-7357
E0D4-736A
E0D5-7368
E0D6-7370
E0D7-7378
E0D8-7375
E0D9-737B
E0DA-737A
E0DB-73C8
E0DC-73B3
E0DD-73CE
E0DE-73BB
E0DF-73C0
E0E0-73E5
E0E1-73EE
E0E2-73DE
E0E3-74A2
E0E4-7405
E0E5-746F
E0E6-7425
E0E7-73F8
E0E8-7432
E0E9-743A
E0EA-7455
E0EB-743F
E0EC-745F
E0ED-7459
E0EE-7441
E0EF-745C
E0F0-7469
E0F1-7470
E0F2-7463
E0F3-746A
E0F4-7476
E0F5-747E
E0F6-748B
E0F7-749E
E0F8-74A7
E0F9-74CA
E0FA-74CF
E0FB-74D4
E0FC-73F1
E140-74E0
E141-74E3
E142-74E7
E143-74E9
E144-74EE
E145-74F2
E146-74F0
E147-74F1
E148-74F8
E149-74F7
E14A-7504
E14B-7503
E14C-7505
E14D-750C
E14E-750E
E14F-750D
E150-7515
E151-7513
E152-751E
E153-7526
E154-752C
E155-753C
E156-7544
E157-754D
E158-754A
E159-7549
E15A-755B
E15B-7546
E15C-755A
E15D-7569
E15E-7564
E15F-7567
E160-756B
E161-756D
E162-7578
E163-7576
E164-7586
E165-7587
E166-7574
E167-758A
E168-7589
E169-7582
E16A-7594
E16B-759A
E16C-759D
E16D-75A5
E16E-75A3
E16F-75C2
E170-75B3
E171-75C3
E172-75B5
E173-75BD
E174-75B8
E175-75BC
E176-75B1
E177-75CD
E178-75CA
E179-75D2
E17A-75D9
E17B-75E3
E17C-75DE
E17D-75FE
E17E-75FF
E180-75FC
E181-7601
E182-75F0
E183-75FA
E184-75F2
E185-75F3
E186-760B
E187-760D
E188-7609
E189-761F
E18A-7627
E18B-7620
E18C-7621
E18D-7622
E18E-7624
E18F-7634
E190-7630
E191-763B
E192-7647
E193-7648
E194-7646
E195-765C
E196-7658
E197-7661
E198-7662
E199-7668
E19A-7669
E19B-766A
E19C-7667
E19D-766C
E19E-7670
E19F-7672
E1A0-7676
E1A1-7678
E1A2-767C
E1A3-7680
E1A4-7683
E1A5-7688
E1A6-768B
E1A7-768E
E1A8-7696
E1A9-7693
E1AA-7699
E1AB-769A
E1AC-76B0
E1AD-76B4
E1AE-76B8
E1AF-76B9
E1B0-76BA
E1B1-76C2
E1B2-76CD
E1B3-76D6
E1B4-76D2
E1B5-76DE
E1B6-76E1
E1B7-76E5
E1B8-76E7
E1B9-76EA
E1BA-862F
E1BB-76FB
E1BC-7708
E1BD-7707
E1BE-7704
E1BF-7729
E1C0-7724
E1C1-771E
E1C2-7725
E1C3-7726
E1C4-771B
E1C5-7737
E1C6-7738
E1C7-7747
E1C8-775A
E1C9-7768
E1CA-776B
E1CB-775B
E1CC-7765
E1CD-777F
E1CE-777E
E1CF-7779
E1D0-778E
E1D1-778B
E1D2-7791
E1D3-77A0
E1D4-779E
E1D5-77B0
E1D6-77B6
E1D7-77B9
E1D8-77BF
E1D9-77BC
E1DA-77BD
E1DB-77BB
E1DC-77C7
E1DD-77CD
E1DE-77D7
E1DF-77DA
E1E0-77DC
E1E1-77E3
E1E2-77EE
E1E3-77FC
E1E4-780C
E1E5-7812
E1E6-7926
E1E7-7820
E1E8-792A
E1E9-7845
E1EA-788E
E1EB-7874
E1EC-7886
E1ED-787C
E1EE-789A
E1EF-788C
E1F0-78A3
E1F1-78B5
E1F2-78AA
E1F3-78AF
E1F4-78D1
E1F5-78C6
E1F6-78CB
E1F7-78D4
E1F8-78BE
E1F9-78BC
E1FA-78C5
E1FB-78CA
E1FC-78EC
E240-78E7
E241-78DA
E242-78FD
E243-78F4
E244-7907
E245-7912
E246-7911
E247-7919
E248-792C
E249-792B
E24A-7940
E24B-7960
E24C-7957
E24D-795F
E24E-795A
E24F-7955
E250-7953
E251-797A
E252-797F
E253-798A
E254-799D
E255-79A7
E256-9F4B
E257-79AA
E258-79AE
E259-79B3
E25A-79B9
E25B-79BA
E25C-79C9
E25D-79D5
E25E-79E7
E25F-79EC
E260-79E1
E261-79E3
E262-7A08
E263-7A0D
E264-7A18
E265-7A19
E266-7A20
E267-7A1F
E268-7980
E269-7A31
E26A-7A3B
E26B-7A3E
E26C-7A37
E26D-7A43
E26E-7A57
E26F-7A49
E270-7A61
E271-7A62
E272-7A69
E273-9F9D
E274-7A70
E275-7A79
E276-7A7D
E277-7A88
E278-7A97
E279-7A95
E27A-7A98
E27B-7A96
E27C-7AA9
E27D-7AC8
E27E-7AB0
E280-7AB6
E281-7AC5
E282-7AC4
E283-7ABF
E284-9083
E285-7AC7
E286-7ACA
E287-7ACD
E288-7ACF
E289-7AD5
E28A-7AD3
E28B-7AD9
E28C-7ADA
E28D-7ADD
E28E-7AE1
E28F-7AE2
E290-7AE6
E291-7AED
E292-7AF0
E293-7B02
E294-7B0F
E295-7B0A
E296-7B06
E297-7B33
E298-7B18
E299-7B19
E29A-7B1E
E29B-7B35
E29C-7B28
E29D-7B36
E29E-7B50
E29F-7B7A
E2A0-7B04
E2A1-7B4D
E2A2-7B0B
E2A3-7B4C
E2A4-7B45
E2A5-7B75
E2A6-7B65
E2A7-7B74
E2A8-7B67
E2A9-7B70
E2AA-7B71
E2AB-7B6C
E2AC-7B6E
E2AD-7B9D
E2AE-7B98
E2AF-7B9F
E2B0-7B8D
E2B1-7B9C
E2B2-7B9A
E2B3-7B8B
E2B4-7B92
E2B5-7B8F
E2B6-7B5D
E2B7-7B99
E2B8-7BCB
E2B9-7BC1
E2BA-7BCC
E2BB-7BCF
E2BC-7BB4
E2BD-7BC6
E2BE-7BDD
E2BF-7BE9
E2C0-7C11
E2C1-7C14
E2C2-7BE6
E2C3-7BE5
E2C4-7C60
E2C5-7C00
E2C6-7C07
E2C7-7C13
E2C8-7BF3
E2C9-7BF7
E2CA-7C17
E2CB-7C0D
E2CC-7BF6
E2CD-7C23
E2CE-7C27
E2CF-7C2A
E2D0-7C1F
E2D1-7C37
E2D2-7C2B
E2D3-7C3D
E2D4-7C4C
E2D5-7C43
E2D6-7C54
E2D7-7C4F
E2D8-7C40
E2D9-7C50
E2DA-7C58
E2DB-7C5F
E2DC-7C64
E2DD-7C56
E2DE-7C65
E2DF-7C6C
E2E0-7C75
E2E1-7C83
E2E2-7C90
E2E3-7CA4
E2E4-7CAD
E2E5-7CA2
E2E6-7CAB
E2E7-7CA1
E2E8-7CA8
E2E9-7CB3
E2EA-7CB2
E2EB-7CB1
E2EC-7CAE
E2ED-7CB9
E2EE-7CBD
E2EF-7CC0
E2F0-7CC5
E2F1-7CC2
E2F2-7CD8
E2F3-7CD2
E2F4-7CDC
E2F5-7CE2
E2F6-9B3B
E2F7-7CEF
E2F8-7CF2
E2F9-7CF4
E2FA-7CF6
E2FB-7CFA
E2FC-7D06
E340-7D02
E341-7D1C
E342-7D15
E343-7D0A
E344-7D45
E345-7D4B
E346-7D2E
E347-7D32
E348-7D3F
E349-7D35
E34A-7D46
E34B-7D73
E34C-7D56
E34D-7D4E
E34E-7D72
E34F-7D68
E350-7D6E
E351-7D4F
E352-7D63
E353-7D93
E354-7D89
E355-7D5B
E356-7D8F
E357-7D7D
E358-7D9B
E359-7DBA
E35A-7DAE
E35B-7DA3
E35C-7DB5
E35D-7DC7
E35E-7DBD
E35F-7DAB
E360-7E3D
E361-7DA2
E362-7DAF
E363-7DDC
E364-7DB8
E365-7D9F
E366-7DB0
E367-7DD8
E368-7DDD
E369-7DE4
E36A-7DDE
E36B-7DFB
E36C-7DF2
E36D-7DE1
E36E-7E05
E36F-7E0A
E370-7E23
E371-7E21
E372-7E12
E373-7E31
E374-7E1F
E375-7E09
E376-7E0B
E377-7E22
E378-7E46
E379-7E66
E37A-7E3B
E37B-7E35
E37C-7E39
E37D-7E43
E37E-7E37
E380-7E32
E381-7E3A
E382-7E67
E383-7E5D
E384-7E56
E385-7E5E
E386-7E59
E387-7E5A
E388-7E79
E389-7E6A
E38A-7E69
E38B-7E7C
E38C-7E7B
E38D-7E83
E38E-7DD5
E38F-7E7D
E390-8FAE
E391-7E7F
E392-7E88
E393-7E89
E394-7E8C
E395-7E92
E396-7E90
E397-7E93
E398-7E94
E399-7E96
E39A-7E8E
E39B-7E9B
E39C-7E9C
E39D-7F38
E39E-7F3A
E39F-7F45
E3A0-7F4C
E3A1-7F4D
E3A2-7F4E
E3A3-7F50
E3A4-7F51
E3A5-7F55
E3A6-7F54
E3A7-7F58
E3A8-7F5F
E3A9-7F60
E3AA-7F68
E3AB-7F69
E3AC-7F67
E3AD-7F78
E3AE-7F82
E3AF-7F86
E3B0-7F83
E3B1-7F88
E3B2-7F87
E3B3-7F8C
E3B4-7F94
E3B5-7F9E
E3B6-7F9D
E3B7-7F9A
E3B8-7FA3
E3B9-7FAF
E3BA-7FB2
E3BB-7FB9
E3BC-7FAE
E3BD-7FB6
E3BE-7FB8
E3BF-8B71
E3C0-7FC5
E3C1-7FC6
E3C2-7FCA
E3C3-7FD5
E3C4-7FD4
E3C5-7FE1
E3C6-7FE6
E3C7-7FE9
E3C8-7FF3
E3C9-7FF9
E3CA-98DC
E3CB-8006
E3CC-8004
E3CD-800B
E3CE-8012
E3CF-8018
E3D0-8019
E3D1-801C
E3D2-8021
E3D3-8028
E3D4-803F
E3D5-803B
E3D6-804A
E3D7-8046
E3D8-8052
E3D9-8058
E3DA-805A
E3DB-805F
E3DC-8062
E3DD-8068
E3DE-8073
E3DF-8072
E3E0-8070
E3E1-8076
E3E2-8079
E3E3-807D
E3E4-807F
E3E5-8084
E3E6-8086
E3E7-8085
E3E8-809B
E3E9-8093
E3EA-809A
E3EB-80AD
E3EC-5190
E3ED-80AC
E3EE-80DB
E3EF-80E5
E3F0-80D9
E3F1-80DD
E3F2-80C4
E3F3-80DA
E3F4-80D6
E3F5-8109
E3F6-80EF
E3F7-80F1
E3F8-811B
E3F9-8129
E3FA-8123
E3FB-812F
E3FC-814B
E440-968B
E441-8146
E442-813E
E443-8153
E444-8151
E445-80FC
E446-8171
E447-816E
E448-8165
E449-8166
E44A-8174
E44B-8183
E44C-8188
E44D-818A
E44E-8180
E44F-8182
E450-81A0
E451-8195
E452-81A4
E453-81A3
E454-815F
E455-8193
E456-81A9
E457-81B0
E458-81B5
E459-81BE
E45A-81B8
E45B-81BD
E45C-81C0
E45D-81C2
E45E-81BA
E45F-81C9
E460-81CD
E461-81D1
E462-81D9
E463-81D8
E464-81C8
E465-81DA
E466-81DF
E467-81E0
E468-81E7
E469-81FA
E46A-81FB
E46B-81FE
E46C-8201
E46D-8202
E46E-8205
E46F-8207
E470-820A
E471-820D
E472-8210
E473-8216
E474-8229
E475-822B
E476-8238
E477-8233
E478-8240
E479-8259
E47A-8258
E47B-825D
E47C-825A
E47D-825F
E47E-8264
E480-8262
E481-8268
E482-826A
E483-826B
E484-822E
E485-8271
E486-8277
E487-8278
E488-827E
E489-828D
E48A-8292
E48B-82AB
E48C-829F
E48D-82BB
E48E-82AC
E48F-82E1
E490-82E3
E491-82DF
E492-82D2
E493-82F4
E494-82F3
E495-82FA
E496-8393
E497-8303
E498-82FB
E499-82F9
E49A-82DE
E49B-8306
E49C-82DC
E49D-8309
E49E-82D9
E49F-8335
E4A0-8334
E4A1-8316
E4A2-8332
E4A3-8331
E4A4-8340
E4A5-8339
E4A6-8350
E4A7-8345
E4A8-832F
E4A9-832B
E4AA-8317
E4AB-8318
E4AC-8385
E4AD-839A
E4AE-83AA
E4AF-839F
E4B0-83A2
E4B1-8396
E4B2-8323
E4B3-838E
E4B4-8387
E4B5-838A
E4B6-837C
E4B7-83B5
E4B8-8373
E4B9-8375
E4BA-83A0
E4BB-8389
E4BC-83A8
E4BD-83F4
E4BE-8413
E4BF-83EB
E4C0-83CE
E4C1-83FD
E4C2-8403
E4C3-83D8
E4C4-840B
E4C5-83C1
E4C6-83F7
E4C7-8407
E4C8-83E0
E4C9-83F2
E4CA-840D
E4CB-8422
E4CC-8420
E4CD-83BD
E4CE-8438
E4CF-8506
E4D0-83FB
E4D1-846D
E4D2-842A
E4D3-843C
E4D4-855A
E4D5-8484
E4D6-8477
E4D7-846B
E4D8-84AD
E4D9-846E
E4DA-8482
E4DB-8469
E4DC-8446
E4DD-842C
E4DE-846F
E4DF-8479
E4E0-8435
E4E1-84CA
E4E2-8462
E4E3-84B9
E4E4-84BF
E4E5-849F
E4E6-84D9
E4E7-84CD
E4E8-84BB
E4E9-84DA
E4EA-84D0
E4EB-84C1
E4EC-84C6
E4ED-84D6
E4EE-84A1
E4EF-8521
E4F0-84FF
E4F1-84F4
E4F2-8517
E4F3-8518
E4F4-852C
E4F5-851F
E4F6-8515
E4F7-8514
E4F8-84FC
E4F9-8540
E4FA-8563
E4FB-8558
E4FC-8548
E540-8541
E541-8602
E542-854B
E543-8555
E544-8580
E545-85A4
E546-8588
E547-8591
E548-858A
E549-85A8
E54A-856D
E54B-8594
E54C-859B
E54D-85EA
E54E-8587
E54F-859C
E550-8577
E551-857E
E552-8590
E553-85C9
E554-85BA
E555-85CF
E556-85B9
E557-85D0
E558-85D5
E559-85DD
E55A-85E5
E55B-85DC
E55C-85F9
E55D-860A
E55E-8613
E55F-860B
E560-85FE
E561-85FA
E562-8606
E563-8622
E564-861A
E565-8630
E566-863F
E567-864D
E568-4E55
E569-8654
E56A-865F
E56B-8667
E56C-8671
E56D-8693
E56E-86A3
E56F-86A9
E570-86AA
E571-868B
E572-868C
E573-86B6
E574-86AF
E575-86C4
E576-86C6
E577-86B0
E578-86C9
E579-8823
E57A-86AB
E57B-86D4
E57C-86DE
E57D-86E9
E57E-86EC
E580-86DF
E581-86DB
E582-86EF
E583-8712
E584-8706
E585-8708
E586-8700
E587-8703
E588-86FB
E589-8711
E58A-8709
E58B-870D
E58C-86F9
E58D-870A
E58E-8734
E58F-873F
E590-8737
E591-873B
E592-8725
E593-8729
E594-871A
E595-8760
E596-875F
E597-8778
E598-874C
E599-874E
E59A-8774
E59B-8757
E59C-8768
E59D-876E
E59E-8759
E59F-8753
E5A0-8763
E5A1-876A
E5A2-8805
E5A3-87A2
E5A4-879F
E5A5-8782
E5A6-87AF
E5A7-87CB
E5A8-87BD
E5A9-87C0
E5AA-87D0
E5AB-96D6
E5AC-87AB
E5AD-87C4
E5AE-87B3
E5AF-87C7
E5B0-87C6
E5B1-87BB
E5B2-87EF
E5B3-87F2
E5B4-87E0
E5B5-880F
E5B6-880D
E5B7-87FE
E5B8-87F6
E5B9-87F7
E5BA-880E
E5BB-87D2
E5BC-8811
E5BD-8816
E5BE-8815
E5BF-8822
E5C0-8821
E5C1-8831
E5C2-8836
E5C3-8839
E5C4-8827
E5C5-883B
E5C6-8844
E5C7-8842
E5C8-8852
E5C9-8859
E5CA-885E
E5CB-8862
E5CC-886B
E5CD-8881
E5CE-887E
E5CF-889E
E5D0-8875
E5D1-887D
E5D2-88B5
E5D3-8872
E5D4-8882
E5D5-8897
E5D6-8892
E5D7-88AE
E5D8-8899
E5D9-88A2
E5DA-888D
E5DB-88A4
E5DC-88B0
E5DD-88BF
E5DE-88B1
E5DF-88C3
E5E0-88C4
E5E1-88D4
E5E2-88D8
E5E3-88D9
E5E4-88DD
E5E5-88F9
E5E6-8902
E5E7-88FC
E5E8-88F4
E5E9-88E8
E5EA-88F2
E5EB-8904
E5EC-890C
E5ED-890A
E5EE-8913
E5EF-8943
E5F0-891E
E5F1-8925
E5F2-892A
E5F3-892B
E5F4-8941
E5F5-8944
E5F6-893B
E5F7-8936
E5F8-8938
E5F9-894C
E5FA-891D
E5FB-8960
E5FC-895E
E640-8966
E641-8964
E642-896D
E643-896A
E644-896F
E645-8974
E646-8977
E647-897E
E648-8983
E649-8988
E64A-898A
E64B-8993
E64C-8998
E64D-89A1
E64E-89A9
E64F-89A6
E650-89AC
E651-89AF
E652-89B2
E653-89BA
E654-89BD
E655-89BF
E656-89C0
E657-89DA
E658-89DC
E659-89DD
E65A-89E7
E65B-89F4
E65C-89F8
E65D-8A03
E65E-8A16
E65F-8A10
E660-8A0C
E661-8A1B
E662-8A1D
E663-8A25
E664-8A36
E665-8A41
E666-8A5B
E667-8A52
E668-8A46
E669-8A48
E66A-8A7C
E66B-8A6D
E66C-8A6C
E66D-8A62
E66E-8A85
E66F-8A82
E670-8A84
E671-8AA8
E672-8AA1
E673-8A91
E674-8AA5
E675-8AA6
E676-8A9A
E677-8AA3
E678-8AC4
E679-8ACD
E67A-8AC2
E67B-8ADA
E67C-8AEB
E67D-8AF3
E67E-8AE7
E680-8AE4
E681-8AF1
E682-8B14
E683-8AE0
E684-8AE2
E685-8AF7
E686-8ADE
E687-8ADB
E688-8B0C
E689-8B07
E68A-8B1A
E68B-8AE1
E68C-8B16
E68D-8B10
E68E-8B17
E68F-8B20
E690-8B33
E691-97AB
E692-8B26
E693-8B2B
E694-8B3E
E695-8B28
E696-8B41
E697-8B4C
E698-8B4F
E699-8B4E
E69A-8B49
E69B-8B56
E69C-8B5B
E69D-8B5A
E69E-8B6B
E69F-8B5F
E6A0-8B6C
E6A1-8B6F
E6A2-8B74
E6A3-8B7D
E6A4-8B80
E6A5-8B8C
E6A6-8B8E
E6A7-8B92
E6A8-8B93
E6A9-8B96
E6AA-8B99
E6AB-8B9A
E6AC-8C3A
E6AD-8C41
E6AE-8C3F
E6AF-8C48
E6B0-8C4C
E6B1-8C4E
E6B2-8C50
E6B3-8C55
E6B4-8C62
E6B5-8C6C
E6B6-8C78
E6B7-8C7A
E6B8-8C82
E6B9-8C89
E6BA-8C85
E6BB-8C8A
E6BC-8C8D
E6BD-8C8E
E6BE-8C94
E6BF-8C7C
E6C0-8C98
E6C1-621D
E6C2-8CAD
E6C3-8CAA
E6C4-8CBD
E6C5-8CB2
E6C6-8CB3
E6C7-8CAE
E6C8-8CB6
E6C9-8CC8
E6CA-8CC1
E6CB-8CE4
E6CC-8CE3
E6CD-8CDA
E6CE-8CFD
E6CF-8CFA
E6D0-8CFB
E6D1-8D04
E6D2-8D05
E6D3-8D0A
E6D4-8D07
E6D5-8D0F
E6D6-8D0D
E6D7-8D10
E6D8-9F4E
E6D9-8D13
E6DA-8CCD
E6DB-8D14
E6DC-8D16
E6DD-8D67
E6DE-8D6D
E6DF-8D71
E6E0-8D73
E6E1-8D81
E6E2-8D99
E6E3-8DC2
E6E4-8DBE
E6E5-8DBA
E6E6-8DCF
E6E7-8DDA
E6E8-8DD6
E6E9-8DCC
E6EA-8DDB
E6EB-8DCB
E6EC-8DEA
E6ED-8DEB
E6EE-8DDF
E6EF-8DE3
E6F0-8DFC
E6F1-8E08
E6F2-8E09
E6F3-8DFF
E6F4-8E1D
E6F5-8E1E
E6F6-8E10
E6F7-8E1F
E6F8-8E42
E6F9-8E35
E6FA-8E30
E6FB-8E34
E6FC-8E4A
E740-8E47
E741-8E49
E742-8E4C
E743-8E50
E744-8E48
E745-8E59
E746-8E64
E747-8E60
E748-8E2A
E749-8E63
E74A-8E55
E74B-8E76
E74C-8E72
E74D-8E7C
E74E-8E81
E74F-8E87
E750-8E85
E751-8E84
E752-8E8B
E753-8E8A
E754-8E93
E755-8E91
E756-8E94
E757-8E99
E758-8EAA
E759-8EA1
E75A-8EAC
E75B-8EB0
E75C-8EC6
E75D-8EB1
E75E-8EBE
E75F-8EC5
E760-8EC8
E761-8ECB
E762-8EDB
E763-8EE3
E764-8EFC
E765-8EFB
E766-8EEB
E767-8EFE
E768-8F0A
E769-8F05
E76A-8F15
E76B-8F12
E76C-8F19
E76D-8F13
E76E-8F1C
E76F-8F1F
E770-8F1B
E771-8F0C
E772-8F26
E773-8F33
E774-8F3B
E775-8F39
E776-8F45
E777-8F42
E778-8F3E
E779-8F4C
E77A-8F49
E77B-8F46
E77C-8F4E
E77D-8F57
E77E-8F5C
E780-8F62
E781-8F63
E782-8F64
E783-8F9C
E784-8F9F
E785-8FA3
E786-8FAD
E787-8FAF
E788-8FB7
E789-8FDA
E78A-8FE5
E78B-8FE2
E78C-8FEA
E78D-8FEF
E78E-9087
E78F-8FF4
E790-9005
E791-8FF9
E792-8FFA
E793-9011
E794-9015
E795-9021
E796-900D
E797-901E
E798-9016
E799-900B
E79A-9027
E79B-9036
E79C-9035
E79D-9039
E79E-8FF8
E79F-904F
E7A0-9050
E7A1-9051
E7A2-9052
E7A3-900E
E7A4-9049
E7A5-903E
E7A6-9056
E7A7-9058
E7A8-905E
E7A9-9068
E7AA-906F
E7AB-9076
E7AC-96A8
E7AD-9072
E7AE-9082
E7AF-907D
E7B0-9081
E7B1-9080
E7B2-908A
E7B3-9089
E7B4-908F
E7B5-90A8
E7B6-90AF
E7B7-90B1
E7B8-90B5
E7B9-90E2
E7BA-90E4
E7BB-6248
E7BC-90DB
E7BD-9102
E7BE-9112
E7BF-9119
E7C0-9132
E7C1-9130
E7C2-914A
E7C3-9156
E7C4-9158
E7C5-9163
E7C6-9165
E7C7-9169
E7C8-9173
E7C9-9172
E7CA-918B
E7CB-9189
E7CC-9182
E7CD-91A2
E7CE-91AB
E7CF-91AF
E7D0-91AA
E7D1-91B5
E7D2-91B4
E7D3-91BA
E7D4-91C0
E7D5-91C1
E7D6-91C9
E7D7-91CB
E7D8-91D0
E7D9-91D6
E7DA-91DF
E7DB-91E1
E7DC-91DB
E7DD-91FC
E7DE-91F5
E7DF-91F6
E7E0-921E
E7E1-91FF
E7E2-9214
E7E3-922C
E7E4-9215
E7E5-9211
E7E6-925E
E7E7-9257
E7E8-9245
E7E9-9249
E7EA-9264
E7EB-9248
E7EC-9295
E7ED-923F
E7EE-924B
E7EF-9250
E7F0-929C
E7F1-9296
E7F2-9293
E7F3-929B
E7F4-925A
E7F5-92CF
E7F6-92B9
E7F7-92B7
E7F8-92E9
E7F9-930F
E7FA-92FA
E7FB-9344
E7FC-932E
E840-9319
E841-9322
E842-931A
E843-9323
E844-933A
E845-9335
E846-933B
E847-935C
E848-9360
E849-937C
E84A-936E
E84B-9356
E84C-93B0
E84D-93AC
E84E-93AD
E84F-9394
E850-93B9
E851-93D6
E852-93D7
E853-93E8
E854-93E5
E855-93D8
E856-93C3
E857-93DD
E858-93D0
E859-93C8
E85A-93E4
E85B-941A
E85C-9414
E85D-9413
E85E-9403
E85F-9407
E860-9410
E861-9436
E862-942B
E863-9435
E864-9421
E865-943A
E866-9441
E867-9452
E868-9444
E869-945B
E86A-9460
E86B-9462
E86C-945E
E86D-946A
E86E-9229
E86F-9470
E870-9475
E871-9477
E872-947D
E873-945A
E874-947C
E875-947E
E876-9481
E877-947F
E878-9582
E879-9587
E87A-958A
E87B-9594
E87C-9596
E87D-9598
E87E-9599
E880-95A0
E881-95A8
E882-95A7
E883-95AD
E884-95BC
E885-95BB
E886-95B9
E887-95BE
E888-95CA
E889-6FF6
E88A-95C3
E88B-95CD
E88C-95CC
E88D-95D5
E88E-95D4
E88F-95D6
E890-95DC
E891-95E1
E892-95E5
E893-95E2
E894-9621
E895-9628
E896-962E
E897-962F
E898-9642
E899-964C
E89A-964F
E89B-964B
E89C-9677
E89D-965C
E89E-965E
E89F-965D
E8A0-965F
E8A1-9666
E8A2-9672
E8A3-966C
E8A4-968D
E8A5-9698
E8A6-9695
E8A7-9697
E8A8-96AA
E8A9-96A7
E8AA-96B1
E8AB-96B2
E8AC-96B0
E8AD-96B4
E8AE-96B6
E8AF-96B8
E8B0-96B9
E8B1-96CE
E8B2-96CB
E8B3-96C9
E8B4-96CD
E8B5-894D
E8B6-96DC
E8B7-970D
E8B8-96D5
E8B9-96F9
E8BA-9704
E8BB-9706
E8BC-9708
E8BD-9713
E8BE-970E
E8BF-9711
E8C0-970F
E8C1-9716
E8C2-9719
E8C3-9724
E8C4-972A
E8C5-9730
E8C6-9739
E8C7-973D
E8C8-973E
E8C9-9744
E8CA-9746
E8CB-9748
E8CC-9742
E8CD-9749
E8CE-975C
E8CF-9760
E8D0-9764
E8D1-9766
E8D2-9768
E8D3-52D2
E8D4-976B
E8D5-9771
E8D6-9779
E8D7-9785
E8D8-977C
E8D9-9781
E8DA-977A
E8DB-9786
E8DC-978B
E8DD-978F
E8DE-9790
E8DF-979C
E8E0-97A8
E8E1-97A6
E8E2-97A3
E8E3-97B3
E8E4-97B4
E8E5-97C3
E8E6-97C6
E8E7-97C8
E8E8-97CB
E8E9-97DC
E8EA-97ED
E8EB-9F4F
E8EC-97F2
E8ED-7ADF
E8EE-97F6
E8EF-97F5
E8F0-980F
E8F1-980C
E8F2-9838
E8F3-9824
E8F4-9821
E8F5-9837
E8F6-983D
E8F7-9846
E8F8-984F
E8F9-984B
E8FA-986B
E8FB-986F
E8FC-9870
E940-9871
E941-9874
E942-9873
E943-98AA
E944-98AF
E945-98B1
E946-98B6
E947-98C4
E948-98C3
E949-98C6
E94A-98E9
E94B-98EB
E94C-9903
E94D-9909
E94E-9912
E94F-9914
E950-9918
E951-9921
E952-991D
E953-991E
E954-9924
E955-9920
E956-992C
E957-992E
E958-993D
E959-993E
E95A-9942
E95B-9949
E95C-9945
E95D-9950
E95E-994B
E95F-9951
E960-9952
E961-994C
E962-9955
E963-9997
E964-9998
E965-99A5
E966-99AD
E967-99AE
E968-99BC
E969-99DF
E96A-99DB
E96B-99DD
E96C-99D8
E96D-99D1
E96E-99ED
E96F-99EE
E970-99F1
E971-99F2
E972-99FB
E973-99F8
E974-9A01
E975-9A0F
E976-9A05
E977-99E2
E978-9A19
E979-9A2B
E97A-9A37
E97B-9A45
E97C-9A42
E97D-9A40
E97E-9A43
E980-9A3E
E981-9A55
E982-9A4D
E983-9A5B
E984-9A57
E985-9A5F
E986-9A62
E987-9A65
E988-9A64
E989-9A69
E98A-9A6B
E98B-9A6A
E98C-9AAD
E98D-9AB0
E98E-9ABC
E98F-9AC0
E990-9ACF
E991-9AD1
E992-9AD3
E993-9AD4
E994-9ADE
E995-9ADF
E996-9AE2
E997-9AE3
E998-9AE6
E999-9AEF
E99A-9AEB
E99B-9AEE
E99C-9AF4
E99D-9AF1
E99E-9AF7
E99F-9AFB
E9A0-9B06
E9A1-9B18
E9A2-9B1A
E9A3-9B1F
E9A4-9B22
E9A5-9B23
E9A6-9B25
E9A7-9B27
E9A8-9B28
E9A9-9B29
E9AA-9B2A
E9AB-9B2E
E9AC-9B2F
E9AD-9B32
E9AE-9B44
E9AF-9B43
E9B0-9B4F
E9B1-9B4D
E9B2-9B4E
E9B3-9B51
E9B4-9B58
E9B5-9B74
E9B6-9B93
E9B7-9B83
E9B8-9B91
E9B9-9B96
E9BA-9B97
E9BB-9B9F
E9BC-9BA0
E9BD-9BA8
E9BE-9BB4
E9BF-9BC0
E9C0-9BCA
E9C1-9BB9
E9C2-9BC6
E9C3-9BCF
E9C4-9BD1
E9C5-9BD2
E9C6-9BE3
E9C7-9BE2
E9C8-9BE4
E9C9-9BD4
E9CA-9BE1
E9CB-9C3A
E9CC-9BF2
E9CD-9BF1
E9CE-9BF0
E9CF-9C15
E9D0-9C14
E9D1-9C09
E9D2-9C13
E9D3-9C0C
E9D4-9C06
E9D5-9C08
E9D6-9C12
E9D7-9C0A
E9D8-9C04
E9D9-9C2E
E9DA-9C1B
E9DB-9C25
E9DC-9C24
E9DD-9C21
E9DE-9C30
E9DF-9C47
E9E0-9C32
E9E1-9C46
E9E2-9C3E
E9E3-9C5A
E9E4-9C60
E9E5-9C67
E9E6-9C76
E9E7-9C78
E9E8-9CE7
E9E9-9CEC
E9EA-9CF0
E9EB-9D09
E9EC-9D08
E9ED-9CEB
E9EE-9D03
E9EF-9D06
E9F0-9D2A
E9F1-9D26
E9F2-9DAF
E9F3-9D23
E9F4-9D1F
E9F5-9D44
E9F6-9D15
E9F7-9D12
E9F8-9D41
E9F9-9D3F
E9FA-9D3E
E9FB-9D46
E9FC-9D48
EA40-9D5D
EA41-9D5E
EA42-9D64
EA43-9D51
EA44-9D50
EA45-9D59
EA46-9D72
EA47-9D89
EA48-9D87
EA49-9DAB
EA4A-9D6F
EA4B-9D7A
EA4C-9D9A
EA4D-9DA4
EA4E-9DA9
EA4F-9DB2
EA50-9DC4
EA51-9DC1
EA52-9DBB
EA53-9DB8
EA54-9DBA
EA55-9DC6
EA56-9DCF
EA57-9DC2
EA58-9DD9
EA59-9DD3
EA5A-9DF8
EA5B-9DE6
EA5C-9DED
EA5D-9DEF
EA5E-9DFD
EA5F-9E1A
EA60-9E1B
EA61-9E1E
EA62-9E75
EA63-9E79
EA64-9E7D
EA65-9E81
EA66-9E88
EA67-9E8B
EA68-9E8C
EA69-9E92
EA6A-9E95
EA6B-9E91
EA6C-9E9D
EA6D-9EA5
EA6E-9EA9
EA6F-9EB8
EA70-9EAA
EA71-9EAD
EA72-9761
EA73-9ECC
EA74-9ECE
EA75-9ECF
EA76-9ED0
EA77-9ED4
EA78-9EDC
EA79-9EDE
EA7A-9EDD
EA7B-9EE0
EA7C-9EE5
EA7D-9EE8
EA7E-9EEF
EA80-9EF4
EA81-9EF6
EA82-9EF7
EA83-9EF9
EA84-9EFB
EA85-9EFC
EA86-9EFD
EA87-9F07
EA88-9F08
EA89-76B7
EA8A-9F15
EA8B-9F21
EA8C-9F2C
EA8D-9F3E
EA8E-9F4A
EA8F-9F52
EA90-9F54
EA91-9F63
EA92-9F5F
EA93-9F60
EA94-9F61
EA95-9F66
EA96-9F67
EA97-9F6C
EA98-9F6A
EA99-9F77
EA9A-9F72
EA9B-9F76
EA9C-9F95
EA9D-9F9C
EA9E-9FA0
EA9F-582F
EAA0-69C7
EAA1-9059
EAA2-7464
EAA3-51DC
EAA4-7199
ED40-7E8A-f
ED41-891C-f
ED42-9348-f
ED43-9288-f
ED44-84DC-f
ED45-4FC9-f
ED46-70BB-f
ED47-6631-f
ED48-68C8-f
ED49-92F9-f
ED4A-66FB-f
ED4B-5F45-f
ED4C-4E28-f
ED4D-4EE1-f
ED4E-4EFC-f
ED4F-4F00-f
ED50-4F03-f
ED51-4F39-f
ED52-4F56-f
ED53-4F92-f
ED54-4F8A-f
ED55-4F9A-f
ED56-4F94-f
ED57-4FCD-f
ED58-5040-f
ED59-5022-f
ED5A-4FFF-f
ED5B-501E-f
ED5C-5046-f
ED5D-5070-f
ED5E-5042-f
ED5F-5094-f
ED60-50F4-f
ED61-50D8-f
ED62-514A-f
ED63-5164-f
ED64-519D-f
ED65-51BE-f
ED66-51EC-f
ED67-5215-f
ED68-529C-f
ED69-52A6-f
ED6A-52C0-f
ED6B-52DB-f
ED6C-5300-f
ED6D-5307-f
ED6E-5324-f
ED6F-5372-f
ED70-5393-f
ED71-53B2-f
ED72-53DD-f
ED73-FA0E-f
ED74-549C-f
ED75-548A-f
ED76-54A9-f
ED77-54FF-f
ED78-5586-f
ED79-5759-f
ED7A-5765-f
ED7B-57AC-f
ED7C-57C8-f
ED7D-57C7-f
ED7E-FA0F-f
ED80-FA10-f
ED81-589E-f
ED82-58B2-f
ED83-590B-f
ED84-5953-f
ED85-595B-f
ED86-595D-f
ED87-5963-f
ED88-59A4-f
ED89-59BA-f
ED8A-5B56-f
ED8B-5BC0-f
ED8C-752F-f
ED8D-5BD8-f
ED8E-5BEC-f
ED8F-5C1E-f
ED90-5CA6-f
ED91-5CBA-f
ED92-5CF5-f
ED93-5D27-f
ED94-5D53-f
ED95-FA11-f
ED96-5D42-f
ED97-5D6D-f
ED98-5DB8-f
ED99-5DB9-f
ED9A-5DD0-f
ED9B-5F21-f
ED9C-5F34-f
ED9D-5F67-f
ED9E-5FB7-f
ED9F-5FDE-f
EDA0-605D-f
EDA1-6085-f
EDA2-608A-f
EDA3-60DE-f
EDA4-60D5-f
EDA5-6120-f
EDA6-60F2-f
EDA7-6111-f
EDA8-6137-f
EDA9-6130-f
EDAA-6198-f
EDAB-6213-f
EDAC-62A6-f
EDAD-63F5-f
EDAE-6460-f
EDAF-649D-f
EDB0-64CE-f
EDB1-654E-f
EDB2-6600-f
EDB3-6615-f
EDB4-663B-f
EDB5-6609-f
EDB6-662E-f
EDB7-661E-f
EDB8-6624-f
EDB9-6665-f
EDBA-6657-f
EDBB-6659-f
EDBC-FA12-f
EDBD-6673-f
EDBE-6699-f
EDBF-66A0-f
EDC0-66B2-f
EDC1-66BF-f
EDC2-66FA-f
EDC3-670E-f
EDC4-F929-f
EDC5-6766-f
EDC6-67BB-f
EDC7-6852-f
EDC8-67C0-f
EDC9-6801-f
EDCA-6844-f
EDCB-68CF-f
EDCC-FA13-f
EDCD-6968-f
EDCE-FA14-f
EDCF-6998-f
EDD0-69E2-f
EDD1-6A30-f
EDD2-6A6B-f
EDD3-6A46-f
EDD4-6A73-f
EDD5-6A7E-f
EDD6-6AE2-f
EDD7-6AE4-f
EDD8-6BD6-f
EDD9-6C3F-f
EDDA-6C5C-f
EDDB-6C86-f
EDDC-6C6F-f
EDDD-6CDA-f
EDDE-6D04-f
EDDF-6D87-f
EDE0-6D6F-f
EDE1-6D96-f
EDE2-6DAC-f
EDE3-6DCF-f
EDE4-6DF8-f
EDE5-6DF2-f
EDE6-6DFC-f
EDE7-6E39-f
EDE8-6E5C-f
EDE9-6E27-f
EDEA-6E3C-f
EDEB-6EBF-f
EDEC-6F88-f
EDED-6FB5-f
EDEE-6FF5-f
EDEF-7005-f
EDF0-7007-f
EDF1-7028-f
EDF2-7085-f
EDF3-70AB-f
EDF4-710F-f
EDF5-7104-f
EDF6-715C-f
EDF7-7146-f
EDF8-7147-f
EDF9-FA15-f
EDFA-71C1-f
EDFB-71FE-f
EDFC-72B1-f
EE40-72BE-f
EE41-7324-f
EE42-FA16-f
EE43-7377-f
EE44-73BD-f
EE45-73C9-f
EE46-73D6-f
EE47-73E3-f
EE48-73D2-f
EE49-7407-f
EE4A-73F5-f
EE4B-7426-f
EE4C-742A-f
EE4D-7429-f
EE4E-742E-f
EE4F-7462-f
EE50-7489-f
EE51-749F-f
EE52-7501-f
EE53-756F-f
EE54-7682-f
EE55-769C-f
EE56-769E-f
EE57-769B-f
EE58-76A6-f
EE59-FA17-f
EE5A-7746-f
EE5B-52AF-f
EE5C-7821-f
EE5D-784E-f
EE5E-7864-f
EE5F-787A-f
EE60-7930-f
EE61-FA18-f
EE62-FA19-f
EE63-FA1A-f
EE64-7994-f
EE65-FA1B-f
EE66-799B-f
EE67-7AD1-f
EE68-7AE7-f
EE69-FA1C-f
EE6A-7AEB-f
EE6B-7B9E-f
EE6C-FA1D-f
EE6D-7D48-f
EE6E-7D5C-f
EE6F-7DB7-f
EE70-7DA0-f
EE71-7DD6-f
EE72-7E52-f
EE73-7F47-f
EE74-7FA1-f
EE75-FA1E-f
EE76-8301-f
EE77-8362-f
EE78-837F-f
EE79-83C7-f
EE7A-83F6-f
EE7B-8448-f
EE7C-84B4-f
EE7D-8553-f
EE7E-8559-f
EE80-856B-f
EE81-FA1F-f
EE82-85B0-f
EE83-FA20-f
EE84-FA21-f
EE85-8807-f
EE86-88F5-f
EE87-8A12-f
EE88-8A37-f
EE89-8A79-f
EE8A-8AA7-f
EE8B-8ABE-f
EE8C-8ADF-f
EE8D-FA22-f
EE8E-8AF6-f
EE8F-8B53-f
EE90-8B7F-f
EE91-8CF0-f
EE92-8CF4-f
EE93-8D12-f
EE94-8D76-f
EE95-FA23-f
EE96-8ECF-f
EE97-FA24-f
EE98-FA25-f
EE99-9067-f
EE9A-90DE-f
EE9B-FA26-f
EE9C-9115-f
EE9D-9127-f
EE9E-91DA-f
EE9F-91D7-f
EEA0-91DE-f
EEA1-91ED-f
EEA2-91EE-f
EEA3-91E4-f
EEA4-91E5-f
EEA5-9206-f
EEA6-9210-f
EEA7-920A-f
EEA8-923A-f
EEA9-9240-f
EEAA-923C-f
EEAB-924E-f
EEAC-9259-f
EEAD-9251-f
EEAE-9239-f
EEAF-9267-f
EEB0-92A7-f
EEB1-9277-f
EEB2-9278-f
EEB3-92E7-f
EEB4-92D7-f
EEB5-92D9-f
EEB6-92D0-f
EEB7-FA27-f
EEB8-92D5-f
EEB9-92E0-f
EEBA-92D3-f
EEBB-9325-f
EEBC-9321-f
EEBD-92FB-f
EEBE-FA28-f
EEBF-931E-f
EEC0-92FF-f
EEC1-931D-f
EEC2-9302-f
EEC3-9370-f
EEC4-9357-f
EEC5-93A4-f
EEC6-93C6-f
EEC7-93DE-f
EEC8-93F8-f
EEC9-9431-f
EECA-9445-f
EECB-9448-f
EECC-9592-f
EECD-F9DC-f
EECE-FA29-f
EECF-969D-f
EED0-96AF-f
EED1-9733-f
EED2-973B-f
EED3-9743-f
EED4-974D-f
EED5-974F-f
EED6-9751-f
EED7-9755-f
EED8-9857-f
EED9-9865-f
EEDA-FA2A-f
EEDB-FA2B-f
EEDC-9927-f
EEDD-FA2C-f
EEDE-999E-f
EEDF-9A4E-f
EEE0-9AD9-f
EEE1-9ADC-f
EEE2-9B75-f
EEE3-9B72-f
EEE4-9B8F-f
EEE5-9BB1-f
EEE6-9BBB-f
EEE7-9C00-f
EEE8-9D70-f
EEE9-9D6B-f
EEEA-FA2D-f
EEEB-9E19-f
EEEC-9ED1-f
EEEF-2170-f
EEF0-2171-f
EEF1-2172-f
EEF2-2173-f
EEF3-2174-f
EEF4-2175-f
EEF5-2176-f
EEF6-2177-f
EEF7-2178-f
EEF8-2179-f
EEF9-FFE2-f
EEFA-FFE4-f
EEFB-FF07-f
EEFC-FF02-f
FA40-2170-t
FA41-2171-t
FA42-2172-t
FA43-2173-t
FA44-2174-t
FA45-2175-t
FA46-2176-t
FA47-2177-t
FA48-2178-t
FA49-2179-t
FA4A-2160-f
FA4B-2161-f
FA4C-2162-f
FA4D-2163-f
FA4E-2164-f
FA4F-2165-f
FA50-2166-f
FA51-2167-f
FA52-2168-f
FA53-2169-f
FA54-FFE2-f
FA55-FFE4-t
FA56-FF07-t
FA57-FF02-t
FA58-3231-f
FA59-2116-f
FA5A-2121-f
FA5B-2235-f
FA5C-7E8A-t
FA5D-891C-t
FA5E-9348-t
FA5F-9288-t
FA60-84DC-t
FA61-4FC9-t
FA62-70BB-t
FA63-6631-t
FA64-68C8-t
FA65-92F9-t
FA66-66FB-t
FA67-5F45-t
FA68-4E28-t
FA69-4EE1-t
FA6A-4EFC-t
FA6B-4F00-t
FA6C-4F03-t
FA6D-4F39-t
FA6E-4F56-t
FA6F-4F92-t
FA70-4F8A-t
FA71-4F9A-t
FA72-4F94-t
FA73-4FCD-t
FA74-5040-t
FA75-5022-t
FA76-4FFF-t
FA77-501E-t
FA78-5046-t
FA79-5070-t
FA7A-5042-t
FA7B-5094-t
FA7C-50F4-t
FA7D-50D8-t
FA7E-514A-t
FA80-5164-t
FA81-519D-t
FA82-51BE-t
FA83-51EC-t
FA84-5215-t
FA85-529C-t
FA86-52A6-t
FA87-52C0-t
FA88-52DB-t
FA89-5300-t
FA8A-5307-t
FA8B-5324-t
FA8C-5372-t
FA8D-5393-t
FA8E-53B2-t
FA8F-53DD-t
FA90-FA0E-t
FA91-549C-t
FA92-548A-t
FA93-54A9-t
FA94-54FF-t
FA95-5586-t
FA96-5759-t
FA97-5765-t
FA98-57AC-t
FA99-57C8-t
FA9A-57C7-t
FA9B-FA0F-t
FA9C-FA10-t
FA9D-589E-t
FA9E-58B2-t
FA9F-590B-t
FAA0-5953-t
FAA1-595B-t
FAA2-595D-t
FAA3-5963-t
FAA4-59A4-t
FAA5-59BA-t
FAA6-5B56-t
FAA7-5BC0-t
FAA8-752F-t
FAA9-5BD8-t
FAAA-5BEC-t
FAAB-5C1E-t
FAAC-5CA6-t
FAAD-5CBA-t
FAAE-5CF5-t
FAAF-5D27-t
FAB0-5D53-t
FAB1-FA11-t
FAB2-5D42-t
FAB3-5D6D-t
FAB4-5DB8-t
FAB5-5DB9-t
FAB6-5DD0-t
FAB7-5F21-t
FAB8-5F34-t
FAB9-5F67-t
FABA-5FB7-t
FABB-5FDE-t
FABC-605D-t
FABD-6085-t
FABE-608A-t
FABF-60DE-t
FAC0-60D5-t
FAC1-6120-t
FAC2-60F2-t
FAC3-6111-t
FAC4-6137-t
FAC5-6130-t
FAC6-6198-t
FAC7-6213-t
FAC8-62A6-t
FAC9-63F5-t
FACA-6460-t
FACB-649D-t
FACC-64CE-t
FACD-654E-t
FACE-6600-t
FACF-6615-t
FAD0-663B-t
FAD1-6609-t
FAD2-662E-t
FAD3-661E-t
FAD4-6624-t
FAD5-6665-t
FAD6-6657-t
FAD7-6659-t
FAD8-FA12-t
FAD9-6673-t
FADA-6699-t
FADB-66A0-t
FADC-66B2-t
FADD-66BF-t
FADE-66FA-t
FADF-670E-t
FAE0-F929-t
FAE1-6766-t
FAE2-67BB-t
FAE3-6852-t
FAE4-67C0-t
FAE5-6801-t
FAE6-6844-t
FAE7-68CF-t
FAE8-FA13-t
FAE9-6968-t
FAEA-FA14-t
FAEB-6998-t
FAEC-69E2-t
FAED-6A30-t
FAEE-6A6B-t
FAEF-6A46-t
FAF0-6A73-t
FAF1-6A7E-t
FAF2-6AE2-t
FAF3-6AE4-t
FAF4-6BD6-t
FAF5-6C3F-t
FAF6-6C5C-t
FAF7-6C86-t
FAF8-6C6F-t
FAF9-6CDA-t
FAFA-6D04-t
FAFB-6D87-t
FAFC-6D6F-t
FB40-6D96-t
FB41-6DAC-t
FB42-6DCF-t
FB43-6DF8-t
FB44-6DF2-t
FB45-6DFC-t
FB46-6E39-t
FB47-6E5C-t
FB48-6E27-t
FB49-6E3C-t
FB4A-6EBF-t
FB4B-6F88-t
FB4C-6FB5-t
FB4D-6FF5-t
FB4E-7005-t
FB4F-7007-t
FB50-7028-t
FB51-7085-t
FB52-70AB-t
FB53-710F-t
FB54-7104-t
FB55-715C-t
FB56-7146-t
FB57-7147-t
FB58-FA15-t
FB59-71C1-t
FB5A-71FE-t
FB5B-72B1-t
FB5C-72BE-t
FB5D-7324-t
FB5E-FA16-t
FB5F-7377-t
FB60-73BD-t
FB61-73C9-t
FB62-73D6-t
FB63-73E3-t
FB64-73D2-t
FB65-7407-t
FB66-73F5-t
FB67-7426-t
FB68-742A-t
FB69-7429-t
FB6A-742E-t
FB6B-7462-t
FB6C-7489-t
FB6D-749F-t
FB6E-7501-t
FB6F-756F-t
FB70-7682-t
FB71-769C-t
FB72-769E-t
FB73-769B-t
FB74-76A6-t
FB75-FA17-t
FB76-7746-t
FB77-52AF-t
FB78-7821-t
FB79-784E-t
FB7A-7864-t
FB7B-787A-t
FB7C-7930-t
FB7D-FA18-t
FB7E-FA19-t
FB80-FA1A-t
FB81-7994-t
FB82-FA1B-t
FB83-799B-t
FB84-7AD1-t
FB85-7AE7-t
FB86-FA1C-t
FB87-7AEB-t
FB88-7B9E-t
FB89-FA1D-t
FB8A-7D48-t
FB8B-7D5C-t
FB8C-7DB7-t
FB8D-7DA0-t
FB8E-7DD6-t
FB8F-7E52-t
FB90-7F47-t
FB91-7FA1-t
FB92-FA1E-t
FB93-8301-t
FB94-8362-t
FB95-837F-t
FB96-83C7-t
FB97-83F6-t
FB98-8448-t
FB99-84B4-t
FB9A-8553-t
FB9B-8559-t
FB9C-856B-t
FB9D-FA1F-t
FB9E-85B0-t
FB9F-FA20-t
FBA0-FA21-t
FBA1-8807-t
FBA2-88F5-t
FBA3-8A12-t
FBA4-8A37-t
FBA5-8A79-t
FBA6-8AA7-t
FBA7-8ABE-t
FBA8-8ADF-t
FBA9-FA22-t
FBAA-8AF6-t
FBAB-8B53-t
FBAC-8B7F-t
FBAD-8CF0-t
FBAE-8CF4-t
FBAF-8D12-t
FBB0-8D76-t
FBB1-FA23-t
FBB2-8ECF-t
FBB3-FA24-t
FBB4-FA25-t
FBB5-9067-t
FBB6-90DE-t
FBB7-FA26-t
FBB8-9115-t
FBB9-9127-t
FBBA-91DA-t
FBBB-91D7-t
FBBC-91DE-t
FBBD-91ED-t
FBBE-91EE-t
FBBF-91E4-t
FBC0-91E5-t
FBC1-9206-t
FBC2-9210-t
FBC3-920A-t
FBC4-923A-t
FBC5-9240-t
FBC6-923C-t
FBC7-924E-t
FBC8-9259-t
FBC9-9251-t
FBCA-9239-t
FBCB-9267-t
FBCC-92A7-t
FBCD-9277-t
FBCE-9278-t
FBCF-92E7-t
FBD0-92D7-t
FBD1-92D9-t
FBD2-92D0-t
FBD3-FA27-t
FBD4-92D5-t
FBD5-92E0-t
FBD6-92D3-t
FBD7-9325-t
FBD8-9321-t
FBD9-92FB-t
FBDA-FA28-t
FBDB-931E-t
FBDC-92FF-t
FBDD-931D-t
FBDE-9302-t
FBDF-9370-t
FBE0-9357-t
FBE1-93A4-t
FBE2-93C6-t
FBE3-93DE-t
FBE4-93F8-t
FBE5-9431-t
FBE6-9445-t
FBE7-9448-t
FBE8-9592-t
FBE9-F9DC-t
FBEA-FA29-t
FBEB-969D-t
FBEC-96AF-t
FBED-9733-t
FBEE-973B-t
FBEF-9743-t
FBF0-974D-t
FBF1-974F-t
FBF2-9751-t
FBF3-9755-t
FBF4-9857-t
FBF5-9865-t
FBF6-FA2A-t
FBF7-FA2B-t
FBF8-9927-t
FBF9-FA2C-t
FBFA-999E-t
FBFB-9A4E-t
FBFC-9AD9-t
FC40-9ADC-t
FC41-9B75-t
FC42-9B72-t
FC43-9B8F-t
FC44-9BB1-t
FC45-9BBB-t
FC46-9C00-t
FC47-9D70-t
FC48-9D6B-t
FC49-FA2D-t
FC4A-9E19-t
FC4B-9ED1-t
/){
  my @t = split /-/, $_, 3;
  $t[0] = hex $t[0];
  $t[1] = hex $t[1];
  my $cp932 = $t[0] > 0xFF ? pack('n', $t[0]) : chr($t[0]);
  my $utf16 = pack('v',$t[1]); # no char > 0x10000.
  my $utf8  = toUTF8($t[1]);
  if(!$t[2] or $t[2] eq 't'){
    $CP932_UTF8{$cp932}  = $utf8;
    $CP932_UTF16{$cp932} = $utf16;
    $UTF8_CP932{$utf8}   = $UTF16_CP932{$utf16} = $cp932;
  } else {
    $CP932_UTF8{$cp932}  = $utf8;
    $CP932_UTF16{$cp932} = $utf16;
  }
}

$UTF16_CP932{"\xFF\xFE"} = '';

sub cp932_to_utf8 {
  my $coderef;
  if(ref $_[0] eq 'CODE'){ $coderef = shift }
  my $str = shift;
  defined $coderef
    ? $str =~ s/($Schar)/
        exists $CP932_UTF8{$1} ? $CP932_UTF8{$1} : &$coderef($1)
      /geo
    : $str =~ s/($Schar)/$CP932_UTF8{$1}/go;
  return $str;
}

sub utf8_to_cp932 {
  my $coderef;
  if(ref $_[0] eq 'CODE'){ $coderef = shift }
  my $str = shift;
  defined $coderef
    ? $str =~ s/($Uchar)/
        exists $UTF8_CP932{$1} ? $UTF8_CP932{$1} : &$coderef(fromUTF8($1))
      /geo
    : $str =~ s/($Uchar)/$UTF8_CP932{$1}/go;
  return $str;
}

sub cp932_to_utf16 {
  my $coderef;
  if(ref $_[0] eq 'CODE'){ $coderef = shift }
  my $str = shift;
  defined $coderef
    ? $str =~ s/($Schar)/
      exists $CP932_UTF16{$1} ? $CP932_UTF16{$1} : &$coderef($1)
     /geo
    : $str =~ s/($Schar)/$CP932_UTF16{$1}/go;
  return $str;
}

sub utf16_to_cp932 {
  my $coderef;
  if(ref $_[0] eq 'CODE'){ $coderef = shift }
  my $str = shift;
  defined $coderef
    ? $str =~ s/($Hchar)/
        exists $UTF16_CP932{$1} ? $UTF16_CP932{$1} : &$coderef(fromUTF16($1))
      /geo
    : $str =~ s/($Hchar)/$UTF16_CP932{$1}/go;
  return $str;
}

1;
__END__

=head1 NAME

ShiftJIS::CP932::MapUTF - Conversion between CP-932 and Unicode

=head1 SYNOPSIS

  use ShiftJIS::CP932::MapUTF;

  $utf8_string  = cp932_to_utf8($cp932_string);

=head1 DESCRIPTION

The Microsoft CodePage 932 (CP-932) table comprises 7915 characters:

  JIS X 0201-1976 single-byte characters (191 characters),
  JIS X 0208-1990 double-byte characters (6879 characters),
  NEC special characters (83 characters from SJIS row 13),
  NEC-selected IBM extended characters (374 characters from SJIS row 89 to 92),
  and IBM extended characters (388 characters from SJIS row 115 to 119).

It contains duplicates that do not round trip
map. These duplicates are due to the characters defined
by vendors, NEC and IBM.

For example, there are two characters that are mapped to U+2252,
namely, 0x81e0 (JIS X 0208) and 0x8790 (NEC special character).

This module provides some functions to maps
from CP-932 to Unicode, and vice versa. 

=head2 FUNCTIONS EXPORTED BY DEFAULT

=over 4

=item C<cp932_to_utf8(STRING)>

=item C<cp932_to_utf8(CODEREF, STRING)>

Converts CP-932 to UTF-8.

For example, converts C<\x81\xe0> or C<\x87\x90> to C<U+2252>
in the UTF-8 encoding.

If C<CODEREF> is not specified, characters that aren't mapped to Unicode
are deleted.
If C<CODEREF> is specified, characters that aren't mapped to Unicode
are converted using C<CODEREF> from the CP-932 character string.

For example, converts C<\x82\xf2> to C<U+3094>, HIRAGANA LETTER VU,
in the UTF-8 encoding.

 cp932_to_utf8(
   sub { $_[0] eq "\x82\xf2" ? toUTF8(0x3094) : "" },
   $cp932_string);

=item C<cp932_to_utf16(STRING)>

=item C<cp932_to_utf16(CODEREF, STRING)>

Converts CP-932 to UTF-16LE.

For example, converts C<\x81\xe0> or C<\x87\x90> to C<U+2252>
in the UTF-16LE encoding.

If C<CODEREF> is not specified, characters that aren't mapped to Unicode
are deleted.
If C<CODEREF> is specified, characters that aren't mapped to Unicode
are converted using C<CODEREF> from the CP-932 character string.

For example, converts C<\x82\xf2> to C<U+3094>, HIRAGANA LETTER VU,
in the UTF-16LE encoding.

 cp932_to_utf16(
   sub { $_[0] eq "\x82\xf2" ? toUTF16(0x3094) : "" },
   $cp932_string);

=item C<utf8_to_cp932(STRING)>

=item C<utf8_to_cp932(CODEREF, STRING)>

Converts UTF-8 to CP-932 (normalized).

For example, C<U+2252> in the UTF-8 encoding is converted
to C<\x81\xe0>, not to C<\x87\x90>.

If C<CODEREF> is not specified, characters that aren't mapped to CP-932
are deleted.
If C<CODEREF> is specified, characters that aren't mapped to CP-932
are converted using C<CODEREF> from its Unicode code point (integer).

For example, characters that aren't mapped to CP-932 are 
converted to numerical character references for HTML 4.01.

  utf8_to_cp932(sub {sprintf "&#x%04x;", shift}, $utf8_string);

=item C<utf16_to_cp932(STRING)>

=item C<utf16_to_cp932(CODEREF, STRING)>

Converts C<UTF-16LE> to C<CP-932> (normalized).

For example, C<U+2252> in the C<UTF-16LE> encoding is converted
to C<\x81\xe0>, not to C<\x87\x90>.

If C<CODEREF> is not specified, characters that aren't mapped to CP-932
are deleted.
If C<CODEREF> is specified, characters that aren't mapped to CP-932
are converted using C<CODEREF> from its Unicode code point (integer).

For example, characters that aren't mapped to CP-932 are 
converted to numerical character references for HTML 4.01.

  utf16_to_cp932(sub {sprintf "&#x%04x;", shift}, $utf16LE_string);

=back

=head2 FUNCTIONS EXPORTED ON REQUEST

=over 4

=item C<toUTF8(CODEPOINT)>

Returns a utf-8 string from codepoint integer.

Valid unicode codepoints are C<0> to C<0x10ffff>,
excepting C<0xD800> to C<0xDFFF>.

If an invalid codepoint is passed in, croaked.

=item C<toUTF16(CODEPOINT)>

Returns a utf-16-le string from codepoint integer.

Valid unicode codepoints are C<0> to C<0x10ffff>,
excepting C<0xD800> to C<0xDFFF>.

If an invalid codepoint is passed in, croaked.

=item C<fromUTF8(STRING)>

Returns the codepoint from a utf-8 string corresponding to B<single character>.

If invalid, croaked.

=item C<fromUTF16(STRING)>

Returns the codepoint from a utf-16-le string corresponding to B<single character>.

If invalid, croaked.

=back

=head1 CAVEAT

This module doesn't understand any logically wide characters (see perlunicode).
Use utf8::decode/utf8::encode (see utf8) on Perl 5.7 or later if necessary.

=head1 AUTHOR

Tomoyuki SADAHIRO

  bqw10602@nifty.com
  http://homepage1.nifty.com/nomenclator/perl/

  Copyright(C) 2001, SADAHIRO Tomoyuki. Japan. All rights reserved.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=over 4

=item 1

Microsoft PRB: Conversion Problem Between Shift-JIS and Unicode
(Article ID: Q170559)

=item 2

Unicode.org

ftp://ftp.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP932.TXT

=item 3

L<ShiftJIS::CP932::Correct>

=back

=cut
