static char acctfile_c_rcsid[]="acctfile.c,v 1.1.1.1 1994/06/18 19:43:30 kerce Exp";

/*-----------------------------------------------------------------------------
 * Kingsley Kerce
 *
 * Copyright (c) 1994
 *
 * Supercomputer Computations Research Institute (SCRI)
 * Florida State University
 *
 * SCRI representatives make no claims about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied
 * warranty.
 *
 * acctfile.c,v
 * Revision 1.1.1.1  1994/06/18  19:43:30  kerce
 * DQS X Distribution
 *
 *---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdio.h>
#include <math.h>
#include "error.h"
#include "strdup.h"
#include "stdstuff.h"
#include "acctfile.h"
#include "request.h"

static BOOLEAN
CalcCPUUsage PROTO((ACCTFILE *AcctFile, dqs_list_type *QComplex,
		    REQUEST *Req));

ACCTFILE *
AcctFileCreate (AcctFileName, IStart, IStop, BinSize, Result)
     char *AcctFileName;
     unsigned long int IStart;
     unsigned long int IStop;
     unsigned int BinSize;
     int *Result;
{
  ACCTFILE *AcctFile;

  DENTER ((DQS_EVENT, "AcctFileCreate"));

  if (IStart < 0)
    {
      *Result = ACCTFILE_SMALLISTART;
      DEXIT;
      return ((ACCTFILE *) NULL);
    }

  if (IStop < IStart)
    {
      *Result = ACCTFILE_SMALLISTOP;
      DEXIT;
      return ((ACCTFILE *) NULL);
    }

  if ((IStop - IStart) < BinSize)
    {
      *Result = ACCTFILE_LARGEBINSIZE;
      DEXIT;
      return ((ACCTFILE *) NULL);
    }

  if (BinSize <= 0)
    {
      *Result = ACCTFILE_SMALLBINSIZE;
      DEXIT;
      return ((ACCTFILE *) NULL);
    }

  AcctFile = (ACCTFILE *) calloc (1, sizeof (ACCTFILE));
  if (AcctFile == (ACCTFILE *) NULL)
    {
      *Result = ACCTFILE_OUTOFMEM;
      DEXIT;
      return ((ACCTFILE *) NULL);
    }

  AcctFile->IStart = IStart;
  AcctFile->IStop = IStop;
  AcctFile->BinSize = BinSize;
  
  AcctFile->AcctFileName = AcctFileName;
  
  AcctFile->DUsage = (dqs_rusage_type *) calloc (1, sizeof (dqs_rusage_type));
  if (AcctFile->DUsage == (dqs_rusage_type *) NULL)
    {
      free (AcctFile);
      *Result = ACCTFILE_OUTOFMEM;
      DEXIT;
      return ((ACCTFILE *) NULL);
    }

  AcctFile->DUsage->qname = (char *) calloc (1, (256 * sizeof (char)));
  AcctFile->DUsage->hostname = (char *) calloc (1, (256 * sizeof (char)));
  AcctFile->DUsage->group = (char *) calloc (1, (256 * sizeof (char)));
  AcctFile->DUsage->owner = (char *) calloc (1, (256 * sizeof (char)));
  AcctFile->DUsage->job_name = (char *) calloc (1, (256 * sizeof (char)));
  AcctFile->DUsage->dqs_job_name = (char *) calloc (1, (256 * sizeof (char)));
  AcctFile->DUsage->account = (char *) calloc (1, (256 * sizeof (char)));  
  if ((AcctFile->DUsage->qname == (char *) NULL)
      || (AcctFile->DUsage->hostname == (char *) NULL)
      || (AcctFile->DUsage->group == (char *) NULL)
      || (AcctFile->DUsage->owner == (char *) NULL)
      || (AcctFile->DUsage->job_name == (char *) NULL)
      || (AcctFile->DUsage->dqs_job_name == (char *) NULL)
      || (AcctFile->DUsage->account == (char *) NULL))
    {
      free (AcctFile);
      *Result = ACCTFILE_OUTOFMEM;
      DEXIT;
      return ((ACCTFILE *) NULL);
    }

  {
    int statResult;

    statResult = stat (AcctFile->AcctFileName, &AcctFile->FileStat);
    if (statResult != 0)
      {
	free (AcctFile);
	*Result = ACCTFILE_NOSTAT;
	DEXIT;
	return ((ACCTFILE *) NULL);
      }
  }
  
  AcctFile->AcctFilePtr = fopen (AcctFile->AcctFileName, "r");
  if (AcctFile->AcctFilePtr == (FILE *) NULL)
    {
      free (AcctFile);
      *Result = ACCTFILE_NOFOPEN;
      DEXIT;
      return ((ACCTFILE *) NULL);
    }
  
  AcctFile->CPU.NumPoints =
    (AcctFile->IStop - AcctFile->IStart) / AcctFile->BinSize;
  if (((AcctFile->IStop - AcctFile->IStart) % AcctFile->BinSize) != 0)
    AcctFile->CPU.NumPoints++;

  AcctFile->CPU.UserDataPoints =
    (double *) calloc (1, AcctFile->CPU.NumPoints * sizeof (double));
  if (AcctFile->CPU.UserDataPoints == NULL)
    {
      free (AcctFile);
      fclose (AcctFile->AcctFilePtr);
      *Result = ACCTFILE_OUTOFMEM;
      DEXIT;
      return ((ACCTFILE *) NULL);
    }
  
  AcctFile->CPU.SysDataPoints =
    (double *) calloc (1, AcctFile->CPU.NumPoints * sizeof (double));
  if (AcctFile->CPU.SysDataPoints == NULL)
    {
      free (AcctFile->CPU.UserDataPoints);
      free (AcctFile);
      fclose (AcctFile->AcctFilePtr);
      *Result = ACCTFILE_OUTOFMEM;
      DEXIT;
      return ((ACCTFILE *) NULL);
    }

  AcctFile->Wallclock.NumPoints = AcctFile->CPU.NumPoints;

  AcctFile->Wallclock.DataPoints =
    (double *) calloc (1, AcctFile->Wallclock.NumPoints * sizeof (double));
  if (AcctFile->Wallclock.DataPoints == NULL)
    {
      free (AcctFile->CPU.UserDataPoints);
      free (AcctFile->CPU.SysDataPoints);
      free (AcctFile);
      fclose (AcctFile->AcctFilePtr);
      *Result = ACCTFILE_OUTOFMEM;
      DEXIT;
      return ((ACCTFILE *) NULL);
    }
  
  {
    unsigned long i;
  
    for (i = 0; i < AcctFile->CPU.NumPoints; i++)
      {
	AcctFile->Wallclock.DataPoints[i] = 0.0;
	AcctFile->CPU.UserDataPoints[i] = 0.0;
	AcctFile->CPU.SysDataPoints[i] = 0.0;
      }
  }
  
  AcctFile->TotalLinesRead = 0;
  AcctFile->PercentageRead = 0;
  AcctFile->Matches = 0;

  *Result = ACCTFILE_SUCCESS;
  DEXIT;
  return ((ACCTFILE *) AcctFile);
}

void
AcctFileDestroy (AcctFile)
     ACCTFILE *AcctFile;
{
  DENTER ((DQS_EVENT, "AcctFileDestroy"));

  free (AcctFile->DUsage->qname);
  free (AcctFile->DUsage->hostname);
  free (AcctFile->DUsage->group);
  free (AcctFile->DUsage->owner);
  free (AcctFile->DUsage->job_name);
  free (AcctFile->DUsage->dqs_job_name);
  free (AcctFile->DUsage);

  free (AcctFile->Wallclock.DataPoints);
  free (AcctFile->CPU.UserDataPoints);
  free (AcctFile->CPU.SysDataPoints);
  free (AcctFile);
  AcctFile = (ACCTFILE *) NULL;

  DEXIT;
}

static BOOLEAN
Match (DUsage, QComplex, Req)
     dqs_rusage_type *DUsage;
     dqs_list_type *QComplex;
     REQUEST *Req;
{
  BOOLEAN QCMatch, QNameMatch, HostNameMatch, OwnerMatch, GroupMatch;
  BOOLEAN JobNameMatch;

  DENTER ((DQS_EVENT, "Match"));

  QCMatch = FALSE;
  QNameMatch = FALSE;
  HostNameMatch = FALSE;
  OwnerMatch = FALSE;
  GroupMatch = FALSE;
  JobNameMatch = FALSE;

  if (Req->QComplex != NULL)
    /* Check for a queue complex match. */
    {
      int Suitable, NumSuitable;
      dqs_list_type *QC, *ReqQC;
      
      QC = QComplex->chain;
      ReqQC = Req->QComplex->chain;
      
      NumSuitable = 0;
      
      while (ReqQC != NULL)
	{
	  while (QC != NULL)
	    {
	      Suitable = dqs_complex_el_suitable (ReqQC, QC);
	      if (Suitable == 1)
		NumSuitable++;
	      QC = QC->next;
	    }
	  
	  ReqQC = ReqQC->next;
	}
      
      if (NumSuitable == Req->QComplex->int1)
	QCMatch = TRUE;
      else
	QCMatch = FALSE;
    }

  if (dqs_wildmat (Req->QName, DUsage->qname) == TRUE)
    QNameMatch = TRUE;
    
  if (dqs_wildmat (Req->HostName, DUsage->hostname) == TRUE)
    HostNameMatch = TRUE;
  
  if (dqs_wildmat (Req->Owner, DUsage->owner) == TRUE)
    OwnerMatch = TRUE;

  if (dqs_wildmat (Req->Group, DUsage->group) == TRUE)
    GroupMatch = TRUE;

  if (dqs_wildmat (Req->JobName, DUsage->job_name) == TRUE)
    JobNameMatch = TRUE;

  if ((QCMatch == TRUE) || (QNameMatch == TRUE) || (HostNameMatch == TRUE)
      || (OwnerMatch == TRUE) || (GroupMatch == TRUE)
      || (JobNameMatch == TRUE))
    {
      DEXIT;
      return (TRUE);
    }
  else
    {
      DEXIT;
      return (FALSE);
    }
}

static BOOLEAN
CalcCPUUsage (AcctFile, QComplex, Req)
     ACCTFILE *AcctFile;
     dqs_list_type *QComplex;
     REQUEST *Req;
{
  unsigned long JStart, JStop, IStart, IStop, StartBin, NextBinEnd;
  double CPUAvgUser, CPUAvgSys;
  dqs_rusage_type *DUsage;
  
  DENTER ((DQS_EVENT, "CalcCPUUsage"));

  DUsage = AcctFile->DUsage;

  if (DUsage->ru_utime > DUsage->ru_wallclock)
    {
      Error (NULL, 0, 0,
	     "WARNING: %s: line %lu: user time (%lu) > wall clock time (%lu)",
	     AcctFile->AcctFileName, AcctFile->TotalLinesRead,
	     DUsage->ru_utime, DUsage->ru_wallclock);
    }
  
  if (DUsage->ru_stime > DUsage->ru_wallclock)
    {
      Error (NULL, 0, 0,
	     "WARNING: %s: line %lu: sys time (%lu) > wall clock time (%lu)",
	     AcctFile->AcctFileName, AcctFile->TotalLinesRead,
	     DUsage->ru_stime, DUsage->ru_wallclock);
    }
  
  /* Interval's start and stop times. */
  IStart = AcctFile->IStart;
  IStop = AcctFile->IStop;

  /* Job's start and stop times. */
  JStart = DUsage->start_time;
  JStop = DUsage->end_time;
  
  if (((JStart <= IStart) && (JStop <= IStart))
      || ((JStart >= IStop) && (JStop >= IStop)))
    {
      DEXIT;
      return (FALSE);
    }

  if (Match (DUsage, QComplex, Req) == FALSE)
    {
      DEXIT;
      return (FALSE);
    }
  
  /*
   * Adjust job start and stop times if either falls outside
   * the interval.
   */
  if (JStart < IStart)
    JStart = IStart;
  
  if (JStop > IStop)
    JStop = IStop + 1;
  
  /* Calc this job's average user and system times. */
  if (DUsage->ru_wallclock > 0.0)
    {
      CPUAvgUser = (double) DUsage->ru_utime / (double) DUsage->ru_wallclock;
      CPUAvgSys = (double) DUsage->ru_stime / (double) DUsage->ru_wallclock;
    }
  else
    {
      CPUAvgUser = 0.0;
      CPUAvgSys = 0.0;
    }
  
  StartBin = (JStart - IStart) / AcctFile->BinSize;
  NextBinEnd = IStart + (StartBin * AcctFile->BinSize);
  
  /*
   * If job begins inside a bin then take care of that job's
   * usage up to the start of the next bin.
   */
  if (((JStart - IStart) % AcctFile->BinSize) != 0)
    {
      NextBinEnd = IStart + ((StartBin + 1) * AcctFile->BinSize);
      if (NextBinEnd > JStop)
	NextBinEnd = JStop;
      AcctFile->Wallclock.DataPoints[StartBin] += NextBinEnd - JStart;
      AcctFile->CPU.UserDataPoints[StartBin] +=
	(NextBinEnd - JStart) * CPUAvgUser;
      AcctFile->CPU.SysDataPoints[StartBin] +=
	(NextBinEnd - JStart) * CPUAvgSys;
    }	
  
  /* NextBinEnd now points to the start of the next bin. */
  if (NextBinEnd != JStop)
    {
      unsigned long UsageStart, UsageStop, Usage;
      
      UsageStart = (NextBinEnd - IStart) / AcctFile->BinSize;
      UsageStop = (JStop - IStart) / AcctFile->BinSize;
      for (Usage = UsageStart; Usage < UsageStop; Usage++)
	{
	  AcctFile->Wallclock.DataPoints[Usage] += AcctFile->BinSize;
	  AcctFile->CPU.UserDataPoints[Usage] +=
	    AcctFile->BinSize * CPUAvgUser;
	  AcctFile->CPU.SysDataPoints[Usage] +=
	    AcctFile->BinSize * CPUAvgSys;
	}
      
      /*
       * If job ends inside a bin then take care of that job's
       * usage from the end of the previous bin.
       */
      if (((JStop - IStart) % AcctFile->BinSize) != 0)
	{
	  unsigned long PrevBinEnd;
	  
	  PrevBinEnd = IStart + (UsageStop * AcctFile->BinSize);
	  AcctFile->Wallclock.DataPoints[UsageStop] += JStop - PrevBinEnd;
	  AcctFile->CPU.UserDataPoints[UsageStop] +=
	    (JStop - PrevBinEnd) * CPUAvgUser;
	  AcctFile->CPU.SysDataPoints[UsageStop] +=
	    (JStop - PrevBinEnd) * CPUAvgSys;
	}
    }

  DEXIT;
  return (TRUE);
}

int
AcctFileRead (AcctFile, Req, LinesToRead)
     ACCTFILE *AcctFile;
     REQUEST *Req;
     int LinesToRead;
{
  int read_dusageResult;
  string QComplexStr;
  dqs_list_type *QComplex;
  unsigned int LinesRead;
  BOOLEAN AnyMatched;

  DENTER ((DQS_EVENT, "AcctFileRead"));

  assert (AcctFile->AcctFilePtr != (FILE *) NULL);

  read_dusageResult = 0;
  LinesRead = 0;

  while ((read_dusageResult == 0)
	 && (((LinesRead < LinesToRead)) || (LinesToRead == 0)))
    {
      read_dusageResult =
	dqs_read_rusage (AcctFile->AcctFilePtr, AcctFile->DUsage, QComplexStr);

      switch (read_dusageResult)
	{
	case -3:
	  /* The file should be open... */
	  Error (__FILE__, __LINE__, 0, "FATAL INTERNAL ERROR");
	  DEXIT;
	  exit (EXIT_FAILURE);
	  break;
	  
	case -2:
	  DEXIT;
	  return (-2);
	  break;

	case -1:
	  DEXIT;
	  return (-1);
	  break;

	case 0:
	  AcctFile->TotalLinesRead++;
	  LinesRead++;
	  
	  AcctFile->PercentageRead =
	    floor ((double) ftell (AcctFile->AcctFilePtr)
		   / AcctFile->FileStat.st_size * 100.0);
	  
	  QComplex = NULL;
	  dqs_parse_resource_list (&QComplex, QComplexStr);

	  /* The following is used to test dqs_build_qcomplex_str(). */
/*
	  {
	    char *QCStr;

	    puts (QCStr = dqs_build_qcomplex_str (QComplex));
	    free (QCStr);
	  }
*/
	  AnyMatched = CalcCPUUsage (AcctFile, QComplex, Req);
	  if (AnyMatched == TRUE)
	    AcctFile->Matches++;

	  dqs_free_list (QComplex);
	  break;

	case 1:
	  if (AcctFile->Matches > 0)
	    {
	      unsigned long i;
	    
	      for (i = 0; i < AcctFile->CPU.NumPoints; i++)
		{
		  AcctFile->Wallclock.DataPoints[i] /= AcctFile->BinSize;
		  AcctFile->CPU.UserDataPoints[i] /= AcctFile->BinSize;
		  AcctFile->CPU.SysDataPoints[i] /= AcctFile->BinSize;
		}
	    }

	  fclose (AcctFile->AcctFilePtr);

	  DEXIT;
	  return (1);
	  break;

	default:
	  Error (__FILE__, __LINE__, 0, "FATAL INTERNAL ERROR");
	  DEXIT;
	  exit (EXIT_FAILURE);
	  break;
	}
    }
  
  DEXIT;
  return (0);
}


/* Set Emacs C-mode style.
   Local Variables:
   c-style: GNU
   End:
 */
