/*
 *  linux/fs/supermount/super.c
 *
 *  Original version:
 *      Copyright (C) 1995, 1997
 *      Stephen Tweedie (sct@@dcs.ed.ac.uk)
 *
 *      from
 *
 *      linux/fs/minix/inode.c
 *      Copyright (C) 1991, 1992  Linus Torvalds
 *
 *      and
 *
 *      linux/fs/ext2/super.c
 *      Copyright (C) 1992, 1993, 1994, 1995  Remy Card
 *
 *  Rewriten for kernel 2.2, 2.4. (C) 1999, 2000 Alexis Mikhailov
 *                                    (alexis@@abc.cap.ru)
 *  Rewritten for kernel 2.4.21 (C) 2003 Andrey Borzenkov
 *                                       (arvidjaar@@mail.ru)
 *
 *  $Id: super.c,v 1.28 2003/12/29 19:28:04 bor Exp $
 */

#define S_DBG_TRACE_CURRENT S_DBG_TRACE_SUPER
#include "supermount.h"

/*
 * vfat and msdos do not have any valididty checks for superblock and must
 * be the last entries!
 */
static char *default_fs_types = "udf:iso9660:ext2:vfat:msdos";

static struct super_operations supermount_sops;

/* ========================= helpers ================================= */

static inline void
supermount_update_inode(struct inode *superi, struct inode *subi)
{
	struct super_block *sb = superi->i_sb;

	ENTER(sb, "superi=%p subi=%p", superi, subi);

	superi->i_ino = subi->i_ino;
	superi->i_mode = subi->i_mode;
	superi->i_uid = subi->i_uid;
	superi->i_gid = subi->i_gid;
	superi->i_nlink = subi->i_nlink;
	superi->i_size = subi->i_size;
	superi->i_atime = subi->i_atime;
	superi->i_ctime = subi->i_ctime;
	superi->i_mtime = subi->i_mtime;
	superi->i_blksize = subi->i_blksize;
	superi->i_blocks = subi->i_blocks;
	superi->i_rdev = subi->i_rdev;
	superi->i_version++;
	set_inode_flags(superi, subi);

	if (S_ISDIR(superi->i_mode)) {
		superi->i_op = &supermount_dir_iops;
		superi->i_fop = &supermount_dir_operations;
	} else if (S_ISLNK(superi->i_mode)) {
		superi->i_op = &supermount_symlink_iops;
		superi->i_mapping = subi->i_mapping;
	} else {
		superi->i_op = &supermount_file_iops;
		superi->i_fop = &supermount_file_operations;
		superi->i_mapping = subi->i_mapping;
	}

	LEAVE(sb, "superi=%p subi=%p", superi, subi);
}

/* this is also called from subfs_mount to reinstantiate root inode */
void
attach_subfs_inode(struct inode *inode, struct inode *subi)
{
	struct super_block *sb = inode->i_sb;
	struct supermount_sb_info *sbi = supermount_sbi(sb);
	struct supermount_inode_info *sii;

	ENTER(sb, "inode=%p subi=%p", inode, subi);

	sii = supermount_i(inode);
	if (!sii->inode) {
		/*
		 * this can be run concurrently. It is executed under
		 * sbi->sem so only one task would actually instantate
		 * inode. See namei.c:prepare_inode
		 * Another user is subfs.c:subfs_real_mount2
		 */
		sii->inode = igrab(subi);
		supermount_update_inode(inode, subi);
		list_add(&sii->list, &sbi->s_inodes);
	} else
		SUPERMOUNT_BUG_LOCKED_ON(sb, sii->inode != subi);

	LEAVE(sb, "inode=%p subi=%p", inode, subi);
}

static int
init_inode_info(struct inode *inode)
{
	struct super_block *sb = inode->i_sb;
	struct supermount_inode_info *sii;
	int rc = 0;

	ENTER(sb, "inode=%p", inode);

	sii = kmalloc(sizeof(*sii), GFP_KERNEL);
	if (!sii) {
		/*
		 * we can't just destroy_inode here because then pointer
		 * to garbage will be returned. Instead we check for bad
		 * inode in namei.c:prepare_inode
		 */
		make_bad_inode(inode);
		rc = 1;
		goto out;
	}

	memset(sii, 0, sizeof(*sii));
	INIT_LIST_HEAD(&sii->list);
	sii->host = inode;
	sii->inode = 0;
	inode->u.generic_ip = sii;

out:
	LEAVE(sb, "inode=%p rc=%d", inode, rc);

	return rc;
}

char *
strdup(const char *val)
{
	char *tmp;
	tmp = kmalloc(1 + strlen(val), GFP_KERNEL);
	if (tmp)
		strcpy(tmp, val);
	return tmp;
}


static struct supermount_sb_info *
create_sbi(struct super_block *sb)
{
	struct supermount_sb_info *sbi = kmalloc(sizeof (*sbi), GFP_KERNEL);

	if (!sbi)
		return NULL;

	memset(sbi, 0, sizeof (*sbi));

	sbi->s_undermount = NULL;
	sbi->s_type = sbi->devname = sbi->s_data = NULL;
	INIT_LIST_HEAD(&sbi->s_inodes);
	INIT_LIST_HEAD(&sbi->s_files);
	INIT_LIST_HEAD(&sbi->s_dentries);
	init_MUTEX(&sbi->sem);
	sbi->host = sb;
	sbi->tray_lock = TRAY_LOCK_ONWRITE;
	sbi->rw = !(sb->s_flags & MS_RDONLY);

	return sbi;
}

static void
free_sbi(struct supermount_sb_info *sbi)
{
	if (sbi->s_type && sbi->s_type != default_fs_types)
		kfree(sbi->s_type);
	if (sbi->devname)
		kfree(sbi->devname);
	if (sbi->s_data)
		kfree(sbi->s_data);
	kfree(sbi);
}

static int
parse_options(char *options, struct super_block *sb)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);
	char *this_char;
	char *value;

	if (!options)
		return 0;

	while ((this_char = options)) {
		if (!strncmp(this_char, "--", 2))
			this_char += 2;
		if (!*this_char)
			break;
		if (*this_char == ',') {
			/* An empty option, or the option "--",
			   introduces options to be passed through to
			   the subfs */
			sbi->s_data = strdup(++this_char);
			if (!sbi->s_data)
				return -ENOMEM;
			return 0;
		}
		if ((options = strchr(this_char, ',')))
			*options++ = 0;

		if ((value = strchr(this_char, '=')))
			*value++ = 0;

		if (!strcmp(this_char, "fs")) {
			if (!value || !*value)
				return -EINVAL;
			
			if (!strcmp(value, "auto")) {
				sbi->s_type = default_fs_types;
				continue;
			}

			sbi->s_type = strdup(value);
			if (!sbi->s_type)
				return -ENOMEM;
		} else if (!strcmp(this_char, "dev")) {
			sbi->devname = strdup(value);
			if (!sbi->devname)
				return -ENOMEM;
		} else if (!strcmp(this_char, "debug")) {
			if (value)
				sbi->s_debug = simple_strtoul(value, 0, 0);
			else
				sbi->s_debug = S_DBG_DEBUG;
		} else if (!strcmp(this_char, "tray_lock")) {
			if (!value || !*value)
				return -EINVAL;
			
			if (!strcmp(value, "always"))
				sbi->tray_lock = TRAY_LOCK_ALWAYS;
			else if (!strcmp(value, "onwrite"))
				sbi->tray_lock = TRAY_LOCK_ONWRITE;
			else if (!strcmp(value, "never"))
				sbi->tray_lock = TRAY_LOCK_NEVER;
			else
				return -EINVAL;
		} else if (!strcmp(this_char, "no_tray_lock")) {
			/* legacy */
			sbi->tray_lock = TRAY_LOCK_NEVER;
		} else if (!strcmp(this_char, "fail_statfs_until_mount")) {
			do { /* legacy */ } while (0);
		} else {
			supermount_error(sb, "Unrecognized mount option \"%s\"",
					 this_char);
			return -EINVAL;
		}
	}
	return 0;
}

void
supermount_init_root_inode(struct inode *inode)
{
	inode->i_mode = 0777 | S_IFDIR;
	inode->i_uid = current->fsuid;
	inode->i_gid = current->fsgid;
	inode->i_blksize = inode->i_sb->s_blocksize;
	inode->i_rdev = NODEV;
	inode->i_atime = inode->i_mtime = inode->i_ctime = CURRENT_TIME;
	inode->i_size = 0;
	inode->i_blocks = 0;
	inode->i_ino = 0;
	inode->i_nlink = 0;

	set_inode_flags(inode, 0);

	inode->i_op = &supermount_dir_iops;
	inode->i_fop = &supermount_dir_operations;
}

static struct inode *
supermount_root_inode(struct super_block *sb)
{
	struct inode *inode = new_inode(sb);

	if (!inode)
		return NULL;

	if (init_inode_info(inode)) {
		make_bad_inode(inode);
		iput(inode);
		return NULL;
	}

	inode->i_version = 0;

	supermount_init_root_inode(inode);

	return inode;
}

/* ========================== used in DECLARE_FSTYPE ================*/

/* read_super: the main mount() entry point into the VFS layer. */
struct super_block *
supermount_read_super(struct super_block *sb, void *data, int silent)
{
	struct inode *root_inode;
	struct dentry *root;
	struct supermount_sb_info *sbi = create_sbi(sb);

	if (!sbi)
		goto fail_no_memory;

	sb->u.generic_sbp = sbi;

	if (parse_options((char *) data, sb))
		goto fail_parsing;

	if (!sbi->devname) {
		supermount_error(sb, "no dev=<device> option");
		goto fail_parsing;
	}

	if (!sbi->s_type)
		sbi->s_type = default_fs_types;

	sb->s_blocksize = 1024;
	sb->s_blocksize_bits = 10;
	sb->s_magic = SUPERMOUNT_SUPER_MAGIC;
	sb->s_op = &supermount_sops;

	root_inode = supermount_root_inode(sb);
	if (!root_inode)
		goto fail_allocating_root_inode;

	root = d_alloc_root(root_inode);
	if (!root)
		goto fail_allocating_root_dentry;
	if (init_dentry_info(root))
		goto fail_init_root_info;

	sb->s_root = root;

	supermount_proc_insert(sbi);

	return sb;

fail_init_root_info:
	dput(root);
fail_allocating_root_dentry:
	iput(root_inode);
fail_parsing:
fail_allocating_root_inode:
	free_sbi(sbi);

fail_no_memory:
	return NULL;

}

/* ======================= super_operations methods ==================== */

/*
 * we never need to create inode except for the special root inode case.
 * Supermount inode is always requested after it is known that subfs inode
 * exists.
 */
static void
supermount_read_inode(struct inode *inode)
{
	struct super_block *sb = inode->i_sb;

	ENTER(sb, "inode=%p", inode);

	init_inode_info(inode);

	LEAVE(sb, "inode=%p", inode);

}

/*
 * FIXME
 * I am still unsure if (or why) this functions is needed; it is likely
 * to go away. So far the only _real_ user seems to be knfsd.
 */
static void
supermount_write_inode(struct inode *inode, int sync)
{
	struct super_block *sb = inode->i_sb;
	struct super_block *subsb;
	struct inode *subi;
	struct vfsmount *submnt;

	ENTER(sb);

	if (subfs_check_disk_change(sb))
		goto out;

	submnt = subfs_prevent_umount(sb);
	if (!submnt)
		goto out;

	subi = get_subfs_inode(inode);
	if (IS_ERR(subi))
		goto allow_umount;

	subsb = subfs_get_sb(sb);
	if (!sb)
		goto put_subi;

	if (subsb->s_op && subsb->s_op->write_inode) {
		if (subfs_get_access(inode, 1))
			goto put_subi;

		write_inode_now(subi, sync);

		subfs_put_access(inode, 1);
	}
put_subi:
	iput(subi);
allow_umount:
	subfs_allow_umount(sb, submnt);
out:
	LEAVE(sb);

	return;
}

/*
 * FIXME
 * subfs_umount has no business here
 * it should be moved into umount_begin; in this case force umount will
 * free subfs (and give false No medium as error ... oh, well)
 */
static void
supermount_put_super(struct super_block *sb)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);

	ENTER(sb);

	subfs_lock(sb);
	if (subfs_is_mounted(sb))
		subfs_umount(sb, SUBFS_UMNT_NORMAL);
	subfs_unlock(sb);

	sb->s_dev = NODEV;

	supermount_proc_remove(sbi);

	LEAVE(sb);

	sb->u.generic_sbp = 0;
	free_sbi(sbi);
}

/*
 * The following has to be done in two steps (unfortunately)
 * First make sure iput(subi) is run just once - and then put
 * read/write access and finally remove inode from list
 *
 * We can't do it in one shot because removing inode from list
 * will remove also possibility to mark it stale. It has been 
 * introduced by me :( removing generation numbers. OTOH so far
 * it is the only place that needs extra treatment because of it.
 */
static void
supermount_clear_inode(struct inode *inode)
{
	struct supermount_inode_info *sii = supermount_i(inode);
	struct super_block *sb = inode->i_sb;
	struct inode *subi;
	struct vfsmount *mnt;

	ENTER(sb, "inode=%p", inode);

	/* it is possible for special case of failed memory allocation */
	if (!sii)
		goto out;
	
	mnt = subfs_prevent_umount(sb);
	if (!mnt)
		goto free_sii;

	subfs_lock(sb);
	subi = sii->inode;
	sii->inode = 0;
	subfs_unlock(sb);


	if (subi) {
		/*
		 * we used to check for subi->i_count != 1 here.
		 * This does not actually work - it is quite possible
		 * that inode has been looked up while we were in this
		 * function. So just ignore it (the worst thing that
		 * may happen is that subfs cannot be remounted)
		 */

		iput(subi);

	}
	subfs_allow_umount(sb, mnt);

	/*
	 * This does not has much to do with subfs but I did not want
	 * to export __ functions
	 */
	subfs_clear_inode(inode);

free_sii:
	kfree(sii);

out:
	LEAVE(sb, "inode=%p", inode);

}
/*
 * FIXME why it is needed?
 */

static void
supermount_write_super(struct super_block *sb)
{
	struct vfsmount *mnt;

	ENTER(sb);

	if (subfs_check_disk_change(sb))
		goto out;

	if (subfs_is_rw(sb)) {
		struct super_block *subsb;

		mnt = subfs_prevent_umount(sb);
		if (!mnt)
			goto out;

		subsb = subfs_get_sb(sb);
		if (subsb && subsb->s_op && subsb->s_op->write_super)
			subsb->s_op->write_super(subsb);

		subfs_allow_umount(sb, mnt);
	}

out:
	LEAVE(sb);

	return;
}

static int
supermount_statfs(struct super_block *sb, struct statfs *buf)
{
	int rc = 0;
	struct super_block *subsb;
	struct vfsmount *mnt;

	ENTER(sb);

	(void)subfs_check_disk_change(sb);

	mnt = subfs_prevent_umount(sb);
	if (!mnt)
		goto out;

	subsb = subfs_get_sb(sb);
	if (subsb && subsb->s_op && subsb->s_op->statfs)
		rc = subsb->s_op->statfs(subsb, buf);

	subfs_allow_umount(sb, mnt);
out:
	buf->f_type = SUPERMOUNT_SUPER_MAGIC;

	LEAVE(sb, "rc=%d", rc);

	return rc;
}

static int
supermount_remount_fs(struct super_block *sb, int *flags, char *data)
{
	return -ENOSYS;
}

/*
 * based on fs/ntfs/inode.c:ntfs_show_options
 */
static int
supermount_show_options(struct seq_file *sf, struct vfsmount *mnt)
{
	struct supermount_sb_info *sbi = supermount_sbi(mnt->mnt_sb);

	seq_printf(sf, ",dev=%s", sbi->devname);
	seq_puts(sf, ",fs=");
	if (sbi->s_type == default_fs_types)
		seq_puts(sf, "auto");
	else 
		seq_puts(sf, sbi->s_type);
	seq_puts(sf, ",tray_lock=");
	if (sbi->tray_lock == TRAY_LOCK_ALWAYS)
		seq_puts(sf, "always");
	else if (sbi->tray_lock == TRAY_LOCK_NEVER)
		seq_puts(sf, "never");
	else if (sbi->tray_lock == TRAY_LOCK_ONWRITE)
		seq_puts(sf, "onwrite");
	else
		SUPERMOUNT_BUG_ON(1);

	if (sbi->s_debug)
		seq_printf(sf, ",debug=0x%lx", sbi->s_debug);
	if (sbi->s_data)
		seq_printf(sf, ",--,%s", sbi->s_data);
	return 0;
}

/*
 * ->alloc_inode:	no needed, we do not provide our own memory
 *  			management
 * ->destroy_inode:	TODO may be for removing inode_info
 * ->read_inode2:	not needed
 * ->dirty_inode:	probably not needed. It appears VFS layer never
 *  			calls mark_inode_dirty itself; exception is
 *  			UPDATE_ATIME and it is already handled specially
 *  			FIXME
 *  			It may be needed for knfsd and similar
 * ->put_inode:		not needed
 * ->delete_inode:	not needed
 * ->write_super:	FIXME not needed, we do not have any backing store
 * ->sync_fs:		not needed
 * ->write_super_lockfs:not needed
 * ->unlockfs:		not needed
 * ->umount_begin:	TODO maybe to move subfs_umount aways from put_super
 * ->fh_to_dentry:	TODO may be
 * ->dentry_to_fh:	TODO may be
 */
static struct super_operations supermount_sops = {
	.read_inode	= supermount_read_inode,
	.write_inode	= supermount_write_inode,
	.clear_inode	= supermount_clear_inode,
	.put_super	= supermount_put_super,
	.write_super	= supermount_write_super,
	.statfs		= supermount_statfs,
	.remount_fs	= supermount_remount_fs,
	.show_options	= supermount_show_options,
};
