#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/locks.h>
#include <linux/slab.h>

#include <asm/system.h>
#include <asm/uaccess.h>
#include <linux/smp_lock.h>

#include "ftpfs.h"
#include "ftpfs_proc.h"

MODULE_AUTHOR("Florin Malita <fmalita@yahoo.com>");
MODULE_DESCRIPTION("FTP File System");

extern struct inode_operations ftp_dir_inode_operations;
extern struct file_operations ftp_dir_operations;
extern struct inode_operations ftp_file_inode_operations;
extern struct file_operations ftp_file_operations;
extern struct address_space_operations ftp_file_aops;
extern struct inode_operations ftp_symlink_inode_operations;

static void ftp_delete_inode(struct inode*);
static void ftp_put_super(struct super_block*);
static int  ftp_statfs(struct super_block*, struct statfs*);
static void ftp_set_inode_attr(struct inode*, struct ftp_fattr*);

static struct super_operations ftp_sops = {
	put_inode:	force_delete,
	delete_inode:	ftp_delete_inode,
	put_super:	ftp_put_super,
	statfs:		ftp_statfs,
};

struct inode*
ftp_iget(struct super_block *sb, struct ftp_fattr *fattr){
	struct inode *res;

	DEBUG("\n");

	res = new_inode(sb);
	if(!res)
		return NULL;
	res->i_ino = fattr->f_ino;
	ftp_set_inode_attr(res, fattr);

	if(S_ISDIR(res->i_mode)){
		DEBUG(" It's a directory\n");
		res->i_op = &ftp_dir_inode_operations;
		res->i_fop = &ftp_dir_operations;;
	}else if(S_ISLNK(res->i_mode)){
		DEBUG(" It's a link\n");
		res->i_op = &ftp_symlink_inode_operations;
	}else{
		DEBUG(" It's a file\n");
		res->i_op = &ftp_file_inode_operations;
		res->i_fop = &ftp_file_operations;
		res->i_data.a_ops = &ftp_file_aops;
	}
	
	insert_inode_hash(res);
	return res;
}

static void 
ftp_set_inode_attr(struct inode *inode, struct ftp_fattr *fattr){
	inode->i_mode 	= fattr->f_mode;
	inode->i_nlink	= fattr->f_nlink;
	inode->i_uid	= fattr->f_uid;
	inode->i_gid	= fattr->f_gid;
	inode->i_rdev	= fattr->f_rdev;
	inode->i_ctime	= fattr->f_ctime;
	inode->i_atime	= fattr->f_atime;
	inode->i_mtime	= fattr->f_mtime;
	inode->i_blksize= fattr->f_blksize;
	inode->i_blocks	= fattr->f_blocks;
	inode->i_size	= fattr->f_size;
}


static void 
ftp_delete_inode(struct inode *inode){
	lock_kernel();
		
	clear_inode(inode);
	unlock_kernel();
}

static void
ftp_put_super(struct super_block *sb){
	struct ftp_sb_info *info = (struct ftp_sb_info*)sb->u.generic_sbp;

	ftp_disconnect(info);
	ftp_cache_empty(info);
	kfree(info);
	VERBOSE("Super block discarded!\n");
}

static int
ftp_statfs(struct super_block *sb, struct statfs *attr){

	DEBUG("\n");

	attr->f_type = FTP_SUPER_MAGIC;
	attr->f_bsize = 1024;
	attr->f_blocks = 0;
	attr->f_namelen = FTP_MAXPATHLEN;
	attr->f_files = -1;
	attr->f_bavail = -1;
	
	return 0;
}

struct super_block*
ftp_read_super(struct super_block *sb, void *opts, int silent){
	struct ftp_sb_info *info;
	struct ftp_fattr root;
	struct inode *root_inode;
	
	info = (struct ftp_sb_info*)kmalloc(sizeof(struct ftp_sb_info), GFP_KERNEL);
	if(!info){
		VERBOSE("Not enough kmem to allocate info!!\n");
		goto out;
	}

	
	memset(info, 0, sizeof(struct ftp_sb_info));

	sb->u.generic_sbp = info;
	sb->s_blocksize = 1024;
	sb->s_blocksize_bits = 10;
	sb->s_magic = FTP_SUPER_MAGIC;
	sb->s_op = &ftp_sops;
	sb->s_flags = 0;
	
	info->mnt.version = FTP_VERSION;
	info->mnt.active = 0;
	info->mnt.file_mode = (S_IRUSR | S_IFREG);
	info->mnt.dir_mode = (S_IRUSR | S_IXUSR | S_IFDIR);
	info->mnt.uid = current->uid;
	info->mnt.gid = current->gid;
	info->ctrl_sock = NULL;
	info->data_sock = NULL;
	init_MUTEX(&info->sem);
	
	info->mnt.mount_point[0] = 0;
	
	ftp_cache_init(info);
	
	if(ftp_parse_options(info, opts) < 0){
		VERBOSE("Invalid options!\n");
		goto out_no_opts;
	}
	
	DEBUG("Mounting %u.%u.%u.%u:%u, user %s, password %s\n",
		info->address.sin_addr.s_addr & 0xff,
		(info->address.sin_addr.s_addr >> 8) & 0xff,
		(info->address.sin_addr.s_addr >> 16) & 0xff,
		(info->address.sin_addr.s_addr >> 24) & 0xff,
		ntohs(info->address.sin_port), info->user, info->pass);

	if(ftp_connect(info) < 0){
		VERBOSE("Could not connect!\n");
		goto out_no_opts;
	}
	
	ftp_init_root_dirent(info, &root);
	root_inode = ftp_iget(sb, &root);
	if(!root_inode) 
		goto out_no_root;
	sb->s_root = d_alloc_root(root_inode);
	if(!sb->s_root) 
		goto out_no_root;

	DEBUG(" Mount succeded!\n");
	return sb;

out_no_root:
	iput(root_inode);
out_no_opts:
	kfree(info);
out:
	DEBUG("Mount failed!!\n");
	return NULL;
}

static DECLARE_FSTYPE (ftp_fs_type, "ftpfs", ftp_read_super, 0);

static int __init 
init_ftp_fs(void){
	VERBOSE("FTP File System\n");
	VERBOSE("Copyright (c) 2001, Florin Malita\n");
	return register_filesystem(&ftp_fs_type);
}

static void __exit 
exit_ftp_fs(void){
	VERBOSE("Unregistering FTPfs.\n");
	unregister_filesystem(&ftp_fs_type);
}

EXPORT_NO_SYMBOLS;

module_init(init_ftp_fs);
module_exit(exit_ftp_fs);

