/***************************************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgaccountobject.h"
#include "skgtestmacro.h"
#include "skgbankincludes.h"
#include "skgimportexportmanager.h"

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
int main(int argc, char** argv)
{
    Q_UNUSED(argc)
    Q_UNUSED(argv)

    // Init test
    SKGINITTEST(true)

    {
        // Test import XML
        SKGDocumentBank document1;
        SKGError err;
        {
            SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)

            for (int i = 1; !err && i <= 2; ++i) {
                {
                    // Scope of the transaction
                    SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_XML"), err)
                    SKGImportExportManager impmissing(&document1, QUrl::fromLocalFile(QLatin1String("/not-existing/missingfile.xml")));
                    SKGTESTERROR(QLatin1String("imp1.importFile") + SKGServices::intToString(i), impmissing.importFile(), false)

                    SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportxml/input.xml"));
                    SKGTESTERROR(QLatin1String("XML.importFile") + SKGServices::intToString(i), imp1.importFile(), true)
                }

                {
                    SKGAccountObject account(&document1);
                    SKGTESTERROR(QLatin1String("XML.setName") + SKGServices::intToString(i), account.setName(QLatin1String("COURANT")), true)
                    SKGTESTERROR(QLatin1String("XML.load") + SKGServices::intToString(i), account.load(), true)
                    SKGTEST(QLatin1String("XML:getCurrentAmount") + SKGServices::intToString(i), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-15255"))
                }

                {
                    SKGAccountObject account(&document1);
                    SKGTESTERROR(QLatin1String("XML.setName") + SKGServices::intToString(i), account.setName(QLatin1String("EPARGNE")), true)
                    SKGTESTERROR(QLatin1String("XML.load") + SKGServices::intToString(i), account.load(), true)
                    SKGTEST(QLatin1String("XML:getCurrentAmount") + SKGServices::intToString(i), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("197.8444982"))
                }
            }
        }
        {
            SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true) {
                SKGTESTERROR(QLatin1String("DOC:changePassword"), document1.changePassword(QLatin1String("test")), true)

                // Scope of the transaction
                SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_ISO20022"), err)

                SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportxml/camt_053_ver_2_extended_uk_account.xml"));
                SKGTESTERROR(QLatin1String("ISO20022.importFile"), imp1.importFile(), true)
            }

            {
                SKGAccountObject account(&document1);
                SKGTESTERROR(QLatin1String("ISO20022.setName"), account.setName(QLatin1String("GB87HAND40516218000025")), true)
                SKGTESTERROR(QLatin1String("ISO20022.load"), account.load(), true)
                SKGTEST(QLatin1String("ISO20022:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-0.1"))
            }
        }
        {
            SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true) {

                // Scope of the transaction
                SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_UNKNOWN"), err)

                SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportxml/unknown.xml"));
                SKGTESTERROR(QLatin1String("ISO20022.importFile"), imp1.importFile(), false)
            }
        }
        {
            SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true) {

                // Scope of the transaction
                SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_UNKNOWN"), err)

                SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportxml/paket_2.xml"));
                SKGTESTERROR(QLatin1String("ISO20022.importFile"), imp1.importFile(), true)
            }
            {
                SKGAccountObject account(&document1);
                SKGTESTERROR(QLatin1String("ISO20022.setName"), account.setName(QLatin1String("SI56123456789012345")), true)
                SKGTESTERROR(QLatin1String("ISO20022.load"), account.load(), true)
                SKGTEST(QLatin1String("ISO20022:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("1999.01"))
            }
        }

        {
            SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true) {

                // Scope of the transaction
                SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_1"), err)

                SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportxml/472392_1.xml"));
                SKGTESTERROR(QLatin1String("ISO20022.importFile"), imp1.importFile(), true)

                // 472390
                SKGAccountObject account(&document1);
                SKGTESTERROR(QLatin1String("ACT:setName"), account.setName("SI5___3"), true)
                SKGTESTBOOL("ACT:exist", account.exist(), false)
            }

            {

                // Scope of the transaction
                SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_2"), err)

                SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportxml/472392_2.xml"));
                SKGTESTERROR(QLatin1String("ISO20022.importFile"), imp1.importFile(), true)
            }
        }
    }

    // End test
    SKGENDTEST()
}
