/***************************************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtestmacro.h"
#include "skgbankincludes.h"
#include "skgimportexportmanager.h"

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
int main(int argc, char** argv)
{
    Q_UNUSED(argc)
    Q_UNUSED(argv)

    // Init test
    SKGINITTEST(true)

    {
        // Test import SKG
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_SKG"), err)
            SKGImportExportManager impmissing(&document1, QUrl::fromLocalFile(QLatin1String("missingfile.skg")));
            SKGTESTERROR(QLatin1String("imp1.importFile"), impmissing.importFile(), false)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportskg/all_types.skg"));
            SKGTESTERROR(QLatin1String("SKG.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("SKG.setName"), account.setName(QLatin1String("FRANC")), true)
            SKGTESTERROR(QLatin1String("SKG.load"), account.load(), true)
            SKGTEST(QLatin1String("SKG:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("1.524490172"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("SKG.setName"), account.setName(QLatin1String("COURANT")), true)
            SKGTESTERROR(QLatin1String("SKG.load"), account.load(), true)
            SKGTEST(QLatin1String("SKG:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-14055"))

            auto properties = account.getProperties();
            SKGTEST(QLatin1String("SKG:getProperties"), SKGServices::intToString(properties.length()), QLatin1String("1"))

            SKGPropertyObject p(account.getPropertyObject(properties.at(0)));
            SKGTEST(QLatin1String("SKG:getPropertyBlob FALSE"), static_cast<unsigned int>(account.getPropertyBlob(properties.at(0)).isNull() || account.getPropertyBlob(properties.at(0)).toByteArray().length() == 0), static_cast<unsigned int>(true))
            SKGTEST(QLatin1String("SKG:getPropertyObject.value"), p.getValue(), "PROP_CPT")
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("SKG.setName"), account.setName(QLatin1String("EPARGNE")), true)
            SKGTESTERROR(QLatin1String("SKG.load"), account.load(), true)
            SKGTEST(QLatin1String("SKG:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("5622.224686"))
        }

        // test multi import
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_SKG"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportskg/all_types.skg"));
            SKGTESTERROR(QLatin1String("SKG.importFile"), imp1.importFile(), true)
        }

        // Export xml
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("EXPORT_XML"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportskg/all_types.xml"));
            SKGTESTERROR(QLatin1String("SKG.exportFile"), imp1.exportFile(), true)
        }

        // Export json
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("EXPORT_JSON"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportskg/all_types.json"));
            SKGTESTERROR(QLatin1String("SKG.exportFile"), imp1.exportFile(), true)
        }

        // Export json
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_JSON"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportskg/all_types.json"));
            SKGTESTERROR(QLatin1String("SKG.importFile"), imp1.importFile(), false)
        }
    }
    {
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        // Export skg from memory
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("EXPORT_SKG"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportskg/all_types_exp_from_mem.skg"));
            SKGTESTERROR(QLatin1String("SKG.exportFile from memory"), imp1.exportFile(), true)
            SKGTESTERROR(QLatin1String("SKG.importFile from file"), imp1.importFile(), true)
        }
        SKGTESTERROR(QLatin1String("SKG.saveAs"), document1.saveAs(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportskg/all_types_save.skg", true), true)

        // Export skg from file
        SKGTESTERROR(QLatin1String("SKG.load"), document1.load(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportskg/all_types_save.skg"), true) {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("EXPORT_SKG"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportskg/all_types_exp_from_file.skg"));
            SKGTESTERROR(QLatin1String("SKG.exportFile from file"), imp1.exportFile(), true)
            SKGTESTERROR(QLatin1String("SKG.importFile from file"), imp1.importFile(), true)
        }
    }
    {
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        // Export sqlite from memory
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("EXPORT_SQLITE"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportskg/all_types_exp_from_mem.sqlite"));
            SKGTESTERROR(QLatin1String("SQLITE.exportFile from memory"), imp1.exportFile(), true)
            SKGTESTERROR(QLatin1String("SQLITE.importFile from file"), imp1.importFile(), true)
        }

        // Export skg from file
        SKGTESTERROR(QLatin1String("SKG.load"), document1.load(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportskg/all_types_save.skg"), true) {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("EXPORT_SQLITE"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportskg/all_types_exp_from_file.sqlite"));
            SKGTESTERROR(QLatin1String("SQLITE.exportFile from file"), imp1.exportFile(), true)
            SKGTESTERROR(QLatin1String("SQLITE.importFile from file"), imp1.importFile(), true)
        }
    }
    {
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        // Export sqlcipher from memory
        {
            // Scope of the transaction
            SKGTESTERROR(QLatin1String("SQLCIPHER.changePassword"), document1.changePassword(QLatin1String("password")), true)

            SKGBEGINTRANSACTION(document1, QLatin1String("EXPORT_SQLCIPHER"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportskg/all_types_exp_from_mem.sqlcipher"));
            SKGTESTERROR(QLatin1String("SQLCIPHER.exportFile from memory"), imp1.exportFile(), true)
            QMap<QString, QString> parameters;
            parameters[QLatin1String("password")] = QLatin1String("password");
            imp1.setImportParameters(parameters);
            SKGTESTERROR(QLatin1String("SQLCIPHER.importFile from file"), imp1.importFile(), true)
        }

        // Export skg from file
        SKGTESTERROR(QLatin1String("SKG.load"), document1.load(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportskg/all_types_save.skg"), true) {
            // Scope of the transaction
            SKGTESTERROR(QLatin1String("SQLCIPHER.changePassword"), document1.changePassword(QLatin1String("password")), true)

            SKGBEGINTRANSACTION(document1, QLatin1String("EXPORT_SQLCIPHER"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportskg/all_types_exp_from_file.sqlcipher"));
            SKGTESTERROR(QLatin1String("SQLCIPHER.exportFile from file"), imp1.exportFile(), true)
            QMap<QString, QString> parameters;
            parameters[QLatin1String("password")] = QLatin1String("password");
            imp1.setImportParameters(parameters);
            SKGTESTERROR(QLatin1String("SQLCIPHER.importFile from file"), imp1.importFile(), true)
        }
    }
    {
        // Test import encrypted SKG
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_SKG"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportskg/encrypted.skg"));
            QMap<QString, QString> parameters;
            parameters[QLatin1String("password")] = QLatin1String("password");
            imp1.setImportParameters(parameters);
            SKGTESTERROR(QLatin1String("SKG.importFile"), imp1.importFile(), false) // Since 2.33.0, no more supported
        }
    }

    {
        // Test import SKG
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_SKG"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportskg/euro_bitcoin_dollar.skg"));
            SKGTESTERROR(QLatin1String("SKG.importFile"), imp1.importFile(), true)
        }
    }

    {
        // Test import SKG
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_SKG"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/advice.skg"));
            SKGTESTERROR(QLatin1String("SKG.importFile"), imp1.importFile(), true)
        }
    }
    // End test
    SKGENDTEST()
}
