/**
 * treeitems.cpp
 *
 * Copyright (C)  2003  Zack Rusin <zack@kde.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "items.h"
#include "entrytoken.h"
#include "grouptoken.h"
#include "applicationtoken.h"
#include "hosttoken.h"

#include <kiconloader.h>
#include <kglobal.h>
#include <klocale.h>
#include <kdebug.h>

#include <qpixmap.h>

using namespace KCfgCreator;

namespace KCfgCreator
{

ConfItem::ConfItem( KListView* parent, const QString& name )
    : KListViewItem( parent, name )
{
}

ConfItem::ConfItem( KListView* parent, ConfItem* after, const QString& label1,
                    const QString& label2, const QString& label3,
                    const QString& label4, const QString& label5,
                    const QString& label6, const QString& label7,
                    const QString& label8 )
    : KListViewItem( parent, after, label1, label2, label3, label4,
                     label5, label6, label7, label8 )
{
}

ConfItem::ConfItem( ConfItem* parent, const QString& name )
    : KListViewItem( parent, name )
{
}

ConfItem::ConfItem( ConfItem* parent, const QString& name, const QString& value )
    : KListViewItem( parent, name, value )
{
}

EntryItem::EntryItem( GroupItem* parent, const QString& name )
    : ConfItem( parent, name )
{
    m_token = parent->group()->createEntry( name, QString::fromLatin1( "String" ) );
}

EntryItem::EntryItem( GroupItem* parent, EntryToken* token )
    : ConfItem( parent, token->key() )
{
    m_token = token;
}

void
EntryItem::update()
{
    setText( 0, m_token->key() );
}

QString
EntryItem::name() const
{
    return m_token->key();
}

QString
EntryItem::value() const
{
    return m_token->valueAsString();
}

void
EntryItem::switchTokens( EntryToken *token )
{
    m_token = token;
    setText( 0, m_token->key() );
}

GroupItem::GroupItem( AppItem* parent, const QString& name )
    : ConfItem( parent, name )
{
    m_token = parent->app()->createGroup( name, false );
}

GroupItem::GroupItem( AppItem* parent, GroupToken* token )
    : ConfItem( parent, token->name() )
{
    m_token = token;
    QPtrList<EntryToken> l( token->entryList() );
    QPtrListIterator<EntryToken> itr( l );

    while ( itr.current() ) {
        new EntryItem( this, itr.current() );
        ++itr;
    }
}

QString
GroupItem::name() const
{
    return m_token->name();
}

EntryToken*
GroupItem::entry( const QString& key )
{
    return m_token->entry( key );
}

GroupToken*
GroupItem::group() const
{
    return m_token;
}

QMap<QString, QString>
GroupItem::entriesMap() const
{
    GroupToken::EntryList l( m_token->entryList() );
    QPtrListIterator<EntryToken> itr( l );
    QMap<QString, QString> emap;

    while ( itr.current() ) {
        emap.insert( itr.current()->key(),
                     itr.current()->valueAsString() );
        ++itr;
    }
    return emap;
}


AppItem::AppItem( KListView *parent, const QString& name )
    : ConfItem( parent, name )
{
    m_token = new ApplicationToken( name, name );
    init( name );
}

AppItem::AppItem( KListView* parent, ApplicationToken* app )
    : ConfItem( parent, app->name() ), m_token( app )
{
    init( app->name() );

    QPtrList<GroupToken> l( app->groupList() );
    QPtrListIterator<GroupToken> itr( l );

    while ( itr.current() ) {
        new GroupItem( this, itr.current() );
        ++itr;
    }
}

void
AppItem::init( const QString& name )
{
    QPixmap pm = KGlobal::iconLoader()->loadIcon( name, KIcon::User, 0,
                                                  KIcon::DefaultState, 0,
                                                  true );
    if ( pm.isNull() )
        pm = KGlobal::iconLoader()->loadIcon( name, KIcon::Small, 0,
                                              KIcon::DefaultState, 0,
                                              true );
    //kdDebug()<<"Name "<<name<<" " << pm.isNull() <<endl;

    if ( pm.isNull() && name.endsWith( ".kcfg" ) ) {
        QString mname = name.left( name.length() - 4 );
        pm = KGlobal::iconLoader()->loadIcon( mname, KIcon::User, 0,
                                              KIcon::DefaultState, 0,
                                              true );
        if ( pm.isNull() )
            pm = KGlobal::iconLoader()->loadIcon( mname, KIcon::Small, 0,
                                                  KIcon::DefaultState, 0,
                                                  true );
        //kdDebug()<<"Name "<<name<<" " << pm.isNull() <<endl;
    }

    if ( pm.isNull() && name.endsWith( "rc" ) ) {
        QString mname = name.left( name.length() - 2 );
        pm = KGlobal::iconLoader()->loadIcon( mname, KIcon::User, 0,
                                              KIcon::DefaultState, 0,
                                              true );
        if ( pm.isNull() )
            pm = KGlobal::iconLoader()->loadIcon( mname, KIcon::Small, 0,
                                                  KIcon::DefaultState, 0,
                                                  true );
        //kdDebug()<<"Name "<<name<<" " << pm.isNull() <<endl;
    }


    if ( pm.isNull() ) {
        pm = KGlobal::iconLoader()->loadIcon( "configure", KIcon::Small, 0,
                                              KIcon::DefaultState, 0,
                                              true );
    }
    if ( !pm.isNull() )
        setPixmap( 0, pm );
}

AppItem::~AppItem()
{
    delete m_token;
    m_token = 0;
}

QString
AppItem::name() const
{
    return m_token->name();
}

void
AppItem::setName( const QString& name )
{
    m_token->setName( name );
    setText( 0, name );
}

QStringList
AppItem::groupList() const
{
    ApplicationToken::GroupList l( m_token->groupList() );
    QPtrListIterator<GroupToken> itr( l );

    QStringList glist;

    while ( itr.current() ) {
        glist.append( itr.current()->name() );
    }

    return glist;
}

GroupToken*
AppItem::group( const QString& name ) const
{
    return m_token->group( name );
}

ApplicationToken*
AppItem::app() const
{
    return m_token;
}

} //end namespace KCfgCreator
