/**
 * entrytoken.h
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 */
#ifndef KONFIG_ENTRYTOKEN_H
#define KONFIG_ENTRYTOKEN_H

#include "token.h"

#include <stdlib.h>

#include <kurl.h>

#include <qstringlist.h>
#include <qvaluelist.h>
#include <qfont.h>
#include <qcolor.h>
#include <qsize.h>
#include <qrect.h>
#include <qpoint.h>
#include <qdatetime.h>
#include <qstring.h>

namespace KCfgCreator
{
    class GroupToken;

    /**
     * EntryToken is the class representing the base config entries.
     * In incapsulates all features of KConfig and KCfg entries.
     * EntryToken's are formed by merging informations from KConfig
     * and KCFG files.
     */
    class EntryToken : public Token
    {
    public:
             enum Type {
            String,
            Password,
            StringList,
            Font,
            Rect,
            Size,
            Color,
            Point,
            Int,
            UInt,
            Bool,
            Double,
            DateTime,
            Int64,
            UInt64,
            IntList,
            Enum,
            Path,
            Invalid
        };

    public:
        struct Value {
            Value() : isCode( false ) {}
            QString value;
            bool isCode;
        };

        struct DefaultValue : public Value {
            QString param;
        };

        struct Parameter {
            Parameter() : max( -1 ) { }
            QString name;
            QString type;//can only be int, uint and Enum
            int max;//required for int and uint
            QStringList values;//for enums
        };
        struct Choice {
            QString name;
            QString label;
            QString whatsThis;
        };

        virtual ~EntryToken() { }

        /**
         * Returns the token as the QVariant
         */
        virtual QVariant variant() const =0;
        /**
         * Sets the token from the QVariant
         * @return true if successful
         */
        virtual void setVariant( const QVariant& ) =0;

        /**
         * Does the best it can to convert the currently held type
         * to a string representation.
         */
        virtual QString  valueAsString() const =0;

        virtual void setValueFromString( const QString& ) = 0;

        /**
         * Returns the name of the type as a string.
         */
        virtual QString typeName() const = 0;

        /**
         * The type of the value held by the token as a QVariant::Type
         */
        virtual Type    type() const =0;

        bool contains( const QString& str ) const;
        GroupToken* parent() const;

        /**
         * These are cfg specific useful only for gui representation of those
         * options.
         */
        QString label() const;
        void setLabel( const QString& );

        /**
         * Represents a more detailed description of an entry.
         */
        QString whatsThis() const;
        void setWhatsThis( const QString& );

        /**
         * Default value as set in the KCfg file
         */
        DefaultValue defaultValue() const;
        void setDefaultValue( const DefaultValue& );

        /**
         * For entries with a predefined set of available entries
         * (enums) represents the set of available options.
         */
        QValueList<Choice> choices() const;
        void setChoices( const QValueList<Choice>& );

        /**
         * Name and key of the given entry. Both will be unique
         * but the name can return a whitespace filled string.
         */
        virtual QString name() const;
        QString key() const;
        void setKey( const QString& );

        /**
         * Indentifies hidden config entries (set in KCfg files)
         */
        void setHidden( bool );
        bool hidden() const;

        Value min() const;
        void setMin( const Value& );

        Value max() const;
        void setMax( const Value& );

        Parameter parameter() const;
        void setParameter( const Parameter& );

        /**
         * If a config entry is default-settable via code, this
         * entry returns the code used to get the default value.
         */
        QString code() const;
        void setCode( const QString& );

        QStringList values() const;
        void setValues( const QStringList& );

	/**
	* Is the token taken from an enviroment variable or script.
	*/
        void setFromEnvironment( bool );
        bool fromEnvironment() const;

    protected:
        friend class GroupToken;
        EntryToken( GroupToken* parent, const QString& key,
                    const QString& name = QString::null, bool hidden=false );
        void copyFromToken( EntryToken* token );
        void addEntryMemento( const QString& data,
                              void (EntryToken::*func)( const QString& ) );
        void addEntryMemento( const QStringList& data,
                              void (EntryToken::*func)( const QStringList& ) );

    private:
        GroupToken*  m_parent;
        QString      m_label;
        QString      m_whatsThis;
        QValueList<Choice> m_choices;
        DefaultValue m_default;
        Parameter    m_param;
        QString      m_key;
        Value        m_max;
        Value        m_min;
        QString      m_code;
        QStringList  m_values;
        bool         m_hidden;
        bool         m_fromEnv;

        class Private;
        Private *d;
    };

    namespace Utils
    {
        EntryToken::Type stringToType( const QString& typeName );
    }

}
#endif
