
/**
 * editorwidget.cpp
 *
 * Copyright (C)  2003  Zack Rusin <zack@kde.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */
#include "editorwidget.h"

#include "items.h"
#include "kcfgentryeditor.h"
#include "classpreview.h"
#include "entrytoken.h"
#include "grouptoken.h"
#include "kcfgcreator.h"
#include "kcfgcoptions.h"

#include <khtml_part.h>
#include <khtmlview.h>
#include <kparts/browserextension.h>
#include <kstandarddirs.h>
#include <klocale.h>
#include <kdebug.h>

#include <qwidgetstack.h>
#include <qlayout.h>
#include <qdom.h>
#include <qstylesheet.h>
#include <qfile.h>
#include <qcstring.h>

using namespace KCfgCreator;

EditorWidget::EditorWidget( QWidget *parent,  const char *name )
    :QWidget( parent, name ), m_item( 0 ), m_currentView( Code )
{
    QHBoxLayout *box = new QHBoxLayout( this );
    m_widgetStack = new QWidgetStack( this );
    m_editor   = new EntryEditor( m_widgetStack );
    m_htmlView = new KHTMLPart( m_widgetStack );
    m_widgetStack->addWidget( m_editor );
    m_widgetStack->addWidget( m_htmlView->view() );

    box->addWidget( m_widgetStack );

    fillHTML();
    writeHTML( i18n( "KCfgCreator helps you create KCFG files for your application." ),
               i18n( "Please create a top level entry or parse an existing file." ) );
    m_widgetStack->raiseWidget( m_htmlView->view() );
    initConnections();
}

void EditorWidget::initConnections()
{
    connect( m_htmlView->browserExtension(),
             SIGNAL(openURLRequest(const KURL &, const KParts::URLArgs &)),
             SLOT(slotUrlOpen(const KURL &)) );
}

void EditorWidget::reset()
{
}

void EditorWidget::treeSelectionChanged( QListViewItem *item )
{
    m_item = dynamic_cast<ConfItem*>( item );
    if ( m_item ) {
        if ( m_item->type() == ConfItem::Entry ) {
            updateEditor();
            m_widgetStack->raiseWidget( m_editor );
        } else {
            updateHTML();
            m_widgetStack->raiseWidget( m_htmlView->view() );
        }
    }
}

void EditorWidget::fillHTML()
{
    QString location = locate( "data", "kcfgcreator/main.html" );
    QFile file( location );
    file.open( IO_ReadOnly );
    QByteArray b = file.readAll();
    m_baseHTML = QString::fromUtf8( QCString( b, b.size() + 1 ) );
    file.close();
}

void EditorWidget::writeHTML( const QString& arg1, const QString& arg2 )
{
    m_htmlView->begin( locate( "data", "kcfgcreator/main.html" ) );
    m_htmlView->write( m_baseHTML.arg( arg1 ).
                       arg( i18n( "Generated Code" ) ).
                       arg( i18n( "Used KCFG code " ) ).
                       arg( arg2 ) );
    m_htmlView->end();
}

void EditorWidget::updateEditor()
{
    EntryItem *item = static_cast<EntryItem*>( m_item );
    if ( item ) {
        m_editor->updateEntry( item );
    }
}

void EditorWidget::updateHTML()
{
    if ( !m_item ) {
        //The only case where this can happen is when we startup
        //and don't have anything in the listview yet
        return;
    }

    switch( m_currentView ) {
    case Code:
    {
        if ( m_item->type() == ConfItem::Group ) {
            writeHTML( i18n( "Generated methods for group: %1." )
                       .arg( m_item->name() ),
                       QString( "<pre>%1</pre>" ).arg(
                           ClassPreview::previewGroup(
                               static_cast<GroupItem*>( m_item )->group() )
                           ) );
        } else if ( m_item->type() == ConfItem::App ) {
            QString arg1 = i18n( "Generated code for class: %1." ).
                           arg( KCfgcOptions::self()->className() );
            QString arg2 = QString( "<pre>%1</pre>" ).arg(
                ClassPreview::previewClass(
                    static_cast<AppItem*>( m_item )->app() ) );
            writeHTML( arg1, arg2 );
        }
    }
    break;
    case KCfgXML:
    {
        CreatorKCfg creator( this );
        if ( m_item->type() == ConfItem::Group ) {
            QDomDocument doc;
            doc.appendChild( creator.createGroup(
                                 doc,
                                 static_cast<GroupItem*>( m_item )->group()
                                 ) );
            QString contents = doc.toString();
            contents = QStyleSheet::escape( contents );
            writeHTML( i18n( "This view shows the generated output for the %1 group." )
                       .arg( m_item->name() ),
                       QString( "<pre>%1</pre>" ).arg( contents ) );
            m_widgetStack->raiseWidget( m_htmlView->view() );
        } else if ( m_item->type() == ConfItem::App ) {
            QString contents = creator.configAsString(
                static_cast<AppItem*>( m_item )->app() );
            contents = QStyleSheet::escape( contents );
            writeHTML( i18n( "This view shows the generated output for the %1 application." )
                       .arg( m_item->name() ),
                       QString( "<pre>%1</pre>" ).arg( contents ) );
            m_widgetStack->raiseWidget( m_htmlView->view() );
        }
    }
    break;
    default:
        break;
    }
}

void EditorWidget::slotUrlOpen( const KURL& url )
{
    kdDebug()<<"Clicked on url "<< url.query()
             << " ref = " << url.htmlRef()
             <<endl;
    QString menu = url.htmlRef();

    ViewType oldView = m_currentView;
    if ( menu == "menu1" ) {
        m_currentView = Code;
    } else
        m_currentView = KCfgXML;

    //Updating takes a second so we don't want to be doing it
    //if we dont have to
    if ( oldView != m_currentView )
        updateHTML();
}

#include "editorwidget.moc"
