#include <kiconloader.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <qwhatsthis.h>
#include <sys/param.h>
#include <qpainter.h>
#include <kstddirs.h>
#include <klocale.h>
#include <qbitmap.h>
#include <kconfig.h>
#include <qfile.h>
#include <ctype.h>
#include "krestoredlg.h"
#include "rogue/dun.h"
#include "kcharfile.h"
#include "kcharfdlg.h"
#include "krogueqh.h"
#include "kinvwnd.h"
#include "kmsgbar.h"
#include "krogue.h"
#include "kscore.h"
#include "roguec.h"
#include "kcmdr.h"
#include "kgame.h"
#include "kdun.h"

#define MSGLEN 160

KDunView::KDunView(QWidget *p)
:QScrollView(p)
{
  resizeContents(DCOLS*PIXMAP_X, DROWS*PIXMAP_Y);
  viewport()->setBackgroundColor(QColor(0, 0, 0));
}

void KDunView::setSize(int x, int y)
{
  setVScrollBarMode(QScrollView::AlwaysOff);
  setHScrollBarMode(QScrollView::AlwaysOff);

  resize(x + width()-visibleWidth(),
         y + height()-visibleHeight());

  setVScrollBarMode(QScrollView::Auto);
  setHScrollBarMode(QScrollView::Auto);
}

void KDunView::writeCfg(KConfig *cfg)
{
  setVScrollBarMode(QScrollView::AlwaysOff);
  setHScrollBarMode(QScrollView::AlwaysOff);
  cfg->writeEntry("W", visibleWidth());
  cfg->writeEntry("H", visibleHeight());
}

void KDunView::resizeEvent(QResizeEvent *e)
{
  QScrollView::resizeEvent(e);

  int hm=QMAX(visibleWidth()-DCOLS*PIXMAP_X, 0),
      vm=QMAX(visibleHeight()-DROWS*PIXMAP_Y, 0);

  moveChild(dun, hm/2, vm/2);
}

void KDunView::makeCellVisible(int row, int col)
{
  ensureVisible(col*PIXMAP_X, row*PIXMAP_Y);
}

KDun::KDun(QWidget *qw, const char *name)
:QWidget(qw, name)
{
  pixmap=NULL;

  QString path=locate("data", "krogue/pics/tiles.png");
  if (path.isEmpty())
    return;

  pixmap=new QPixmap(path);
  if (!pixmap)
    return;

  dung=new QPixmap(DCOLS*PIXMAP_X, DROWS*PIXMAP_Y);

  setBackgroundColor(QColor(0, 0, 0));
  setFocusPolicy(QWidget::StrongFocus);
  whatx=whaty=-1;
  killoth=0;
  whatsthis.insertItem(i18n("&What's this?"), this, SLOT(whats()));
  setFixedSize(DCOLS*PIXMAP_X, DROWS*PIXMAP_Y);
}

int KDun::initDungeon(const char *fn)
{
  int i, id;
  if (!pixmap || pixmap->isNull())
  {
    KMessageBox::error(NULL, i18n("Can't load tiles.png"), i18n("Rogue"));
    return 0;
  }
  QBitmap mask(PIXMAP_Y-2, PIXMAP_Y-2, true);
  itempix[0]=new QPixmap;
  for (i=1; i<10; i++)
  {
    mask.fill(QColor(0, 0, 0));
    if (i==V_GOLD) id=V_PROTARMOR; else id=i;
    itempix[i]=new QPixmap(PIXMAP_Y-2, PIXMAP_Y-2);
    if (!itempix[i] || !dung)
    {
      KMessageBox::error(kapp->mainWidget(),
        i18n("Can't create pixmap (out of memory?)"), i18n("Rogue"));
      return 0;
    }
    bitBlt(itempix[i], (PIXMAP_Y-PIXMAP_X-2)/2, 0, pixmap, (id&0xf)*(PIXMAP_X+1),
      ((id>>4)+2)*(PIXMAP_Y+1)+2, PIXMAP_X, PIXMAP_Y-2, CopyROP, true);
    bitBlt(&mask, (PIXMAP_Y-PIXMAP_X-2)/2, 0, pixmap->mask(), (id&0xf)*(PIXMAP_X+1),
      ((id>>4)+2)*(PIXMAP_Y+1)+2, PIXMAP_X, PIXMAP_Y-2, OrROP, true);
    itempix[i]->setMask(mask);
  }
  dung->fill(backgroundColor());

  if (fn==NULL)
  {
    init(0);
    initLevel(0);
  } else
  {
    char *str;
    init(1);
    if (!restore(fn, 0, &str))
    {
      KMessageBox::error(kapp->mainWidget(), i18n("Rogue"), QString::fromUtf8(str));
      return 0;
    }
    initLevel(1);
    updatePixmap(0, 0, DROWS-1, DCOLS-1);
  }
  return 1;
}

void KDun::storeMap(const QString &fn)
{
  if (!dung->save(fn, "PNG"))
  {
    KMessageBox::error(kapp->mainWidget(),
      i18n("Cannot save dungeon image.\n"
           "The page may be incomplete"));
  }
}

void KDun::storeItemPix(int num, const QString &fn)
{
  if (!itempix[num]->save(fn, "PNG"))
  {
    KMessageBox::error(kapp->mainWidget(),
      i18n("Cannot save item icon image.\n"
           "The page may be incomplete"));
  }
}

void KDun::saveProperties(KConfig *cfg)
{
  QString path=locateLocal("config", "krsavXXXXXX");
  char buf[PATH_MAX];
  strncpy(buf, QFile::encodeName(path), PATH_MAX);
  buf[PATH_MAX-1]=0;
  if (mktemp(buf)==NULL)
  {
    KMessageBox::error(top, i18n("Cannot create temporary save file."));
    return;
  }
  if (gam->save(buf)!=0) return;
  cfg->writeEntry("SaveFile", buf);
}

void KDun::readProperties(KConfig *cfg)
{
  const char *c=QFile::encodeName(cfg->readEntry("SaveFile"));
  initDungeon(c);
}

char *getDesc(object *obj)
{
  static char buf[MSGLEN];
  get_desc(obj, buf);
  return buf;
}    

void KDun::initLevel(bool restore)
{
  if (!restore)
  {
    clear_level();
    make_level();
    put_objects();
    put_stairs();
    put_mons();
    add_traps();
    put_player(party_room);
  }
  print_stats(STAT_ALL);
  cmdr->check();
}

void KDun::initLevel()
{
  initLevel(false);
}

void KDun::paintEvent(QPaintEvent *pe)
{
  QRect r=pe->rect();
  bitBlt(this, r.x(), r.y(), dung, r.x(), r.y(), r.width(), r.height());
}

void KDun::paintPixmap(const QRect &r)
{
  bitBlt(this, r.x(), r.y(), dung, r.x(), r.y(), r.width(), r.height());
}

void KDun::clear()
{
  dung->fill(backgroundColor());
  erase();
}

void KDun::keyPressEvent(QKeyEvent *k)
{
  cmdr->interpretKey(k);
}

void KDun::updatePixmap(int row1, int col1, int row2, int col2)
{
  int i, j, fl;
/*  printf("updatePixmap(%d, %d, %d, %d)\n", row1, col1, row2, col2);*/
  for (i=row1; i<=row2; i++)
    for (j=col1; j<=col2; j++)
    {
      fl=vdungeon[i][j]>>FLSH;  
      bitBlt(dung, j*PIXMAP_X, i*PIXMAP_Y, pixmap, (fl&0xf)*(PIXMAP_X+1),
	(fl>>4)*(PIXMAP_Y+1), PIXMAP_X, PIXMAP_Y);
      fl=(vdungeon[i][j]&OBJMASK)+32;
      if (fl)
        bitBlt(dung, j*PIXMAP_X, i*PIXMAP_Y, pixmap, (fl&0xf)*(PIXMAP_X+1),
	  (fl>>4)*(PIXMAP_Y+1), PIXMAP_X, PIXMAP_Y);    
    }
  paintPixmap(QRect(col1*PIXMAP_X, row1*PIXMAP_Y, (col2-col1+1)*PIXMAP_X, (row2-row1+1)*PIXMAP_Y));
}

void KDun::killedBy(object *monster, short other, bool really)
{
  int level;
  killoth=(monster?10:other);
  if (!cmdr->readOnly())
  {
    char buf[MSGLEN];
    if (msg->willQuit) return;
    if (other!=WIN)
    {
      strncpy(buf, killed_by_setbuf(monster, other), MSGLEN);
      KHTMLDlg khdlg(QSize(350, 150), top);
      khdlg.setVar("NAME", gam->rogueName());
      khdlg.setVar("KILLEDBY", QString::fromUtf8(buf));
      khdlg.setVar("CREATE", i18n("Create character file"));
      connect(&khdlg, SIGNAL(action(const char *)), SLOT(charfile()));
      if (other==QUIT) khdlg.loadFile("quit.qml"); else
      {
        khdlg.loadFile("killedby.qml");
        khdlg.setIcon(UserIcon("rip"));
      }
      if (khdlg.loaded() && (cmdr->cmd_executed>0)) khdlg.exec();
      level=cur_level;
    } else level=max_level;
    
    KScoreDlg top10(false, top);
    if (!score_only)
      top10.putScore(rogue.gold, login_name, nick_name, other, level, 
        (monster?monster->m_char:0), has_amulet());
    top10.listScores();
    if (top10.loaded() && really && top10.listCount()>1) top10.exec();
    
    KScoreDlg user10(true, top);
    if (!score_only)
      user10.putScore(rogue.gold, login_name, nick_name, other, level, 
        (monster?monster->m_char:0), has_amulet());
    user10.listScores();
    if (user10.loaded() && really) user10.exec();
  }
  killoth=0;
  msg->willQuit=1;
  if (really) kapp->quit();
}

void KDun::highscores()
{
  KScoreDlg top10(false, top);
  top10.listScores();
  if (top10.loaded()) top10.exec();
}

void KDun::uhighscores()
{
  KScoreDlg top10(true, top);
  top10.listScores();
  if (top10.loaded()) top10.exec();
}

void KDun::win()
{
  KHTMLDlg win(QSize(530, 150));
  win.setVar("YOU_WON", i18n("YOU WON!"));
  win.setVar("CONGRATULATIONS",
    i18n("Congratulations, you have been admitted to the Fighter's Guild. "
         "You return home, sell all your treasure at great profit and retire "
         "into comfort."));
  win.loadFile("win.qml");
  if (win.loaded()) win.exec();
  win_unwear();
  KSellItemDlg dlg(top);
  object *obj=&rogue.pack;
  char buf[MSGLEN];
  int value;
  while ((obj=sell_pack(obj, buf, &value))!=NULL)
    if (buf[0]) dlg.add(QString::fromUtf8(buf), value);
  dlg.exec();
  killedBy(NULL, WIN);
}

void KDun::quit(bool really)
{
  if (!cmdr->readOnly() && (cmdr->cmd_executed>0))
    if (KMessageBox::questionYesNo(top,
      i18n("Are you sure that you want to quit?"))!=KMessageBox::Yes) return;
  killedBy(NULL, QUIT, really);
}

void KDun::quit()
{
  quit(true);
}

void KDun::charfile(void)
{
  KCharFDlg dlg(this);
  KCharFile *cf=NULL;
  dlg.exec();
  if (dlg.result()==QDialog::Rejected) return;
  switch (dlg.current)
  {
    case 1: cf=new KTextCharFile; break;
    case 2: cf=new KHTMLCharFile; break;
    case 3: cf=new KUnicodeCharFile; break;
  }
  CHECK_PTR(cf);
  QString fn;
  fn=KFileDialog::getSaveFileName(cfg->readEntry("CharDir"),
    QString(cf->mask())+"\n*", this);
  cfg->setGroup("Settings");
  if (fn.isNull() || fn.isEmpty()) return;
  if (QFile::exists(fn))
  {
    if (KMessageBox::questionYesNo(kapp->mainWidget(),
      i18n("File already exists.\nDo you want to overwrite it?"),
      i18n("Rogue"))!=KMessageBox::Yes)
    {
      delete cf;
      return;
    }
  }
  cf->store(fn, killoth);
  fn.truncate(fn.findRev('/'));
  cfg->writeEntry("CharDir", fn);
  delete cf;
}

int KDun::restoreFile(const char *fn)
{
  char *retstr;
  bool ro;
  KRestoreDlg dlg(fn, kapp->mainWidget());
  dlg.exec();
  if (dlg.result()==QDialog::Rejected || dlg.item()==2)
    return 0;
  free_stuff(&level_objects);
  free_stuff(&level_monsters);
  cmdr->restart();
  msg->invMsg();
  ro=dlg.item();
  if (!restore(fn, ro, &retstr))
  {
    dlg.setLabel(retstr);
    dlg.exec();
    if (dlg.result()==QDialog::Rejected || dlg.item()==2)
      return 0;
    restore(fn, 1, &retstr);
    ro=1;
  }
  cmdr->setReadOnly(ro);
  if (ro) score_only=1;
  initLevel(1);
  updatePixmap(0, 0, DROWS-1, DCOLS-1);
  top->newTitle();
  return 1;
}

void KDun::load()
{
  if (msg->more()) return;
  quit(false);
  if (!msg->willQuit) return;
  msg->willQuit=0;
  cfg->setGroup("Settings");
  QString fn=KFileDialog::getOpenFileName(cfg->readEntry("SaveDir"), "*.krogue\n*");
  cfg->setGroup("Settings");
  if (fn.isNull() || fn.isEmpty())
  {
    kapp->quit();
    return;
  }
  QString dir=fn;
  dir.truncate(dir.findRev('/'));
  cfg->writeEntry("SaveDir", dir);
  if (!restoreFile(QFile::encodeName(fn)))
    kapp->quit();
}

void KDun::focusInEvent(QFocusEvent *)
{
  //do nothing
}

void KDun::focusOutEvent(QFocusEvent *)
{
  //do nothing
}

void KDun::focusObject(int x, int y)
{
  QPainter p;
  p.begin(this);
  p.drawWinFocusRect(x*PIXMAP_X, y*PIXMAP_Y, PIXMAP_X, PIXMAP_Y);
  p.end();
}

void KDun::qhHyperlink(QString str)
{
  kapp->invokeHTMLHelp("krogue/"+str, "");
}

void KDun::qhClose()
{
  if (whatx!=-1) focusObject(whatx, whaty);
  whatx=whaty=-1;
}

bool KDun::customWhatsThis() const
{
  return true;
}

void KDun::whats()
{
  focusObject(whatx, whaty);
  QWhatsThis::enterWhatsThisMode();
  QWhatsThis::leaveWhatsThisMode(krwtGetDItemInfo(whatx, whaty),
    mapToGlobal(QPoint((whatx+1)*PIXMAP_X, (whaty+1)*PIXMAP_Y)));
}

void KDun::mousePressEvent(QMouseEvent *e)
{
  if (e->button()==LeftButton && QWhatsThis::inWhatsThisMode())
  {
    whatx=e->x()/PIXMAP_X;
    whaty=e->y()/PIXMAP_Y;
    if (whatx<0 || whatx>=DCOLS || whaty<0 || whaty>=DROWS)
    {
      QWhatsThis::leaveWhatsThisMode();
      return;
    }
    focusObject(whatx, whaty);
    QWhatsThis::leaveWhatsThisMode(krwtGetDItemInfo(whatx, whaty));
    focusObject(whatx, whaty);
    whatx=whaty=-1;
    return;
  }
  if (e->button()==RightButton && !QWhatsThis::inWhatsThisMode())
  {
    whatx=e->x()/PIXMAP_X;
    whaty=e->y()/PIXMAP_Y;
    focusObject(whatx, whaty);
    whatsthis.exec(QPoint(e->globalX(), e->globalY()));
  } else QWidget::mousePressEvent(e);
}

