/***************************************************************************
                          cmudstream.cpp
                      -------------------
    description          : Class for wrapping telnet I/O
    begin                : Sat Jan 22 2000
    copyright            : (C) 2000 by Kmud Developer Team
    email                : kmud-devel@kmud.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include <kprocess.h>
#include <qmessagebox.h>

#include "kmud.h"
#include "cmudstream.h"
#include "kmudplugin.h"
#include "cpluginmanager.h"



CMudStream::CMudStream(KmudApp* app)
{
	kmudapp=app;
	view=app->getView();
	doc=app->getDoc();
	connection=app->getConnection();
	log=app->getMudLog();
	pluginmanager=app->getPluginManager();

	mapper=app->getMapper();

	LoginHasBeenSent=false;
	PasswordHasBeenSent=false;

	triggerLoopLastTime.start();
	triggerLoopNum=0;
	triggerLoop=false;
	triggerLoopShowMessageBox=false;
}

CMudStream::~CMudStream()
{
}

void CMudStream::setCharDBEntry(CharDBEntry *e)
{
        entry = e;
}

void CMudStream::setMapper(CMapWindow *map)
{
        mapper = map;
}

bool CMudStream::triggerLoopDetect()
{
	bool ret=false;


	// if already loop detected always return true
	if (triggerLoop==true) return true;

	if (triggerLoopLastTime.elapsed() < 1000)
		triggerLoopNum++;
	else
		triggerLoopNum=0;


	if (triggerLoopNum > 20)
	{
		ret=true;
		triggerLoopNum=0;
	}

	triggerLoopLastTime.start();	


	return ret;
}


void CMudStream::slotProcessOutput(QString s)
{
  TriggerEntry* trigger=NULL;
  CCharacterProfile* charprof=NULL;
	CMudProfile* mudprof=NULL;
  QString rawoutput=s.copy();
  QString beautyfiedoutput,output,output2;
	PluginList pluginlist;
	PluginInfo* plugininfo;

	// for triggers
	QColor tOldFgColor, tOldBgColor;
	QFont tOldFont;
	QString tLeft,tMatch,tRight,tDummy;
	int tPos,ti,tj;
	TriggerEntry* tFirstTrigger;
	bool tLoop;
	QRegExp tre;


	if (doc) charprof = doc->getCharacterProfile (doc->getCurrentCharacterID ());
	if ((doc) && (entry)) mudprof = doc->getMudProfile(entry->mudfile);

  /* here goes your alias/trigger/whatever filter-function call */
  /* take "output" and return the result of your filter to "output" again */


	// plugins, raw output
	output=rawoutput;
	pluginlist=pluginmanager->getRunningPlugins();
	plugininfo=pluginlist.first();
	while (plugininfo!=NULL)
	{
		if (plugininfo->pointer!=NULL)
			output2 = plugininfo->pointer->processRawOutput(output);
		output=output2;
		plugininfo=pluginlist.next();
	}
	rawoutput=output;


	// mud view
  if (view)
	{
		// color triggers
		if (charprof)
		{
			// start with complete output
			tRight=rawoutput;
			// collect return values to one beautifiedoutput
			beautyfiedoutput="";

			// loop until no more triggers match the remaining string
			tLoop=true;
			while (tLoop)
			{
				// find first (regarding the position in string) matching color trigger
				tPos=tRight.length()+1; tFirstTrigger=NULL;
				trigger = charprof->firstTrigger ();
				while (!doc->getTriggersDisabled() && trigger)
				{
					if (trigger->type > 0)
					{
						// get index
						if (trigger->isRegEx)
						{
							tDummy=CMudView::stripANSI(tRight);
							tj=tDummy.find(QRegExp(trigger->name));
							ti=CMudView::translateIndexANSI(tj,tRight);
						}
						else
						{
							tDummy=CMudView::stripANSI(tRight);
							tj=tDummy.find(trigger->name);
							ti=CMudView::translateIndexANSI(tj,tRight);
						}

						if ((ti<tPos) && (ti>=0))
						{
							tPos=ti;
							tFirstTrigger=trigger;
						}
					}
					trigger = charprof->nextTrigger ();
				}

				// found one? -> split it and handle substrings seperately
				if (tFirstTrigger!=NULL)
				{
					// split string to left, matching and remaining substring
					if (tFirstTrigger->isRegEx)
					{
						tre=QRegExp(tFirstTrigger->name);
						tPos=tre.match(tRight,0,&ti);
					}
					else
					{
						tDummy=CMudView::stripANSI(tRight);
						tPos=tDummy.find(tFirstTrigger->name);
						tj=tPos;
						tPos=CMudView::translateIndexANSI(tj,tRight);
						ti=tFirstTrigger->name.length();
						ti=CMudView::translateIndexANSI(tj+ti,tRight)-tPos;
					}

					if (tFirstTrigger->type==2) // type 2, entire line
					{
						tPos=tRight.findRev('\n',tPos)+1;
						ti=tRight.find('\n',tPos+ti)-tPos; // length
						if ((tPos<0) || (ti<0))
						{ // something went wrong -> skip
							tLeft=tRight;
							tMatch="";
							tRight="";
						}
						else
						{
							tLeft=tRight.left(tPos);
							tMatch=tRight.mid(tPos,ti);
							tRight=tRight.right(tRight.length()-tLeft.length()-tMatch.length());
						}
					}
					else // default to type 1 (match only)
					{
						tLeft=tRight.left(tPos);
						tMatch=tRight.mid(tPos,ti);
						tRight=tRight.right(tRight.length()-tLeft.length()-tMatch.length());
					}


					// non-matching strings are easy, just do it as normal
					beautyfiedoutput+=view->displayMudOutput(tLeft);

					// save fg/bg/font
					tOldFgColor=view->getOutputActualForegroundColor();
					tOldBgColor=view->getOutputActualBackgroundColor();
					tOldFont=view->getOutputActualFont();

					// set fg/bg/font values to the one of the trigger
					if (tFirstTrigger->setFgc)
						view->setOutputActualForegroundColor(tFirstTrigger->fgc);
					if (tFirstTrigger->setBgc)
						view->setOutputActualBackgroundColor(tFirstTrigger->bgc);
					if (tFirstTrigger->setFont)
						view->setOutputActualFont(tFirstTrigger->font);

					// now display the string
					// ignore ANSI Colors from MUD and use our colors
					beautyfiedoutput+=view->displayMudOutput(tMatch,true);

					// restore previously saved fg/bg/font
					view->setOutputActualForegroundColor(tOldFgColor);
					view->setOutputActualBackgroundColor(tOldBgColor);
					view->setOutputActualFont(tOldFont);

				}
				// if no color trigger found, fallback to standard handling
				else
				{
					beautyfiedoutput+=view->displayMudOutput(tRight);
					tLoop=false; // we're done
				}

			} // while (tLoop)

		} else
  	{
			beautyfiedoutput = view->displayMudOutput(rawoutput);
		}
	}

	// plugins, parsed output
	pluginlist=pluginmanager->getRunningPlugins();
	plugininfo=pluginlist.first();
	output=beautyfiedoutput;
	while (plugininfo!=NULL)
	{
		if (plugininfo->pointer!=NULL)
			output2 = plugininfo->pointer->processParsedOutput(output);
		output=output2;
		plugininfo=pluginlist.next();
	}
	beautyfiedoutput=output;


  // logging
  if ((charprof) && (log))
  {
		if (charprof->getLogANSIEnabled()==true)
			log->append(rawoutput);
		else
			log->append(beautyfiedoutput);
  }



	// start normal triggers
  if ((charprof!=NULL) && (triggerLoop==false) && !doc->getTriggersDisabled())
  {
		tRight=beautyfiedoutput;
		tLoop=true;

		while (tLoop)
		{
			// find first (regarding the position in string) matching normal trigger
			tPos=tRight.length()+1; tFirstTrigger=NULL;
			trigger = charprof->firstTrigger ();
			while (trigger)
			{
				if (trigger->type == 0)
				{
					// get index
					if (trigger->isRegEx)
						ti=tRight.find(QRegExp(trigger->name));
					else
						ti=tRight.find(trigger->name);
					if ((ti<tPos) && (ti>=0))
					{
						tPos=ti;
						tFirstTrigger=trigger;
					}
				}
				trigger = charprof->nextTrigger ();
			}

			// found one? -> split it and handle substrings seperately
			if (tFirstTrigger!=NULL)
			{
				triggerLoop=triggerLoopDetect();

				// split string to left, matching and remaining substring
				if (tFirstTrigger->isRegEx)
				{
					tre=QRegExp(tFirstTrigger->name);
					tPos=tre.match(tRight,0,&ti);
				} else
				{
					tPos=tRight.find(tFirstTrigger->name);
					ti=tFirstTrigger->name.length();
				}

				if ((tPos<0) || (ti<0))
				{ // something went wrong -> skip
					tLeft=tRight;
					tMatch="";
					tRight="";
				}
				else
				{
					tLeft=tRight.left(tPos);
					tMatch=tRight.mid(tPos,ti);
					tRight=tRight.right(tRight.length()-tLeft.length()-tMatch.length());
					if (!doc->getTriggersDisabled())
					{
					  slotProcessInput (tFirstTrigger->command + "\n");
					}
				}
			}
			else
			{
				tLoop=false;
			}
		} // while (tLoop)
  }
	else // if ((prof!=NULL) && (triggerLoop==false))
	{
		// the thing about triggerLoopShowMessageBox is ugly but we have to do it because
		// this method is a slot and can (and surely will) be called more than once
		if ((triggerLoop==true) && (triggerLoopShowMessageBox==false))
		{
			triggerLoopShowMessageBox=true;
			QMessageBox::warning(view, i18n("Trigger loop detected"), i18n("Your triggers seem to loop.\nIgnoring triggers until you close this messagebox."));
			triggerLoop=false;
			triggerLoopShowMessageBox=false;
		}
	}
	// end normal trigger


	// valid move checking
	if (mapper!=NULL)
	{
		QString *dirCmd=directionCmdQueue.dequeue();

		if ((dirCmd!=NULL) && (mudprof!=NULL))
		{			
			if (mudprof->getDoValidMoveChecking())
			{
				bool movePlayer=true;

				if ((mudprof->getFailedMoveMsg1()!="") &&
						(output.contains(mudprof->getFailedMoveMsg1())))
						movePlayer=false;

				if ((mudprof->getFailedMoveMsg2()!="") &&
						(output.contains(mudprof->getFailedMoveMsg2())))
						movePlayer=false;

				if ((mudprof->getFailedMoveMsg3()!="") &&
						(output.contains(mudprof->getFailedMoveMsg3())))
						movePlayer=false;

				if (movePlayer)
				{
					mapper->movePlayerBy(*dirCmd);
				}
			}
			delete dirCmd;
		}
  }
		
  autologin(beautyfiedoutput);

  emit(processedOutput());

} // slotProcessOutput

void CMudStream::slotProcessInput(QString s)
{
	QString input=s.copy();
	QString output;
	CCharacterProfile* charprof=NULL;
	CMudProfile* mudprof=NULL;
	PluginList pluginlist;
	PluginInfo* plugininfo;
	bool doFancyStuff=true;
	int start,end;
	QString command;


	if (doc) charprof = doc->getCharacterProfile (doc->getCurrentCharacterID ());
	if ((doc) && (entry)) mudprof = doc->getMudProfile(entry->mudfile);

	/* here goes your alias/trigger/whatever filter-function call */
	/* take "input" and return the result of your filter to "input" again */

	// login/password not sent yet but it's in the inpout now
	// seems that the user is loggin in now -> don't do fancy stuff because
	// otherwise logging in with a passwd containing special char won't be possible
	if (entry!=NULL)
	{
		if ((LoginHasBeenSent==false) && (s.contains(entry->login)>0))
		{
			doFancyStuff=false;
			LoginHasBeenSent=true;
		}
		if ((PasswordHasBeenSent==false) && (s.contains(entry->password)>0))
		{
			doFancyStuff=false;
			PasswordHasBeenSent=true;
		}
 	}

	// do all the fancy stuff with the special chars only if we want
	// e.g. we don't want it when still entering login and password
	if (doFancyStuff==true)
	{

		if (view->getEnableSpecialChars()==true)
		{
			// speedwalking
			QString speedwalkingChar = view->getSpeedwalkingChar ();
			uint slen=speedwalkingChar.length();
			if ((slen> 0) && ((slen+ 1) < input.length ()) && (input.left (slen) == speedwalkingChar))
				inSpdWalk(input,speedwalkingChar);

			// extern commands
			QString externChar = view->getExternChar ();
			slen = externChar.length ();
			if ((slen > 0) && ((slen + 1) < input.length ()) && (input.left (slen) == externChar))
				inExternCommand (input, externChar);

			// separator-char-recognition
			QString separatorChar = view->getSeparator ();
			if (separatorChar != "")
				inSeparator (input, separatorChar);
		}

		// backslash recognition
		inBackslash(input);

	} // doFancyStuff


	// split input into single commands (seperator: \n) and send them individually
	start=0;
	end=input.find('\n',start);
	while (end!=-1)
	{
		command = input.mid(start,end-start);
		// if it's a mapper command send the mapper stuff before and after
		if ((mapper) && (mudprof) && (mudprof->getFollowMode()) && (mapper->validMoveCmd(command)))
		{
			mapper->executeBeforeExitCommand();
			mapper->executeBeforeEnterCommand(command);
			if (connection->isConnected ())
				connection->sendData(command+'\n',command.length()+1);
			mapper->executeAfterExitCommand();
			mapper->executeAfterEnterCommand(command);						

			// if we do valid move checking, queue movement of player on map
			if ((mudprof->getDoValidMoveChecking()==true) && (mapper->getMapperData()->speedwalkActive==false))
				directionCmdQueue.enqueue(new QString(command));	
			else
				mapper->movePlayerBy(command);	
		}
		else // otherwise just send it
		{
			if (connection->isConnected ())
				connection->sendData(command+'\n',command.length()+1);
		}

		// logging
		if ((log) && (connection->isEchoOn()))
			log->append(command+'\n');

		start=end+1;
		end=input.find('\n',start);
	} // while(end!=-1)
	




	// plugins
	pluginlist = pluginmanager->getRunningPlugins();
	plugininfo=pluginlist.first();
	while (plugininfo!=NULL)
	{
		if (plugininfo->pointer!=NULL)
			output = plugininfo->pointer->processInput(input);
		input=output;
		plugininfo=pluginlist.next();
	}


	view->slotAppendToOutput(input);
	emit(processedInput());
} // slotProcessInput

/** proccess speedwalking */
void CMudStream::inSpdWalk(QString & input, QString speedwalkingChar){
	int m, n;
	uint l;
	bool b;
	QString num="",com="",out="",temp=input.right(input.length()-speedwalkingChar.length()).stripWhiteSpace();
	m = 0;
	for (l = 0; l < temp.length (); l++){
		if ((temp[l] >= '0') && (temp[l] <= '9'))
		{
			if (com != "")
				for (n = 0; n < m; n++)
					out += com + "\n";					
			com = "";
			num += temp[l];
		} // if
		else
		{
			if (num != "")
			{
				m = num.toInt (&b);
				if (!b)
					m = 0;
			} // if
			num = "";
			com += temp[l];
		} // else
	}//for
	for (n = 0; n < m; n++)
		out += com + "\n";					
	input = out;
}

/** proccess backslashes */
void CMudStream::inBackslash(QString & input){
	QString temp= input.copy ();
	input = "";
	uint l;
	bool b = false;
	if (temp[0] != '\\')
		input += temp[0];
	for (l = 1; l < temp.length (); l++)
	{
		if ((temp[l - 1] == '\\') && (!b))
		{
			switch (temp[l]) {
				case 'n':
					input += '\n';
					break;
				case 'r':
					input += '\r';
					break;
				case 't':
					input += '\t';
					break;
				case 'a':
					input += '\a';
					break;
				case '\\':
					input += '\\';
					b = true;
					break;
				default:
					input += temp[l];
			}
		} // if
		else
		{
			if (temp[l] != '\\')
			{
				input += temp[l];
			} // if
			b = false;
		} // else
	} // for
}

/** splits the input into pieces, using the separator-char */
void CMudStream::inSeparator (QString & input, QString separatorChar)
{
	int spos = input.find (separatorChar);
	uint slen = separatorChar.length ();
	while (spos > -1)
	{
		input.replace (spos, slen, "\n");
		spos = input.find (separatorChar);
	} // while
} // inSeparator

/** processes call to external commands */
void CMudStream::inExternCommand (QString & input, QString externChar)
{
	KShellProcess proc;
	QString temp = input.copy ();
	temp.remove (0, externChar.length ());
	proc << temp;
	proc.start (proc.DontCare);
	input = "";
} // inExternCommand

/** reset's the autologin */
void CMudStream::reset ()
{
	LoginHasBeenSent=false;
	PasswordHasBeenSent=false;
	entry = NULL;
} // reset

void CMudStream::autologin(QString output)
{
	if (entry)
	{
			CMudProfile* selectedMud=doc->getMudProfile(entry->mudfile);
	
		  if (!LoginHasBeenSent)
		  {
				QString NamePrompt=selectedMud->getNamePrompt();
				if (NamePrompt!="")
				{
					if (output.contains(NamePrompt,false))
					{
						slotProcessInput(entry->login+"\n" );
			  		LoginHasBeenSent=true;
					}
				}
				else
				{
				  LoginHasBeenSent=true;
				}
			}
		  	
			if (!PasswordHasBeenSent)
			{
				if (entry->password!="")
				{
					QString PasswordPrompt=selectedMud->getPasswordPrompt();
					if (PasswordPrompt!="")
					{
						if (output.contains(PasswordPrompt,false))
						{
							slotProcessInput(entry->password+"\n" );
							PasswordHasBeenSent=true;
						}
					}
					else
					{
					  PasswordHasBeenSent=true;
					}
				}
				else
				{
				  PasswordHasBeenSent=true;
				}
			}
	}
}


